# FindPkgVersionWin.cmake
# --------------------
# description:
#   Defines the function `find_pkg_version_win`, which obtains the version of a
#   library from the `pkgconfig` (*.pc) files in a given path.
#   This is currently used only under Windows, as a simple replacement for
#   `PkgConfig` from MSYS2 platform.
#
# usage:
#   find_pkg_version_win(<library-name> <config-filename> <search-paths> <output-var-name>)
#
# parameters:
#   * library-name: name of the library (eg. "cerf")
#   * config-filename: name of the package config file(eg. "libcerf.pc").
#     If an empty string is given the default will be used:
#     'lib' + <library-name> + '.pc'
#     eg. 'libcerf.pc'
#   * search-paths: List of paths to search for the package config file.
#     The file will be searched also within `pkgconfig` subfolder of the
#     given paths.
#   * output-var-name: Name of the output variable.
#     If an empty string is given the default will be used:
#     <library-name> + '_VERSION'
#     eg. 'cerf_VERSION'
#
# output:
#   The function pushes the following variables into the parent scope:
#   * <library-name>_PATH: Path of the discovered package config file.
#   * output-var-name: The version of the package.
#
# example:
#   * Find the version of the 'cerf' package in the `CMAKE_LIBRARY_PATH` folders:
#     find_pkg_version_win("cerf" "" "${CMAKE_LIBRARY_PATH}" Cerf_VERSION)

function(find_pkg_version_win libname conffile paths output_var)

    string(STRIP "${paths}" _paths)
    string(STRIP "${conffile}" _pkgconffile)
    set(_msghdr "find_pkg_version_win::")  # message header

    # find the configuration file in the given paths
    if(NOT _pkgconffile)
        set(_pkgconffile "lib${libname}.pc")
    endif()

    # initialize the output variable
    set(_outvar ${output_var})
    if(NOT _outvar)
        set(_outvar ${libname}_VERSION)
    endif()
    set(${_outvar} "${_outvar}-NOTFOUND" PARENT_SCOPE)

    set(_path_sfx pkgconfig)  # search subdirectories

    # NOTE: ${libname}_PATH variable will be cached
    find_path("${libname}_PATH" "${_pkgconffile}"
        PATHS "${_paths}" PATH_SUFFIXES "${_path_sfx}"
        DOC "Package configuration for ${_pkgconffile}")

    if(NOT ${libname}_PATH)
        message(WARNING "${_msghdr} '${_pkgconffile}' not found in paths {${_paths}}")
        return()
    endif()

    message(STATUS "${_msghdr} '${_pkgconffile}' found in '${${libname}_PATH}'")

    # extract the configuration information
    file(READ "${${libname}_PATH}/${_pkgconffile}" _conf)
    string(TOLOWER "${_conf}" _conf_lc)
    string(REGEX MATCH "version[: \t]*([1-9.]+)" _rxmatch ${_conf_lc})

    if(CMAKE_MATCH_1)
        set(${_outvar} "${CMAKE_MATCH_1}" PARENT_SCOPE)
        message(STATUS "${_msghdr} ${_outvar} => ${CMAKE_MATCH_1}")
    else()
        message(WARNING "${_msghdr} ${_outvar} not found")
    endif()

endfunction()
