/**
 * Copyright (C) 2009-2014 Paul Fretwell - aka 'Footleg' (drfootleg@gmail.com)
 * 
 * This file is part of Cave Converter.
 * 
 * Cave Converter is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Cave Converter is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Cave Converter.  If not, see <http://www.gnu.org/licenses/>.
 */
package cavesurvey.regression;

import java.io.FileNotFoundException;
import java.text.ParseException;

import footleg.cavesurvey.cmdline.CaveConverter;
import footleg.cavesurvey.cmdline.CaveConverter.CmdlineOpt;
import footleg.cavesurvey.tools.UtilityFunctions;

/**
 * Regression helper class which runs data file conversions and compares the output with reference
 * files. Tests pass if the output file is identical to the reference file. A file name prefix
 * is given as the argument for each test conversion. The test will look for an input file with
 * this prefix and the suffix '_in._extn_' and compare the output with a reference file with
 * this prefix and the suffix '_ref._extn_' (where _extn_ is the appropriate file extension for
 * the file format.
 *  
 * @author      Footleg <drfootleg@gmail.com>
 * @version     2014.05.20                                (ISO 8601 YYYY.MM.DD)
 * @since       1.6                                       (The Java version used)
 */
public final class RunFileConverter {
	final static String refFilesLocation = "./test/data/regression/";
	final static String genFilesLocation = "./build/test/data/";
	private static boolean oldJRE = false;
	
	public static void setOldJRE(boolean oldJRE) {
		RunFileConverter.oldJRE = oldJRE;
	}

	public static String convertCompassToSurvex( String filePrefix, CmdlineOpt splaysOpt, CmdlineOpt genLRUDOpt ) throws ParseException, FileNotFoundException {
		//Compare resulting file with expected file
		String fileCompare = convertFile(filePrefix, UtilityFunctions.compassFormat, UtilityFunctions.survexFormat, 
				splaysOpt, genLRUDOpt);
		return fileCompare;
	}

	public static String convertDxfToSurvex( String filePrefix, CmdlineOpt splaysOpt, CmdlineOpt genLRUDOpt ) throws ParseException, FileNotFoundException {
		//Compare resulting file with expected file
		String fileCompare = convertFile(filePrefix, UtilityFunctions.dxfFormat, UtilityFunctions.survexFormat, 
				splaysOpt, genLRUDOpt);
		return fileCompare;
	}
	
	public static String convertSurvexToSurvex( String filePrefix, CmdlineOpt splaysOpt, CmdlineOpt genLRUDOpt ) throws ParseException, FileNotFoundException {
		//Compare resulting file with expected file
		String fileCompare = convertFile(filePrefix, UtilityFunctions.survexFormat, UtilityFunctions.survexFormat, 
				splaysOpt, genLRUDOpt);
		return fileCompare;
	}
	
	public static String convertSurvexToTopoRobot( String filePrefix, CmdlineOpt splaysOpt, CmdlineOpt genLRUDOpt ) throws ParseException, FileNotFoundException {
		//Compare resulting file with expected file
		String fileCompare = convertFile(filePrefix, UtilityFunctions.survexFormat, UtilityFunctions.toporobotFormat, 
				splaysOpt, genLRUDOpt);
		return fileCompare;
	}
	
	public static String convertDxfToTopoRobot( String filePrefix, CmdlineOpt splaysOpt, CmdlineOpt genLRUDOpt ) throws ParseException, FileNotFoundException {
		//Compare resulting file with expected file
		String fileCompare = convertFile(filePrefix, UtilityFunctions.dxfFormat, UtilityFunctions.toporobotFormat, 
				splaysOpt, genLRUDOpt);
		return fileCompare;
	}
	
	public static String convertPocketTopoToSurvex( String filePrefix, CmdlineOpt splaysOpt, CmdlineOpt genLRUDOpt ) throws ParseException, FileNotFoundException {
		//Compare resulting file with expected file
		String fileCompare = convertFile(filePrefix, UtilityFunctions.pocketTopoFormat, UtilityFunctions.survexFormat, 
				splaysOpt, genLRUDOpt);
		return fileCompare;
	}
	
	public static String convertPocketTopoToToporobot( String filePrefix, CmdlineOpt splaysOpt, CmdlineOpt genLRUDOpt ) throws ParseException, FileNotFoundException {
		//Compare resulting file with expected file
		String fileCompare = convertFile(filePrefix, UtilityFunctions.pocketTopoFormat, UtilityFunctions.toporobotFormat, 
				splaysOpt, genLRUDOpt);
		return fileCompare;
	}
	
	private static String generateFileSuffix( char format ) {
		String fileSuffix = "";
		
		if( format == UtilityFunctions.pocketTopoFormat ) {
			fileSuffix = ".txt";
		}
		else if( format == UtilityFunctions.dxfFormat ) {
			fileSuffix = ".dxf";
		}
		else if( format == UtilityFunctions.survexFormat ) {
			fileSuffix = ".svx";
		}
		else if( format == UtilityFunctions.toporobotFormat ) {
			fileSuffix = ".text";
		}
		else if( format == UtilityFunctions.compassFormat ) {
			fileSuffix = ".dat";
		}
		else {
			fileSuffix = ".txt";
		}
		
		return fileSuffix;
	}
	
	
	private static String generateInputFilename( String filePrefix, char fromFormat ) {
		return refFilesLocation + filePrefix + "_in" + generateFileSuffix( fromFormat );
	}
	
	private static String generateFilename( String location, String filePrefix, char fromFormat, char toFormat, 
			CmdlineOpt splaysOpt, String suffix ) {
		String splayOptString = "";
		if ( splaysOpt == CmdlineOpt.T ) {
			splayOptString = "_spl";
		}
		else if ( splaysOpt == CmdlineOpt.F ) {
			splayOptString = "_nsp";
		}
		return location + filePrefix + "_" + fromFormat + toFormat + splayOptString + "_" + suffix + generateFileSuffix( toFormat );
	}
	
	private static String generateRefFilename( String filePrefix, char fromFormat, char toFormat, CmdlineOpt splaysOpt, boolean oldJRE ) {
		String suffix = "ref";
		if ( oldJRE ) {
			suffix += "7";
		}
		return generateFilename( refFilesLocation, filePrefix, fromFormat, toFormat, splaysOpt, suffix );
	}
	
	private static String generateOutputFilename( String filePrefix, char fromFormat, char toFormat, CmdlineOpt splaysOpt ) {
		return generateFilename( genFilesLocation, filePrefix, fromFormat, toFormat, splaysOpt, "testgen" );
	}

	/**
	 * Sets the date used as 'today' in conversions so that output files match reference files
	 * with dates in them
	 */
	private static void setDateForConversions() {
		//Set date to a fixed date rather than the actual date at runtime so that test output uses a known date
		try {
			CaveConverter.setToday( UtilityFunctions.stringToDate("16/08/2012 13:14:15", "dd/MM/yyyy HH:mm:ss") );
		} catch (ParseException e) {
			//Just print out error (will never happen with hard coded arguments used in method call above)
			e.printStackTrace();
		}
	}
	
	private static String convertFile( String filePrefix, char fromFormat, char toFormat, 
			CmdlineOpt splaysOpt, CmdlineOpt genLRUDOpt ) throws ParseException, FileNotFoundException {
		
		String inputFile = generateInputFilename(filePrefix, fromFormat);
		String outputFile = generateOutputFilename(filePrefix, fromFormat, toFormat, splaysOpt);
		String referenceFile = generateRefFilename(filePrefix, fromFormat, toFormat, splaysOpt, oldJRE);
		
		setDateForConversions();
		 
		//Do file conversion
		CaveConverter.convertFile(inputFile, outputFile, fromFormat, toFormat, splaysOpt, genLRUDOpt);

		//Compare resulting file with expected file
		String fileCompare = DiffFiles.diffFiles( referenceFile, outputFile );
		
		return fileCompare;
	}
	
	public static String convertUsingCmdLine( String filePrefix, char fromFormat, char toFormat, String options ) throws ParseException, FileNotFoundException {
		String inputFile = generateInputFilename(filePrefix, fromFormat);
		String outputFile = generateOutputFilename(filePrefix, fromFormat, toFormat, CmdlineOpt.F);
		String referenceFile = generateRefFilename(filePrefix, fromFormat, toFormat, CmdlineOpt.F, oldJRE);
		
		//Set file suffix from command line parameters
		String fileSuffix = "_cmd";
		if ( options.contains("nosplays") ) {
			fileSuffix += "n";
		}
		else if ( options.contains("splays") ) {
			fileSuffix += "s";
		}
		if ( options.contains("lrud") ) {
			fileSuffix += "l";
		}
		
		//Change output and ref filenames to cmdline suffixes
		outputFile = outputFile.replace("_nsp", fileSuffix);
		referenceFile = referenceFile.replace("_nsp", fileSuffix);
		
		setDateForConversions();

		String[] opts = options.split(" ");
		int numArgs = 4 + opts.length;
		String[] args = new String[numArgs];
		args[0] = inputFile;
		args[1] = outputFile;
		args[2] = "" + fromFormat;
		args[3] = "" + toFormat;
		for (int iArg = 0; iArg < opts.length; iArg++ ) {
			args[4 + iArg] = opts[iArg];
		}

		CaveConverter.main(args);
		
		//Compare resulting file with expected file
		String fileCompare = DiffFiles.diffFiles( referenceFile, outputFile );
		
		return fileCompare;
	}
}
