/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2014 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#define _BSD_SOURCE /* lstat() */

#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
/* open() */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>


#include "cdw_config.h"
#include "cdw_fs.h"
#include "cdw_iso9660.h"
#include "isosize.h"
#include "cdw_debug.h"
#include "cdw_widgets.h"
#include "gettext.h"
#include "cdw_string.h"


/* FIXME: this is a global variable that sometimes is copied into
   temporary variable, which is further processed. Make sure that you
   operate on variable that holds a current configuration. */
extern cdw_config_t global_config;      /* variable holding cdw configuration */


/**
  \brief Function returns size of ISO9660 file system in current ISO image file

  Size of the file system is counted in 2048-byte sectors.
  Size of file system may be different than size of file on a disc
*/
long long cdw_iso_image_get_n_sectors(void)
{
	cdw_vdm ("INFO: inspecting file \"%s\"\n", global_config.general.image_fullpath);
	int f = open(global_config.general.image_fullpath, O_RDONLY);
	if (f == -1) {
		cdw_fs_errno_handler(errno);
		cdw_vdm ("ERROR: failed to open iso file image\n");
		return CDW_ERROR;
	}
	long long iso_size = isosize(f);
	close(f);

	long long fs_size = cdw_fs_get_file_size(global_config.general.image_fullpath);
	cdw_vdm ("INFO: iso_size = %lld, fs_size = %lld\n", iso_size, fs_size);

	if (iso_size % 2048) {
		cdw_vdm ("ERROR: iso_size is not multiple of 2048\n");
	}
	if (fs_size % 2048) {
		cdw_vdm ("WARNING: fs_size is not multiple of 2048\n");
	}

	long long size = iso_size;
	long long sectors = size / 2048;

	/* WARNING: sector size is set as constant (2048), but:
	   - 2048 is at the moment the only sector size supported by cdw
	   - mkisofs man page describes "-print-size" as: "Print
	   estimated filesystem size in multiples of the sector
	   size (2048 bytes) and exit". Why only 2048? Who should
	   re-calculate the size when another size should be used?
	   cdrecord or caller of cdrecord? */

	/* 524288 * 2048-byte sectors = 1GB */
	cdw_vdm ("INFO: returning %lld sectors (%.2f%s)\n",
		 sectors,
		 sectors >= 524288 ? (double) size / (1024 * 1024 * 1024) : (double) size / (1024 * 1024),
		 sectors >= 524288 ? "GB" : "MB");
	return sectors;
}





double cdw_iso_image_get_size_mb(const char *fullpath)
{
	struct stat stbuf;
	int rv = lstat(fullpath, &stbuf);
	if (rv != 0) {
		int e = errno;
		cdw_vdm ("ERROR: can't lstat() iso file \"%s\", error = \"%s\"\n", fullpath, strerror(e));
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("There seems to be problem with iso file. Please check the file location."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		cdw_fs_errno_handler(e);

		return CDW_ERROR;
	}

	long long iso_file_size_b = (long long) stbuf.st_size;
	double iso_file_size_mb = (double) (iso_file_size_b / (1024 * 1024));
	cdw_vdm ("INFO: size in bytes = %lld, size in megabytes = %f\n", iso_file_size_b, iso_file_size_mb);

	return iso_file_size_mb;
}





void cdw_iso9660_debug_print_options(cdw_iso9660_t *iso)
{
	cdw_vdm ("INFO: ISO9660 options:\n");
	cdw_vdm ("INFO:        joliet information = %s\n",     iso->joliet_information ? "true" : "false");
	cdw_vdm ("INFO:               joliet long = %s\n",     iso->joliet_long ? "true" : "false");
	cdw_vdm ("INFO:                rock ridge = %lld\n",   iso->rock_ridge);
	cdw_vdm ("INFO:           follow symlinks = %s\n",     iso->follow_symlinks ? "true" : "false");
	cdw_vdm ("INFO:                       pad = %s\n",     iso->pad ? "true" : "false");
	cdw_vdm ("INFO:                 iso level = %lld\n",   iso->iso_level);

	cdw_vdm ("INFO:                  root dir = \"%s\"\n", iso->root_dir);

	cdw_vdm ("INFO:     other mkisofs options = \"%s\"\n", iso->mkisofs_other_options);
	cdw_vdm ("INFO: other xorriso ISO options = \"%s\"\n", iso->xorriso_iso_other_options);

	cdw_vdm ("INFO:         boot disc options = \"%s\"\n", iso->boot_disc_options);

	cdw_vdm ("INFO:                 volume ID = \"%s\"\n", iso->volume_id);
	cdw_vdm ("INFO:             volume set ID = \"%s\"\n", iso->volume_set_id);
	cdw_vdm ("INFO:                  preparer = \"%s\"\n", iso->preparer);
	cdw_vdm ("INFO:                 publisher = \"%s\"\n", iso->publisher);
	cdw_vdm ("INFO:                 system id = \"%s\"\n", iso->system_id);
	cdw_vdm ("INFO:                 copyright = \"%s\"\n", iso->copyright);
	cdw_vdm ("INFO:                  abstract = \"%s\"\n", iso->abstract);

	return;
}





cdw_rv_t cdw_iso9660_copy(cdw_iso9660_t *target, cdw_iso9660_t *source)
{
	target->joliet_information  = source->joliet_information;
	target->joliet_long         = source->joliet_long;
	target->rock_ridge          = source->rock_ridge;
	target->follow_symlinks     = source->follow_symlinks;
	target->pad                 = source->pad;
	target->iso_level           = source->iso_level;

	int crv = cdw_string_set(&(target->root_dir), source->root_dir);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set root_dir with \"%s\"\n", source->root_dir);
		return CDW_ERROR;
	}

	crv = cdw_string_set(&(target->mkisofs_other_options), source->mkisofs_other_options);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set mkisofs_other_options with \"%s\"\n", source->mkisofs_other_options);
		return CDW_ERROR;
	}

	crv = cdw_string_set(&(target->xorriso_iso_other_options), source->xorriso_iso_other_options);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set xorriso_iso_other_options with \"%s\"\n", source->xorriso_iso_other_options);
		return CDW_ERROR;
	}

	crv = cdw_string_set(&(target->boot_disc_options), source->boot_disc_options);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set boot_disc_options with \"%s\"\n", source->boot_disc_options);
		return CDW_ERROR;
	}

	strncpy(target->volume_id, source->volume_id, CDW_ISO9660_VOLI_LEN);
	        target->volume_id[CDW_ISO9660_VOLI_LEN] = '\0';

	strncpy(target->volume_set_id, source->volume_set_id, CDW_ISO9660_VOLS_LEN);
	        target->volume_set_id[CDW_ISO9660_VOLS_LEN] = '\0';

	strncpy(target->preparer, source->preparer, CDW_ISO9660_PREP_LEN);
	        target->preparer[CDW_ISO9660_PREP_LEN] = '\0';

	strncpy(target->publisher, source->publisher, CDW_ISO9660_PUBL_LEN);
	        target->publisher[CDW_ISO9660_PUBL_LEN] = '\0';

	strncpy(target->system_id, source->system_id, CDW_ISO9660_SYSI_LEN);
	        target->system_id[CDW_ISO9660_SYSI_LEN] = '\0';

	strncpy(target->copyright, source->copyright, CDW_ISO9660_COPY_LEN);
	        target->copyright[CDW_ISO9660_COPY_LEN] = '\0';

	strncpy(target->abstract, source->abstract, CDW_ISO9660_ABST_LEN);
	        target->abstract[CDW_ISO9660_ABST_LEN] = '\0';

	return CDW_OK;
}





/*
  Initialize pointers from \p iso with NULL, so that \p iso can be
  later safely passed to cdw_iso9660_set_defaults().
*/
cdw_rv_t cdw_iso9660_init(cdw_iso9660_t *iso)
{
	iso->root_dir = (char *) NULL;
	iso->mkisofs_other_options = (char *) NULL;
	iso->xorriso_iso_other_options = (char *) NULL;
	iso->boot_disc_options = (char *) NULL;

	return CDW_OK;
}





/*
  Set default values for all fields in \p iso.
  You have to initialize \p iso with cw_iso9660_init() before calling this function.
*/
cdw_rv_t cdw_iso9660_set_defaults(cdw_iso9660_t *iso)
{
	iso->joliet_information = true;
	iso->joliet_long = false;         /* "Joliet long" breaks a standard, use with care. */
	iso->rock_ridge = CDW_ISO9660_RR_FULL;
	iso->follow_symlinks = false;
	iso->pad = true;
	iso->iso_level = 3;

	cdw_rv_t crv = cdw_string_set(&(iso->root_dir), "");
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set root_dir with empty string\n");
		return CDW_ERROR;
	}

	crv = cdw_string_set(&(iso->mkisofs_other_options), "");
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set mkisofs_other_options with empty string\n");
		return CDW_ERROR;
	}

	crv = cdw_string_set(&(iso->xorriso_iso_other_options), "");
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set xorriso_iso_other_options with empty string\n");
		return CDW_ERROR;
	}

	crv = cdw_string_set(&(iso->boot_disc_options), "");
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set boot_disc_options with empty string\n");
		return CDW_ERROR;
	}

	/* 2TRANS: this is a default label for ISO9660 volume */
	strncpy(iso->volume_id, _("my volume"), CDW_ISO9660_VOLI_LEN);
	        iso->volume_id[CDW_ISO9660_VOLI_LEN] = '\0';
	iso->volume_set_id[0] = '\0';
	iso->preparer[0]      = '\0';
	iso->publisher[0]     = '\0';
	iso->system_id[0]     = '\0';
	iso->copyright[0]     = '\0';
	iso->abstract[0]      = '\0';

	return CDW_OK;
}





/*
  Deallocate all pointers from \p iso, but not the \p iso itself.
*/
cdw_rv_t cdw_iso9660_clean(cdw_iso9660_t *iso)
{
	cdw_string_delete(&iso->root_dir);
	cdw_string_delete(&iso->mkisofs_other_options);
	cdw_string_delete(&iso->xorriso_iso_other_options);
	cdw_string_delete(&iso->boot_disc_options);

	return CDW_OK;
}
