/*
 * Author: Andrei Zavada <johnhommer@gmail.com>
 *         building on original work by Thomas Nowotny
 *
 * License: GPL-2+
 *
 * Initial version: 2008-09-02
 *
 * Class CModel
 */

/*--------------------------------------------------------------------------

The wrapper class which takes lists of pointers to neurons and synapses
which are networked to a neural system and assembles a common state
vector and handles the derivatives. At the same time it serves the neurons
and synapses their state at any given time and allows them to adjust their
parameters.

--------------------------------------------------------------------------*/


#ifndef LIBCN_MODEL_H
#define LIBCN_MODEL_H

#include <csignal>
#include <list>
#include <vector>
#include <string>

#include "libxml/parser.h"
#include "libxml/tree.h"

#include "gsl/gsl_rng.h"

#include "base-neuron.hh"
#include "base-synapse.hh"
#include "hosted-neurons.hh"
#include "hosted-synapses.hh"
#include "standalone-neurons.hh"
#include "standalone-synapses.hh"
#include "integrate-rk65.hh"

#include "config.h"


using namespace std;
namespace CNRun {

// CModel _status bits
#define CN_MDL_LOGDT		(1 << 0)
#define CN_MDL_LOGSPIKERS	(1 << 1)
#define CN_MDL_LOGUSINGID	(1 << 2)
#define CN_MDL_SORTUNITS	(1 << 3)
#define CN_MDL_ALLOWNOISE	(1 << 4)
#define CN_MDL_NOTREADY		(1 << 5)
#define CN_MDL_DISKLESS		(1 << 6)
#define CN_MDL_HAS_DDTB_UNITS	(1 << 7)
#define CN_MDL_DISPLAY_PROGRESS_PERCENT	(1 << 8)
#define CN_MDL_DISPLAY_PROGRESS_TIME	(1 << 9)
#define CN_MDL_DONT_COALESCE		(1 << 10)


class CModel {

    public:
	string	name;

    private:
	int	_status;
    public:
	int status()	{ return _status; }

      // structure ------
    friend class C_BaseSynapse;
    friend class C_HostedNeuron;
    friend class C_HostedConductanceBasedNeuron;
    friend class C_HostedRateBasedNeuron;
    friend class C_HostedSynapse;
    friend class CNeuronMap;
    friend class CSynapseMap;

    public:
	size_t units() const __attribute__ ((pure))
		{ return unit_list.size(); }

    private:
	unsigned long
		_global_unit_id_reservoir;
    private:
	list<C_BaseUnit*>		unit_list; // all units together
      // these have derivative(), are churned in _integrator->cycle()
	list<C_HostedNeuron*>		hosted_neu_list;
	list<C_HostedSynapse*>	hosted_syn_list;
      // these need preadvance() and fixate()
	list<C_StandaloneNeuron*>	standalone_neu_list;
	list<C_StandaloneSynapse*>	standalone_syn_list;
      // ... also these, but at discrete dt only
      // (only the standalone map units currently)
	list<C_StandaloneNeuron*>	ddtbound_neu_list;
	list<C_StandaloneSynapse*>	ddtbound_syn_list;

      // neurons that can possibly_fire() (various oscillators), and
      // have no inputs, and hence not dependent on anything else
	list<C_BaseNeuron*>		conscious_neu_list;

      // various lists to avoid traversing all of them in unit_list:
      // listeners, spikeloggers & readers
	list<C_BaseUnit*>		lisn_unit_list;
      // uses a meaningful do_spikelogging_or_whatever
	list<C_BaseNeuron*>		spikelogging_neu_list;
      // `Multiplexing AB' synapses are treated very specially
	list<C_BaseSynapse*>		mx_syn_list;

      // those for which apprise_from_source( model_time()) will be called
	list<C_BaseUnit*>		units_with_continuous_sources;
      // same, but not every cycle
	list<C_BaseUnit*>		units_with_periodic_sources;
	list<double>		regular_periods;
	list<unsigned>		regular_periods_last_checked;

    public:
	C_BaseUnit    *unit_by_label( const char *) const __attribute__ ((pure));
	C_BaseNeuron  *neuron_by_label( const char *) const __attribute__ ((pure));
	C_BaseSynapse *synapse_by_label( const char *) const __attribute__ ((pure));

	size_t hosted_unit_cnt() const __attribute__ ((pure))
		{  return hosted_neu_list.size() + hosted_syn_list.size();		}
	size_t standalone_unit_cnt() const __attribute__ ((pure))
		{  return standalone_neu_list.size() + standalone_syn_list.size();	}
	size_t ddtbound_unit_cnt() const __attribute__ ((pure))
		{  return ddtbound_neu_list.size() + ddtbound_syn_list.size();		}
	size_t total_neuron_cnt() const __attribute__ ((pure))
		{  return hosted_neu_list.size()
				+ standalone_neu_list.size()
				+ ddtbound_neu_list.size();		}
	size_t total_synapse_cnt() const __attribute__ ((pure))
		{  return hosted_syn_list.size()
				+ standalone_syn_list.size()
				+ ddtbound_syn_list.size();		}
	size_t conscious_neuron_cnt() const __attribute__ ((pure))
		{  return conscious_neu_list.size();	}
	size_t listening_unit_cnt() const __attribute__ ((pure))
		{  return lisn_unit_list.size();	}
	size_t spikelogging_neuron_cnt() const __attribute__ ((pure))
		{  return spikelogging_neu_list.size();	}

	size_t unit_with_continuous_sources_cnt() const __attribute__ ((pure))
		{  return units_with_continuous_sources.size();	}
	size_t unit_with_periodic_sources_cnt() const __attribute__ ((pure))
		{  return units_with_periodic_sources.size();	}

      // if is_last == true, do allocations of hosted units' vars immediately
      // otherwise defer until addition is done with is_last == true
      // or the user calls finalize_additions
	int include_unit( C_HostedNeuron*, bool is_last = true);
	int include_unit( C_HostedSynapse*, bool is_last = true);
	int include_unit( C_StandaloneNeuron*);
	int include_unit( C_StandaloneSynapse*);

	C_BaseNeuron *add_neuron_species( TUnitType type, const char *label, bool finalize = true,
					  double x = 0., double y = 0., double z = 0.);
	C_BaseNeuron *add_neuron_species( const char *type, const char *label, bool finalize = true,
					  double x = 0., double y = 0., double z = 0.);
	C_BaseSynapse *add_synapse_species( const char *type, const char *src_l, const char *tgt_l,
					    double g, bool allow_clone = true, bool finalize = true);
	void finalize_additions();
    private:
	C_BaseSynapse *add_synapse_species( TUnitType type, C_BaseNeuron *src, C_BaseNeuron *tgt,
					    double g, bool allow_clone, bool finalize);
	void _include_base_unit( C_BaseUnit*);
//	int _check_new_synapse( C_BaseSynapse*);
    public:
	C_BaseUnit* exclude_unit( C_BaseUnit*, bool do_delete = false);
	// return nullptr if do_delete == true, the excluded unit otherwise, even if it was not owned
	void delete_unit( C_BaseUnit* u)
		{  exclude_unit( u, true);  }
    private:
	friend class C_BaseUnit;
	void register_listener( C_BaseUnit*);
	void unregister_listener( C_BaseUnit*);
	friend class C_BaseNeuron;
	friend class SSpikeloggerService;
	void register_spikelogger( C_BaseNeuron*);
	void unregister_spikelogger( C_BaseNeuron*);
	void register_mx_synapse( C_BaseSynapse*);
	void unregister_mx_synapse( C_BaseSynapse*);

	void register_unit_with_sources( C_BaseUnit*);
	void unregister_unit_with_sources( C_BaseUnit*);

    private:
	unsigned short	_longest_label;
    public:
	unsigned short longest_label()  { return _longest_label; }

    public:
      // ctor, dtor
	CModel( const char *inname, CIntegrate_base *inRK65Setup, int instatus);
       ~CModel();

	void reset( bool also_reset_params = false);
	void reset_state_all_units();

    public:
      // NeuroMl interface
	int import_NetworkML( const char *fname, bool appending = false);
	int import_NetworkML( xmlDoc *doc, const char *fname, bool appending = false);  // fname is merely informational here
	void cull_deaf_synapses();  // those with gsyn == 0
	void cull_blind_synapses(); // those with _source == nullptr
	int export_NetworkML( const char *fname);
	int export_NetworkML( xmlDoc *doc);
	void dump_metrics( FILE *strm = stdout);
	void dump_state( FILE *strm = stdout);
	void dump_units( FILE *strm = stdout);
    private:
	int _process_populations( xmlNode*);
	int _process_population_instances( xmlNode*, const xmlChar*, const xmlChar*);

	int _process_projections( xmlNode*);
	int _process_projection_connections( xmlNode*, const xmlChar*, const xmlChar*,
					     const xmlChar *src_grp_prefix, const xmlChar *tgt_grp_prefix);

      // the essential mechanical parts: ----
      // hosted unit variables
    private:
	vector<double> V,	// contains catenated var vectors of all constituent neurons and synapses
		       W;	// V and W alternate in the capacity of the main vector, so avoiding many a memcpy
	size_t	_var_cnt;	// total # of variables (to be) allocated in V an W, plus one for model_time
    public:
	size_t vars()			{ return _var_cnt;         }

      // integrator interface
    private:
	friend class CIntegrate_base;
	friend class CIntegrateRK65;
    public:
	CIntegrate_base
		*_integrator;
	const double& model_time() const		{ return V[0];            }

	double& dt() const				{ return _integrator->dt;	}
	double& dt_min() const				{ return _integrator->_dt_min;	}
	double& dt_max() const				{ return _integrator->_dt_max;	}
      // this one is internal
	friend class CSynapseMxAB_dd;
    private:
	const double& model_time( vector<double> &x)	{ return x[0];            }

    private:
	unsigned long
		_cycle;
	double	_discrete_time;
	double	_discrete_dt;
    public:
	unsigned long cycle()				{ return _cycle;          }
	const double& model_discrete_time()		{ return _discrete_time;  }
	const double& discrete_dt()			{ return _discrete_dt;    }

      // simulation
    private:
	void _setup_schedulers();
	void prepare_advance();
//	void ctrl_c_handler( int);
	unsigned _do_advance_on_pure_hosted( double, double*)  __attribute__ ((hot));
	unsigned _do_advance_on_pure_standalone( double, double*) __attribute__ ((hot));
	unsigned _do_advance_on_pure_ddtbound( double, double*) __attribute__ ((hot));
	unsigned _do_advance_on_mixed( double, double*) __attribute__ ((hot));
    public:
	unsigned advance( double dist, double *cpu_time_p = nullptr) __attribute__ ((hot));

    public:
	double	spike_threshold, // above which neurons will detect a spike
		spike_lapse;  // if occurs less than this after the unit's _last_spike_end
	// (begs to be moved to SSpikeloggerService)

    public:
	float	listen_dt;
    private:
	ofstream
		*_dt_logger, *_spike_logger;

    public:
      // high-level functions to manipulate unit behaviour, set params, & connect sources
	struct STagGroup {
		string pattern;
		bool enable;
		STagGroup( const char *a, bool b = true)
		      : pattern (a), enable (b) {}
	};
	struct STagGroupListener : STagGroup {
		int bits;
		STagGroupListener( const char *a, bool b, int c = 0)
		      : STagGroup (a, b), bits (c) {}
	};
	int process_listener_tags( const list<STagGroupListener>&);

	struct STagGroupSpikelogger : STagGroup {
		double period, sigma, from;
		STagGroupSpikelogger( const char *a, bool b,
				      double c = 0., double d = 0., double e = 0.)  // defaults disable sdf computation
		      : STagGroup (a, b), period (c), sigma (d), from (e) {}
	};
	int process_spikelogger_tags( const list<STagGroupSpikelogger>&);

	int process_putout_tags( const list<STagGroup>&);

	struct STagGroupDecimate : STagGroup {
		float fraction;
		STagGroupDecimate( const char *a, double c)
		      : STagGroup (a), fraction (c) {}
	};
	int process_decimate_tags( const list<STagGroupDecimate>&);

	struct STagGroupNeuronParmSet : STagGroup {
		string parm;
		double value;
		STagGroupNeuronParmSet( const char *a, bool b, const char *c, double d)  // b == false to revert to stock
		      : STagGroup (a, b), parm (c), value (d)
			{}
	};
	struct STagGroupSynapseParmSet : STagGroupNeuronParmSet {
		string target;
		STagGroupSynapseParmSet( const char *a, const char *z, bool b, const char *c, double d)
		      : STagGroupNeuronParmSet (a, b, c, d), target (z)
			{}
	};
	int process_paramset_static_tags( const list<STagGroupNeuronParmSet>&);
	int process_paramset_static_tags( const list<STagGroupSynapseParmSet>&);
    private:
	void coalesce_synapses();  // those which used to be clones then made independent

    public:
	struct STagGroupSource : STagGroup {
		string parm;
		C_BaseSource *source;
		STagGroupSource( const char *a, bool b, const char *c, C_BaseSource *d)  // b == false to revert to stock
		      :  STagGroup (a, b), parm (c), source (d)
			{}
	};
	int process_paramset_source_tags( const list<STagGroupSource>&);

	list<C_BaseSource*> Sources;
	C_BaseSource* source_by_id( const char *id)
		{
			auto K = Sources.begin();
			while ( K != Sources.end() ) {
				if ( (*K)->name == id )
					return *K;
				K++;
			}
			return nullptr;
		}

    public:
	int	verbosely;

	gsl_rng	*_rng;

	double rng_sample()
		{
			return gsl_rng_uniform_pos( _rng);
		}

      // various convenience fields and members
	typedef list<C_BaseUnit*>::iterator lBUi;
	typedef list<C_BaseUnit*>::const_iterator lBUci;
	typedef list<C_BaseUnit*>::reverse_iterator lBUri;
	typedef list<C_BaseUnit*>::const_reverse_iterator lBUcri;

	lBUi  ulist_begin()	{ return unit_list.begin();		}
	lBUi  ulist_end()	{ return unit_list.end();		}
	lBUri ulist_rbegin()	{ return unit_list.rbegin();		}
	lBUri ulist_rend()	{ return unit_list.rend();		}

	lBUi lulist_begin()	{ return lisn_unit_list.begin();	}
	lBUi lulist_end()	{ return lisn_unit_list.end();		}
	list<C_BaseNeuron*>::		iterator knlist_begin()	{ return spikelogging_neu_list.begin();	}
	list<C_BaseNeuron*>::		iterator knlist_end()	{ return spikelogging_neu_list.end();	}

	// lBUi rlist_rbegin()	{ return reader_unit_list.rbegin();	}
	// lBUi rlist_rend()	{ return reader_unit_list.rend();	}
};




// by popular demand
#define for_all_units(U) \
	for ( auto U = ulist_begin(); U != ulist_end(); ++U )
#define for_all_units_const(U) \
	for ( auto U = unit_list.begin(); U != unit_list.end(); ++U )
#define for_all_neurons(U) \
	for ( auto U = ulist_begin(); U != ulist_end(); ++U ) if ( (*U)->is_neuron() )
#define for_all_synapses(U) \
	for ( auto U = ulist_begin(); U != ulist_end(); ++U ) if ( (*U)->is_synapse() )
#define for_all_neurons_reversed(U) \
	for ( auto U = ulist_rbegin(); U != ulist_rend(); ++U ) if ( (*U)->is_neuron() )
#define for_all_synapses_reversed(U) \
	for ( auto U = ulist_rbegin(); U != ulist_rend(); ++U ) if ( (*U)->is_synapse() )

#define for_all_hosted_neurons(U) \
	for ( auto U = hosted_neu_list.begin(); U != hosted_neu_list.end(); ++U )
#define for_all_hosted_synapses(U) \
	for ( auto U = hosted_syn_list.begin(); U != hosted_syn_list.end(); ++U )
#define for_all_standalone_neurons(U) \
	for ( auto U = standalone_neu_list.begin(); U != standalone_neu_list.end(); ++U )
#define for_all_standalone_synapses(U) \
	for ( auto U = standalone_syn_list.begin(); U != standalone_syn_list.end(); ++U )
#define for_all_ddtbound_neurons(U) \
	for ( auto U = ddtbound_neu_list.begin(); U != ddtbound_neu_list.end(); ++U )
#define for_all_ddtbound_synapses(U) \
	for ( auto U = ddtbound_syn_list.begin(); U != ddtbound_syn_list.end(); ++U )

#define for_all_units_with_contiuous_sources(U) \
	for ( auto U = units_with_continuous_sources.begin(); U != units_with_continuous_sources.end(); ++U )
#define for_all_units_with_periodic_sources(U) \
	for ( auto U = units_with_periodic_sources.begin(); U != units_with_periodic_sources.end(); ++U )

#define for_all_units_reversed(U) \
	for ( auto U = ulist_rbegin(); U != ulist_rend(); ++U )
#define for_all_readers_reversed(U) \
	for ( auto U = rlist_rbegin(); U != rlist_rend(); ++U )

#define for_all_hosted_neurons_reversed(U)       for ( auto U = hosted_neu_list.rbegin(); U != hosted_neu_list.rend(); ++U )
#define for_all_hosted_synapses_reversed(U)      for ( auto U = hosted_syn_list.rbegin(); U != hosted_syn_list.rend(); ++U )
#define for_all_standalone_synapses_reversed(U)  for ( auto U = standalone_syn_list.rbegin(); U != standalone_syn_list.rend(); ++U )

#define for_all_listening_units(U) \
	for ( auto U = lulist_begin(); U != lulist_end(); ++U )
#define for_all_conscious_neurons(N) \
	for ( auto N = conscious_neu_list.begin(); N != conscious_neu_list.end(); ++N )
#define for_all_spikelogging_neurons(N)	\
	for ( auto N = knlist_begin(); N != knlist_end(); ++N )
#define for_all_mx_synapses(N)	\
	for ( auto Y = mx_syn_list.begin(); Y != mx_syn_list.end(); ++Y )


#define for_model_units(M,U) \
	for ( auto U = M->ulist_begin(); U != M->ulist_end(); ++U )
#define for_model_units_reversed(M,U)  \
	for ( auto U = M->ulist_rbegin(); U != M->ulist_rend(); ++U )

#define for_model_hosted_neurons(M,U)  \
	for ( auto U = M->hosted_neu_list.begin(); U != M->hosted_neu_list.end(); ++U )
#define for_model_hosted_synapses(M,U)  \
	for ( auto U = M->hosted_syn_list.begin(); U != M->hosted_syn_list.end(); ++U )
#define for_model_hosted_neurons_reversed(M,U)   for ( auto U = M->hnlist_rbegin(); U != M->hnlist_rend(); ++U )
#define for_model_hosted_synapses_reversed(M,U)  for ( auto U = M->hylist_rbegin(); U != M->hylist_rend(); ++U )

#define for_model_standalone_neurons(M,U)   for ( auto U = M->snlist_begin(); U != M->snlist_end(); ++U )
#define for_model_standalone_synapses(M,U)  for ( auto U = M->sylist_begin(); U != M->sylist_end(); ++U )
#define for_model_standalone_neurons_reversed(M,U)   for ( auto U = M->snlist_rbegin(); U != M->snlist_rend(); ++U )
#define for_model_standalone_synapses_reversed(M,U)  for ( auto U = M->sylist_rbegin(); U != M->sylist_rend(); ++U )

#define for_model_neuron_units(M,U)   for_model_units(M,U) if ( (*U)->is_neuron() )
#define for_model_synapse_units(M,U)  for_model_units(M,U) if ( (*U)->is_synapse() )

#define for_model_spikelogging_neurons(M,N)	for ( auto N = M->knlist_begin(); N != M->knlist_end(); ++N )


// return values for import_NetworkML
#define CN_NMLIN_NOFILE		-1
#define CN_NMLIN_NOELEM		-2
#define CN_NMLIN_BADATTR	-3
#define CN_NMLIN_BADCELLTYPE	-4
#define CN_NMLIN_BIGLABEL	-5
#define CN_NMLIN_STRUCTERROR	-6






inline void
CIntegrateRK65::fixate()
{
	swap( model->V, model->W);
}


// various CUnit & CNeuron methods accessing CModel members
// that we want to have inline

inline const double&
C_BaseUnit::model_time() const
{
	return M->model_time();
}

inline void
C_BaseUnit::pause_listening()
{
	if ( M )
		M->unregister_listener( this);
}

inline void
C_BaseUnit::resume_listening()
{
	if ( M )
		M->register_listener( this);
}



template <class T>
void
C_BaseUnit::attach_source( T *s, TSinkType t, unsigned short idx)
{
	sources.push_back( SSourceInterface<T>( s, t, idx));
	M->register_unit_with_sources(this);
}





inline SSpikeloggerService*
C_BaseNeuron::enable_spikelogging_service( int s_mask)
{
	if ( !_spikelogger_agent )
		_spikelogger_agent = new SSpikeloggerService( this, s_mask);
	M->register_spikelogger( this);
	return _spikelogger_agent;
}
inline SSpikeloggerService*
C_BaseNeuron::enable_spikelogging_service( double sample_period, double sigma, double from, int s_mask)
{
	if ( !_spikelogger_agent )
		_spikelogger_agent = new SSpikeloggerService( this, sample_period, sigma, from, s_mask);
	M->register_spikelogger( this);
	return _spikelogger_agent;
}

inline void
C_BaseNeuron::disable_spikelogging_service()
{
	if ( _spikelogger_agent && !(_spikelogger_agent->_status & CN_KL_PERSIST)) {
		_spikelogger_agent->sync_history();
		M->unregister_spikelogger( this);

		delete _spikelogger_agent;
		_spikelogger_agent = nullptr;
	}
}






inline void
C_HostedNeuron::reset_vars()
{
//	cout << "reset_vars() on " << label << " (idx " << idx << ")\n";
	if ( M && idx < M->_var_cnt )
		memcpy( &M->V[idx],
			__CNUDT[_type].stock_var_values,
			__CNUDT[_type].vno * sizeof(double));
}

inline double&
C_HostedNeuron::var_value( size_t v)
{
	return M->V[idx + v];
}

inline const double&
C_HostedNeuron::get_var_value( size_t v) const
{
	return M->V[idx + v];
}



inline unsigned
C_HostedConductanceBasedNeuron::n_spikes_in_last_dt() const
{
	return E() >= M->spike_threshold;
}

inline unsigned
C_HostedRateBasedNeuron::n_spikes_in_last_dt() const
{
	return round(E() * M->dt() * M->rng_sample());
}


inline unsigned
C_StandaloneConductanceBasedNeuron::n_spikes_in_last_dt() const
{
	return E() >= M->spike_threshold;
}

inline unsigned
C_StandaloneRateBasedNeuron::n_spikes_in_last_dt() const
{
	return round(E() * M->dt() * M->rng_sample());
}








inline void
C_HostedSynapse::reset_vars()
{
//	cout << "reset_vars() on " << label << " (idx " << idx << ")\n";
	if ( M && M->_var_cnt > idx )
		memcpy( &M->V[idx],
			__CNUDT[_type].stock_var_values,
			__CNUDT[_type].vno * sizeof(double));
}



inline double&
C_HostedSynapse::var_value( size_t v)
{
	return M->V[idx + v];
}

inline const double&
C_HostedSynapse::get_var_value( size_t v) const
{
	return M->V[idx + v];
}



inline double
C_HostedConductanceBasedNeuron::E() const
{
	return M->V[idx+0];
}

// F is computed on the fly, so far usually


inline double
C_HostedSynapse::S() const
{
	return M->V[idx+0];
}






inline
CNeuronMap::CNeuronMap( const char *inlabel, double x, double y, double z, CModel *inM, int s_mask)
      : C_StandaloneConductanceBasedNeuron( NT_MAP, inlabel, x, y, z, inM, s_mask)
{
	if ( inM ) {
		if ( isfinite( inM->_discrete_dt) && inM->_discrete_dt != fixed_dt ) {
			printf( "Inappropriate discrete dt\n");
			_status |= CN_UERROR;
		}
		inM -> _discrete_dt = fixed_dt;
	}
}


inline
CSynapseMap::CSynapseMap( C_BaseNeuron *insource, C_BaseNeuron *intarget,
			  double ing, CModel *inM, int s_mask, TUnitType alt_type)
      : C_StandaloneSynapse( alt_type, insource, intarget, ing, inM, s_mask),
	_source_was_spiking (false)
{
	if ( !inM )
		fprintf( stderr, "A MxMap synapse is created unattached to a model: preadvance() will cause a segfault!\n");
	else {
		if ( isfinite( inM->_discrete_dt) && inM->_discrete_dt != fixed_dt ) {
			printf( "Inappropriate discrete dt\n");
			_status |= CN_UERROR;
		}
		inM -> _discrete_dt = fixed_dt;
	}
}


inline void
CSynapseMap::preadvance()
{
//	printf( "fafa %s\n", label);
	V_next[0] = S() * exp( -M->discrete_dt() / P[_tau_])
		+ (_source->n_spikes_in_last_dt() ? P[_delta_] : 0);

//	V_next[1] = ;
}



inline void
CSynapseMxMap::preadvance()
{
	V_next[0] = S() * exp( -M->discrete_dt() / P[_tau_]) + q() * P[_delta_];
}

}

#endif

// EOF
