/*
 * Copyright © 2011 Canonical Limited
 *
 * This library is free software: you can redistribute it and/or modify it under the terms of version 3 of the
 * GNU Lesser General Public License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this program.  If not,
 * see <http://www.gnu.org/licenses/>.
 *
 * Author: Ryan Lortie <desrt@desrt.ca>
 */

#include "qconf.h"

#include "qconfschema.h"
#include "qconftypes.h"

#include <dconf-dbus-1.h>
#include <QtDBus>

struct _QConfPrivate {
  const QConfSchema *schema;
  QString           *path;
  bool               dynamic;
};

static DConfDBusClient *dconfClient;

static void qconf_notify(DConfDBusClient *, const char *key, void *user_data) {
    QConf *self = (QConf *) user_data;

    self->notify(key);
}

void QConf::notify(const char *key) {
    const char *rel = key + priv->path->size();
    int id = priv->schema->findKey(rel);

    if (id < 0)
        return;

    GVariant *value = dconf_dbus_client_read(dconfClient, key);

    if (value == NULL)
        value = priv->schema->defaultValue(id);

    if (!g_variant_is_of_type(value, priv->schema->valueType(id))) {
        g_variant_unref(value);
        value = priv->schema->defaultValue(id);
    }

    QVariant qvar = qconf_types_to_qvariant(value);
    g_variant_unref(value);

    void *args[2] = { 0, (void *) qvar.constData() };

    QMetaObject::activate(this, priv->schema->metaObject(), id, args);
}

static void dconfClientInitialise()
{
    if (dconfClient == NULL)
        dconfClient = dconf_dbus_client_new(NULL,
                                            (DBusConnection *) QDBusConnection::sessionBus().internalPointer(),
                                            (DBusConnection *) QDBusConnection::systemBus().internalPointer());
}

void QConf::setSchema(const QString& schema) {
    const char *schema_path;

    priv->schema = QConfSchema::getSchema(schema);

    schema_path = priv->schema->path();

    if (schema_path)
        setPath(schema_path);
}

void QConf::setPath (const QString& path) {
    if (priv->path != NULL) {
        if (priv->schema != NULL && priv->schema->path() != NULL)
            // Give a more helpful error message in the more-likely case.
            qFatal("QConf path specified, but schema '%s' has its own path",
                   priv->schema->name().toUtf8().constData());

        else
            // This really shouldn't happen...
            qFatal("QConf path has already been specified");
    }

    priv->path = new QString(path);

    dconf_dbus_client_subscribe(dconfClient, priv->path->toUtf8(), qconf_notify, this);
}

QConf::QConf(const QString& schema, const QString& path)
{
    dconfClientInitialise();

    priv = new QConfPrivate;
    priv->schema = NULL;
    priv->path = NULL;
    priv->dynamic = false;

    setSchema(schema);
    setPath(path);
}

QConf::QConf(const QString & schema)
{
    dconfClientInitialise();

    priv = new QConfPrivate;
    priv->schema = NULL;
    priv->path = NULL;
    priv->dynamic = false;
    setSchema(schema);
}

QConf::QConf()
{
    dconfClientInitialise();

    priv = new QConfPrivate;
    priv->schema = NULL;
    priv->path = NULL;
    priv->dynamic = false;
}

QConf::~QConf() {
    if (priv->path != NULL)
        delete priv->path;

    delete priv;
}

const QMetaObject *QConf::metaObject() const
{
    /* First, we respond to qt_metacall() requests as per our staticMetaObject table.  After this function is
     * called, we handle them dynamically, as per the GSettings schema.
     *
     * This gives QML a chance to set the "schema" and "path" properties
     * before we start sending the property sets to dconf.
     */
    priv->dynamic = true;

    if (priv->schema == NULL)
        qFatal("QConf must have schema specified");

    if (priv->path == NULL)
        qFatal("QConf schema '%s' has no path and none specified",
                priv->schema->name().toUtf8().constData());

    return priv->schema->metaObject();
}

void *QConf::qt_metacast(const char *name)
{
    if (name == NULL)
        return NULL;

    if (strcmp(name, "QConf") == 0)
        return static_cast<void*>(const_cast<QConf*>(this));

    return QObject::qt_metacast(name);
}

int
QConf::qt_metacall(QMetaObject::Call   call,
                   int                 id,
                   void              **params)
{
    id = QObject::qt_metacall(call, id, params);

    if (id < 0)
        return id;

    if (priv->dynamic) {
        switch (call) {
        case QMetaObject::WriteProperty:
            dconf_dbus_client_write(dconfClient, (*priv->path + priv->schema->keyName(id)).toUtf8(),
                                    qconf_types_collect(priv->schema->valueType(id), params[0]));
            id -= priv->schema->count();
            break;

        case QMetaObject::ReadProperty:
            {
                GVariant *value;

                value = dconf_dbus_client_read(dconfClient,
                                               (*priv->path + priv->schema->keyName(id)).toUtf8());

                /* if the value was missing from dconf, fill it in */
                if (value == NULL)
                    value = priv->schema->defaultValue(id);

                /* maybe the value was not missing from dconf, but had the wrong type.  in that case, we want to
                 * ignore it and replace it with the default value.
                 */
                if (!g_variant_is_of_type(value, priv->schema->valueType(id))) {
                    g_variant_unref(value);
                    value = priv->schema->defaultValue(id);
                }

                qconf_types_unpack(value, params[0]);
                g_variant_unref(value);
            }
            id -= priv->schema->count();
            break;

        case QMetaObject::ResetProperty:
            dconf_dbus_client_write(dconfClient,
                                    (*priv->path + priv->schema->keyName(id)).toUtf8(), NULL);
            id -= priv->schema->count();
            break;

        case QMetaObject::QueryPropertyDesignable:
        case QMetaObject::QueryPropertyScriptable:
        case QMetaObject::QueryPropertyStored:
        case QMetaObject::QueryPropertyEditable:
        case QMetaObject::QueryPropertyUser:
            id -= priv->schema->count();
            break;

        case QMetaObject::CreateInstance:
            Q_ASSERT(false);

        case QMetaObject::InvokeMetaMethod:
            if (id < priv->schema->count()) {
                // People should not be invoking signals like this
                Q_ASSERT(false);
            } else {
                id -= priv->schema->count();
                dconf_dbus_client_write(dconfClient, (*priv->path + priv->schema->keyName(id)).toUtf8(),
                                        qconf_types_collect(priv->schema->valueType(id), params[1]));
                id -= priv->schema->count();
            }
        }
    } else {
        // handle static properties (for QML construction), see below
        switch (call) {
        case QMetaObject::WriteProperty:
            if (id == 0) {
                setSchema(*reinterpret_cast<QString *>(params[0]));
            } else if (id == 1) {
                setPath(*reinterpret_cast<QString *>(params[0]));
            }

            // fall through
        case QMetaObject::ResetProperty:
        case QMetaObject::ReadProperty:
        case QMetaObject::QueryPropertyDesignable:
        case QMetaObject::QueryPropertyScriptable:
        case QMetaObject::QueryPropertyStored:
        case QMetaObject::QueryPropertyEditable:
        case QMetaObject::QueryPropertyUser:
            id -= 2;
            break;

        case QMetaObject::CreateInstance:
        case QMetaObject::InvokeMetaMethod:
            Q_ASSERT(false);
        }
    }

    return id;
}

// These definitions are placed here for the benefit of QML construction
static const char static_string_data[] = {              // offset
    "QConf\0"                                           // 0
    "QString\0"                                         // 6
    "schema\0"                                          // 14
    "path"                                              // 21
};

static const uint static_data[] = {
     5,                                                 // revision
     0,                                                 // classname
     0,    0,                                           // classinfo
     0,    0,                                           // methods
     2,   14,                                           // properties
     0,    0,                                           // enums/sets
     0,    0,                                           // constructors
     0,                                                 // flags
     0,                                                 // signalCount
    14,    6, (QVariant::String << 24) | 0x4002,        // QString schema (Scriptable, Writable)
    21,    6, (QVariant::String << 24) | 0x4002,        // QString path (Scriptable, Writable)
     0                                                  // eod
};

const QMetaObject QConf::staticMetaObject = {
    {
        &QObject::staticMetaObject,                     // superclass data
        static_string_data,                             // string data
        static_data,                                    // data
        NULL                                            // extra data
    }
};

// vim:sw=4
