/*
 * Licensed to Elasticsearch under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.elasticsearch.search.aggregations.support.values;

import org.elasticsearch.index.fielddata.SortedBinaryDocValues;
import org.elasticsearch.index.fielddata.SortingBinaryDocValues;
import org.elasticsearch.script.SearchScript;
import org.elasticsearch.search.aggregations.support.ScriptValues;

import java.lang.reflect.Array;
import java.util.Collection;

/**
 * {@link SortedBinaryDocValues} implementation that reads values from a script.
 */
public class ScriptBytesValues extends SortingBinaryDocValues implements ScriptValues {

    private final SearchScript script;

    public ScriptBytesValues(SearchScript script) {
        super();
        this.script = script;
    }

    @Override
    public SearchScript script() {
        return script;
    }

    private void set(int i, Object o) {
        if (o == null) {
            values[i].clear();
        } else {
            values[i].copyChars(o.toString());
        }
    }

    @Override
    public void setDocument(int docId) {
        script.setNextDocId(docId);
        final Object value = script.run();

        if (value == null) {
            count = 0;
        } else if (value.getClass().isArray()) {
            count = Array.getLength(value);
            grow();
            for (int i = 0; i < count; ++i) {
                set(i, Array.get(value, i));
            }
        } else if (value instanceof Collection) {
            final Collection<?> coll = (Collection<?>) value;
            count = coll.size();
            grow();
            int i = 0;
            for (Object v : coll) {
                set(i++, v);
            }
        } else {
            count = 1;
            set(0, value);
        }
        sort();
    }
}
