// Copyright 2015 Canonical Ltd.
// Licensed under the AGPLv3, see LICENCE file for details.

package dependency

import (
	"github.com/juju/errors"

	"github.com/juju/juju/worker"
)

// Engine is a mechanism for persistently running named workers and managing
// dependencies between them.
type Engine interface {

	// Install causes the Engine to accept responsibility for maintaining a
	// worker corresponding to the supplied manifold, restarting it when it
	// fails and when its inputs' workers change, until the Engine shuts down.
	Install(name string, manifold Manifold) error

	// Engine is just another Worker.
	worker.Worker
}

// Manifold defines the behaviour of a node in an Engine's dependency graph. It's
// named for the "device that connects multiple inputs or outputs" sense of the
// word.
type Manifold struct {

	// Inputs lists the names of the manifolds which this manifold might use.
	// An engine will attempt to start a worker independent of the availability
	// of these inputs, and will restart the worker as the available inputs
	// change. If a worker has no dependencies, it should declare empty inputs.
	Inputs []string

	// Start is used to create a worker for the manifold. It must not be nil.
	Start StartFunc

	// Output is used to implement a GetResourceFunc for manifolds that declare
	// a dependency on this one; it can be nil if your manifold is a leaf node,
	// or if it exposes no services to its dependents.
	Output OutputFunc
}

// Manifolds conveniently represents several Manifolds.
type Manifolds map[string]Manifold

// Install is a convenience function for installing multiple manifolds into an
// engine at once. It returns the first error it encounters (and installs no more
// manifolds).
func Install(engine Engine, manifolds Manifolds) error {
	for name, manifold := range manifolds {
		if err := engine.Install(name, manifold); err != nil {
			return errors.Trace(err)
		}
	}
	return nil
}

// StartFunc returns a worker or an error. All the worker's dependencies should
// be taken from the supplied GetResourceFunc; if no worker can be started due
// to unmet dependencies, it should return ErrMissing, in which case it will
// not be called again until its declared inputs change.
type StartFunc func(getResource GetResourceFunc) (worker.Worker, error)

// GetResourceFunc returns an indication of whether a named dependency can be
// satisfied. In particular:
//
//  * if the named resource does not exist, it returns ErrMissing
//  * if the named resource exists, and out is nil, it returns nil
//  * if the named resource exists, and out is non-nil, it returns the error
//    from the named resource manifold's output func (hopefully nil)
//
// Appropriate types for the out pointer depend upon the resource in question.
type GetResourceFunc func(name string, out interface{}) error

// ErrMissing can be returned by a StartFunc or a worker to indicate to
// the engine that it can't be usefully restarted until at least one of its
// dependencies changes. There's no way to specify *which* dependency you need,
// because that's a lot of implementation hassle for little practical gain.
var ErrMissing = errors.New("dependency not available")

// OutputFunc is a type coercion function for a worker generated by a StartFunc.
// When passed an out pointer to a type it recognises, it will assign a suitable
// value and return no error.
type OutputFunc func(in worker.Worker, out interface{}) error

// IsFatalFunc is used to configure an Engine such that, if any worker returns
// an error that satisfies the engine's IsFatalFunc, the engine will stop all
// its workers, shut itself down, and return the original fatal error via Wait().
type IsFatalFunc func(err error) bool

// MoreImportantFunc is used to determine which of two errors is more important.
type MoreImportantFunc func(err0, err1 error) error
