#include "rrsig_rr.h"


namespace ADNS {

	RRSIG_RR::RRSIG_RR()
	{
		signer = gcnew DOMAIN_NAME();
		rr_type = RR_TYPE::RRSIG;
		rr_class = RR_CLASS::IN;
		UpdateRdata();
	}

	RR_TYPE RRSIG_RR::GetTypeCovered()
	{
		return TypeCovered;
	}

	Void RRSIG_RR::SetTypeCovered(RR_TYPE t)
	{
		TypeCovered = t;
		UpdateRdata();
	}

	SIGN_ALGORITHM RRSIG_RR::GetAlgorithm()
	{
		return sig_algo;
	}
	
	Void RRSIG_RR::SetAlgorithm(SIGN_ALGORITHM sa)
	{
		sig_algo = sa;
		UpdateRdata();
	}

	Byte RRSIG_RR::GetLabels()
	{
		return Labels;
	}

	Void RRSIG_RR::SetLabels(Byte l)
	{
		Labels = l;
		UpdateRdata();
	}

	UInt32 RRSIG_RR::GetOriginalTTL()
	{
		return originalttl;
	}

	Void RRSIG_RR::SetOriginalTTL(UInt32 newttl)
	{
		originalttl = newttl;
		UpdateRdata();
	}

	UInt32 RRSIG_RR::GetSignatureExpiration()
	{
		return SigExpiration;
	}

	Void RRSIG_RR::SetSignatureExpiration(UInt32 sigexp)
	{
		SigExpiration = sigexp;
		UpdateRdata();
	}

	UInt32 RRSIG_RR::GetSignatureInception()
	{
		return SigInception;
	}

	Void RRSIG_RR::SetSignatureInception(UInt32 siginc)
	{
		SigInception = siginc;
		UpdateRdata();
	}

	UInt16 RRSIG_RR::GetKeyTag()
	{
		return KeyTag;
	}

	Void RRSIG_RR::SetKeyTag(UInt16 kt)
	{
		KeyTag = kt;
		UpdateRdata();
	}

	DOMAIN_NAME^ RRSIG_RR::GetSignerName()
	{
		return signer;
	}

	Void RRSIG_RR::SetSignerName(DOMAIN_NAME^ sn)
	{
		signer = sn->Clone();
		UpdateRdata();
	}

	array<Byte>^ RRSIG_RR::GetSignature()
	{
		return signature;
	}

	Void RRSIG_RR::SetSignature(array<Byte>^ sig)
	{
		if (!sig)
			return;

		if (!signature)
			signature = gcnew array<Byte>(sig->Length);
		else
			signature->Resize(signature,sig->Length);
		sig->CopyTo(signature,0);
		UpdateRdata();
	}

	Void RRSIG_RR::UpdateRdata()
	{

		int size_needed = 0;
		int siglen = 0;
		array<Byte>^ b;

		if (signature)
			siglen = signature->Length;
		
		size_needed = 18 + signer->GetName()->Length + siglen;

		if (!rdata)
			rdata = gcnew array<Byte>(size_needed);
		else
			rdata->Resize(rdata,size_needed);

		BitConverter::GetBytes(IPAddress::HostToNetworkOrder((short int) TypeCovered))->CopyTo(rdata,0);
		rdata[2] = (Byte) sig_algo;
		rdata[3] = (Byte) Labels;
		BitConverter::GetBytes(IPAddress::HostToNetworkOrder((int) originalttl))->CopyTo(rdata,4);
		BitConverter::GetBytes(IPAddress::HostToNetworkOrder((int) SigExpiration))->CopyTo(rdata,8);
		BitConverter::GetBytes(IPAddress::HostToNetworkOrder((int) SigInception))->CopyTo(rdata,12);
		BitConverter::GetBytes(IPAddress::HostToNetworkOrder((short int) KeyTag))->CopyTo(rdata,16);
		b = signer->GetName();
		b->Copy(b,0,rdata,18,b->Length);
		if (signature)
			signature->CopyTo(rdata,18 + signer->GetName()->Length);

	}

	array<Byte>^ RRSIG_RR::RdataNoSignature()
	{

		int size_needed = 0;
		int siglen = 0;
		array<Byte>^ b;
		array<Byte>^ output;
		DOMAIN_NAME^ dname;

		size_needed = 18 + signer->GetName()->Length ;

		output = gcnew array<Byte>(size_needed);
			
		BitConverter::GetBytes(IPAddress::HostToNetworkOrder((short int) TypeCovered))->CopyTo(output,0);
		output[2] = (Byte) sig_algo;
		output[3] = (Byte) Labels;
		BitConverter::GetBytes(IPAddress::HostToNetworkOrder((int) originalttl))->CopyTo(output,4);
		BitConverter::GetBytes(IPAddress::HostToNetworkOrder((int) SigExpiration))->CopyTo(output,8);
		BitConverter::GetBytes(IPAddress::HostToNetworkOrder((int) SigInception))->CopyTo(output,12);
		BitConverter::GetBytes(IPAddress::HostToNetworkOrder((short int) KeyTag))->CopyTo(output,16);
		dname = signer->Clone();
		dname->MakeCanonical();
		b = dname->GetName();
		b->Copy(b,0,output,18,b->Length);

		return output;

	}

	String^ RRSIG_RR::Print()
	{
		String^ output;

		output = PrintHeader();
		output += Enum::GetName(RR_TYPE::typeid,TypeCovered);
		output += " ";
		output += Convert::ToString((Byte) sig_algo);
		output += " ";
		output += Convert::ToString((Byte) Labels);
		output += " ";
		output += Convert::ToString((unsigned int) originalttl);
		output += " ";
		output += TimeStampToDateTime(SigExpiration)->ToString("yyyyMMddHHmmss");
		output += " ";
		output += TimeStampToDateTime(SigInception)->ToString("yyyyMMddHHmmss");
		output += " ";
		output += Convert::ToString((unsigned short int) KeyTag);
		output += " ";
		output += signer->Print();
		output += " ";
		if (signature)
			output += Convert::ToBase64String(signature);

		return output;
	}

	Void RRSIG_RR::ToCanonical()
	{
		signer->MakeCanonical();
		owner->MakeCanonical();
		UpdateRdata();
		return;
	}

	//This function assumes that the input array of resource records in already in canonical form and order
	//and concatenated.  All we should need to do is prepend the RRSIG with no RDATA, and validate the hash.
	
	bool RRSIG_RR::ValidateSignature(array<Byte>^ CanonicalByteArray, DNSKEY_RR^ key)
	{
		int i = 0;
		array<Byte>^ thingtomatch;
		int oldlen = 0;

		if ((CanonicalByteArray == nullptr) || (key == nullptr))
			return false;

		if ((Byte) key->GetAlgorithm() != (Byte) sig_algo)
			return false;

		thingtomatch = RdataNoSignature();
		oldlen = thingtomatch->Length;
		thingtomatch->Resize(thingtomatch,oldlen + CanonicalByteArray->Length);
		CanonicalByteArray->CopyTo(thingtomatch,oldlen);

		switch(sig_algo)
		{
		case SIGN_ALGORITHM::RSASHA1:
		case SIGN_ALGORITHM::RSASHA1_NSEC3:
			
			return key->ValidateRSASHA1Signature(signature, thingtomatch);

		case SIGN_ALGORITHM::RSASHA256:

			return key->ValidateRSASHA256Signature(signature, thingtomatch);

		default:
			break;
		}

		return false;
	}
		
	RRSIG_RR^ RRSIG_RR::Clone()
	{
		RRSIG_RR^ newrr = gcnew RRSIG_RR();
		newrr->rr_type = rr_type;
		newrr->owner = owner->Clone();
		newrr->ttl = ttl;
		newrr->rr_class = rr_class;
		newrr->TypeCovered = TypeCovered;
		newrr->signer = signer->Clone();
		newrr->signature = gcnew array<Byte>(signature->Length);
		signature->CopyTo(newrr->signature,0);
		newrr->Labels = Labels;
		newrr->originalttl = originalttl;
		newrr->SigExpiration = SigExpiration;
		newrr->SigInception = SigInception;
		newrr->KeyTag = KeyTag;
		newrr->sig_algo = sig_algo;
		newrr->UpdateRdata();
		return newrr;
	}

	ResourceRecord^ RRSIG_RR::ParseResourceRecord(array<Byte>^ domainname, UInt16 rr_type, UInt16 rr_class, UInt32 ttl, UInt16 rdata_len, array<Byte>^ packet, int rdata_start)
	{
		RRSIG_RR^ rrsigout;
		array<Byte>^ tmparray;
		int len;
		int pos;
		int reallen;

		rrsigout = gcnew RRSIG_RR();
		rrsigout->owner = gcnew DOMAIN_NAME(domainname);
		rrsigout->rr_class = (RR_CLASS) rr_class;
		rrsigout->ttl = ttl;
		rrsigout->SetTypeCovered((RR_TYPE) IPAddress::NetworkToHostOrder((short int) BitConverter::ToUInt16(packet,rdata_start)));
		rrsigout->SetAlgorithm((SIGN_ALGORITHM) packet[rdata_start + 2]);
		rrsigout->SetLabels(packet[rdata_start + 3]);
		rrsigout->SetOriginalTTL(IPAddress::NetworkToHostOrder((int) BitConverter::ToUInt32(packet,rdata_start + 4)));
		rrsigout->SetSignatureExpiration(IPAddress::NetworkToHostOrder((int) BitConverter::ToUInt32(packet,rdata_start + 8)));
		rrsigout->SetSignatureInception(IPAddress::NetworkToHostOrder((int) BitConverter::ToUInt32(packet,rdata_start + 12)));
		rrsigout->SetKeyTag(IPAddress::NetworkToHostOrder((short int) BitConverter::ToUInt16(packet,rdata_start + 16)));
		pos = rdata_start + 18;
		tmparray = ReadDomainFromPacket(packet, pos, reallen);
		rrsigout->SetSignerName(gcnew DOMAIN_NAME(tmparray));
		pos += reallen;
		len = rdata_start + rdata_len - pos;
		tmparray->Resize(tmparray,len);
		packet->Copy(packet,pos,tmparray,0,len);
		rrsigout->SetSignature(tmparray);
		return rrsigout;
	}

	String^ RRSIG_RR::PrintRR(ResourceRecord^ rec)
	{
		return safe_cast<RRSIG_RR^>(rec)->Print();
	}

	ResourceRecord^ RRSIG_RR::CloneRR(ResourceRecord^ rec)
	{
		return safe_cast<RRSIG_RR^>(rec)->Clone();
	}


}