
// graphBuilder (gb) program
// This program processes an input file containing mobile "Node" positions  
// using a communications "range" value and builds a linked graph reflecting
// the "Nodes" that are within communications "range" of each other.
// The graph is output as a text file using NRL Scripted Display Tool (sdt)
// "node pos" (position) and "link" commands.

// The input format is expected to be either an "sdt" file containing "node pos"
// commands or can be an ns-2 mobility trace file where the waypoint locations
// given by ns-2 "setdest" commands.
// Note that in either case the graph _may_ change over time (as node positions
// change) and the SDT file that is output can contain "wait" commands that
// indicate the pace of change along with "link" and "unlink" commands to
// describe connectivity.


#include "manetGraph.h"
#include "protoSpace.h"  // we use this for its bounding box iteration
#include <protoDebug.h>
#include <protoDefs.h>

#include <stdio.h>   // for sprintf()
#include <stdlib.h>  // for rand(), srand()
#include <ctype.h>   // for "isprint()"
#include <math.h>    // for "sqrt()"



void Usage()
{
    fprintf(stderr, "Usage: gb {ns|sdt} <mobilityTraceFile> range <commsRange>\n");
}

const unsigned int MAX_LINE = 256;

class Node : public ManetGraph::Node, public ProtoSpace::Node, public ProtoTree::Item
{
    public:
        Node();
        ~Node();

        // _MUST_ call init to create a default interface
        bool Init(UINT32 nodeId, const char* nodeName = NULL);

        unsigned int GetId() const
        {
            ManetGraph::Interface* iface = GetDefaultInterface();
            if (NULL != iface)
            {
                return (iface->GetAddress().GetEndIdentifier());
            }
            else
            {
                ASSERT(0);
                return ((unsigned int)-1);
            }
        }
        
        const char* GetName() const
            {return node_name;}

        void SetRtrPriority(UINT8 value)
            {rtr_priority = value;}
        UINT8 GetRtrPriority() const
            {return rtr_priority;}

        void SetRelayStatus(bool state)
            {relay_status = state;}
        bool GetRelayStatus() const
            {return relay_status;}

        void SetVisited(bool state)
            {visited = state;}
        bool WasVisited() const
            {return visited;}

        void SetPosition(double xPos, double yPos)
        {
            ordinate[0] = xPos;
            ordinate[1] = yPos;
        }
        double GetOrdinateX() const
            {return ordinate[0];}
        double GetOrdinateY() const
            {return ordinate[1];}

        // ProtoSpace::Node required overrides
        unsigned int GetDimensions() const
            {return 2;}
        double GetOrdinate(unsigned int  index) const
            {return (ordinate[index]);}
        const double* GetOrdinatePtr() const
            {return ordinate;}

    private:
        // ProtoTree::Item overrides so nodes
        // can be cached by name
        const char* GetKey() const
            {return node_name;}   
        unsigned int GetKeysize() const
            {return node_name_bits;} 
            
        char*           node_name;
        unsigned int    node_name_bits;
        UINT8           rtr_priority;
        bool            relay_status;
        bool            visited;
        double          ordinate[2];  // x,y coordinates

};  // end class Node

Node::Node()
 : node_name(NULL), node_name_bits(0), rtr_priority(0)
{
    ordinate[0] = ordinate[1] = 0.0;
}

Node::~Node()
{
}

bool Node::Init(UINT32 nodeId, const char* nodeName)
{
    if (NULL != nodeName)
    {
        if (NULL != node_name) delete[] node_name;
        size_t nameLen = strlen(nodeName) + 1;
        if (NULL == (node_name = new char[nameLen]))
        {
            PLOG(PL_ERROR, "Node::Init() new node_name error: %s\n",
                 GetErrorString());
            return false;
        }
        strcpy(node_name, nodeName);
        node_name_bits = nameLen << 3;
    }
    
    ProtoAddress addr;
    addr.SetEndIdentifier(nodeId);
    ManetGraph::Interface* iface = new ManetGraph::Interface(*this, addr);
    if (NULL != iface)
    {
        return AppendInterface(*iface);
    }
    else
    {
        PLOG(PL_ERROR, "Node::Init() new ManetGraph::Interface() error: %s\n",
             GetErrorString());
        return false;
    }
}  // end Node::Init()


// This class encapsulates most of the functionality of
// our "gb" (GraphBuilder) application.  I bothered to do
// this in case we want to re-use this elsewhere.

class GraphBuilder
{
    public:
        GraphBuilder();
        ~GraphBuilder();
        
        bool SetInputFile(const char* fileName)
            {return fast_reader.SetFile(fileName);}
        
        void SetCommsRange(double value) 
            {comms_range = value;}
        
        // This updates the "space" with data from the next time "epoch"
        // from our input file
        double ReadNextEpoch();
        
        // This updates the "graph" connectivity from the "space" base 
        // on the "comms_range"
        bool UpdateGraphFromSpace();
        
    private:
        // "FastReader" is handy class I use for doing
        //  buffered (fast) reading of a usually text input file.
        class FastReader
        {
            public:
                enum Result {OK, ERROR_, DONE, TIMEOUT};
                FastReader(FILE* filePtr = NULL);
                ~FastReader();
                
                bool SetFile(const char* fileName)
                {
                    if (NULL != file_ptr) fclose(file_ptr);
                    return (NULL != (file_ptr = fopen(fileName, "r")));
                }
                
                Result Read(char*           buffer,
                            unsigned int*   len,
                            double timeout = -1.0);
                
                Result Readline(char*           buffer,
                                unsigned int*   len,
                                double          timeout = -1.0);

                bool Seek(int offset);

            private:
                enum {BUFSIZE = 2048};
                FILE*        file_ptr;
                char         savebuf[BUFSIZE];
                char*        saveptr;
                unsigned int savecount;
        };  // end class GraphBuilder::FastReader
        
        static double CalculateDistance(double x1, double y1, double x2,double y2)
        {
            double dx = x1 - x2;
            double dy = y1 - y2;
            return sqrt(dx*dx + dy*dy);
        }  // end CalculateDistance()
        
        // member variables
        FastReader                                  fast_reader;
        double                                      comms_range;     
        ProtoSpace                                  space;
        ManetGraph                                  graph;  
        ProtoTree                                   node_tree;
        UINT32                                      node_id_index;
        double                                      next_epoch_time; 
        ProtoGraph::Vertice::SortedList::ItemPool   vertice_item_pool;
        bool                                        bound_init;
        double                                      bound_min;
        double                                      bound_max;
                        
};  // end class GraphBuilder

GraphBuilder::GraphBuilder()
 : comms_range(0.0), node_id_index(0), next_epoch_time(-1.0),
   bound_init(true)
{
}

GraphBuilder::~GraphBuilder()
{
}

// returns the start time (in seconds) of the "epoch"
double GraphBuilder::ReadNextEpoch()
{
    double lastTime = next_epoch_time;
    if (lastTime < 0.0) next_epoch_time = 0.0;
    bool boundsChanged = false;
    // Read the SDT "node pos" or NS2 "setdest" lines from the file, finding
    // new nodes and their x,y positions versus time
    // SDT line format we want is "wait <delayTime" or "node <name> pos x,y"
    // ns-2 line format we want is '<sim> at <time> "<nodeName> setdest x y velocity"'
    bool gotLine = false; // this is set to true if we find any useful content
    bool reading = true;
    while (reading)
    {
        char buffer[MAX_LINE];
        unsigned int len = MAX_LINE;
        switch (fast_reader.Readline(buffer, &len))
        {
            case GraphBuilder::FastReader::OK:
                break;
            case GraphBuilder::FastReader::ERROR_:
                PLOG(PL_ERROR, "gb: error reading file\n");
                return -1.0;
            case GraphBuilder::FastReader::DONE:
                reading = false;
                continue;
            case GraphBuilder::FastReader::TIMEOUT:
                return -1.0; // should never occur for this program
        }

        // Is this line the start of a new "epoch"?
        // (we also detect if it is an ns-2 file here)
        bool ns2 = false;
        char sim[32];
        double time;
        if (1 == sscanf(buffer, "wait %lf\n", &time))
        {
            // It's an SDT "wait" command ...
            time /= 1000.0;
            next_epoch_time += time;
            // first epoch is at time "0.0"
            if (lastTime < 0.0) lastTime = 0.0;
            break;
        }
        else if (2 == sscanf(buffer, "%s at %lf", sim, &time))
        {
            // It's an NS2 script
            ns2 = true;
            // Init or update "lastTime", if applicable
            if (lastTime < 0.0) 
            {
                lastTime = time;
            }
            else if (time != lastTime)
            {
                // It's a new epoch, so seek backwards length of this line and return
                next_epoch_time = time;
                fast_reader.Seek(-len);
                break;
            }
        }
        
        char nodeName[256];
        double x, y;
        if (ns2)
        {
            // Find leading quote of "<nodeName> setdest <x> <y> ..." command
            char* ptr = strchr(buffer, '\"');
            if (NULL == ptr)
                continue;  // go to next line
            else
                ptr++;
            if (3 != sscanf(ptr, "%s setdest %lf %lf", nodeName, &x, &y))
                continue;
        }
        else
        {
            // It's likely an SDT  command line, so see if it is a "node <name> pos x,y" line
            // Note the trick here that allows for the sometimes abbreviated form of
            // the node "position" attribute name
            char posString[16];
            if (4 == sscanf(buffer, "node %s %s %lf,%lf", nodeName, posString, &x, &y))
            {
                if (0 != strncmp("position", posString, strlen(posString)))
                    continue; // not a line of interest
            }
            else
            {
                continue; // not a line of interest
            }
        }
        
        gotLine = true;
        
        if (bound_init)
        {
            bound_min = (x < y) ? x : y;
            bound_max = (x > y) ? x : y;
            boundsChanged = true;
            bound_init = false;
        }
        else
        {
            double ordMin = (x < y) ? x : y;
            if (ordMin < bound_min)
            {
                bound_min = ordMin;
                boundsChanged = true;
            }
            double ordMax = (x > y) ? x : y;
            if (ordMax > bound_max)
            {
                bound_max = ordMax;
                boundsChanged = true;
            }
        }
        
        // Do we know this node already?
        unsigned int nameBits = (unsigned int)(strlen(nodeName)+1) << 3;
        Node* node = static_cast<Node*>(node_tree.Find(nodeName, nameBits));
        if (NULL == node)
        {
            // It's a new node
            // Find numeric portion of "nodeName", if applicable, to use as an identifier
            unsigned int nodeId;
            bool nameHasNumber = (1 == sscanf(nodeName, "%u", &nodeId));
            ASSERT(nameHasNumber);
            char nodeNamePrefix[32];
            if (!nameHasNumber)
                nameHasNumber = (2 == sscanf(nodeName, "%[^0-9]%u", nodeNamePrefix, &nodeId));
            if (nameHasNumber)
            {
                // Make sure it doesn't collide with existing node
                ProtoAddress addr;
                addr.SetEndIdentifier((UINT32)nodeId);
                //  If collision, assign a unique identifier from our space
                if (NULL != graph.FindInterface(addr))
                    nodeId = node_id_index++; 
            }
            else
            {
                //  Assign an identifier from our space
                nodeId = node_id_index++; 
            }
            // Adjust "node_id_index" if necessary to guarantee uniqueness
            // of assigned identifiers
            if (nodeId >= node_id_index) 
                node_id_index = nodeId + 1;
            
            // Create and insert new node into "space", "graph", and "node_tree"
            Node* node = new Node();
            if (!node->Init(nodeId, nodeName))
            {
                PLOG(PL_ERROR, "gb error: Node initialization failure!\n");
                return -1.0;
            }
            node->SetPosition(x, y);
            graph.InsertNode(*node);
            if (!space.InsertNode(*node))
            {
                PLOG(PL_ERROR, "gb: error inserting node into space\n");
                return -1.0;
            }
            node_tree.Insert(*node);
            // Output new node position for SDT visualization
            printf("node %d position %f,%f symbol circle,red,3 label white\n", node->GetId(), x, y);
        }
        else
        {
            // Existing node, update position in "space", etc
            space.RemoveNode(*node);
            node->SetPosition(x, y);
            space.InsertNode(*node);
            // Update node position for SDT visualization
            printf("node %d position %f,%f\n", node->GetId(), x, y);
        }
    }  // end while reading()
    
    if (boundsChanged)
    {
        double bMin = bound_min - 10.0;
        double bMax = bound_max + 10.0;
        printf("bgbounds %f,%f,%f,%f\n", bMin, bMin, bMax, bMax);
    }
    return (gotLine ? lastTime : -1.0);
}  // end ReadNextEpoch()

bool GraphBuilder::UpdateGraphFromSpace()
{
    ManetGraph::InterfaceIterator it(graph);
    ManetGraph::Interface* iface;
    unsigned int count = 0;
    while (NULL != (iface = it.GetNextInterface()))
    {
        Node& node = static_cast<Node&>(iface->GetNode());

        // Cache list of node's current neighbors into
        // a list we can retrieve from as we determine
        // our "within range" nodes.
        ManetGraph::Interface::SortedList nbrList(&vertice_item_pool);
        ManetGraph::Interface::AdjacencyIterator nbrIterator(*iface);
        ManetGraph::Interface* nbrIface;
        while (NULL != (nbrIface = nbrIterator.GetNextInterface()))
        {
            if (!nbrList.Insert(*nbrIface))
            {
                PLOG(PL_ERROR, "gb: error adding neighbor to temp list\n");
                return false;
            }
        }

        // Iterate through our ProtoSpace from this node's location,
        // looking for neighbor nodes within RANGE_MAX distance
        ProtoSpace::Iterator sit(space);
        if (!sit.Init(node.GetOrdinatePtr()))
        {
            PLOG(PL_ERROR, "gb: error initializing space iterator\n");
            return false;
        }

        Node* nbr;
        count = 0;
        double lastDist = -1.0;
        double distance;
        while (NULL != (nbr = static_cast<Node*>(sit.GetNextNode(&distance))))
        {
            count++;

            ASSERT(lastDist <= distance);
            lastDist = distance;

            if (nbr == &node)
            {
                continue;
            }

            if (distance <= comms_range)
            {
                // Is this "nbr" already connected as a neighbor in our graph?
                ManetGraph::Interface* defaultIface = nbr->GetDefaultInterface();
                ASSERT(NULL != defaultIface);
                const ProtoAddress& nbrAddr = defaultIface->GetAddress();
                nbrIface = nbrList.FindInterface(nbrAddr);
                if (NULL == nbrIface)
                {
                    ManetGraph::SimpleCostDouble cost(1.0);
                    if (!graph.Connect(*iface, *defaultIface, cost, true))
                    {
                        PLOG(PL_ERROR, "gb error: unable to connect interfaces in graph\n");
                        return false;
                    }
                    printf("link %d,%d,green\n", node.GetId(), nbr->GetId());
                }
                else
                {
                    nbrList.Remove(*nbrIface);
                }
            }
            else
            {
                // Disconnect any former neighbors that were no longer in range
                if (!nbrList.IsEmpty())
                {
                    ManetGraph::Interface::SortedList::Iterator listIterator(nbrList);
                    while (NULL != (nbrIface = listIterator.GetNextInterface()))
                    {
                        graph.Disconnect(*iface, *nbrIface, true);
                        printf("unlink %d,%d\n", node.GetId(), static_cast<Node&>(nbrIface->GetNode()).GetId());
                    }
                }
                break;
            }
        }  // end while (NULL != nbr)
    }  // end while (NULL != node)  (graph update)
    return true;
}  // end UpdateGraphFromSpace()


//////////////////////////////////////////////////////////////////////////////////////////
// "GraphBuilder::FastReader" implementation
GraphBuilder::FastReader::FastReader(FILE* filePtr)
 : file_ptr(filePtr), savecount(0)
{
}

GraphBuilder::FastReader::~FastReader()
{
    if (NULL != file_ptr)
    {
        fclose(file_ptr);
        file_ptr = NULL;
    }
}

GraphBuilder::FastReader::Result GraphBuilder::FastReader::Read(char*           buffer,
                                                                unsigned int*   len,
                                                                double          timeout)
{
    unsigned int want = *len;
    if (savecount)
    {
        unsigned int ncopy = MIN(want, savecount);
        memcpy(buffer, saveptr, ncopy);
        savecount -= ncopy;
        saveptr += ncopy;
        buffer += ncopy;
        want -= ncopy;
    }
    while (want)
    {
        unsigned int result;
#ifndef WIN32 // no real-time input for WIN32 yet
        if (timeout >= 0.0)
        {
            int fd = fileno(file_ptr);
            fd_set input;
            FD_ZERO(&input);
            struct timeval t;
            t.tv_sec = (unsigned long)timeout;
            t.tv_usec = (unsigned long)((1.0e+06 * (timeout - (double)t.tv_sec)) + 0.5);
            FD_SET(fd, &input);
            int status = select(fd+1, &input, NULL, NULL, &t);
            switch (status)
            {
                case -1:
                    if (EINTR != errno)
                    {
                        perror("trpr: GraphBuilder::FastReader::Read() select() error");
                        return ERROR_;
                    }
                    else
                    {
                        continue;
                    }
                    break;

                case 0:
                    return TIMEOUT;

                default:
                    result = fread(savebuf, sizeof(char), 1, file_ptr);
                    break;
            }
        }
        else
#endif // !WIN32
        {
            // Perform buffered read when there is no "timeout"
            result = fread(savebuf, sizeof(char), BUFSIZE, file_ptr);
        }
        if (result)
        {
            
            // This check skips NULLs that have been read on some
            // use of trpr via tail from an NFS mounted file
            if (!isprint(*savebuf) &&
                    ('\t' != *savebuf) &&
                    ('\n' != *savebuf) &&
                    ('\r' != *savebuf))
                continue;
            unsigned int ncopy= MIN(want, result);
            memcpy(buffer, savebuf, ncopy);
            savecount = result - ncopy;
            saveptr = savebuf + ncopy;
            buffer += ncopy;
            want -= ncopy;
        }
        else  // end-of-file
        {
#ifndef WIN32
            if (ferror(file_ptr))
            {
                if (EINTR == errno) continue;
            }
#endif // !WIN32
            *len -= want;
            if (*len)
                return OK;  // we read at least something
            else
                return DONE; // we read nothing
        }
    }  // end while(want)
    return OK;
}  // end GraphBuilder::FastReader::Read()

// An OK text readline() routine (reads what will fit into buffer incl. NULL termination)
// if *len is unchanged on return, it means the line is bigger than the buffer and
// requires multiple reads
GraphBuilder::FastReader::Result GraphBuilder::FastReader::Readline(char*         buffer,
                                                                    unsigned int* len,
                                                                    double        timeout)
{
    unsigned int count = 0;
    unsigned int length = *len;
    char* ptr = buffer;
    while (count < length)
    {
        unsigned int one = 1;
        switch (Read(ptr, &one, timeout))
        {
            case OK:
                if (('\n' == *ptr) || ('\r' == *ptr))
                {
                    *ptr = '\0';
                    *len = count;
                    return OK;
                }
                count++;
                ptr++;
                break;

            case TIMEOUT:
                // On timeout, save any partial line collected
                if (count)
                {
                    savecount = MIN(count, BUFSIZE);
                    if (count < BUFSIZE)
                    {
                        memcpy(savebuf, buffer, count);
                        savecount = count;
                        saveptr = savebuf;
                        *len = 0;
                    }
                    else
                    {
                        memcpy(savebuf, buffer+count-BUFSIZE, BUFSIZE);
                        savecount = BUFSIZE;
                        saveptr = savebuf;
                        *len = count - BUFSIZE;
                    }
                }
                return TIMEOUT;

            case ERROR_:
                return ERROR_;

            case DONE:
                return DONE;
        }
    }
    // We've filled up the buffer provided with no end-of-line
    return ERROR_;
}  // end GraphBuilder::FastReader::Readline()

bool GraphBuilder::FastReader::Seek(int offset)
{
    bool result = true;
    if (offset < 0)
    {
        if (0 != savecount)
        {
            int avail = saveptr - savebuf;
            if (avail >= abs(offset))
            {
                savecount += abs(offset);
                saveptr -= abs(offset);
                offset = 0;
            }
            else
            {
                offset -= savecount;
                savecount = 0;
            }
        }
        if (0 != offset)
            result = (0 == fseek(file_ptr, offset, SEEK_CUR));
    }
    else if (offset > 0)
    {
        if ((unsigned int)offset < savecount)
        {
            savecount -= offset;
            saveptr += offset;
        }
        else
        {
            if ((unsigned int)offset > savecount)
            {
                result = (0 == fseek(file_ptr, offset - savecount, SEEK_CUR));
            }
            savecount = 0;
        }
    }
    return result;
}  // end GraphBuilder::FastReader::Seek()


int main(int argc, char* argv[])
{
    GraphBuilder graphBuilder;
    bool gotInput = false;
    bool gotRange = false;
    // Parse the command line
    int i = 1;
    while (i < argc)
    {
        size_t len = strlen(argv[i]);
        if (0 == strncmp(argv[i], "range", len))
        {
            if (++i == argc)
            {
                fprintf(stderr, "gb error: missing \"range\" argument!\n");
                return -1;
            }
            double commsRange;
            if (1 != sscanf(argv[i], "%lf", &commsRange))
            {
                fprintf(stderr, "gb error: invalid \"range\" argument!\n");
                return -1;
            }
            graphBuilder.SetCommsRange(commsRange);
            gotRange = true;
        }   
        else if (0 == strncmp(argv[i], "input", len))
        {
            if (++i == argc)
            {
                fprintf(stderr, "gb error: missing \"input\" argument!\n");
                Usage();
                return -1;
            }
            if (!graphBuilder.SetInputFile(argv[i]))
            {
                perror("gb error: unable to open input file");
                return -1;
            }
            gotInput = true;
        }
        else
        {
            fprintf(stderr, "gb error: invalid command: %s\n", argv[i]);
            Usage();
            return -1;
        }
        i++;
    }
    
    if (!gotInput)
    {
        fprintf(stderr, "gb error: no input file specified!\n");
        Usage();
        return -1;
    }
    if (!gotRange)
    {
        fprintf(stderr, "gb error: no communication range specified!\n");
        Usage();
        return -1;
    }

    double lastEpochTime = -1.0;
    double nextEpochTime;
    while ((nextEpochTime = graphBuilder.ReadNextEpoch()) >= 0.0)
    {
        // Output "wait" command for SDT visualization, if applicable
        if (lastEpochTime >= 0.0)
            printf("wait %f\n", (float)(1000.0 * (nextEpochTime - lastEpochTime)));
        lastEpochTime = nextEpochTime;
        // "UpdateGraphFromSpace()" adds/removes "links" between the Nodes depending
        // upon their spatial distance and the communications range
        if (!graphBuilder.UpdateGraphFromSpace())
        {
            PLOG(PL_ERROR, "gb: error updating graph ...\n");
            return -1;
        }
    }
    fprintf(stderr, "gb: Done.\n");
    return 0;
}  // end main()

