## Heads up!

It's suprising how many libraries are in the average dependency tree that don't belong there for one reason or another. Either because they were accidentally listed as `dependencies` instead of `devDepedencies`, or they are required in a file as variables, but the variable is never actually used (poor linting), and so on. Or because the maintainer made the decision to add the deps, even though they will never ([or can't ever](https://github.com/felixge/node-dateformat/issues/36)) be used by 99.9% of users.

Worse, many libraries like chalk and [shelljs](https://github.com/eslint/eslint/issues/7316) actually execute code when `require()` is called!? (shelljs was modifying the `String.prototype`, and chalk loops over some objects to dynamically create methods). In other words, they do something like this:

```js
// in the main export of a library, if you do this it will
// automatically modify the String.prototype _globally_, 
// the moment node.js loads the dependency tree
String.prototype.foo = function() {};

// same if you do something like this
// (dont' do this, ever. wrap this kind of code in a function
// and allow implementors to decide when to call it)
while (foo) {
  // do stuff
}
```

In any case, just having these libraries in your dependency tree somewhere means that their code will excecute the moment you run your application _even if the libraries are never called by your application or any other code anywhere in the tree_.

**solution**

lazy-cache doesn't use any "magic", it uses native, plain-vanilla, tried and true javascript getters to call node's `require()` system. 

**Faster, safer code**

There main advantage to this, the main is that `require`s are loaded on demand, so only code that is actually used will ever be loaded. As a result, applications will load faster (sometimes much faster - we've seen load times drop from ~1 second to less than 50 milliseconds). 

Moreover, in some cases this also avoids inadvertently loading libraries that execute code or modifies globals, etc.

**webpack users**

If you use webpack and are experiencing issues with lazy-cache, this is a known bug caused by webpack, not lazy-cache. There is a solution though, you can use [unlazy-loader](https://github.com/doowb/unlazy-loader), a webpack loader that _fixes the webpack bug_.

## Usage

```js
var utils = require('{%= name %}')(require);
```

**Use as a property on `lazy`**

The module is also added as a property to the `lazy` function so it can be called without having to call a function first.

```js
var utils = require('{%= name %}')(require);

// `npm install glob`
utils('glob');

// glob sync
console.log(utils.glob.sync('*.js'));

// glob async
utils.glob('*.js', function (err, files) {
  console.log(files);
});
```

**Use as a function**

```js
var utils = require('{%= name %}')(require);
var glob = utils('glob');

// `glob` is a now a function that may be called when needed
glob().sync('foo/*.js');
```

## Aliases

An alias may be passed as the second argument if you don't want to use the automatically camel-cased variable name.

**Example**

```js
var utils = require('{%= name %}')(require);

// alias `ansi-yellow` as `yellow`
utils('ansi-yellow', 'yellow');
console.log(utils.yellow('foo'));
```

Dot notation may also be used in the alias to create an object hierarchy.

**Example**

```js
var utils = require('{%= name %}')(require);
utils('ansi-cyan', 'color.cyan');
utils('ansi-yellow', 'color.yellow');
utils('ansi-magenta', 'color.magenta');
console.log(utils.color.cyan('foo'));
console.log(utils.color.yellow('bar'));
console.log(utils.color.magenta('baz'));
```

## Browserify usage

**Example**

```js
var utils = require('{%= name %}')(require);
// temporarily re-assign `require` to trick browserify
var fn = require;
require = utils;
// list module dependencies (here, `require` is actually `lazy-cache`)
require('glob');
require = fn; // restore the native `require` function

/**
 * Now you can use glob with the `utils.glob` variable
 */

// sync
console.log(utils.glob.sync('*.js'));

// async
utils.glob('*.js', function (err, files) {
  console.log(files.join('\n'));
});
```

## Kill switch

To force lazy-cache to immediately invoke all dependencies, do:

```js
process.env.UNLAZY = true;
```
