/* -*- coding: utf-8 -*- */
/* Copyright (C) 1996-2015 Piotr Held <pjheld@gmail.com>
 *
 * This file is part of Octave.
 *
 * Octave is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation;
 * either version 3 of the License, or (at your option) any
 * later version.
 *
 * Octave is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public
 * License along with Octave; see the file COPYING.  If not,
 * see <http://www.gnu.org/licenses/>.
 */
/* Author: Piotr Held <pjheld@gmail.com>.
 * This function is based on c1 of TISEAN 3.0.1 
 * https://github.com/heggus/Tisean"
 */
/******************************************************************************/
/******************************************************************************/

#define HELPTEXT "Part of tisean package\n\
No argument checking\n\
FOR INTERNAL USE ONLY"

#include <octave/oct.h>
#include <octave/f77-fcn.h>
#include <octave/oct-map.h>

extern "C"
{
  F77_RET_T
  F77_FUNC (d1, D1)
            (const octave_idx_type& nmax, const octave_idx_type& mmax,
             const octave_idx_type& nxx, const double *y,
             const octave_idx_type& delay, const octave_idx_type& m,
             const octave_idx_type& ncmin, const double& pr,
             double& pln, double& eln, 
             const octave_idx_type& nmin, const octave_idx_type& kmax,
             const octave_idx_type& iverb);

  F77_RET_T
  F77_FUNC (rand, RAND)
            (const double& R);
}


DEFUN_DLD (__c1__, args, nargout, HELPTEXT)
{
  octave_value_list retval;
  int nargin = args.length ();


  if (nargin != 10)
    {
      print_usage ();
    }
  else
    {
    // Assigning inputs
      Matrix input           = args(0).matrix_value ();
      octave_idx_type mindim = args(1).idx_type_value ();
      octave_idx_type maxdim = args(2).idx_type_value ();
      octave_idx_type delay  = args(3).idx_type_value ();
      octave_idx_type tmin   = args(4).idx_type_value ();
      octave_idx_type cmin   = args(5).idx_type_value ();
      double resolution      = args(6).double_value ();
      double seed            = args(7).double_value ();
      octave_idx_type kmax   = args(8).idx_type_value ();
      bool verbose           = args(9).bool_value ();
      octave_idx_type iverb  = verbose;


      if (! error_state)
        {

          octave_idx_type lines_read   = input.rows (); //nmax in d1()
          octave_idx_type columns_read = input.columns ();


          dim_vector dv (maxdim - mindim + 1, 1);
          string_vector keys;
          keys.append (std::string("dim"));
          keys.append (std::string("c1"));
          octave_map output (dv, keys);

          // Seed the rand() function for d1()
          F77_XFCN (rand, RAND, (sqrt(seed)));

          for (octave_idx_type m = mindim; m <= maxdim; m++)
            {
              octave_scalar_map tmp (keys);
              tmp.setfield ("dim", m);

              // Creat c1 output
              Matrix c1_out ((octave_idx_type) ((0 - log (1./(lines_read 
                                                         -(m-1) * delay)) +
                                                (log (2.) /resolution)) 
                                                     / (log (2.) /resolution))
                             , 2);

              double pr = 0.0;
              octave_idx_type current_row = 0;
              for (double pl = log (1./(lines_read - (m-1)*delay));
                   pl <= 0.0; pl += log (2.) / resolution)
                {
                  double pln = pl;
                  double rln;

                  F77_XFCN (d1, D1,
                            (lines_read, columns_read, lines_read,
                             input.fortran_vec (), delay, m, cmin,
                             pr, pln, rln, tmin, kmax, iverb));

                  if (pln != pr)
                    {
                      pr = pln;
                      c1_out(current_row,0) = exp (rln);
                      c1_out(current_row,1) = exp (pln);
                      current_row += 1;
                    }

                }
              // Resize output
              c1_out.resize (current_row, 2);
              tmp.setfield ("c1", c1_out);

              output.assign (idx_vector(m-mindim), tmp);
            }

          retval(0) = output;
        }
    }
  return retval;
}
