#include <tjutils/tjnumeric.h>

#include "seqdiffweight.h"
#include "seqgradchanparallel.h"

float* get_dti_dirarr(unsigned int ndir); // forward declaration

void calc_dw_grads(fvector& gradtrims, double& delta, const fvector& bvals, float maxgradstrength, float middur, float gamma) {
  Log<Seq> odinlog("SeqDiffWeight","calc_grads");

  float maxbval=bvals.maxabs();

  // solve cubic equation for delta
  double a=3.0/2.0*middur;
  double c=-3.0/2.0*secureDivision(maxbval,gamma*gamma*maxgradstrength*maxgradstrength);
  ODINLOG(odinlog,normalDebug) << "a/c=" << a << "/" << c << STD_endl;

  double x0=0.0;
  double x1=0.0;
  double x2=0.0;
  int nroots=solve_cubic (a,0.0,c,&x0,&x1,&x2);
  ODINLOG(odinlog,normalDebug) << "nroots/x0/x1/x2=" << nroots << "/" << x0 << "/" << x1 << "/" << x2 << STD_endl;

  delta=maxof3(x0,x1,x2);
  ODINLOG(odinlog,normalDebug) << "delta=" << delta << STD_endl;

  unsigned int ntrims=bvals.length();
  gradtrims.resize(ntrims);
  for(unsigned int i=0; i<ntrims; i++) {
    float sign=1.0;
    if(bvals[i]<0.0) sign=-1.0;
    float strength=sqrt( secureDivision ( fabs(bvals[i]) , fabs( gamma*gamma*delta*delta*(middur + 2.0/3.0*delta ) ) ) );
    gradtrims[i]=sign*strength/maxgradstrength;
  }
}



SeqDiffWeight::SeqDiffWeight(const STD_string& object_label, const fvector& bvals, float maxgradstrength,
                    const SeqObjBase& midpart, direction chan, bool stejskalTanner, const STD_string& nucleus)
  : SeqObjList(object_label), SeqSimultanVector(object_label),
   par1(object_label+"_par1"), par2(object_label+"_par2") {
  Log<Seq> odinlog(this,"SeqDiffWeight(...)");

  middle_part=midpart;

  fvector gradtrims1;
  double delta;
  calc_dw_grads(gradtrims1, delta, bvals, maxgradstrength, middle_part.get_duration(), systemInfo->get_gamma(nucleus));

  fvector gradtrims2(gradtrims1);
  if(!stejskalTanner) gradtrims2=-gradtrims1;

  // reset
  for(int idim=0; idim<n_directions; idim++) {
    pfg1[idim].set_strength(0.0);
    pfg2[idim].set_strength(0.0);
  }

  pfg1[chan]=SeqGradVectorPulse(object_label+"_pfg1_"+directionLabel[chan],chan,maxgradstrength,gradtrims1,delta);
  pfg2[chan]=SeqGradVectorPulse(object_label+"_pfg2_"+directionLabel[chan],chan,maxgradstrength,gradtrims2,delta);

//  SeqSimultanVector::set_reorder_scheme(rotateReorder,gradtrims1.size());

  build_seq();
}


SeqDiffWeight::SeqDiffWeight(const STD_string& object_label, unsigned int ndir, const fvector& bvals, float maxgradstrength,
                    const SeqObjBase& midpart, unsigned int baseline_rep, bool stejskalTanner, const STD_string& nucleus)
  : SeqObjList(object_label), SeqSimultanVector(object_label),
   par1(object_label+"_par1"), par2(object_label+"_par2") {
  Log<Seq> odinlog(this,"SeqDiffWeight(...)");

  float* arr=get_dti_dirarr(ndir);
  if(!arr) {
    ODINLOG(odinlog,errorLog) << "array not available for ndir=" << ndir << STD_endl;
    return;
  }

  middle_part=midpart;

  STD_list<float> blist[n_directions];

  unsigned int idim; // single declaration for MSVC6
  for(idim=0; idim<n_directions; idim++) blist[idim].push_back(0.0); // start with b0 scan

  unsigned int repcounter=0;
  for(unsigned int idir=0; idir<ndir; idir++) {
    float* base=arr+3*idir;
    for(unsigned int ib=0; ib<bvals.size(); ib++) {

      for(idim=0; idim<n_directions; idim++) blist[idim].push_back(bvals[ib]*base[idim]);

      repcounter++;
      if(baseline_rep && repcounter>=baseline_rep && idir<(ndir-1)) {
        repcounter=0;
        for(idim=0; idim<n_directions; idim++) blist[idim].push_back(0.0);
      }
    }
  }

  fvector bvec[n_directions];
  for(idim=0; idim<n_directions; idim++) {
    bvec[idim]=list2vector(blist[idim]);
  }

  unsigned int ndti=bvec[0].size();
  b_vectors_cache.redim(ndti,n_directions);
  for(unsigned int idti=0; idti<ndti; idti++) {
    for(idim=0; idim<n_directions; idim++) {
      b_vectors_cache(idti,idim)=bvec[idim][idti];
    }
  }


  for(idim=0; idim<n_directions; idim++) {

    fvector gradtrims1;
    double delta;
    calc_dw_grads(gradtrims1, delta, bvec[idim], maxgradstrength, middle_part.get_duration(), systemInfo->get_gamma(nucleus));

    fvector gradtrims2(gradtrims1);
    if(!stejskalTanner) gradtrims2=-gradtrims1;

    ODINLOG(odinlog,normalDebug) << "gradtrims1[" << idim << "]=" << gradtrims1 << STD_endl;
    ODINLOG(odinlog,normalDebug) << "gradtrims2[" << idim << "]=" << gradtrims2 << STD_endl;

    pfg1[idim]=SeqGradVectorPulse(object_label+"_pfg1_"+directionLabel[idim],direction(idim),maxgradstrength,gradtrims1,delta);
    pfg2[idim]=SeqGradVectorPulse(object_label+"_pfg2_"+directionLabel[idim],direction(idim),maxgradstrength,gradtrims2,delta);
  }

  build_seq();
}



SeqDiffWeight::SeqDiffWeight(const STD_string& object_label )
  : SeqObjList(object_label), SeqSimultanVector(object_label) {
}


SeqDiffWeight& SeqDiffWeight::operator = (const SeqDiffWeight& sgdw) {
  SeqSimultanVector::operator = (sgdw);
  SeqObjList::operator = (sgdw);
  for(int idim=0; idim<n_directions; idim++) {
    pfg1[idim]=sgdw.pfg1[idim];
    pfg2[idim]=sgdw.pfg2[idim];
  }
  par1=sgdw.par1;
  par2=sgdw.par2;
  middle_part=sgdw.middle_part;
  b_vectors_cache=sgdw.b_vectors_cache;
  build_seq();
  return *this;
}


SeqGradInterface& SeqDiffWeight::set_strength(float gradstrength) {
  par1.set_strength(gradstrength);
  par2.set_strength(gradstrength);
  return *this;
}


SeqGradInterface& SeqDiffWeight::invert_strength() {
  par1.invert_strength();
  par2.invert_strength();
  return *this;
}


float SeqDiffWeight::get_strength() const {
  return par1.get_strength();
}

fvector SeqDiffWeight::get_gradintegral() const {
  return par1.get_gradintegral()+par2.get_gradintegral();
}


double SeqDiffWeight::get_gradduration() const {
  return par1.get_gradduration()+par2.get_gradduration();
}

SeqGradInterface& SeqDiffWeight::set_gradrotmatrix(const RotMatrix& matrix) {
  par1.set_gradrotmatrix(matrix);
  par2.set_gradrotmatrix(matrix);
  return *this;
}


void SeqDiffWeight::build_seq() {


//  par1 /= (pfg1[readDirection] / pfg1[phaseDirection] / pfg1[sliceDirection]);
//  par2 /= (pfg2[readDirection] / pfg2[phaseDirection] / pfg2[sliceDirection]);

  SeqGradChanParallel* sgcp1=new SeqGradChanParallel(get_label()+"_sgcp1");
  sgcp1->set_temporary();

  SeqGradChanParallel* sgcp2=new SeqGradChanParallel(get_label()+"_sgcp2");
  sgcp2->set_temporary();

  SeqSimultanVector::clear();

  for(int idim=0; idim<n_directions; idim++) {
    if(pfg1[idim].get_strength()) {
      (*sgcp1) /= pfg1[idim];
      SeqSimultanVector::operator += (pfg1[idim]);
    }
    if(pfg2[idim].get_strength()) {
      (*sgcp2) /= pfg2[idim];
      SeqSimultanVector::operator += (pfg2[idim]);
    }
  }

  par1 /= (*sgcp1);
  par2 /= (*sgcp2);

  SeqObjList::operator += (par1);
  SeqObjList::operator += (middle_part);
  SeqObjList::operator += (par2);
}





//////////////////////////////////////////////////////////////////////////////////////////




SeqDiffWeightFlowComp::SeqDiffWeightFlowComp(const STD_string& object_label,
                    const fvector& bvals, float maxgradstrength,
                    direction chan, double stimdelay, const STD_string& nucleus)
  : SeqGradChanList(object_label), SeqSimultanVector(object_label),
    delay(object_label+"_delay", chan, stimdelay) {
  Log<Seq> odinlog(this,"SeqDiffWeightFlowComp()");

  fvector gradtrims;
  double delta;
  calc_dw_grads(gradtrims, delta, 0.5*bvals, maxgradstrength, 0.0, systemInfo->get_gamma(nucleus));

  pfg1=SeqGradVectorPulse(object_label+"_pfg1",chan,maxgradstrength, gradtrims,delta);
  pfg2=SeqGradVectorPulse(object_label+"_pfg2",chan,-maxgradstrength,gradtrims,2.0*delta);
  pfg3=SeqGradVectorPulse(object_label+"_pfg3",chan,maxgradstrength, gradtrims,delta);

  build_seq();
}


SeqDiffWeightFlowComp::SeqDiffWeightFlowComp(const STD_string& object_label)
  : SeqGradChanList(object_label), SeqSimultanVector(object_label) {}


SeqDiffWeightFlowComp& SeqDiffWeightFlowComp::operator = (const SeqDiffWeightFlowComp& sgdwfc) {
  SeqSimultanVector::operator = (sgdwfc);
  SeqGradChanList::operator = (sgdwfc);
  pfg1=sgdwfc.pfg1;
  pfg2=sgdwfc.pfg2;
  pfg3=sgdwfc.pfg3;
  delay=sgdwfc.delay;
  build_seq();
  return *this;
}

void SeqDiffWeightFlowComp::build_seq() {

  SeqSimultanVector::clear();
  SeqSimultanVector::operator += (pfg1);
  SeqSimultanVector::operator += (pfg2);
  SeqSimultanVector::operator += (pfg3);

  SeqGradChanList::clear();
  SeqGradChanList::operator += (pfg1);
  SeqGradChanList::operator += (delay);
  SeqGradChanList::operator += (pfg2);
  SeqGradChanList::operator += (delay);
  SeqGradChanList::operator += (pfg3);
}



//////////////////////////////////////////////////////////////////////////////////////////





static float ps3 [9] = { 1.0,  0.0,  0.0, 0.0, 1.0, 0.0, 0.0, 0.0,  1.0};
//static float ps3 [9] = { -0.100796,  0.994902,  0.003109, -0.994900, -0.100783, -0.004049, -0.003715, -0.003501,  0.999987};
static float ps4 [12] = { -0.021560, -0.304929,  0.952131, -0.949220, -0.064610,  0.307906,  0.491865,  0.656790,  0.571572,  0.435796, -0.897109,  0.072653};
static float ps5 [15] = { -0.810867, -0.496465,  0.309866,  0.307318, -0.931254,  0.195761, -0.171601, -0.145829,  0.974314, -0.894745,  0.169209, -0.413280,  0.139610,  0.736549,  0.661819};
static float ps6 [18] = { -0.006226, -0.064447,  0.997902,  0.636679, -0.653135,  0.409945, -0.399917, -0.828419,  0.392157,  0.789559,  0.384929,  0.477940, -0.887689,  0.101313,  0.449159,  0.152552, -0.851204, -0.502175};
static float ps7 [21] = {  0.055396, -0.824375,  0.563326,  0.811301, -0.575497,  0.102927, -0.229063,  0.783589,  0.577511, -0.881413, -0.159849,  0.444477,  0.564952,  0.817041,  0.115211,  0.707747,  0.119062,  0.696361, -0.150399, -0.035322,  0.987994};
static float ps8 [24] = { -0.590425, -0.804665, -0.062549,  0.027408, -0.732319, -0.680410,  0.756064, -0.645224,  0.109787,  0.952343,  0.144115,  0.268837,  0.391908, -0.032990,  0.919413,  0.101437, -0.919831,  0.378974, -0.389232, -0.418658,  0.820502, -0.751151,  0.241893,  0.614215};
static float ps9 [27] = {  0.393918, -0.259108,  0.881868,  0.625857,  0.591521,  0.508337,  0.517317, -0.817156,  0.254244, -0.758341,  0.441699,  0.479396, -0.234181,  0.114957,  0.965373,  0.958459, -0.080740,  0.273564, -0.041437,  0.894887,  0.444364, -0.237557, -0.849808,  0.470524,  0.818447,  0.402205, -0.410335};
static float ps10 [30] = {  0.557836,  0.789965,  0.254507,  0.569859, -0.258762, -0.779937, -0.195499, -0.880727,  0.431393,  0.932196, -0.349524, -0.094038,  0.549517, -0.660507,  0.511627, -0.015038, -0.296982,  0.954765,  0.818590,  0.095700,  0.566350, -0.833628, -0.392682,  0.388415,  0.175157,  0.478983,  0.860172, -0.299722,  0.912303,  0.279053};
static float ps11 [33] = {  0.511351,  0.804163,  0.303054, -0.794769, -0.503466,  0.338916,  0.174571,  0.221229,  0.959470, -0.234636, -0.431883,  0.870875,  0.815945,  0.188147,  0.546657, -0.981514,  0.181966,  0.059311, -0.599540,  0.800343,  0.001858, -0.081149, -0.946839,  0.311306, -0.636726,  0.218439,  0.739503,  0.517936, -0.498952,  0.694830, -0.161200,  0.795247,  0.584462};
static float ps12 [36] = {  0.425180,  0.315653,  0.848284,  0.185196, -0.382898,  0.905037, -0.480267, -0.117082,  0.869273, -0.477164, -0.715637,  0.510077, -0.702985,  0.558424, -0.440426, -0.707505, -0.690260, -0.151584,  0.944077,  0.041899, -0.327051, -0.264785,  0.510148,  0.818314,  0.918467,  0.068806,  0.389467,  0.085525,  0.922253,  0.377009, -0.144817,  0.947629, -0.284652, -0.704094,  0.656651,  0.270298};
static float ps13 [39] = {  0.608489,  0.354732,  0.709863,  0.840005,  0.542193,  0.020474, -0.380743, -0.814648,  0.437474,  0.207531, -0.233095,  0.950051, -0.789643,  0.613015, -0.025996, -0.937460, -0.021576,  0.347425,  0.304986, -0.753712,  0.582152,  0.873326, -0.161334,  0.459645,  0.321048,  0.871575,  0.370521, -0.486692, -0.248656,  0.837437, -0.041337,  0.427363,  0.903135, -0.582926,  0.527413,  0.618088, -0.220542,  0.971625,  0.085473};
static float ps14 [42] = { -0.529230, -0.838809,  0.127733, -0.502637,  0.855008, -0.127741, -0.313344,  0.635274,  0.705864, -0.000007,  0.000003,  1.000000,  0.601859,  0.028092, -0.798108, -0.333120, -0.625125, -0.705868,  0.923984,  0.258254, -0.282062,  0.301096, -0.657339,  0.690830,  0.821113, -0.458657, -0.339715,  0.835101,  0.432657,  0.339727, -0.015439, -0.983550, -0.179977,  0.915418, -0.287130,  0.282075,  0.321591,  0.647564, -0.690826,  0.600669, -0.046970,  0.798117};
static float ps15 [45] = {  0.957387, -0.187071,  0.220034,  0.190618,  0.585245,  0.788133,  0.325492, -0.844583, -0.425129,  0.022796, -0.678409,  0.734331, -0.596977, -0.258722,  0.759395, -0.416250,  0.343103,  0.842031,  0.734448,  0.292307,  0.612489,  0.581881, -0.362502,  0.728016,  0.928822,  0.354407, -0.108098, -0.861661,  0.322776,  0.391608,  0.525770,  0.761172, -0.379714, -0.034273,  0.991162, -0.128153,  0.084508, -0.056169,  0.994838,  0.630413, -0.756662,  0.173326,  0.491239,  0.826626,  0.274542};
static float ps16 [48] = {  0.554077,  0.486008,  0.675866, -0.495011,  0.455826,  0.739721,  0.944451,  0.224351,  0.240166, -0.818498, -0.461979,  0.341521,  0.629291, -0.144331,  0.763650, -0.302298, -0.787745,  0.536725,  0.011914,  0.814134,  0.580555,  0.030663, -0.999031,  0.031565,  0.589202, -0.783569, -0.197131,  0.562891,  0.816715,  0.127007,  0.052951,  0.220175,  0.974022,  0.067168, -0.416946,  0.906446,  0.878961, -0.413143,  0.238202, -0.522802, -0.177477,  0.833775,  0.409735, -0.767260,  0.493386, -0.919674,  0.170720,  0.353630};
static float ps17 [51] = { -0.084677, -0.798978,  0.595368, -0.285378,  0.382557,  0.878755,  0.239575, -0.008537,  0.970840, -0.826274,  0.479411, -0.295697, -0.933600,  0.272403,  0.232780, -0.585702,  0.695361,  0.416445,  0.348167,  0.536125,  0.768993,  0.687326,  0.597291, -0.413312, -0.788627, -0.069187, -0.610967,  0.672858,  0.682522,  0.285353, -0.729422, -0.025495,  0.683588, -0.028902,  0.887279,  0.460327,  0.432026, -0.525413,  0.733004, -0.386679,  0.910660, -0.145523, -0.269733, -0.320655,  0.907978, -0.266447, -0.962145,  0.057290, -0.969994, -0.241445, -0.028553};
static float ps18 [54] = { -0.358616,  0.232844,  0.903979,  0.990653, -0.112342,  0.077367, -0.282930,  0.716688,  0.637423, -0.891249, -0.417614,  0.176844, -0.733882, -0.178601,  0.655377,  0.579701, -0.807043, -0.112374,  0.000650,  0.985575,  0.169239,  0.319924, -0.498679,  0.805586, -0.822530,  0.367692,  0.433874, -0.412439, -0.753502,  0.511984,  0.153276, -0.903274,  0.400754,  0.228998,  0.150756,  0.961682,  0.720077, -0.052737,  0.691887,  0.530172,  0.845386,  0.065124,  0.737068, -0.568030,  0.366160,  0.309857,  0.667672,  0.676907, -0.209598, -0.358489,  0.909700,  0.795763,  0.431108,  0.425331};
static float ps19 [57] = { -0.506190, -0.182434,  0.842905, -0.353038, -0.682139,  0.640352,  0.416262,  0.804302,  0.424057, -0.364266, -0.916701,  0.164226,  0.265893, -0.875818,  0.402794, -0.788937, -0.296829, -0.538025, -0.635112,  0.742750,  0.212027, -0.853341, -0.307549,  0.420979,  0.000000,  0.000000,  1.000000, -0.170383,  0.527122, -0.832533,  0.617996, -0.195486,  0.761490, -0.126832,  0.982490,  0.136478,  0.755313, -0.565523,  0.331189, -0.890859,  0.267459,  0.367200,  0.987215, -0.062418,  0.146668,  0.317816,  0.401218,  0.859079, -0.500303,  0.374591,  0.780627,  0.131853, -0.737897, -0.661909, -0.813935, -0.580022, -0.032945};
static float ps20 [60] = { -0.947965,  0.033315,  0.316625, -0.399719, -0.680997,  0.613570, -0.224707,  0.635190,  0.738945, -0.625805, -0.223904,  0.747152,  0.664622,  0.112881,  0.738604, -0.855031, -0.480158,  0.195886, -0.256130,  0.083729,  0.963009, -0.198427,  0.183236, -0.962835,  0.168344,  0.896561,  0.409682, -0.505636, -0.860974,  0.055276,  0.690315, -0.403910,  0.600269, -0.002786, -0.980688,  0.195559, -0.966326, -0.047679, -0.252866, -0.428878,  0.867093,  0.253402,  0.490998, -0.811800,  0.316073,  0.109684, -0.655863,  0.746869,  0.701264,  0.583572,  0.409477, -0.862283,  0.506426,  0.000331, -0.688646,  0.405965,  0.600798,  0.266429,  0.453089,  0.850721};
static float ps21 [63] = {  0.281102,  0.664650,  0.692259,  0.912517,  0.330300, -0.241279, -0.523441, -0.498410,  0.691084,  0.959417, -0.211461, -0.186559, -0.737697, -0.614913, -0.278721, -0.671940,  0.628624,  0.391572, -0.593050, -0.773400,  0.223929,  0.214257,  0.116502,  0.969805,  0.409038, -0.735521,  0.540090,  0.953042,  0.036026,  0.300687, -0.738786,  0.080476,  0.669118,  0.629538, -0.272009,  0.727800, -0.348436,  0.936659, -0.035541, -0.297481, -0.074546,  0.951813, -0.282864,  0.460513,  0.841377,  0.065572, -0.476145,  0.876918,  0.240577,  0.958924,  0.150293, -0.804358,  0.556622, -0.207800,  0.661996,  0.272751,  0.698118, -0.108159, -0.886921,  0.449079, -0.183756,  0.858018,  0.479623};
static float ps22 [66] = { -0.882341,  0.332209, -0.333333, -0.352361, -0.935864,  0.000000,  0.986663,  0.162779,  0.000000,  0.341948,  0.773385,  0.533805, -0.498797,  0.682828, -0.533805,  0.728872,  0.598025, -0.333333, -0.000000, -0.000000,  1.000000,  0.634302, -0.773086,  0.000000,  0.158690, -0.961879,  0.222724,  0.087078, -0.527812,  0.844886,  0.413560,  0.339318,  0.844886, -0.840745, -0.090557,  0.533805,  0.446341,  0.366214, -0.816497, -0.540321,  0.203436, -0.816497, -0.825279, -0.184302, -0.533805,  0.912357, -0.343510, -0.222724, -0.753667, -0.618369, -0.222724,  0.253029,  0.806863, -0.533805, -0.572250,  0.622562,  0.533805, -0.153469,  0.930234,  0.333333, -0.093980,  0.569650,  0.816497,  0.500637, -0.188494, -0.844886};
static float ps23 [69] = {  0.461401, -0.470194,  0.752347, -0.001035, -0.702076,  0.712101,  0.335322,  0.027683,  0.941697, -0.165846,  0.814522,  0.555922,  0.663968,  0.730147, -0.161344,  0.801328, -0.534568,  0.268534,  0.604447, -0.778192, -0.170471,  0.411647,  0.852388,  0.322461, -0.640902, -0.026697,  0.767158,  0.788165, -0.084222,  0.609674, -0.080094,  0.992163,  0.095902,  0.223489,  0.927836, -0.298619,  0.991384, -0.050116,  0.121023,  0.903221,  0.247278, -0.350777, -0.871316, -0.449293, -0.197343,  0.896749, -0.315784, -0.310034, -0.525387, -0.541060,  0.656675, -0.188149,  0.259579,  0.947216,  0.200720,  0.574644,  0.793408, -0.128638, -0.262416,  0.956342,  0.378311, -0.859291,  0.344239,  0.644524,  0.418362,  0.639971, -0.564167,  0.497659,  0.658826};
static float ps24 [72] = {  0.132723,  0.739879, -0.659517, -0.918278, -0.379929,  0.111440, -0.965426,  0.153303,  0.210835,  0.608607, -0.784469,  0.119187,  0.639150,  0.437609,  0.632444,  0.789542,  0.209334, -0.576891,  0.135008, -0.362540,  0.922138,  0.344276,  0.093474,  0.934204,  0.682349,  0.706470,  0.187884,  0.141671,  0.550473,  0.822745,  0.699577, -0.167742,  0.694590, -0.591356, -0.690452,  0.416621,  0.935168,  0.136207,  0.326968,  0.644819, -0.248547, -0.722795, -0.727042,  0.600675,  0.332565, -0.156172,  0.940576, -0.301540, -0.196180,  0.149388,  0.969121,  0.293367,  0.952754, -0.078708,  0.467415, -0.628758,  0.621439,  0.242627,  0.862959,  0.443209, -0.321942,  0.668161,  0.670756,  0.229728, -0.943726, -0.237921, -0.391999, -0.321717,  0.861879,  0.887686, -0.387799,  0.248245};
static float ps25 [75] = { -0.390766,  0.445739,  0.805368, -0.207574, -0.064662,  0.976080, -0.695120, -0.716447,  0.059252,  0.603415, -0.536675,  0.589806,  0.869788, -0.131387,  0.475612,  0.348154,  0.902751,  0.252644, -0.841901, -0.429795, -0.326312,  0.026932,  0.992875, -0.116078,  0.199282, -0.822464,  0.532766, -0.866657, -0.349331,  0.356193,  0.328448, -0.243190,  0.912678, -0.536597, -0.448887,  0.714537, -0.639476,  0.653882,  0.404362,  0.137832,  0.291341,  0.946638,  0.900413, -0.230926, -0.368687, -0.372910, -0.822287,  0.429862, -0.668219,  0.036606,  0.743064, -0.229908,  0.922003,  0.311534,  0.002286,  0.727683,  0.685910,  0.487929,  0.607072,  0.627207, -0.868181,  0.488192, -0.089047, -0.997594, -0.063085, -0.028748, -0.057425, -0.548469,  0.834196,  0.613080,  0.142607,  0.777043, -0.522776,  0.844196, -0.118489};
static float ps26 [78] = { -0.170472, -0.438955,  0.882189,  0.554683, -0.770371,  0.314414,  0.345112, -0.727674, -0.592780, -0.416733,  0.254294,  0.872736, -0.763875,  0.349438,  0.542576,  0.747465, -0.374683,  0.548551,  0.284346, -0.587046,  0.757974,  0.636315, -0.740927, -0.214778, -0.000000,  0.000000,  1.000000,  0.432938,  0.379714,  0.817546,  0.006321,  0.520762,  0.853679,  0.451156, -0.136178,  0.881994, -0.150889, -0.863141, -0.481893,  0.599057,  0.676130,  0.428928,  0.226535,  0.973861, -0.016605, -0.946988, -0.026404,  0.320181, -0.625272, -0.775587,  0.086600, -0.765174, -0.460884,  0.449549, -0.363201, -0.745474,  0.558887,  0.895953, -0.443971,  0.012599, -0.592938, -0.176069,  0.785763,  0.079483, -0.899040,  0.430592, -0.903853, -0.419147, -0.085820,  0.978816, -0.037941,  0.201195,  0.791067,  0.176959,  0.585575, -0.233199,  0.968071,  0.091965};
static float ps27 [81] = {  0.463021, -0.849503,  0.252896,  0.091872, -0.435100,  0.895683,  0.113035,  0.782286,  0.612578, -0.782141, -0.478793,  0.398763,  0.618257, -0.507255,  0.600376,  0.383776,  0.378376,  0.842347, -0.202217,  0.957680,  0.204833,  0.605451,  0.671866,  0.426645,  0.905808,  0.410461,  0.105043,  0.994425, -0.009222, -0.105043,  0.506335, -0.118428,  0.854166,  0.908941, -0.121717,  0.398763,  0.867348, -0.482561,  0.121824,  0.636480, -0.747878, -0.188604, -0.598218, -0.792019,  0.121824, -0.415234, -0.313019,  0.854166, -0.503985,  0.190923,  0.842346, -0.000000, -0.000000,  1.000000, -0.094212,  0.446180,  0.889970,  0.162534, -0.769747,  0.617310,  0.769074,  0.251434,  0.587627,  0.825389, -0.369741, -0.426645, -0.080058, -0.964176,  0.252896, -0.805072, -0.080951,  0.587627, -0.279795, -0.941352, -0.188604, -0.419649,  0.669808,  0.612578, -0.360407, -0.713902,  0.600376};
static float ps28 [84] = { -0.200535,  0.176758,  0.963609, -0.789706,  0.093773, -0.606277, -0.634769,  0.434030,  0.639286,  0.371816, -0.791058,  0.485778,  0.295843,  0.516228,  0.803732,  0.558554,  0.721872,  0.408556,  0.757931,  0.348475,  0.551458, -0.769718,  0.520916, -0.369026, -0.072981, -0.922968,  0.377894, -0.814183, -0.367904,  0.449169,  0.836984, -0.511974, -0.193235, -0.567532,  0.822113, -0.045132, -0.936615,  0.064113,  0.344444, -0.951494, -0.292660, -0.094916,  0.097786,  0.871555,  0.480448,  0.744405,  0.665315, -0.056724,  0.414048,  0.099130,  0.904841, -0.010740, -0.636290,  0.771375,  0.459723, -0.405579,  0.790038,  0.048340, -0.194793,  0.979652, -0.418882,  0.803886,  0.422262, -0.173571,  0.588245,  0.789836, -0.487511, -0.721184,  0.492166, -0.406398, -0.371802,  0.834628,  0.357460,  0.933785,  0.016391, -0.132835,  0.989660,  0.054113, -0.977329,  0.156583, -0.142513, -0.641476,  0.004168,  0.767132};
static float ps29 [87] = { -0.005354,  0.838993,  0.544116, -0.813080,  0.581660,  0.023934, -0.704906, -0.614780,  0.353769, -0.075777,  0.995211,  0.061751,  0.481696, -0.869085,  0.112513,  0.861387,  0.263749,  0.434106, -0.139595, -0.323021,  0.936040, -0.137055,  0.154882,  0.978380,  0.736312,  0.666344,  0.117604,  0.148443, -0.885577,  0.440135,  0.274554, -0.090726,  0.957282, -0.517675, -0.645708, -0.561314,  0.556108,  0.381572, -0.738341, -0.347114,  0.549350,  0.760083,  0.555705,  0.242041,  0.795367, -0.974030,  0.168711,  0.151007,  0.338337,  0.907573, -0.248675,  0.150825,  0.499473,  0.853100, -0.225735, -0.715029,  0.661647,  0.928918, -0.196774,  0.313674,  0.958288,  0.284151, -0.030686,  0.674417, -0.210137,  0.707817, -0.349431, -0.911036, -0.218888,  0.274834, -0.561372,  0.780594, -0.469528,  0.803671,  0.365589, -0.861943, -0.128115,  0.490552,  0.642962, -0.608202,  0.465499, -0.569978,  0.095177,  0.816129, -0.754108,  0.419020,  0.505711};
static float ps30 [90] = { -0.546405,  0.619202,  0.563943, -0.398931, -0.600006,  0.693432,  0.587973,  0.521686,  0.618168,  0.055894, -0.991971, -0.113444, -0.666933, -0.677984,  0.309094,  0.163684,  0.533013,  0.830123,  0.542826,  0.133898,  0.829102, -0.074751, -0.350412,  0.933608,  0.845751, -0.478624, -0.235847,  0.767148, -0.610673,  0.196372, -0.283810,  0.381633,  0.879663,  0.537228, -0.616249,  0.575868, -0.711387,  0.197778,  0.674398,  0.886511,  0.219025,  0.407586,  0.296061,  0.842985,  0.449136, -0.937540, -0.340990,  0.068877,  0.398833,  0.917023,  0.000835,  0.097278, -0.711949,  0.695460, -0.311534,  0.908623, -0.278121, -0.432043, -0.089758,  0.897375, -0.949980,  0.030810,  0.310788,  0.146722, -0.811981, -0.564942, -0.172201, -0.908573,  0.380580,  0.507209, -0.848578, -0.150513, -0.730808, -0.654136, -0.194999,  0.077744,  0.094961,  0.992441,  0.383976, -0.293959,  0.875300,  0.788208, -0.213656,  0.577130, -0.752333, -0.301447,  0.585769, -0.975732,  0.165497, -0.143382};
static float ps31 [93] = {  0.204365,  0.723592,  0.659280,  0.156581,  0.952862,  0.259878,  0.502093,  0.845243, -0.182942, -0.462199,  0.146202,  0.874641, -0.828951, -0.557577,  0.044144,  0.759598, -0.365808,  0.537769, -0.461678, -0.305697,  0.832708,  0.293945,  0.300853,  0.907239, -0.744891, -0.458029,  0.485125, -0.053573, -0.007571,  0.998535, -0.801661,  0.004602,  0.597761,  0.906689, -0.326546, -0.266990, -0.634552,  0.733992,  0.242072,  0.898661,  0.303062,  0.317116,  0.386692, -0.157930,  0.908585, -0.147760,  0.482285,  0.863463, -0.282948,  0.955956,  0.078033,  0.077351,  0.977208, -0.197689,  0.415644, -0.565839,  0.712086, -0.367403, -0.708805,  0.602171,  0.809329, -0.571428,  0.135854, -0.024719, -0.459890,  0.887632,  0.971604, -0.128709,  0.198542,  0.605429,  0.490255,  0.626981, -0.247787,  0.807881,  0.534723,  0.072291, -0.816662,  0.572570,  0.474533, -0.823107,  0.311951,  0.728476,  0.053702,  0.682963, -0.610171,  0.488179,  0.623997,  0.973963,  0.137190, -0.180485,  0.571718,  0.771067,  0.280348};
static float ps32 [96] = {  0.519013,  0.854199,  0.031151,  0.801211, -0.063809,  0.594970,  0.514837,  0.726299,  0.455448,  0.117967,  0.965760,  0.231065, -0.467151, -0.549242,  0.692895,  0.716315,  0.197382, -0.669278,  0.599777, -0.492419,  0.630707, -0.800020,  0.223072,  0.556963,  0.029535,  0.733272, -0.679294,  0.379870,  0.402820,  0.832727, -0.951171, -0.161172,  0.263244, -0.102942,  0.448113,  0.888030, -0.760744, -0.566441,  0.316879, -0.697256,  0.653755,  0.294005, -0.944892,  0.293928,  0.144174, -0.353468,  0.934011,  0.051810, -0.680714,  0.715159, -0.158670,  0.747590,  0.362889,  0.556256,  0.890215, -0.360105,  0.279001, -0.378146, -0.845537,  0.376926, -0.281684,  0.832306,  0.477411,  0.841861,  0.525064,  0.124811,  0.478308, -0.041353,  0.877218,  0.231306, -0.428135,  0.873612, -0.206770, -0.303548,  0.930110,  0.978177,  0.099085,  0.182624,  0.118710,  0.750307,  0.650344,  0.004364,  0.977355, -0.211562, -0.348364,  0.817823, -0.458049, -0.435353,  0.090815,  0.895667, -0.498520,  0.514078,  0.697998,  0.058008,  0.049572,  0.997085};
static float ps33 [99] = {  0.089846, -0.796132,  0.598416, -0.889197, -0.439400,  0.127498,  0.878732,  0.057482, -0.473842,  0.298808,  0.383830,  0.873721, -0.667462,  0.734674,  0.121444,  0.902709, -0.168444,  0.395908, -0.466351,  0.711849,  0.525154, -0.012642,  0.898708,  0.438366, -0.796352,  0.378713,  0.471593,  0.998344,  0.021989, -0.053149,  0.921461, -0.385101, -0.051050,  0.287662, -0.049396,  0.956457,  0.511047, -0.533415,  0.674017, -0.317868, -0.543955,  0.776578,  0.671652,  0.316978,  0.669633, -0.194675, -0.169063,  0.966188,  0.655941, -0.120039,  0.745206, -0.412577,  0.858600, -0.304281, -0.117792,  0.238027,  0.964089,  0.139886, -0.457650,  0.878060, -0.697865, -0.651979, -0.296492, -0.512479,  0.326764,  0.794098,  0.288795, -0.949244, -0.124633, -0.096132,  0.630120,  0.770524,  0.752046, -0.576139,  0.320141, -0.910968, -0.265341, -0.315802, -0.338708, -0.827785,  0.447268,  0.375133,  0.714447,  0.590627, -0.692855, -0.487474,  0.531338,  0.325491,  0.932049,  0.159189, -0.594859, -0.109432,  0.796347, -0.614860, -0.777249,  0.133535, -0.036549, -0.984926,  0.169073};
static float ps34 [102] = { -0.227292,  0.546703,  0.805887,  0.441153,  0.200379,  0.874776, -0.384424,  0.088677,  0.918888, -0.067113,  0.675224, -0.734553,  0.289281,  0.820849,  0.492466,  0.215005, -0.899033,  0.381460,  0.057075, -0.994756, -0.084875, -0.102712, -0.320427,  0.941688,  0.626386,  0.750060, -0.212252,  0.960624,  0.097251, -0.260276,  0.910037,  0.414253,  0.015075,  0.018249,  0.157257,  0.987389, -0.565995,  0.716767,  0.407302, -0.879234,  0.337122,  0.336596,  0.896546,  0.191369,  0.399480,  0.688778,  0.678433,  0.255565, -0.152798,  0.855448,  0.494835,  0.192637,  0.562150,  0.804287, -0.816991,  0.576633,  0.004479, -0.766316, -0.448832,  0.459684,  0.709943, -0.075851,  0.700162, -0.501383, -0.327837,  0.800711,  0.360787,  0.928343,  0.089511, -0.364122, -0.688358,  0.627358, -0.606184,  0.401621,  0.686471,  0.304947, -0.219571,  0.926712, -0.457250,  0.888461,  0.039500, -0.597904,  0.722092, -0.347985,  0.482321, -0.522788,  0.702893,  0.837085, -0.353846,  0.417230,  0.763822,  0.008667, -0.645369,  0.630903,  0.474301,  0.614003,  0.196633,  0.925540, -0.323591,  0.987330, -0.123742,  0.099330};
static float ps35 [105] = {  0.322845,  0.739190,  0.591075,  0.952764, -0.275224,  0.128425, -0.435735,  0.871470, -0.225112,  0.822539, -0.152951,  0.547755, -0.163205,  0.785170,  0.597387,  0.552607, -0.029060,  0.832935,  0.220823, -0.329362,  0.918019,  0.265105,  0.866584, -0.422791,  0.659292,  0.675265,  0.330685, -0.203207, -0.263415,  0.943037, -0.011602,  0.507891,  0.861343, -0.491542,  0.444266,  0.749009,  0.829314,  0.329633, -0.451199, -0.589689, -0.148796,  0.793805, -0.786395,  0.142270,  0.601118,  0.443586,  0.895704,  0.030746,  0.752972,  0.329843,  0.569418,  0.655995,  0.703730, -0.272826,  0.955898,  0.155214,  0.249334,  0.506293, -0.505363,  0.698767, -0.004827,  0.990475, -0.137610,  0.141870,  0.096952,  0.985126,  0.421916,  0.400296,  0.813480, -0.313556,  0.929918,  0.192185, -0.579323,  0.685339,  0.441243,  0.980918,  0.052353, -0.187243,  0.122013,  0.948813,  0.291320, -0.290694,  0.157443,  0.943774, -0.078173, -0.635601,  0.768050, -0.211441,  0.807358, -0.550877, -0.497337, -0.557741,  0.664515,  0.734731, -0.565231,  0.375080, -0.878318,  0.379100,  0.291275,  0.873713,  0.486357,  0.009089, -0.723907,  0.688934,  0.036449};
static float ps36 [108] = { -0.442021,  0.233720,  0.866021,  0.265297,  0.963214, -0.042849,  0.014198,  0.835225,  0.549726, -0.918570, -0.395226, -0.005005, -0.579713,  0.813187, -0.051581,  0.892537,  0.236984,  0.383687,  0.843422, -0.180847,  0.505899, -0.745608,  0.303434,  0.593293,  0.133929, -0.265066,  0.954884, -0.309793,  0.894950,  0.321082,  0.327869,  0.180018,  0.927414,  0.643428, -0.565080,  0.516415,  0.345805,  0.874587,  0.339876,  0.057298,  0.888801, -0.454697, -0.316097, -0.214039,  0.924267,  0.344138, -0.857246,  0.383013,  0.663879,  0.219903,  0.714778, -0.937860, -0.079616,  0.337756,  0.682960,  0.605834,  0.408082, -0.019210,  0.522484,  0.852433,  0.993716, -0.058561,  0.095385,  0.661588,  0.749862,  0.002750, -0.397201,  0.624181,  0.672778, -0.513865, -0.506286,  0.692544,  0.290595, -0.606963,  0.739696,  0.917284, -0.336405, -0.213123, -0.122197, -0.615176,  0.778862, -0.126908,  0.991809, -0.014502, -0.065870,  0.123218,  0.990191, -0.673790,  0.661289,  0.329703, -0.788497, -0.485536,  0.377527,  0.540353, -0.214506,  0.813638,  0.382021,  0.587004,  0.713783,  0.850692, -0.499691,  0.163194, -0.462873, -0.796214,  0.389606, -0.703715, -0.115243,  0.701074};
static float ps37 [111] = { -0.954059, -0.257965,  0.152396,  0.385177,  0.401219,  0.831061, -0.044825,  0.566975,  0.822514, -0.671872,  0.605565, -0.426473,  0.175678,  0.946160,  0.271878, -0.980206,  0.191084,  0.051793, -0.802935,  0.485758,  0.345448,  0.299326,  0.945908, -0.125147,  0.603043, -0.331384,  0.725619, -0.685088, -0.701521,  0.196273,  0.758506,  0.089442,  0.645499, -0.894808,  0.023151,  0.445850,  0.801753, -0.596312,  0.040054, -0.607684, -0.102556,  0.787529,  0.843509,  0.515612,  0.150454,  0.190019, -0.349851,  0.917331, -0.675261,  0.288604,  0.678771,  0.041084, -0.916783,  0.397267, -0.195784, -0.189168,  0.962229,  0.058929,  0.149113,  0.987063, -0.952760, -0.136443, -0.271352,  0.682823,  0.489769,  0.542106, -0.557306, -0.797681, -0.230468,  0.153620, -0.987511, -0.034965, -0.389228, -0.780049,  0.489923, -0.326981,  0.694221, -0.641202, -0.438140, -0.474772,  0.763299, -0.762006, -0.416709,  0.495682, -0.155024,  0.856206,  0.492828, -0.449239,  0.868714, -0.208615, -0.064280, -0.649466,  0.757669, -0.331738,  0.249426,  0.909800,  0.297073,  0.735335,  0.609122,  0.433035, -0.002622,  0.901373,  0.522125, -0.820246, -0.233628, -0.892303,  0.257373, -0.370883, -0.451826,  0.626854,  0.634749};
static float ps38 [114] = { -0.676206,  0.728972, -0.106518,  0.072231, -0.908443,  0.411720, -0.389288, -0.080230,  0.917615, -0.676402,  0.167071,  0.717334, -0.460690, -0.867151,  0.189245,  0.448263,  0.400819,  0.799003, -0.246186, -0.764755, -0.595435,  0.068469,  0.500765,  0.862871, -0.823912,  0.499391,  0.267915,  0.062498, -0.653811,  0.754073, -0.290923, -0.781577,  0.551817, -0.015735,  0.115881,  0.993138,  0.441939, -0.752626,  0.488102,  0.920465, -0.071320,  0.384262, -0.909908,  0.106315,  0.400955,  0.736330, -0.473462,  0.483377,  0.013905, -0.296972,  0.954785,  0.640695,  0.626934,  0.443242,  0.908488,  0.332755, -0.252833, -0.210124,  0.738705,  0.640440,  0.759884,  0.259220,  0.596138,  0.061906, -0.944189, -0.323535, -0.334017,  0.366930,  0.868214, -0.751032, -0.660191, -0.009921, -0.399527, -0.886702, -0.232673, -0.662679, -0.597622,  0.451337, -0.491780,  0.821935,  0.287360,  0.915422, -0.389028,  0.103243,  0.097537,  0.994018, -0.049148, -0.362266, -0.007743, -0.932042,  0.325649, -0.942891,  0.070062,  0.675680, -0.125663,  0.726405,  0.408413, -0.434792,  0.802592, -0.370998, -0.468629,  0.801715, -0.592198,  0.551955,  0.587066,  0.915494,  0.345473,  0.206201, -0.724062, -0.239792,  0.646710,  0.999626, -0.004145, -0.027025};
static float ps39 [117] = {  0.347153,  0.508688,  0.787859, -0.078282,  0.947920,  0.308739,  0.631909,  0.764615,  0.126708,  0.988533,  0.143494, -0.047031,  0.192917, -0.976621,  0.094843,  0.291491,  0.834533,  0.467534,  0.498874, -0.585235,  0.639237,  0.801485, -0.597122, -0.032657, -0.144644, -0.937323,  0.317023,  0.012081, -0.114745,  0.993322, -0.625179, -0.176163,  0.760340,  0.702102, -0.175371,  0.690144,  0.800827, -0.453784,  0.390839, -0.014742,  0.694437,  0.719403, -0.878699, -0.177390,  0.443193,  0.192259, -0.833283,  0.518339, -0.921493,  0.222000,  0.318694, -0.366381,  0.765325,  0.529190,  0.385601,  0.118180,  0.915066,  0.549586, -0.791079,  0.268604,  0.920391,  0.018063,  0.390582, -0.349828,  0.074056,  0.933882,  0.364182, -0.286254,  0.886245, -0.632981, -0.531079,  0.563285, -0.001472,  0.320596,  0.947215,  0.877627,  0.410926,  0.246800,  0.076526, -0.565476,  0.821207, -0.530028, -0.803470,  0.271120,  0.500102, -0.852700, -0.150998, -0.835436, -0.520972,  0.175031, -0.388262,  0.461512,  0.797658,  0.703022,  0.242172,  0.668665, -0.264781, -0.961753, -0.070160, -0.293581, -0.379303,  0.877462,  0.623223,  0.602331,  0.498789, -0.705112,  0.213058,  0.676330, -0.703858,  0.557815,  0.439803, -0.265884, -0.721690,  0.639116,  0.965898, -0.244797,  0.084355};
static float ps40 [120] = {  0.558523, -0.643113,  0.523886,  0.118807,  0.947226,  0.297739,  0.147131, -0.682186,  0.716222,  0.285042, -0.883821,  0.370960,  0.335250, -0.044445,  0.941080,  0.873413,  0.184236,  0.450784,  0.499484,  0.804442,  0.321540,  0.031536, -0.994644,  0.098427, -0.832278,  0.540204,  0.124473, -0.832799, -0.464545,  0.301072, -0.350077, -0.199964,  0.915129,  0.822155, -0.501563,  0.269251, -0.896134, -0.097505,  0.432939,  0.354502,  0.931427, -0.082298, -0.382487,  0.224761,  0.896207,  0.972570, -0.171596, -0.157042, -0.675831,  0.026545,  0.736579,  0.577818, -0.810686,  0.094418,  0.179984, -0.807595, -0.561602, -0.480516,  0.539719,  0.691236, -0.138537, -0.871912,  0.469657, -0.025579,  0.097641,  0.994893,  0.519215,  0.729128, -0.445857,  0.637324,  0.120443,  0.761125,  0.442010, -0.392731,  0.806467, -0.962447,  0.132955, -0.236682,  0.761366, -0.236438,  0.603671,  0.635300,  0.388715, -0.667304, -0.268758, -0.589576,  0.761689,  0.686201,  0.725438, -0.053554,  0.973461,  0.228837,  0.002569,  0.818706,  0.518341,  0.247070,  0.304040,  0.375695,  0.875450,  0.299135, -0.937617, -0.177181, -0.581244,  0.729844,  0.359837,  0.030601, -0.342747,  0.938929,  0.789913, -0.345411, -0.506684, -0.076637,  0.518479,  0.851649,  0.232583,  0.727661,  0.645302,  0.602447,  0.514123,  0.610520};
static float ps41 [123] = {  0.570240, -0.503812,  0.648845, -0.213208, -0.976102,  0.042047, -0.457635, -0.131533,  0.879357,  0.316770, -0.807971,  0.496830,  0.624689, -0.735275,  0.262935, -0.195457, -0.699901,  0.686975,  0.311562, -0.246405,  0.917722, -0.825116,  0.560457,  0.071214,  0.889796, -0.039074,  0.454683, -0.652207, -0.285222, -0.702335, -0.900608,  0.221247,  0.374105,  0.526334,  0.487232, -0.696834,  0.244279, -0.961976,  0.122185,  0.000000,  0.000000,  1.000000,  0.041279,  0.443230,  0.895457,  0.351628,  0.178778,  0.918910, -0.077842,  0.967549,  0.240394, -0.465230, -0.801424,  0.375873,  0.975310, -0.219224,  0.026687,  0.835478, -0.413691,  0.361714,  0.768507,  0.143169, -0.623618, -0.573996, -0.818799, -0.009854, -0.535477,  0.836017,  0.119751, -0.983473, -0.153943, -0.095302,  0.381211,  0.589665,  0.712021,  0.178675, -0.592369,  0.785604, -0.943716, -0.173476,  0.281615, -0.355285,  0.812979,  0.461343, -0.620613, -0.677921, -0.394034,  0.844211,  0.371366,  0.386516, -0.137437, -0.366020,  0.920402, -0.357821,  0.558649,  0.748248, -0.843430, -0.537235, -0.002233,  0.649835, -0.123856,  0.749916, -0.307755,  0.222194,  0.925158, -0.295019, -0.882404, -0.366506, -0.687405,  0.565538,  0.455675, -0.057362, -0.915503,  0.398200,  0.759638,  0.528964, -0.378348, -0.661319,  0.241290,  0.710237,  0.015192,  0.767478,  0.640896};
static float ps42 [126] = { -0.398081,  0.759978,  0.513774, -0.552704,  0.436246,  0.710076,  0.033739, -0.058664,  0.997707, -0.826846,  0.256869,  0.500344, -0.109287,  0.949319,  0.294702, -0.862751, -0.174568,  0.474538,  0.830917,  0.018557,  0.556086,  0.131261, -0.708656,  0.693237,  0.636126,  0.638353,  0.433416,  0.349266,  0.589509,  0.728349, -0.075739, -0.911351,  0.404602,  0.963324,  0.254117, -0.086206,  0.807793,  0.588220,  0.038308,  0.538507,  0.840319,  0.062237,  0.669202, -0.714242,  0.205003,  0.884388,  0.330360,  0.329728,  0.353187, -0.847851,  0.395483,  0.828046, -0.387895,  0.404818,  0.633020,  0.323865,  0.703134,  0.471747,  0.820072, -0.323940,  0.886390, -0.462936,  0.001550,  0.132193,  0.338336,  0.931694, -0.186235,  0.539339,  0.821237, -0.206069,  0.975165, -0.081175,  0.624803, -0.219936,  0.749166,  0.288831, -0.358754,  0.887622, -0.364703, -0.204777,  0.908327, -0.600328, -0.420416,  0.680336,  0.731451, -0.597018, -0.329468,  0.185236,  0.982546, -0.017063,  0.415947,  0.072692,  0.906479,  0.019880,  0.777833,  0.628157,  0.305069,  0.872705,  0.381208,  0.534377, -0.581454,  0.613475,  0.973964, -0.129556, -0.186035, -0.755350, -0.554569,  0.349141, -0.646828,  0.019773,  0.762380,  0.977793, -0.068381,  0.198101, -0.095416, -0.456877,  0.884398, -0.514263,  0.848324,  0.126016, -0.290488,  0.183591,  0.939101, -0.316725, -0.685919,  0.655134};
static float ps43 [129] = {  0.327172, -0.332909,  0.884381, -0.707217,  0.605258,  0.365386,  0.621561,  0.368700,  0.691175, -0.868663, -0.397352,  0.295864,  0.669593, -0.295823,  0.681274, -0.094650,  0.957781,  0.271471,  0.123828, -0.869340,  0.478450,  0.861175,  0.349879,  0.368730,  0.460114, -0.633911,  0.621653, -0.957033,  0.170378, -0.234645, -0.525769, -0.848901, -0.054167, -0.601393, -0.740231,  0.300641,  0.787918, -0.487166,  0.376636, -0.479980,  0.867981,  0.127389, -0.372140,  0.780485,  0.502350, -0.485079,  0.493392,  0.721985, -0.132336,  0.463386,  0.876219, -0.805300,  0.274682,  0.525397, -0.510420,  0.810062, -0.288568, -0.220230,  0.099346,  0.970376, -0.777915, -0.136173,  0.613437,  0.086576, -0.631340,  0.770658,  0.146883,  0.008276,  0.989119, -0.184836,  0.975408, -0.120059, -0.089089, -0.314400,  0.945101, -0.288923, -0.629213,  0.721536,  0.240141,  0.364283,  0.899795, -0.024708,  0.758329,  0.651404,  0.625107,  0.671246,  0.398334, -0.935819,  0.333248,  0.114840, -0.624029, -0.504833,  0.596432, -0.767853,  0.639891, -0.030674, -0.983440, -0.171816, -0.057664,  0.835538,  0.017897,  0.549141,  0.338465,  0.653977,  0.676576, -0.258020, -0.869623,  0.420929, -0.452894, -0.249633,  0.855903,  0.523477,  0.038123,  0.851186, -0.959435, -0.005811,  0.281869,  0.818358,  0.573503,  0.037221, -0.269155, -0.893014, -0.360669, -0.188642, -0.980358,  0.057555, -0.563722,  0.134783,  0.814894};
static float ps44 [132] = { -0.596036, -0.656008, -0.463028, -0.022036, -0.214364,  0.976505,  0.596617, -0.729594,  0.334275,  0.090359,  0.890559, -0.445803,  0.887970,  0.455759,  0.061578,  0.311082,  0.919639,  0.239775,  0.096871,  0.625006, -0.774586,  0.587039, -0.487625,  0.646225, -0.667177, -0.203258, -0.716632, -0.746048,  0.665787,  0.011841, -0.101943,  0.941299,  0.321813, -0.689537,  0.244984,  0.681558,  0.954141, -0.299066, -0.013245,  0.181979, -0.745978, -0.640625, -0.408008, -0.733166,  0.544057,  0.535432, -0.578156, -0.615669,  0.646409,  0.754447,  0.113862,  0.846978, -0.418792,  0.327478,  0.465343, -0.142186,  0.873636,  0.995382,  0.093084, -0.023446, -0.394107, -0.903817,  0.166716, -0.731363, -0.120353,  0.671285,  0.770722, -0.147546,  0.619853, -0.646607, -0.466494,  0.603558,  0.057423,  0.516086,  0.854610, -0.892784, -0.307478,  0.329234,  0.809853, -0.476385, -0.342338, -0.922154,  0.096921,  0.374484, -0.209954, -0.788475, -0.578123, -0.031432, -0.998855,  0.036057,  0.836691,  0.354882,  0.417143,  0.242734, -0.434238,  0.867478, -0.945514,  0.006311, -0.325521,  0.504769, -0.795924, -0.334236, -0.366949, -0.370753,  0.853165, -0.062489,  0.144212,  0.987572,  0.430909,  0.496456,  0.753557,  0.300561,  0.170102,  0.938471,  0.398928, -0.916734, -0.021336,  0.253944, -0.917845,  0.305078,  0.324389, -0.390635, -0.861496, -0.707550, -0.658526,  0.256351,  0.426736, -0.000145, -0.904376, -0.267070,  0.730281, -0.628778};
static float ps45 [135] = {  0.298955, -0.215433,  0.929631, -0.297442, -0.873770,  0.384777,  0.632052, -0.773892, -0.040010,  0.968554,  0.133869, -0.209718,  0.072197,  0.787748,  0.611752, -0.837044,  0.290110, -0.463889,  0.019865,  0.076051,  0.996906, -0.587741, -0.598292,  0.544616,  0.044730, -0.863975,  0.501544, -0.011159, -0.993184,  0.116018, -0.321757, -0.037976,  0.946060,  0.984630, -0.062219,  0.163196,  0.359032, -0.912025,  0.198257, -0.773305,  0.546841,  0.320879, -0.087410, -0.337063,  0.937416,  0.728136,  0.520034,  0.446524, -0.255219,  0.676121,  0.691175,  0.162332, -0.576820,  0.800579,  0.435270,  0.804446,  0.404235,  0.677012, -0.648409,  0.348167, -0.793036, -0.246694,  0.556988,  0.929835,  0.340690,  0.139060, -0.848564,  0.155392,  0.505760,  0.353515,  0.189803,  0.915971,  0.385974, -0.745330,  0.543606,  0.638512,  0.289412,  0.713122, -0.605468,  0.449624,  0.656694, -0.058665, -0.952812, -0.297839, -0.486302, -0.348482,  0.801294, -0.471106,  0.772195,  0.426349, -0.598824, -0.773521,  0.207543,  0.551288, -0.444212,  0.706228, -0.628567,  0.057489,  0.775628,  0.643981, -0.087707,  0.759998,  0.714346,  0.688239,  0.126635,  0.874696, -0.474150,  0.100445,  0.277115, -0.943200, -0.183254, -0.954924,  0.243797,  0.169360,  0.054173,  0.468508,  0.881797, -0.227749, -0.653133,  0.722182,  0.386933,  0.580055,  0.716812,  0.358034,  0.932553,  0.046443,  0.844205,  0.484161, -0.230014, -0.312274,  0.341673,  0.886423,  0.874433,  0.115438,  0.471212};
static float ps46 [138] = {  0.695433,  0.252224,  0.672871, -0.315506,  0.800360,  0.509785,  0.693347,  0.576227,  0.432703,  0.820390, -0.114337,  0.560256,  0.810799,  0.041318, -0.583865,  0.892979, -0.444929, -0.068017, -0.381466, -0.466025,  0.798313, -0.660932,  0.694959,  0.283197, -0.032971, -0.886047,  0.462421,  0.663946, -0.743873,  0.076341, -0.024528,  0.938980,  0.343095,  0.848983,  0.524501,  0.064240, -0.568647,  0.551431,  0.610380,  0.831876, -0.445710,  0.330642,  0.395598,  0.236689,  0.887401, -0.360723, -0.904350,  0.228100,  0.577381, -0.693169,  0.431448, -0.322062,  0.906745, -0.272193, -0.892349, -0.326999,  0.311104, -0.232602,  0.135109,  0.963142,  0.142252, -0.011309,  0.989766,  0.278696, -0.709243,  0.647536, -0.125915, -0.253818,  0.959021,  0.408239,  0.564120,  0.717711, -0.266410,  0.509866,  0.817962,  0.366453,  0.811359,  0.455421, -0.825321,  0.339325,  0.451336, -0.581493,  0.224931,  0.781839, -0.684826, -0.424360,  0.592395, -0.052206, -0.630855,  0.774142, -0.968247,  0.085558,  0.234901,  0.593779, -0.424308,  0.683659,  0.028069, -0.996452,  0.079344, -0.687420, -0.672427,  0.274403,  0.042562,  0.730393,  0.681700, -0.397632, -0.736256,  0.547555,  0.604551,  0.789957,  0.102403, -0.516896, -0.126197,  0.846695,  0.248122, -0.386963,  0.888085, -0.898079, -0.234008, -0.372417,  0.365573, -0.921872, -0.128483,  0.066355,  0.401248,  0.913563,  0.971788, -0.143029,  0.187538,  0.531045, -0.103411,  0.841010, -0.274516, -0.953110, -0.127367, -0.982801, -0.183905, -0.016781};
static float ps47 [141] = { -0.235568, -0.268822,  0.933939, -0.354819, -0.154863, -0.922020, -0.945400,  0.252190, -0.206446, -0.753631, -0.405973,  0.516940, -0.347493,  0.929743,  0.121764, -0.857169,  0.515027,  0.002750,  0.014396,  0.369920,  0.928952, -0.863474, -0.020151,  0.503991, -0.106241, -0.915288,  0.388537, -0.480319, -0.738059,  0.473881,  0.629100, -0.767652,  0.122246, -0.168153, -0.696892,  0.697184,  0.295252,  0.790159,  0.537099, -0.020288,  0.697935,  0.715873, -0.841834, -0.330980, -0.426343,  0.274851, -0.925622,  0.260157, -0.603880, -0.138815,  0.784894, -0.324496,  0.482091,  0.813812, -0.308871, -0.929494, -0.201591,  0.633596, -0.155534,  0.757869, -0.850171,  0.373363,  0.371227,  0.299348, -0.209948,  0.930759, -0.661090,  0.222194,  0.716652,  0.068233, -0.481456,  0.873810, -0.769491,  0.440603, -0.462334,  0.000000, -0.000000,  1.000000, -0.535437,  0.712932, -0.452808, -0.730775, -0.648636,  0.212697,  0.861626, -0.042472,  0.505764,  0.629290,  0.758702,  0.168419, -0.643732,  0.722412,  0.252449, -0.463002, -0.479482,  0.745470,  0.645513,  0.241953,  0.724411,  0.015388,  0.999488, -0.028046, -0.600231,  0.551472,  0.579312, -0.348733,  0.123167,  0.929094,  0.347116,  0.510927,  0.786425, -0.935614, -0.277447,  0.218286, -0.980745,  0.115406,  0.157544,  0.035249, -0.931276, -0.362606,  0.459658, -0.493888,  0.738098, -0.865999, -0.490811, -0.095657, -0.982639, -0.120539, -0.141032, -0.413147, -0.898248,  0.149865,  0.614240,  0.603194,  0.508788, -0.192843,  0.766002, -0.613231, -0.333980,  0.793001,  0.509516};
static float ps48 [144] = { -0.900302,  0.079145,  0.428010,  0.726183,  0.538020,  0.428010,  0.913389,  0.376477,  0.154874,  0.188308,  0.565148,  0.803211, -0.103414,  0.354087,  0.929477,  0.333389, -0.942753,  0.008279,  0.692523,  0.712545,  0.112654, -0.684916,  0.058848,  0.726242,  0.611951,  0.029447,  0.790347, -0.372141,  0.046290,  0.927021,  0.286024,  0.840328, -0.460476, -0.545125, -0.295394,  0.784589,  0.974370, -0.203717, -0.095406,  0.681627, -0.566802,  0.462731,  0.833467,  0.189789,  0.518954, -0.170331,  0.668514,  0.723931, -0.251530,  0.886634,  0.388089,  0.096344,  0.851841,  0.514864, -0.269231, -0.605852,  0.748638, -0.464602,  0.382792,  0.798508,  0.972997,  0.174193, -0.151437,  0.975729,  0.001727,  0.218977,  0.433122,  0.708525,  0.557133,  0.290163,  0.216666,  0.932127, -0.287807, -0.952924,  0.095406,  0.671102, -0.730095,  0.128781,  0.374314, -0.629541,  0.680858,  0.756250, -0.398574, -0.518869, -0.832060, -0.513260,  0.210333, -0.580721,  0.782528,  0.224529,  0.394574,  0.885895,  0.243930,  0.574960, -0.334517,  0.746673,  0.052557, -0.793071,  0.606858,  0.048811, -0.963948,  0.261575,  0.834781, -0.191764,  0.516109,  0.552354,  0.392401,  0.735477,  0.393069, -0.841244,  0.371221,  0.302647, -0.187492,  0.934479, -0.580868, -0.782419,  0.224529,  0.831553, -0.525224, -0.180717, -0.812671, -0.275965,  0.513234,  0.098654, -0.487438,  0.867567, -0.596348, -0.588589,  0.545832,  0.015015,  0.988624,  0.149657, -0.506653,  0.657956,  0.557133,  0.891416, -0.406151,  0.201047, -0.200646, -0.295024,  0.934185, -0.000000, -0.000000,  1.000000};
static float ps49 [147] = {  0.357717, -0.137210,  0.923695,  0.838543, -0.337831,  0.427454, -0.317996, -0.267279,  0.909638, -0.529443,  0.828200, -0.183778, -0.689971,  0.530826,  0.492101, -0.279028, -0.942153,  0.185720, -0.046442,  0.913995,  0.403058, -0.630851, -0.109360,  0.768158, -0.276853, -0.599762,  0.750758,  0.304543,  0.223179,  0.925983, -0.854643,  0.161266,  0.493538,  0.610467,  0.705494,  0.360012, -0.621237,  0.254478,  0.741151,  0.890673, -0.410740, -0.194920,  0.350941,  0.914569,  0.201009,  0.041261, -0.355820,  0.933643,  0.826642,  0.551816,  0.110282, -0.000000,  0.000000,  1.000000, -0.407353,  0.781370,  0.472783, -0.091752, -0.865118,  0.493106,  0.075960, -0.659638,  0.747735,  0.629056,  0.147965,  0.763148,  0.844883,  0.217315, -0.488823,  0.932100,  0.329770, -0.149805,  0.399940, -0.488068,  0.775782,  0.542299,  0.497945,  0.676730,  0.805637,  0.380433,  0.454115,  0.284114,  0.791333,  0.541361,  0.970225, -0.205600,  0.128031,  0.316663, -0.797706,  0.513216,  0.602345,  0.796324, -0.055209,  0.660486, -0.730183, -0.174901, -0.746285, -0.582639,  0.321856,  0.164213, -0.959169,  0.230280,  0.213400,  0.548647,  0.808360, -0.091659,  0.711400,  0.696784,  0.806564, -0.575548,  0.134902,  0.621773, -0.608293,  0.493334,  0.653689, -0.226907,  0.721945, -0.346165,  0.108949,  0.931826,  0.969087,  0.169674,  0.179112,  0.982386, -0.028355, -0.184700,  0.866461,  0.018986,  0.498883,  0.345004, -0.929492, -0.130452, -0.388419,  0.518785,  0.761572, -0.599309, -0.459575,  0.655454,  0.021711,  0.997509,  0.067112, -0.459078, -0.763562,  0.454115, -0.083373,  0.366642,  0.926619};
static float ps50 [150] = {  0.292398,  0.632726,  0.717050, -0.843833,  0.220921, -0.489019,  0.785259,  0.598608, -0.158229, -0.246495,  0.274622, -0.929421, -0.000000,  0.000000, -1.000000,  0.686000,  0.466526,  0.558353,  0.126829,  0.336422,  0.933132,  0.535711, -0.811680,  0.232785, -0.360725, -0.732516,  0.577319, -0.874102,  0.387697,  0.292638,  0.444951, -0.885998, -0.130480, -0.591382,  0.275693, -0.757800, -0.583170,  0.586267, -0.562321,  0.039175,  0.686594, -0.725985,  0.572177, -0.134776, -0.808980, -0.937808,  0.343315, -0.051486,  0.615523, -0.680490, -0.397574,  0.721149,  0.068523,  0.689383,  0.954292,  0.298630,  0.012096, -0.797347, -0.562403, -0.218954,  0.571225,  0.817386,  0.074716, -0.791417,  0.531793, -0.301421,  0.077086, -0.611571, -0.787426, -0.348912, -0.109463,  0.930741,  0.274444, -0.824626,  0.494644, -0.480310, -0.345587, -0.806146,  0.488682,  0.830487, -0.267359,  0.851245, -0.055223, -0.521854,  0.266684,  0.931720,  0.246531,  0.888295,  0.223094,  0.401449,  0.192710,  0.976944, -0.091891,  0.292776, -0.817780, -0.495498,  0.097361, -0.974745, -0.200979, -0.420389, -0.449396,  0.788236,  0.670427,  0.545344, -0.503117, -0.681322, -0.212859,  0.700351, -0.067466, -0.844666, -0.531024,  0.743502, -0.666714, -0.051930,  0.078278,  0.905458, -0.417154,  0.090735,  0.383555, -0.919050, -0.886715, -0.285804,  0.363391, -0.500435, -0.744752, -0.441486,  0.983486, -0.030143, -0.178457, -0.194325,  0.968880, -0.153330, -0.294107,  0.572993, -0.764971, -0.978598,  0.035285, -0.202734,  0.701930, -0.383871, -0.599947,  0.249265, -0.278339, -0.927574, -0.417932,  0.543917,  0.727658,  0.396822,  0.032728,  0.917312};
static float ps51 [153] = { -0.182631,  0.665442,  0.723763,  0.947000, -0.276145, -0.164122, -0.466354,  0.469307,  0.749844, -0.888561,  0.047701,  0.456272,  0.986230, -0.074324,  0.147735,  0.981014,  0.115685, -0.155654,  0.497602,  0.749709,  0.436266,  0.636566,  0.628530, -0.446916, -0.685683,  0.199945,  0.699900, -0.398021,  0.116063,  0.910005,  0.121705, -0.686485,  0.716887,  0.511679,  0.462981, -0.723763,  0.168346,  0.525634, -0.833888,  0.471613,  0.513474,  0.716887, -0.302804,  0.930227, -0.207333,  0.871604, -0.467420,  0.147735,  0.281395,  0.226883,  0.932385, -0.379594, -0.879849,  0.285961,  0.861218,  0.093932,  0.499481, -0.281355, -0.754449,  0.592998, -0.668385, -0.735021,  0.114045, -0.131659,  0.335564,  0.932772,  0.000000, -0.000000,  1.000000,  0.669329, -0.195176,  0.716872,  0.496954, -0.856187, -0.141351,  0.396115, -0.765549,  0.506978, -0.774932,  0.437373,  0.456272,  0.615082,  0.170587,  0.769789, -0.645511, -0.145087,  0.749844, -0.168555,  0.979072,  0.114045,  0.107153,  0.988909, -0.102850, -0.745510, -0.432652, -0.506978,  0.140565,  0.533734,  0.833888,  0.168256,  0.787428,  0.592998,  0.821783,  0.313391, -0.475876,  0.441176,  0.891508,  0.102850, -0.291387, -0.212204,  0.932772,  0.765076, -0.624844, -0.155654, -0.016646,  0.899664, -0.436266,  0.936466,  0.293148,  0.192600,  0.152816,  0.945978,  0.285961,  0.632125, -0.750549,  0.192600, -0.524536,  0.705978,  0.475876,  0.384023, -0.111981,  0.916508, -0.879262, -0.454883,  0.141351, -0.198962,  0.872170,  0.446916, -0.675748,  0.542111, -0.499481,  0.115342, -0.342572,  0.932385,  0.867127, -0.252854,  0.429134,  0.755338,  0.621674,  0.207333,  0.426988, -0.474454,  0.769789};
static float ps52 [156] = { -0.380828,  0.125600,  0.916076,  0.775463,  0.466398,  0.425594,  0.051490,  0.997061,  0.056727, -0.838226,  0.490955,  0.237362,  0.181976,  0.759386,  0.624674,  0.330409, -0.227234,  0.916076,  0.000000, -0.000000,  1.000000, -0.378272, -0.762516,  0.524862, -0.080269,  0.336748,  0.938167,  0.233251,  0.470185,  0.851187, -0.398125,  0.749051,  0.529545, -0.218447, -0.938897,  0.265996,  0.676634, -0.600863,  0.425594,  0.316689,  0.139824,  0.938167,  0.554318, -0.434200,  0.710072,  0.247499, -0.597866,  0.762431,  0.762649, -0.643302,  0.067299, -0.501304,  0.851612, -0.153137,  0.444405,  0.895826,  0.000000, -0.615405, -0.741972,  0.265996,  0.590752,  0.297563,  0.749978,  0.837278,  0.136188,  0.529545,  0.616183, -0.057059,  0.785533, -0.121326, -0.637128,  0.761149,  0.494541,  0.604328,  0.624674,  0.257553,  0.911733,  0.320013, -0.973618,  0.218032,  0.067299,  0.593075, -0.769300, -0.237569, -0.418228,  0.456096,  0.785533,  0.382369, -0.780812,  0.494091, -0.887788,  0.175222,  0.425594,  0.014067, -0.364205,  0.931212, -0.625771, -0.164650,  0.762431, -0.433890, -0.482070,  0.761149, -0.298498, -0.209147,  0.931212,  0.866300,  0.447124, -0.222720,  0.762727,  0.644228,  0.056727,  0.570118,  0.756675,  0.320013, -0.809388,  0.230579, -0.540115, -0.168111,  0.960278, -0.222720, -0.981363, -0.116085,  0.153137,  0.898041, -0.370379,  0.237362,  0.022325, -0.848267,  0.529098, -0.120485,  0.650397,  0.749978,  0.971347, -0.006784,  0.237569,  0.933261,  0.329669,  0.142626, -0.673278,  0.504948,  0.540115, -0.688913, -0.495434,  0.529098, -0.097807,  0.899613,  0.425594, -0.681085,  0.178664,  0.710072, -0.853034, -0.167949,  0.494091, -0.302142,  0.942533,  0.142626};
static float ps53 [159] = { -0.348682,  0.599436,  0.720484,  0.000000, -0.000000,  1.000000,  0.372542, -0.268634,  0.888284, -0.057626, -0.977512, -0.202854,  0.036699, -0.338665,  0.940191, -0.244555, -0.662520,  0.707997, -0.331527,  0.825257,  0.457210, -0.802318, -0.451736,  0.390155, -0.969400,  0.241289, -0.045207,  0.012489,  0.855232,  0.518094,  0.217138, -0.955800,  0.198234, -0.789741,  0.253663,  0.558538,  0.866327, -0.360922,  0.345272, -0.108084, -0.984138,  0.140676, -0.801560,  0.593495, -0.072562,  0.635958,  0.001051,  0.771723, -0.343472, -0.864386,  0.367239,  0.770181,  0.271893,  0.576971, -0.981093, -0.085486, -0.173635, -0.608463,  0.765359,  0.209756, -0.602214, -0.353568,  0.715771, -0.640018, -0.018789,  0.768131, -0.954667,  0.121600,  0.271705, -0.632384,  0.578023,  0.515733,  0.159638,  0.366365,  0.916675,  0.538947, -0.831151,  0.136838,  0.337613, -0.805040,  0.487778, -0.405776, -0.913468, -0.030360, -0.191793,  0.309247,  0.931440,  0.333518,  0.066039,  0.940428, -0.847538,  0.468937,  0.248550,  0.497733,  0.365845,  0.786396, -0.545561, -0.640483,  0.540504,  0.620704,  0.596871,  0.508401,  0.649557, -0.633433,  0.420521, -0.968139, -0.201519,  0.148650, -0.013376, -0.865067,  0.501478,  0.096149, -0.632834,  0.768295, -0.340704, -0.029687,  0.939702,  0.637917,  0.745616, -0.192663,  0.878769, -0.040591,  0.475518,  0.312339,  0.660340,  0.682931, -0.297469, -0.367782,  0.881050, -0.023460,  0.627774,  0.778042,  0.286390, -0.947536, -0.141969,  0.871706,  0.401712,  0.280634, -0.521447,  0.307019,  0.796136,  0.685060, -0.322135,  0.653392,  0.666681,  0.721428,  0.187289,  0.363034,  0.850010,  0.381693,  0.427974, -0.555655,  0.712802,  0.858522,  0.511214, -0.040007, -0.851421, -0.128550,  0.508485};
static float ps54 [162] = {  0.709865,  0.241411,  0.661674, -0.748020,  0.028172,  0.663077, -0.376106, -0.617862,  0.690501, -0.091531, -0.099182,  0.990851, -0.110710, -0.925445, -0.362346, -0.234756,  0.562199,  0.792983, -0.050816, -0.471016,  0.880660,  0.724650,  0.516268,  0.456452, -0.185310,  0.246558,  0.951246, -0.527518,  0.305721,  0.792628,  0.817754,  0.554723, -0.153493,  0.175059,  0.123698,  0.976757, -0.064560, -0.949174,  0.308061, -0.344522, -0.323431,  0.881304, -0.651019, -0.595072,  0.471236,  0.578307, -0.764202,  0.285582,  0.937600, -0.056872, -0.343032, -0.898571, -0.401443, -0.177242,  0.239467, -0.579742,  0.778816, -0.454078, -0.002350,  0.890959,  0.578286, -0.814557, -0.045618, -0.060241, -0.763322,  0.643204, -0.363894, -0.930405, -0.043917,  0.288465, -0.946959,  0.141625, -0.546578,  0.576549,  0.607326,  0.782565, -0.115962,  0.611674,  0.544881, -0.337552,  0.767570, -0.959831, -0.244713,  0.137263,  0.438414,  0.814427,  0.380135,  0.943764, -0.311961, -0.109501, -0.631694, -0.310992,  0.710103,  0.685430,  0.710182,  0.160709,  0.096748,  0.471056,  0.876782,  0.513770, -0.003168,  0.857922, -0.575420, -0.800158,  0.169228, -0.233129,  0.948335,  0.215202, -0.821116,  0.559553, -0.112557, -0.796056,  0.339694,  0.500903,  0.910650,  0.133184,  0.391125,  0.240406, -0.239312,  0.940710,  0.498580, -0.639217,  0.585508,  0.997852, -0.001922,  0.065473, -0.490770,  0.788039,  0.371670,  0.137120,  0.736330,  0.662583, -0.044953, -0.998846, -0.016883, -0.354060, -0.840741,  0.409630, -0.455729, -0.606172, -0.651818, -0.770915,  0.583057,  0.256388,  0.236443, -0.847294,  0.475592,  0.929074, -0.239849,  0.281593, -0.187444,  0.808212,  0.558263, -0.845490, -0.281278,  0.453904,  0.758200, -0.456413,  0.465640,  0.421717,  0.337689,  0.841499};
static float ps55 [165] = { -0.015909,  0.448575,  0.893603,  0.666417, -0.339498,  0.663799,  0.875151, -0.316481,  0.365993, -0.469502, -0.807589,  0.356885,  0.300782,  0.265277,  0.916056, -0.641373,  0.018210,  0.767013,  0.612013,  0.309996,  0.727560, -0.877396,  0.407800,  0.252736, -0.971698,  0.000287,  0.236227,  0.401943,  0.914315, -0.049703, -0.621052, -0.590604, -0.515249, -0.597297, -0.309859,  0.739746,  0.698082, -0.595744,  0.397204, -0.977334, -0.201859, -0.063799, -0.733807, -0.522444,  0.434257,  0.976089, -0.215734,  0.026650, -0.903853, -0.381318,  0.194029, -0.014070, -0.277146,  0.960725, -0.345197, -0.116239,  0.931304,  0.552654, -0.029334,  0.832894, -0.058518,  0.914358,  0.400655,  0.436426, -0.811225,  0.389162,  0.219989, -0.959757,  0.174560,  0.337259, -0.926936, -0.164459,  0.422095, -0.616459,  0.664691,  0.080279, -0.594703,  0.799927, -0.841672, -0.180149,  0.509053, -0.832739,  0.549684, -0.066282, -0.660227,  0.563262,  0.496826,  0.130126, -0.838760,  0.528723,  0.020582,  0.998479,  0.051142, -0.600168,  0.347650,  0.720374,  0.243140, -0.058419,  0.968230, -0.131827, -0.947270,  0.292064,  0.950626,  0.005396,  0.310291, -0.340449,  0.247329,  0.907151,  0.355941, -0.358806,  0.862881,  0.273073,  0.926311,  0.259575,  0.314832,  0.781978,  0.537951, -0.841584,  0.185419,  0.507304, -0.324866,  0.578840,  0.747934, -0.183515, -0.769541,  0.611661, -0.838030, -0.520321, -0.164233, -0.053234,  0.104383,  0.993111,  0.837970,  0.322207,  0.440441,  0.653416, -0.725000, -0.217765,  0.596317,  0.769839,  0.227494, -0.377327,  0.788545,  0.485614,  0.324956,  0.554280,  0.766275, -0.479965, -0.608981,  0.631487,  0.709230,  0.695980, -0.112268,  0.794714, -0.003197,  0.606976, -0.012685,  0.730467,  0.682831, -0.261559, -0.465580,  0.845472, -0.577327,  0.811806, -0.087549};
static float ps56 [168] = { -0.520068, -0.353303,  0.777629, -0.166496, -0.886506,  0.431724, -0.315085,  0.221733,  0.922798, -0.003832,  0.352646,  0.935749, -0.846862,  0.528154,  0.062274, -0.356128, -0.657155,  0.664319,  0.816219,  0.411889,  0.405134, -0.950968, -0.027798,  0.308038, -0.182603, -0.441971,  0.878247, -0.980866,  0.194460, -0.009339, -0.006141, -0.706497,  0.707689,  0.044773,  0.974405,  0.220297, -0.576024,  0.003527,  0.817425,  0.308583,  0.544317,  0.780061,  0.507227, -0.348368,  0.788264, -0.452919,  0.885578, -0.103034, -0.684323, -0.696510, -0.215812,  0.664361, -0.524628,  0.532344,  0.160124, -0.885473,  0.436232, -0.713710,  0.670900, -0.201277,  0.375576,  0.887838,  0.265869, -0.443845,  0.738719,  0.507244, -0.795474,  0.145941,  0.588151,  0.000000, -0.000000,  1.000000,  0.130077, -0.985644,  0.107640, -0.011531,  0.650254,  0.759629,  0.322444, -0.627657,  0.708573,  0.950047,  0.009801,  0.311954, -0.781659, -0.204361,  0.589276, -0.622445,  0.761227,  0.181919,  0.482789,  0.793530, -0.370438,  0.201864,  0.974290, -0.100045, -0.284166, -0.136822,  0.948962,  0.806896,  0.135313,  0.574986, -0.885639,  0.364493, -0.287728, -0.889178, -0.361076,  0.281046, -0.897715,  0.311783,  0.311287,  0.587032,  0.007783,  0.809526, -0.745122, -0.646269,  0.164713, -0.125835,  0.853303,  0.506003,  0.132472, -0.398015,  0.907764,  0.507452,  0.860771, -0.039559, -0.318724,  0.553237,  0.769638,  0.316005,  0.226166,  0.921407, -0.707147,  0.543845,  0.451858,  0.221071,  0.809247,  0.544286, -0.308431,  0.926689,  0.214751, -0.889993, -0.447593, -0.087021,  0.546896,  0.653271,  0.523585,  0.594357,  0.364880,  0.716660, -0.463003,  0.775496, -0.429226, -0.988053, -0.152969, -0.018744,  0.784792, -0.207287,  0.584066, -0.671699, -0.534738,  0.512714,  0.298089, -0.120230,  0.946936, -0.584389,  0.369064,  0.722691};
static float ps57 [171] = {  0.665408,  0.458591,  0.589005,  0.378260,  0.547667,  0.746311,  0.772045, -0.254063,  0.582579, -0.208126, -0.742529,  0.636659,  0.138385, -0.798334,  0.586099,  0.750396,  0.123783,  0.649295, -0.901149,  0.066572,  0.428368, -0.462962,  0.211995,  0.860654,  0.271035, -0.920438,  0.281662,  0.105101, -0.577745,  0.809423, -0.210078,  0.748062,  0.629500,  0.193096,  0.952612,  0.235043,  0.880229,  0.474446, -0.009834, -0.439789,  0.807861,  0.392360, -0.933100,  0.279155, -0.226708, -0.741437,  0.537681, -0.401460, -0.376305,  0.925006,  0.052517, -0.883412,  0.389487,  0.260543, -0.661648, -0.742220,  0.106450,  0.158412,  0.321376,  0.933608,  0.875021,  0.335607,  0.348864,  0.502572,  0.848903,  0.163658,  0.664228, -0.736310, -0.129031, -0.685850,  0.583474,  0.434934, -0.029410, -0.248062,  0.968298,  0.243574, -0.007366,  0.969854, -0.353224, -0.111956,  0.928816,  0.138545,  0.828568,  0.542476,  0.945681,  0.218998, -0.240264, -0.795832, -0.272062,  0.540956,  0.488159,  0.228953,  0.842188,  0.040276, -0.998730,  0.030281, -0.201511,  0.424307,  0.882812,  0.919985, -0.004926,  0.391922,  0.591938, -0.779162,  0.206194, -0.449253, -0.811861,  0.372900, -0.683536, -0.011523,  0.729826, -0.309256, -0.945490,  0.102025,  0.534944,  0.594904, -0.599937,  0.450533, -0.722702,  0.524139, -0.239671, -0.481254,  0.843180, -0.844352,  0.533305, -0.051537,  0.985801,  0.142638,  0.088608, -0.987091,  0.137053,  0.082872,  0.733312,  0.626785,  0.263427, -0.088311, -0.928230,  0.361373,  0.765942,  0.541243, -0.346972,  0.054159,  0.615469,  0.786298,  0.297773, -0.335848,  0.893609,  0.560419, -0.119224,  0.819583, -0.105514,  0.104676,  0.988893,  0.463308,  0.743448,  0.482318, -0.134864,  0.937811,  0.319879, -0.730466,  0.300858,  0.613110,  0.522285, -0.476888,  0.706962, -0.470911,  0.537748,  0.699335, -0.543084, -0.326176,  0.773737};
static float ps58 [174] = {  0.820117, -0.014660,  0.572008,  0.075174,  0.322588,  0.943550,  0.712592, -0.597969,  0.366941,  0.990451, -0.120492, -0.066989, -0.244907,  0.909521,  0.335845,  0.206946, -0.961050,  0.183182,  0.166497,  0.923535,  0.345488, -0.287177, -0.184945,  0.939854, -0.869715, -0.474466,  0.135934, -0.253629, -0.521925,  0.814412, -0.536671,  0.019953,  0.843556, -0.775379, -0.397640,  0.490580,  0.152543, -0.853345,  0.498531, -0.658757, -0.558407, -0.504203,  0.433341,  0.475421,  0.765631, -0.050036,  0.809510,  0.584971,  0.966651, -0.034297,  0.253789, -0.111704, -0.959148,  0.259919,  0.132551,  0.609991,  0.781243, -0.264386,  0.202911,  0.942829, -0.413268, -0.856470,  0.309305, -0.851318,  0.479498,  0.212931,  0.523185,  0.802034,  0.288129,  0.000000,  0.000000,  1.000000,  0.376379,  0.178894,  0.909030,  0.350548, -0.444177,  0.824513, -0.186802, -0.794590,  0.577695, -0.393594,  0.708904,  0.585268,  0.869675,  0.291154,  0.398616,  0.854829, -0.305202,  0.419666, -0.025552,  0.995595,  0.090211, -0.591713,  0.749755,  0.296215, -0.946301, -0.157041,  0.282581,  0.590209, -0.092048,  0.801986,  0.639519, -0.392490,  0.661035,  0.017423, -0.355465,  0.934527, -0.793401, -0.095373,  0.601181, -0.482708,  0.407404,  0.775252, -0.429416,  0.901447,  0.054727,  0.675238,  0.661677, -0.325942,  0.323937,  0.945635,  0.028978, -0.969457, -0.236566, -0.064723,  0.293938, -0.141491,  0.945294,  0.348290,  0.749225,  0.563344,  0.079284, -0.649286,  0.756401, -0.908323,  0.404821, -0.105210,  0.670186,  0.253293,  0.697634, -0.690301,  0.517339,  0.505811, -0.491313, -0.626713,  0.604848,  0.805850,  0.560120,  0.192018, -0.634032, -0.773099,  0.017950, -0.907334,  0.186460,  0.376798, -0.726473,  0.221679,  0.650457,  0.431296, -0.678583,  0.594567, -0.188334,  0.543588,  0.817950, -0.716180,  0.697283, -0.029708, -0.488552,  0.823272, -0.289034, -0.558508, -0.323605,  0.763773};
static float ps59 [177] = {  0.338641,  0.463168,  0.819023, -0.912123, -0.337929,  0.232025, -0.170514,  0.950043,  0.261426,  0.148012,  0.935326,  0.321337,  0.344380,  0.105944,  0.932833, -0.367740,  0.617429,  0.695376,  0.603382,  0.246002,  0.758560, -0.949114,  0.026120,  0.313847,  0.268861, -0.239866,  0.932833,  0.238882,  0.745278,  0.622492,  0.073615,  0.308465,  0.948383,  0.267109, -0.756717,  0.596685, -0.858898,  0.512110,  0.006108, -0.061681, -0.311071,  0.948383,  0.795432,  0.007442,  0.605997,  0.114724, -0.562175,  0.819023,  0.065139, -0.927031,  0.369284, -0.255386, -0.911878,  0.321337, -0.007441,  0.604442,  0.796614, -0.819837, -0.198732,  0.537003,  0.719913, -0.338368,  0.605997, -0.994194, -0.107426,  0.006108, -0.903924, -0.405416, -0.136230,  0.652619, -0.745339,  0.136230, -0.804261,  0.500179, -0.320913,  0.112239, -0.992161,  0.054952,  0.791324,  0.364309,  0.491005,  0.445673,  0.815478,  0.369284, -0.381172, -0.241300,  0.892458, -0.245876,  0.378237,  0.892458, -0.591658, -0.407913,  0.695376,  0.558255,  0.576471,  0.596685,  0.921455, -0.201230,  0.332305, -0.688206,  0.687414,  0.232025, -0.797389,  0.174136,  0.577795,  0.551049,  0.792466, -0.261425,  0.715674,  0.641155,  0.277003, -0.728519, -0.539288,  0.422408,  0.445895, -0.475147,  0.758560, -0.324654,  0.070899,  0.943172,  0.939558,  0.119358,  0.320913,  0.564491, -0.123275,  0.816182, -0.436847, -0.702046,  0.562402, -0.629803, -0.043613,  0.775529,  0.567406, -0.661033,  0.491005, -0.000000, -0.000000,  1.000000,  0.104404, -0.820246, -0.562402,  0.213355,  0.976975,  0.000000, -0.554284,  0.302197,  0.775529,  0.383231, -0.881137,  0.277003, -0.662350,  0.522417,  0.537003,  0.437373, -0.793899, -0.422408, -0.258746, -0.546312,  0.796614,  0.774380,  0.628334, -0.074373, -0.873595,  0.371929,  0.313847, -0.976975,  0.213355,  0.000000, -0.093561, -0.777014,  0.622492,  0.441937, -0.893958, -0.074373,  0.515637,  0.855043,  0.054952};
static float ps60 [180] = {  0.654875,  0.355659,  0.666817,  0.271924,  0.933965,  0.231877, -0.118810,  0.110437,  0.986756, -0.326879,  0.866547,  0.377155, -0.394817, -0.108337,  0.912350,  0.906253, -0.349334,  0.238058,  0.566330, -0.470917,  0.676393, -0.558540,  0.665203,  0.495518,  0.648904,  0.611989,  0.452098, -0.435216,  0.497941,  0.750095, -0.060085,  0.972644,  0.224396,  0.290295, -0.619397,  0.729435, -0.008761, -0.548745,  0.835944,  0.098843,  0.637960,  0.763700, -0.162887, -0.917294,  0.363371,  0.095045, -0.821836,  0.561740, -0.214960,  0.721063,  0.658681,  0.741627,  0.039918,  0.669624, -0.847852, -0.200363,  0.490920, -0.943158,  0.304222,  0.133799, -0.215650, -0.732737,  0.645439, -0.060360, -0.223624,  0.972805,  0.814723,  0.561120,  0.146185,  0.265965, -0.335097,  0.903866,  0.441982, -0.764517,  0.469218, -0.566124,  0.807614, -0.165116,  0.716591, -0.584614,  0.380425,  0.331272, -0.943028, -0.030940,  0.465735,  0.168038,  0.868823, -0.480716, -0.789299,  0.381994,  0.961362,  0.253123,  0.108216,  0.920360, -0.003999,  0.391053,  0.393331,  0.915435, -0.085260,  0.852276,  0.324551,  0.410233,  0.571183,  0.799445,  0.186110,  0.535413, -0.167226,  0.827870, -0.442791,  0.203222,  0.873291, -0.143848,  0.431269,  0.890682,  0.893460,  0.414910, -0.171983, -0.795825,  0.511407,  0.324230,  0.399108,  0.523871,  0.752511,  0.806535, -0.590581,  0.026762, -0.691380, -0.710203,  0.132686, -0.973847, -0.080055,  0.212633, -0.052107, -0.996034,  0.072118, -0.696694,  0.034360,  0.716545, -0.626272, -0.286200,  0.725171,  0.994108, -0.072683,  0.080414, -0.738939, -0.519516,  0.429037, -0.702182,  0.353175,  0.618230, -0.505053, -0.581614,  0.637689,  0.371230,  0.770180,  0.518664,  0.050870,  0.858370,  0.510502,  0.170569,  0.325484,  0.930036, -0.312160, -0.422488,  0.850917, -0.887018,  0.152189,  0.435932, -0.605767,  0.776332,  0.174228,  0.215791, -0.937112,  0.274326,  0.210754, -0.013433,  0.977447,  0.787853, -0.274495,  0.551307};
static float ps61 [183] = {  0.711156, -0.521238,  0.471772,  0.825805,  0.103849,  0.554312,  0.236383,  0.809263,  0.537788,  0.667514, -0.712896,  0.214953, -0.024701,  0.933635,  0.357372, -0.529212, -0.780306,  0.333253, -0.768617,  0.156595,  0.620247,  0.811014, -0.221622,  0.541425,  0.256857, -0.585829,  0.768654, -0.783756, -0.168342,  0.597819,  0.922436,  0.260420,  0.285120,  0.155602, -0.836028,  0.526161, -0.063482, -0.691472,  0.719608, -0.702744, -0.507748,  0.498341,  0.470004,  0.590868,  0.655722, -0.864986,  0.376390,  0.331859, -0.379188,  0.892039, -0.245933,  0.348611,  0.065688,  0.934963,  0.046808, -0.967541,  0.248340, -0.199968, -0.865116,  0.459987, -0.647223,  0.681775,  0.341006,  0.036415, -0.055852,  0.997775,  0.255655, -0.964692, -0.063331, -0.197936,  0.495458,  0.845779,  0.555199,  0.740447,  0.378803,  0.615572, -0.082775,  0.783722, -0.313221, -0.938246,  0.146922,  0.580413,  0.813958,  0.024380,  0.547354, -0.412122,  0.728395,  0.336118,  0.380412,  0.861575, -0.356683,  0.857520,  0.370725, -0.556651, -0.007596,  0.830712,  0.739159,  0.451895,  0.499434,  0.612270,  0.256188,  0.747993, -0.672867,  0.462511,  0.577351,  0.025205, -0.387849,  0.921378, -0.780170, -0.595100, -0.192850,  0.959601, -0.078081,  0.270313,  0.118420,  0.607212,  0.785666, -0.996699, -0.074532,  0.032204, -0.280011, -0.170904,  0.944662, -0.921415, -0.388399,  0.011887, -0.416270,  0.650719,  0.635047,  0.885276, -0.407510,  0.224100,  0.813508, -0.578233, -0.062058, -0.937095,  0.043562,  0.346345, -0.967061,  0.249731,  0.049274, -0.423511, -0.662178,  0.618190, -0.557333, -0.340449,  0.757281, -0.326883, -0.917291, -0.227428,  0.552853, -0.830607, -0.066672, -0.274164, -0.475434,  0.835941,  0.043395,  0.282633,  0.958246, -0.113553,  0.771535,  0.625970, -0.766507, -0.616055,  0.181501, -0.256259,  0.154151,  0.954237,  0.458667, -0.715745,  0.526625,  0.066042,  0.996389,  0.053365, -0.491295,  0.326140,  0.807628,  0.324296, -0.258502,  0.909950, -0.900329, -0.294326,  0.320592};
static float ps62 [186] = {  0.613428, -0.404937,  0.678036,  0.663289, -0.066595,  0.745394, -0.146632,  0.988109, -0.046257,  0.775482, -0.507281,  0.375890,  0.851827,  0.434292, -0.292885,  0.659022, -0.598760, -0.455166,  0.046419, -0.938290,  0.342719, -0.418627, -0.805342,  0.419734, -0.175218,  0.772659,  0.610161, -0.145225, -0.355369, -0.923376, -0.703753,  0.331392,  0.628419,  0.988511,  0.063222,  0.137295, -0.862812,  0.409674,  0.296181, -0.134150, -0.810783,  0.569767,  0.121550,  0.653868,  0.746782, -0.256544, -0.379826,  0.888773,  0.653627,  0.523931,  0.546140,  0.407762, -0.210195,  0.888566, -0.159395,  0.511415,  0.844422, -0.016639, -0.606255,  0.795096,  0.692030, -0.705754, -0.151673,  0.212407, -0.779178,  0.589716, -0.114627, -0.110167,  0.987281,  0.109259, -0.337709,  0.934888,  0.441278, -0.895450, -0.058678, -0.812304, -0.217626,  0.541111, -0.563562, -0.306087,  0.767273,  0.134489,  0.984787,  0.110033, -0.158282,  0.944964,  0.286338, -0.238416, -0.951502,  0.194425,  0.782679,  0.621810,  0.027685,  0.831801, -0.205354,  0.515691,  0.523338, -0.681507,  0.511533, -0.502676, -0.864166,  0.023107,  0.329227, -0.536210,  0.777231,  0.376913,  0.884276,  0.275668, -0.452906,  0.286030,  0.844431, -0.884005,  0.111203,  0.454059,  0.390861,  0.725412,  0.566573, -0.964569, -0.119653,  0.235136, -0.977954,  0.180028,  0.105814,  0.449170,  0.153922,  0.880088,  0.662248,  0.704074, -0.256333, -0.684393,  0.021474,  0.728797, -0.436913,  0.580414,  0.687188,  0.704373,  0.242667,  0.667062,  0.643867,  0.709475,  0.286496, -0.370502, -0.611014,  0.699564, -0.418336, -0.040287,  0.907399,  0.649737, -0.738626,  0.179650, -0.151495,  0.208735,  0.966167,  0.374461, -0.881523,  0.287570, -0.102890, -0.876525, -0.470232, -0.647456, -0.532607,  0.545097,  0.420003,  0.463570,  0.780192, -0.430112,  0.813479,  0.391478,  0.176156,  0.023979,  0.984070,  0.944071, -0.243921,  0.221884, -0.890639, -0.102158, -0.443086,  0.865679, -0.499390,  0.034792, -0.938963, -0.343729, -0.014139,  0.856123,  0.403896,  0.322367};
static float ps63 [189] = {  0.557048,  0.254557,  0.790505,  0.501828, -0.856777,  0.118753,  0.366133, -0.832181,  0.416439,  0.458902, -0.342363,  0.819876,  0.876786, -0.393604,  0.276264,  0.684179, -0.418767,  0.597104, -0.504411, -0.622034,  0.598868,  0.763420,  0.588533,  0.266118,  0.378565,  0.925263,  0.024027, -0.973098, -0.005409,  0.230330,  0.036207,  0.539129,  0.841445, -0.153898,  0.260962,  0.953003,  0.459908,  0.699647,  0.546790,  0.540504,  0.799090,  0.263271, -0.893126,  0.264375,  0.363912, -0.734197,  0.548949,  0.399512,  0.189445, -0.236455,  0.952995,  0.082573, -0.753309,  0.652463, -0.094532,  0.914517,  0.393347,  0.404328, -0.625886,  0.666922, -0.893846, -0.316945,  0.317153, -0.629940,  0.756256,  0.176786, -0.333618,  0.925140,  0.181148, -0.445620,  0.770837,  0.455230,  0.293398, -0.505611, -0.811341, -0.729767,  0.241250,  0.639717,  0.060323, -0.918212,  0.391468, -0.210101, -0.597555,  0.773812, -0.552993,  0.522508,  0.648987,  0.868961,  0.493684, -0.034392,  0.639407, -0.655156,  0.402404, -0.481084,  0.225299,  0.847230,  0.126826, -0.519656,  0.844910, -0.625969, -0.073665,  0.776361, -0.479368, -0.821721,  0.308191, -0.181997, -0.960933,  0.208531,  0.802857,  0.156738,  0.575198,  0.237577,  0.906435,  0.349188,  0.980397,  0.196412,  0.015613,  0.144655,  0.763767,  0.629074, -0.724790, -0.595435,  0.346607, -0.844126, -0.054426,  0.533376, -0.186549,  0.743670,  0.641993,  0.845555, -0.166505,  0.507261,  0.648531, -0.082191,  0.756737, -0.144061, -0.319350,  0.936623, -0.237930, -0.814969,  0.528408,  0.754640, -0.646528,  0.111892, -0.333825, -0.053261,  0.941129,  0.383661,  0.012884,  0.923384,  0.897479,  0.313172,  0.310570, -0.196705,  0.974233, -0.110349, -0.454846, -0.360333,  0.814417,  0.200471,  0.270688,  0.941562,  0.680149,  0.467463,  0.564691,  0.979258, -0.201661,  0.019660,  0.364240,  0.509525,  0.779560, -0.875779,  0.470288,  0.108808, -0.664927, -0.745382,  0.047733, -0.713694, -0.359820,  0.600975, -0.000000, -0.000000,  1.000000,  0.962848, -0.010571,  0.269836,  0.039092,  0.994194,  0.100253};
static float ps64 [192] = {  0.797023,  0.136705,  0.588273, -0.278590,  0.948576,  0.150306,  0.269452, -0.711937,  0.648492, -0.645004, -0.682074,  0.344595, -0.997625,  0.026724, -0.063488, -0.015834,  0.014472,  0.999770,  0.839210,  0.379789, -0.389213,  0.208215, -0.891081,  0.403263,  0.539573, -0.252383,  0.803221, -0.132838, -0.935923,  0.326194,  0.827364, -0.548564, -0.120610,  0.307510, -0.090828,  0.947200,  0.218118,  0.504810,  0.835219,  0.726209, -0.666170,  0.169821, -0.332668, -0.557425,  0.760663, -0.328467,  0.034648,  0.943880,  0.291451,  0.863987,  0.410588, -0.524955,  0.742151, -0.416694, -0.122014,  0.589755,  0.798312,  0.949992,  0.086143, -0.300156, -0.055576,  0.316879,  0.946836, -0.141862,  0.824696,  0.547496,  0.601245, -0.789791, -0.121382,  0.734088,  0.444365,  0.513473,  0.661510,  0.749911,  0.006173, -0.611822, -0.502190,  0.611129, -0.830952,  0.381428,  0.405009, -0.370819,  0.344175,  0.862576, -0.762479, -0.186326,  0.619604,  0.065038, -0.295413,  0.953153,  0.926285, -0.117378,  0.358076, -0.063469, -0.795508,  0.602610,  0.768559, -0.194100,  0.609624,  0.446139, -0.883423,  0.143262, -0.834875, -0.510044, -0.206975, -0.423944, -0.882753,  0.202533, -0.522214, -0.248724,  0.815738, -0.378698, -0.766359,  0.518923, -0.637493,  0.376534,  0.672179, -0.959665,  0.216672,  0.179153,  0.304862, -0.464001,  0.831722, -0.239236, -0.285484,  0.928044,  0.532848,  0.370235,  0.760920, -0.407192,  0.822667,  0.396753, -0.835774,  0.104423,  0.539052,  0.927325,  0.216334,  0.305397,  0.923215, -0.362997,  0.126124, -0.125640,  0.985406, -0.114845, -0.419397, -0.897620, -0.135587,  0.456964,  0.642255,  0.615380,  0.611904,  0.712455,  0.343485,  0.580539,  0.068549,  0.811342, -0.154722, -0.987867, -0.013398, -0.651714,  0.630831,  0.421095,  0.276048,  0.217697,  0.936165, -0.837437, -0.532713,  0.122130, -0.413285,  0.619494,  0.667400,  0.576347, -0.522314,  0.628500,  0.792823, -0.455369,  0.405057, -0.607712,  0.077141,  0.790402,  0.963101,  0.267540, -0.029315,  0.006778,  0.955284,  0.295611, -0.020026, -0.576391,  0.816928,  0.137929,  0.746604,  0.650813};
static float ps65 [195] = {  0.062708, -0.858600,  0.508797,  0.584679, -0.677329, -0.446514,  0.223552, -0.677808,  0.700429, -0.797685, -0.192161, -0.571640, -0.833387,  0.549920, -0.055262, -0.110259, -0.990666, -0.080144, -0.440792, -0.643039,  0.626261, -0.668866,  0.731483,  0.132480, -0.295096,  0.303587,  0.905954, -0.418542, -0.895656, -0.150407,  0.582057,  0.146941,  0.799761,  0.647646,  0.760598, -0.045227, -0.142844,  0.594825,  0.791062,  0.151006, -0.177234,  0.972515, -0.005207,  0.469083, -0.883138,  0.326648, -0.424895,  0.844254,  0.794224, -0.398328,  0.458850,  0.201001,  0.633134,  0.747489, -0.478759,  0.146635, -0.865614, -0.863722, -0.251902,  0.436497, -0.313970,  0.888937, -0.333487, -0.105369,  0.945447,  0.308265,  0.911225, -0.055942,  0.408093, -0.762107, -0.036203,  0.646438, -0.069980,  0.088294,  0.993633, -0.955156,  0.269870,  0.121848,  0.364629,  0.391243,  0.844970, -0.261306, -0.854740,  0.448484,  0.044670,  0.377010,  0.925131,  0.733289, -0.136689,  0.666036,  0.005037,  0.808941,  0.587869, -0.357268, -0.920554,  0.157927, -0.967919, -0.030735,  0.249375,  0.835161,  0.542783,  0.088843, -0.129061, -0.689712,  0.712489,  0.270791,  0.111467,  0.956163, -0.342540, -0.435450,  0.832496, -0.581814, -0.245630,  0.775344, -0.934059, -0.250560, -0.254468,  0.949492,  0.302087, -0.084904,  0.433742,  0.712721,  0.551268,  0.306000, -0.781460, -0.543769,  0.623541,  0.721745,  0.300468,  0.872864, -0.180568, -0.453324, -0.522232,  0.845263, -0.113160, -0.024672, -0.973867,  0.225775,  0.805004,  0.534289, -0.257882,  0.480688, -0.700513,  0.527467,  0.778134,  0.477959,  0.407508, -0.410622, -0.015340,  0.911677, -0.223380,  0.974566, -0.017964, -0.446061,  0.525145,  0.724743,  0.228546,  0.890852,  0.392619,  0.595995,  0.464027,  0.655338,  0.682614,  0.462275, -0.565986, -0.591730,  0.195033,  0.782188,  0.996991,  0.000177,  0.077517,  0.585962, -0.442002,  0.679178, -0.176851, -0.209334,  0.961719, -0.573411, -0.738938,  0.353794, -0.407794,  0.883175,  0.231746, -0.820888,  0.490498,  0.292497,  0.932807, -0.297778,  0.202977, -0.703978,  0.402970,  0.584833,  0.688012, -0.657035,  0.308131};
static float ps66 [198] = { -0.521329, -0.419036,  0.743387, -0.269227, -0.644814,  0.715355, -0.997659,  0.020268,  0.065314,  0.168269, -0.024156,  0.985445, -0.796049, -0.537148,  0.278889,  0.841662,  0.479152,  0.249036,  0.913646, -0.030944,  0.405332, -0.572064,  0.768372, -0.286963,  0.742061, -0.040706,  0.669096,  0.600592, -0.589687,  0.539962, -0.808281,  0.527345, -0.261894, -0.580852, -0.776337,  0.244770, -0.790538,  0.523367,  0.318019, -0.779942, -0.353154,  0.516694,  0.968496,  0.174376,  0.177785,  0.318739, -0.779021,  0.539936, -0.303277, -0.827253,  0.472944,  0.037843, -0.634374,  0.772099,  0.742627, -0.351341, -0.570144, -0.177771,  0.528871,  0.829875,  0.427570,  0.496317,  0.755549, -0.011512,  0.265032,  0.964171, -0.569251, -0.632513,  0.525243,  0.747034,  0.664545,  0.017896, -0.141830, -0.986789, -0.078302,  0.008223, -0.842871,  0.538053,  0.485149,  0.871816,  0.067582,  0.833340,  0.260442,  0.487559, -0.606835,  0.157875,  0.778991, -0.468262,  0.449670,  0.760610,  0.732211, -0.681072,  0.002805,  0.012707,  0.969010, -0.246693,  0.296985,  0.893786,  0.336076, -0.589146,  0.763637,  0.264133, -0.935940, -0.170165,  0.308320, -0.315512,  0.212198,  0.924891,  0.128832,  0.541673,  0.830658, -0.132611, -0.065210,  0.989021,  0.929384, -0.213945, -0.300788,  0.341311, -0.560699,  0.754403,  0.625323,  0.251953,  0.738574, -0.318457,  0.887049,  0.334260,  0.661159,  0.526278,  0.534696, -0.680925, -0.145050,  0.717845,  0.283771, -0.920780,  0.267655, -0.461396,  0.887194,  0.000685,  0.599663, -0.327682,  0.730089, -0.166145,  0.984635,  0.053762,  0.604293,  0.730771,  0.317496,  0.338320, -0.284010,  0.897150, -0.426997, -0.105397,  0.898090, -0.255969,  0.747019,  0.613549, -0.910636,  0.410196,  0.049805,  0.368493,  0.730404,  0.575085, -0.958369,  0.223458, -0.177752, -0.310263, -0.932279,  0.185992,  0.048358, -0.358445,  0.932298,  0.308507,  0.255294,  0.916323, -0.018286,  0.928071,  0.371955, -0.246306, -0.379783,  0.891683, -0.546466,  0.639696,  0.540522,  0.928879,  0.365676, -0.058868,  0.057272,  0.772644,  0.632251,  0.809762, -0.325116,  0.488451,  0.491885, -0.010475,  0.870597, -0.844861,  0.035702,  0.533794};
static float ps67 [201] = {  0.317637,  0.442274,  0.838749, -0.849332, -0.085926,  0.520818,  0.172297,  0.187230,  0.967088, -0.812030,  0.237130,  0.533269, -0.036417, -0.992662,  0.115304,  0.937603,  0.159011,  0.309220,  0.847487, -0.101735,  0.520976, -0.423868, -0.129491,  0.896420, -0.281093, -0.762496,  0.582741, -0.760362, -0.421726,  0.493960, -0.562359, -0.660399,  0.497620,  0.059769,  0.974510,  0.216237,  0.532808, -0.796335,  0.286299, -0.182366, -0.294414,  0.938117,  0.360629,  0.694556,  0.622526,  0.051750, -0.727669,  0.683974,  0.527281, -0.849672, -0.005697, -0.212257,  0.894780,  0.392830, -0.323703, -0.895439,  0.305621, -0.648645,  0.075837,  0.757304, -0.020148, -0.904853,  0.425246, -0.807626, -0.548734,  0.215942, -0.160576,  0.736564,  0.657030,  0.608971,  0.691374,  0.388789,  0.296908, -0.810142,  0.505485,  0.608542, -0.611151,  0.506132, -0.303007,  0.498676,  0.812101, -0.654211, -0.239455,  0.717405,  0.241860, -0.965874, -0.092686,  0.252992, -0.942078,  0.220191,  0.960217, -0.151645,  0.234494,  0.938324,  0.342940,  0.044055,  0.749387, -0.648263, -0.134811, -0.133208,  0.037626,  0.990374,  0.110554, -0.162880,  0.980433, -0.887420,  0.383659,  0.255523,  0.570765,  0.815779,  0.093449,  0.768473, -0.616979,  0.169664,  0.369721, -0.590993,  0.716962, -0.930385,  0.365088, -0.033082, -0.189909, -0.565571,  0.802536, -0.453758,  0.675898,  0.580746, -0.587100,  0.391177,  0.708727,  0.351245,  0.875159,  0.332751,  0.830102, -0.403780,  0.384569,  0.536545,  0.236778,  0.809972, -0.594974, -0.775441,  0.211419, -0.065176,  0.360860,  0.930340,  0.112134, -0.465838,  0.877736, -0.999576, -0.028807,  0.004191, -0.315475, -0.948922, -0.004622,  0.100032,  0.848008,  0.520458,  0.650611, -0.059055,  0.757111,  0.385669, -0.017099,  0.922479,  0.656407, -0.355616,  0.665332, -0.469371, -0.473257,  0.745465, -0.506338,  0.806431,  0.305435,  0.604214,  0.487827,  0.630040,  0.077453,  0.610754,  0.788023,  0.787944,  0.608906,  0.091532,  0.779666,  0.211016,  0.589571, -0.704219,  0.552831,  0.445481, -0.384157,  0.199442,  0.901469, -0.932572, -0.249715,  0.260678,  0.816668,  0.447074,  0.364937,  0.404433, -0.316817,  0.857940, -0.959215,  0.088990,  0.268305};
static float ps68 [204] = { -0.085763, -0.964111,  0.251266,  0.938348,  0.244179, -0.244702,  0.015517,  0.939415,  0.342430,  0.199039,  0.975076,  0.098036, -0.590522,  0.786338,  0.181540,  0.321964, -0.452468,  0.831632, -0.933574, -0.355526, -0.045173,  0.045990, -0.611951,  0.789557, -0.820674,  0.552150,  0.147056, -0.164075,  0.809871,  0.563195,  0.653538,  0.494670,  0.572878,  0.409726,  0.662238,  0.627348,  0.294572,  0.168039,  0.940739, -0.015767, -0.831829,  0.554808, -0.768714,  0.142691,  0.623473,  0.227683,  0.501252,  0.834809,  0.407978, -0.910495,  0.067478,  0.005374,  0.308107,  0.951336,  0.680297, -0.726162,  0.099423, -0.744245, -0.411600,  0.526009,  0.961121, -0.268478, -0.064549,  0.590157,  0.043909,  0.806093,  0.874945, -0.455584,  0.164055, -0.372325, -0.915384,  0.153120,  0.282219, -0.734125,  0.617586,  0.538673, -0.552163,  0.636355, -0.681672,  0.579772,  0.446304, -0.500118, -0.441654,  0.744865, -0.203562, -0.425113,  0.881953, -0.385692,  0.603716,  0.697688,  0.046522, -0.026691,  0.998561, -0.632651, -0.161122,  0.757491,  0.499006,  0.346401,  0.794355, -0.622833, -0.753101,  0.211937,  0.474217, -0.798668,  0.370469, -0.617152, -0.704248, -0.350940, -0.315320,  0.364049,  0.876380, -0.322145, -0.835813,  0.444567,  0.771345,  0.223650,  0.595824, -0.348680, -0.177220,  0.920334, -0.283111,  0.929385,  0.236836, -0.563997, -0.654477,  0.503555, -0.518349, -0.848582, -0.105937, -0.949384,  0.164767, -0.267437, -0.855276,  0.338669,  0.392181,  0.129119,  0.768832,  0.626279, -0.842388, -0.129177,  0.523159,  0.603799, -0.265127,  0.751754, -0.269839, -0.666224,  0.695221, -0.817205, -0.523754,  0.240536,  0.203765, -0.922101,  0.328953, -0.610245,  0.393163,  0.687767, -0.446865,  0.773167,  0.450027,  0.818317, -0.064306,  0.571159, -0.952002,  0.065113,  0.299086,  0.100906, -0.994893,  0.002310, -0.225876,  0.080369,  0.970835, -0.078474,  0.588641,  0.804577,  0.701967, -0.602378,  0.379978,  0.796508, -0.346349,  0.495598, -0.768169, -0.637231, -0.062079, -0.835166, -0.445522, -0.322502, -0.524787,  0.117184,  0.843129,  0.935737,  0.144377,  0.321795,  0.352995, -0.146893,  0.924022,  0.059074, -0.311021,  0.948565,  0.329624,  0.864448,  0.379576, -0.998993, -0.040829, -0.018605};
static float ps69 [207] = {  0.420628, -0.420531,  0.803881, -0.955900, -0.252851,  0.149406,  0.445538,  0.386313,  0.807625, -0.062686, -0.284350,  0.956669,  0.797773, -0.154844,  0.582736,  0.410010,  0.647265,  0.642605, -0.125944,  0.909929,  0.395181, -0.940963,  0.014955,  0.338179,  0.838786,  0.435577,  0.326668,  0.072243,  0.016055,  0.997258,  0.756281, -0.642462,  0.123619, -0.962368, -0.205928, -0.177315, -0.820002,  0.543571,  0.179242,  0.181352,  0.953236,  0.241772,  0.823745, -0.424932,  0.375336, -0.624278,  0.763645,  0.164690,  0.607192, -0.684901,  0.402776, -0.420986,  0.794663,  0.437357,  0.685692,  0.391997,  0.613322, -0.662210,  0.592995,  0.458078,  0.153836,  0.825716,  0.542704, -0.640720, -0.751820,  0.155708, -0.780969,  0.044197,  0.623005, -0.924932,  0.365390, -0.104839,  0.256451, -0.209806,  0.943512, -0.999426,  0.033783, -0.002508,  0.140922,  0.609548,  0.780123,  0.427064,  0.831815,  0.354540, -0.640135, -0.230680,  0.732813, -0.642727,  0.336852,  0.688064,  0.427344,  0.903572,  0.030584,  0.569900, -0.156051,  0.806760, -0.538308, -0.715705,  0.444961, -0.687274, -0.473515,  0.550852,  0.138662,  0.985251, -0.100271, -0.830188,  0.321164,  0.455676, -0.440392, -0.549435,  0.710054,  0.312408, -0.852421,  0.419260, -0.852919, -0.238280,  0.464492, -0.432723,  0.587829,  0.683527, -0.349580, -0.890891,  0.290012, -0.373471,  0.916881,  0.140885, -0.265879, -0.027223,  0.963622, -0.120879,  0.243468,  0.962347, -0.084791,  0.990814,  0.105346,  0.863643,  0.143914,  0.483124,  0.652587, -0.442915,  0.614781, -0.021002, -0.920479,  0.390228, -0.241028, -0.779617,  0.578016, -0.809956, -0.519053,  0.273046,  0.661961,  0.120415,  0.739802,  0.383598, -0.667796,  0.637887, -0.373924, -0.298775,  0.878017,  0.136056, -0.518754,  0.844028,  0.209411, -0.963582,  0.166303, -0.150185,  0.748629,  0.645754,  0.169462,  0.336732,  0.926226, -0.945715,  0.271061,  0.179300,  0.073097, -0.760690,  0.644986,  0.944898, -0.112964,  0.307257,  0.675529,  0.722847,  0.145442, -0.543015,  0.037762,  0.838874,  0.637760,  0.634245,  0.437031,  0.385752,  0.090555,  0.918148,  0.511837, -0.846823,  0.144616,  0.862678,  0.505056,  0.026565, -0.157444, -0.564627,  0.810190, -0.148503,  0.520199,  0.841035, -0.399512,  0.310350,  0.862597};
static float ps70 [210] = { -0.632083, -0.100805,  0.768316, -0.326319,  0.315787,  0.890952, -0.164525,  0.595720,  0.786161, -0.794836,  0.604268, -0.055646, -0.949016, -0.310929,  0.051889,  0.332731,  0.791680,  0.512379,  0.214754,  0.145336,  0.965794, -0.652071, -0.449212,  0.610747, -0.820826,  0.102655,  0.561878, -0.399404,  0.006705,  0.916751, -0.844610, -0.413921,  0.339563,  0.852303,  0.436193,  0.288643,  0.220917, -0.945682,  0.238498, -0.106029,  0.075760,  0.991473, -0.445136, -0.662910,  0.602000,  0.524653,  0.256073,  0.811890, -0.954620,  0.128510, -0.268675, -0.744274,  0.395867,  0.537908,  0.470998,  0.852656,  0.226138,  0.469785,  0.558145,  0.683941, -0.168574, -0.797514,  0.579271,  0.649231, -0.497422,  0.575388, -0.023370,  0.371988,  0.927943, -0.604009,  0.653568,  0.456095, -0.098207, -0.949890,  0.296756, -0.854094,  0.414387, -0.314337, -0.454483, -0.349542,  0.819308, -0.409544,  0.852290,  0.325385,  0.140685,  0.656402,  0.741177, -0.927526,  0.227126,  0.296833,  0.326536, -0.370787,  0.869420,  0.717048,  0.676585,  0.167558, -0.813972, -0.575288,  0.080573,  0.089537, -0.663921,  0.742423, -0.304166, -0.951807,  0.039325, -0.818931,  0.516109,  0.250964, -0.117196,  0.945919,  0.302494,  0.522428, -0.841223,  0.139331,  0.185582,  0.944552,  0.270890, -0.941888,  0.335890, -0.005052, -0.229541, -0.559632,  0.796318, -0.390544, -0.853893,  0.344009, -0.631330,  0.762385,  0.142099, -0.204785, -0.232598,  0.950769, -0.723491, -0.370106, -0.582738,  0.623317,  0.642327,  0.445973, -0.602419,  0.222504,  0.766540, -0.314794,  0.947708,  0.052484,  0.814766, -0.202553,  0.543258,  0.000187,  0.999997,  0.002594,  0.670239, -0.664066,  0.331355,  0.591760, -0.273477,  0.758308, -0.577294, -0.811804,  0.087789,  0.271795,  0.421008,  0.865378,  0.695603,  0.033665,  0.717637, -0.478670,  0.518876,  0.708268, -0.294052,  0.759162,  0.580695,  0.434338, -0.051156,  0.899296, -0.809663, -0.198506,  0.552306,  0.420325, -0.786484,  0.452514,  0.653847,  0.666597, -0.357956,  0.998864, -0.032942, -0.034444,  0.123168, -0.146201,  0.981557, -0.867207, -0.121609, -0.482870,  0.129424, -0.848570,  0.513010,  0.025912, -0.426779,  0.903984,  0.962944,  0.174971,  0.205243, -0.949079, -0.095741,  0.300138,  0.010281,  0.835840,  0.548877,  0.386863, -0.607474,  0.693767};
static float ps71 [213] = { -0.237232, -0.434598,  0.868819,  0.925021, -0.046467,  0.377062, -0.623051, -0.748151,  0.228205, -0.763643, -0.644402, -0.039950, -0.627166, -0.589647,  0.508900,  0.626948,  0.696147,  0.349736, -0.377969, -0.808529,  0.451022,  0.413352, -0.436620,  0.799064, -0.635981, -0.171910, -0.752313, -0.195374,  0.078648,  0.977570,  0.172595,  0.798800,  0.576306, -0.836589,  0.003710,  0.547819,  0.490691, -0.662234,  0.566276, -0.706667,  0.611619, -0.355730,  0.268016, -0.198439,  0.942756, -0.367488, -0.912757,  0.178402,  0.494805, -0.820738,  0.285581, -0.956870,  0.075733,  0.280470,  0.423657,  0.414067,  0.805645, -0.972735,  0.195551, -0.124685, -0.654653,  0.547333,  0.521398,  0.367707,  0.863828,  0.344373,  0.997433,  0.067619, -0.023562,  0.852443, -0.343659,  0.394005,  0.388250,  0.118823,  0.913862, -0.839948,  0.324036,  0.435303, -0.286392,  0.906302,  0.310800, -0.154186,  0.616159,  0.772382, -0.063527, -0.206550,  0.976372,  0.825931,  0.215863,  0.520807, -0.812209, -0.512991,  0.277771, -0.409927,  0.716178,  0.564844,  0.226566, -0.951331,  0.208893, -0.177766,  0.364900,  0.913918, -0.681855,  0.257483,  0.684672, -0.856609,  0.504149, -0.109793,  0.103191,  0.059261,  0.992895, -0.386083, -0.155965,  0.909183,  0.222553, -0.847523,  0.481845, -0.925340, -0.228220,  0.302757, -0.665478,  0.401600, -0.629171, -0.460813,  0.173957,  0.870282,  0.103300, -0.992530, -0.064906,  0.081495,  0.868377, -0.489163,  0.129235,  0.344965,  0.929676,  0.439136,  0.661044,  0.608424, -0.223298, -0.971323, -0.081669, -0.674205,  0.737768, -0.033851,  0.219195, -0.667398,  0.711711, -0.772370, -0.308235,  0.555371,  0.152815,  0.596224,  0.788140,  0.547974, -0.145367,  0.823767,  0.769829, -0.100602,  0.630272,  0.374860,  0.628824, -0.681220,  0.074174,  0.974348, -0.212473, -0.564281,  0.773499,  0.288594,  0.044396,  0.943433,  0.328577, -0.921173, -0.388710,  0.018544, -0.425498,  0.904480,  0.029470,  0.089366, -0.455806,  0.885582, -0.545513, -0.374357,  0.749848, -0.533367, -0.842216, -0.078686, -0.446624,  0.470578,  0.760976, -0.081657, -0.686126,  0.722885, -0.643255, -0.068015,  0.762625, -0.125970,  0.818188,  0.560982, -0.782325,  0.581925,  0.222106,  0.663183,  0.461760,  0.589038,  0.949374,  0.213024,  0.230889, -0.932518,  0.335223,  0.134297,  0.824093,  0.477613,  0.304559};
static float ps72 [216] = { -0.115683, -0.463514,  0.878506,  0.871651, -0.266696,  0.411216,  0.005362, -0.687842,  0.725840,  0.799129,  0.584576, -0.140228, -0.649736, -0.758601, -0.048660,  0.431880, -0.896164,  0.101836, -0.000000, -0.000000,  1.000000, -0.913278,  0.111970,  0.391646,  0.107896,  0.746541,  0.656533,  0.234467, -0.521836,  0.820190, -0.686573, -0.303320,  0.660768,  0.335792, -0.919666, -0.203613, -0.674853,  0.620353,  0.399669, -0.346446, -0.836199,  0.425142,  0.725959,  0.224216,  0.650162, -0.132922,  0.858998,  0.494423, -0.878673,  0.444506, -0.174208,  0.425553, -0.293043,  0.856172, -0.934986, -0.319437,  0.154148, -0.049458, -0.867813,  0.494423,  0.823822,  0.388803,  0.412492, -0.857950,  0.396556,  0.326596,  0.192224,  0.894635,  0.403333,  0.261320, -0.766661,  0.586467,  0.446831,  0.862338,  0.238151,  0.784505, -0.070008,  0.616158, -0.127314, -0.966738,  0.221829,  0.991593,  0.025836, -0.126790, -0.044679,  0.971523,  0.232696,  0.302897, -0.006729,  0.953000,  0.415382,  0.732569,  0.539260,  0.912175,  0.090560,  0.399669, -0.529859,  0.536810,  0.656570,  0.598362,  0.502728,  0.623880, -0.426697,  0.777461,  0.462043, -0.685548,  0.710807, -0.157408, -0.810119,  0.579164,  0.090973, -0.208560,  0.921158, -0.328590,  0.596191, -0.788724, -0.149903, -0.529766, -0.112242,  0.840684,  0.664924, -0.337324,  0.666400, -0.757805,  0.007565,  0.652437,  0.484887,  0.273006,  0.830875, -0.603669, -0.735601,  0.307368, -0.885309, -0.180781,  0.428423, -0.287977, -0.673464,  0.680820,  0.497403, -0.775087,  0.389655,  0.321098,  0.541650,  0.776860,  0.115288, -0.273553,  0.954923, -0.232611, -0.201155,  0.951540, -0.958779,  0.274375,  0.073902, -0.779535, -0.466871,  0.417560,  0.491042, -0.576940,  0.652701, -0.562242,  0.212891,  0.799100,  0.966253,  0.219906,  0.134151,  0.717726, -0.547846,  0.429807, -0.427449, -0.896692,  0.115022,  0.207746,  0.975387,  0.073902,  0.664070,  0.658827,  0.353494,  0.571507, -0.025829,  0.820190,  0.207840,  0.286397,  0.935296, -0.555678, -0.594828,  0.580863, -0.740983,  0.321920,  0.589332,  0.850789,  0.506061,  0.141638, -0.322145,  0.090482,  0.942356, -0.088550,  0.281301,  0.955525,  0.006787,  0.538938,  0.842318, -0.424248, -0.411413,  0.806693, -0.235907,  0.670057,  0.703826, -0.979564,  0.103765, -0.172299, -0.331316,  0.416163,  0.846781, -0.129065,  0.990294, -0.051572};
static float ps73 [219] = { -0.994967,  0.012813,  0.099380,  0.071611,  0.284941, -0.955867, -0.652946, -0.725634,  0.217063, -0.642952,  0.765884,  0.005760, -0.958319, -0.282800, -0.040593, -0.214846,  0.874605, -0.434634, -0.393449, -0.573014,  0.718925, -0.000000,  0.000000, -1.000000, -0.847351, -0.469907, -0.247354,  0.463081, -0.359894,  0.809958, -0.850617,  0.520060, -0.077380,  0.456275,  0.888855, -0.041838,  0.248690, -0.181439,  0.951438,  0.979327,  0.037588,  0.198758,  0.535688, -0.057285,  0.842470, -0.750192,  0.107871,  0.652362,  0.810339,  0.183801, -0.556388, -0.351481,  0.496081,  0.793955, -0.313636, -0.119247, -0.942026, -0.169881, -0.692253, -0.701374,  0.692426, -0.662771,  0.285098, -0.937894,  0.306732,  0.162079,  0.957219, -0.250554,  0.144758,  0.249156, -0.887861, -0.386813,  0.659943, -0.584594, -0.471937, -0.302976, -0.043869,  0.951988, -0.114980, -0.560008,  0.820470,  0.356831, -0.929805, -0.090187, -0.919866,  0.053299,  0.388595, -0.829551,  0.335054,  0.446748,  0.549116,  0.676293, -0.491019, -0.371582, -0.869305,  0.325937, -0.539171, -0.270770, -0.797482,  0.789239, -0.577789, -0.207993,  0.234807,  0.949156,  0.209684,  0.535451, -0.792896, -0.290875,  0.717303,  0.691691,  0.083904,  0.611875, -0.392096, -0.686927,  0.068897,  0.924527, -0.374837, -0.652556, -0.634856, -0.413676,  0.852618,  0.510045, -0.113565, -0.480061, -0.573877, -0.663481, -0.065692,  0.746945, -0.661633,  0.747319,  0.082595,  0.659312,  0.717025,  0.391022,  0.577042,  0.145233, -0.979083,  0.142489, -0.440116,  0.876581, -0.194691, -0.506307, -0.827685, -0.242055,  0.059144, -0.985625, -0.158255, -0.056459, -0.889558, -0.453320, -0.227322, -0.778858,  0.584555, -0.053303,  0.532527,  0.844733,  0.591974,  0.119894, -0.796989,  0.930507,  0.243396, -0.273706,  0.166370, -0.466212,  0.868889,  0.414515, -0.699337, -0.582327, -0.286147, -0.445461, -0.848342,  0.367589,  0.322668, -0.872218,  0.162665,  0.982784, -0.087609, -0.886161, -0.219460, -0.408112, -0.503293,  0.170464,  0.847136,  0.629724,  0.396707, -0.667886, -0.340048,  0.630705, -0.697552,  0.830058, -0.401390,  0.387156,  0.230393, -0.256843, -0.938590, -0.776250, -0.478876,  0.410016,  0.887439, -0.107201,  0.448286, -0.626687,  0.513127, -0.586484,  0.351655,  0.797019,  0.491019,  0.475870, -0.746172,  0.465590,  0.714010, -0.233997,  0.659875, -0.132433,  0.747801,  0.650580, -0.064481, -0.284249, -0.956580};
static float ps74 [222] = {  0.874230, -0.347852,  0.338704,  0.349258,  0.846485, -0.401847,  0.915364,  0.371333, -0.155627,  0.724771, -0.316082,  0.612208, -0.888913, -0.438982, -0.130874,  0.881132, -0.063362,  0.468605, -0.789870,  0.590730, -0.164754, -0.315993,  0.937649,  0.144782,  0.575189, -0.779000,  0.249633,  0.562140,  0.715595,  0.414636, -0.333130, -0.424024,  0.842157, -0.358447, -0.666754,  0.653417, -0.503976,  0.362388,  0.784017, -0.592166, -0.466925,  0.656749, -0.287516,  0.843278,  0.454111,  0.780015,  0.239874,  0.577960, -0.705531,  0.569562,  0.421693, -0.298734, -0.947700,  0.112350,  0.470280, -0.016577,  0.882362, -0.080291, -0.595540,  0.799303, -0.881994,  0.112355,  0.457670,  0.479014, -0.080292, -0.874128,  0.989233,  0.144990,  0.019918, -0.093928, -0.812306,  0.575618, -0.200477,  0.698867,  0.686581, -0.596823, -0.679097,  0.427353,  0.775834,  0.487582,  0.400431, -0.709497,  0.064535,  0.701747, -0.478035, -0.865612, -0.148988,  0.000000,  0.000000,  1.000000,  0.497722, -0.297430,  0.814744,  0.472669, -0.551722,  0.687158,  0.032795,  0.346562,  0.937454, -0.239166,  0.469138,  0.850123, -0.065402,  0.997432, -0.029186, -0.559410, -0.815957,  0.145860, -0.535985,  0.779665,  0.323795, -0.483434,  0.617803,  0.620170, -0.886157,  0.362617,  0.288505,  0.334411,  0.687960,  0.644112,  0.172797, -0.892486,  0.416664,  0.557513,  0.251011,  0.791311,  0.218547, -0.470344,  0.854993, -0.726718,  0.344314,  0.594415, -0.714763, -0.684634, -0.142791,  0.420995, -0.763433,  0.489830,  0.682310, -0.568538,  0.459583,  0.256829,  0.162876,  0.952633,  0.707277, -0.030322,  0.706286,  0.180691,  0.972522,  0.146806, -0.224757,  0.188639,  0.955981, -0.070121, -0.954679,  0.289260, -0.291777, -0.867656, -0.402542, -0.044269, -0.326651,  0.944108,  0.582860,  0.504223,  0.637207,  0.933904,  0.141978, -0.328125, -0.983467,  0.106716,  0.146302, -0.277012, -0.137653,  0.950955, -0.046923, -0.824479, -0.563944,  0.977837, -0.104650,  0.181338,  0.325962, -0.930723,  0.165843, -0.054163,  0.950031,  0.307421,  0.931218, -0.363930,  0.019678,  0.328483,  0.442605,  0.834386, -0.782435, -0.200321,  0.589633, -0.779235,  0.612964,  0.130646,  0.229022, -0.183791,  0.955913, -0.555237, -0.208885,  0.805033,  0.180674, -0.710811,  0.679783,  0.768685,  0.621658, -0.150546,  0.071617,  0.607828,  0.790832,  0.927482,  0.197284,  0.317578,  0.576757, -0.816235, -0.033334, -0.790385, -0.444782,  0.421260};
static float ps75 [225] = { -0.352631, -0.561433,  0.748628, -0.374110, -0.926709,  0.035378, -0.961478, -0.227706,  0.153980,  0.427431, -0.593109,  0.682294, -0.281127,  0.869375,  0.406393,  0.321475,  0.836461,  0.443832, -0.463292, -0.312069,  0.829442,  0.111170, -0.263922,  0.958116,  0.419366, -0.299479,  0.856997,  0.209318,  0.956801,  0.201789, -0.158145,  0.272300,  0.949127, -0.797216, -0.401470, -0.450854,  0.596501,  0.422743,  0.682257,  0.724351, -0.658694,  0.203564,  0.015491,  0.892463,  0.450854,  0.471533,  0.212443,  0.855876, -0.318919,  0.506918,  0.800828,  0.801219, -0.130176,  0.584040, -0.163985, -0.778869,  0.605370,  0.121806,  0.720894,  0.682257,  0.641660, -0.373791,  0.669741, -0.970788,  0.054989,  0.233555, -0.672349, -0.739389,  0.035378, -0.715882, -0.609791,  0.340099,  0.330354, -0.815804,  0.474688,  0.889668, -0.439263,  0.124653,  0.465160, -0.665224, -0.584040, -0.554995,  0.767867,  0.319938,  0.513248,  0.817158, -0.262354, -0.719319, -0.212505,  0.661379,  0.990183,  0.085524,  0.110559, -0.719874,  0.127501,  0.682294,  0.595115, -0.067177,  0.800828,  0.613907,  0.652787,  0.443832, -0.471059,  0.850621, -0.233555, -0.845127,  0.262590,  0.465626,  0.204014, -0.507568,  0.837110, -0.982331,  0.185407, -0.025489, -0.860019, -0.500764,  0.097991,  0.771010,  0.604007,  0.201789, -0.212363,  0.964983, -0.153980, -0.238465, -0.909652,  0.340099,  0.052957, -0.914790,  0.400443, -0.352906,  0.929104,  0.110559,  0.000000,  0.000000,  1.000000, -0.726625,  0.541864,  0.422373,  0.932688, -0.166532,  0.319938,  0.120907, -0.740243,  0.661379, -0.451867,  0.247733,  0.856997, -0.630410, -0.485912,  0.605370, -0.185402, -0.295183,  0.937280, -0.453111, -0.721412,  0.523694, -0.615330,  0.415712,  0.669741,  0.920024,  0.354645,  0.166680,  0.405728,  0.645972,  0.646611,  0.734680,  0.161285,  0.658963,  0.603506, -0.647282,  0.465626, -0.079950, -0.552842,  0.829442,  0.905211,  0.124248,  0.406393, -0.286015, -0.014454,  0.958116,  0.291940,  0.464806,  0.835898,  0.519710,  0.827445,  0.212688,  0.077666,  0.992152, -0.097991,  0.154912,  0.246641,  0.956645,  0.803625, -0.419271,  0.422373, -0.593554,  0.804391,  0.025489,  0.907879, -0.366493, -0.203564, -0.013373,  0.517007,  0.855876,  0.313959, -0.024224,  0.949127, -0.847049, -0.349506,  0.400443, -0.781995,  0.610692,  0.124653, -0.878329, -0.056642,  0.474688, -0.173717,  0.731840,  0.658963,  0.080015, -0.982759, -0.166680, -0.545809, -0.036612,  0.837110};
static float ps76 [228] = {  0.818184,  0.350134,  0.456049, -0.125723, -0.777783,  0.615830,  0.846214,  0.525447,  0.088474,  0.379488,  0.910114,  0.166379, -0.985134, -0.165693,  0.045352,  0.490001,  0.858268, -0.152561,  0.130656, -0.846533,  0.516053, -0.777836, -0.461409,  0.426699, -0.705261,  0.610504,  0.360406, -0.000000, -0.000000,  1.000000,  0.014731,  0.301156,  0.953461,  0.477353, -0.825398,  0.301416, -0.579778, -0.688839,  0.435153,  0.391238, -0.730260,  0.560047,  0.890262, -0.242389,  0.385591, -0.252197,  0.941426,  0.223859,  0.727816,  0.598632,  0.334549,  0.078075,  0.971958,  0.221816, -0.490450,  0.075742,  0.868171, -0.829189,  0.498764, -0.252348, -0.816703,  0.572705,  0.070751, -0.765258,  0.367343,  0.528620,  0.707466,  0.201338,  0.677462, -0.730282,  0.066840,  0.679868, -0.923814, -0.167673,  0.344172,  0.275056,  0.122731, -0.953562,  0.894766,  0.414191, -0.166852,  0.521001, -0.246022,  0.817331, -0.338895, -0.396947,  0.852985, -0.341206,  0.453063,  0.823598, -0.027940, -0.298363,  0.954044, -0.580818,  0.324240,  0.746672,  0.594086,  0.457164, -0.661864, -0.259322,  0.713097,  0.651341,  0.745854, -0.097337,  0.658959, -0.559909, -0.197335,  0.804711, -0.373776, -0.645276,  0.666266,  0.608481,  0.516043,  0.602868,  0.938743,  0.271007,  0.212879, -0.248677,  0.198623,  0.948002,  0.892713,  0.054419,  0.447328,  0.224560,  0.865405,  0.447936,  0.072843, -0.997153,  0.019475,  0.228553,  0.970733, -0.073770, -0.428202,  0.802295,  0.415892,  0.524081,  0.041767,  0.850644,  0.476792,  0.353221,  0.804925,  0.274403,  0.174939,  0.945568, -0.534379,  0.590366,  0.604903, -0.097360, -0.555246,  0.825968,  0.216772, -0.941671,  0.257422, -0.724290, -0.667093,  0.174330,  0.267173, -0.126517,  0.955307,  0.673694, -0.729051,  0.120917,  0.650242, -0.616890,  0.443432,  0.223898,  0.492599,  0.840962, -0.778168, -0.206747,  0.593051,  0.363436,  0.664932,  0.652518,  0.069054,  0.739617,  0.669476, -0.983185,  0.103269,  0.150607,  0.475128, -0.509681,  0.717272, -0.404898,  0.914187, -0.017864,  0.633390,  0.767516,  0.098668, -0.339284, -0.848400,  0.406330, -0.587265,  0.791918,  0.167290, -0.075637,  0.549654,  0.831961, -0.891524,  0.104964,  0.440644,  0.718458, -0.376431,  0.584908,  0.099193, -0.880934, -0.462727,  0.946144, -0.318612,  0.057426,  0.082647,  0.943908, -0.319699,  0.162697, -0.646611,  0.745268,  0.502117,  0.766016,  0.401370,  0.237708, -0.405737,  0.882537,  0.983612, -0.046004,  0.174330, -0.893607,  0.366475,  0.259157};
static float ps77 [231] = { -0.839647, -0.135974,  0.525836, -0.645936, -0.250283,  0.721197, -0.832479,  0.146739,  0.534272,  0.952662, -0.104855,  0.285379, -0.152737, -0.175599,  0.972541, -0.631494, -0.741833, -0.225607, -0.383695, -0.900449,  0.204867,  0.125547,  0.228414,  0.965435, -0.538356,  0.776560,  0.327302,  0.696647, -0.683395,  0.218298,  0.395251,  0.608707,  0.687934,  0.642865, -0.156059,  0.749913, -0.379443,  0.924510,  0.036113, -0.350104, -0.805765,  0.477672, -0.124268,  0.399851,  0.908117, -0.059710,  0.982874,  0.174339,  0.811915,  0.531507, -0.241443,  0.810447, -0.262968,  0.523472, -0.089571,  0.663931,  0.742410,  0.596721,  0.401550,  0.694753, -0.146726,  0.107903,  0.983274,  0.577895, -0.439657,  0.687561,  0.642171,  0.596750,  0.481150, -0.799233, -0.600144, -0.032463, -0.754939,  0.600232,  0.264176,  0.939509,  0.342331,  0.011531, -0.655125,  0.041023,  0.754406, -0.605759, -0.697927,  0.382040, -0.292712,  0.903294,  0.313656,  0.245484, -0.756945,  0.605618,  0.166207,  0.986032, -0.010744, -0.269296, -0.650645,  0.710028, -0.756364, -0.423059,  0.498934, -0.609095,  0.327947,  0.722118,  0.454222, -0.857747,  0.240733, -0.998412, -0.056328,  0.000424, -0.397638,  0.223486,  0.889909,  0.418307,  0.796608,  0.436389, -0.414108, -0.364483,  0.834066, -0.616997, -0.780917,  0.097383,  0.417231,  0.901390,  0.115823,  0.332565, -0.535640,  0.776203,  0.148365,  0.521145,  0.840474, -0.055361, -0.824560,  0.563060,  0.800264,  0.300704,  0.518801,  0.629457, -0.775965, -0.040768, -0.766507,  0.416409,  0.488948, -0.910980,  0.328293,  0.249679,  0.929146,  0.254367, -0.268301,  0.193332,  0.928544,  0.316904, -0.534605, -0.557948,  0.634738, -0.199889,  0.905647, -0.373962,  0.041875, -0.629929,  0.775523,  0.170225,  0.767210,  0.618395, -0.562583,  0.601451,  0.567237,  0.972471, -0.232772,  0.010797,  0.638594,  0.125850,  0.759183, -0.363095,  0.511682,  0.778680,  0.497357, -0.709512,  0.499228, -0.103559, -0.946662,  0.305134, -0.424182, -0.072101,  0.902702,  0.132128, -0.067668,  0.988920,  0.128451, -0.355849,  0.925674,  0.838145,  0.020115,  0.545076,  0.715687, -0.530623,  0.454127,  0.883008, -0.396602,  0.251005,  0.833627,  0.472294,  0.286365, -0.959737,  0.040691,  0.277936,  0.397373, -0.250448,  0.882820,  0.400313,  0.038452,  0.915571,  0.848172, -0.529716, -0.002192,  0.136341, -0.984531,  0.110042,  0.378281,  0.327954,  0.865650, -0.149076, -0.444203,  0.883436,  0.943478,  0.184047,  0.275639, -0.315052,  0.752072,  0.578904, -0.052539,  0.873164,  0.484586};
static float ps78 [234] = {  0.427950,  0.624180,  0.653649,  0.637135,  0.763607, -0.104710,  0.938404,  0.167906,  0.302004,  0.000000, -0.000000,  1.000000,  0.244328,  0.185147,  0.951853,  0.417663, -0.879414,  0.228449,  0.655808,  0.470002,  0.590774, -0.307821, -0.677172,  0.668345,  0.309610,  0.442718,  0.841512,  0.476515,  0.040045,  0.878254,  0.169653,  0.740945,  0.649783,  0.048505,  0.554181,  0.830982,  0.704635, -0.005531,  0.709548,  0.917957,  0.390341,  0.070635,  0.283882, -0.430635,  0.856718, -0.015331,  0.301597,  0.953312,  0.260096, -0.139652,  0.955430,  0.081261,  0.917834,  0.388558,  0.260793, -0.690283,  0.674905,  0.797290,  0.215115,  0.563963, -0.560917, -0.551829,  0.617136, -0.283153,  0.381579,  0.879899, -0.513984, -0.067825,  0.855114, -0.405343,  0.737584,  0.540061,  0.709640, -0.529780,  0.464483, -0.217352,  0.620897,  0.753157, -0.502017,  0.508920,  0.699271, -0.880731,  0.148079,  0.449872,  0.880508, -0.346366,  0.323631,  0.399539,  0.904000, -0.152160, -0.723243,  0.342684,  0.599573,  0.019848, -0.309281,  0.950763, -0.700903, -0.259407,  0.664412,  0.825218,  0.442810,  0.350621,  0.949229, -0.310654, -0.049584, -0.277271,  0.104727,  0.955067,  0.145428, -0.974556,  0.170559,  0.264621,  0.951463,  0.157142,  0.553176,  0.291381,  0.780444, -0.248189, -0.186250,  0.950638,  0.029770, -0.586252,  0.809582, -0.101639,  0.807259,  0.581380,  0.737990, -0.284245,  0.612025, -0.937567, -0.270791,  0.218265,  0.521346,  0.839473,  0.153244, -0.877885, -0.139149,  0.458208, -0.783730, -0.437626,  0.440739, -0.343816,  0.936970,  0.062272,  0.514929, -0.513966,  0.686067, -0.864391,  0.394878,  0.311287,  0.192926, -0.873289,  0.447377,  0.759455,  0.636989,  0.132182, -0.770510,  0.616981,  0.160151, -0.367661, -0.823404,  0.432239, -0.977576,  0.061455,  0.201415,  0.522190, -0.246984,  0.816282, -0.844153,  0.526146, -0.102840, -0.222682, -0.475696,  0.850956,  0.474315, -0.737522,  0.480714, -0.470510, -0.356080,  0.807358,  0.670303, -0.699700,  0.247210,  0.821461,  0.544485, -0.169522, -0.668847,  0.593715,  0.447377, -0.046814, -0.786505,  0.615806, -0.534186,  0.227742,  0.814112,  0.604713, -0.796422, -0.005790, -0.528284,  0.800755,  0.282325,  0.615674,  0.683091,  0.392851,  0.136866,  0.985733, -0.097969, -0.616214, -0.691605,  0.376779,  0.977017, -0.128466,  0.170099, -0.243990,  0.907235,  0.342628, -0.730716,  0.038072,  0.681619,  0.884113, -0.072925,  0.461547, -0.104174, -0.922230,  0.372344, -0.993772, -0.109695, -0.019626,  0.362323,  0.827896,  0.428148, -0.052860,  0.990009,  0.130718};
static float ps79 [237] = { -0.118538,  0.490723,  0.863215,  0.312615, -0.770199,  0.555937,  0.583600, -0.668099,  0.461578,  0.493701, -0.869459, -0.017316, -0.990635,  0.131585, -0.036426, -0.278806, -0.741675,  0.610070,  0.161552,  0.405033,  0.899916,  0.581561,  0.616211,  0.531103, -0.045226, -0.095966,  0.994357, -0.304287, -0.018495,  0.952401,  0.786934,  0.441205,  0.431361,  0.983487,  0.167954,  0.067419, -0.802266,  0.348785,  0.484477,  0.952378,  0.030066,  0.303434,  0.119874,  0.657098,  0.744212, -0.638940, -0.742088,  0.202637,  0.391877, -0.873958,  0.287455, -0.125887,  0.734403,  0.666937, -0.235439, -0.325420,  0.915789, -0.321815,  0.916893,  0.236099,  0.899230,  0.394687,  0.188698, -0.538759,  0.065040,  0.839945,  0.018780, -0.792556,  0.609510,  0.367353, -0.365469,  0.855269, -0.977038, -0.068465,  0.201767, -0.083699,  0.208652,  0.974402,  0.458738,  0.823017,  0.334965,  0.917991, -0.265041,  0.295036,  0.910293, -0.413917,  0.006182, -0.575560,  0.376971,  0.725688,  0.633069, -0.747459,  0.201319,  0.328185,  0.740866,  0.586014,  0.445773,  0.236930,  0.863221, -0.867219, -0.262734,  0.422968, -0.620004,  0.585692,  0.522073, -0.774953, -0.530420,  0.343660, -0.153309, -0.911354,  0.382009, -0.919106, -0.367417,  0.142300,  0.590584, -0.757135, -0.279208, -0.732360, -0.139592,  0.666456,  0.847298,  0.181935,  0.498985,  0.492004, -0.538642,  0.683957, -0.391282,  0.771101,  0.502296,  0.707762,  0.660352,  0.251015,  0.149001,  0.884746,  0.441614,  0.068185,  0.992090, -0.105396, -0.024944,  0.984082,  0.175956, -0.801560, -0.597733,  0.014713,  0.350028,  0.930809, -0.105239,  0.197643,  0.130319,  0.971573,  0.401212,  0.509383,  0.761287,  0.055222, -0.389862,  0.919216,  0.689228,  0.060349,  0.722027, -0.503185, -0.237308,  0.830957,  0.940615, -0.243065, -0.236987, -0.411010, -0.511811,  0.754400, -0.734750,  0.146018,  0.662436, -0.892335,  0.035918,  0.449942,  0.747296, -0.663409, -0.037909,  0.829424, -0.128498,  0.543639,  0.645532,  0.349790,  0.678922,  0.124445, -0.927778,  0.351769,  0.121531, -0.889765, -0.439940, -0.347307,  0.295611,  0.889939, -0.422582, -0.832198,  0.358984, -0.559995, -0.645701,  0.519110,  0.809131, -0.527528,  0.258884, -0.221099,  0.973638, -0.056077,  0.225905, -0.150850,  0.962399, -0.805735,  0.526317,  0.271628,  0.476826, -0.052047,  0.877456, -0.116359, -0.590346,  0.798719,  0.628459, -0.256539,  0.734320,  0.195116, -0.606664,  0.770642,  0.261323,  0.950623,  0.167411, -0.670049, -0.408805,  0.619607,  0.743908, -0.417185,  0.522071, -0.369595,  0.576026,  0.729104, -0.560134, -0.826174, -0.060711};
static float ps80 [240] = {  0.818898, -0.387108,  0.423737,  0.764945,  0.595369,  0.245754, -0.795110, -0.283210,  0.536276, -0.531255,  0.169946,  0.829992, -0.079992,  0.926088,  0.368730,  0.190091, -0.489581,  0.850985, -0.972785, -0.024246,  0.230438,  0.856415, -0.484623, -0.178028,  0.601702, -0.615171,  0.509430, -0.764076,  0.203314,  0.612251,  0.777064,  0.114137,  0.618987, -0.687366, -0.061993,  0.723660, -0.099456,  0.610501,  0.785746,  0.233614,  0.968439, -0.086893,  0.395813,  0.599609, -0.695558, -0.898885,  0.424931, -0.106955, -0.585402, -0.341551,  0.735287,  0.827600, -0.142215,  0.543004,  0.545592, -0.799581,  0.250996,  0.930518,  0.337342,  0.142608, -0.231136,  0.794946,  0.560925, -0.949104, -0.277659,  0.148684,  0.749593, -0.605032,  0.268414,  0.393566,  0.909936,  0.130850,  0.362963, -0.001446,  0.931802,  0.835381,  0.443258, -0.325055,  0.682188,  0.682011, -0.263592,  0.355428,  0.721737,  0.593941,  0.066754,  0.789911,  0.609578,  0.160161,  0.787210, -0.595524, -0.975333,  0.211931,  0.061730,  0.726898, -0.686728,  0.004871, -0.109798, -0.602680,  0.790393,  0.835011,  0.548811, -0.039529,  0.139325, -0.726251,  0.673163,  0.437275,  0.485146,  0.757248,  0.423611, -0.298320,  0.855312, -0.477788, -0.859200,  0.183015, -0.491041,  0.870382,  0.036247,  0.040425, -0.987470,  0.152540, -0.995738, -0.058675, -0.071149,  0.524042,  0.783481,  0.333972,  0.266312,  0.294573,  0.917772, -0.640141, -0.553013,  0.533288, -0.876386, -0.035358,  0.480309,  0.645397, -0.366145,  0.670373, -0.902541,  0.247173,  0.352598,  0.618297,  0.598295,  0.509659,  0.218060, -0.971785, -0.089908,  0.533942,  0.190360,  0.823813,  0.750692, -0.482590, -0.451186,  0.232613,  0.892635,  0.386127,  0.164172, -0.193427,  0.967282, -0.383727,  0.868789,  0.312985,  0.445840,  0.767420, -0.460752,  0.175649,  0.569765,  0.802817, -0.153972, -0.118803,  0.980907, -0.076628, -0.986035, -0.147861, -0.202574, -0.912194,  0.356183, -0.055451, -0.369860,  0.927431,  0.951409, -0.174971,  0.253388, -0.928567, -0.096404, -0.358426, -0.531634,  0.683704,  0.499913, -0.243368,  0.137019,  0.960207,  0.643347,  0.764463,  0.041253, -0.373093,  0.598786,  0.708701, -0.622101,  0.091479, -0.777574,  0.411986, -0.574193,  0.707510, -0.318036,  0.381509,  0.867931,  0.670601,  0.365653,  0.645439,  0.304664, -0.929068,  0.209794, -0.022869,  0.361766,  0.931988, -0.593125,  0.417802,  0.688218,  0.442526,  0.100633, -0.891092,  0.361061, -0.794237,  0.488695,  0.067036,  0.087603,  0.993897, -0.334474, -0.370951,  0.866327, -0.652200,  0.713259,  0.256702,  0.834880,  0.366510,  0.410666,  0.088211, -0.896039,  0.435124};
static float ps81 [243] = { -0.737993, -0.666655, -0.104586,  0.651782,  0.323950, -0.685738, -0.920548, -0.311625,  0.235544,  0.007978, -0.114228,  0.993423,  0.652743,  0.633431,  0.415562, -0.211509,  0.418791,  0.883107, -0.262988, -0.102494,  0.959339,  0.936107,  0.213585,  0.279438, -0.043615,  0.928216,  0.369477, -0.750363,  0.506315, -0.424970,  0.775185,  0.353418,  0.523626,  0.399884,  0.359322, -0.843197, -0.835418, -0.473198, -0.279573,  0.888300, -0.452558, -0.078199, -0.859190, -0.192790,  0.473946,  0.555534,  0.499470,  0.664764,  0.841217, -0.228781,  0.489911, -0.920579,  0.324239, -0.217722,  0.964373, -0.057474,  0.258228,  0.984393, -0.172982, -0.032370,  0.540259, -0.800477, -0.259532, -0.251049, -0.913580, -0.319916, -0.113872,  0.790873,  0.601292, -0.598766, -0.708915,  0.372719, -0.636745,  0.196912,  0.745508,  0.375637, -0.540731,  0.752667,  0.424114,  0.743003,  0.517759, -0.319030,  0.633610,  0.704811, -0.849896,  0.129120,  0.510885, -0.732615,  0.401607,  0.549533,  0.639840, -0.404621,  0.653366, -0.196523,  0.975054, -0.103191, -0.100270,  0.164398,  0.981284, -0.250736,  0.759543, -0.600188, -0.753498,  0.577485,  0.314248,  0.163793,  0.797056,  0.581269,  0.400892,  0.334052,  0.853050, -0.612885,  0.747164, -0.257135,  0.070910, -0.597216,  0.798940, -0.730712, -0.056970,  0.680304, -0.467622,  0.883281, -0.033824, -0.046584, -0.776099,  0.628888, -0.787197, -0.589755,  0.180304,  0.198143,  0.147154,  0.969064,  0.377723, -0.863822,  0.333373,  0.586649,  0.806347, -0.075145, -0.559849,  0.639074,  0.527403,  0.162650,  0.898642, -0.407417,  0.289654,  0.588445,  0.754873,  0.516930,  0.824604,  0.229807, -0.114338,  0.915933, -0.384700, -0.757871, -0.473176,  0.449150,  0.514303,  0.083101, -0.853573, -0.695019,  0.101741, -0.711757,  0.283289, -0.117402,  0.951821, -0.384275,  0.181324,  0.905237, -0.342016,  0.827201,  0.445830,  0.099581,  0.395856,  0.912897,  0.913121,  0.407176,  0.020457, -0.525105, -0.561695,  0.639346, -0.273972,  0.944870,  0.179332, -0.963510, -0.002897,  0.267656,  0.028671,  0.993072,  0.113957,  0.115879,  0.376248, -0.919244,  0.328498,  0.943764,  0.037393, -0.496434,  0.431377,  0.753304,  0.481507, -0.272692,  0.832941,  0.183106, -0.373707,  0.909294,  0.657560,  0.209131,  0.723795,  0.490800,  0.049218,  0.869881,  0.521835, -0.667770,  0.530821, -0.698134,  0.713940,  0.053840,  0.359191,  0.757469, -0.545182,  0.992836,  0.119476,  0.001190, -0.006423,  0.614979,  0.788517, -0.391471, -0.885788,  0.249261, -0.902829,  0.302127,  0.305974,  0.098814,  0.983724, -0.150078,  0.849661,  0.063676,  0.523471, -0.805271,  0.570979, -0.159751, -0.240779, -0.595060,  0.766765};
static float ps82 [246] = {  0.159779,  0.694686,  0.701343, -0.677855,  0.388286,  0.624297,  0.073351,  0.490925,  0.868108,  0.490070,  0.871564,  0.014423,  0.057701, -0.266218,  0.962184, -0.242590, -0.842802, -0.480452, -0.679152, -0.578511, -0.451750, -0.945255, -0.050982,  0.322325, -0.330518,  0.773583,  0.540673, -0.230152,  0.928461,  0.291532, -0.058722,  0.998195, -0.012614,  0.900102, -0.206647,  0.383553, -0.568252,  0.821758,  0.042469, -0.433986, -0.366368,  0.823062,  0.735394, -0.666070,  0.124683,  0.338268, -0.356668,  0.870840,  0.372207, -0.797760,  0.474385,  0.739997, -0.425214,  0.521150, -0.292310,  0.109704,  0.950010,  0.835250,  0.546108, -0.064220, -0.362238, -0.623506,  0.692838,  0.286951, -0.069565,  0.955416, -0.888059,  0.457805,  0.042025,  0.474178,  0.203732,  0.856533, -0.477750,  0.288559,  0.829752, -0.212210,  0.941215, -0.262833, -0.491875, -0.064347,  0.868285,  0.214871,  0.975706, -0.042752, -0.489797,  0.842147, -0.225582, -0.853264,  0.203297,  0.480219,  0.135162, -0.670926, -0.729102,  0.636635, -0.658025,  0.402117,  0.110975, -0.513751,  0.850732, -0.519877, -0.720976,  0.458172, -0.075940, -0.698058,  0.712002,  0.381287,  0.885287, -0.266244,  0.752945, -0.147081,  0.641436,  0.354073,  0.465253,  0.811278, -0.062745,  0.267298,  0.961569, -0.488182,  0.816649,  0.307834, -0.516040, -0.781924, -0.349710,  0.323595, -0.945952, -0.021466, -0.659677, -0.234075,  0.714168,  0.833596,  0.323785,  0.447528,  0.074936, -0.853585,  0.515536, -0.733335,  0.652357,  0.191443,  0.537244, -0.073768,  0.840195,  0.756906,  0.552973, -0.348301,  0.440804,  0.655872,  0.612800,  0.080088,  0.952188, -0.294830, -0.232835, -0.184747,  0.954807,  0.950162,  0.295169,  0.100340, -0.955474,  0.219532,  0.197169,  0.237508, -0.680718,  0.692974,  0.641856,  0.744835, -0.182322,  0.218502,  0.219499,  0.950830, -0.618358,  0.627685,  0.472911,  0.870446,  0.062931,  0.488225, -0.310444, -0.924578, -0.220863,  0.615702,  0.412218,  0.671556, -0.824474,  0.444765,  0.349897, -0.971767,  0.213213, -0.101041,  0.688443,  0.713143,  0.132186,  0.853686, -0.454545,  0.254184, -0.827292, -0.316113,  0.464393, -0.171900, -0.461931,  0.870098,  0.489103, -0.579730,  0.651683, -0.817262, -0.056330,  0.573506, -0.669382,  0.113422,  0.734209, -0.244549,  0.447249,  0.860328,  0.040329,  0.965840,  0.255980,  0.969153,  0.060920,  0.238813,  0.702363,  0.147529,  0.696363,  0.573799, -0.339479,  0.745325, -0.445317,  0.563361,  0.695929, -0.045029,  0.853328,  0.519426, -0.618188, -0.497134,  0.608853, -0.237357, -0.825979,  0.511293, -0.930045, -0.312737,  0.192902,  0.836195,  0.497642,  0.230501,  0.000000, -0.000000,  1.000000, -0.997990, -0.041366,  0.048018};
static float ps83 [249] = {  0.196848, -0.201914,  0.959417, -0.445878, -0.198878,  0.872720,  0.585404,  0.113311,  0.802784, -0.702680, -0.711453, -0.008611,  0.595504,  0.398453,  0.697575,  0.820789, -0.503415,  0.269961,  0.045276, -0.423014,  0.904991, -0.041182,  0.974571,  0.220263,  0.652397, -0.565043,  0.505079,  0.906810,  0.238530,  0.347561, -0.871281, -0.137300,  0.471188, -0.934829,  0.105128,  0.339181,  0.833546,  0.506915,  0.219630,  0.298742,  0.437456, -0.848166,  0.747038,  0.601179, -0.283758, -0.271851,  0.894561,  0.354764, -0.289946,  0.040972,  0.956166,  0.986182,  0.082579, -0.143618,  0.035273,  0.887065,  0.460294, -0.546818, -0.452810,  0.704240,  0.866745,  0.495221, -0.059239, -0.906858, -0.322699,  0.271059, -0.338496, -0.848075,  0.407662,  0.710552, -0.703635,  0.003683, -0.880488,  0.474064, -0.002214, -0.191582,  0.547069,  0.814870,  0.939656, -0.255241,  0.227811,  0.384810,  0.326064,  0.863484, -0.491202,  0.743760,  0.453368, -0.166751,  0.656195, -0.735936, -0.512114,  0.840021,  0.179174, -0.641511, -0.710036, -0.290366, -0.813284,  0.292775,  0.502843, -0.550775,  0.063343,  0.832247, -0.977713,  0.201181,  0.060023,  0.987531,  0.008323,  0.157207, -0.131982,  0.287406,  0.948672,  0.335259,  0.040100,  0.941272, -0.764961,  0.060355,  0.641243, -0.268913,  0.961623,  0.054473, -0.354254, -0.671133,  0.651218, -0.720435,  0.642274,  0.261644,  0.423445, -0.637255,  0.643895, -0.571774, -0.662381,  0.484072, -0.672517,  0.534354,  0.512042, -0.227640,  0.765801,  0.601440, -0.390081,  0.320357,  0.863255, -0.100412, -0.810333,  0.577303,  0.336546,  0.576227,  0.744782, -0.152053, -0.226219,  0.962136,  0.599994, -0.379153,  0.704450,  0.338302, -0.430464,  0.836811, -0.877257,  0.403199,  0.260483,  0.028425,  0.712093,  0.701510,  0.491446, -0.173397,  0.853472,  0.138813,  0.236167,  0.961747,  0.265425,  0.780269,  0.566330, -0.680873, -0.200831,  0.704329, -0.289449, -0.948073,  0.131819, -0.472785,  0.874013, -0.112138,  0.544697,  0.810099, -0.216897, -0.062885, -0.941429,  0.331295,  0.092595,  0.493380,  0.864871, -0.619585,  0.325729,  0.714153, -0.102625, -0.618414,  0.779122, -0.170177,  0.839106, -0.516663, -0.441735,  0.572124,  0.691046,  0.392961,  0.851531,  0.347097,  0.000000,  0.000000,  1.000000,  0.482232,  0.873191,  0.070644,  0.221118,  0.960632,  0.168207,  0.779531,  0.181671,  0.599439,  0.012065,  0.998155, -0.059514,  0.897891, -0.039275,  0.438461,  0.534220,  0.647661,  0.543271, -0.755790, -0.412837,  0.508280,  0.810739, -0.310604,  0.496213,  0.750055,  0.460034,  0.475170,  0.641838, -0.717495,  0.270639,  0.422370, -0.806442,  0.413830,  0.960383,  0.270634,  0.066496,  0.727855, -0.103983,  0.677801, -0.217853,  0.945328, -0.242685};
static float ps84 [252] = {  0.613955,  0.551610,  0.564611,  0.400037, -0.482886,  0.778968,  0.795429,  0.362609,  0.485600,  0.356110,  0.509661,  0.783219,  0.849259, -0.067099,  0.523695, -0.193281,  0.382603,  0.903470,  0.388740,  0.707336,  0.590387,  0.195736, -0.265895,  0.943921, -0.175161, -0.982924,  0.056383, -0.303006, -0.491343,  0.816559, -0.303347, -0.198985,  0.931872, -0.008103, -0.845547,  0.533839, -0.082521, -0.656918,  0.749433,  0.571015,  0.339044,  0.747657, -0.507438, -0.027865,  0.861238, -0.930379,  0.248278,  0.269729, -0.534379, -0.313294,  0.785039,  0.280901, -0.853405,  0.439084, -0.284471, -0.753881,  0.592233, -0.622090,  0.419815,  0.660877, -0.953009,  0.141572, -0.267826, -0.479438,  0.817522,  0.319056, -0.562773,  0.826451, -0.016315,  0.821420,  0.569888,  0.022304, -0.498250, -0.576604,  0.647514,  0.455899, -0.217649,  0.863009, -0.094058,  0.995490,  0.012373,  0.027690,  0.197328,  0.979946, -0.388874,  0.545503,  0.742430, -0.310980,  0.767051,  0.561181, -0.343164,  0.932693,  0.111006,  0.435950,  0.893086, -0.111104, -0.188251,  0.922537,  0.336878, -0.464880, -0.797787,  0.383957, -0.451909,  0.269645,  0.850335, -0.572273,  0.644426,  0.507167,  0.486460, -0.667186,  0.564109, -0.810706,  0.274763,  0.516973,  0.834547,  0.487838, -0.256018, -0.027521,  0.840888,  0.540509,  0.920267,  0.145025,  0.363423,  0.052641, -0.960575,  0.272991,  0.618515,  0.779284,  0.100776,  0.739083,  0.145011,  0.657821, -0.130100,  0.638829,  0.758269, -0.979155,  0.203093,  0.002989,  0.135277, -0.518400,  0.844371,  0.312407,  0.267807,  0.911417, -0.252488,  0.096283,  0.962798,  0.216435, -0.713272,  0.666632, -0.732643, -0.131113,  0.667865,  0.541392, -0.784310,  0.302906, -0.082294, -0.366165,  0.926904,  0.880912, -0.012496, -0.473116, -0.666918, -0.725789,  0.168676, -0.908642, -0.367934, -0.197470,  0.889203, -0.415178,  0.192212,  0.821220, -0.333963,  0.462674, -0.712500,  0.580600, -0.394015, -0.753029,  0.648178, -0.113194, -0.875301,  0.474588,  0.092814, -0.890785, -0.245514,  0.382393, -0.680843,  0.135690,  0.719751,  0.992484,  0.084286,  0.088719, -0.662658, -0.612363,  0.431156, -0.030905, -0.083347,  0.996041, -0.217508, -0.916002,  0.337092,  0.071854,  0.969293,  0.235176,  0.522795,  0.067969,  0.849744,  0.632976, -0.423009,  0.648387,  0.136832,  0.693516,  0.707328,  0.949555,  0.302214, -0.083736,  0.233369,  0.870714,  0.432893,  0.979811,  0.030816, -0.197539, -0.727905, -0.384050,  0.568032,  0.746816,  0.591882,  0.303219, -0.696913,  0.692042,  0.188122,  0.081897,  0.462159,  0.883007,  0.523709,  0.772686,  0.358728,  0.677550, -0.152559,  0.719480, -0.778939,  0.505631,  0.370934,  0.356762,  0.918924,  0.168224,  0.257723,  0.003361,  0.966213,  0.324600, -0.929644,  0.174345};
static float ps85 [255] = { -0.614093, -0.338016,  0.713187, -0.784859,  0.550295,  0.284908,  0.891086, -0.258574,  0.372967, -0.915459,  0.391802,  0.091788, -0.276317,  0.062317,  0.959044, -0.927581, -0.292636, -0.232287,  0.541038, -0.840716,  0.021768, -0.293922, -0.812010, -0.504232, -0.385241,  0.899590,  0.205735,  0.180474,  0.461586,  0.868543, -0.953445, -0.151259,  0.260890, -0.151812,  0.735553,  0.660238, -0.471052, -0.151089,  0.869070, -0.096675,  0.528448,  0.843443,  0.665517, -0.707514,  0.237720,  0.434990, -0.305681,  0.846961, -0.168398, -0.949859,  0.263459,  0.551400,  0.680711,  0.482276, -0.973335,  0.201881, -0.108916, -0.086628, -0.488843,  0.868060,  0.738086, -0.294622,  0.606982, -0.873268,  0.301182,  0.383005, -0.884503,  0.031104,  0.465497,  0.778215,  0.497683,  0.383005, -0.823467,  0.028957, -0.566625,  0.210289,  0.219844, -0.952600, -0.053181, -0.856937,  0.512671,  0.388833,  0.600514,  0.698707,  0.436344,  0.861127,  0.260890, -0.689800,  0.491454,  0.531647, -0.333852, -0.809951,  0.482206,  0.638702,  0.445473,  0.627386,  0.338467, -0.643918,  0.686156,  0.015816,  0.875915,  0.482206, -0.168637, -0.951626, -0.256846,  0.517101, -0.709868,  0.478220, -0.483218, -0.604903,  0.632923,  0.621286, -0.116941,  0.774809,  0.834154,  0.244112,  0.494566,  0.259779, -0.964916,  0.038117,  0.708462,  0.064149, -0.702827,  0.027097,  0.999516, -0.015297, -0.368659,  0.585435,  0.722050,  0.123223,  0.695768,  0.707618, -0.532458,  0.122655,  0.837523,  0.066829, -0.271017,  0.960252, -0.975287,  0.123358,  0.183298,  0.215837, -0.471332,  0.855137, -0.760777,  0.648649,  0.021768, -0.112539,  0.966115,  0.232287, -0.218986, -0.676410,  0.703217, -0.506193,  0.694895,  0.510774,  0.219739,  0.192068,  0.956465, -0.050348,  0.275215,  0.960063, -0.578757, -0.815495,  0.002919, -0.744912,  0.210982,  0.632923, -0.113450,  0.953484, -0.279279,  0.307989,  0.951327,  0.010971,  0.280761, -0.104966,  0.954021, -0.941521, -0.333758,  0.046300,  0.949203,  0.019221,  0.314076, -0.556362,  0.483307, -0.675926,  0.708093,  0.497795, -0.500804, -0.598141, -0.709540,  0.372534,  0.363682,  0.430686, -0.825981,  0.744682,  0.650907, -0.147538,  0.466448,  0.078279,  0.881078, -0.316414,  0.336381,  0.886978, -0.605493,  0.751865,  0.260916,  0.062738, -0.693890,  0.717343,  0.825089,  0.241459, -0.510808, -0.000000, -0.000000,  1.000000,  0.436455,  0.870695, -0.226708, -0.854358, -0.437154,  0.281014, -0.669883, -0.706658, -0.227796,  0.674101,  0.176090,  0.717343, -0.834975, -0.543109, -0.088601, -0.551430,  0.386257,  0.739412, -0.736839,  0.517800, -0.434685,  0.861839, -0.473781,  0.181013,  0.445392,  0.353269,  0.822695,  0.235406, -0.831687,  0.502872,  0.405047, -0.873050,  0.271514, -0.997095, -0.068868, -0.032548, -0.284352,  0.845846,  0.451319};
static float ps86 [258] = { -0.112237, -0.294066,  0.949172,  0.258861,  0.590611,  0.764310,  0.348854, -0.837672,  0.420246,  0.223743,  0.202621,  0.953354, -0.371315, -0.915498,  0.154881, -0.865391, -0.448933, -0.222616,  0.060394, -0.040851,  0.997338, -0.964293,  0.236669,  0.118857, -0.497528,  0.716847,  0.488463, -0.371317, -0.684902,  0.626924, -0.701570,  0.645447,  0.301991, -0.602888, -0.796149,  0.051710, -0.240469, -0.516533,  0.821808, -0.933178,  0.095642,  0.346456,  0.720881, -0.513663,  0.465275,  0.416251, -0.296743,  0.859464,  0.456312,  0.875196,  0.160659, -0.217108, -0.022226,  0.975895,  0.923483, -0.366988,  0.111798,  0.234574,  0.910917,  0.339418,  0.339520, -0.044389,  0.939551, -0.226160, -0.862066,  0.453536,  0.667955, -0.744199,  0.002049, -0.001713,  0.676517,  0.736425,  0.194913, -0.736613,  0.647619, -0.076966,  0.220201,  0.972413,  0.713561,  0.539005,  0.447554, -0.436223,  0.538666,  0.720797,  0.775638, -0.212326,  0.594394,  0.757401,  0.084493,  0.647460,  0.040797, -0.543144,  0.838648,  0.932774,  0.188701, -0.307122, -0.814625, -0.579612,  0.020870,  0.319142, -0.536600,  0.781159,  0.212359,  0.975041,  0.064804,  0.657747,  0.349442,  0.667277, -0.252937,  0.729822,  0.635124,  0.260314,  0.777635,  0.572294, -0.342174,  0.270097,  0.899980, -0.626888, -0.581882,  0.518097, -0.104013, -0.967452,  0.230689,  0.596438, -0.414561,  0.687314, -0.036157,  0.962981,  0.267134, -0.207799,  0.953468, -0.218445, -0.479774, -0.793847,  0.373663, -0.012333,  0.854396,  0.519476,  0.529095,  0.142897,  0.836444,  0.457321, -0.881828,  0.115054, -0.697580, -0.665341,  0.265899,  0.057772, -0.887070,  0.458005,  0.981371, -0.097655,  0.165452,  0.962306,  0.180609,  0.203341,  0.079302,  0.433073,  0.897863, -0.847178, -0.444171,  0.291550,  0.879677, -0.301934,  0.367430,  0.486111, -0.654230,  0.579379, -0.469013,  0.021693,  0.882925,  0.500337,  0.760694,  0.413530,  0.606420, -0.115211,  0.786753,  0.590847, -0.741995,  0.316768,  0.158067, -0.304520,  0.939299, -0.684719,  0.061278,  0.726226, -0.672425,  0.496330,  0.549091,  0.506284,  0.586857,  0.631882, -0.297127,  0.949323,  0.102477,  0.905704, -0.007818,  0.423838,  0.050955, -0.998693,  0.003957,  0.281390, -0.877943, -0.387345, -0.630481, -0.208909,  0.747563, -0.192572,  0.495663,  0.846897,  0.846532,  0.278330,  0.453779, -0.380967, -0.254928,  0.888750, -0.944437, -0.326198,  0.040408,  0.407394,  0.378538,  0.831107, -0.514312,  0.830880,  0.212419, -0.090427, -0.732716,  0.674499, -0.798481,  0.239579,  0.552295, -0.765237, -0.347929,  0.541626, -0.689914, -0.692920, -0.209478, -0.576935,  0.310041,  0.755659, -0.839635,  0.535712,  0.089592, -0.508669, -0.455021,  0.730898, -0.834224, -0.079047,  0.545730, -0.855707,  0.396289,  0.332747,  0.785360, -0.584226,  0.204671,  0.996237,  0.046194, -0.073334};
static float ps87 [261] = { -0.898778, -0.347801,  0.266894, -0.823343, -0.557229,  0.107711, -0.186537, -0.969920,  0.156395, -0.515034, -0.714826,  0.473036,  0.358522, -0.204202,  0.910914,  0.405771,  0.737452,  0.539921, -0.194273,  0.419022,  0.886949, -0.293514,  0.152884,  0.943651,  0.144728, -0.445496,  0.883508, -0.568604, -0.260373,  0.780318,  0.769920,  0.323612, -0.549999, -0.026972, -0.791930,  0.610016,  0.768022,  0.265751,  0.582683,  0.660783,  0.745782,  0.084704, -0.520057,  0.711210,  0.472992,  0.192964,  0.972594,  0.129710, -0.058443,  0.966876,  0.248468, -0.416850, -0.868746,  0.267425,  0.432804,  0.860502,  0.268733, -0.094620,  0.651373,  0.752835, -0.387268,  0.564483,  0.728960,  0.820795, -0.420679,  0.386425, -0.313866, -0.680810,  0.661805,  0.410420, -0.453498,  0.791136,  0.429817, -0.666882,  0.608708, -0.536905,  0.840500, -0.072749,  0.936853, -0.190223,  0.293464, -0.347302, -0.153871,  0.925043,  0.548825, -0.810788, -0.203506, -0.980706,  0.194983, -0.014060, -0.469791,  0.322934,  0.821590,  0.159811,  0.725714,  0.669178,  0.595427, -0.199314,  0.778293,  0.477031,  0.056901,  0.877043,  0.643610, -0.425587,  0.636114, -0.914806, -0.273094, -0.297573, -0.687034,  0.226155,  0.690534,  0.642688, -0.620480,  0.449395, -0.988173, -0.082450,  0.129288,  0.643188,  0.740173, -0.196094, -0.359853, -0.447339,  0.818775, -0.853103,  0.115528,  0.508791,  0.199012,  0.891203,  0.407617,  0.020320, -0.940321,  0.339681, -0.229818, -0.860445,  0.454773, -0.628428,  0.488005,  0.605747,  0.004090,  0.242722,  0.970087,  0.708822,  0.038513,  0.704335, -0.738259, -0.075060,  0.670328, -0.721929,  0.662383, -0.200168, -0.278935,  0.763554,  0.582392,  0.891332,  0.042732,  0.451333, -0.888297,  0.443908, -0.117793, -0.309158,  0.888429,  0.339286,  0.184780, -0.662784,  0.725654, -0.057785,  0.997718, -0.034925,  0.557146,  0.302244,  0.773458, -0.948655, -0.315876, -0.016631,  0.208672,  0.054664,  0.976457,  0.097213,  0.491091,  0.865667, -0.735848, -0.551019,  0.393580,  0.750524, -0.657862, -0.062706, -0.907749, -0.120526,  0.401827, -0.102160, -0.022438,  0.994515,  0.096634, -0.196442,  0.975742,  0.355762,  0.540765,  0.762238,  0.802876, -0.196134,  0.562957, -0.573218, -0.507713,  0.643155, -0.536895,  0.032327,  0.843029, -0.136738, -0.335177,  0.932180, -0.952797,  0.161996,  0.256779, -0.724849,  0.604762,  0.329935,  0.626052,  0.692334,  0.358794, -0.475385,  0.810520, -0.342148,  0.586309,  0.528188,  0.614215,  0.829376,  0.534442,  0.162808, -0.773472, -0.474692, -0.420009, -0.819824,  0.372882,  0.434566,  0.986765,  0.050761,  0.154008,  0.273362, -0.939062,  0.208413, -0.093804, -0.587363,  0.803869,  0.303638,  0.301499,  0.903826, -0.891447,  0.424093,  0.159586,  0.232042, -0.840398,  0.489783, -0.041231,  0.857455,  0.512904, -0.439863, -0.898065,  0.000108,  0.314889, -0.946448, -0.071289};
static float ps88 [264] = { -0.329296, -0.363572,  0.871424, -0.077793, -0.269098,  0.959966,  0.670002, -0.492828,  0.555175, -0.295075,  0.631780,  0.716788, -0.724696, -0.248340,  0.642762, -0.467008,  0.124074,  0.875505,  0.902010, -0.005019,  0.431686,  0.986002,  0.071614,  0.150569,  0.657418, -0.660719,  0.362287,  0.580107,  0.814255, -0.021554, -0.635473,  0.319916,  0.702729,  0.723972, -0.075027,  0.685737, -0.979271, -0.166320,  0.115611, -0.127633,  0.434756,  0.891458,  0.150167, -0.426797,  0.891792,  0.427047, -0.825211,  0.369672, -0.128019,  0.937782,  0.322763, -0.300393, -0.794609,  0.527598, -0.397529,  0.391040,  0.830095,  0.410590,  0.581915,  0.701990, -0.548565, -0.656361,  0.517944, -0.876724, -0.442778,  0.187891,  0.615202, -0.310420,  0.724684,  0.604168,  0.620305,  0.500202, -0.893609, -0.248124,  0.374028, -0.286579, -0.084961,  0.954282,  0.462631, -0.150078,  0.873756, -0.528001, -0.168886,  0.832282, -0.950934,  0.201745, -0.234572,  0.587457, -0.796364,  0.143870,  0.384718, -0.921877,  0.046216, -0.333501, -0.941783,  0.042682, -0.953296,  0.047088,  0.298345,  0.203098, -0.945327,  0.255163, -0.709117, -0.654520,  0.262216,  0.592202,  0.374149,  0.713659,  0.043980,  0.990196, -0.132584, -0.497801, -0.817822,  0.288725,  0.293700,  0.075933,  0.952877, -0.824992,  0.231338,  0.515627,  0.402503, -0.445289,  0.799818,  0.176748,  0.505533,  0.844510, -0.093778, -0.536422,  0.838724, -0.681288,  0.045865,  0.730577, -0.151955,  0.986711,  0.057535, -0.330798, -0.609067,  0.720840,  0.402881,  0.788313,  0.465027, -0.031608,  0.649842,  0.759411,  0.617063, -0.776933, -0.124934, -0.592324,  0.707920,  0.384710,  0.990884, -0.128131, -0.041621, -0.746043,  0.481996,  0.459455,  0.194605,  0.743889,  0.639342, -0.920656,  0.390093, -0.014820,  0.387173,  0.896223,  0.216523,  0.826454, -0.261841,  0.498411, -0.780443, -0.625180, -0.007700, -0.536750,  0.568618,  0.623357, -0.058240, -0.747410,  0.661805, -0.796388,  0.570836,  0.199781, -0.928228, -0.364408, -0.074828,  0.802511,  0.536286,  0.261482,  0.192279, -0.644753,  0.739813,  0.249746,  0.921777, -0.296569,  0.210652, -0.184575,  0.959978, -0.845726, -0.035438,  0.532439,  0.762430,  0.412222,  0.498772, -0.389129,  0.890186,  0.236953, -0.353901,  0.793177,  0.495606, -0.552049, -0.443524,  0.706065,  0.147579,  0.902122,  0.405459,  0.017561,  0.902574, -0.430176, -0.748708, -0.468285,  0.469198,  0.773121,  0.169875,  0.611086,  0.085001,  0.264197,  0.960716,  0.000000,  0.000000,  1.000000,  0.907542,  0.255726,  0.333124,  0.372352,  0.323526,  0.869876,  0.451987, -0.653257,  0.607423, -0.910113,  0.322731,  0.259883,  0.215362, -0.816866,  0.535116,  0.118498,  0.982243,  0.145451,  0.620014,  0.745310,  0.245144,  0.840770, -0.453077,  0.296356, -0.086326,  0.823203,  0.561146,  0.559386,  0.108322,  0.821799, -0.207360,  0.185217,  0.960571, -0.782936,  0.617340, -0.076822};
static float ps89 [267] = {  0.394237,  0.801920,  0.448890,  0.871911,  0.189679,  0.451435,  0.053024, -0.995932,  0.072862,  0.483488,  0.229926,  0.844615,  0.829289, -0.050835,  0.556502,  0.771800, -0.460856,  0.438105,  0.591275,  0.776111,  0.219192, -0.756801,  0.126552,  0.641278, -0.201887,  0.560163,  0.803405,  0.619880,  0.730437, -0.286724, -0.168336, -0.846191,  0.505592, -0.116245, -0.903202, -0.413174,  0.711879,  0.202713,  0.672411, -0.762301, -0.141751,  0.631509,  0.351663, -0.836616,  0.420009,  0.606121,  0.642441,  0.468922, -0.756047,  0.615616, -0.222281, -0.414288, -0.264373,  0.870903, -0.421374,  0.832314,  0.360135, -0.429410, -0.540015,  0.723872,  0.768953,  0.434173,  0.469260, -0.547780,  0.228059,  0.804939,  0.631140, -0.031582,  0.775026,  0.343761,  0.914513,  0.213295,  0.836417, -0.541540, -0.084503, -0.134618, -0.954266,  0.266936,  0.112429, -0.925745,  0.361048, -0.698454,  0.385355,  0.603046,  0.307140, -0.941433,  0.139175,  0.061947, -0.601037,  0.796817, -0.208555, -0.450214,  0.868224, -0.904145, -0.056486,  0.423476, -0.937562, -0.261819,  0.228970, -0.988020, -0.152913, -0.020832,  0.195940,  0.192148,  0.961606, -0.587929,  0.622512,  0.516545,  0.515324, -0.274835,  0.811731,  0.295534,  0.425349,  0.855417, -0.055594, -0.983034, -0.174793,  0.580283,  0.438921,  0.686017, -0.426158, -0.735575,  0.526611, -0.465319,  0.488385,  0.738213, -0.617866, -0.353363,  0.702407,  0.806002,  0.530440, -0.262667,  0.536820, -0.816259,  0.213412, -0.582758, -0.053934,  0.810854,  0.033522, -0.360603,  0.932117,  0.557063, -0.506248,  0.658326, -0.938712,  0.306973,  0.156805, -0.799786,  0.481031,  0.359098, -0.653839,  0.706678,  0.270372,  0.159317, -0.897038, -0.412239,  0.725110, -0.280019,  0.629131,  0.917651,  0.328693,  0.223332, -0.222682,  0.962237,  0.156568, -0.904484,  0.401171, -0.144813,  0.962981,  0.022129,  0.268660,  0.582528, -0.675529,  0.452019, -0.982113,  0.033148,  0.185353, -0.644673, -0.565551,  0.514343,  0.082897,  0.586824,  0.805460, -0.298353,  0.315237,  0.900895, -0.013538,  0.371138,  0.928479, -0.348268,  0.718260,  0.602339, -0.172465, -0.169514,  0.970320,  0.388813, -0.006817,  0.921291,  0.088866, -0.063521,  0.994016, -0.185534, -0.678776,  0.710521,  0.515373,  0.856242, -0.035215, -0.255901, -0.966218,  0.030625, -0.088448,  0.764513,  0.638511,  0.086338, -0.793288,  0.602694,  0.309187, -0.489053,  0.815617,  0.675833, -0.737042, -0.004391, -0.986037,  0.151360, -0.069438, -0.902983,  0.232657, -0.361238, -0.360902,  0.035306,  0.931935, -0.895582, -0.444529,  0.018098,  0.336776, -0.688759,  0.642022, -0.466487,  0.880145,  0.087941,  0.273591, -0.247529,  0.929450, -0.179289, -0.759922, -0.624799,  0.784406,  0.577250,  0.226914, -0.882708,  0.221388,  0.414504, -0.105113,  0.121822,  0.986970, -0.392389, -0.873400,  0.288450, -0.814075, -0.348567,  0.464525,  0.392515,  0.616705,  0.682354,  0.731864,  0.680939, -0.026395};
static float ps90 [270] = {  0.876356, -0.199226,  0.438531,  0.255493, -0.061115,  0.964877,  0.143704, -0.218048, -0.965300, -0.104453, -0.993046, -0.054300,  0.166079,  0.833431,  0.527078, -0.189773, -0.945424,  0.264875, -0.958006, -0.176794,  0.225760,  0.014731, -0.892302,  0.451199, -0.016691,  0.732764, -0.680279, -0.640357, -0.687917, -0.341634,  0.507245,  0.847081,  0.158606,  0.000000,  0.000000,  1.000000,  0.773851,  0.613406, -0.157758, -0.616326,  0.030864,  0.786886,  0.273225, -0.570599,  0.774445, -0.965234,  0.018286, -0.260748,  0.852775,  0.189261, -0.486780,  0.037685, -0.277267,  0.960054, -0.346952, -0.936704,  0.047014,  0.020790, -0.521810,  0.852808, -0.000988,  0.461547,  0.887115,  0.293599, -0.887321, -0.355612,  0.935551, -0.062385, -0.347639,  0.578971,  0.564453,  0.588375, -0.942361,  0.283991,  0.176931, -0.574469,  0.738741,  0.352487,  0.497023, -0.117474,  0.859749,  0.638823,  0.103235,  0.762396,  0.779140,  0.473554,  0.410716, -0.803463,  0.057014,  0.592619,  0.148899,  0.230034,  0.961724, -0.168282,  0.978007,  0.123222, -0.273987,  0.423198,  0.863617, -0.712027,  0.155429, -0.684734,  0.961676, -0.246848,  0.119355,  0.053696, -0.985430,  0.161383,  0.404637,  0.165594,  0.899360, -0.396883,  0.731729,  0.554127, -0.735365, -0.301063, -0.607124,  0.875623, -0.482929, -0.008007,  0.904763,  0.233078,  0.356480,  0.747498, -0.645259,  0.157758, -0.111987,  0.823198,  0.556601,  0.837719,  0.049154,  0.543885, -0.481829,  0.525168,  0.701455,  0.692529, -0.713765, -0.104609, -0.689299,  0.303247,  0.657958,  0.875421,  0.453854,  0.166296,  0.679879, -0.607517,  0.410716, -0.462806, -0.667810,  0.582957, -0.667948,  0.542006,  0.509975, -0.201461,  0.644691,  0.737419,  0.801869, -0.531576, -0.272823,  0.642855,  0.663573, -0.382633, -0.554298,  0.822145, -0.129738, -0.000355, -0.943811, -0.330484, -0.466439, -0.200650,  0.861495, -0.429369, -0.837699,  0.337496,  0.524962, -0.377018,  0.763068,  0.517202,  0.376563,  0.768571,  0.997701, -0.040141, -0.054597,  0.269387,  0.926994,  0.260985,  0.488816, -0.272836, -0.828625, -0.255761,  0.904363, -0.341634, -0.475111, -0.450154,  0.756063,  0.342089,  0.647041,  0.681405, -0.685044, -0.219667,  0.694595,  0.857876, -0.422574,  0.292371,  0.460513, -0.876156, -0.142401, -0.912634, -0.399043,  0.088676, -0.214829, -0.150362,  0.965008,  0.485326, -0.783647,  0.387757,  0.420214,  0.787842,  0.450251,  0.266830, -0.768051,  0.582150,  0.076285,  0.672761,  0.735916,  0.722072, -0.397224,  0.566414,  0.291855, -0.332087,  0.896961, -0.843770,  0.311013,  0.437405, -0.498847,  0.605659, -0.619942,  0.272023,  0.450197,  0.850486, -0.580274, -0.806320,  0.114590,  0.323380, -0.942105,  0.088676, -0.232344, -0.402559,  0.885416,  0.733861,  0.672506,  0.095831,  0.977793,  0.192317,  0.083283, -0.677662, -0.459739,  0.573947, -0.230689, -0.815249,  0.531180,  0.832093,  0.423171, -0.358535, -0.362831,  0.064678,  0.929608, -0.234759, -0.629659,  0.740553};
static float ps91 [273] = { -0.069414,  0.511496,  0.856477,  0.119132, -0.636741,  0.761819, -0.758800, -0.584610,  0.287146, -0.930099,  0.151200,  0.334747, -0.675341,  0.296810,  0.675144,  0.439606,  0.857508, -0.267257,  0.166433, -0.975826, -0.141644, -0.167846, -0.205938,  0.964063,  0.532516,  0.798552,  0.280609, -0.316301, -0.405247,  0.857746,  0.348728, -0.872651,  0.341861,  0.528091, -0.720402,  0.449600,  0.262976, -0.045152,  0.963745,  0.811366, -0.088480, -0.577803, -0.175993, -0.920155,  0.349774,  0.486560,  0.086032,  0.869401, -0.175935, -0.643388,  0.745050,  0.920479,  0.176889,  0.348467, -0.434891, -0.583661,  0.685718,  0.281386, -0.954277,  0.100883, -0.003535,  0.929551,  0.368677, -0.687623, -0.598277, -0.411387,  0.673746, -0.516942,  0.528050,  0.449510,  0.729786,  0.515125, -0.421589, -0.118486,  0.899013,  0.092299, -0.923538,  0.372235,  0.910901,  0.109469, -0.397840,  0.957579,  0.275522,  0.084439,  0.061514,  0.697956,  0.713494, -0.492441,  0.191809,  0.848947, -0.487267,  0.481845,  0.728283, -0.294976, -0.911465, -0.286741, -0.566036, -0.698873,  0.437241,  0.787523,  0.346730,  0.509496,  0.826076,  0.365931, -0.428594, -0.041153, -0.458421,  0.887782,  0.903368, -0.393954, -0.169492,  0.436666, -0.213263,  0.873980,  0.575155, -0.372854,  0.728132,  0.444512,  0.358932,  0.820717, -0.301444,  0.884303,  0.356567, -0.984155,  0.162394,  0.071189,  0.854165,  0.453210,  0.254957,  0.944068, -0.295752,  0.145827,  0.658127, -0.084675,  0.748131,  0.780706, -0.250667,  0.572420,  0.118722, -0.266396,  0.956524, -0.036937, -0.809408,  0.586084,  0.845679,  0.533199, -0.022925,  0.918491, -0.127003,  0.374493,  0.714183,  0.686757,  0.135307,  0.109371,  0.986061,  0.125388, -0.683395,  0.728473,  0.047937,  0.288227, -0.443369,  0.848734,  0.762942, -0.589141, -0.266144,  0.193113,  0.828079,  0.526300,  0.423371, -0.599835,  0.678936,  0.446832, -0.888056, -0.108161,  0.466509,  0.883586,  0.040561, -0.318748, -0.783477,  0.533445, -0.643791,  0.561328,  0.520043,  0.541443, -0.826311,  0.155079, -0.563271,  0.763560,  0.315757,  0.666638,  0.218157,  0.712742,  0.247697, -0.778558,  0.576622, -0.840332,  0.415963, -0.347588, -0.037976,  0.263719,  0.963852,  0.585444,  0.496897,  0.640585, -0.261238,  0.085888,  0.961446, -0.556689, -0.312132,  0.769851, -0.756212, -0.193276,  0.625130, -0.423260,  0.723171,  0.545779,  0.822018,  0.046025,  0.567598,  0.332428,  0.616638,  0.713617,  0.986829,  0.091300, -0.133539, -0.294985,  0.364160,  0.883386,  0.988901,  0.004365,  0.148513, -0.015412,  0.992269, -0.123144, -0.258999, -0.961735,  0.089364,  0.815914, -0.365863, -0.447692, -0.636541, -0.008280,  0.771198,  0.186971,  0.461900,  0.867001,  0.844110, -0.533738,  0.051005, -0.644704, -0.754673,  0.121759, -0.258249,  0.633106,  0.729715,  0.232655,  0.213906,  0.948744,  0.000000, -0.000000,  1.000000, -0.133868,  0.812657,  0.567158,  0.717251, -0.647610,  0.257200, -0.654544, -0.484715,  0.580192, -0.920855, -0.342594,  0.186160};
static float ps92 [276] = { -0.553960, -0.437925,  0.708061,  0.912246,  0.334723,  0.236152,  0.588257, -0.302998,  0.749764, -0.516962,  0.675997,  0.525146,  0.779357, -0.611841,  0.135105, -0.248488,  0.737523,  0.627944,  0.760596, -0.342689,  0.551415, -0.731304,  0.073523,  0.678078,  0.994923,  0.094072,  0.035744, -0.841731, -0.533684, -0.081672, -0.884577,  0.385773,  0.262112,  0.347600, -0.256946,  0.901750,  0.142569,  0.523736,  0.839866, -0.337800,  0.338101,  0.878396, -0.594963,  0.755914,  0.273156, -0.200789,  0.979283, -0.026234,  0.403769,  0.834059,  0.375920, -0.750157, -0.290811,  0.593880, -0.230479,  0.119464,  0.965716,  0.786176,  0.403854,  0.467792, -0.231968, -0.757785,  0.609879, -0.779731,  0.610043,  0.140952, -0.392673, -0.292769,  0.871834, -0.621369, -0.721408,  0.305730,  0.348011,  0.660223,  0.665578,  0.914306, -0.404852,  0.011813, -0.241170, -0.957814,  0.156297,  0.136935, -0.466545,  0.873833, -0.132738, -0.903835,  0.406770,  0.144407, -0.960896, -0.236276, -0.404950,  0.544413,  0.734595,  0.030506,  0.715483,  0.697964,  0.567294, -0.557972,  0.605677, -0.952644, -0.084762, -0.292034, -0.620894,  0.426573,  0.657667, -0.042615,  0.972195, -0.230263, -0.342390, -0.932189, -0.117438, -0.815102, -0.540734,  0.207886, -0.555131,  0.212608,  0.804132, -0.728028,  0.543631,  0.417661, -0.488290, -0.867009,  0.099337, -0.392328, -0.841103,  0.372322, -0.613864,  0.789306, -0.012965, -0.487131, -0.665683,  0.565305, -0.706996, -0.526806,  0.471839,  0.580779,  0.791717,  0.189418,  0.136045,  0.945197,  0.296807,  0.734295,  0.605614,  0.306664,  0.396072,  0.443535,  0.803992,  0.576520, -0.761440,  0.296367,  0.941135,  0.331827, -0.064461, -0.349370,  0.850857,  0.392407,  0.562361,  0.660787,  0.497103,  0.675620,  0.200184,  0.709552, -0.878946, -0.323593,  0.350345, -0.795531,  0.288701,  0.532712,  0.967111,  0.099736, -0.233986,  0.234238, -0.684244,  0.690611, -0.619948, -0.150854,  0.770005,  0.972708, -0.163746,  0.164395,  0.387343, -0.493242,  0.778895,  0.240694, -0.891664,  0.383409, -0.144317,  0.549843,  0.822706, -0.920551,  0.137357,  0.365675,  0.289211, -0.002524,  0.957262, -0.051178,  0.329913,  0.942623, -0.400958,  0.900043, -0.170750,  0.028559,  0.053899,  0.998138, -0.313756, -0.552588,  0.772143,  0.417650, -0.751960,  0.510024, -0.067973, -0.996826, -0.041437,  0.464821,  0.199384,  0.862663, -0.072517,  0.874429,  0.479704, -0.443766, -0.011761,  0.896066,  0.208911,  0.267703,  0.940580,  0.606856,  0.453196,  0.652947,  0.843134,  0.150345,  0.516256,  0.719292, -0.569935,  0.397230,  0.543632, -0.052328,  0.837691, -0.039449, -0.643126,  0.764743, -0.696153, -0.717181,  0.031971, -0.981965,  0.158458,  0.103131,  0.878057, -0.381694,  0.288660,  0.057974, -0.823400,  0.564492,  0.082540, -0.215940,  0.972912, -0.128733, -0.406516,  0.904529,  0.198063,  0.823698,  0.531312,  0.889254, -0.135744,  0.436808, -0.411256,  0.902449,  0.128276, -0.860110, -0.074252,  0.504676,  0.746354, -0.072648,  0.661573, -0.185789, -0.147262,  0.971492};
static float ps93 [279] = {  0.309878, -0.602967,  0.735124,  0.661656,  0.595358,  0.455807,  0.431368, -0.743552,  0.510932, -0.194172,  0.183582,  0.963636, -0.099391,  0.668528,  0.737016,  0.994929,  0.077243,  0.064421,  0.918183, -0.396130, -0.004661, -0.327822, -0.811355,  0.483979, -0.877192, -0.409811,  0.250180, -0.874096, -0.205885,  0.439963,  0.037291, -0.616614,  0.786382, -0.278009,  0.773923,  0.568994,  0.895651, -0.343295, -0.282768, -0.179927, -0.485224,  0.855678,  0.976819, -0.181201,  0.113977, -0.804304,  0.579439,  0.131702,  0.346121,  0.588009,  0.731058, -0.857604, -0.458412, -0.233180,  0.548917, -0.538275,  0.639492,  0.439437, -0.892499, -0.101688,  0.626383,  0.779226, -0.021245,  0.465276, -0.102365,  0.879227, -0.366621,  0.572660,  0.733246,  0.939327, -0.066254,  0.336565, -0.041194, -0.895220,  0.443717,  0.794367,  0.372255,  0.480007, -0.818103, -0.574076,  0.033812,  0.501966,  0.168423,  0.848330,  0.980616, -0.128633, -0.147801, -0.146521,  0.987537,  0.057471,  0.448757,  0.722987,  0.525269,  0.749905, -0.435976,  0.497562, -0.504992, -0.816506,  0.279823, -0.942910, -0.332944, -0.008253, -0.213332,  0.928194,  0.304868,  0.328680, -0.936079,  0.125398,  0.150409,  0.738358,  0.657423, -0.435044,  0.076740,  0.897133, -0.128809, -0.991609,  0.010937, -0.416558,  0.337329,  0.844209, -0.969164, -0.142712,  0.200886,  0.821537, -0.186119,  0.538922, -0.160807,  0.440722,  0.883123, -0.065319, -0.270746,  0.960432,  0.157223, -0.784208,  0.600249,  0.264344,  0.096892,  0.959549,  0.066541,  0.277143,  0.958522,  0.712383,  0.647639, -0.270324, -0.384069, -0.320799,  0.865782,  0.884571, -0.364013,  0.291598, -0.000000,  0.000000,  1.000000, -0.543457,  0.647653,  0.534042, -0.639981,  0.726426,  0.250460, -0.613109,  0.426674,  0.664866,  0.518843, -0.817154,  0.251122,  0.632024, -0.288939,  0.719069, -0.237226, -0.936417,  0.258549,  0.103349,  0.519924,  0.847937,  0.683957, -0.026646,  0.729035, -0.553900, -0.657621,  0.510616, -0.566949, -0.128036,  0.813742,  0.683872,  0.245868,  0.686926,  0.405183, -0.365254,  0.838103, -0.636783,  0.169301,  0.752226,  0.562735,  0.466448,  0.682463, -0.742843, -0.467831,  0.478873,  0.934278,  0.208420,  0.289286, -0.716346, -0.257238,  0.648596,  0.914588, -0.068742, -0.398502, -0.551562, -0.446622,  0.704492,  0.213526, -0.168637,  0.962272,  0.662695, -0.646364,  0.378219,  0.334533,  0.361628,  0.870237, -0.132631, -0.748115,  0.650180, -0.246591, -0.863629, -0.439702,  0.843159,  0.096285,  0.528972, -0.033192,  0.861399,  0.506843, -0.793579,  0.586680, -0.161364, -0.515257, -0.813405, -0.269968, -0.713468, -0.669994, -0.205113, -0.746293,  0.520620,  0.414730,  0.643362, -0.765026,  0.028644,  0.052566, -0.976906,  0.207103,  0.147217, -0.415862,  0.897433,  0.436168, -0.823091, -0.363702, -0.263566, -0.104697,  0.958943, -0.780242, -0.001021,  0.625477,  0.050100,  0.964243,  0.260242, -0.354832, -0.614472,  0.704641, -0.798404,  0.273158,  0.536596,  0.251223, -0.891716,  0.376471,  0.308354,  0.932159,  0.189728, -0.407977, -0.911913,  0.044375};
static float ps94 [282] = {  0.758827,  0.029781,  0.650611, -0.066061,  0.937111,  0.342723,  0.062325, -0.015840,  0.997930, -0.072117,  0.819955,  0.567867,  0.114964, -0.968610,  0.220406,  0.314155, -0.526638,  0.789911,  0.034085,  0.660594,  0.749969,  0.177710,  0.825785,  0.535256, -0.847119, -0.233710,  0.477251, -0.104334,  0.993040,  0.054640,  0.463934,  0.721627,  0.513829, -0.877992,  0.049230,  0.476136,  0.140302,  0.987784, -0.067813,  0.459289, -0.303239,  0.834925, -0.952350,  0.188947,  0.239435,  0.213865,  0.210751,  0.953858,  0.561149,  0.787445,  0.255034,  0.752367,  0.642212,  0.146653, -0.316797,  0.829459,  0.460040,  0.759871, -0.589042,  0.275002, -0.728106, -0.084945,  0.680181, -0.732299,  0.170595,  0.659269, -0.333435, -0.210484,  0.918977, -0.999138,  0.040678,  0.008299,  0.458745,  0.174114,  0.871342, -0.587717, -0.752361,  0.297559,  0.856700, -0.168922,  0.487370,  0.163849, -0.874185,  0.457115, -0.544961,  0.255231,  0.798670, -0.064920, -0.243000,  0.967851, -0.716569,  0.580329,  0.386971, -0.048571,  0.242988,  0.968813,  0.339597, -0.936316,  0.089371, -0.771226, -0.620823,  0.140672,  0.379847,  0.921767,  0.077857, -0.412511,  0.481886,  0.773060,  0.298095,  0.642928,  0.705537,  0.643285,  0.257911,  0.720879, -0.250780,  0.673041,  0.695791,  0.963847, -0.088735,  0.251249, -0.324915, -0.681834,  0.655388, -0.895578,  0.348483, -0.276584, -0.356550, -0.825257,  0.437977, -0.311101,  0.252566,  0.916202,  0.685400, -0.257779,  0.681012,  0.958967,  0.087222, -0.269768, -0.500418,  0.654243,  0.567052, -0.559377, -0.232523,  0.795632, -0.309544,  0.927531,  0.209449, -0.974965, -0.218006,  0.043777,  0.063093, -0.481063,  0.874413,  0.586268, -0.681014,  0.438760,  0.376538, -0.722792,  0.579475, -0.214664,  0.013769,  0.976591,  0.575069, -0.055641,  0.816211, -0.496250,  0.016278,  0.868027, -0.557023, -0.631818,  0.539010, -0.832613,  0.326276,  0.447548,  0.390686,  0.422240,  0.817971,  0.130486,  0.455006,  0.880876, -0.224273, -0.445742,  0.866611, -0.719002,  0.684457,  0.120642,  0.888296,  0.114899,  0.444666,  0.892075,  0.451004,  0.028241, -0.087905, -0.828399,  0.553198,  0.570048,  0.510597,  0.643689, -0.598399, -0.800765,  0.026327, -0.458873, -0.471769,  0.752907, -0.687327, -0.386523,  0.614965, -0.522869,  0.849761,  0.067187, -0.751592, -0.527889,  0.395528, -0.542102,  0.775637,  0.323284,  0.134095,  0.970176,  0.201933,  0.208436, -0.278804,  0.937456,  0.613716, -0.773269,  0.159399,  0.960929,  0.181963,  0.208580,  0.381358,  0.905375, -0.186714, -0.778914, -0.344756, -0.523866, -0.147642,  0.478220,  0.865741, -0.652151,  0.440804,  0.616759, -0.954661,  0.296959, -0.020918,  0.840466, -0.539812,  0.047121, -0.869197,  0.449538,  0.205941, -0.096307, -0.648038,  0.755495,  0.329278, -0.054176,  0.942677,  0.326170,  0.880458,  0.344104,  0.143583, -0.714049,  0.685213,  0.691072,  0.592768,  0.413575,  0.549691, -0.510313,  0.661378,  0.862957,  0.413873,  0.289853, -0.893864, -0.373271,  0.248346, -0.127132, -0.937072,  0.325166,  0.750481, -0.436693,  0.496062,  0.402156, -0.855170,  0.327039};
static float ps95 [285] = {  0.472850,  0.874495,  0.108034, -0.705798,  0.198813,  0.679944, -0.719260,  0.582792,  0.378179, -0.817418,  0.327503,  0.473888,  0.591500,  0.455493,  0.665322, -0.452978, -0.161340,  0.876801,  0.709187,  0.648699, -0.276122,  0.678740, -0.275871,  0.680593,  0.785295,  0.258622,  0.562518,  0.079182, -0.957213, -0.278341,  0.482109, -0.474711,  0.736356, -0.951755, -0.293148, -0.090702, -0.209380, -0.611263,  0.763228,  0.233831, -0.120305,  0.964806,  0.425333,  0.836017,  0.346651, -0.460483,  0.688025,  0.560872,  0.800905, -0.003476,  0.598782, -0.451921, -0.584160,  0.674184, -0.304376, -0.402247,  0.863454,  0.304204,  0.891437, -0.335857,  0.990249,  0.048362, -0.130645,  0.166986, -0.947079,  0.274149, -0.000000,  0.000000,  1.000000,  0.254460, -0.402379,  0.879398, -0.401706,  0.843230, -0.357205, -0.420741, -0.062969, -0.904993,  0.937500, -0.143540, -0.317001,  0.209056, -0.821221,  0.530934,  0.453639, -0.681564,  0.574179,  0.990903,  0.033871,  0.130245,  0.269497,  0.778857,  0.566351,  0.898421,  0.282638,  0.336088, -0.634424,  0.489336,  0.598377, -0.503970,  0.097671,  0.858181,  0.757610, -0.640169,  0.127317,  0.831048,  0.423877, -0.360121, -0.181966,  0.726622,  0.662502, -0.664482, -0.522851,  0.533939, -0.831157,  0.457779, -0.315622, -0.843162, -0.532313,  0.075635,  0.846983, -0.241332,  0.473686, -0.346404,  0.935141, -0.074271,  0.652937,  0.705038,  0.276758, -0.510223, -0.735912,  0.445092,  0.352510,  0.583361,  0.731729, -0.673546, -0.080000,  0.734804, -0.568763, -0.354440,  0.742214,  0.914293,  0.146749, -0.377535,  0.455288, -0.232036,  0.859577,  0.004601, -0.518332,  0.855167, -0.771895, -0.265061,  0.577859,  0.000641,  0.853132,  0.521695, -0.520952,  0.348953,  0.779000, -0.078197, -0.980405,  0.180805, -0.743812, -0.501724, -0.441607, -0.105462,  0.994300,  0.015651,  0.899567, -0.434485,  0.044744, -0.188074, -0.921180, -0.340670,  0.831984,  0.522055,  0.187779,  0.687829,  0.725708,  0.015429, -0.357146,  0.551775,  0.753652,  0.020682, -0.288374,  0.957294, -0.204907, -0.188722,  0.960415,  0.839116, -0.030404, -0.543102, -0.536524, -0.824143,  0.181469,  0.528090,  0.667176,  0.525355, -0.265839, -0.779228,  0.567568, -0.043911,  0.271811,  0.961348, -0.322226, -0.943354,  0.079078,  0.978301, -0.201715, -0.047317,  0.084942,  0.678099,  0.730045,  0.155692,  0.982253,  0.104589,  0.925687,  0.020397,  0.377739,  0.201344,  0.182957,  0.962282, -0.276095,  0.853456,  0.442023,  0.045432,  0.895652, -0.442429, -0.291716,  0.321826,  0.900739,  0.240351, -0.630776,  0.737803, -0.102429,  0.522600,  0.846402,  0.637704, -0.673251,  0.374255,  0.939566,  0.302986, -0.159423,  0.949061, -0.220949,  0.224643, -0.261067,  0.068232,  0.962906,  0.336421, -0.922291, -0.190266,  0.886537, -0.403680, -0.226042,  0.682755, -0.488768,  0.543094,  0.562899, -0.811469,  0.157044,  0.156589,  0.451683,  0.878329,  0.763927, -0.631632, -0.132125,  0.401435,  0.342444,  0.849460,  0.617538,  0.211278,  0.757634, -0.573788,  0.813690,  0.093146,  0.624356, -0.045344,  0.779823,  0.001022, -0.742830,  0.669480,  0.535999, -0.774141, -0.336764};
static float ps96 [288] = {  0.629447, -0.366066,  0.685414,  0.181079, -0.983468, -0.001226, -0.117583,  0.547931,  0.828219, -0.562286, -0.668479,  0.486795, -0.209346, -0.566021,  0.797367, -0.854242,  0.518757,  0.034096, -0.288161,  0.143813,  0.946721,  0.661689,  0.162289,  0.732004,  0.196943,  0.908374,  0.368877, -0.381142,  0.605037,  0.699043,  0.019746, -0.155880,  0.987579,  0.557727,  0.647053,  0.519868,  0.540259, -0.793022, -0.281488,  0.674368, -0.736353, -0.054875,  0.801320,  0.262022,  0.537802, -0.590876, -0.280892,  0.756284,  0.256246, -0.693922,  0.672911,  0.441517, -0.896409, -0.038914, -0.757433, -0.574644,  0.309968,  0.140553, -0.861444,  0.488015, -0.746543, -0.663990,  0.042331,  0.841313,  0.008512,  0.540482,  0.124890,  0.637267,  0.760456,  0.458661,  0.824301,  0.331901,  0.998558, -0.014257,  0.051751,  0.513196, -0.597983,  0.615667,  0.847910,  0.511944,  0.137699, -0.069362,  0.329003,  0.941778, -0.483067, -0.066207,  0.873076,  0.679473, -0.106109,  0.725987, -0.782988, -0.212414,  0.584645, -0.366629,  0.792476,  0.487406,  0.208673, -0.328990,  0.920989, -0.875516,  0.192221,  0.443309, -0.969638, -0.154944,  0.189195,  0.185865,  0.948815, -0.255352,  0.330663,  0.751878,  0.570387,  0.879644, -0.408953,  0.242867,  0.481736,  0.046238,  0.875096,  0.745870, -0.478465,  0.463410, -0.338943,  0.389847,  0.856234, -0.304394, -0.942590, -0.137364, -0.896144,  0.370177,  0.244734,  0.948312,  0.078845,  0.307389, -0.898702, -0.050447,  0.435649,  0.079987,  0.810999,  0.579554, -0.573976,  0.428931,  0.697545, -0.313751, -0.721788,  0.616913,  0.092174, -0.546872,  0.832127,  0.468321, -0.226788,  0.853957, -0.893520, -0.433058,  0.118671, -0.745729,  0.605732,  0.277447,  0.393224,  0.913082, -0.107966, -0.006070,  0.976957,  0.213348,  0.612971, -0.684624,  0.394407, -0.757414,  0.626704, -0.183211, -0.357943, -0.335384,  0.871432,  0.551862, -0.814235,  0.180192,  0.377161, -0.476556,  0.794131, -0.695816, -0.456694,  0.554320, -0.022205, -0.727716,  0.685519,  0.943034, -0.169671,  0.286179,  0.115378, -0.893837, -0.433293,  0.070363, -0.964544,  0.254370, -0.527665,  0.190746,  0.827760,  0.175630,  0.424980,  0.888001,  0.966975,  0.248151,  0.058145, -0.559700, -0.825739, -0.069937,  0.613339,  0.424052,  0.666330,  0.388730, -0.801564,  0.454295, -0.389044, -0.836741,  0.385369, -0.097136, -0.995091,  0.018949, -0.594066, -0.772910,  0.222926, -0.973401,  0.109456,  0.201272,  0.750628,  0.503939,  0.427322,  0.437614,  0.309834,  0.844095, -0.093563, -0.374373,  0.922546, -0.702212, -0.014613,  0.711818,  0.319799, -0.919743,  0.227602,  0.680249,  0.687740,  0.253525,  0.265572, -0.067452,  0.961729, -0.960869,  0.276426, -0.017891,  0.897169,  0.323099,  0.301157, -0.731014,  0.218574,  0.646409, -0.870639, -0.331912,  0.363074, -0.021189,  0.090236,  0.995695,  0.223935,  0.185758,  0.956738,  0.384966,  0.551065,  0.740357, -0.582666,  0.635470,  0.506634,  0.822288, -0.242989,  0.514586, -0.158738,  0.735885,  0.658236, -0.293053,  0.923484,  0.247582, -0.239788, -0.115533,  0.963926, -0.122016, -0.863621,  0.489153, -0.473880, -0.522932,  0.708505, -0.758522,  0.437549,  0.482903};
static float ps97 [291] = {  0.480667,  0.584228,  0.653940,  0.683600,  0.454256,  0.571264, -0.386581,  0.290338,  0.875362,  0.994438, -0.058025,  0.087904,  0.681773, -0.730512,  0.039209,  0.976469,  0.198025,  0.085407, -0.330643,  0.900182,  0.283454,  0.057105,  0.338974,  0.939061, -0.626575, -0.107711,  0.771882, -0.793576,  0.049762,  0.606432,  0.926478, -0.180973,  0.329981,  0.943575,  0.094697,  0.317331,  0.063637,  0.075670,  0.995100, -0.362247,  0.579986,  0.729653, -0.160594, -0.358809,  0.919492, -0.204549, -0.881101,  0.426405,  0.516357, -0.848561, -0.115413,  0.799152,  0.597744,  0.063702,  0.948125, -0.306615,  0.083936,  0.716912,  0.186638,  0.671717, -0.917980, -0.390287,  0.070628, -0.586242,  0.594913,  0.549908, -0.369080,  0.767798,  0.523705, -0.183920,  0.452174,  0.872761, -0.171432,  0.181288,  0.968373, -0.572071,  0.744178,  0.344868,  0.184167, -0.663273,  0.725363,  0.288433, -0.957441, -0.010634,  0.533551,  0.083676,  0.841619,  0.128590,  0.929651,  0.345274, -0.044965,  0.994994, -0.089246,  0.618758,  0.765473,  0.176603,  0.842511, -0.534521,  0.066805,  0.266461, -0.927146,  0.263435,  0.691994, -0.662374,  0.287062,  0.847923, -0.004696,  0.530098,  0.027364,  0.581835,  0.812846, -0.899072,  0.402694,  0.171777, -0.746338,  0.314312,  0.586675,  0.891293,  0.394995,  0.222654,  0.483097, -0.776600,  0.404364, -0.889999, -0.310201,  0.334181, -0.773952,  0.491494,  0.399290,  0.840953, -0.436372,  0.319963,  0.308771,  0.222855,  0.924660,  0.242679,  0.668907,  0.702617,  0.979536,  0.120377, -0.161304,  0.749792,  0.568185,  0.339083,  0.031608, -0.933917,  0.356090,  0.559549,  0.707118,  0.432307,  0.444147, -0.647749,  0.618994,  0.302413, -0.028398,  0.952754, -0.532304, -0.581702,  0.615041,  0.176936,  0.980207,  0.088819,  0.388296,  0.883513,  0.261974,  0.042173, -0.188463,  0.981174, -0.193188, -0.086925,  0.977304, -0.354684, -0.491620,  0.795305, -0.917349, -0.038666,  0.396201, -0.081526,  0.979735,  0.182959,  0.203014,  0.963780, -0.172954, -0.087064, -0.602218,  0.793570, -0.891609,  0.229280,  0.390467, -0.647754, -0.756387,  0.091072, -0.750204,  0.633652,  0.188887, -0.417130,  0.024676,  0.908512, -0.132661,  0.886989,  0.442325,  0.488927, -0.857725,  0.158928,  0.665298, -0.544146,  0.511159,  0.275446, -0.281453,  0.919192, -0.976316,  0.144973,  0.160590, -0.802764, -0.554524,  0.219257,  0.841039,  0.288001,  0.457941, -0.612204,  0.153466,  0.775664, -0.738421, -0.466047,  0.487375,  0.287989,  0.459184,  0.840364,  0.095503, -0.457826,  0.883897, -0.790155, -0.217133,  0.573156, -0.598913, -0.357308,  0.716683,  0.697380, -0.098283,  0.709931,  0.374978, -0.479380,  0.793465,  0.526203,  0.345265,  0.777112,  0.242048, -0.822838,  0.514151, -0.638079, -0.682786,  0.355891,  0.502592, -0.191069,  0.843145, -0.429793, -0.769944,  0.471661, -0.428956, -0.903225, -0.013489,  0.328388,  0.803259,  0.496926, -0.407041, -0.242664,  0.880586,  0.601990, -0.384165,  0.700018, -0.553822,  0.410741,  0.724274, -0.153567,  0.716289,  0.680696, -0.011649, -0.798777,  0.601515,  0.066392,  0.809726,  0.583040,  0.791846, -0.287182,  0.538986, -0.265597, -0.707608,  0.654789, -0.442768, -0.865226,  0.235246};
static float ps98 [294] = {  0.614078, -0.522903,  0.591169,  0.552818,  0.259746,  0.791786,  0.648852,  0.443690,  0.618167,  0.679308, -0.711828,  0.178444,  0.331393,  0.921580,  0.202161, -0.448546, -0.857321,  0.252602, -0.662257, -0.262973,  0.701613,  0.760592, -0.543091,  0.355741,  0.402044, -0.683925,  0.608775, -0.528567, -0.087536,  0.844366,  0.979602,  0.007674,  0.200800,  0.154923, -0.789218,  0.594251, -0.114042, -0.828611,  0.548086, -0.509107, -0.860265, -0.027451,  0.698207,  0.598339,  0.393061, -0.727090,  0.010342,  0.686464, -0.434772,  0.895182,  0.098092, -0.996827, -0.063016,  0.048624, -0.012085, -0.990532,  0.136746, -0.201970, -0.925369,  0.320780, -0.592954,  0.736597,  0.325316, -0.032655, -0.247989,  0.968212,  0.130044, -0.926493, -0.353129, -0.368129,  0.349825,  0.861454,  0.847469,  0.492772,  0.197413,  0.440339, -0.183049,  0.878974, -0.699087, -0.703194, -0.129594,  0.836095,  0.545899, -0.054219,  0.329596, -0.858317,  0.393265, -0.295340, -0.184468,  0.937414, -0.734869,  0.282982,  0.616352, -0.118750,  0.467087,  0.876201,  0.825544,  0.360300,  0.434352,  0.318165,  0.314612,  0.894310, -0.548644,  0.455752,  0.700914, -0.134352,  0.242059,  0.960915, -0.834551, -0.177270,  0.521632,  0.637883, -0.271725,  0.720605,  0.041695,  0.824370,  0.564514, -0.854922,  0.349388,  0.383452, -0.349293,  0.083156,  0.933316, -0.949390, -0.098645,  0.298206, -0.227872,  0.788631,  0.571083,  0.889018,  0.344140, -0.302018,  0.782363, -0.575446, -0.238265, -0.962341,  0.151656,  0.225613, -0.914509,  0.173766, -0.365348,  0.073753, -0.923923,  0.375402, -0.534927, -0.549371,  0.641907, -0.527766, -0.789349, -0.313672, -0.276132, -0.958995,  0.063879, -0.049877, -0.673818,  0.737212,  0.416739,  0.506428,  0.754890, -0.313927,  0.593246,  0.741289, -0.733296, -0.434974,  0.522565, -0.224929, -0.442052,  0.868330,  0.556517, -0.724635,  0.406439,  0.124166,  0.929230,  0.348015, -0.567251,  0.187380,  0.801945,  0.226359, -0.593343,  0.772467, -0.457611, -0.366340,  0.810177, -0.065602, -0.991742, -0.110197,  0.026585, -0.478194,  0.877852,  0.113364,  0.195278,  0.974174,  0.749418,  0.187130,  0.635102, -0.915921,  0.384084,  0.116479,  0.442155, -0.445433,  0.778517,  0.899395, -0.382558,  0.211513, -0.304762, -0.633489,  0.711205,  0.947752,  0.314898, -0.051050,  0.315144,  0.816585,  0.483604, -0.372579,  0.860086,  0.348478,  0.498806,  0.660220,  0.561518,  0.243759, -0.958392,  0.148549,  0.386365,  0.068346,  0.919810, -0.767827, -0.564907,  0.302194, -0.042550,  0.668333,  0.742645,  0.193208, -0.975469, -0.105499, -0.650747,  0.755646,  0.074346,  0.944491,  0.258668,  0.202554,  0.617895, -0.002811,  0.786256, -0.475103,  0.683578,  0.554074, -0.089211, -0.004669,  0.996002, -0.690989,  0.531035,  0.490445,  0.230338,  0.676131,  0.699851,  0.794821, -0.082814,  0.601166,  0.477418, -0.864097,  0.159400, -0.585179, -0.690167,  0.425716,  0.826191, -0.562031,  0.039118,  0.789753, -0.339675,  0.510795,  0.895511,  0.101072,  0.433409,  0.133150,  0.480659,  0.866740, -0.364391, -0.781494,  0.506446,  0.178497, -0.071581,  0.981333,  0.657996,  0.739014, -0.144564, -0.872551,  0.095125,  0.479172,  0.235759, -0.328399,  0.914643,  0.978633, -0.199401,  0.050165};
static float ps99 [297] = {  0.805055,  0.591947,  0.038544,  0.400332,  0.652555,  0.643355, -0.415404,  0.759269,  0.500950,  0.808173, -0.203275, -0.552753, -0.501568, -0.171721, -0.847904, -0.859239,  0.510659, -0.030607, -0.810419,  0.495893, -0.311947,  0.712379, -0.686224,  0.147011, -0.614950, -0.639024,  0.462044, -0.295812, -0.921893,  0.250216, -0.638795, -0.439359,  0.631589, -0.759829, -0.025812,  0.649611, -0.734265, -0.643706,  0.215633,  0.710646, -0.204854,  0.673065, -0.630958,  0.650673, -0.422512, -0.792708, -0.490699, -0.361701,  0.907230,  0.391033,  0.155009, -0.386223,  0.161559,  0.908147,  0.279892, -0.488418,  0.826504,  0.044684,  0.899845,  0.433915, -0.202012,  0.465726,  0.861563,  0.146089,  0.757516,  0.636261, -0.756701,  0.445124,  0.478819,  0.547411,  0.709785,  0.443336, -0.205435, -0.509963,  0.835305, -0.195401, -0.714996,  0.671267, -0.846460, -0.265729, -0.461404, -0.174646, -0.019156,  0.984445, -0.928871, -0.229052,  0.291092,  0.393421, -0.784796,  0.478868, -0.156075, -0.872640,  0.462752,  0.687402,  0.042773,  0.725016,  0.511469, -0.110419,  0.852178, -0.798695, -0.459035,  0.389067, -0.708123,  0.695125,  0.123952, -0.319305,  0.933337, -0.164095, -0.556104,  0.556233,  0.617538,  0.253986, -0.695788,  0.671841,  0.051883,  0.401229,  0.914507, -0.233967,  0.899662,  0.368602, -0.639944,  0.307777,  0.704092, -0.431987, -0.595076,  0.677696,  0.225196,  0.948058,  0.224660,  0.486987, -0.584826,  0.648708, -0.314935,  0.636212,  0.704308,  0.053716, -0.811865,  0.581369,  0.191646, -0.902878,  0.384815, -0.649227,  0.658382,  0.380838, -0.804326, -0.253064,  0.537605,  0.943683, -0.074442,  0.322366,  0.029927, -0.617933,  0.785661,  0.531242, -0.847218, -0.001968, -0.493149,  0.829472,  0.262259,  0.231979,  0.547788,  0.803812, -0.427382, -0.373806,  0.823173, -0.956089,  0.283820,  0.073072, -0.616359, -0.210269,  0.758873,  0.285720,  0.293242,  0.912345, -0.904843, -0.011112,  0.425601,  0.953519,  0.165578,  0.251765, -0.980146, -0.197065,  0.021900,  0.845738,  0.017089,  0.533325,  0.107138, -0.993026,  0.049211,  0.982805, -0.025582, -0.182864,  0.077577,  0.127737,  0.988769, -0.039247,  0.978752,  0.201257, -0.022099, -0.963597,  0.266445,  0.530494,  0.808425, -0.255001, -0.397500, -0.778998,  0.484929, -0.590490,  0.072882,  0.803748, -0.894628, -0.426480,  0.133247,  0.044219, -0.391477,  0.919125, -0.911990,  0.253896,  0.322198, -0.154182,  0.236875,  0.959228,  0.060706, -0.138439,  0.988509, -0.150452, -0.988376,  0.021838,  0.688131,  0.685269,  0.238498, -0.306570,  0.944581,  0.117393, -0.192999, -0.273175,  0.942405,  0.995983, -0.054409,  0.071116, -0.848306,  0.267815, -0.456784, -0.412314, -0.113310,  0.903968,  0.691883, -0.443745,  0.569551,  0.477418,  0.426997,  0.767949, -0.516835,  0.816007, -0.258872, -0.029489,  0.634423,  0.772423, -0.435506,  0.396375,  0.808221,  0.933975, -0.303659,  0.188368,  0.837865, -0.493886, -0.232506, -0.473114, -0.851294, -0.226853,  0.512881, -0.359809,  0.779417, -0.643557, -0.511628, -0.569272, -0.138963,  0.795081,  0.590369,  0.303111,  0.014832,  0.952840, -0.621392, -0.783476,  0.006110,  0.291095, -0.250808,  0.923233,  0.310817,  0.837377,  0.449658, -0.399799, -0.916520,  0.012340,  0.691265,  0.285524,  0.663798};
static float ps100 [300] = { -0.347525, -0.619580,  0.703809,  0.302255, -0.336903,  0.891705,  0.810019, -0.424441,  0.404623, -0.600593,  0.332731,  0.727033, -0.434892, -0.384048,  0.814479, -0.256787,  0.030420,  0.965989,  0.913518,  0.162240, -0.373045,  0.515225,  0.162765,  0.841458,  0.641736, -0.756502,  0.126012,  0.870980,  0.486810,  0.066405, -0.707903,  0.466592,  0.530250,  0.750779, -0.268205,  0.603653, -0.805454, -0.058584,  0.589755,  0.108435,  0.357285,  0.927680,  0.545766, -0.310945,  0.778109,  0.279921, -0.943915,  0.175125, -0.075707,  0.838424,  0.539735, -0.580116, -0.500152,  0.642894,  0.282654,  0.143715,  0.948395, -0.012972,  0.687364,  0.726197, -0.172335, -0.485701,  0.856968, -0.847951, -0.405076,  0.341896,  0.848909,  0.216426,  0.482197,  0.585397,  0.650722,  0.483602, -0.648836,  0.061633,  0.758428, -0.465187, -0.096043,  0.879987,  0.442818, -0.091827,  0.891897, -0.255133, -0.252746,  0.933288, -0.936416,  0.322432, -0.138428,  0.477187, -0.825644,  0.301004,  0.204821, -0.975873, -0.075637, -0.406616, -0.868907,  0.282248, -0.920668,  0.091004,  0.379591, -0.171379, -0.932429,  0.318128, -0.619191, -0.751820,  0.226647,  0.387047,  0.811989,  0.436885, -0.130286,  0.938937,  0.318470, -0.252552, -0.805905,  0.535476,  0.657576, -0.060744,  0.750936, -0.787903,  0.210572,  0.578678,  0.663375, -0.646106,  0.377466,  0.022999,  0.127615,  0.991557, -0.081256, -0.682385,  0.726463, -0.632713, -0.218934,  0.742793, -0.381851,  0.885528,  0.264631,  0.004028, -0.837408,  0.546564,  0.331933,  0.923102,  0.194175, -0.101692,  0.496560,  0.862025,  0.456007, -0.889975,  0.001310, -0.600511,  0.778704,  0.181678,  0.988136,  0.060440, -0.141185, -0.757130,  0.577446,  0.305466, -0.367402,  0.436311,  0.821370, -0.971135,  0.195638,  0.136467,  0.259222,  0.963787, -0.062600,  0.992405, -0.076185,  0.096587, -0.799574, -0.578270,  0.162129, -0.502208,  0.580272,  0.641149,  0.123621, -0.544719,  0.829457, -0.870695,  0.343400,  0.352089, -0.702141, -0.710927, -0.039746,  0.888663,  0.361589,  0.282012,  0.393042, -0.533759,  0.748745,  0.138882,  0.910217,  0.390149,  0.203278, -0.106825,  0.973276, -0.445810,  0.185248,  0.875749,  0.939730,  0.319030, -0.122991, -0.561525,  0.707684,  0.428803,  0.411147,  0.621389,  0.666959,  0.208082, -0.714432,  0.668048,  0.495994,  0.868190, -0.015352,  0.075475,  0.985665,  0.150890, -0.689419, -0.586649,  0.424904,  0.831703, -0.031012,  0.554355,  0.034403, -0.342848,  0.938761, -0.049357, -0.114276,  0.992222,  0.975069,  0.199391,  0.097385,  0.195800,  0.565090,  0.801459,  0.077541, -0.942158,  0.326077, -0.897550,  0.430220,  0.096511,  0.371071,  0.382385,  0.846220,  0.200032,  0.770110,  0.605738,  0.749951,  0.599573,  0.279438,  0.911711, -0.194980,  0.361615, -0.331241,  0.799874,  0.500481,  0.575745,  0.424972,  0.698510, -0.187513,  0.276080,  0.942666, -0.768325,  0.637733,  0.054531,  0.560691,  0.791563,  0.243008,  0.463487, -0.694610,  0.550179, -0.760861, -0.325422,  0.561418, -0.487143, -0.722273,  0.490931,  0.818877, -0.545138,  0.179625,  0.274155, -0.846696,  0.456009,  0.703729,  0.197627,  0.682429,  0.948433,  0.057595,  0.311701,  0.000724, -0.995887,  0.090597,  0.625131, -0.497044,  0.601797, -0.267172,  0.658836,  0.703246,  0.738950,  0.447524,  0.503662};
static float ps101 [303] = { -0.656303,  0.538641,  0.528330,  0.098306,  0.728407,  0.678055, -0.493388,  0.805891, -0.327273,  0.587785,  0.466172,  0.661205, -0.998272,  0.038547, -0.044350, -0.971100, -0.237759, -0.020870,  0.133141,  0.331410,  0.934046, -0.015359,  0.863622,  0.503906, -0.851914,  0.377565,  0.362887,  0.413174, -0.254902,  0.874250, -0.950875,  0.291869, -0.103194, -0.287432,  0.832904,  0.472920, -0.431758,  0.005418,  0.901973,  0.051812, -0.766546,  0.640096,  0.282559,  0.811107,  0.512120,  0.508184, -0.022274,  0.860960,  0.876657,  0.306806,  0.370598, -0.289768, -0.210493,  0.933663, -0.425760,  0.656065,  0.623143,  0.381553,  0.633494,  0.673129,  0.163414,  0.931373,  0.325331,  0.556765,  0.828265,  0.063166, -0.795840,  0.047622,  0.603631, -0.636459, -0.053547,  0.769449, -0.785501, -0.563910,  0.254941, -0.784360, -0.414121,  0.461826,  0.858257, -0.370529,  0.355110, -0.649744,  0.747129, -0.140110,  0.871941,  0.059943,  0.485929, -0.885632, -0.141672,  0.442250,  0.576910,  0.216855,  0.787495,  0.127669, -0.318003,  0.939454, -0.911037,  0.123658,  0.393346, -0.372432,  0.900085,  0.226146, -0.592591, -0.466387,  0.656749, -0.175060, -0.636676,  0.750997,  0.736481,  0.508008,  0.446681,  0.196797,  0.543211,  0.816206,  0.856434,  0.477153,  0.197093,  0.545928,  0.681083,  0.487942, -0.422424, -0.893414,  0.152873,  0.332638,  0.157314,  0.929841, -0.216858,  0.975842,  0.026545, -0.126435,  0.948167,  0.291536, -0.358793, -0.466044,  0.808747,  0.718826,  0.013772,  0.695053,  0.516336, -0.470278,  0.715706, -0.701732,  0.613263, -0.362603, -0.621282, -0.754588,  0.211201, -0.035616, -0.993848, -0.104870, -0.377649,  0.265341,  0.887116, -0.194216, -0.976840,  0.089794,  0.042107, -0.983895,  0.173719, -0.764545,  0.642105,  0.056330,  0.637885, -0.234463,  0.733574, -0.171619,  0.715481,  0.677225, -0.195923, -0.806074,  0.558443, -0.893306,  0.434919,  0.113353, -0.735318, -0.240169,  0.633740, -0.450061,  0.891439, -0.052735, -0.967113,  0.198245,  0.159350, -0.300868,  0.505600,  0.808608,  0.296623, -0.670736,  0.679800, -0.053436,  0.542979,  0.838044, -0.201672, -0.920457,  0.334794,  0.323610,  0.938278,  0.122111,  0.963031,  0.126479,  0.237856,  0.682211,  0.684504,  0.256987,  0.043160, -0.905565,  0.422006,  0.943128, -0.126612,  0.307375, -0.605640,  0.196222,  0.771166,  0.060021,  0.098872,  0.993288, -0.535178,  0.433116,  0.725256, -0.754313,  0.302278,  0.582786,  0.292969, -0.471827,  0.831594, -0.529404,  0.748080,  0.400134, -0.887915, -0.457874,  0.044255, -0.200119,  0.066142,  0.977537,  0.451560,  0.835615,  0.312796, -0.276665,  0.929264, -0.244797,  0.707255, -0.437884,  0.555021, -0.121347, -0.405033,  0.906214, -0.587702,  0.795559,  0.147282,  0.241647, -0.083957,  0.966725, -0.413656, -0.659874,  0.627260, -0.044156, -0.143141,  0.988717, -0.513951, -0.268495,  0.814717,  0.398157,  0.390696,  0.829956, -0.915800, -0.312148,  0.252734,  0.518510, -0.661979,  0.541231,  0.836733, -0.524940,  0.155936, -0.127647,  0.314671,  0.940579,  0.623069,  0.634809, -0.456948, -0.977995, -0.066981,  0.197586,  0.283490, -0.827186,  0.485177, -0.746765, -0.665051, -0.007045,  0.061074, -0.572579,  0.817572,  0.752708,  0.270938,  0.600019,  0.816775, -0.190721,  0.544521, -0.736727,  0.604457,  0.303094, -0.429855, -0.806139,  0.406650};
static float ps102 [306] = { -0.562294,  0.181234,  0.806833,  0.624838, -0.283908,  0.727306,  0.583800, -0.537572,  0.608436,  0.186654, -0.320199,  0.928780,  0.272833,  0.757214,  0.593456,  0.433207, -0.901035,  0.021616, -0.060932, -0.843222,  0.534101, -0.743328,  0.063136,  0.665941,  0.065083, -0.941568,  0.330474, -0.544125, -0.066036,  0.836401, -0.677796, -0.694572,  0.241168, -0.925122,  0.311877,  0.216523,  0.666079,  0.718104,  0.201656, -0.311792, -0.161185,  0.936379,  0.361697,  0.063877,  0.930105,  0.670777,  0.405322,  0.621105,  0.220647, -0.833758,  0.506124,  0.078865, -0.714587,  0.695087,  0.744913,  0.159633,  0.647782,  0.977170,  0.009468,  0.212248, -0.381028,  0.602804,  0.701032, -0.312771, -0.775907,  0.547853,  0.465379, -0.849416, -0.248828, -0.536816, -0.676261,  0.504480, -0.630069, -0.439548,  0.640165, -0.027917, -0.495203,  0.868328,  0.432099, -0.426489,  0.794605,  0.989981, -0.136510, -0.036079, -0.253410,  0.952857,  0.166876,  0.785096,  0.619291, -0.010168,  0.149813, -0.079837,  0.985486,  0.890433,  0.090247,  0.446076,  0.259486,  0.573665,  0.776901, -0.709673, -0.192609,  0.677691, -0.965848,  0.025809,  0.257821,  0.257671,  0.891226,  0.373259,  0.436461, -0.753920, -0.491026,  0.547970,  0.208489,  0.810100, -0.413090, -0.568038,  0.711822, -0.492578, -0.315391,  0.811108,  0.067166,  0.251198, -0.965603, -0.467544, -0.829246,  0.306193, -0.347723,  0.356238,  0.867285,  0.005653,  0.948548,  0.316582,  0.296470,  0.316136,  0.901201,  0.193642, -0.974967,  0.109277,  0.358826,  0.928647, -0.094118,  0.469128,  0.440206,  0.765596, -0.583937,  0.786040, -0.202875,  0.939625,  0.231665, -0.251867, -0.128732,  0.272758,  0.953431,  0.349673, -0.892220,  0.285783, -0.618095,  0.571327,  0.539948, -0.734262,  0.314706,  0.601514,  0.484894, -0.749951,  0.449946, -0.806444,  0.571244,  0.152738, -0.727557, -0.535809,  0.428451,  0.034265,  0.672876,  0.738961,  0.212188, -0.541227,  0.813664, -0.853035, -0.472747,  0.221001,  0.105040,  0.172025,  0.979476,  0.834088,  0.505431,  0.220990,  0.924929,  0.380139, -0.000543, -0.772071,  0.624658, -0.117085,  0.891589, -0.172910,  0.418534, -0.191737,  0.749749,  0.633335, -0.877507,  0.190435,  0.440131,  0.451608,  0.876276,  0.167902,  0.585681,  0.809574, -0.039583, -0.825300, -0.311085,  0.471281,  0.952550, -0.240224,  0.186927, -0.024774,  0.996916,  0.074458,  0.651358, -0.688369, -0.319188,  0.594329, -0.036078,  0.803413,  0.907728, -0.419196,  0.017467, -0.225026, -0.911956,  0.343073, -0.220473, -0.967329, -0.125168,  0.065113,  0.439306,  0.895974, -0.343914,  0.099122,  0.933755,  0.034008,  0.841290,  0.539513, -0.172437, -0.659523,  0.731638, -0.872835, -0.066953,  0.483401,  0.829927,  0.344994,  0.438407,  0.847960, -0.441306,  0.293620,  0.991588,  0.127668, -0.021301, -0.265580, -0.418631,  0.868456,  0.495538,  0.627148,  0.600939, -0.165289,  0.534584,  0.828794,  0.690738, -0.614048,  0.381872,  0.690707,  0.584372,  0.425950, -0.106074,  0.006599,  0.994336,  0.939678,  0.258455,  0.224069, -0.545520,  0.417913,  0.726469, -0.104045, -0.984948,  0.138031,  0.764111, -0.371442,  0.527413,  0.360011, -0.666803,  0.652507,  0.405602, -0.188075,  0.894491,  0.774829, -0.096475,  0.624766, -0.799613,  0.457733,  0.388714, -0.226094,  0.882004,  0.413462,  0.490679,  0.774429,  0.399367, -0.641239,  0.764842,  0.061884};
static float ps103 [309] = {  0.307533,  0.322794,  0.895113,  0.885169,  0.130899,  0.446476, -0.233530,  0.417911,  0.877960, -0.529006, -0.842498, -0.101730,  0.640094,  0.394975,  0.658995, -0.132620,  0.989785,  0.052324,  0.048589,  0.972047,  0.229705,  0.028784, -0.320235,  0.946901,  0.913282, -0.105658,  0.393386, -0.044933,  0.891352,  0.451079,  0.195244, -0.915289,  0.352314, -0.539758, -0.003430,  0.841813, -0.631090, -0.249166,  0.734603,  0.800203,  0.241348, -0.549023, -0.807408, -0.441295,  0.391601, -0.735371, -0.675289, -0.056699, -0.881365, -0.003309,  0.472424, -0.426414, -0.239027,  0.872374,  0.733400,  0.006601, -0.679766, -0.304295,  0.621580,  0.721833,  0.651975, -0.515406,  0.556134, -0.023625,  0.982328, -0.185671, -0.885071, -0.461200, -0.062802, -0.929565,  0.343076,  0.134935,  0.468355, -0.472691,  0.746463,  0.799747,  0.367603,  0.474630, -0.233683,  0.930277,  0.282802,  0.602509, -0.032711,  0.797442,  0.929518, -0.184207, -0.319475, -0.674852,  0.457682,  0.578880, -0.444396, -0.824456,  0.350406, -0.044840,  0.578578,  0.814393, -0.800686,  0.232723,  0.552034, -0.782899, -0.596042,  0.178334, -0.651681, -0.473489,  0.592554,  0.305963, -0.943133,  0.129949,  0.881505, -0.466458,  0.073251,  0.407722, -0.907784, -0.098442,  0.150257,  0.749266,  0.644998,  0.807635, -0.321434,  0.494375,  0.440150,  0.813081,  0.381008,  0.981546,  0.027285,  0.189268, -0.826116,  0.426945,  0.367765,  0.207372,  0.877320,  0.432789, -0.304350,  0.806148,  0.507441,  0.981497,  0.191470, -0.001448, -0.629400,  0.231272,  0.741869,  0.256024, -0.400767,  0.879680, -0.644463, -0.656899,  0.391346,  0.735525,  0.156655,  0.659137, -0.913139, -0.373537,  0.163240, -0.986876,  0.159616, -0.024460, -0.929119, -0.189294,  0.317658,  0.928279,  0.281951,  0.242491,  0.011740, -0.551648,  0.833994, -0.028423, -0.888772,  0.457467,  0.390973,  0.697578,  0.600438,  0.986828,  0.000707, -0.161769, -0.540006,  0.831811, -0.128391, -0.483380,  0.810671,  0.330390,  0.444708, -0.223821,  0.867259, -0.132384,  0.198867,  0.971044, -0.011355, -0.742711,  0.669516,  0.243879, -0.619294,  0.746323, -0.243503, -0.646015,  0.723444,  0.440244,  0.497772,  0.747267,  0.800426, -0.506835,  0.320057,  0.307352,  0.933234,  0.186033,  0.929260, -0.285748,  0.234145,  0.631369, -0.689233,  0.355430,  0.426731, -0.827033,  0.365947,  0.255663,  0.810383, -0.527177,  0.640000,  0.710556,  0.292419,  0.455053, -0.677070,  0.578363,  0.529190,  0.225993,  0.817854, -0.468016, -0.668007,  0.578557,  0.149988,  0.988639, -0.009806,  0.614906,  0.597720,  0.514414, -0.383886, -0.918609,  0.093752,  0.384146,  0.043964,  0.922225,  0.048768,  0.371686,  0.927077, -0.023321, -0.037563,  0.999022,  0.150242,  0.137329,  0.979065, -0.507264,  0.653045,  0.562330, -0.602266, -0.783868,  0.151086, -0.800790,  0.580296,  0.148298, -0.483271,  0.447123,  0.752681, -0.799700, -0.524212, -0.292715,  0.644745, -0.289168,  0.707592, -0.674906,  0.640322,  0.366727,  0.454710,  0.471385, -0.755668,  0.207467,  0.558469,  0.803163, -0.097724,  0.754212,  0.649319, -0.733601,  0.671043, -0.107377, -0.305685,  0.011640,  0.952061,  0.783103, -0.087535,  0.615701, -0.629560,  0.767849,  0.118588, -0.214620, -0.934867,  0.282774,  0.221465, -0.792451,  0.568308, -0.407521,  0.232212,  0.883179, -0.221096, -0.444229,  0.868203, -0.194909, -0.212365,  0.957555,  0.214941, -0.140584,  0.966456};
static float ps104 [312] = {  0.430465,  0.723375,  0.539840, -0.061692, -0.885862,  0.459829,  0.655255, -0.701076,  0.281307, -0.028043,  0.722434,  0.690871,  0.392440,  0.553661,  0.734473, -0.665191, -0.734729,  0.133023, -0.902448, -0.149218,  0.404131,  0.570577,  0.363340,  0.736495,  0.019347, -0.973906,  0.226127, -0.924884, -0.314105,  0.214306, -0.236831, -0.938990,  0.249416, -0.500865, -0.808534,  0.308881, -0.720798,  0.502844,  0.477072, -0.640321, -0.141941,  0.754878,  0.758614,  0.377968,  0.530702, -0.267767,  0.664985,  0.697205,  0.533328, -0.672458,  0.513187,  0.870474, -0.291182,  0.396848, -0.368105, -0.552447,  0.747865,  0.085602,  0.329483,  0.940273, -0.980179, -0.019286,  0.197172, -0.223324, -0.395212,  0.891029, -0.352808, -0.194518,  0.915253, -0.811215,  0.001605,  0.584746,  0.523267, -0.291339,  0.800821,  0.775340,  0.624075,  0.096841, -0.914821,  0.134882,  0.380671,  0.259102,  0.117704,  0.958651,  0.057956, -0.758889,  0.648636, -0.524199,  0.577202,  0.626142, -0.808066, -0.425317,  0.407596,  0.153195,  0.550954,  0.820354,  0.164892, -0.574496,  0.801726, -0.204298, -0.728764,  0.653579,  0.907203,  0.417834,  0.048959, -0.429140, -0.899276,  0.084502,  0.592236,  0.800235,  0.094240, -0.084245, -0.581895,  0.808889, -0.355496,  0.464597,  0.811032, -0.761601, -0.265779,  0.591038,  0.290269, -0.342719,  0.893469, -0.344447,  0.937124,  0.056172, -0.150976,  0.951524,  0.267972,  0.743607, -0.506885,  0.436024,  0.158909, -0.130816,  0.978588, -0.244578,  0.052071,  0.968230, -0.600458,  0.366304,  0.710824,  0.337011,  0.347518,  0.875017, -0.821817, -0.545414,  0.164745, -0.392794,  0.873092,  0.288832, -0.097827, -0.162963,  0.981770,  0.425958, -0.831759,  0.356000,  0.835847, -0.509012,  0.205590,  0.356565,  0.921117,  0.156223, -0.091945,  0.995532,  0.021517,  0.006970,  0.087643,  0.996128,  0.040299, -0.372569,  0.927129,  0.716531,  0.159897,  0.678982, -0.788536,  0.271713,  0.551709,  0.606729,  0.561023,  0.563146, -0.887246,  0.459326,  0.042602, -0.481376, -0.675535,  0.558506,  0.504496,  0.141300,  0.851773, -0.607869,  0.703741,  0.367755, -0.660364,  0.128522,  0.739866, -0.871633,  0.381604,  0.307627,  0.532662,  0.781669,  0.324445, -0.096269,  0.518284,  0.849773,  0.955999,  0.184857,  0.227802,  0.640710, -0.066781,  0.764873,  0.406255, -0.092278,  0.909088,  0.827887, -0.052452,  0.558437,  0.946598, -0.285425,  0.149951, -0.507308, -0.349150,  0.787866,  0.860095,  0.414640,  0.297170,  0.713832,  0.612377,  0.339762, -0.182850,  0.845981,  0.500882,  0.097153,  0.969107,  0.226700, -0.483622,  0.003622,  0.875270,  0.396554, -0.537494,  0.744207, -0.521030,  0.845843, -0.114358,  0.286445,  0.874041,  0.392429,  0.738229, -0.268459,  0.618828, -0.583948,  0.801542,  0.128592, -0.309527, -0.831806,  0.460751, -0.872359, -0.171337, -0.457857,  0.606279, -0.471538,  0.640373,  0.944300, -0.060980,  0.323387,  0.207535,  0.740539,  0.639165,  0.996016, -0.053020,  0.071697, -0.426311,  0.241447,  0.871758,  0.052513,  0.880455,  0.471213, -0.628012, -0.480927,  0.611809, -0.172776,  0.294284,  0.939971,  0.270368, -0.945056,  0.183765,  0.769411, -0.602506, -0.212115, -0.410876,  0.757936,  0.506669, -0.734123,  0.678053, -0.036163, -0.982540, -0.185570,  0.013380,  0.186531, -0.883903,  0.428861, -0.673324, -0.631863,  0.383906,  0.305611, -0.735957,  0.604127, -0.965955,  0.228545,  0.121236,  0.166078,  0.985983, -0.015971};
static float ps105 [315] = { -0.660053,  0.282320,  0.696151,  0.228505,  0.152326,  0.961552,  0.443662,  0.795300,  0.413112, -0.888348, -0.107785,  0.446341, -0.173847,  0.933987,  0.312162,  0.088304, -0.434199,  0.896478,  0.668816, -0.665909,  0.330530, -0.209140,  0.628920,  0.748813, -0.741690, -0.179777,  0.646201,  0.468896, -0.830915,  0.299528,  0.695031, -0.275391,  0.664148,  0.608804, -0.042373,  0.792188, -0.447356,  0.461154,  0.766296, -0.251971,  0.187220,  0.949452, -0.195148,  0.979178,  0.055919, -0.790035, -0.370868,  0.488162, -0.440060, -0.027983,  0.897532,  0.531100, -0.503684,  0.681348, -0.415833, -0.897380,  0.147620,  0.666822,  0.195458,  0.719127,  0.060911, -0.639173,  0.766647, -0.982651,  0.171235,  0.071244,  0.672035,  0.420277,  0.609702, -0.588754,  0.701759,  0.401128, -0.980093, -0.047725,  0.192717,  0.767205,  0.639479, -0.049627, -0.614678, -0.437908,  0.656054, -0.926402,  0.138063,  0.350310,  0.605516,  0.760926,  0.233113,  0.594519,  0.804021, -0.009868, -0.402315, -0.494725,  0.770318, -0.200999, -0.839426,  0.504939, -0.978478, -0.206261,  0.006122, -0.219562, -0.975580, -0.005981,  0.232102, -0.948230,  0.216770,  0.788489, -0.035805,  0.614005,  0.007030,  0.323228,  0.946295,  0.787360,  0.537899, -0.301211,  0.829767,  0.196524,  0.522365,  0.267997, -0.849910,  0.453686, -0.623231, -0.738138,  0.258330,  0.035977,  0.983717,  0.176086,  0.250317,  0.787397,  0.563336,  0.903436, -0.361105, -0.231100, -0.472787,  0.227035,  0.851427,  0.478702,  0.424966,  0.768276, -0.019359,  0.085332,  0.996165,  0.287744, -0.700820,  0.652728, -0.818404, -0.414687, -0.397805, -0.186511, -0.684940,  0.704323,  0.463349,  0.194653,  0.864533, -0.629688,  0.509368,  0.586546,  0.408648,  0.901459,  0.142753,  0.844367, -0.446451,  0.296185,  0.391647, -0.046709,  0.918929,  0.950106, -0.227608,  0.213292, -0.225810,  0.419753,  0.879100, -0.422345,  0.665319,  0.615610, -0.789928,  0.602638, -0.113318, -0.762360,  0.535136,  0.363917,  0.623525, -0.777233,  0.084405, -0.499318,  0.693798, -0.518966,  0.041413, -0.807893,  0.587873,  0.917315, -0.397240,  0.027082,  0.550350,  0.241053, -0.799380, -0.325787, -0.287908,  0.900540, -0.940042, -0.189712, -0.283427,  0.280815, -0.280884,  0.917740,  0.001647, -0.994166,  0.107846,  0.262999,  0.605298,  0.751296,  0.317859, -0.507979,  0.800577,  0.133255, -0.101889,  0.985831,  0.707724, -0.489663,  0.509271, -0.413968,  0.890731,  0.187693, -0.993101,  0.004183, -0.117187,  0.024244,  0.542975,  0.839399,  0.894764,  0.437719, -0.088311, -0.806202,  0.084775,  0.585536,  0.503409, -0.282360,  0.816610,  0.921103, -0.030969,  0.388085,  0.474986,  0.629361,  0.615056,  0.254023,  0.390463,  0.884879, -0.641864,  0.025712,  0.766387,  0.236192,  0.919254,  0.314939, -0.417682, -0.678246,  0.604587,  0.625405, -0.763001, -0.163395, -0.379984,  0.824666,  0.418972, -0.426196, -0.814755,  0.393104,  0.033660,  0.734446,  0.677832,  0.778449,  0.595673,  0.197968, -0.655263, -0.613722, -0.440428, -0.062937, -0.258043,  0.964081,  0.623813,  0.614690, -0.482714, -0.798311,  0.588261,  0.129029, -0.181697,  0.805357,  0.564257, -0.205071, -0.941390,  0.267827,  0.025037, -0.929344,  0.368364, -0.419100,  0.906258, -0.055246, -0.905690, -0.394711, -0.154689, -0.810122,  0.323294,  0.489064,  0.033242,  0.890975,  0.452834,  0.846794, -0.257586,  0.465391, -0.161658, -0.488147,  0.857659, -0.211941, -0.067897,  0.974921, -0.916804, -0.286190,  0.278505};
static float ps106 [318] = {  0.008146, -0.774933,  0.631991, -0.760255,  0.503664,  0.410287, -0.866262, -0.369114, -0.336666, -0.833553,  0.233310,  0.500755,  0.806285, -0.587363,  0.070070,  0.391804,  0.753060,  0.528573, -0.337652,  0.752343,  0.565660,  0.258827,  0.043355,  0.964950,  0.310822, -0.742526,  0.593333,  0.707807,  0.690422,  0.149419, -0.368652,  0.177911,  0.912383, -0.480601, -0.715685,  0.506772, -0.131621,  0.203582,  0.970170, -0.916530, -0.370584,  0.150464, -0.286942, -0.419053,  0.861429,  0.615717, -0.120877,  0.778640,  0.495195,  0.085087,  0.864605,  0.404319,  0.546452,  0.733427, -0.264465, -0.039946,  0.963568, -0.515002,  0.844294, -0.148121,  0.210034,  0.707383,  0.674903,  0.000000,  0.000000,  1.000000, -0.061801,  0.429630,  0.900888,  0.845908, -0.293453,  0.445337, -0.799896, -0.589996,  0.109870, -0.410926,  0.841555,  0.350608,  0.734544,  0.569395,  0.369101,  0.566008,  0.351456,  0.745730, -0.763826, -0.302901,  0.569931, -0.235343,  0.621164,  0.747509, -0.309926, -0.881278,  0.356784,  0.947176,  0.148470,  0.284280,  0.792289, -0.087570,  0.603829,  0.688829,  0.133925,  0.712445,  0.677897, -0.689112,  0.256087, -0.474592,  0.537709,  0.696872, -0.982819,  0.150150,  0.107340, -0.310461,  0.411557,  0.856875, -0.151264, -0.975015,  0.162678, -0.158651,  0.896268,  0.414166, -0.808087,  0.560970,  0.179747,  0.010967,  0.627129,  0.778838,  0.027728, -0.999334, -0.023709,  0.924748, -0.379827, -0.023927,  0.346624, -0.883602,  0.314800,  0.907263,  0.201203, -0.369312, -0.681017,  0.402790,  0.611536, -0.523494, -0.806479,  0.274854, -0.545954,  0.295245,  0.784069, -0.550392, -0.378713,  0.744073, -0.982677, -0.113845,  0.146239, -0.959715, -0.272323, -0.069194, -0.690252, -0.642984,  0.331849, -0.133634, -0.240828,  0.961324, -0.863072,  0.440058, -0.247901,  0.499120, -0.354260,  0.790809, -0.532581,  0.014017,  0.846263,  0.386907, -0.161178,  0.907923,  0.159992, -0.872085,  0.462462, -0.388159, -0.917038,  0.091504, -0.709411,  0.125899,  0.693460,  0.367213,  0.293527,  0.882608,  0.097161, -0.969041,  0.226980, -0.285692,  0.956147, -0.064524, -0.403611, -0.582434,  0.705598,  0.843535,  0.150589,  0.515530, -0.008694, -0.440406,  0.897756, -0.245442, -0.770569,  0.588203,  0.315008,  0.892960,  0.321547,  0.226088,  0.969385,  0.095794, -0.568239,  0.651539,  0.502595,  0.512161, -0.740262,  0.435549, -0.424247, -0.213459,  0.880028, -0.902272,  0.328064,  0.279785, -0.937776,  0.062032,  0.341654, -0.220107,  0.957859,  0.184550,  0.123192,  0.249363,  0.960542, -0.630731, -0.773494,  0.062338,  0.996119,  0.006144,  0.087801,  0.247101, -0.392459,  0.885955, -0.630116,  0.724630,  0.279043, -0.150212, -0.616628,  0.772791,  0.860877,  0.495684,  0.114840, -0.831844, -0.041194,  0.553479,  0.485009,  0.864166,  0.134104,  0.924099, -0.073913,  0.374937,  0.724477, -0.530201,  0.440476, -0.080271, -0.906086,  0.415409, -0.666255, -0.148712,  0.730746, -0.458728,  0.881560,  0.111444,  0.130438,  0.863969,  0.486358,  0.555881,  0.754258,  0.349416,  0.693550, -0.337114,  0.636665,  0.736204,  0.373880,  0.564107,  0.128769, -0.201585,  0.970970,  0.556716, -0.563767,  0.610109,  0.582796,  0.579959,  0.569207,  0.043318,  0.961468,  0.271482,  0.358585, -0.561484,  0.745757,  0.121685, -0.620043,  0.775074, -0.818746, -0.442216,  0.366197, -0.071696,  0.788694,  0.610591,  0.200733,  0.478006,  0.855112,  0.959404, -0.222272,  0.173607, -0.639155, -0.526279,  0.560813,  0.659828, -0.750659, -0.033736};
static float ps107 [321] = {  0.567828, -0.311393,  0.761975, -0.587614, -0.806896, -0.060233, -0.727714, -0.673344,  0.130542,  0.145223, -0.634488,  0.759168,  0.603371,  0.725247,  0.331601,  0.490557, -0.744278,  0.453216, -0.628589,  0.740597,  0.237472, -0.213097,  0.870355,  0.443927,  0.245060,  0.421384,  0.873144, -0.433351,  0.828663,  0.354294,  0.871667,  0.448463,  0.197681, -0.298308,  0.360910,  0.883604, -0.708717,  0.216449,  0.671468,  0.922006,  0.028756,  0.386106, -0.442601,  0.529550,  0.723658,  0.727559, -0.359905,  0.584061,  0.497237,  0.404278,  0.767668,  0.025290,  0.866909,  0.497824, -0.076153, -0.517385,  0.852358, -0.452808, -0.786915,  0.419200,  0.400381,  0.214565,  0.890874,  0.810694,  0.487831, -0.323722, -0.485663, -0.628704,  0.607340,  0.693257,  0.362729,  0.622753,  0.897960, -0.393938, -0.196163,  0.854892, -0.387023,  0.345503,  0.090780, -0.960324,  0.263698, -0.848867,  0.167105, -0.501498,  0.950987, -0.189970,  0.243999, -0.488644, -0.426204,  0.761299,  0.000000,  0.000000,  1.000000, -0.291980, -0.911374,  0.290078,  0.796902,  0.100449,  0.595699, -0.336545,  0.721371,  0.605278, -0.702462,  0.563935, -0.434193,  0.083420,  0.598912,  0.796458, -0.600402,  0.024277,  0.799330,  0.457741, -0.870504,  0.180820,  0.129059,  0.218395,  0.967289, -0.196822,  0.568435,  0.798839, -0.290999, -0.358210,  0.887133,  0.843254,  0.301052,  0.445297, -0.648237, -0.248718,  0.719673, -0.239879, -0.084593,  0.967110, -0.924119, -0.291888,  0.246587,  0.374130, -0.250935,  0.892781,  0.416091,  0.762400,  0.495596,  0.329006, -0.709603,  0.623072, -0.521878,  0.296488,  0.799836,  0.251910, -0.948008, -0.194479, -0.075670, -0.275455,  0.958331,  0.194430,  0.972339,  0.129434,  0.547047, -0.550999,  0.630190,  0.151623, -0.186406,  0.970702,  0.634462, -0.727644,  0.260751, -0.771605, -0.063569,  0.632918,  0.944997,  0.221419,  0.240737, -0.674959, -0.470421,  0.568450, -0.391752,  0.105591,  0.913992, -0.263562,  0.961333, -0.079840,  0.993607,  0.017199,  0.111575,  0.124174, -0.831872,  0.540897, -0.869398, -0.490283,  0.061396,  0.512757, -0.043564,  0.857428, -0.557685,  0.661954,  0.500803,  0.196007,  0.753946,  0.627014,  0.916715,  0.086939, -0.389969,  0.916421, -0.393680,  0.072027,  0.370187, -0.493053,  0.787312,  0.745103,  0.532323,  0.401812, -0.150775,  0.184520,  0.971195, -0.536134, -0.822842,  0.188391,  0.651992, -0.758073,  0.015244, -0.855386,  0.123274,  0.503109,  0.988111,  0.053865, -0.143999, -0.802138,  0.589946,  0.092405, -0.652200,  0.456151,  0.605443, -0.981676,  0.187996,  0.031147, -0.060605, -0.730842,  0.679851,  0.153241, -0.420746,  0.894142,  0.571241,  0.584818,  0.575909, -0.946713,  0.163179,  0.277680, -0.458255, -0.167053,  0.872981, -0.283197, -0.596259,  0.751182, -0.963608, -0.267317,  0.000713,  0.295975, -0.879771,  0.372023,  0.050621, -0.998377, -0.026104, -0.126049, -0.982572,  0.136618,  0.800028, -0.575310,  0.170215, -0.460654,  0.883312,  0.086939, -0.745704,  0.563926,  0.354842, -0.029857,  0.406398,  0.913208,  0.354335,  0.598480,  0.718518, -0.816372, -0.286315,  0.501558, -0.088567,  0.738433,  0.668485,  0.369589,  0.928571, -0.034053, -0.751400, -0.645539, -0.136667, -0.082464, -0.900401,  0.427174,  0.284573,  0.015513,  0.958529, -0.259165, -0.785148,  0.562474, -0.821944,  0.356138,  0.444493, -0.652527, -0.654494,  0.381898,  0.622090,  0.160859,  0.766243,  0.425476,  0.875481,  0.229136,  0.702801, -0.103760,  0.703778, -0.012580,  0.961802,  0.273456,  0.236703,  0.894876,  0.378377};
static float ps108 [324] = {  0.845597,  0.479498,  0.234623, -0.988546, -0.007665, -0.150722, -0.140823, -0.466127,  0.873438,  0.372367,  0.771573,  0.515769, -0.567468,  0.821768, -0.051751,  0.116797,  0.239394,  0.963872, -0.697794, -0.681242, -0.221344, -0.925848,  0.213979, -0.311478, -0.843060, -0.044533,  0.535972, -0.098726,  0.803662,  0.586839,  0.707402, -0.588584, -0.391346,  0.666066, -0.691740,  0.279021,  0.837134, -0.370947, -0.402001,  0.920882,  0.036858,  0.388094, -0.479067, -0.060023,  0.875724, -0.246452, -0.033095,  0.968590,  0.664253, -0.335397,  0.668040, -0.340946,  0.939091, -0.043167, -0.158688, -0.983950,  0.081613,  0.238832,  0.441767,  0.864755, -0.136908,  0.209042,  0.968276, -0.000000,  0.000000,  1.000000, -0.750011, -0.478952,  0.456167, -0.168256, -0.670313,  0.722752, -0.569341, -0.820896, -0.044503,  0.857307,  0.263159, -0.442462,  0.592117,  0.147565,  0.792226,  0.485892, -0.719412,  0.496342, -0.179961,  0.972058,  0.150726,  0.856173,  0.274385,  0.437815, -0.014020,  0.446976,  0.894436, -0.206705, -0.930138,  0.303507, -0.075657,  0.993150, -0.089044,  0.261061, -0.836985,  0.480940, -0.366663, -0.500956,  0.783965,  0.808182, -0.357930,  0.467684, -0.141324,  0.630791,  0.762974, -0.601218, -0.670345,  0.434942,  0.742101,  0.494344,  0.452669, -0.045565,  0.800415, -0.597713,  0.312632, -0.477078,  0.821375,  0.550027,  0.551089,  0.627511,  0.508307, -0.530124,  0.678669, -0.832254,  0.486740, -0.265402, -0.823905,  0.536312,  0.183165,  0.946061,  0.249609,  0.206552,  0.689065,  0.332515,  0.643913,  0.245498, -0.231240,  0.941413,  0.659677, -0.097172,  0.745241, -0.190533, -0.827593,  0.528003, -0.754687,  0.654112, -0.050831, -0.437838,  0.881874,  0.174915,  0.614646,  0.763199, -0.199342, -0.369809,  0.570167,  0.733588, -0.921533,  0.133632,  0.364580,  0.764190,  0.644970, -0.005300,  0.019871, -0.929408,  0.368519,  0.094589, -0.415532,  0.904647, -0.269663,  0.888987,  0.370113, -0.044864, -0.228663,  0.972471,  0.468737,  0.839670,  0.274300, -0.709422, -0.292453,  0.641242, -0.900872, -0.434054, -0.005160, -0.614901,  0.158009,  0.772612, -0.512491,  0.768061,  0.383972,  0.585019,  0.686961,  0.431090, -0.023105,  0.925160,  0.378874, -0.520062, -0.294585,  0.801720, -0.464139,  0.840633, -0.279126,  0.954183,  0.101541, -0.281467, -0.096973, -0.646977, -0.756318,  0.474969,  0.386695,  0.790488, -0.786614,  0.181207,  0.590256, -0.892587,  0.449037, -0.040675, -0.297508, -0.270501,  0.915597,  0.068192,  0.986008,  0.152111, -0.552099,  0.600616,  0.578314,  0.072420, -0.626229,  0.776268,  0.292205, -0.681187,  0.671268, -0.389199,  0.176319,  0.904122, -0.325172, -0.940109, -0.102261,  0.992364, -0.064819, -0.104938,  0.331601,  0.621303,  0.709946,  0.236032, -0.935729,  0.262106,  0.672913, -0.556929,  0.486845,  0.457617, -0.051838,  0.887637, -0.683048, -0.067973,  0.727203, -0.260797,  0.400959,  0.878190,  0.231719,  0.001977,  0.972781, -0.969171,  0.232479, -0.081613,  0.781745,  0.102709,  0.615082, -0.334290,  0.752663,  0.567228,  0.228568,  0.910573,  0.344403, -0.396699, -0.691055,  0.604212,  0.981319,  0.188610, -0.037940, -0.905947, -0.359528,  0.223607,  0.476931, -0.291719,  0.829118,  0.823328, -0.130846,  0.552277,  0.783171,  0.575738, -0.234880,  0.418775,  0.823829, -0.382013,  0.140942,  0.809731,  0.569623, -0.408921, -0.902348,  0.136206, -0.655973,  0.734335,  0.174502, -0.687641,  0.399876,  0.606011, -0.572378, -0.508766,  0.643071,  0.936384, -0.303995, -0.175422, -0.507932,  0.373552,  0.776186,  0.366238,  0.202420,  0.908238};
static float ps109 [327] = {  0.610026, -0.233011,  0.757346,  0.731705,  0.574946,  0.366122, -0.008585,  0.044474,  0.998974, -0.105568,  0.938088,  0.329919, -0.029742, -0.903237,  0.428110, -0.371585, -0.662425,  0.650475,  0.077914,  0.877016,  0.474102,  0.215616,  0.943187, -0.252803, -0.727064,  0.127484,  0.674630,  0.150699,  0.957837,  0.244617,  0.460917,  0.413076,  0.785445, -0.778696,  0.517881,  0.354162, -0.009458,  0.984039, -0.177701, -0.193713, -0.583244,  0.788861, -0.904694,  0.302605,  0.299931,  0.463362,  0.187468,  0.866113,  0.595572,  0.767030,  0.238661,  0.906877, -0.059250,  0.417209, -0.362540,  0.848445,  0.385625,  0.454558, -0.890716, -0.001114, -0.607300,  0.314958,  0.729375, -0.782908,  0.332914,  0.525569, -0.605596, -0.511622,  0.609506, -0.776550,  0.523142, -0.351130,  0.210343, -0.086800,  0.973767, -0.400634, -0.903335, -0.153228,  0.715899, -0.399026,  0.572947, -0.011234, -0.465862,  0.884786,  0.006121,  0.529310,  0.848407, -0.269233, -0.827376,  0.492912, -0.730226,  0.664681,  0.158017,  0.007173,  0.295157,  0.955422,  0.235493, -0.967528,  0.091834, -0.535472, -0.699099,  0.473847,  0.881414,  0.389322, -0.267466, -0.209150,  0.644668,  0.735296,  0.262932,  0.766005,  0.586603,  0.784889,  0.607035,  0.124330,  0.023091,  0.730384,  0.682647,  0.870019, -0.299867,  0.391340, -0.876313,  0.099638,  0.471325, -0.648234,  0.759384, -0.055935, -0.444783, -0.208271,  0.871086, -0.174113,  0.817985,  0.548256,  0.163971, -0.315119,  0.934780,  0.780856, -0.153529,  0.605551, -0.970673, -0.194577,  0.141186, -0.624989, -0.056532,  0.778584,  0.643526,  0.491435,  0.586827, -0.972648,  0.117826, -0.200184, -0.428687, -0.451346,  0.782633,  0.804635, -0.583664,  0.109078,  0.163355, -0.808080,  0.565970, -0.414448,  0.296274,  0.860497,  0.444110, -0.050794,  0.894531, -0.428041,  0.517089,  0.741215, -0.794624, -0.113345,  0.596427, -0.911959, -0.152467,  0.380900,  0.656949,  0.269158,  0.704253,  0.233636,  0.149604,  0.960746, -0.237129, -0.345888,  0.907817, -0.283890,  0.944115,  0.167493,  0.805510,  0.107759,  0.582702,  0.806305,  0.341775,  0.482765, -0.464623,  0.070697,  0.882682,  0.227686,  0.591803,  0.773258, -0.735197, -0.559290,  0.382988, -0.635103, -0.295033,  0.713862, -0.219185,  0.174167,  0.960012, -0.042973,  0.995710,  0.081951, -0.915236, -0.402915,  0.001524, -0.624532,  0.522689,  0.580307,  0.197724,  0.980255, -0.002328,  0.401130, -0.289186,  0.869175,  0.410351, -0.787854,  0.459236,  0.923661,  0.163297,  0.346675,  0.800004,  0.587082, -0.123811, -0.999865,  0.014452,  0.007843, -0.210206,  0.428355,  0.878821, -0.413150,  0.702815,  0.579101, -0.538356,  0.815993,  0.210543,  0.631532, -0.715356,  0.299054, -0.967147,  0.060205,  0.246987,  0.881673,  0.402261,  0.246655,  0.515106, -0.470042,  0.716747,  0.531046,  0.712562,  0.458525,  0.090781, -0.652837,  0.752039,  0.442101,  0.612880,  0.654924,  0.355594, -0.664709,  0.657050,  0.647258,  0.019856,  0.762012, -0.975313, -0.183368, -0.123051, -0.092754, -0.761754,  0.641192,  0.589300, -0.615291,  0.523587,  0.441451, -0.865652,  0.236150,  0.272952, -0.504702,  0.819007, -0.635439, -0.736610,  0.231566,  0.343859,  0.859048,  0.379206,  0.209447, -0.916189,  0.341656, -0.431239, -0.846234,  0.312922,  0.238483,  0.379724,  0.893832, -0.915893,  0.369008, -0.158027, -0.431515, -0.898494,  0.080646, -0.632172, -0.774809, -0.005381, -0.050035, -0.199852,  0.978548, -0.261095, -0.070861,  0.962709, -0.874816,  0.472996,  0.104749,  0.788310,  0.347915, -0.507467, -0.966270,  0.251969,  0.053231, -0.598588,  0.690880,  0.405434};
static float ps110 [330] = {  0.098197, -0.720142,  0.686843,  0.676501,  0.724281,  0.133282, -0.546768,  0.741721,  0.388452, -0.226819,  0.616965,  0.753597, -0.893995,  0.378121,  0.240410,  0.713458,  0.497591,  0.493336, -0.439062,  0.254572,  0.861637,  0.037663, -0.493429,  0.868970,  0.761574,  0.263801,  0.591959, -0.422529, -0.016964,  0.906191,  0.225525,  0.117554,  0.967119,  0.642823, -0.136393,  0.753774, -0.247430,  0.380401,  0.891108, -0.609915,  0.563644,  0.557054,  0.168797,  0.566335,  0.806705, -0.714497, -0.690841,  0.110601,  0.368177, -0.703182,  0.608260,  0.938196,  0.318093, -0.136400, -0.395107, -0.289964,  0.871672, -0.440631,  0.495782,  0.748361,  0.630646,  0.126672,  0.765663,  0.238424, -0.121670,  0.963509, -0.788697,  0.189565,  0.584827,  0.000000,  0.000000,  1.000000,  0.204172, -0.846961,  0.490888, -0.128203,  0.989060, -0.072974,  0.796331,  0.551168,  0.249138, -0.106743, -0.986930,  0.120726, -0.200238, -0.906126,  0.372612, -0.230650,  0.125265,  0.964940, -0.649793,  0.745791,  0.146850, -0.913132,  0.169357,  0.370820,  0.378985,  0.803161,  0.459676, -0.568589, -0.411039,  0.712568, -0.041436,  0.485172,  0.873437, -0.322342,  0.863397,  0.388125, -0.752151, -0.266326,  0.602776, -0.720105, -0.477841,  0.503107,  0.452457, -0.004005,  0.891777,  0.394844,  0.486812,  0.779174,  0.460038, -0.793793,  0.397816, -0.057122,  0.927627,  0.369114, -0.607100,  0.113747,  0.786443,  0.516177,  0.616838,  0.594200, -0.648694, -0.662206,  0.375072, -0.360895,  0.932302, -0.023838,  0.247577, -0.355595,  0.901253,  0.511941,  0.858950,  0.011015,  0.455493,  0.856539,  0.242625,  0.788098, -0.010118,  0.615466,  0.609900,  0.698415,  0.374483,  0.254457, -0.570435,  0.780933,  0.203654,  0.350301,  0.914229,  0.658534, -0.366403,  0.657329,  0.663751, -0.674280,  0.323697, -0.526341,  0.836846, -0.150511, -0.183232, -0.398537,  0.898663,  0.034139,  0.990590,  0.132539,  0.848643,  0.528952,  0.003844, -0.935014,  0.354226, -0.016508, -0.333773, -0.927084,  0.170617, -0.291482,  0.918719, -0.266446,  0.468050, -0.477289,  0.743723, -0.416340, -0.800974,  0.430234,  0.000806,  0.707966,  0.706246,  0.768876, -0.469462,  0.434091, -0.745043, -0.033709,  0.666164, -0.055020, -0.834935,  0.547591, -0.701293,  0.708790, -0.076188,  0.594840,  0.385576,  0.705334, -0.820852, -0.509865,  0.257373, -0.277151, -0.740465,  0.612290,  0.995189,  0.096005, -0.019523,  0.110927,  0.841773,  0.528311,  0.925884, -0.099970,  0.364341, -0.631257,  0.351403,  0.691398, -0.740641,  0.588735,  0.323794,  0.905373, -0.322356,  0.276380,  0.969552,  0.113825,  0.216825, -0.017932,  0.248702,  0.968414, -0.985030,  0.127262, -0.116276, -0.512568, -0.623983,  0.589847,  0.460016, -0.253124,  0.851066,  0.867998,  0.348255,  0.353975,  0.204174,  0.926866,  0.315013, -0.538403, -0.812240,  0.224472, -0.398001,  0.702790,  0.589645, -0.123840, -0.636495,  0.761274, -0.964224, -0.078814,  0.253102,  0.015906, -0.250917,  0.967878, -0.825920,  0.557369,  0.084831, -0.354492, -0.533863,  0.767676, -0.935508, -0.334193, -0.114631,  0.887516,  0.124044,  0.443766,  0.433471,  0.246932,  0.866676,  0.278856,  0.957152,  0.078098, -0.786608,  0.400325,  0.470094, -0.875886, -0.297216,  0.380113, -0.211426, -0.138826,  0.967485, -0.043858,  0.945522, -0.322589,  0.584243, -0.597399,  0.549340, -0.879610, -0.058220,  0.472120, -0.211546,  0.960778,  0.179316, -0.979793,  0.149452,  0.132930,  0.814215, -0.240018,  0.528626, -0.585827, -0.168086,  0.792814,  0.282379,  0.704763,  0.650823, -0.165971,  0.807846,  0.565543, -0.829343,  0.528678, -0.180803, -0.460752,  0.869919,  0.175923};
static float ps111 [333] = {  0.504365,  0.022619,  0.863194,  0.392879, -0.918347,  0.047789,  0.875803, -0.482644,  0.004820,  0.796504,  0.217017,  0.564344,  0.050794,  0.887623,  0.457760, -0.939385,  0.097776,  0.328627, -0.697691,  0.384488,  0.604480, -0.290807, -0.941080,  0.172626,  0.259798,  0.004237,  0.965654,  0.579708, -0.813041, -0.053871, -0.000000,  0.000000,  1.000000,  0.723288,  0.517021, -0.457760,  0.682011, -0.642330,  0.349675, -0.965557, -0.123855,  0.228824, -0.214359, -0.429011,  0.877496, -0.410993,  0.882454,  0.228824, -0.391889,  0.786721,  0.476962,  0.376407, -0.207433,  0.902934, -0.555447,  0.606668,  0.568711,  0.469873, -0.752513,  0.461458, -0.113525, -0.217167,  0.969511, -0.276115,  0.431892,  0.858621, -0.512604,  0.002761,  0.858621,  0.135176, -0.221902,  0.965654,  0.779808, -0.429742,  0.455215, -0.168502,  0.793377,  0.584945,  0.308638,  0.626748,  0.715493, -0.268732, -0.913322, -0.305984, -0.272614, -0.765381,  0.582986, -0.852037,  0.469547,  0.231426,  0.098882, -0.908743,  0.405472,  0.907301, -0.233519,  0.349675,  0.580983, -0.576670,  0.574378,  0.721937,  0.022057,  0.691607,  0.293259,  0.797779,  0.526828,  0.996985, -0.055854, -0.053871,  0.733873, -0.670911,  0.106348, -0.723716, -0.035363,  0.689191, -0.599452, -0.216792,  0.770493,  0.335004, -0.886453,  0.319334, -0.501708,  0.423893,  0.754058, -0.194355,  0.910365,  0.365324,  0.240876,  0.968596,  0.061643, -0.149802,  0.214786,  0.965104,  0.887150,  0.004674,  0.461458,  0.665668,  0.618461,  0.417603, -0.517831, -0.674020,  0.526828, -0.839987, -0.542575,  0.005950, -0.261594,  0.011929,  0.965104, -0.809554,  0.145567,  0.568711,  0.241941, -0.789291,  0.564344, -0.365023, -0.595674,  0.715493,  0.959163, -0.262100,  0.106348,  0.385956,  0.221498,  0.895531,  0.473376,  0.439646,  0.763300, -0.058865, -0.973041,  0.222995, -0.760735, -0.281286,  0.584945, -0.210622,  0.970322,  0.118798, -0.735043,  0.661570,  0.148447, -0.402437,  0.221778,  0.888177,  0.918574,  0.364446,  0.152974,  0.027132,  0.970350,  0.240177,  0.932825,  0.340183, -0.118798, -0.029476,  0.406863,  0.913013,  0.248207,  0.410358,  0.877496, -0.599732, -0.466330,  0.650276,  0.850977, -0.468963,  0.236456,  0.018879, -0.444598,  0.895531,  0.611671,  0.233672,  0.755815, -0.010069, -0.999932, -0.005950,  0.874419,  0.088875, -0.476962,  0.663626, -0.365716,  0.652574,  0.797832, -0.183178,  0.574378, -0.690221, -0.720968,  0.061643,  0.821085,  0.401762,  0.405472, -0.854027,  0.315901,  0.413333,  0.485352,  0.793945,  0.366175, -0.012367, -0.796443,  0.604587, -0.439540, -0.389486,  0.809386, -0.873448, -0.321912,  0.365324,  0.791190,  0.569466,  0.222995,  0.928352,  0.190234,  0.319334, -0.628613, -0.714996,  0.305984,  0.073930,  0.756092,  0.650276, -0.504266, -0.855413,  0.118256,  0.453857,  0.883193,  0.118256,  0.366938, -0.622122,  0.691607,  0.976682, -0.033353,  0.212083, -0.136886,  0.622578,  0.770493, -0.359715, -0.192385,  0.913013,  0.501468,  0.639263,  0.582986,  0.468678, -0.413025,  0.780866,  0.549842, -0.807895,  0.212083, -0.666725, -0.435834, -0.604587,  0.122964,  0.211965,  0.969511,  0.723205, -0.553291, -0.413333, -0.626330,  0.197754,  0.754058, -0.167328, -0.893089,  0.417603,  0.986228,  0.158339,  0.047789,  0.129141, -0.641924,  0.755815,  0.951892, -0.268078, -0.148447,  0.250258, -0.438482,  0.863194,  0.805864,  0.541201, -0.240177, -0.411993, -0.834373,  0.366175,  0.094523,  0.579621,  0.809386, -0.584386,  0.741955,  0.328627, -0.182499,  0.971233, -0.152974,  0.599501, -0.175635,  0.780866,  0.640291,  0.748484,  0.172626, -0.356639,  0.630733,  0.689191, -0.118857, -0.635017,  0.763300};
static float ps112 [336] = { -0.709460,  0.012066,  0.704643,  0.135896, -0.620790,  0.772109,  0.825207,  0.533834,  0.184540, -0.795102, -0.176393,  0.580257, -0.950507,  0.273572,  0.147293, -0.262830,  0.751240,  0.605442,  0.257674, -0.011975,  0.966158,  0.510212, -0.712197,  0.482140, -0.724884,  0.672759,  0.148116,  0.912894,  0.315280,  0.259275,  0.514605,  0.381285,  0.767986,  0.727951, -0.521997,  0.444529, -0.997048,  0.076726,  0.002714,  0.449620, -0.851201,  0.270737, -0.217741,  0.946721, -0.237292, -0.417842,  0.811115,  0.409268, -0.242571, -0.036221,  0.969457, -0.336031,  0.177487,  0.924977, -0.583067, -0.725193,  0.366235,  0.900095,  0.142227,  0.411825, -0.807746, -0.392492,  0.439883, -0.866316,  0.049072,  0.497080, -0.879356,  0.274817,  0.388856, -0.884042,  0.466810,  0.023645, -0.674494, -0.390662,  0.626452,  0.689375, -0.103706,  0.716943,  0.094874,  0.913710,  0.395136, -0.170032,  0.884720,  0.434004,  0.757929, -0.647064,  0.082770,  0.031531, -0.913632,  0.405317, -0.309707,  0.927253,  0.210436,  0.559081, -0.303127,  0.771714,  0.319817,  0.860955,  0.395569,  0.611298,  0.532161,  0.585764, -0.194679,  0.425594,  0.883725,  0.653011, -0.698581,  0.292509,  0.785318,  0.096932,  0.611457,  0.101024, -0.213382,  0.971732, -0.439881, -0.867503,  0.232259,  0.738127, -0.320752,  0.593537, -0.818498, -0.570611,  0.066813,  0.359483, -0.237471,  0.902430,  0.335557, -0.687062,  0.644475, -0.213954, -0.916537,  0.337910, -0.354999, -0.646926,  0.674880, -0.032945, -0.984213,  0.173895, -0.686507, -0.718889, -0.109117,  0.200190, -0.415161,  0.887449,  0.391924,  0.191931,  0.899754, -0.981789, -0.149744,  0.116904,  0.679802,  0.638024,  0.361656,  0.285601,  0.418395,  0.862193, -0.584277,  0.809504, -0.057657,  0.150958,  0.215303,  0.964809,  0.387818,  0.591977,  0.706513, -0.584267,  0.195603,  0.787636,  0.933704,  0.357394,  0.021584, -0.032436, -0.459797,  0.887431,  0.522246,  0.811248,  0.262936, -0.900664, -0.377421,  0.215309, -0.378849,  0.924945, -0.030832, -0.487103, -0.015283,  0.873211, -0.381823, -0.244865,  0.891208, -0.983076, -0.120570, -0.137928,  0.797384,  0.423718,  0.429700, -0.299483, -0.937578, -0.176797, -0.348567,  0.576802,  0.738783, -0.425544,  0.370335,  0.825690, -0.511918,  0.643037,  0.569599,  0.875714, -0.301378,  0.377222, -0.382666, -0.788062,  0.482208, -0.832926,  0.486609,  0.263525,  0.843321, -0.484542,  0.232443,  0.962956, -0.057594, -0.263435,  0.957836, -0.073358,  0.277792, -0.263828, -0.468205,  0.843314, -0.734941,  0.476868,  0.482140, -0.068544,  0.969301,  0.236131,  0.480552,  0.716700,  0.505381,  0.496258, -0.047006,  0.866902,  0.088225, -0.788417,  0.608781, -0.651134, -0.747779,  0.129810, -0.099395,  0.216294,  0.971256, -0.597272,  0.437778,  0.672024,  0.917144,  0.168286, -0.361285,  0.000000,  0.000000,  1.000000,  0.719231,  0.316362,  0.618564, -0.638906,  0.674764,  0.369449,  0.607840,  0.148753,  0.780002, -0.004759,  0.796211,  0.605000,  0.143661,  0.620832,  0.770668, -0.580778, -0.594138,  0.556503,  0.855121, -0.105425,  0.507596,  0.043473,  0.433634,  0.900040, -0.140245, -0.252043,  0.957500, -0.602597, -0.205050,  0.771253, -0.271874, -0.958399,  0.086923,  0.393309, -0.490606,  0.777569, -0.098398,  0.631087,  0.769446,  0.097330,  0.992072,  0.079495, -0.481928, -0.442770,  0.756109, -0.755658,  0.248923,  0.605821,  0.575063, -0.523828,  0.628417, -0.148006,  0.988986, -0.000279,  0.488766,  0.872162,  0.020993,  0.281287, -0.843873,  0.456899, -0.113878, -0.658130,  0.744242, -0.531089,  0.827237,  0.183368,  0.952682, -0.275387,  0.128681, -0.154783, -0.812497,  0.562042,  0.233712,  0.766269,  0.598506, -0.754443, -0.577340,  0.312240};
static float ps113 [339] = {  0.271998,  0.595449,  0.755948, -0.964973,  0.260717,  0.029227, -0.666514,  0.565136,  0.486191, -0.135317,  0.473109,  0.870550,  0.611103, -0.206851,  0.764046, -0.258454, -0.886484,  0.383859, -0.530809,  0.754344,  0.386272, -0.138988,  0.921773,  0.361964, -0.936040,  0.217374,  0.276726, -0.727504,  0.631469,  0.268299,  0.898124,  0.393519,  0.196255, -0.357144,  0.486170,  0.797551,  0.256111,  0.957190,  0.134884, -0.260713, -0.955833,  0.135689,  0.836090, -0.299628,  0.459540,  0.840017,  0.121763,  0.528721,  0.184273, -0.901763,  0.390983,  0.474230, -0.428318,  0.769187, -0.502703, -0.380199,  0.776362,  0.573362,  0.556648, -0.601165, -0.214685,  0.672771,  0.708018, -0.368511,  0.893189,  0.257705, -0.800010, -0.598247, -0.045656, -0.931619, -0.021257,  0.362815, -0.822197,  0.394097,  0.410706, -0.574482,  0.805650,  0.144565, -0.090256,  0.243916,  0.965587,  0.784964, -0.123227,  0.607163, -0.605783, -0.147378,  0.781861, -0.055156,  0.814225,  0.577924, -0.462894, -0.844781,  0.268466,  0.334178,  0.389840,  0.858108, -0.640599, -0.759683,  0.111865, -0.746753,  0.664302,  0.032598,  0.113129, -0.233211,  0.965823,  0.710555,  0.313604,  0.629892, -0.420598, -0.746828,  0.515116, -0.678481,  0.090677,  0.729000,  0.918393, -0.070346,  0.389365, -0.039138, -0.977403, -0.207728,  0.943053, -0.253349,  0.215559, -0.443013,  0.661663,  0.604932, -0.385314, -0.188250,  0.903380,  0.989424, -0.034976,  0.140771,  0.690086,  0.062903,  0.720989,  0.438502,  0.680451,  0.587114,  0.347895, -0.781367,  0.518108, -0.644078, -0.752358, -0.138277,  0.098843,  0.442201,  0.891453,  0.393404, -0.919365,  0.000525,  0.140516,  0.210155,  0.967518,  0.550861, -0.745125,  0.375952, -0.186131,  0.971760, -0.145047, -0.471353,  0.042292,  0.880930,  0.877138, -0.477485,  0.051347, -0.843009,  0.457858, -0.282317, -0.745364, -0.488253,  0.453918, -0.274112, -0.409366,  0.870219,  0.023151,  0.646971,  0.762163, -0.036287, -0.872195,  0.487810, -0.137384, -0.610431,  0.780064, -0.369296, -0.592944,  0.715568, -0.243377,  0.027602,  0.969539,  0.247079, -0.038876,  0.968215, -0.184668,  0.976977,  0.106833, -0.963511, -0.187714,  0.190813, -0.761535, -0.577318, -0.294565, -0.876640, -0.392088,  0.278873,  0.523729, -0.608539,  0.596144,  0.701653, -0.553004,  0.449299,  0.389938, -0.883034,  0.261150,  0.317319, -0.624202,  0.713919, -0.789704, -0.077974,  0.608513, -0.038814, -0.963438,  0.265106,  0.120149,  0.904171,  0.409926, -0.700234, -0.324412,  0.635947,  0.677066, -0.387846,  0.625426, -0.722875,  0.309966,  0.617554, -0.191599, -0.763015,  0.617331,  0.829256,  0.362007,  0.425777,  0.662659,  0.536013,  0.523042,  0.340488,  0.840252,  0.421953,  0.938854,  0.164229,  0.302626,  0.365864,  0.160933,  0.916648,  0.381672, -0.232540,  0.894568,  0.551630,  0.251087,  0.795399, -0.449437, -0.864922, -0.223422,  0.239394, -0.428493,  0.871254,  0.513498,  0.482789,  0.709391,  0.506369, -0.019796,  0.862090,  0.188620,  0.768839,  0.610990,  0.090177, -0.606074,  0.790280, -0.145211, -0.204272,  0.968084, -0.912600, -0.406465,  0.044130, -0.993533,  0.034934,  0.108038, -0.857893, -0.238446,  0.455151,  0.112652, -0.769934,  0.628101, -0.458412, -0.888560,  0.017884, -0.848476,  0.143974,  0.509274, -0.000000,  0.000000,  1.000000, -0.777124, -0.594814,  0.205608,  0.576362,  0.720475,  0.385646, -0.042384, -0.998689,  0.028698, -0.563356,  0.471171,  0.678695, -0.618558, -0.689887,  0.376088, -0.299969,  0.813839,  0.497680, -0.872596,  0.452268,  0.184472,  0.723981, -0.656131,  0.212938,  0.576257, -0.809690,  0.111038, -0.316946,  0.263845,  0.911006, -0.529810,  0.268956,  0.804341, -0.027867, -0.418289,  0.907887,  0.980449,  0.188553,  0.056282};
static float ps114 [342] = { -0.712121,  0.364658,  0.599923, -0.499167, -0.300794,  0.812622, -0.315110, -0.686014,  0.655813,  0.219962,  0.192997,  0.956226, -0.353049,  0.669347,  0.653705, -0.461401,  0.133972,  0.877018, -0.082413, -0.976536,  0.198963,  0.534392, -0.740339,  0.407827, -0.515086,  0.342274,  0.785834,  0.927795, -0.356741,  0.109232,  0.498229, -0.145108,  0.854817, -0.630822, -0.773134,  0.065786, -0.383145, -0.080889,  0.920140,  0.330786, -0.325713,  0.885715, -0.074662,  0.475961,  0.876291, -0.300541, -0.505831,  0.808585, -0.079394, -0.611185,  0.787496, -0.684476,  0.574834,  0.448395,  0.950833, -0.164458,  0.262432,  0.324302,  0.922222,  0.210559, -0.882470, -0.458462, -0.105161,  0.352437, -0.550280,  0.756954, -0.070660, -0.400817,  0.913429,  0.113101, -0.244191,  0.963109, -0.990446, -0.029119, -0.134791,  0.447355, -0.889380, -0.094218, -0.788716, -0.606380,  0.101149,  0.059942, -0.998141,  0.010985,  0.143831, -0.490720,  0.859364, -0.523670, -0.685054,  0.506429,  0.702115, -0.563426,  0.435414, -0.512664, -0.512236,  0.689050, -0.091151, -0.902010,  0.421983, -0.202861,  0.078305,  0.976072,  0.854452, -0.350386,  0.383590, -0.738358, -0.659253, -0.142171, -0.823227, -0.453530,  0.341479,  0.550180,  0.815248,  0.180754,  0.170535,  0.421760,  0.890526, -0.317129,  0.494836,  0.809053,  0.922768,  0.059125,  0.380792,  0.492939, -0.730481, -0.472661,  0.542577, -0.574625,  0.612712,  0.343806, -0.870015,  0.353370,  0.539003, -0.369526,  0.756919,  0.692631,  0.486851,  0.532201,  0.985209, -0.170427, -0.017811,  0.689603,  0.491027, -0.532297, -0.320735, -0.826065,  0.463407,  0.021151,  0.810170,  0.585813,  0.772658, -0.591752, -0.229849, -0.286960,  0.953938, -0.087506, -0.436542, -0.899275,  0.027103,  0.082372, -0.923206, -0.375374,  0.667151,  0.273553,  0.692877,  0.138148, -0.847936,  0.511782, -0.513594, -0.810547,  0.281486,  0.225630, -0.819821, -0.526294, -0.833210, -0.255479,  0.490399,  0.535024,  0.460541,  0.708273,  0.849425, -0.146588,  0.506941,  0.816318, -0.577602, -0.001097, -0.684135, -0.285209,  0.671278, -0.066400, -0.977162, -0.201856,  0.351252, -0.736691,  0.577848,  0.638036,  0.675357,  0.369869,  0.438819,  0.103292,  0.892619, -0.545741,  0.532519,  0.646986,  0.263609,  0.754145,  0.601477, -0.905979, -0.404419,  0.125091,  0.491786,  0.656825,  0.571601,  0.281448, -0.071958,  0.956875, -0.023027,  0.258142,  0.965832,  0.360233, -0.873188, -0.328292, -0.113943, -0.146613,  0.982610,  0.717577, -0.371386,  0.589198, -0.088308, -0.779089,  0.620663, -0.839266,  0.389226,  0.379651, -0.781035, -0.048278,  0.622618,  0.593652, -0.761053, -0.261486,  0.628128,  0.063458,  0.775518,  0.420378,  0.812909,  0.403065, -0.602122, -0.071077,  0.795234,  0.101927,  0.635956,  0.764965, -0.272344,  0.294839,  0.915914, -0.308938, -0.920521,  0.239163,  0.211268, -0.963075, -0.166891,  0.055130,  0.020550,  0.998268, -0.837148,  0.176155,  0.517834, -0.285514, -0.294798,  0.911908,  0.492823, -0.856022,  0.156050,  0.326828,  0.563825,  0.758476,  0.652239, -0.757358, -0.031523,  0.809859,  0.486990,  0.327060,  0.208786,  0.977949,  0.004855, -0.672980,  0.154631,  0.723317, -0.989149, -0.025749,  0.144641, -0.974770, -0.222991, -0.009924,  0.797634,  0.071442,  0.598896, -0.913701, -0.022548,  0.405761,  0.142647, -0.692796,  0.706884,  0.413460,  0.326936,  0.849802,  0.695356, -0.159054,  0.700844,  0.910344, -0.389335, -0.140330,  0.827992,  0.283416,  0.483843, -0.937189, -0.220477,  0.270309, -0.943019,  0.185412,  0.276291, -0.128038,  0.671096,  0.730231,  0.818357, -0.527077,  0.229087, -0.176190, -0.893944, -0.412093, -0.686848, -0.652858,  0.319400,  0.136437, -0.949899,  0.281204,  0.930116,  0.264533,  0.254766, -0.672570,  0.710596, -0.206649};
static float ps115 [345] = {  0.609985,  0.369208,  0.701144,  0.756728, -0.245878,  0.605728,  0.219885, -0.130739,  0.966725, -0.245924, -0.763118,  0.597639,  0.649097,  0.739264,  0.179337,  0.594527, -0.374241,  0.711675, -0.011158, -0.245463,  0.969342, -0.490605, -0.871359, -0.006252, -0.566700,  0.582990, -0.582215,  0.425137, -0.252778,  0.869116, -0.727134,  0.676537, -0.116509, -0.791251,  0.186297,  0.582422,  0.951636, -0.281658,  0.122710, -0.578415,  0.815607,  0.014866, -0.230861,  0.383306,  0.894304,  0.360145, -0.681372,  0.637203, -0.904747,  0.221516,  0.363818, -0.811180, -0.252102,  0.527666, -0.676421, -0.734495,  0.054522,  0.921022,  0.374787,  0.106085, -0.067546, -0.674201,  0.735453, -0.821010, -0.570144, -0.029638, -0.471304, -0.438309,  0.765348, -0.959195,  0.253890,  0.124436, -0.461685, -0.638366,  0.615902, -0.424034, -0.226081,  0.876974,  0.768862,  0.021052,  0.639068,  0.037345,  0.700269,  0.712902, -0.407579, -0.805571,  0.430041, -0.658019, -0.477985,  0.581844,  0.215799,  0.342500,  0.914398, -0.236040, -0.360315,  0.902474,  0.135602, -0.751911,  0.645169, -0.183418,  0.611895,  0.769378,  0.213438,  0.927007,  0.308385, -0.194933, -0.911617,  0.361877, -0.116737,  0.808554,  0.576726,  0.998825, -0.046918,  0.012110, -0.410697,  0.886471,  0.213300, -0.033036, -0.476161,  0.878737, -0.619136,  0.377355,  0.688675, -0.568391,  0.582152,  0.581405,  0.617122, -0.125083,  0.776862,  0.436295, -0.007315,  0.899774,  0.520097,  0.600099,  0.607766, -0.438772,  0.277333,  0.854731, -0.187636,  0.966807,  0.173429,  0.299752,  0.695869,  0.652621, -0.045062, -0.991282, -0.123811,  0.444776,  0.861732,  0.244113, -0.979371, -0.189926,  0.069003,  0.597309,  0.692761,  0.404110,  0.617606,  0.133443,  0.775084, -0.761800, -0.605349,  0.230680, -0.626772,  0.154477,  0.763737,  0.715586,  0.487401,  0.500376, -0.860665,  0.440522,  0.255337, -0.924922, -0.222300,  0.308385, -0.114991, -0.985542,  0.124436,  0.869234,  0.352400,  0.346766,  0.457591, -0.774913,  0.436027,  0.229085, -0.858683,  0.458458,  0.430802,  0.239657,  0.870043,  0.053749, -0.951343,  0.303411,  0.126312, -0.989588,  0.069003, -0.707199,  0.595766,  0.380699, -0.027866,  0.931053,  0.363818,  0.195027,  0.555325,  0.808442,  0.202894, -0.362397,  0.909671,  0.412446,  0.469130,  0.780900, -0.033322, -0.854283,  0.518739,  0.174121, -0.574339,  0.799886,  0.382344,  0.800186,  0.462077,  0.974484, -0.016339, -0.223859, -0.409652,  0.506268,  0.758867,  0.882614, -0.470097,  0.000559, -0.899374, -0.411012,  0.148981, -0.266015,  0.873650,  0.407396,  0.965708, -0.074947,  0.248579,  0.901752,  0.128849,  0.412603,  0.830351, -0.502940,  0.239934, -0.000000, -0.000000,  1.000000,  0.783322,  0.559488,  0.270888,  0.304148, -0.919985,  0.247228, -0.411899,  0.040850,  0.910313, -0.770001,  0.403139,  0.494548,  0.881974, -0.306215,  0.358264, -0.000168,  0.246883,  0.969045, -0.274146, -0.568892,  0.775375, -0.481910,  0.758227,  0.439152, -0.624579, -0.653874,  0.427026, -0.015246,  0.476301,  0.879150, -0.663553,  0.657233, -0.357412, -0.608663,  0.756041,  0.240689, -0.363260,  0.931470, -0.020142, -0.894394, -0.015175,  0.447022, -0.552592, -0.796644,  0.244950,  0.397934, -0.483387,  0.779735,  0.766409, -0.630972, -0.120381,  0.528484, -0.821265,  0.215008, -0.275182, -0.959047, -0.067110, -0.338753,  0.708455,  0.619143, -0.971565, -0.156741, -0.177466,  0.877624, -0.098489,  0.469124,  0.743776, -0.458682,  0.486218,  0.135137,  0.839035,  0.527028,  0.762336,  0.045974, -0.645547, -0.347531, -0.919553,  0.183426,  0.224296,  0.108398,  0.968474, -0.587654, -0.073921,  0.805728, -0.219982, -0.111048,  0.969163, -0.808408, -0.441745,  0.389022, -0.212961,  0.147369,  0.965883,  0.776308,  0.256960,  0.575602, -0.646549, -0.280587,  0.709398};
static float ps116 [348] = {  0.062688, -0.684546,  0.726270, -0.008877, -0.837344,  0.546604,  0.550973,  0.572054,  0.607605, -0.778273, -0.053162,  0.625671, -0.316153,  0.911886,  0.261747, -0.087479, -0.950560,  0.297966, -0.066940,  0.827807,  0.557005,  0.543815, -0.497510,  0.675833,  0.361620,  0.869010,  0.337717, -0.302434,  0.822746,  0.481272,  0.398456, -0.833739,  0.382246,  0.569101,  0.717933,  0.400869, -0.088821,  0.932022,  0.351349,  0.153827,  0.873978,  0.460977,  0.945440,  0.325648, -0.009884,  0.380769,  0.745260,  0.547359,  0.533225, -0.021244,  0.845706,  0.586817, -0.273532,  0.762119,  0.832140, -0.402491,  0.381503, -0.311783,  0.949620,  0.031828,  0.727165,  0.649706,  0.221615, -0.051814, -0.317930,  0.946697, -0.501201, -0.363270,  0.785387, -0.599092,  0.457069,  0.657401,  0.130198,  0.962452,  0.238191,  0.691567,  0.369162,  0.620850,  0.280172, -0.020585,  0.959729,  0.363426, -0.428842,  0.827052,  0.778999, -0.585537,  0.224292, -0.503275,  0.854475, -0.128793, -0.391850,  0.498731,  0.773124,  0.115349,  0.537363,  0.835425,  0.264661,  0.369301,  0.890826,  0.731223, -0.342227,  0.590079,  0.846320,  0.309216,  0.433737,  0.829716, -0.557221, -0.032810, -0.408585, -0.711456,  0.571742,  0.785859,  0.130722,  0.604431,  0.141617,  0.167556,  0.975638, -0.607737,  0.730732, -0.310944,  0.041244, -0.072629,  0.996506,  0.292686, -0.935053,  0.200026, -0.680706,  0.226037,  0.696812,  0.089651, -0.988750, -0.119730,  0.478388, -0.687208,  0.546708,  0.173773,  0.733446,  0.657161,  0.306570, -0.624901,  0.717993, -0.159560,  0.508440,  0.846185,  0.823673, -0.494477, -0.277589, -0.772003, -0.593948,  0.226353, -0.689866, -0.723937, -0.000085, -0.769935,  0.393498,  0.502355,  0.402401,  0.180546,  0.897483,  0.137952, -0.915708,  0.377422, -0.574445, -0.526311,  0.626905, -0.109588,  0.120932,  0.986593, -0.229534, -0.855253,  0.464603,  0.618726,  0.182265,  0.764171, -0.626296, -0.158395,  0.763325, -0.181479, -0.714904,  0.675261, -0.482948,  0.284813,  0.828035,  0.482989,  0.873599, -0.059548, -0.958839, -0.167844,  0.229032, -0.115169, -0.535894,  0.836393, -0.915823, -0.005845,  0.401539, -0.280839, -0.352330,  0.892744,  0.978273,  0.000845,  0.207318, -0.421443, -0.146103,  0.895008,  0.078022, -0.989200,  0.124081,  0.905200, -0.260006, -0.336168,  0.729587,  0.527463,  0.435299,  0.236680, -0.790552,  0.564809, -0.667751,  0.603014,  0.436444,  0.608554,  0.762124, -0.220972, -0.003496,  0.347226,  0.937775,  0.862257,  0.453728,  0.225041,  0.683682, -0.728179,  0.048324,  0.491891,  0.390901,  0.777971,  0.906079,  0.065028,  0.418081,  0.125516, -0.492969,  0.860946, -0.763774, -0.470484,  0.441922, -0.826476,  0.157999,  0.540346, -0.521429,  0.845027,  0.118496, -0.041052,  0.678116,  0.733808,  0.845697, -0.145999,  0.513304,  0.843548,  0.537049, -0.002422,  0.996106,  0.087624, -0.009745,  0.401168, -0.209195,  0.891797,  0.543082,  0.820463,  0.178612, -0.611077, -0.660826,  0.435768, -0.931432,  0.346194,  0.112179, -0.888586, -0.390281,  0.241029, -0.483476,  0.658763,  0.576439,  0.947394,  0.235489,  0.216768, -0.354871,  0.083043,  0.931220, -0.693989,  0.691280,  0.201273,  0.714212, -0.077757,  0.695597,  0.178594, -0.269755,  0.946222, -0.713746, -0.321414,  0.622302,  0.342217,  0.575818,  0.742510, -0.428725, -0.832944,  0.349856, -0.350655, -0.554413,  0.754763, -0.129994, -0.991369,  0.016979,  0.677912, -0.563698,  0.471891, -0.859476, -0.234793,  0.454062, -0.258045,  0.305163,  0.916672, -0.196034, -0.116018,  0.973710,  0.292411,  0.938332, -0.184470,  0.329114,  0.938583,  0.103666,  0.983404, -0.175694,  0.045257, -0.269773,  0.685391,  0.676359, -0.513320,  0.780850,  0.356056,  0.577407, -0.056771, -0.814480,  0.908048, -0.395877,  0.136858, -0.979114,  0.089367,  0.182617,  0.931014, -0.207048,  0.300574};
static float ps117 [351] = { -0.779124,  0.079168,  0.621850, -0.633129,  0.683542,  0.363206,  0.598839, -0.694510,  0.398807,  0.129440, -0.799176,  0.586994,  0.564760, -0.220557,  0.795236, -0.460851, -0.498849,  0.734007, -0.129343,  0.235834,  0.963147, -0.517758, -0.836603, -0.178949,  0.816471,  0.512401,  0.266121, -0.164689,  0.954632,  0.248105,  0.280992, -0.949048,  0.142660, -0.916948,  0.012598,  0.398807, -0.456785,  0.724586,  0.516064, -0.986896,  0.160352, -0.017983, -0.934836, -0.297690, -0.193552,  0.475363, -0.021876,  0.879518, -0.372435, -0.927593,  0.029370, -0.988831, -0.042466,  0.142865, -0.113215, -0.772233,  0.625171,  0.963110,  0.263728, -0.053537, -0.744250,  0.490187,  0.453661,  0.026379,  0.771176,  0.636076, -0.070324,  0.888048,  0.454340, -0.036343, -0.901974,  0.430258,  0.348437, -0.243062,  0.905269,  0.234567,  0.403753,  0.884286, -0.140554, -0.990044,  0.007617, -0.045461, -0.437350,  0.898141, -0.340426,  0.934947,  0.099919,  0.706539,  0.308703,  0.636792,  0.877750,  0.477712,  0.036704,  0.265758,  0.749383,  0.606464, -0.933626, -0.102049, -0.343407,  0.500423, -0.617992,  0.606352,  0.769506,  0.628777, -0.111803,  0.066565,  0.971253,  0.228553, -0.822694, -0.150654,  0.548159,  0.503176, -0.862836,  0.048244,  0.920571, -0.385582, -0.062258, -0.502046,  0.831439,  0.238032,  0.037224, -0.977315,  0.208494, -0.114421, -0.213569,  0.970204, -0.421175, -0.869846,  0.256865,  0.349878,  0.177580,  0.919810, -0.486229, -0.007131,  0.873802,  0.847366,  0.318229,  0.425088,  0.187818, -0.459030,  0.868341, -0.356457, -0.686478,  0.633787,  0.602332,  0.154463,  0.783159, -0.323094,  0.862264,  0.390014, -0.814154,  0.579374, -0.038468, -0.667390, -0.096003,  0.738495,  0.168420,  0.885566,  0.432907, -0.811172, -0.369832,  0.453017,  0.789604, -0.549266,  0.273555,  0.467473,  0.346328,  0.813342,  0.237211, -0.027118,  0.971080,  0.710349,  0.516942,  0.477677,  0.722697,  0.680766,  0.119446, -0.494967, -0.677922, -0.543535, -0.008060,  0.433431,  0.901151, -0.224276,  0.765208,  0.603454, -0.875390,  0.387034,  0.289650,  0.112524,  0.210497,  0.971097, -0.379496,  0.215353,  0.899781,  0.754483, -0.253941,  0.605203, -0.493062,  0.389625,  0.777871,  0.604800, -0.418578,  0.677502,  0.785219, -0.584827, -0.203488, -0.698909,  0.329659,  0.634705,  0.703888, -0.042805,  0.709020,  0.859235, -0.325875,  0.394361, -0.662175,  0.744467,  0.085401, -0.916204,  0.363657, -0.168298,  0.407136, -0.451032,  0.794236, -0.238707,  0.027849,  0.970692,  0.877760,  0.429288, -0.212717, -0.361407,  0.595045,  0.717848, -0.194948, -0.597299,  0.777964,  0.812729,  0.116070,  0.570964,  0.365001, -0.781703,  0.505682, -0.615568, -0.548634,  0.565753, -0.199246, -0.950308,  0.239198,  0.294812,  0.934474,  0.199609,  0.046109, -0.639699,  0.767241,  0.117373, -0.240956,  0.963412, -0.287350, -0.401665,  0.869538,  0.280100, -0.646615,  0.709530,  0.207854, -0.904564,  0.372236, -0.257359,  0.427422,  0.866647,  0.503271,  0.729836, -0.462664, -0.399928, -0.828086, -0.392851,  0.714885, -0.499669,  0.489152, -0.661457,  0.730597, -0.169422,  0.449506, -0.846000,  0.286754,  0.991928,  0.070585,  0.105340, -0.927774, -0.209952,  0.308472, -0.123117,  0.618279,  0.776256,  0.585469,  0.809588, -0.042348, -0.276905, -0.845832,  0.455951,  0.618928,  0.740468, -0.261983,  0.630238,  0.697250,  0.341531,  0.565539,  0.490942,  0.662677,  0.352980,  0.569605,  0.742264, -0.339867, -0.180050,  0.923078,  0.957696, -0.132154,  0.255642, -0.099359,  0.994841,  0.020468,  0.000000, -0.000000,  1.000000, -0.736624, -0.570130,  0.363780, -0.519413, -0.276580,  0.808526,  0.680829,  0.336406, -0.650617,  0.122050,  0.608393,  0.784195, -0.576344,  0.545659,  0.608345,  0.871594, -0.092148,  0.481491, -0.851818,  0.237844,  0.466730, -0.606522,  0.165860,  0.777574, -0.958412,  0.175780,  0.224830};
static float ps118 [354] = {  0.138204, -0.151299,  0.978779,  0.076798, -0.762167,  0.642809, -0.994415, -0.032019,  0.100563,  0.207528, -0.918396, -0.336870,  0.093478, -0.593739,  0.799210, -0.301947, -0.131003,  0.944281,  0.801561,  0.465258,  0.375546, -0.059443, -0.028434,  0.997827,  0.905349, -0.392073,  0.163162, -0.522171, -0.749866,  0.406249,  0.795329,  0.560424, -0.231033,  0.809280,  0.022866,  0.586978,  0.324264, -0.282806,  0.902703, -0.423838,  0.813704,  0.397803, -0.740073,  0.625514,  0.247033,  0.315930,  0.900864,  0.297713, -0.540240, -0.371732,  0.754954, -0.817406,  0.041635,  0.574555, -0.843972, -0.383290,  0.375235, -0.702443, -0.167608,  0.691724, -0.122696, -0.486736,  0.864890, -0.863764,  0.421485,  0.276156, -0.748892,  0.466343,  0.470834, -0.075715,  0.239656,  0.967901,  0.920597,  0.035585,  0.388890,  0.685014,  0.233345,  0.690149, -0.686537, -0.581414,  0.436607,  0.055343, -0.890362,  0.451877, -0.157518, -0.832254,  0.531546,  0.319345, -0.504560,  0.802146, -0.932839, -0.006880,  0.360229,  0.655411,  0.005370,  0.755253, -0.257367, -0.958805,  0.120232,  0.604601, -0.760861,  0.235688,  0.649212,  0.733686, -0.200569,  0.536577,  0.798728,  0.272249, -0.264700,  0.963359, -0.043284, -0.675826,  0.603283, -0.423448, -0.255928,  0.094643,  0.962052, -0.946101, -0.216371,  0.240991,  0.985801, -0.160439,  0.049558, -0.868147,  0.235798,  0.436715,  0.482802,  0.585401,  0.651312, -0.906761, -0.409574,  0.100170, -0.606942,  0.661128,  0.441056,  0.879119,  0.454340,  0.143962,  0.109580,  0.372532,  0.921527, -0.959549,  0.177062,  0.218895,  0.499465,  0.139638,  0.855006, -0.470840,  0.081540,  0.878442,  0.316792,  0.263165,  0.911256, -0.924641, -0.260523, -0.277788,  0.777486, -0.595454,  0.202364,  0.279965, -0.838743,  0.467044, -0.189086,  0.644234,  0.741087,  0.694636, -0.422403,  0.582286,  0.840625, -0.202055,  0.502518, -0.136869,  0.458411,  0.878138, -0.239478,  0.791652,  0.562083, -0.827477,  0.411092, -0.382473, -0.169471, -0.980840, -0.096084, -0.103364, -0.950065,  0.294437,  0.574025, -0.791709, -0.209024, -0.430257,  0.666722,  0.608573, -0.660249,  0.064776,  0.748248,  0.985608,  0.042348,  0.163654, -0.933063,  0.353499,  0.066572, -0.097254, -0.265756,  0.959122,  0.367775, -0.914589, -0.168133,  0.163758, -0.953346,  0.253602,  0.503757, -0.143673,  0.851814,  0.001128, -0.996897,  0.078708,  0.613389,  0.788748,  0.040374, -0.589798,  0.488754,  0.642851,  0.425501,  0.761515,  0.488922, -0.140977, -0.679310,  0.720183,  0.521287, -0.375691,  0.766235, -0.034051,  0.873026,  0.486484,  0.202209,  0.840933,  0.501939, -0.331202, -0.349314,  0.876519,  0.489301, -0.742928,  0.456774, -0.539866, -0.572490,  0.617089,  0.829489, -0.557453, -0.034552,  0.103055, -0.985520, -0.134649,  0.627398,  0.637199,  0.447603,  0.675703,  0.446647,  0.586457,  0.304565,  0.494236,  0.814230,  0.071380,  0.951477,  0.299327, -0.713088, -0.385185,  0.585780, -0.326520,  0.307669,  0.893714,  0.019114,  0.744137,  0.667754, -0.359848, -0.727961,  0.583594, -0.939307,  0.186595, -0.287896, -0.847756, -0.185026,  0.497067,  0.134683,  0.116675,  0.983996,  0.509280,  0.375750,  0.774239, -0.325532, -0.873776,  0.361310,  0.306074, -0.691011,  0.654845,  0.067609,  0.576751,  0.814118,  0.514464, -0.578102,  0.633343,  0.262003,  0.686937,  0.677844,  0.782994,  0.622030, -0.000511,  0.682116, -0.731244, -0.000127,  0.726479,  0.648073,  0.228539, -0.727297,  0.274125,  0.629202,  0.396432, -0.881865,  0.255257, -0.544002,  0.296455,  0.784969,  0.330663, -0.009846,  0.943697, -0.345252, -0.552548,  0.758612, -0.469224, -0.867860,  0.163240,  0.693141, -0.215054,  0.687973, -0.493981,  0.868863, -0.032571, -0.380981,  0.502071,  0.776388, -0.971165, -0.236191, -0.032429,  0.107855, -0.389209,  0.914813, -0.518376, -0.151192,  0.841681,  0.828937,  0.255268,  0.497696, -0.409173, -0.909215, -0.076846};
static float ps119 [357] = { -0.236302, -0.397010,  0.886873,  0.663619,  0.619735,  0.418973, -0.683725, -0.354866,  0.637644,  0.683611, -0.579292,  0.443956, -0.046716, -0.978389, -0.201425, -0.860313, -0.508734,  0.032434,  0.907963, -0.190925,  0.373031, -0.156014, -0.741169,  0.652938, -0.214469, -0.856590,  0.469315, -0.996238, -0.014832,  0.085378,  0.803243,  0.436265,  0.405553, -0.243220,  0.867767,  0.433386, -0.579476,  0.561673,  0.590534, -0.356363,  0.608767,  0.708807, -0.296521,  0.944319, -0.142606, -0.560847, -0.555593,  0.613814, -0.579300,  0.220527,  0.784716,  0.384269,  0.230143,  0.894076,  0.138749,  0.227503,  0.963842, -0.848790, -0.274218,  0.452063,  0.145584, -0.182419,  0.972383,  0.810728, -0.540844,  0.224072, -0.403605, -0.729656,  0.552001, -0.875758, -0.407372,  0.259028,  0.020868, -0.883602,  0.467774,  0.465665,  0.769059,  0.437840,  0.076215, -0.752891,  0.653718,  0.268306,  0.783922,  0.559891,  0.446187, -0.849486,  0.281585, -0.513174, -0.857829, -0.027957,  0.256334, -0.382688,  0.887605,  0.313940,  0.618161,  0.720638,  0.481463, -0.380704,  0.789467, -0.753175, -0.616460,  0.229575, -0.911868, -0.375415, -0.166013, -0.736316,  0.676016, -0.028997, -0.950181, -0.149407,  0.273559,  0.391104, -0.179779,  0.902617,  0.528221, -0.569972,  0.629377,  0.777812,  0.238400,  0.581527, -0.758767,  0.491324,  0.427638, -0.094835, -0.994116,  0.052341, -0.609083,  0.778207,  0.153009,  0.889027, -0.388395, -0.242447, -0.572174, -0.206840,  0.793621,  0.675160, -0.387209,  0.627876, -0.857770,  0.264368,  0.440841,  0.417436,  0.874871,  0.245657, -0.464692, -0.405772,  0.787026, -0.582918, -0.791025,  0.185705, -0.455253,  0.010960,  0.890295, -0.000254,  0.424047,  0.905640, -0.892365, -0.005476,  0.451281,  0.467962,  0.429081,  0.772594,  0.336371,  0.590260, -0.733790,  0.264789,  0.958795,  0.102953, -0.954821,  0.128922,  0.267760, -0.597450, -0.690358,  0.407995,  0.981925,  0.151902,  0.112915,  0.050629,  0.781582,  0.621745, -0.190115,  0.953903,  0.232217, -0.955961, -0.283871,  0.074542,  0.952878,  0.022192,  0.302541, -0.923756,  0.337581, -0.180869, -0.126646,  0.606142,  0.785209, -0.360413,  0.226161,  0.904961,  0.508497,  0.612971,  0.604729,  0.119566, -0.579115,  0.806430,  0.346553,  0.930042, -0.122160,  0.245789, -0.881183,  0.403861, -0.695613,  0.358130,  0.622789,  0.661446,  0.435303,  0.610738,  0.773459, -0.600044, -0.204228,  0.216008,  0.904937,  0.366646, -0.876810,  0.480691,  0.011819,  0.623501,  0.749462,  0.222605, -0.438760,  0.847593,  0.298455, -0.008913,  0.902519,  0.430558,  0.293891, -0.750550,  0.591863,  0.855032,  0.025049,  0.517969, -0.410479, -0.848579,  0.333796,  0.333864, -0.576811,  0.745536, -0.353304, -0.200164,  0.913844,  0.127946, -0.991721, -0.010964,  0.784920, -0.187926,  0.590410, -0.964941,  0.254715,  0.063320,  0.079153, -0.967052,  0.241960,  0.228060, -0.016482, -0.973508, -0.742270, -0.498101,  0.448253, -0.478502,  0.422483,  0.769769, -0.158384, -0.949211,  0.271871, -0.541455,  0.838343, -0.063298, -0.778659,  0.132971,  0.613196,  0.609982, -0.183159,  0.770957,  0.705476,  0.026519,  0.708238,  0.647975, -0.718348,  0.253188,  0.273008,  0.024590,  0.961697, -0.792099, -0.576422, -0.200793, -0.181079,  0.762105,  0.621616,  0.508364,  0.026410,  0.860737, -0.377437,  0.922257,  0.083566,  0.900837,  0.236443,  0.364126, -0.651673,  0.004938,  0.758484,  0.817053, -0.391990,  0.422811,  0.986362, -0.114263,  0.118461,  0.713360,  0.700798,  0.000310, -0.116683,  0.227591,  0.966741,  0.019746,  0.021156,  0.999581,  0.620406, -0.686456, -0.379308,  0.428079, -0.738154, -0.521418,  0.099642,  0.615857,  0.781532, -0.109705, -0.190898,  0.975460, -0.106922, -0.576801,  0.809857, -0.774221, -0.133194,  0.618742, -0.250016,  0.425502,  0.869736,  0.242823,  0.427647,  0.870721,  0.013248, -0.385738,  0.922513,  0.490730, -0.734215,  0.469162,  0.603005,  0.235329,  0.762237};
static float ps120 [360] = {  0.535275, -0.120662,  0.836015, -0.968000,  0.228788,  0.103117,  0.081375,  0.944102,  0.319452,  0.809120, -0.577430,  0.109087,  0.315658, -0.476320,  0.820658,  0.846069,  0.505203,  0.170109, -0.121051, -0.820173,  0.559162, -0.717055, -0.524466,  0.459095, -0.071812, -0.449370,  0.890455,  0.322178, -0.815621,  0.480588, -0.991206, -0.008725,  0.132042,  0.321829, -0.231948,  0.917947,  0.107783,  0.361110,  0.926273,  0.497697, -0.341559,  0.797267, -0.449224,  0.687995,  0.569965,  0.640053,  0.596064,  0.484810, -0.013123,  0.856803,  0.515478,  0.861140,  0.399683, -0.314152,  0.844946,  0.199476,  0.496262,  0.941588,  0.331077,  0.061645,  0.714426,  0.141262,  0.685304,  0.706480,  0.656998,  0.263134, -0.470899, -0.466779,  0.748580, -0.663905,  0.328394,  0.671854,  0.107207,  0.132300,  0.985395,  0.866498,  0.490146, -0.094541, -0.362476, -0.810655,  0.459836,  0.822309, -0.274757,  0.498314, -0.919337, -0.269455, -0.286729, -0.924648, -0.144591,  0.352306, -0.341420,  0.925633,  0.163206,  0.328144,  0.222398,  0.918074,  0.573289,  0.730513, -0.371065, -0.083719, -0.672376,  0.735460, -0.301259, -0.322638,  0.897300, -0.879008,  0.466778,  0.097277, -0.416915,  0.828563,  0.373717, -0.957388, -0.246425,  0.150606, -0.211730, -0.967743,  0.136540,  0.106071,  0.570639,  0.814322, -0.834158, -0.046443,  0.549567, -0.522456, -0.000810,  0.852666,  0.987687, -0.118477,  0.102169, -0.654800, -0.385430,  0.650139,  0.577462,  0.808240,  0.115266,  0.705859,  0.370994,  0.603429, -0.189826, -0.974912, -0.116242,  0.540612,  0.316668,  0.779397, -0.510628, -0.238764,  0.825985,  0.118495, -0.581494,  0.804875,  0.838792, -0.444936,  0.313784, -0.319375,  0.381169,  0.867589,  0.153658,  0.758189,  0.633671, -0.612809,  0.709370,  0.348223,  0.704189, -0.505322,  0.498766, -0.289645,  0.602908,  0.743376, -0.780016,  0.400361,  0.480922,  0.424216,  0.723213,  0.544981, -0.927602,  0.356967, -0.110133,  0.677508, -0.320426,  0.662050,  0.283107, -0.919578,  0.272445,  0.406916,  0.879835, -0.245581, -0.070208,  0.696905,  0.713719,  0.334087, -0.665368,  0.667586, -0.110160,  0.024212,  0.993619, -0.098338, -0.217524,  0.971088, -0.642314,  0.760608, -0.094384, -0.691505,  0.653509, -0.307810,  0.534622,  0.100408,  0.839105,  0.257632,  0.857717,  0.444914, -0.109235,  0.489946,  0.864882,  0.630401, -0.559841, -0.537748,  0.121601, -0.761710,  0.636405,  0.314834,  0.621677,  0.717215, -0.164813, -0.919312,  0.357354, -0.008204, -0.999722,  0.022127, -0.490954,  0.481749,  0.725866, -0.320566, -0.093333,  0.942617, -0.321872,  0.144097,  0.935753,  0.125320, -0.347587,  0.929235, -0.827745,  0.181870,  0.530812,  0.044038, -0.969487,  0.241153,  0.528176, -0.532360,  0.661531, -0.101235,  0.979218,  0.175738, -0.535326, -0.618720,  0.574989, -0.742175, -0.624873,  0.242300, -0.241396,  0.784549,  0.571148,  0.715716, -0.090344,  0.692523, -0.111942,  0.264494,  0.957868, -0.990211, -0.109670, -0.086339,  0.406394,  0.913691, -0.003708, -0.732758,  0.671997,  0.107167,  0.927836, -0.205887,  0.311017,  0.524792,  0.521933,  0.672443, -0.193720,  0.908027,  0.371429,  0.088756, -0.889854,  0.447530,  0.363559,  0.900063,  0.240234,  0.596071,  0.792122, -0.131308, -0.552312,  0.821157,  0.143706, -0.446565,  0.892265, -0.066656, -0.263438, -0.535609,  0.802324,  0.521977,  0.777730,  0.350252, -0.227142,  0.972669, -0.048175,  0.529760, -0.692425,  0.489798,  0.948727,  0.033660,  0.314299, -0.897871,  0.323126,  0.299028, -0.692735,  0.091724,  0.715336,  0.109501, -0.106162,  0.988301,  0.330443,  0.431504,  0.839412,  0.500141, -0.815836,  0.290293, -0.511225,  0.240031,  0.825248, -0.936397,  0.092572,  0.338512, -0.808935, -0.284447,  0.514504,  0.802547,  0.443308,  0.399244,  0.745117,  0.666844,  0.010981,  0.327451, -0.002967,  0.944863, -0.690066, -0.149839,  0.708066, -0.777906,  0.551017,  0.302064, -0.323441, -0.691857,  0.645539,  0.855144, -0.037584,  0.517026};
static float ps121 [363] = { -0.600352, -0.319758,  0.733030, -0.846943, -0.445872,  0.289630, -0.738878,  0.439422,  0.510849,  0.781240,  0.519106,  0.346688, -0.771247,  0.564913,  0.293343,  0.717659, -0.420918,  0.554792,  0.085923,  0.991313, -0.099578, -0.170408,  0.436652,  0.883344, -0.422126,  0.805228,  0.416435, -0.499159, -0.864448,  0.059750, -0.102169,  0.062677,  0.992791,  0.994960,  0.011165, -0.099652, -0.060049,  0.952900, -0.297280,  0.890397, -0.447112, -0.085350, -0.937677,  0.146648,  0.315050,  0.619128,  0.750238, -0.231999,  0.365922, -0.687151,  0.627634,  0.445106,  0.797001,  0.408252,  0.060129,  0.988720,  0.137178, -0.348621, -0.656077,  0.669348, -0.494467,  0.850312, -0.180196, -0.948902, -0.263581,  0.173523,  0.289260,  0.953996,  0.078872,  0.639588, -0.268835,  0.720177, -0.025036, -0.856571,  0.515421,  0.720182, -0.046751,  0.692208, -0.272976,  0.223996,  0.935580,  0.878176, -0.363756, -0.310625,  0.762572,  0.179421,  0.621524,  0.484716,  0.855701,  0.181179,  0.204650, -0.347241,  0.915173,  0.260677,  0.917345,  0.300874,  0.716384,  0.432549,  0.547444, -0.226566,  0.765062,  0.602783, -0.652577,  0.289298,  0.700321, -0.174133, -0.553875,  0.814187,  0.867534, -0.351021,  0.352376,  0.979181,  0.195085,  0.056092,  0.421991,  0.324721,  0.846451, -0.749748, -0.174748,  0.638233, -0.421154, -0.848296,  0.320972, -0.969604,  0.226359,  0.092896,  0.372362,  0.693543,  0.616721,  0.201245, -0.851428,  0.484326, -0.719308,  0.070992,  0.691054,  0.580627,  0.112780,  0.806321, -0.019952,  0.277466,  0.960528,  0.908027,  0.363461,  0.208288, -0.505894, -0.522349, -0.686457, -0.357650,  0.004662,  0.933844, -0.169179,  0.968505,  0.182695, -0.101101, -0.722440,  0.684002, -0.618166,  0.636190,  0.461663, -0.352108,  0.935914, -0.009164,  0.370432,  0.102460,  0.923191,  0.515632, -0.479913,  0.709793, -0.736927, -0.410347,  0.537172,  0.575657, -0.615579,  0.538221,  0.991493, -0.059041,  0.115997, -0.008131,  0.818888,  0.573896,  0.667789,  0.743662,  0.032008,  0.209381,  0.309649,  0.927511,  0.952649,  0.118913,  0.279857,  0.805543,  0.579822,  0.122095, -0.303246, -0.941216,  0.148841,  0.606378, -0.720071,  0.337348, -0.082402,  0.647981,  0.757186,  0.703394, -0.698492,  0.131707,  0.282682, -0.925405,  0.252421,  0.066344,  0.497031,  0.865193,  0.517532, -0.102047,  0.849557, -0.247409, -0.355826,  0.901209,  0.215316,  0.831627,  0.511895, -0.823041,  0.226928,  0.520679,  0.293687,  0.523056,  0.800100,  0.063639, -0.558190,  0.827269,  0.815365, -0.204452,  0.541644, -0.605738,  0.753029,  0.256962, -0.423979,  0.368175,  0.827459,  0.935056, -0.129962,  0.329819, -0.864117, -0.256487,  0.433031, -0.395991,  0.893676,  0.211033, -0.902161, -0.430536,  0.027282, -0.780864, -0.609989,  0.134777, -0.699732, -0.597807,  0.391154, -0.586234, -0.074292,  0.806728, -0.426779, -0.216334,  0.878100, -0.417954, -0.461053,  0.782780,  0.859594,  0.281627,  0.426362, -0.434755,  0.670906,  0.600727, -0.569106, -0.542904,  0.617555, -0.310232,  0.567990,  0.762328, -0.515531,  0.152689,  0.843157,  0.946202, -0.296133,  0.130407, -0.201160,  0.893017,  0.402562, -0.571401,  0.819655,  0.040824,  0.141918,  0.082856,  0.986405,  0.061564, -0.141013,  0.988092,  0.596005,  0.633787,  0.493043, -0.509064, -0.714777,  0.479529, -0.555751,  0.490482,  0.671244,  0.413625, -0.804626,  0.426017,  0.139323, -0.722211,  0.677496,  0.424872, -0.316829,  0.847999,  0.147403,  0.687960,  0.710622,  0.295177, -0.121902,  0.947634,  0.849113, -0.510171,  0.136870,  0.757091, -0.549124,  0.353943,  0.027370,  0.931447,  0.362847,  0.294894, -0.534691,  0.791924, -0.170518, -0.925064,  0.339382,  0.757660, -0.647627, -0.080810, -0.021097, -0.360990,  0.932331, -0.946702, -0.081545,  0.311619, -0.134648,  0.989399, -0.054407, -0.273968, -0.810401,  0.517872,  0.613623,  0.322850,  0.720580,  0.870960,  0.027404,  0.490589, -0.858163, -0.006355,  0.513338, -0.179287, -0.149644,  0.972349,  0.648462,  0.710245,  0.273951};
static float ps122 [366] = {  0.350814, -0.500984,  0.791167,  0.765580,  0.634311,  0.107408,  0.172218, -0.390264,  0.904453,  0.741241,  0.196159,  0.641937, -0.934285, -0.338285,  0.112581,  0.958551, -0.280597,  0.049455, -0.783227,  0.618886, -0.059464, -0.142898,  0.886105,  0.440905,  0.211900, -0.140050, -0.967204,  0.447876,  0.608904,  0.654709, -0.524725, -0.818916,  0.232468,  0.096709, -0.790687,  0.604534, -0.608454,  0.306148,  0.732159, -0.746987, -0.098712,  0.657470, -0.190095,  0.580805,  0.791536,  0.988150,  0.110020, -0.107031, -0.584882,  0.760398,  0.282327, -0.117661, -0.941320,  0.316343,  0.561616, -0.397887,  0.725447,  0.059177,  0.834021,  0.548550, -0.962059,  0.034760,  0.270620,  0.457594,  0.756063,  0.467950,  0.895942,  0.431101,  0.106963,  0.257463, -0.937636,  0.233564, -0.528820,  0.693123,  0.489827, -0.532322, -0.712391,  0.457310,  0.620934,  0.444365,  0.645740, -0.609471,  0.080171,  0.788745, -0.406496, -0.913653,  0.000158,  0.255391,  0.874298,  0.412769,  0.874622,  0.118898,  0.470000,  0.457496, -0.843685,  0.280879, -0.278091, -0.556295,  0.783072, -0.420911,  0.250977,  0.871691, -0.013707,  0.995381,  0.095015, -0.339671, -0.839911,  0.423289, -0.374000,  0.662389,  0.649126, -0.583250,  0.512771,  0.629988, -0.626899,  0.777316,  0.052702, -0.176460,  0.761171,  0.624084,  0.240084, -0.970745, -0.003667,  0.220273,  0.571605,  0.790410,  0.134032, -0.611703,  0.779650, -0.884135,  0.205688,  0.419520, -0.831286, -0.544600,  0.111235,  0.451891, -0.890573,  0.051706,  0.742818, -0.587088,  0.321792,  0.000000, -0.000000,  1.000000,  0.945087, -0.197459,  0.260425,  0.257457,  0.739152,  0.622391, -0.199817, -0.102582,  0.974449,  0.420844, -0.888259, -0.184081,  0.046790, -0.987146,  0.152819, -0.092008, -0.689556,  0.718365,  0.786859,  0.370946,  0.493205,  0.876363, -0.118542,  0.466837, -0.759281,  0.622552,  0.189529, -0.456042, -0.402550,  0.793712,  0.584363,  0.808058,  0.074585, -0.324050, -0.921970,  0.212046,  0.362363, -0.831425, -0.421220,  0.047825,  0.946842,  0.318123,  0.885498, -0.457900, -0.078870, -0.208478,  0.370469,  0.905146, -0.923281, -0.210215,  0.321499, -0.711929,  0.566923,  0.414434,  0.741857, -0.272576,  0.612658, -0.581606, -0.156973,  0.798182, -0.122770, -0.839433,  0.529414,  0.997553, -0.057691,  0.039494,  0.024707,  0.681951,  0.730981,  0.906042,  0.290037,  0.308166,  0.974132,  0.200850,  0.103566, -0.838415, -0.436175,  0.326821,  0.875093, -0.444887,  0.190496,  0.205094, -0.950313, -0.234182, -0.406619,  0.466263,  0.785659,  0.757409, -0.038487,  0.651805,  0.167352,  0.984384, -0.054596, -0.388756, -0.207380,  0.897698,  0.403547,  0.433359,  0.805823, -0.856798,  0.412410,  0.309541,  0.009454,  0.479447,  0.877520,  0.559229, -0.706468,  0.433781,  0.307402, -0.698925,  0.645761,  0.689563, -0.494226,  0.529380, -0.001404,  0.248369,  0.968664,  0.197846, -0.150083,  0.968676,  0.389558, -0.281581,  0.876902, -0.414854,  0.024112,  0.909568, -0.681090, -0.724694,  0.104572, -0.219447, -0.358751,  0.907271,  0.391974,  0.193347,  0.899429, -0.320842, -0.717211,  0.618602, -0.755984,  0.368554,  0.540978,  0.840240, -0.357874,  0.407337,  0.592248,  0.047142,  0.804375,  0.630907,  0.719788,  0.289587, -0.506753, -0.571867,  0.645112,  0.320760, -0.825716,  0.464011,  0.402924, -0.047319,  0.914009, -0.013143, -0.255376,  0.966753, -0.768741,  0.142826,  0.623408, -0.650330, -0.347730,  0.675393,  0.205872,  0.348960,  0.914245,  0.503213, -0.601009,  0.620938,  0.638177, -0.747257,  0.185302,  0.962836,  0.035272,  0.267775,  0.590624, -0.179237,  0.786789,  0.200243,  0.100133,  0.974616, -0.702541, -0.637581,  0.316112,  0.232839,  0.957290,  0.171412,  0.795874,  0.519250,  0.311391,  0.642596,  0.596158,  0.481318, -0.692554, -0.509351,  0.510814, -0.809107, -0.283331,  0.514849,  0.568899,  0.271502,  0.776300,  0.433996,  0.864829,  0.252425, -0.043210, -0.496190,  0.867138,  0.097624, -0.909721,  0.403581, -0.876260, -0.033880,  0.480645, -0.957339,  0.231658,  0.172731};
static float ps123 [369] = { -0.186370, -0.654884,  0.732389,  0.035850, -0.621385,  0.782685, -0.308443,  0.351821,  0.883790,  0.662907,  0.093127,  0.742888,  0.493791, -0.830124,  0.258969, -0.503014, -0.280078,  0.817639, -0.859233, -0.294409, -0.418381, -0.955228,  0.131718,  0.264933, -0.429761,  0.822780,  0.371938, -0.263423, -0.806010,  0.530053, -0.748601, -0.653551,  0.111658,  0.217305,  0.820375,  0.528927, -0.290361, -0.265741,  0.919278,  0.629643, -0.516542,  0.580288,  0.085474,  0.232989,  0.968716, -0.685030,  0.696849, -0.212451, -0.407230, -0.909635,  0.082018, -0.029999, -0.999345,  0.020236, -0.158306,  0.179331,  0.970968,  0.373084,  0.498751,  0.782340,  0.574918, -0.817896,  0.022690, -0.683886, -0.340988,  0.645002, -0.358312, -0.876123,  0.322523, -0.870238,  0.488093,  0.066721, -0.629583, -0.561087,  0.537408,  0.606612,  0.751801,  0.258490,  0.463395,  0.103426,  0.880095,  0.001600,  0.785043,  0.619440, -0.610736,  0.105631,  0.784757, -0.396955,  0.143783,  0.906506, -0.506234, -0.740171,  0.442577, -0.037992, -0.792019,  0.609314, -0.420670,  0.666972, -0.614967, -0.410507,  0.898643,  0.154678,  0.986561,  0.128761,  0.100584, -0.875160, -0.482942, -0.029359, -0.954700, -0.289549,  0.068624,  0.160219,  0.461860,  0.872362,  0.575498, -0.694260,  0.432209, -0.399861, -0.659725,  0.636297,  0.315898,  0.288291,  0.903934, -0.109584, -0.912310,  0.394565,  0.373899, -0.116634,  0.920107,  0.567014,  0.501032,  0.653806, -0.593848, -0.804452, -0.014196,  0.606553, -0.765916, -0.213229,  0.000000, -0.000000,  1.000000,  0.186897, -0.982037, -0.025946,  0.002416,  0.905926,  0.423429,  0.256845, -0.561779,  0.786407,  0.610021, -0.310146,  0.729167,  0.419941,  0.813178,  0.402979,  0.053422, -0.972647,  0.226061, -0.508308, -0.479854,  0.715097,  0.940390, -0.315465,  0.127080, -0.223240, -0.923173, -0.312915,  0.375086, -0.924218,  0.071640, -0.066506, -0.233162,  0.970161,  0.429535,  0.886450,  0.172356, -0.236420, -0.968549, -0.077575, -0.759168,  0.451944,  0.468412,  0.118795, -0.883028,  0.454036, -0.914702, -0.250165,  0.317392,  0.883431, -0.117260,  0.453652, -0.997162,  0.075242, -0.002571,  0.218546,  0.666766,  0.712503, -0.413002,  0.535346,  0.736773,  0.132614, -0.403663,  0.905246, -0.796792, -0.413714,  0.440412,  0.732806,  0.469690,  0.492328,  0.836234, -0.522809,  0.165481,  0.009197,  0.619531,  0.784919, -0.201680,  0.547914,  0.811859,  0.956457, -0.276572, -0.093267, -0.236882, -0.043403,  0.970568, -0.063581,  0.396315,  0.915910, -0.923372, -0.329756, -0.196585, -0.526207,  0.326639,  0.785120,  0.531755,  0.308639,  0.788656,  0.363317, -0.330557,  0.871053,  0.828326,  0.093650,  0.552364,  0.741314, -0.670684, -0.025242, -0.982254, -0.082168,  0.168601, -0.211287,  0.940648,  0.265591, -0.431330,  0.700296,  0.568806,  0.465897, -0.480661,  0.742903, -0.197334, -0.963812,  0.179235,  0.769904, -0.326769,  0.548152, -0.221321,  0.849831,  0.478334, -0.710992,  0.282233,  0.644076, -0.570351, -0.790848,  0.221943,  0.568666, -0.113939,  0.814639,  0.263575, -0.928097,  0.262992, -0.709141, -0.621386,  0.333164,  0.419946,  0.680578,  0.600383, -0.601376,  0.504694,  0.619380, -0.817734, -0.187521,  0.544193,  0.755243, -0.530506,  0.384930, -0.616296,  0.659177,  0.430887, -0.764307,  0.589967,  0.260335, -0.802981, -0.521903, -0.287816,  0.234013,  0.060972,  0.970320,  0.878764, -0.330259,  0.344533,  0.158833, -0.182665,  0.970261, -0.860597, -0.462499,  0.213229, -0.093089, -0.452021,  0.887137, -0.908877, -0.026957,  0.416191,  0.966616, -0.107956,  0.232376, -0.784195,  0.047655,  0.618681,  0.716403,  0.289310,  0.634875, -0.609084, -0.651647, -0.452076, -0.879559,  0.376286,  0.291178,  0.194766, -0.744666,  0.638388,  0.938568,  0.098353,  0.330782, -0.665500, -0.119554,  0.736760, -0.308893, -0.477046,  0.822808,  0.011507,  0.978386,  0.206467, -0.748135, -0.652619, -0.119928, -0.218375,  0.717066,  0.661912,  0.352034, -0.817863,  0.455162, -0.463449, -0.069563,  0.883389, -0.856165,  0.216428,  0.469192,  0.750044, -0.113551,  0.651567};
static float ps124 [372] = { -0.669303, -0.213104,  0.711772,  0.785895,  0.617771,  0.026979,  0.893998, -0.355829, -0.272311,  0.798230,  0.384216,  0.463905, -0.280080,  0.288247, -0.915680,  0.799712, -0.225256,  0.556525,  0.997948, -0.060912,  0.019750, -0.738404,  0.204694,  0.642542,  0.868162, -0.343409,  0.358281, -0.453668,  0.062882,  0.888950,  0.478092, -0.324262,  0.816261, -0.668865, -0.440811,  0.598586, -0.218450,  0.974042, -0.059352,  0.325255, -0.257847, -0.909793, -0.133599,  0.901369,  0.411930,  0.644572, -0.136068,  0.752338, -0.022588, -0.770511,  0.637027,  0.535466,  0.289211,  0.793494,  0.724563,  0.653633, -0.218568,  0.097764,  0.737447,  0.668292,  0.480193,  0.862932,  0.157366,  0.625882,  0.779511, -0.025196,  0.436220,  0.484533,  0.758248, -0.409871,  0.450683,  0.793026,  0.508448,  0.775722,  0.373813,  0.040891,  0.378449,  0.924718,  0.653553, -0.347648,  0.672316,  0.588610, -0.807917,  0.028439,  0.420879, -0.890754,  0.171515,  0.874564,  0.149844,  0.461178, -0.154454, -0.176668,  0.972076,  0.556996,  0.787267, -0.264512,  0.861202, -0.500885, -0.086286,  0.637421,  0.449161,  0.626059, -0.348123,  0.881455,  0.319136,  0.508255,  0.643143,  0.572752, -0.915038, -0.290778, -0.279559, -0.266504, -0.686942,  0.676081, -0.294266, -0.899936, -0.321751, -0.072051, -0.604142,  0.793613,  0.037304, -0.980733,  0.191758, -0.550563,  0.248795,  0.796858, -0.648574,  0.019077,  0.760913,  0.783759,  0.005612,  0.621040,  0.982508,  0.075849, -0.170076,  0.760369, -0.644479,  0.080533,  0.764021, -0.572783,  0.296970, -0.879333,  0.164647,  0.446839, -0.117742,  0.780111,  0.614462,  0.888720, -0.436026,  0.141624, -0.769211,  0.557895,  0.311556, -0.804168, -0.025671,  0.593847,  0.161318, -0.660807,  0.733015, -0.362110, -0.882763,  0.299343, -0.089942, -0.873242, -0.478915,  0.528817, -0.531861,  0.661420, -0.258967,  0.630773,  0.731479, -0.789425,  0.375716,  0.485434, -0.146497, -0.936231,  0.319392,  0.617842, -0.419148, -0.665271, -0.467874,  0.618961,  0.630858,  0.312983,  0.796571,  0.517220, -0.389604,  0.917905,  0.075225,  0.379873, -0.689040,  0.617187, -0.904694, -0.418605, -0.079365,  0.958921, -0.203015,  0.198128,  0.115192, -0.466590,  0.876940,  0.153011,  0.175665,  0.972486,  0.213138, -0.812478,  0.542634, -0.212621, -0.832391,  0.511779,  0.234438, -0.921094,  0.310845, -0.917424, -0.058284,  0.393620,  0.330626, -0.509659,  0.794314, -0.430586, -0.899854,  0.069702, -0.332560,  0.780560,  0.529273, -0.105832,  0.206481,  0.972710, -0.963905,  0.142400,  0.224964, -0.109825, -0.404377,  0.907975, -0.901019, -0.282181,  0.329453,  0.300768,  0.339426,  0.891251,  0.306536,  0.504636, -0.807080, -0.712786,  0.685611,  0.147898, -0.434139, -0.755545,  0.490586, -0.038814,  0.601194,  0.798160,  0.060454,  0.962008,  0.266244, -0.067241,  0.238837, -0.968729, -0.541015,  0.734306,  0.409996, -0.956859,  0.288397,  0.035323,  0.448928, -0.098529,  0.888119,  0.305833,  0.654720,  0.691237,  0.950508,  0.290719, -0.109622,  0.587588, -0.660082,  0.468008, -0.806474, -0.256220,  0.532871,  0.667814,  0.714431,  0.208836,  0.812297,  0.521912,  0.260350,  0.678460,  0.597790,  0.427013, -0.256506,  0.013924,  0.966442,  0.484720,  0.586910, -0.648524,  0.605284,  0.079179,  0.792062,  0.607778, -0.752332,  0.254170, -0.627770, -0.635466,  0.449542,  0.857790,  0.486922, -0.164631,  0.231830, -0.053291,  0.971295,  0.728083, -0.472771,  0.496370, -0.544985,  0.812167,  0.208269,  0.390570,  0.126148,  0.911889,  0.420390, -0.812091,  0.404698, -0.489322, -0.150792,  0.858968, -0.171582,  0.968157,  0.182297, -0.785991, -0.475766,  0.394797,  0.728848,  0.233330,  0.643690,  0.257497,  0.959783,  0.111855, -0.654042,  0.570232,  0.497056,  0.035266, -0.896410,  0.441821,  0.180071,  0.529045,  0.829269,  0.904746, -0.093243,  0.415622,  0.981869,  0.172752,  0.078044, -0.328407, -0.287644,  0.899672,  0.000000,  0.000000,  1.000000,  0.966266,  0.041636,  0.254158, -0.015441, -0.999068, -0.040300, -0.512210, -0.382769,  0.768849, -0.206613, -0.972855,  0.104230, -0.183237,  0.439024,  0.879592};
static float ps125 [375] = { -0.654707,  0.002136,  0.755879,  0.398473,  0.468800,  0.788319,  0.805669, -0.228269,  0.546618, -0.404532, -0.256016,  0.877958,  0.322539, -0.939628,  0.114311,  0.373323, -0.099230,  0.922379,  0.922817, -0.098706,  0.372379,  0.919198,  0.392623, -0.030374,  0.071260, -0.656356,  0.751079,  0.638481,  0.751608,  0.165612,  0.678676,  0.145176,  0.719947,  0.051577,  0.975464,  0.214030, -0.612734,  0.669885, -0.419300,  0.913173,  0.134733,  0.384657, -0.969535,  0.194491, -0.148913, -0.145443,  0.161209,  0.976145, -0.566779,  0.208926,  0.796939,  0.124467,  0.965120, -0.230327, -0.152841, -0.884368,  0.441058, -0.985465, -0.163264,  0.046933,  0.063989,  0.692635,  0.718444,  0.850280, -0.515296, -0.107207,  0.782624,  0.260283,  0.565467,  0.918688,  0.314449, -0.239028,  0.698406,  0.498272,  0.513764,  0.451088,  0.851326,  0.267889, -0.985771, -0.024759, -0.166258, -0.795906, -0.455353,  0.398983,  0.236427,  0.905918,  0.351304, -0.649849,  0.735321, -0.192355,  0.862852, -0.408268, -0.298000,  0.513306,  0.597462,  0.616081, -0.182518, -0.643040,  0.743765, -0.962611, -0.071024,  0.261411,  0.576527,  0.705545,  0.412095,  0.609314,  0.382828,  0.694391,  0.479909,  0.255062,  0.839423,  0.192411, -0.795340,  0.574815, -0.523979, -0.823061,  0.219125, -0.141588,  0.988774,  0.047736, -0.422654, -0.610839,  0.669507, -0.386333,  0.846844,  0.365517,  0.474525, -0.834236,  0.280850, -0.307975,  0.579962,  0.754185, -0.305501, -0.463660,  0.831678, -0.552516,  0.803428,  0.221879,  0.182007, -0.493578,  0.850444, -0.944057,  0.173453,  0.280482, -0.450111,  0.403617,  0.796551,  0.692263, -0.109876,  0.713231, -0.528703, -0.426930,  0.733623,  0.284527, -0.306029,  0.908510, -0.378065,  0.721829,  0.579681,  0.640012, -0.357332,  0.680219,  0.300461,  0.658760,  0.689752, -0.140763,  0.732232,  0.666350, -0.103938,  0.977688, -0.182544, -0.500943, -0.733870,  0.458793, -0.362119,  0.923268,  0.128245, -0.727559,  0.625817,  0.281091, -0.677432, -0.658959,  0.326893,  0.728448, -0.460349,  0.507388,  0.085219,  0.140166,  0.986454, -0.178584,  0.944999,  0.274017,  0.379072,  0.779780,  0.498244,  0.064955, -0.910520,  0.408332, -0.568359,  0.695044,  0.440321, -0.783807, -0.115783,  0.610115, -0.995538,  0.069959,  0.063320,  0.848119,  0.364941,  0.384072, -0.276846,  0.893337, -0.353986, -0.087758,  0.548365,  0.831622,  0.953584,  0.243932,  0.176563, -0.295328, -0.765397,  0.571795,  0.285329,  0.950369,  0.124037, -0.791300, -0.609655, -0.046523, -0.474532, -0.026574,  0.879837,  0.158765,  0.521137,  0.838576, -0.608333, -0.220632,  0.762399, -0.763518,  0.146101,  0.629043, -0.627608, -0.555914,  0.545039,  0.673712,  0.733593, -0.089180,  0.322767,  0.940253, -0.108378, -0.058956, -0.072076,  0.995655, -0.042705, -0.788200,  0.613936, -0.534262,  0.556870,  0.635972, -0.664727,  0.362963,  0.652990,  0.819132,  0.543746, -0.182655,  0.159246, -0.087915,  0.983317, -0.888255,  0.033018,  0.458163, -0.366258,  0.184622,  0.912014, -0.520945,  0.853092, -0.029161, -0.724393, -0.347988,  0.595113,  0.005541,  0.900597,  0.434620, -0.273189, -0.051581,  0.960576,  0.413508, -0.755323,  0.508427,  0.781737, -0.541623,  0.309083,  0.520145, -0.236276,  0.820745,  0.950583, -0.298957, -0.083772,  0.050768, -0.299369,  0.952786, -0.348310, -0.874350,  0.337923,  0.152828,  0.812590,  0.562442,  0.314009, -0.640321,  0.700991, -0.090707, -0.995832,  0.009520, -0.833578,  0.283261,  0.474248, -0.064444, -0.486388,  0.871363,  0.536816, -0.573891,  0.618447, -0.181766, -0.282030,  0.942030, -0.235634,  0.378434,  0.895134,  0.243647,  0.327705,  0.912823,  0.824361,  0.015673,  0.565847,  0.742517,  0.595824,  0.306043,  0.425574, -0.452289,  0.783787, -0.496790, -0.867515, -0.024850,  0.905731, -0.405173,  0.124446,  0.871600,  0.454897,  0.182711, -0.690494,  0.720180,  0.067518, -0.195347,  0.847593,  0.493381,  0.878608, -0.326984,  0.348037, -0.719206,  0.504043,  0.478209, -0.872811, -0.221319,  0.434992, -0.795388,  0.599097, -0.091877,  0.001765,  0.354384,  0.935098,  0.540715,  0.023936,  0.840865,  0.316265,  0.118622,  0.941225};
static float ps126 [378] = { -0.275059, -0.708536,  0.649861,  0.861162,  0.171625,  0.478481,  0.712368,  0.166471,  0.681777, -0.575421, -0.444855,  0.686291,  0.058834,  0.133742,  0.989268, -0.374730, -0.808194,  0.454312,  0.310382,  0.161036,  0.936873, -0.114791,  0.639237,  0.760394,  0.314992, -0.227658,  0.921386, -0.543214,  0.237716,  0.805239,  0.094409, -0.287922,  0.952989, -0.329634,  0.650156,  0.684572,  0.558257, -0.427153,  0.711258, -0.463341, -0.158333,  0.871921, -0.586182, -0.724101,  0.363412,  0.690992,  0.536301, -0.484676, -0.131580,  0.902134,  0.410902, -0.135830, -0.837544,  0.529216,  0.902205, -0.417051,  0.109973,  0.725908, -0.170058, -0.666437, -0.658256, -0.746937, -0.093725, -0.479860, -0.877314, -0.007441,  0.293907,  0.870412,  0.394970,  0.780379, -0.604980,  0.158137, -0.134913, -0.331243,  0.933850,  0.537702, -0.818572, -0.202032,  0.511020, -0.248929,  0.822735,  0.717396, -0.441037,  0.539286,  0.049844, -0.496679,  0.866502,  0.860265, -0.078663, -0.503743, -0.179209,  0.088788,  0.979796, -0.871595, -0.363689, -0.328714, -0.331779,  0.941778,  0.054557,  0.923691, -0.027698,  0.382137, -0.763743, -0.084092,  0.640019,  0.804512, -0.032050,  0.593072,  0.977463, -0.206821,  0.042316, -0.345219,  0.794042,  0.500321, -0.278804, -0.956844,  0.081966,  0.695232, -0.718099, -0.031416,  0.088909,  0.949112,  0.302128,  0.385376,  0.373876,  0.843625, -0.091660,  0.305914,  0.947637, -0.406149,  0.060241,  0.911819,  0.173090, -0.644531,  0.744728,  0.636543,  0.758694, -0.138553,  0.483124, -0.780428,  0.396893,  0.439607, -0.027945,  0.897755,  0.183572, -0.957747,  0.221411, -0.752172, -0.327968,  0.571554,  0.710740, -0.654192, -0.258615,  0.746859,  0.529076,  0.402841, -0.344201,  0.892989,  0.289994,  0.333843, -0.738984,  0.585193,  0.099365, -0.807652,  0.581227, -0.927452,  0.210974,  0.308742, -0.130078,  0.970690,  0.202090,  0.198896, -0.044960,  0.978989, -0.874439, -0.174729,  0.452578,  0.581939,  0.381401,  0.718248, -0.127848,  0.790437,  0.599053,  0.390982, -0.566106,  0.725712, -0.543035,  0.732725,  0.410154, -0.060857, -0.983670,  0.169383,  0.993681, -0.039259, -0.105151,  0.016576,  0.495908,  0.868217, -0.442783, -0.866398,  0.230862,  0.952813,  0.180245,  0.244252, -0.387094, -0.541001,  0.746643, -0.988991,  0.012445, -0.147454,  0.549939, -0.620887,  0.558629, -0.394707,  0.896431, -0.201537, -0.810997,  0.315606,  0.492623,  0.528861,  0.179586,  0.829491,  0.285131, -0.432231,  0.855498,  0.950400,  0.013760, -0.310726,  0.425319,  0.574530,  0.699299, -0.328940,  0.272676,  0.904127, -0.950822, -0.219824,  0.218209,  0.261989, -0.871043,  0.415506, -0.164626, -0.335972, -0.927373, -0.052723, -0.691813,  0.720149,  0.019120, -0.922537,  0.385434, -0.176245, -0.530868,  0.828925,  0.093366,  0.705291,  0.702743, -0.805709, -0.564281, -0.180057,  0.475605,  0.739590,  0.476243, -0.611612,  0.009638,  0.791099, -0.238093,  0.472091,  0.848788, -0.029496, -0.098546,  0.994695,  0.292786,  0.942650,  0.160271, -0.833775,  0.438073, -0.336022,  0.702138, -0.538906, -0.465385,  0.229909,  0.549551,  0.803204,  0.635388, -0.038045,  0.771255,  0.839551, -0.450848, -0.303136, -0.518679,  0.854763, -0.018768,  0.687001, -0.621052,  0.377259, -0.919811, -0.381132, -0.093202,  0.702328, -0.237180,  0.671179,  0.599708,  0.566924,  0.564754,  0.478567,  0.843356,  0.244387, -0.748916, -0.357900, -0.557703, -0.847034, -0.392657,  0.358266, -0.983926, -0.178572,  0.000901,  0.090024,  0.852093,  0.515590, -0.620446, -0.239363,  0.746828, -0.453259,  0.451643,  0.768489,  0.836113, -0.542605, -0.080590, -0.253703, -0.137845,  0.957410, -0.368180, -0.354231,  0.859630,  0.089163,  0.993505,  0.070690,  0.841887, -0.240256,  0.483221,  0.224214,  0.920531, -0.319922,  0.612053, -0.763212,  0.207121, -0.898529, -0.419987,  0.127501, -0.638809, -0.700629, -0.317872, -0.934496,  0.231267, -0.270617,  0.291828,  0.740056,  0.605932, -0.649770,  0.395074,  0.649396,  0.934690, -0.331690, -0.127809, -0.761836, -0.590455,  0.266401, -0.527748,  0.603430,  0.597791,  0.797028,  0.602659, -0.039348, -0.126343,  0.991803, -0.019079, -0.504023, -0.649234,  0.569611};
static float ps127 [381] = { -0.215903, -0.611411,  0.761290, -0.713941,  0.472645,  0.516618, -0.209880, -0.939078, -0.272182,  0.179240, -0.770564,  0.611640, -0.551908,  0.499856,  0.667489, -0.018775,  0.553810,  0.832431, -0.842142, -0.417740,  0.341014, -0.510498, -0.859651,  0.019828, -0.172029,  0.071741,  0.982476, -0.811534,  0.253677,  0.526365,  0.177020,  0.017140,  0.984058, -0.865946,  0.026610,  0.499429, -0.353892, -0.915534,  0.191203,  0.720778, -0.086452, -0.687754, -0.014822, -0.092611,  0.995592,  0.925587, -0.347473, -0.150172,  0.440068,  0.547061,  0.712085, -0.400144, -0.467179,  0.788434, -0.386416, -0.009741,  0.922273,  0.748688,  0.271401,  0.604820,  0.214190,  0.572017,  0.791782,  0.883546,  0.407172,  0.231424,  0.939607, -0.211319,  0.269227,  0.992697, -0.036792,  0.114889, -0.100126,  0.844152,  0.526670,  0.924298, -0.188864, -0.331667, -0.326549,  0.781108,  0.532200, -0.653165,  0.299446,  0.695491,  0.027962,  0.183822,  0.982562,  0.934910,  0.354651,  0.012924, -0.186472, -0.798532,  0.572341,  0.412144, -0.898614, -0.150433,  0.062964,  0.730213,  0.680312,  0.417317, -0.265985,  0.868964, -0.260124,  0.900916,  0.347399,  0.719973,  0.635941, -0.277882,  0.732438,  0.598945,  0.323729,  0.128067,  0.388895,  0.912337,  0.100207, -0.992588,  0.068756,  0.495914,  0.679225,  0.541038,  0.209595, -0.416266,  0.884756, -0.114059,  0.345112,  0.931605, -0.442235,  0.373703,  0.815337,  0.568544,  0.743165,  0.352794,  0.991400,  0.120355, -0.051383, -0.939334, -0.273147,  0.207464,  0.701287, -0.568957,  0.429516,  0.390817, -0.026796,  0.920078, -0.694992,  0.632011,  0.342853,  0.812239,  0.575343,  0.096168,  0.014006, -0.959342,  0.281898,  0.960125, -0.272062,  0.064361,  0.858907, -0.181675,  0.478825, -0.809883,  0.565578,  0.155601, -0.584096, -0.079152,  0.807816,  0.498187,  0.168374,  0.850565,  0.392063, -0.668090,  0.632410, -0.296804, -0.953752, -0.047580, -0.158383,  0.694466,  0.701877, -0.220628, -0.176807,  0.959199, -0.314263,  0.229785,  0.921107, -0.625553,  0.761519,  0.169623, -0.752856, -0.129939,  0.645232, -0.247493,  0.493020,  0.834073,  0.283669,  0.726094,  0.626354,  0.986893, -0.114278, -0.113943, -0.525926,  0.721226, -0.450816, -0.210954, -0.401192,  0.891371, -0.418722, -0.246766,  0.873944,  0.599494, -0.325089,  0.731386, -0.543664,  0.838438, -0.038096, -0.557912, -0.520126,  0.646686,  0.596040,  0.306538, -0.742140, -0.440304, -0.873804, -0.206394,  0.878997, -0.473807,  0.053595, -0.143106, -0.982790,  0.116807,  0.444132, -0.858920,  0.254956,  0.351179,  0.834806,  0.423995, -0.377405, -0.824484,  0.421653,  0.648078,  0.748078,  0.142741,  0.734075, -0.143406,  0.663753, -0.631892,  0.736491, -0.241442,  0.280773,  0.207251,  0.937130, -0.370212,  0.620369,  0.691438, -0.549343, -0.799106,  0.244235,  0.316429, -0.832121,  0.455463,  0.581325,  0.351130,  0.734009, -0.009530, -0.529623,  0.848180, -0.747501, -0.339103,  0.571184, -0.531181,  0.150348,  0.833812,  0.728267, -0.684975,  0.020891,  0.840213, -0.419538, -0.343556,  0.748843, -0.370093,  0.549786, -0.541681,  0.667494,  0.510915,  0.032745, -0.872632,  0.487280,  0.675836,  0.096309,  0.730733, -0.020797, -0.715264,  0.698545, -0.877375, -0.199206,  0.436498, -0.641772, -0.515468, -0.567822,  0.199802, -0.607789,  0.768552, -0.327388,  0.942880, -0.061590, -0.787595,  0.576112, -0.218606,  0.184513, -0.972305, -0.143446, -0.184879, -0.913081,  0.363461,  0.965312,  0.189011,  0.180130,  0.411214, -0.478924,  0.775587,  0.855245, -0.399010,  0.330675, -0.025397,  0.945559,  0.324459,  0.573466, -0.092225,  0.814021,  0.551453,  0.695768, -0.460224,  0.128355,  0.862368,  0.489741,  0.225926, -0.932340,  0.282311, -0.057543, -0.992650, -0.106461,  0.368198,  0.386577,  0.845570, -0.382823, -0.679179,  0.626229,  0.849803,  0.511969, -0.125389,  0.701522,  0.530706, -0.475624, -0.700616, -0.710102,  0.069947, -0.938265, -0.008419, -0.345816,  0.828982,  0.047301,  0.557271,  0.792041,  0.422728,  0.440423,  0.887897,  0.221368,  0.403280,  0.211535, -0.204171,  0.955807, -0.957935, -0.043490,  0.283670, -0.483043,  0.804663,  0.345235, -0.002843, -0.317772,  0.948163,  0.579728, -0.531037,  0.617993};
static float ps128 [384] = {  0.680572,  0.430595,  0.592798, -0.369574, -0.196648,  0.908155, -0.717697, -0.695887, -0.025539, -0.208498,  0.926839,  0.312248,  0.813695,  0.430095,  0.391048, -0.954180,  0.051360,  0.294793, -0.202543, -0.627574,  0.751750, -0.625803,  0.549378,  0.553674,  0.062421,  0.489628, -0.869694,  0.247721,  0.877274, -0.411127,  0.912471,  0.376545,  0.160032,  0.338715,  0.720555,  0.605039, -0.071143,  0.750842,  0.656639, -0.019123,  0.118879,  0.992725,  0.524454,  0.590472,  0.613426, -0.583159,  0.784487, -0.210962,  0.529833, -0.273453,  0.802808, -0.915542, -0.232294, -0.328362, -0.940251, -0.186441,  0.284900,  0.795706,  0.248099,  0.552539,  0.472566,  0.146590,  0.869019,  0.263397, -0.670869,  0.693222,  0.681861, -0.561698,  0.468575, -0.616910,  0.693781,  0.371603,  0.525604,  0.384741,  0.758759, -0.734033,  0.629134, -0.255707,  0.882684,  0.058417,  0.466321,  0.991184, -0.110551,  0.073023, -0.451282,  0.524873,  0.721702,  0.772414,  0.606499, -0.188506,  0.017165,  0.967040,  0.254047, -0.640781, -0.681813,  0.352888,  0.175655,  0.963381, -0.202591,  0.333106, -0.795409,  0.506325, -0.601535,  0.365282,  0.710439, -0.791113, -0.573901, -0.211609,  0.003979, -0.098336,  0.995145,  0.317228, -0.317101,  0.893764, -0.333633, -0.733596,  0.592053, -0.030191, -0.917793,  0.395911,  0.627048,  0.741689,  0.238135, -0.606944,  0.039290, -0.793773, -0.954982, -0.288682,  0.068353, -0.527178, -0.848796, -0.040354, -0.292444, -0.421367,  0.858444,  0.255943,  0.876374,  0.407998, -0.148291, -0.271309,  0.951000, -0.597134, -0.786054,  0.159841,  0.420903, -0.086303,  0.902991, -0.112042, -0.795511,  0.595491, -0.270808,  0.957094, -0.103120,  0.713192, -0.221118,  0.665180,  0.936626, -0.343297,  0.069854,  0.762574,  0.010710,  0.646812,  0.224246,  0.953248,  0.202565, -0.617331,  0.434827, -0.655613, -0.270596,  0.655515,  0.705037,  0.341112,  0.328527,  0.880746,  0.668486,  0.610090,  0.425343, -0.994254, -0.048290,  0.095533, -0.968921,  0.211637,  0.128074, -0.742441, -0.026887,  0.669372,  0.790561, -0.370423,  0.487647,  0.482375,  0.760003,  0.435557, -0.135845,  0.987667,  0.077848, -0.827873, -0.304387,  0.471141,  0.398743,  0.895103, -0.199484,  0.140518,  0.474415,  0.869014, -0.733702,  0.193367,  0.651376, -0.750004, -0.504733,  0.427479, -0.542773, -0.633672,  0.551233,  0.348531,  0.540102,  0.766039,  0.140821,  0.656711,  0.740878, -0.073006,  0.573515,  0.815935, -0.877387, -0.408459,  0.251700,  0.082236, -0.325695,  0.941892, -0.778807,  0.518443,  0.353094,  0.113182, -0.816239,  0.566519, -0.584789,  0.123165,  0.801780,  0.390551, -0.876209,  0.282360, -0.710150, -0.242419,  0.660999, -0.567219,  0.805197,  0.172976, -0.749869,  0.639254,  0.170443, -0.408482,  0.485241, -0.773100,  0.684170, -0.729047,  0.020033, -0.135517, -0.808686, -0.572417, -0.271312,  0.451202,  0.850181, -0.865263,  0.138739,  0.481738, -0.431830,  0.292560,  0.853189, -0.851333,  0.444939, -0.277959,  0.514445,  0.337784, -0.788193, -0.537805,  0.730107, -0.421556,  0.055372, -0.983741,  0.170841,  0.256297,  0.089453,  0.962450,  0.482429, -0.625386,  0.613315, -0.450480,  0.701835,  0.551811, -0.570078, -0.113018,  0.813780,  0.321847,  0.946791,  0.001522, -0.423381,  0.832822,  0.356588, -0.085848, -0.996220, -0.013243,  0.178746, -0.512357,  0.839965,  0.859230, -0.161831,  0.485319,  0.134262,  0.267120,  0.954264,  0.652121,  0.200881,  0.731017, -0.424988, -0.544568,  0.723071,  0.430432,  0.869577,  0.242000, -0.982618, -0.147763, -0.112372, -0.353841,  0.924567,  0.141322, -0.641860, -0.446805,  0.623202, -0.238193,  0.208160,  0.948648, -0.832223,  0.551727, -0.054787,  0.961865, -0.013127,  0.273209,  0.863888,  0.503458, -0.015056,  0.211682, -0.125492,  0.969249,  0.478825, -0.877195,  0.035442, -0.025000,  0.890822,  0.453664,  0.929556, -0.238003,  0.281568,  0.868590,  0.087872, -0.487677, -0.885296,  0.442664,  0.142478, -0.073772,  0.363454,  0.928687, -0.186877,  0.916613, -0.353410, -0.769714,  0.371480,  0.519175, -0.211412, -0.032449,  0.976858, -0.454821, -0.798266,  0.394854, -0.895252,  0.303884,  0.325852, -0.409280,  0.045477,  0.911275, -0.248460,  0.817522,  0.519544,  0.030867, -0.670979,  0.740833};
static float ps129 [387] = { -0.667184, -0.218186,  0.712222,  0.281627,  0.113713,  0.952762, -0.026093, -0.796221,  0.604443,  0.793408, -0.328088,  0.512700, -0.950696, -0.242747,  0.193003, -0.294903, -0.147566,  0.944064, -0.829220,  0.554163,  0.072784, -0.387689, -0.565681,  0.727807,  0.221423,  0.789965,  0.571775, -0.560354,  0.572586,  0.598456,  0.370265,  0.625746,  0.686546,  0.813755, -0.097384,  0.572992, -0.941026,  0.325068,  0.093817,  0.782414,  0.136256,  0.607670, -0.853940, -0.354592,  0.380857,  0.065375,  0.976883,  0.203532,  0.337925, -0.119855,  0.933510,  0.980758, -0.008495,  0.195041,  0.285900,  0.948829,  0.134108,  0.494366,  0.046604,  0.868003,  0.507070,  0.430392,  0.746755, -0.753262,  0.151337,  0.640073, -0.124946, -0.326376,  0.936946, -0.936276,  0.208896,  0.282399, -0.012033, -0.918040,  0.396304, -0.222926, -0.845719,  0.484833, -0.089678,  0.618988,  0.780264,  0.521247, -0.193817,  0.831105,  0.610452,  0.767763,  0.194650,  0.505878, -0.860778,  0.056122, -0.491822, -0.186695,  0.850445,  0.010752,  0.134734,  0.990824, -0.724271,  0.523579,  0.448661,  0.133117, -0.049738,  0.989852,  0.160390,  0.301141,  0.939994, -0.350303,  0.929794,  0.113006, -0.047620,  0.982835, -0.178233, -0.578206, -0.716567,  0.390140, -0.733090, -0.542374,  0.410377,  0.951774,  0.208360,  0.225193,  0.312560, -0.336565,  0.888274,  0.674683, -0.265509,  0.688700, -0.897283,  0.430505, -0.097719, -0.635259, -0.000215,  0.772299, -0.895794, -0.149704,  0.418500, -0.408989,  0.743394,  0.529239,  0.577341,  0.566754,  0.587765,  0.754259,  0.493900,  0.432616,  0.624948, -0.743813,  0.237028,  0.007963,  0.786631,  0.617372,  0.447237, -0.615613,  0.648845, -0.202982,  0.866062,  0.456875, -0.343399, -0.371919,  0.862411,  0.739630, -0.545810,  0.393749,  0.290721, -0.951314,  0.102388, -0.168296, -0.528075,  0.832354, -0.343484, -0.932591,  0.110875, -0.355628,  0.262945,  0.896877,  0.779717, -0.600878,  0.176034, -0.207178,  0.740948,  0.638806,  0.702087,  0.361125,  0.613728, -0.413828, -0.724978,  0.550593, -0.795344, -0.086228,  0.599993,  0.225200, -0.922360,  0.313906, -0.552820, -0.814394,  0.176500,  0.871002, -0.380421,  0.310861,  0.557835, -0.812038, -0.171505, -0.358243,  0.597709,  0.717221,  0.226555, -0.703367,  0.673756, -0.479659,  0.418068,  0.771457,  0.034988,  0.471117,  0.881377, -0.586071, -0.563308,  0.582413, -0.221884,  0.088076,  0.971087, -0.393720,  0.859694,  0.325440,  0.105244, -0.266721,  0.958010,  0.672256, -0.027104,  0.739822, -0.544071, -0.402581,  0.736149, -0.730106, -0.649393,  0.212681,  0.939438,  0.342134,  0.020020,  0.068022, -0.467514,  0.881365,  0.613498,  0.680844,  0.400090, -0.408210, -0.276369, -0.870049,  0.993101,  0.117237, -0.002312, -0.487501, -0.872069, -0.042876, -0.865246, -0.465372,  0.186491, -0.439169,  0.042175,  0.897414, -0.125477, -0.991736,  0.026742,  0.431392,  0.853114,  0.293423, -0.673946,  0.372347,  0.638087, -0.964564,  0.221149, -0.143911, -0.393847, -0.855024,  0.337371, -0.722498,  0.648862,  0.238695,  0.877915,  0.416894,  0.235510,  0.763301,  0.606490,  0.222581,  0.145507,  0.644211,  0.750880, -0.165508,  0.952797,  0.254529, -0.884790,  0.080046,  0.459064, -0.581035,  0.712256,  0.393815,  0.171655, -0.841721,  0.511899,  0.023099, -0.646172,  0.762842, -0.835829, -0.548732, -0.016817, -0.183496, -0.947863,  0.260547,  0.017019,  0.906084,  0.422756, -0.825348,  0.309102,  0.472501, -0.689992, -0.723769,  0.008301,  0.614607,  0.210072,  0.760347, -0.115700,  0.307231,  0.944575, -0.687446,  0.726235,  0.000883,  0.493143, -0.417080,  0.763449,  0.110456, -0.993051, -0.040620,  0.637800, -0.488348,  0.595590, -0.084220, -0.101178,  0.991297, -0.913877,  0.160914, -0.372741,  0.230944,  0.900178,  0.369248, -0.571969,  0.215106,  0.791569, -0.240818,  0.462954,  0.853042,  0.852985,  0.285469,  0.436948,  0.384180, -0.781834,  0.491061, -0.739389, -0.360608,  0.568565, -0.274840,  0.533018, -0.800222, -0.853534,  0.440311,  0.278579, -0.993273,  0.103751,  0.051431, -0.213901, -0.700400,  0.680945,  0.575306, -0.681226,  0.452718,  0.438597, -0.853435,  0.281570,  0.425822,  0.754128,  0.499966, -0.970255, -0.018338,  0.241390,  0.280395,  0.471936,  0.835856,  0.906805,  0.064202,  0.416632};
static float ps130 [390] = {  0.776960,  0.156761,  0.609721, -0.689703, -0.534667,  0.488304,  0.251783,  0.790779,  0.557919, -0.028613,  0.578419,  0.815238,  0.952520,  0.136821,  0.272004, -0.780406,  0.418162,  0.464873,  0.140386, -0.573782,  0.806887,  0.682564, -0.291601,  0.670131, -0.747007, -0.662685,  0.053189, -0.505280, -0.616725,  0.603608, -0.912390,  0.272626,  0.305318, -0.107395,  0.385920,  0.916260,  0.838974, -0.204691,  0.504207,  0.638897,  0.296906,  0.709688,  0.857430,  0.274017,  0.435578,  0.353900, -0.177517,  0.918282, -0.414429,  0.644795,  0.642252,  0.747890, -0.069688,  0.660155,  0.970961,  0.231452,  0.060528,  0.125939, -0.212152,  0.969088, -0.940017,  0.045091,  0.338136, -0.199074, -0.484159,  0.852032, -0.535160,  0.814775,  0.223039, -0.841397,  0.194038,  0.504381, -0.586170,  0.004538,  0.810175,  0.949148, -0.102942,  0.297524,  0.886632, -0.328356,  0.325679, -0.551677,  0.235572,  0.800099, -0.984320,  0.131740,  0.117296, -0.315573, -0.306368,  0.898083, -0.432856, -0.900713,  0.036757, -0.572090,  0.820029, -0.016284, -0.222621, -0.058008,  0.973178,  0.626661,  0.719344,  0.299734, -0.191759,  0.693992,  0.693977, -0.191172,  0.977835, -0.085394, -0.228583, -0.963383,  0.140154,  0.598808,  0.618585,  0.508705, -0.613026,  0.551071,  0.566144, -0.836155, -0.405775,  0.369040,  0.276784,  0.902963,  0.328708, -0.832830,  0.488017,  0.261215, -0.316056,  0.805724,  0.500916, -0.688601,  0.629772,  0.359466, -0.549616, -0.731774,  0.403025,  0.021998, -0.896986,  0.441511,  0.265003, -0.693593,  0.669852,  0.586421, -0.499388,  0.637748,  0.755448, -0.430664,  0.493788, -0.729802,  0.106052,  0.675382, -0.360263, -0.786424,  0.501745,  0.257245, -0.385126,  0.886286, -0.257815, -0.961472, -0.095405,  0.529548,  0.487270,  0.694368,  0.466782,  0.264429,  0.843915,  0.514020, -0.737083, -0.438740, -0.400605, -0.873447,  0.276778,  0.378390, -0.923459, -0.063591,  0.894624,  0.374483,  0.243741,  0.632207, -0.637163,  0.440837, -0.705744, -0.145776,  0.693308, -0.631768, -0.409718,  0.658028,  0.883431,  0.023830,  0.467955, -0.968093,  0.225533, -0.109231,  0.985929,  0.087524, -0.142421,  0.769034,  0.525767,  0.363533,  0.242972, -0.833003,  0.497061,  0.057869,  0.741393,  0.668571,  0.067298,  0.972480,  0.223054,  0.218074, -0.929106,  0.298674,  0.566745, -0.135109,  0.812739, -0.586269, -0.792338,  0.168785,  0.087889,  0.891474,  0.444465,  0.212752,  0.012680,  0.977024, -0.074117, -0.640574,  0.764311, -0.914322, -0.183567,  0.360997,  0.049753,  0.209251,  0.976596,  0.734149, -0.670741,  0.105510,  0.207747,  0.612908,  0.762355, -0.844203, -0.045247,  0.534110, -0.104735, -0.249175,  0.962779, -0.426914, -0.473261,  0.770564,  0.891050,  0.452185,  0.039494, -0.172246,  0.170294,  0.970222, -0.339041,  0.327617,  0.881883,  0.598681, -0.761071,  0.249702, -0.129561,  0.928885,  0.346968, -0.342851,  0.893081,  0.291307, -0.195445, -0.908743,  0.368764,  0.266075,  0.238123,  0.934078,  0.424161,  0.042683,  0.904580, -0.939969, -0.301153,  0.160513, -0.492610,  0.452194,  0.743543, -0.996568, -0.002719, -0.082733, -0.847710, -0.502160,  0.170951, -0.008008, -0.973805,  0.227244,  0.488138, -0.341126,  0.803340, -0.390406,  0.112189,  0.913782,  0.124799,  0.424134,  0.896959, -0.271737,  0.524077,  0.807157, -0.296378, -0.652665,  0.697272,  0.016400, -0.423408,  0.905791, -0.101651,  0.831696,  0.545847,  0.886201, -0.442896,  0.135982, -0.422735, -0.114155,  0.899035,  0.618247,  0.080435,  0.781857,  0.057445, -0.768604,  0.637140, -0.154667, -0.799722,  0.580106,  0.455146,  0.866641,  0.204392,  0.613752,  0.785997,  0.074287,  0.448426,  0.781212,  0.434306, -0.006221, -0.018435,  0.999811,  0.433222, -0.816678,  0.381255,  0.723401,  0.419981,  0.548003, -0.160093,  0.978124,  0.132828,  0.776998,  0.608469,  0.161365,  0.399862,  0.653911,  0.642270,  0.776832, -0.555179,  0.297168,  0.344653,  0.452009,  0.822741, -0.684219,  0.335525,  0.647508,  0.386490, -0.529360,  0.755251,  0.409058, -0.897532,  0.164646, -0.542780, -0.267279,  0.796211, -0.700793,  0.698941,  0.142726, -0.788665, -0.284902,  0.544828, -0.833546,  0.550228,  0.049497,  0.038127,  0.999268, -0.003096, -0.933749,  0.347444,  0.085997, -0.715084, -0.635697,  0.290765,  0.459418, -0.684522,  0.566008};
static float ps131 [393] = { -0.480577, -0.092817,  0.872027,  0.665642,  0.416231,  0.619413, -0.343972, -0.636355,  0.690460,  0.804387, -0.454237, -0.382924,  0.113393,  0.419368,  0.900707, -0.734984,  0.636884,  0.232760,  0.064044,  0.923713,  0.377693,  0.889469,  0.428182,  0.159703,  0.089989, -0.276293,  0.956851,  0.647022,  0.586655,  0.487030,  0.446565, -0.773800,  0.449237, -0.113666, -0.973340,  0.199222,  0.268310, -0.882134,  0.387104,  0.446510, -0.512880,  0.733200,  0.057026, -0.996665,  0.058366, -0.547541,  0.769104,  0.329663, -0.794662, -0.531042,  0.294121,  0.498175, -0.082374,  0.863155, -0.278281,  0.746942,  0.603852,  0.233158, -0.969501, -0.075530,  0.959405, -0.238414,  0.150669, -0.131070, -0.758652,  0.638175, -0.603390, -0.792488, -0.088789,  0.486452,  0.145659,  0.861480, -0.520592, -0.306920,  0.796734,  0.636394, -0.418674,  0.647854,  0.907583, -0.172770,  0.382680,  0.783016, -0.583477,  0.215499,  0.968264, -0.069988, -0.239929, -0.643448, -0.693448,  0.324199, -0.046906, -0.880436,  0.471839,  0.266552, -0.950832,  0.157698,  0.451957,  0.847275,  0.279035,  0.449015, -0.893306,  0.019748,  0.464781, -0.852513,  0.239164, -0.109668,  0.857699,  0.502320,  0.631551, -0.696980,  0.339651, -0.748453,  0.070512,  0.659429, -0.654774,  0.611010,  0.444901,  0.638955, -0.760204,  0.117586,  0.637260,  0.710729,  0.297935, -0.058734,  0.690743, -0.720711, -0.129249, -0.378780,  0.916417,  0.486496,  0.380149,  0.786644,  0.646258,  0.229663,  0.727740,  0.733745, -0.514594,  0.443634, -0.668737,  0.450420,  0.591534, -0.660361, -0.114215,  0.742212,  0.473638,  0.736965,  0.482234,  0.302141,  0.057575,  0.951523,  0.079837, -0.949774,  0.302581,  0.282298, -0.385940,  0.878270, -0.325398,  0.186446,  0.927013,  0.406646,  0.911345,  0.063946,  0.609548, -0.785456, -0.107282,  0.110710,  0.193030,  0.974927, -0.292765, -0.020465,  0.955965, -0.543427, -0.829510,  0.128843,  0.182226,  0.983067,  0.019299,  0.670304,  0.010703,  0.742010, -0.907924, -0.339084,  0.246368,  0.100257, -0.044116,  0.993983, -0.097788,  0.087342,  0.991367, -0.077461,  0.526309,  0.846758, -0.077748,  0.710074,  0.699822, -0.320476, -0.786671,  0.527678,  0.668103, -0.208147,  0.714362, -0.813045, -0.134947,  0.566346,  0.101200,  0.803144,  0.587127,  0.793479, -0.306097,  0.526018, -0.692355, -0.327404,  0.643001,  0.796368,  0.159465,  0.583412, -0.945461, -0.324733,  0.025526, -0.093584,  0.305425,  0.947606,  0.973538, -0.022293,  0.227435,  0.059263, -0.490668,  0.869329, -0.339981, -0.927843,  0.153364, -0.288008,  0.589835,  0.754418, -0.806762,  0.267990,  0.526608,  0.486382,  0.574017,  0.658739, -0.521329, -0.674580,  0.522645, -0.884755,  0.067829,  0.461094, -0.535562, -0.502564,  0.678677, -0.236612, -0.901108,  0.363344,  0.241372,  0.941758,  0.234163,  0.919123,  0.256910,  0.298681, -0.174162,  0.939128,  0.296153, -0.315375, -0.249809,  0.915497, -0.142310, -0.585127,  0.798357,  0.812833,  0.348210,  0.466961,  0.486943, -0.303174,  0.819129, -0.104016, -0.153789,  0.982614,  0.872154, -0.461975, -0.161018,  0.777729, -0.628500, -0.011210,  0.307392,  0.293996,  0.905028,  0.301204, -0.169700,  0.938338,  0.869543, -0.387898,  0.305663, -0.547138,  0.103500,  0.830619,  0.768194,  0.629022,  0.119202,  0.960495, -0.264846, -0.085477, -0.655927,  0.263980,  0.707159, -0.981777, -0.130795,  0.137861,  0.556746, -0.614246,  0.559228,  0.895038, -0.440607,  0.069088, -0.981350, -0.158465, -0.108820, -0.475771,  0.323338,  0.817982,  0.301294,  0.705714,  0.641241,  0.305209,  0.514861,  0.801103, -0.998387,  0.056683,  0.003183,  0.906140,  0.070495,  0.417063,  0.278195,  0.850420,  0.446535, -0.479977,  0.679042,  0.555449,  0.788050,  0.520045,  0.329439, -0.004506, -0.986418, -0.164190, -0.823254, -0.340206,  0.454437,  0.143655, -0.816939,  0.558547, -0.690349, -0.522322,  0.500597, -0.851425, -0.519606,  0.071302, -0.489902,  0.512363,  0.705323, -0.910438,  0.269317,  0.313960,  0.812955, -0.077868,  0.577097, -0.457624, -0.821699,  0.339692,  0.330471, -0.710023,  0.621817,  0.240944, -0.583904,  0.775244, -0.345929,  0.841520,  0.414943,  0.120071,  0.630174,  0.767114, -0.277355,  0.404524,  0.871456,  0.715688,  0.690393, -0.105583, -0.921928, -0.136671,  0.362449,  0.403666, -0.892603, -0.200785, -0.340447, -0.455559,  0.822534};
static float ps132 [396] = { -0.095450, -0.226704,  0.969275,  0.492696, -0.859472,  0.136228, -0.806069, -0.070207,  0.587642, -0.807236, -0.577833,  0.120329, -0.800708,  0.308696,  0.513394, -0.127387,  0.172737,  0.976696,  0.292944, -0.939365,  0.178259,  0.504156,  0.740376,  0.444601,  0.172478,  0.136439,  0.975518,  0.667320,  0.559298,  0.491802, -0.810418,  0.466847,  0.353945, -0.777356,  0.603650,  0.176988,  0.999948, -0.002223,  0.009914, -0.315757, -0.940110, -0.128413,  0.661031, -0.224285,  0.716055,  0.149054, -0.831770,  0.534734,  0.511574, -0.787082,  0.344666, -0.646810, -0.757650, -0.087198,  0.481768, -0.378095,  0.790534,  0.683998, -0.666404,  0.296736, -0.185822, -0.623644,  0.759301,  0.582885, -0.632608,  0.509953,  0.381108, -0.741891,  0.551683, -0.367571,  0.144602,  0.918685,  0.075359, -0.992835,  0.092736, -0.500468,  0.697996,  0.512185, -0.690491, -0.266263,  0.672552, -0.718229, -0.454802,  0.526595,  0.794593,  0.346931,  0.498259,  0.440980,  0.433175,  0.786064, -0.274968, -0.892239,  0.358193,  0.296383,  0.609094,  0.735637,  0.122921, -0.177697,  0.976378,  0.294171, -0.046986,  0.954597, -0.112335,  0.664265,  0.739008,  0.465722,  0.063812,  0.882628,  0.795745, -0.275792,  0.539192,  0.647897,  0.004296,  0.761715,  0.924413, -0.369404,  0.094878,  0.492649,  0.601825,  0.628572,  0.971248, -0.159819,  0.176452,  0.898599, -0.419121, -0.129838,  0.101470, -0.940938,  0.323017, -0.019556, -0.751174,  0.659814,  0.634548, -0.439369,  0.635849, -0.499737,  0.538357,  0.678554,  0.306488, -0.289254,  0.906861,  0.037985, -0.584970,  0.810165,  0.213991, -0.682901,  0.698466,  0.670389,  0.679648,  0.297754,  0.268051, -0.500086,  0.823446,  0.910864,  0.272436,  0.310008,  0.315626,  0.875985,  0.364734, -0.743408,  0.134966,  0.655079, -0.794197, -0.599486, -0.099335, -0.089385,  0.924544,  0.370445,  0.350209,  0.246699,  0.903600,  0.083729, -0.392067,  0.916118, -0.884287,  0.101990,  0.455668, -0.070360,  0.986271,  0.149395,  0.125381,  0.957833,  0.258524, -0.314459,  0.701062,  0.640022,  0.223448,  0.419762,  0.879699, -0.657441,  0.663998,  0.356200, -0.044404, -0.309855, -0.949746, -0.476660, -0.844442,  0.244362, -0.354984, -0.457861,  0.815077, -0.900496, -0.413928, -0.133309,  0.491243, -0.159053,  0.856378,  0.825231, -0.511306,  0.239915, -0.569796,  0.147343,  0.808469, -0.104302,  0.815584,  0.569161, -0.287845,  0.923501,  0.253558,  0.751231, -0.483214,  0.449619,  0.908083, -0.103054,  0.405913, -0.658487, -0.741413,  0.129236, -0.233665, -0.038993,  0.971535, -0.461714,  0.350600,  0.814800,  0.102123,  0.745369,  0.658783, -0.433509, -0.773479,  0.462384, -0.105043, -0.967506,  0.229995,  0.568874,  0.238392,  0.787116,  0.973993,  0.168831, -0.151105,  0.812686, -0.581619,  0.035503, -0.974710,  0.213733,  0.065253, -0.391022, -0.635676,  0.665596, -0.140826, -0.437066,  0.888336, -0.667293,  0.739500, -0.088660, -0.635471,  0.761233,  0.129236, -0.086574,  0.470402,  0.878195,  0.312304,  0.763285,  0.565564, -0.915837,  0.266146,  0.300681, -0.828994, -0.272958,  0.488122, -0.487876,  0.816644,  0.308333,  0.124224,  0.992098,  0.017619, -0.234873, -0.780962,  0.578735, -0.301842,  0.531161,  0.791682,  0.795858, -0.054238,  0.603050, -0.513386, -0.265006,  0.816215,  0.741576,  0.181210,  0.645932, -0.893287, -0.342378,  0.291231,  0.889633, -0.314235,  0.331376,  0.807160,  0.494409,  0.322571,  0.484530,  0.874673, -0.013347, -0.305815,  0.834597,  0.458176,  0.966656,  0.056228,  0.249829,  0.111744,  0.873341,  0.474119, -0.454266, -0.059580,  0.888872,  0.636234,  0.406810,  0.655524, -0.463178,  0.882460,  0.082037,  0.088982,  0.577764,  0.811339, -0.922542, -0.108289,  0.370390, -0.265669,  0.963425,  0.035096,  0.974631,  0.209673,  0.078308, -0.644937,  0.343142,  0.682869, -0.550973, -0.456704,  0.698463,  0.917233,  0.389814, -0.082026,  0.439773, -0.575647,  0.689370, -0.671677,  0.516012,  0.531584, -0.254164,  0.336456,  0.906751, -0.781377, -0.529422,  0.330396, -0.634405, -0.694862,  0.338669, -0.310947, -0.255063,  0.915563,  0.972484, -0.055356, -0.226299,  0.312796, -0.869573,  0.382100, -0.301900, -0.945532,  0.121759,  0.875257,  0.117787,  0.469096, -0.576268, -0.617966,  0.534821, -0.069179, -0.883944,  0.462447,  0.000000, -0.000000,  1.000000, -0.646236, -0.061119,  0.760686,  0.493887,  0.837653,  0.233266};
static float ps133 [399] = { -0.013833, -0.801492,  0.597846, -0.438776, -0.613095,  0.656955,  0.245728, -0.349300,  0.904216, -0.930602, -0.365851, -0.011508, -0.619187, -0.418317,  0.664544, -0.869659, -0.455680,  0.189865, -0.616318, -0.590665,  0.520833, -0.446550,  0.313484,  0.838046, -0.374945, -0.107657,  0.920775,  0.197254, -0.087492,  0.976440, -0.243913, -0.546616,  0.801073,  0.203163,  0.857873, -0.471994, -0.947713, -0.063155,  0.312812, -0.492359, -0.865761, -0.089671,  0.299073,  0.669573,  0.679873, -0.889164, -0.298553,  0.346776,  0.189636,  0.148141,  0.970614, -0.761752, -0.534701,  0.365826,  0.562364,  0.774975,  0.288376,  0.158621,  0.927309,  0.339024,  0.667628,  0.743278,  0.042543,  0.171620,  0.368581,  0.913616, -0.089979, -0.416354,  0.904739,  0.856160, -0.459066,  0.237168,  0.248305, -0.918386,  0.308079,  0.322308, -0.536251,  0.780098, -0.459248,  0.840302,  0.288068, -0.416750, -0.760517,  0.497929,  0.718804, -0.011490,  0.695118, -0.149502, -0.168031,  0.974379,  0.417631, -0.808389,  0.414839, -0.169001, -0.946554,  0.274725,  0.634752,  0.621720,  0.458862,  0.286683, -0.942327, -0.172723,  0.390663,  0.050938,  0.919124,  0.469788, -0.352882,  0.809181,  0.447638,  0.738902,  0.503631, -0.988870,  0.058120, -0.136958,  0.851728,  0.042846,  0.522229, -0.002135,  0.875346,  0.483492,  0.381923, -0.176907,  0.907105,  0.372733,  0.869354,  0.324492,  0.855616,  0.484017,  0.183436, -0.293349, -0.947929, -0.124001,  0.952217, -0.167786, -0.255208,  0.760424, -0.632265,  0.148312, -0.858865,  0.326496,  0.394654, -0.758358,  0.022810,  0.651439,  0.413076, -0.760554, -0.500926, -0.233587, -0.723550,  0.649548,  0.638153, -0.751173, -0.168820,  0.896484, -0.442546,  0.021669,  0.786522, -0.613090, -0.074185,  0.547474,  0.362573,  0.754197, -0.777650,  0.559309,  0.287115, -0.771703, -0.369366,  0.517730, -0.590081, -0.032880,  0.806674,  0.609566, -0.737028,  0.291923,  0.064987, -0.979604,  0.190136,  0.053408, -0.258382,  0.964565,  0.921095,  0.291621,  0.257957, -0.526510, -0.241613,  0.815114,  0.945120, -0.265137,  0.190919,  0.569958,  0.142061,  0.809300,  0.457165, -0.868429,  0.191914,  0.234641,  0.819843,  0.522303,  0.945192,  0.084459,  0.315403,  0.164339, -0.691043,  0.703883,  0.686373,  0.427463,  0.588360, -0.707806, -0.201726,  0.676991,  0.130103,  0.572870,  0.809255,  0.701558, -0.429008,  0.569007,  0.616827, -0.689354, -0.379889, -0.252216,  0.482743,  0.838657, -0.348836, -0.932168,  0.096834,  0.842851, -0.337369,  0.419267, -0.628292,  0.390771,  0.672716,  0.071062,  0.744921,  0.663357,  0.000000,  0.000000,  1.000000, -0.215141,  0.059901,  0.974744, -0.555319,  0.603604,  0.572087, -0.849480, -0.138985,  0.508986, -0.466549,  0.882784,  0.054994, -0.254125,  0.877694,  0.406294, -0.585013, -0.731620,  0.349989, -0.094278,  0.633374,  0.768081,  0.919409, -0.128348,  0.371772, -0.964166, -0.218863,  0.149946, -0.042317,  0.439112,  0.897435, -0.289233,  0.953879, -0.080363,  0.373070, -0.694380,  0.615350, -0.427609,  0.106403,  0.897680, -0.387549, -0.869131,  0.307274,  0.566585, -0.651638,  0.504331,  0.104179, -0.508489,  0.854743,  0.730449, -0.562191,  0.387795,  0.077065,  0.986498,  0.144512,  0.724810,  0.646143,  0.239060, -0.110219,  0.993321,  0.034130,  0.799693, -0.183204,  0.571776,  0.528246, -0.515267,  0.674874,  0.791119,  0.466065,  0.396124, -0.436720, -0.438399,  0.785546, -0.995695,  0.005229,  0.092547,  0.506834,  0.566848,  0.649464,  0.562470, -0.093330,  0.821533, -0.881501,  0.092396,  0.463053, -0.162386,  0.786560,  0.595780, -0.293284, -0.319208,  0.901161,  0.649232, -0.267907,  0.711845,  0.206709, -0.829645,  0.518614,  0.024075, -0.914801,  0.403186, -0.722970, -0.214547, -0.656722, -0.771644,  0.246611,  0.586301, -0.728270,  0.481224,  0.487900,  0.626820, -0.777199,  0.055296,  0.128009,  0.989513, -0.066915, -0.984425, -0.156364, -0.080356, -0.449416,  0.504878,  0.736969,  0.898230, -0.396061, -0.190576,  0.348232,  0.484047,  0.802766,  0.842650,  0.260598,  0.471200, -0.813341, -0.581726,  0.008434,  0.972861, -0.228593, -0.035868,  0.548861,  0.824187, -0.139524, -0.024370,  0.226494,  0.973708, -0.621869,  0.182041,  0.761669, -0.314173,  0.662752,  0.679747,  0.379345,  0.272966,  0.884074, -0.048736, -0.641802,  0.765320, -0.088973,  0.954041,  0.286164, -0.239616,  0.277898,  0.930245, -0.720241, -0.667980,  0.187231};
static float ps134 [402] = {  0.525053, -0.265196,  0.808697,  0.519761, -0.611767, -0.596314,  0.130884,  0.739045,  0.660819,  0.714835,  0.693771,  0.087710, -0.556259, -0.455018,  0.695367,  0.738033, -0.448285,  0.504328,  0.082141, -0.389038,  0.917552, -0.309743,  0.891864, -0.329603,  0.187040, -0.716672,  0.671861,  0.937505, -0.211744,  0.276132,  0.399529, -0.630131,  0.665817, -0.409394, -0.908484,  0.083986,  0.214994, -0.539443,  0.814113, -0.715928,  0.511211,  0.475511, -0.919134, -0.064221,  0.388674, -0.690920,  0.203655,  0.693653, -0.212922, -0.975505,  0.055262,  0.323735,  0.257956,  0.910305, -0.796655,  0.305565,  0.521508, -0.490128, -0.830528,  0.264571, -0.396320,  0.788284,  0.470679,  0.909757,  0.191001,  0.368593, -0.080613, -0.887964,  0.452793, -0.263236,  0.915964,  0.302848, -0.148487,  0.982360,  0.113672,  0.045863,  0.862090,  0.504676,  0.875186, -0.476257, -0.085028, -0.337515, -0.483599,  0.807599,  0.093978, -0.950808,  0.295181,  0.143926, -0.855052,  0.498168,  0.952609, -0.270367, -0.139420, -0.751051,  0.659680,  0.027293, -0.869152, -0.299649,  0.393429,  0.694431,  0.357858,  0.624262, -0.980063,  0.048239,  0.192740, -0.957670, -0.197171,  0.209744, -0.426575,  0.491147,  0.759479,  0.930729, -0.353305,  0.094439, -0.087177, -0.986800, -0.136476,  0.420125,  0.847745,  0.323765, -0.094606,  0.731356,  0.675403,  0.364523, -0.778171,  0.511441,  0.572580, -0.635851,  0.517538, -0.728410, -0.377792,  0.571569,  0.821426, -0.551107,  0.146767,  0.611592, -0.067386,  0.788299, -0.201878, -0.648339,  0.734099, -0.218371,  0.548751,  0.806962, -0.618781, -0.780286,  0.090910,  0.773022,  0.572307,  0.273682, -0.832191,  0.075406,  0.549338, -0.527881, -0.842065, -0.110758,  0.713377, -0.611049,  0.343091, -0.032376, -0.769418,  0.637924, -0.364000,  0.928542,  0.072902, -0.075522,  0.384793,  0.919908, -0.126707, -0.960799,  0.246597,  0.670195,  0.138297,  0.729186,  0.383517,  0.452997,  0.804803, -0.300919, -0.890656,  0.340852,  0.777662, -0.051309,  0.626586,  0.506109,  0.709225,  0.490769, -0.840664,  0.241099, -0.484927, -0.866147,  0.386417,  0.316973, -0.996791, -0.078126,  0.017454, -0.772838, -0.507518,  0.380982,  0.545414,  0.520831,  0.656703,  0.989006, -0.142037,  0.041144,  0.851398,  0.523177,  0.037518,  0.589960, -0.461012,  0.662884, -0.312887,  0.688503,  0.654267,  0.701842, -0.261223,  0.662707, -0.505687,  0.283759,  0.814715, -0.773790,  0.575381,  0.264926, -0.915365,  0.173693,  0.363233, -0.442723, -0.101989,  0.890839,  0.615891,  0.730347,  0.295418,  0.192776,  0.923459,  0.331753,  0.286043,  0.813551,  0.506276, -0.880038, -0.430483,  0.200545, -0.268483, -0.271866,  0.924124,  0.321512,  0.936440,  0.140390, -0.042720,  0.946336,  0.320348, -0.614866, -0.587524,  0.526076, -0.644170,  0.737808,  0.201704, -0.594987,  0.691806,  0.409140,  0.451318, -0.878337,  0.157593, -0.706770, -0.027572,  0.706906, -0.650318, -0.687813,  0.322488,  0.412994, -0.451978,  0.790665,  0.655698, -0.736533,  0.166067, -0.296232,  0.343575,  0.891181, -0.181032,  0.844862,  0.503424, -0.000000, -0.000000,  1.000000, -0.210771, -0.580805, -0.786283, -0.362948,  0.124780,  0.923417, -0.945040, -0.326275,  0.021064, -0.058898, -0.223752,  0.972865,  0.294182, -0.309803,  0.904146, -0.464340,  0.846732,  0.259679,  0.810859,  0.168913,  0.560336, -0.225681, -0.051538,  0.972837, -0.558517,  0.060703,  0.827269,  0.246917,  0.053767,  0.967544, -0.625688,  0.416476,  0.659593,  0.368480,  0.660082,  0.654610, -0.256950, -0.790687,  0.555689,  0.812456,  0.386599,  0.436414, -0.415243, -0.634020,  0.652374,  0.526498, -0.775666,  0.348054,  0.898919, -0.027521,  0.437250, -0.771007, -0.619432,  0.147828,  0.241650, -0.963712,  0.113421, -0.808544, -0.166434,  0.564408,  0.007918, -0.603710,  0.797164,  0.083121,  0.207628,  0.974670,  0.853393, -0.413663,  0.317180,  0.151868,  0.403214,  0.902416, -0.639618, -0.224670,  0.735127, -0.130574, -0.449698,  0.883585, -0.466905, -0.304360,  0.830280, -0.561375,  0.827271,  0.021922,  0.675562,  0.562071,  0.477172,  0.397033, -0.106025,  0.911660, -0.147862,  0.174204,  0.973545,  0.021756, -0.997176,  0.071875,  0.461969,  0.752437, -0.469492,  0.890442,  0.397266,  0.222021, -0.974927,  0.004503, -0.222480,  0.160045, -0.154584,  0.974930, -0.004403,  0.578617,  0.815587,  0.476591,  0.099535,  0.873472,  0.537740,  0.304863,  0.786062,  0.966165,  0.203994,  0.157835};
static float ps135 [405] = { -0.389809,  0.464682,  0.795059, -0.661547, -0.721572,  0.204180, -0.470425,  0.618695,  0.629220, -0.304422, -0.910847,  0.278722,  0.193945,  0.273429,  0.942137,  0.300776,  0.445329,  0.843336, -0.841034,  0.145692,  0.520996,  0.785554, -0.608232,  0.113836,  0.625635, -0.764418,  0.155713,  0.494394,  0.474461,  0.728328, -0.057783,  0.636436,  0.769162, -0.084790, -0.819786,  0.566358,  0.708260, -0.701682, -0.077528, -0.489538, -0.852838,  0.181713, -0.480152, -0.149222,  0.864400,  0.166546, -0.622556,  0.764648,  0.050659,  0.443919,  0.894634,  0.136802, -0.776265,  0.615385,  0.798089,  0.454271,  0.395843,  0.078692,  0.090925,  0.992744, -0.563060, -0.350192,  0.748551, -0.826420,  0.456919,  0.329021,  0.433627,  0.274015,  0.858419,  0.979552, -0.196770,  0.041940, -0.964417, -0.264386, -0.000903, -0.053973, -0.675063,  0.735783,  0.789737,  0.579573,  0.201025,  0.513901,  0.652701,  0.556675,  0.966275,  0.154324,  0.206150, -0.364643,  0.923963,  0.115445,  0.310398,  0.794356,  0.522161, -0.796834, -0.444500,  0.409237,  0.328489,  0.094411,  0.939777,  0.552044, -0.509437,  0.660092,  0.205865, -0.089753,  0.974456,  0.528877, -0.680281,  0.507451,  0.558346,  0.091699,  0.824525, -0.351261,  0.873691,  0.336571, -0.149447, -0.318653,  0.936016,  0.101363, -0.895880,  0.432580,  0.623563,  0.287956,  0.726809, -0.750064, -0.312393,  0.582936,  0.358391, -0.667894,  0.652284,  0.903696,  0.384141,  0.189130,  0.322399, -0.813286,  0.484380,  0.636757, -0.114052,  0.762583, -0.168419,  0.966315,  0.194601,  0.121262,  0.757166,  0.641869,  0.007906, -0.999438, -0.032565, -0.493839,  0.267795,  0.827290,  0.324223, -0.269065,  0.906909,  0.445558, -0.089502,  0.890768, -0.665006, -0.161537,  0.729159, -0.229665, -0.523312,  0.820609, -0.381035,  0.069536,  0.921942, -0.736052,  0.035995,  0.675967,  0.040358,  0.962801,  0.267179, -0.175278,  0.468352,  0.865982,  0.354127,  0.934649, -0.032016,  0.265996,  0.905198,  0.331455, -0.891262, -0.240233,  0.384631, -0.267971,  0.643786,  0.716750,  0.904021, -0.124977,  0.408810, -0.910620, -0.361285,  0.200610,  0.925662, -0.285138,  0.248688,  0.190082, -0.438401,  0.878450, -0.907214,  0.245811,  0.341378,  0.215651, -0.975219,  0.049431, -0.268503, -0.692865,  0.669211,  0.973058, -0.052346,  0.224541,  0.677350, -0.655297,  0.334339, -0.832776, -0.105911,  0.543385, -0.678316,  0.244009,  0.693070,  0.380122, -0.485702,  0.787147, -0.631336, -0.746653, -0.209582, -0.838264,  0.532502,  0.117282,  0.203850,  0.971762,  0.118847, -0.546611,  0.800833,  0.244709,  0.790339, -0.122653,  0.600267,  0.874642,  0.484703, -0.008042,  0.777197,  0.277764,  0.564634, -0.631780, -0.501458,  0.591096,  0.484276, -0.810730,  0.328927,  0.480553,  0.792067,  0.376429, -0.475928, -0.670075,  0.569642, -0.666477,  0.555677,  0.497023, -0.432812, -0.527279,  0.731198,  0.414362, -0.900591,  0.131299, -0.046306, -0.101087,  0.993799, -0.295620, -0.821596,  0.487431, -0.704916,  0.644221,  0.296771, -0.058537,  0.271568,  0.960638,  0.156069,  0.597899,  0.786231,  0.876365,  0.071154,  0.476363,  0.265947, -0.920208,  0.287211,  0.823647, -0.324426,  0.465138,  0.082446,  0.879870,  0.468008, -0.316880,  0.781084,  0.538047, -0.571587,  0.054100,  0.818756,  0.519989, -0.305367,  0.797723, -0.802510, -0.556855,  0.214222,  0.662211,  0.475636,  0.579005, -0.020707, -0.493174,  0.869684, -0.743648, -0.668571, -0.000216,  0.074252, -0.274903,  0.958601,  0.735978,  0.083936,  0.671782, -0.933341,  0.330985,  0.139006, -0.588974,  0.439353,  0.678291, -0.272927, -0.131753,  0.952970, -0.494760, -0.778950,  0.385292, -0.363349, -0.341516,  0.866801,  0.999397,  0.033589,  0.008846,  0.824592, -0.482401,  0.295528, -0.663217, -0.626246,  0.409828, -0.151459, -0.980577,  0.124612, -0.979675,  0.103866,  0.171607,  0.891594,  0.253671,  0.375115,  0.428594,  0.885799,  0.177956,  0.343155,  0.640616,  0.686918, -0.141220,  0.899551,  0.413357, -0.167372,  0.081936,  0.982483, -0.102035,  0.789299,  0.605472,  0.772161, -0.360849, -0.523025,  0.573667,  0.818955,  0.014811,  0.688388, -0.321942,  0.649981, -0.106320, -0.925715,  0.362969,  0.904354, -0.420576,  0.072518, -0.930088,  0.002161,  0.367330,  0.701797, -0.508654,  0.498750, -0.545028,  0.837835,  0.031263, -0.286433,  0.279742,  0.916352, -0.056943,  0.976387, -0.208390,  0.971819,  0.135479, -0.192906,  0.664320,  0.633215,  0.397137, -0.521803,  0.727643,  0.445261};
static float ps136 [408] = {  0.603118, -0.447541,  0.660270,  0.538612, -0.255123,  0.803000, -0.846727, -0.496518,  0.191109, -0.399120, -0.005889,  0.916880,  0.218033, -0.883957,  0.413621, -0.554135, -0.814231,  0.173094,  0.229960, -0.434865,  0.870638,  0.283124,  0.114775,  0.952191, -0.272992,  0.861457,  0.428214,  0.951634, -0.162950,  0.260461,  0.386513, -0.881211,  0.272168, -0.704745, -0.428964,  0.565088, -0.466637, -0.235774,  0.852444,  0.781507, -0.035322,  0.622895,  0.091744, -0.290195,  0.952560, -0.014229, -0.994882,  0.100040,  0.860082, -0.496287,  0.118148,  0.878332, -0.370312,  0.302328, -0.798684,  0.596969,  0.075705,  0.587039, -0.743368,  0.320606, -0.257196,  0.419388,  0.870611, -0.795790, -0.605305, -0.017977,  0.500824,  0.145669,  0.853203,  0.641210,  0.766740,  0.030985,  0.977722,  0.054896,  0.202598, -0.348931,  0.936558, -0.033251, -0.218849, -0.975751,  0.003860, -0.582365, -0.044904,  0.811686, -0.727359, -0.565362,  0.388993, -0.370553,  0.583290,  0.722816, -0.337207,  0.250773, -0.907416,  0.913147,  0.183365,  0.364061,  0.232847,  0.668845,  0.705995, -0.850279, -0.371939,  0.372408,  0.968299, -0.205026, -0.142695,  0.363660,  0.316129,  0.876250, -0.305293,  0.928620,  0.210860, -0.934109,  0.111816,  0.339024, -0.655278,  0.338047,  0.675526,  0.296581,  0.932760,  0.204935, -0.625033,  0.639280,  0.447945,  0.903864, -0.413049, -0.111443,  0.883490, -0.335614, -0.326816, -0.432609,  0.717016,  0.546568, -0.923491, -0.135259,  0.358984, -0.100246,  0.952710,  0.286870, -0.170392, -0.571766,  0.802527, -0.790723,  0.536772,  0.294334,  0.050840,  0.598770,  0.799306,  0.402956,  0.707516,  0.580558,  0.233180, -0.766181,  0.598827,  0.623148, -0.049027,  0.780565,  0.655639, -0.754205, -0.036220,  0.934678,  0.303286,  0.185457, -0.177145, -0.951408,  0.251876,  0.451841,  0.890971,  0.044831, -0.820575,  0.239450,  0.518960, -0.327374, -0.411069,  0.850793,  0.902031, -0.021877,  0.431117, -0.164321,  0.605060,  0.779038,  0.533044, -0.837352,  0.121265,  0.722993, -0.673743,  0.152813,  0.710478, -0.244390,  0.659920, -0.367229, -0.602015,  0.709028, -0.912791, -0.408427,  0.000643, -0.267211, -0.192019,  0.944313,  0.491340, -0.861695, -0.126752, -0.739603, -0.082066,  0.668021, -0.384363, -0.851584,  0.356469,  0.237081, -0.615324,  0.751777, -0.563936,  0.533108,  0.630692,  0.021483, -0.837846,  0.545484,  0.130299,  0.278388,  0.951589,  0.424262, -0.773159,  0.471409,  0.428180, -0.056195,  0.901944,  0.027462, -0.695395,  0.718103, -0.562300, -0.601677,  0.567277, -0.571852, -0.726255,  0.381497,  0.709920,  0.377780,  0.594387,  0.552518,  0.358554,  0.752438,  0.835804,  0.505464,  0.214333, -0.655515,  0.713130,  0.248486, -0.538224,  0.177328,  0.823935,  0.812207,  0.179064,  0.555208,  0.557812,  0.713260,  0.424390, -0.148790,  0.987435,  0.053239, -0.032793,  0.428462,  0.902965,  0.985191,  0.171274,  0.007955,  0.029863, -0.515499,  0.856369,  0.996621, -0.075328,  0.032742,  0.754519, -0.421863,  0.502726,  0.018575, -0.940301,  0.339836, -0.179983, -0.740792,  0.647173,  0.985030,  0.034088, -0.168978,  0.713567,  0.557883,  0.423779, -0.465718,  0.387259,  0.795700,  0.062645,  0.987710,  0.143192,  0.756451, -0.562570,  0.333611,  0.435696, -0.629832,  0.643024, -0.223639,  0.753612,  0.618106, -0.050812,  0.872630,  0.485731, -0.337842,  0.215563,  0.916185,  0.403342,  0.525470,  0.749131, -0.744295,  0.448550,  0.494800, -0.814018, -0.232856,  0.532120,  0.516414,  0.822596,  0.238020,  0.193982, -0.089102,  0.976950, -0.384996, -0.745886,  0.543537, -0.114395,  0.239354,  0.964170,  0.184596,  0.819174,  0.543027,  0.696336,  0.679915,  0.229851,  0.849208, -0.224849,  0.477796, -0.192454,  0.034329,  0.980705,  0.356312,  0.843717,  0.401477, -0.474815,  0.808874,  0.346803, -0.857062,  0.015534,  0.514979, -0.049875, -0.132971,  0.989864, -0.943181, -0.275582,  0.185643,  0.832384,  0.380026,  0.403381, -0.365817, -0.918251,  0.151638,  0.129788,  0.926143,  0.354138, -0.184356, -0.867254,  0.462475,  0.572135,  0.553937,  0.604826, -0.713893,  0.131695,  0.687760, -0.121926, -0.361205,  0.924481, -0.644246, -0.270209,  0.715496,  0.212670,  0.467883,  0.857821,  0.044476,  0.076448,  0.996081,  0.608938, -0.614092,  0.502081, -0.519637, -0.444685,  0.729543,  0.427031, -0.451714,  0.783326,  0.674271,  0.166497,  0.719471, -0.000146,  0.753625,  0.657304, -0.715637, -0.673140,  0.186402,  0.189336, -0.963768,  0.187893,  0.951922, -0.295670,  0.080154};
static float ps137 [411] = {  0.355304, -0.564163,  0.745305,  0.126109, -0.204864,  0.970632, -0.799464, -0.589124,  0.117432,  0.439636,  0.686103,  0.579640, -0.078776, -0.942545,  0.324659, -0.837657, -0.272157,  0.473562,  0.742146, -0.105901,  0.661819, -0.563647,  0.167314,  0.808893, -0.520776, -0.781564,  0.343439,  0.985058,  0.064126,  0.159837,  0.898186, -0.302546,  0.318949, -0.511862,  0.704953,  0.490957, -0.307538,  0.530331,  0.790044,  0.538635,  0.209147,  0.816168, -0.447475,  0.355080,  0.820783,  0.657819, -0.752669,  0.027635, -0.909193, -0.066347,  0.411055,  0.020688,  0.954223,  0.298380,  0.225388,  0.402301,  0.887330,  0.605038,  0.755999,  0.249791, -0.552552,  0.819050,  0.154417, -0.381443, -0.765849,  0.517665, -0.540809,  0.817747, -0.197018,  0.001751, -0.741264,  0.671212, -0.785459,  0.522807,  0.331251, -0.912812, -0.400305,  0.080809,  0.760164, -0.417069,  0.498201, -0.875157,  0.469535,  0.116780,  0.923669,  0.172493,  0.342173,  0.839227, -0.206774,  0.502935,  0.233953,  0.936364,  0.261705, -0.728647, -0.460872,  0.506627, -0.325472,  0.933649,  0.149560, -0.801255, -0.059030,  0.595404, -0.980597, -0.193943,  0.028539,  0.753615,  0.591995,  0.285666,  0.736576,  0.135405,  0.662662, -0.349137,  0.178048,  0.920001, -0.152734, -0.846515,  0.509985,  0.638126, -0.698804, -0.323215,  0.671059, -0.314830,  0.671240, -0.441580,  0.896879, -0.024792,  0.571369, -0.508577,  0.644118, -0.747091,  0.649520,  0.141347, -0.055003,  0.862974,  0.502245, -0.959546,  0.269861,  0.080294, -0.437479, -0.884668,  0.161165,  0.348000, -0.902446,  0.253942, -0.072905,  0.864383, -0.497521,  0.165511,  0.869547,  0.465290, -0.457786, -0.017947,  0.888881, -0.806753,  0.585953, -0.076213, -0.925098, -0.271091,  0.265899,  0.248004, -0.399001,  0.882776, -0.742086,  0.153302,  0.652539,  0.855898,  0.401085,  0.326449,  0.612399,  0.010976,  0.790473,  0.438952, -0.683109,  0.583680,  0.434484,  0.873604,  0.219178,  0.462509, -0.383302,  0.799478,  0.333091,  0.942250,  0.034866, -0.093543,  0.566563,  0.818692, -0.644403,  0.345838,  0.682012, -0.998075,  0.053866,  0.030731,  0.657329,  0.338574,  0.673265, -0.104597,  0.988343,  0.110621,  0.553534,  0.521471,  0.649360, -0.163719,  0.721036,  0.673278, -0.100427, -0.195307,  0.975587,  0.389729,  0.819581,  0.419999,  0.812816, -0.500780,  0.297573, -0.441175, -0.601351,  0.666140,  0.242097,  0.743248,  0.623676,  0.854790,  0.508399,  0.104235, -0.666183,  0.538922,  0.515522,  0.143952, -0.584557,  0.798481, -0.955097,  0.134907,  0.263797, -0.220476,  0.367195,  0.903636, -0.368618,  0.674157,  0.640026, -0.972928,  0.180778, -0.143984, -0.833615, -0.459733,  0.306157,  0.599002,  0.667262,  0.442672,  0.501918, -0.766755,  0.400206, -0.213047, -0.965915,  0.147037, -0.115572,  0.189424,  0.975070,  0.028614, -0.398921,  0.916539, -0.280536,  0.821095,  0.497094, -0.869655,  0.147457,  0.471122, -0.426985,  0.841295,  0.331521,  0.441133,  0.396064,  0.805317, -0.721712, -0.688953, -0.066903,  0.425734,  0.005968,  0.904829,  0.004100,  0.393457,  0.919334,  0.195267, -0.925844, -0.323548, -0.786694,  0.350619,  0.508113,  0.292535, -0.838272,  0.460135,  0.219865,  0.001785,  0.975529,  0.548845, -0.190803,  0.813857, -0.640635, -0.753040,  0.150058, -0.549192, -0.216828,  0.807077,  0.041882,  0.743082,  0.667888,  0.010408, -0.993725,  0.111362, -0.912223,  0.395706, -0.106139,  0.113438,  0.205246,  0.972114, -0.116353, -0.990371, -0.075019,  0.815325,  0.276143,  0.508910, -0.193742, -0.377838,  0.905374, -0.710702, -0.254862,  0.655704, -0.067976, -0.577195,  0.813772, -0.939018,  0.070489, -0.336566, -0.220182, -0.712140,  0.666616,  0.940705,  0.303831,  0.150865,  0.647865, -0.605164,  0.462653,  0.120813,  0.583542,  0.803046, -0.336071, -0.202790,  0.919746,  0.331042,  0.573050,  0.749683, -0.278121, -0.543283,  0.792144, -0.591205, -0.437281,  0.677689, -0.306124, -0.887849,  0.343530, -0.583223, -0.631961,  0.510369,  0.000000, -0.000000,  1.000000,  0.344140, -0.202282,  0.916869, -0.652548, -0.032658,  0.757043,  0.221637, -0.733799,  0.642196,  0.548399,  0.835680,  0.029971, -0.230502,  0.970342, -0.072833,  0.692555, -0.674220,  0.256506,  0.139625, -0.944853,  0.296239, -0.234041, -0.008023,  0.972194, -0.891064,  0.334621,  0.306650, -0.519479,  0.520942,  0.677319,  0.858954,  0.033313,  0.510968,  0.720717,  0.484817,  0.495499, -0.424703, -0.388931,  0.817533,  0.332581,  0.209792,  0.919444, -0.975913, -0.071483,  0.206117, -0.702088, -0.632272,  0.327573};
static float ps138 [414] = {  0.025243,  0.611025,  0.791209, -0.985741, -0.147452,  0.081069,  0.846665, -0.363624,  0.388506,  0.661409,  0.746442,  0.073234,  0.796044,  0.275655,  0.538821,  0.489356, -0.866225,  0.100925,  0.907901,  0.235456,  0.346809, -0.946762, -0.113975,  0.301083, -0.760809,  0.399582,  0.511375, -0.147147,  0.981430,  0.123058, -0.608348,  0.741758,  0.282325, -0.434906, -0.728077,  0.529870, -0.084275, -0.741766,  0.665343, -0.716202, -0.619942,  0.320509,  0.950619,  0.019430,  0.309752, -0.839293,  0.473862,  0.266537, -0.206609,  0.617268,  0.759140,  0.033683, -0.433079,  0.900726,  0.038443,  0.867066,  0.496707,  0.282973, -0.935540,  0.211405, -0.094450,  0.760916,  0.641939, -0.324795,  0.737856,  0.591673,  0.722976,  0.201753, -0.660759, -0.401521, -0.032146,  0.915285, -0.066830, -0.992062, -0.106517, -0.537732,  0.669149,  0.512917,  0.135282, -0.787480,  0.601310,  0.647269,  0.640039,  0.413996, -0.134741, -0.564190,  0.814576,  0.017423, -0.221299,  0.975050, -0.857822, -0.046054,  0.511880,  0.529037,  0.584369,  0.615331,  0.601379, -0.692226,  0.398957,  0.768759, -0.567434,  0.295004,  0.296237, -0.917864, -0.264140, -0.402771,  0.195392,  0.894202,  0.698489,  0.471966,  0.537924,  0.580793,  0.007531,  0.814016,  0.757359,  0.611641,  0.228695, -0.120285, -0.986766,  0.108739, -0.341980, -0.465023,  0.816581,  0.436610,  0.462390,  0.771730, -0.574616,  0.103812,  0.811813, -0.843158, -0.507172,  0.178497,  0.875104, -0.144006,  0.462013,  0.380541, -0.768125,  0.514949, -0.200181,  0.862826,  0.464176,  0.052837, -0.955434,  0.290436,  0.645005, -0.739397,  0.193029, -0.994959,  0.028035, -0.096280,  0.855517,  0.068759,  0.513189, -0.090873,  0.457681,  0.884461, -0.946177,  0.277802,  0.166057, -0.169876, -0.933211,  0.316639,  0.476146,  0.876352,  0.072749,  0.967867, -0.247151,  0.046376, -0.236298, -0.822090,  0.518007,  0.639815, -0.767737, -0.034883, -0.195934, -0.146122,  0.969669,  0.944800, -0.202048,  0.257934, -0.573183, -0.126034,  0.809677, -0.325444,  0.441249,  0.836293,  0.750796, -0.071423,  0.656662,  0.479847, -0.864119, -0.151805,  0.758527, -0.284945,  0.586040,  0.431823, -0.328622,  0.839962,  0.208618, -0.068529,  0.975593, -0.616918, -0.597247,  0.512550, -0.530574, -0.836825,  0.134965,  0.898005,  0.404766,  0.172485,  0.427593, -0.516973,  0.741555,  0.516478,  0.227440,  0.825543,  0.704195,  0.698980, -0.124646, -0.786579,  0.613048, -0.073932, -0.971964, -0.189743, -0.138863,  0.117867,  0.397249,  0.910110, -0.293767, -0.661565,  0.689951,  0.326203, -0.945234, -0.011123,  0.818206,  0.444918,  0.364124, -0.427385,  0.575434,  0.697293, -0.209371,  0.070182,  0.975314,  0.076586, -0.630465,  0.772430,  0.416715, -0.126852,  0.900143,  0.449435, -0.834578,  0.318573, -0.880299,  0.293670,  0.372601, -0.276450, -0.956855, -0.089468,  0.321122,  0.321242,  0.890889, -0.757988,  0.633742,  0.154356, -0.936374,  0.091773,  0.338794,  0.607808, -0.409895,  0.680114,  0.102089, -0.991036,  0.086175, -0.842747, -0.251741,  0.475820,  0.706712, -0.501688,  0.498865,  0.183933,  0.159480,  0.969915, -0.333458, -0.934232,  0.126554, -0.684120, -0.407716,  0.604771, -0.705255,  0.584263,  0.401562,  0.286994, -0.667402,  0.687174, -0.726379,  0.016115,  0.687105, -0.610146,  0.486579,  0.625271, -0.515674,  0.351476,  0.781374,  0.236576,  0.547513,  0.802659,  0.890559, -0.419528,  0.175784,  0.235895, -0.485901,  0.841578, -0.555821, -0.758977,  0.339142, -0.822038,  0.166861,  0.544437,  0.459982,  0.753497,  0.469744,  0.135020,  0.739728,  0.659221, -0.887983,  0.457722,  0.044454, -0.568368, -0.778932, -0.264994, -0.427730,  0.812944,  0.395183, -0.934082, -0.355458,  0.033772, -0.990343,  0.065511,  0.122185,  0.621280,  0.357036,  0.697521, -0.375671, -0.881008, -0.287570,  0.381739,  0.086856,  0.920180, -0.823721, -0.566236, -0.029336,  0.682815, -0.258350, -0.683388, -0.382632, -0.253042,  0.888573, -0.021682, -0.875646,  0.482467,  0.259362,  0.833223,  0.488334,  0.228108, -0.283446,  0.931464,  0.214186, -0.285446, -0.934155,  0.056134, -0.946943, -0.316461,  0.220914, -0.881280,  0.417784, -0.015052,  0.226992,  0.973780,  0.339622,  0.675274,  0.654723, -0.913308, -0.315906,  0.257043, -0.545271, -0.342806,  0.764960, -0.000000, -0.000000,  1.000000,  0.166196,  0.937701,  0.305116, -0.372655, -0.864808,  0.336505,  0.608857, -0.205634,  0.766164,  0.705782,  0.144238,  0.693590, -0.490741, -0.545242,  0.679621, -0.794505, -0.445508,  0.412656,  0.527162, -0.618320,  0.582907, -0.173620, -0.358330,  0.917309};
static float ps139 [417] = {  0.748102, -0.265722,  0.608059, -0.263203,  0.742514,  0.615952,  0.718025, -0.681562, -0.141113, -0.497436, -0.860565,  0.109478, -0.818075, -0.569096,  0.082960, -0.843272,  0.374655,  0.385391, -0.018611, -0.888802,  0.457914,  0.855607, -0.457294,  0.242528,  0.592142,  0.145886,  0.792518, -0.385815, -0.872379,  0.300169,  0.875478,  0.135921,  0.463750,  0.367502,  0.673346,  0.641519, -0.210833, -0.095891,  0.972807,  0.556598, -0.785726,  0.269878, -0.805756,  0.195364,  0.559097, -0.111298,  0.100758,  0.988666,  0.461189, -0.740625, -0.488650,  0.473366, -0.434359,  0.766327,  0.292833, -0.585072,  0.756267,  0.531192,  0.690333,  0.491198,  0.688408, -0.057372,  0.723051, -0.348834,  0.071945,  0.934419, -0.226256,  0.968260, -0.106209,  0.995290,  0.091516, -0.031991, -0.324704, -0.687489,  0.649558,  0.860747, -0.281798,  0.423915,  0.548429, -0.829134, -0.108456, -0.442405,  0.598876,  0.667552,  0.650360,  0.693262,  0.310514,  0.592970, -0.251882,  0.764815,  0.983341, -0.077318, -0.164508, -0.862722, -0.424093,  0.275420, -0.060671, -0.254006,  0.965298, -0.954766, -0.221677,  0.198197,  0.162216,  0.781631,  0.602278, -0.141992, -0.446009,  0.883693, -0.766237, -0.431398,  0.476210,  0.829790, -0.067657,  0.553959,  0.650130,  0.343147,  0.677924, -0.258932, -0.842834,  0.471789, -0.633144, -0.601780,  0.486816,  0.941301, -0.258215,  0.217434,  0.033758,  0.969290,  0.243594, -0.932433,  0.305334,  0.193234,  0.332077,  0.493308,  0.803973,  0.854250,  0.023125, -0.519347,  0.037257, -0.056423,  0.997712, -0.241508,  0.595274,  0.766370,  0.749659,  0.651288,  0.117621,  0.989823, -0.042874,  0.135693,  0.790625,  0.342150,  0.507785, -0.068022,  0.894348,  0.442170,  0.519134,  0.520705,  0.677766, -0.434756, -0.133727,  0.890564, -0.336541, -0.508829,  0.792359,  0.209333, -0.867843,  0.450587,  0.956586,  0.121281,  0.265015,  0.100499,  0.465641,  0.879249, -0.646912, -0.393992,  0.652898, -0.740775, -0.604079,  0.293839,  0.276846, -0.407498,  0.870231,  0.580672,  0.755635, -0.303045, -0.280939, -0.952764,  0.115382, -0.245504,  0.266890,  0.931932, -0.276022,  0.855733,  0.437645, -0.722806,  0.400133,  0.563422,  0.871589,  0.471212,  0.135248, -0.470768,  0.243086,  0.848108,  0.651826, -0.754892,  0.072538,  0.761869, -0.473510,  0.441977,  0.437647, -0.778784,  0.449399, -0.456570,  0.884951, -0.091677, -0.961248, -0.256121, -0.102005, -0.364503,  0.429258,  0.826363,  0.162601,  0.984335,  0.068155, -0.943823, -0.053459,  0.326099, -0.051688,  0.780096,  0.623521,  0.401473, -0.232813,  0.885786,  0.640475, -0.453151,  0.620037, -0.615685,  0.584879,  0.528062,  0.929463, -0.079555,  0.360235, -0.561072,  0.421381,  0.712486,  0.348718,  0.815095,  0.462617,  0.618787, -0.641519,  0.453383,  0.096817, -0.955332,  0.279226, -0.978338,  0.206999, -0.002625,  0.478168,  0.829844,  0.287602, -0.127445,  0.444682,  0.886575,  0.334520, -0.938626, -0.084134,  0.388675,  0.139758,  0.910714, -0.766591, -0.214442,  0.605271,  0.802578, -0.594374,  0.050866, -0.194186,  0.948331,  0.250920,  0.726708, -0.635423,  0.261022, -0.657250,  0.222935,  0.719946,  0.251616,  0.312599,  0.915954,  0.922072,  0.381739, -0.063709,  0.153810,  0.125135,  0.980145,  0.507657, -0.047074,  0.860273, -0.506402, -0.558605,  0.656900,  0.167273, -0.233047,  0.957971, -0.421905,  0.862433,  0.279653, -0.038852, -0.992637,  0.114730, -0.461419, -0.740843,  0.488103,  0.082018, -0.618930,  0.781152, -0.154736, -0.942543,  0.296089,  0.589355,  0.801774,  0.099090, -0.132587, -0.624189,  0.769941, -0.874559, -0.245433,  0.418221,  0.142815,  0.894244,  0.424183,  0.672207,  0.528403,  0.518583,  0.286287, -0.050181,  0.956829, -0.393984, -0.915788, -0.078165,  0.069077, -0.437911,  0.896361,  0.298597, -0.735778,  0.607841, -0.120241, -0.774290,  0.621303, -0.032965,  0.629797,  0.776060, -0.675976, -0.730326,  0.098390, -0.744653,  0.564992,  0.355353, -0.602037,  0.730520,  0.322322,  0.091169, -0.771350,  0.629848,  0.089501, -0.994204, -0.059567, -0.557119,  0.040213,  0.829458,  0.471644,  0.329740,  0.817816,  0.783712,  0.524254,  0.333095, -0.846722,  0.504503,  0.168933, -0.274674, -0.926482, -0.257263,  0.342385, -0.896834,  0.280111,  0.911653, -0.410050,  0.027362,  0.913649, -0.163585, -0.372136, -0.487235, -0.351206,  0.799535, -0.295107, -0.296863,  0.908176,  0.890373,  0.329462,  0.314150,  0.005008,  0.281008,  0.959692, -0.618972, -0.177584,  0.765074,  0.177330,  0.633786,  0.752907,  0.753637,  0.144278,  0.641261,  0.485204, -0.613019,  0.623526, -0.724918,  0.008326,  0.688785};
static float ps140 [420] = {  0.080095, -0.572887,  0.815711,  0.114949, -0.177343,  0.977413,  0.207564, -0.819906,  0.533546, -0.260627, -0.856956,  0.444635,  0.758050,  0.643824,  0.104171,  0.345938, -0.860973,  0.372896, -0.652071, -0.574575,  0.494638, -0.297334,  0.267242,  0.916610, -0.262774, -0.118674,  0.957531,  0.865353, -0.251323,  0.433592, -0.853992,  0.492899,  0.166577, -0.302617, -0.323999,  0.896352, -0.435007,  0.607244,  0.664849, -0.617339,  0.571559,  0.540567, -0.872458,  0.339547,  0.351459,  0.485936,  0.767876,  0.417411, -0.832877,  0.190332,  0.519701, -0.782757, -0.551066,  0.289168, -0.014041,  0.584416,  0.811333,  0.072833,  0.945391,  0.317696,  0.698529, -0.609848,  0.374357, -0.870750,  0.037016, -0.490330, -0.410445,  0.065679,  0.909517, -0.597324,  0.038778,  0.801062,  0.642289, -0.389257,  0.660260,  0.720991, -0.673370, -0.163538,  0.043967, -0.989582,  0.137090, -0.334260,  0.924379,  0.183831,  0.322458,  0.635857,  0.701218,  0.993781,  0.017402,  0.109981, -0.959911, -0.234608, -0.153395,  0.192613, -0.980940, -0.025623, -0.820309,  0.522213, -0.233209, -0.975061,  0.209517, -0.073206,  0.282541, -0.943984,  0.170485,  0.880822,  0.328861,  0.340593, -0.938226, -0.105595, -0.329519, -0.887210, -0.227012,  0.401652,  0.450012,  0.866324,  0.216730, -0.789253, -0.410375,  0.456807,  0.523912,  0.273762,  0.806580,  0.536135, -0.748866,  0.389562, -0.089660, -0.233217,  0.968282, -0.147067, -0.781971,  0.605716, -0.108427, -0.948767,  0.296791,  0.762345,  0.397616,  0.510619,  0.307421, -0.261036,  0.915070, -0.229158,  0.610897,  0.757821,  0.671500, -0.717782,  0.184056,  0.000000, -0.000000,  1.000000,  0.953041, -0.111560,  0.281544, -0.281912,  0.759034,  0.586850, -0.702266, -0.703173,  0.111219, -0.497481, -0.358171,  0.790080,  0.610729,  0.441396,  0.657404,  0.283868,  0.889011,  0.359275, -0.936170, -0.349511,  0.037783,  0.388898,  0.921261,  0.006012, -0.507312, -0.546821,  0.666049, -0.166515, -0.984046,  0.062661,  0.096620, -0.383169,  0.918611, -0.471305,  0.741473,  0.477586,  0.836306,  0.183093,  0.516788, -0.464903, -0.149844,  0.872589,  0.305783,  0.781897,  0.543262, -0.960569, -0.089425,  0.263267,  0.408988, -0.718314,  0.562809,  0.124136,  0.712837,  0.690256, -0.327007, -0.688234,  0.647611, -0.637846, -0.720376, -0.272417,  0.428341, -0.903617,  0.001152, -0.639642, -0.178444,  0.747673,  0.095689,  0.851799,  0.515055, -0.527087, -0.838083,  0.140700, -0.756269,  0.011392,  0.654161, -0.376626,  0.446837,  0.811474,  0.395118,  0.127658,  0.909717, -0.627984,  0.695154,  0.349853, -0.743286,  0.388407,  0.544670, -0.443947, -0.828964,  0.340190, -0.946423,  0.289179,  0.143732,  0.655050,  0.594899,  0.465838,  0.908403, -0.336560,  0.248056,  0.451210, -0.099138,  0.886894, -0.116677,  0.877324,  0.465498, -0.766290,  0.528533,  0.365312,  0.057999, -0.740650,  0.669383,  0.768575, -0.431853,  0.472013,  0.747863,  0.011927,  0.663746,  0.133784, -0.933756,  0.331967, -0.902052, -0.357786,  0.241435, -0.686050,  0.223317,  0.692434, -0.839394, -0.537723,  0.079197,  0.593357, -0.580189,  0.557950,  0.905621, -0.422426,  0.037505, -0.939627,  0.132380,  0.315558,  0.783542,  0.536952,  0.312640, -0.519627,  0.824702,  0.223282,  0.788508, -0.614221,  0.031411, -0.613170,  0.789780,  0.016462, -0.008710, -0.877905,  0.478756, -0.309391, -0.927547,  0.209602, -0.016891, -0.995144, -0.096971, -0.505332,  0.252182,  0.825253,  0.253887, -0.021666,  0.966991, -0.629628, -0.708252,  0.319293,  0.985943,  0.151980, -0.069415,  0.070293,  0.384931,  0.920265,  0.152821,  0.170047,  0.973514, -0.477643, -0.715741,  0.509482, -0.991966,  0.073893,  0.102679,  0.234820,  0.959657,  0.154652, -0.662914, -0.387253,  0.640765, -0.155924,  0.433827,  0.887402, -0.198089,  0.073798,  0.977402, -0.880902, -0.017385,  0.472980, -0.574292,  0.422973,  0.700915, -0.593950, -0.801718, -0.066877,  0.500029,  0.627785,  0.596537, -0.129565, -0.628237,  0.767158,  0.297222,  0.322300,  0.898767, -0.134892,  0.956223,  0.259694,  0.287927, -0.467794,  0.835624,  0.589652,  0.060679,  0.805374, -0.322502, -0.516136,  0.793471,  0.485720, -0.332165,  0.808543, -0.064910,  0.234391,  0.969973,  0.455455, -0.533622,  0.712606,  0.697981,  0.233056,  0.677132, -0.782194, -0.203880,  0.588732, -0.326289,  0.863362,  0.384892,  0.879311,  0.454313,  0.142871,  0.261640, -0.656550,  0.707451,  0.426342,  0.461375,  0.778053,  0.760920, -0.213250,  0.612801,  0.200819,  0.519961,  0.830248,  0.621903, -0.163941,  0.765741, -0.079500,  0.752988,  0.653214,  0.493865, -0.847052,  0.196469, -0.114699, -0.444700,  0.888305};
static float ps141 [423] = { -0.238685,  0.809396,  0.536571,  0.141187,  0.285999,  0.947771,  0.352335, -0.471245,  0.808572,  0.207844,  0.761711,  0.613675, -0.375049,  0.457864,  0.806039, -0.978509, -0.035145,  0.203189,  0.796490, -0.513705,  0.318922,  0.498482, -0.689520,  0.525430, -0.087598, -0.929786,  0.357525,  0.958959,  0.242369,  0.147154,  0.634645, -0.682235,  0.363017, -0.518126,  0.797798,  0.308324, -0.031237,  0.999389,  0.015680, -0.199669, -0.549318,  0.811408,  0.800373,  0.223405,  0.556321, -0.051412, -0.267455,  0.962198, -0.262785,  0.091169,  0.960537, -0.263556, -0.847014,  0.461634,  0.998916,  0.040084,  0.023647, -0.846326,  0.112384,  0.520675, -0.439728, -0.219727,  0.870838,  0.529834, -0.334963,  0.779151,  0.335309, -0.265916,  0.903801, -0.750191, -0.005870,  0.661195,  0.979306, -0.158520,  0.125821,  0.041978,  0.969693,  0.240692,  0.470672, -0.821398,  0.322139, -0.207847, -0.736017,  0.644265,  0.343787,  0.820381,  0.456930,  0.939871, -0.341263, -0.013498, -0.430717, -0.723017,  0.540119, -0.055240,  0.387136,  0.920366,  0.422706, -0.901867,  0.089195, -0.695647,  0.445778,  0.563345, -0.799021, -0.590786,  0.111969, -0.885334, -0.086914,  0.456759, -0.252045,  0.315495,  0.914842, -0.608127,  0.118514,  0.784943,  0.159290, -0.380562,  0.910933,  0.816623, -0.170147,  0.551522, -0.663654, -0.706737,  0.245125,  0.564268,  0.706720,  0.426788,  0.384594, -0.913002, -0.136070,  0.679302, -0.522703,  0.515102,  0.233625, -0.971559,  0.038621, -0.441634,  0.738004,  0.510205,  0.923446, -0.163638,  0.347088, -0.609344, -0.323731,  0.723808,  0.286867, -0.919237,  0.269649, -0.912220,  0.331462,  0.240806,  0.177650, -0.965991, -0.187889,  0.339902, -0.656739,  0.673172, -0.525877,  0.554555,  0.644920, -0.764948, -0.213501,  0.607678,  0.590675, -0.795636,  0.134409, -0.311450, -0.913646,  0.261247,  0.733906,  0.588142,  0.339810,  0.144195,  0.896978,  0.417898, -0.553381, -0.531963,  0.640924, -0.038989, -0.832143,  0.553189, -0.064691,  0.176166,  0.982232, -0.620074, -0.639336,  0.454707, -0.239965, -0.351077,  0.905075, -0.712609, -0.437784,  0.548209, -0.784793,  0.505919,  0.357974, -0.489895, -0.867960, -0.081538, -0.513918, -0.845711,  0.143740, -0.732638,  0.679413,  0.040498, -0.888807, -0.411321,  0.202084,  0.982410, -0.155348, -0.103625,  0.564348, -0.357466, -0.744130,  0.083139, -0.973794,  0.211693, -0.438676,  0.227216,  0.869446,  0.511821,  0.068619,  0.856348,  0.692001, -0.337101,  0.638356,  0.332596,  0.184581,  0.924830, -0.902583, -0.429880, -0.023405,  0.430150,  0.674128,  0.600435,  0.911250, -0.346452,  0.222698,  0.734841, -0.655561,  0.173923, -0.666331, -0.745451,  0.017518,  0.969466,  0.039194,  0.242072,  0.968828,  0.237355, -0.070953,  0.775680,  0.020433,  0.630796,  0.902520,  0.238413,  0.358631,  0.482248,  0.495402,  0.722505, -0.366914, -0.605983,  0.705804,  0.627398,  0.546949,  0.554273,  0.868393,  0.429898,  0.247148, -0.099642,  0.912853,  0.395944, -0.256923, -0.133544,  0.957161, -0.723175,  0.241863,  0.646931, -0.021535, -0.476123,  0.879115,  0.856543, -0.510040,  0.078698, -0.935250,  0.126238,  0.330713,  0.322024, -0.880721, -0.347319,  0.314292,  0.405621,  0.858308, -0.612267, -0.114615,  0.782299,  0.658099,  0.149203,  0.738001,  0.165153, -0.584858,  0.794145, -0.666510,  0.126582, -0.734671, -0.836658,  0.315559,  0.447689, -0.570221,  0.816458,  0.090800,  0.339791,  0.939513, -0.043094,  0.108909,  0.499296,  0.859559, -0.422333, -0.420297,  0.803110, -0.841032, -0.327732,  0.430414,  0.057247,  0.662718,  0.746677,  0.130858,  0.980245, -0.148309,  0.132188,  0.067431,  0.988928,  0.287178,  0.927619,  0.238855,  0.823886, -0.350260,  0.445566,  0.894195,  0.032928,  0.446466,  0.001660,  0.822444,  0.568843, -0.137117,  0.706010,  0.694801,  0.500521,  0.294306,  0.814164, -0.063427, -0.045677,  0.996941, -0.338160,  0.641547,  0.688524,  0.319985, -0.812516,  0.487265,  0.468586,  0.835813,  0.286083, -0.843062,  0.517437,  0.146646,  0.500055, -0.136922,  0.855101, -0.774593, -0.531975,  0.342066, -0.623742,  0.634824,  0.456009,  0.793025,  0.598413,  0.114073, -0.161110,  0.539965,  0.826124, -0.933493, -0.212680,  0.288718,  0.320293, -0.037545,  0.946574,  0.154449, -0.762380,  0.628428,  0.645565,  0.736627,  0.201558, -0.691525,  0.677604,  0.250291,  0.137931, -0.161736,  0.977147, -0.482576, -0.802196,  0.351570, -0.021404, -0.674317,  0.738131,  0.658655,  0.360091,  0.660687,  0.784395,  0.415413,  0.460604,  0.527386, -0.522494,  0.669973, -0.448048,  0.000345,  0.894009,  0.176783,  0.981527,  0.073165, -0.281873, -0.599450, -0.749138,  0.128371, -0.895784,  0.425549};
static float ps142 [426] = { -0.654083,  0.519868,  0.549466,  0.033724,  0.410635,  0.911176, -0.379102, -0.277866,  0.882651, -0.879756, -0.370819,  0.297527,  0.368395, -0.091178,  0.925187,  0.588742,  0.188570,  0.786018,  0.762464, -0.524738,  0.378549, -0.902048,  0.293330,  0.316650,  0.053115, -0.998588, -0.001185, -0.732851,  0.100294,  0.672956, -0.078811, -0.820842,  0.565692, -0.789032,  0.509202,  0.343718, -0.565392, -0.196803,  0.801000,  0.739504, -0.343548,  0.578885, -0.750202,  0.636850,  0.177816,  0.238391,  0.797257,  0.554573,  0.160947, -0.537035,  0.828064, -0.759668, -0.455276,  0.464358, -0.377977,  0.167582,  0.910521,  0.407605,  0.132748,  0.903458,  0.061852, -0.378322,  0.923605,  0.282007, -0.959380, -0.007903, -0.495067,  0.868789,  0.010689,  0.868609,  0.063293,  0.491439, -0.638694,  0.666445,  0.384606,  0.058019, -0.928754, -0.366128, -0.884388,  0.439730,  0.156511,  0.678756, -0.734364,  0.000400, -0.583023,  0.786670,  0.203064,  0.555313, -0.028551,  0.831151, -0.163179,  0.504163,  0.848052,  0.141776, -0.816119,  0.560223,  0.989641,  0.131114,  0.058479,  0.156120, -0.969140,  0.190773,  0.989622,  0.034964, -0.139373,  0.828432, -0.559752,  0.019431, -0.390676,  0.897287,  0.205544,  0.723425,  0.025713,  0.689924,  0.382210,  0.923295,  0.037969,  0.942011,  0.206762,  0.264320,  0.564978, -0.409228,  0.716472, -0.093042,  0.833272,  0.544979, -0.680219,  0.308666,  0.664851,  0.908370, -0.125395,  0.398924,  0.579763, -0.793248,  0.186098,  0.175371, -0.965849, -0.190736, -0.235876, -0.124681,  0.963752, -0.597183, -0.532006,  0.600285,  0.970588, -0.196004, -0.139792, -0.489961,  0.652711,  0.577846, -0.177072, -0.357735,  0.916881,  0.870622, -0.449231,  0.200522,  0.785096,  0.566875, -0.249554,  0.957764,  0.260689, -0.121368,  0.932933,  0.157353, -0.323846, -0.197087, -0.902352,  0.383298, -0.680701, -0.332558,  0.652726, -0.195527,  0.086795,  0.976850,  0.139407,  0.900849,  0.411140,  0.241414, -0.894705,  0.375796, -0.541339,  0.242576,  0.805052, -0.000000, -0.000000,  1.000000, -0.375692, -0.840970, -0.389390, -0.061738, -0.979299, -0.192775,  0.208822,  0.064346,  0.975835,  0.600557,  0.576282,  0.554284,  0.378127, -0.491937,  0.784231,  0.361937, -0.767175,  0.529569,  0.018829,  0.209701,  0.977584,  0.948883, -0.234428,  0.211338, -0.588060,  0.030035,  0.808259,  0.682455,  0.708062,  0.181393,  0.967951,  0.004198,  0.251106,  0.061188,  0.756648,  0.650953,  0.417017,  0.838913, -0.349745,  0.271660,  0.935609,  0.225470, -0.277249, -0.942915,  0.184509,  0.473773,  0.867903, -0.149274,  0.731090,  0.583975,  0.352818,  0.744926,  0.226429,  0.627547, -0.269397,  0.879549,  0.392198, -0.302259,  0.758285,  0.577619,  0.033826, -0.695602,  0.717630, -0.173904, -0.984741, -0.006507,  0.606434,  0.391417,  0.692120,  0.253764,  0.647766,  0.718334, -0.857395, -0.047929,  0.512422,  0.942489, -0.066736, -0.327507, -0.499723, -0.415255,  0.760158,  0.651476, -0.197098,  0.732620, -0.341686,  0.586328,  0.734486, -0.299540, -0.498043,  0.813774,  0.596282,  0.802724,  0.009029, -0.747058, -0.662531,  0.054375, -0.623828, -0.746593,  0.231165,  0.734599, -0.646343,  0.206410,  0.429425,  0.343620,  0.835176, -0.600553,  0.687271, -0.408650,  0.473395, -0.620315,  0.625385, -0.230288, -0.278443, -0.932436, -0.522294,  0.460722,  0.717596,  0.451411, -0.819075,  0.354039,  0.258889, -0.670835,  0.694951, -0.420202, -0.046080,  0.906260, -0.143526,  0.682977,  0.716200, -0.826908, -0.256884,  0.500233, -0.463384,  0.790070,  0.401329, -0.063504, -0.978443,  0.196509,  0.857011,  0.261804,  0.443835, -0.072020, -0.540243,  0.838422, -0.405273, -0.619066,  0.672689,  0.023847, -0.920644,  0.389675, -0.790199,  0.362493,  0.494150,  0.854943, -0.331942,  0.398607, -0.178865,  0.302663,  0.936164, -0.493499, -0.714784,  0.495523, -0.858497, -0.422267, -0.290986,  0.441063,  0.533185,  0.721926, -0.848206,  0.162956,  0.503975,  0.172921, -0.162333,  0.971466, -0.373985,  0.910430, -0.176786, -0.041269, -0.202715,  0.978368,  0.278967, -0.336728,  0.899328,  0.994189, -0.095211,  0.050230,  0.932742,  0.349002,  0.090504,  0.430438,  0.701070,  0.568528, -0.727492, -0.119013,  0.675716,  0.568414,  0.727188,  0.384840,  0.823905,  0.553602,  0.121265,  0.802485, -0.138478,  0.580380, -0.940818,  0.338240, -0.021359,  0.047200,  0.594907,  0.802408, -0.361932,  0.386654,  0.848236,  0.470657, -0.260542,  0.842971,  0.497034,  0.843028,  0.205575,  0.744888,  0.428537,  0.511369,  0.879103,  0.469471, -0.082315,  0.643177, -0.531312,  0.551390, -0.656501, -0.636841,  0.404277, -0.190888, -0.677228,  0.710580, -0.293729, -0.785150,  0.545218,  0.244501,  0.474823,  0.845436};
static float ps143 [429] = {  0.089966, -0.562737,  0.821726, -0.264487,  0.690007,  0.673748, -0.158967,  0.160674,  0.974122,  0.158457,  0.405658,  0.900185,  0.337815, -0.299853,  0.892171, -0.493448, -0.526557,  0.692276,  0.296687,  0.538242,  0.788843,  0.803188, -0.261253,  0.535385, -0.975115,  0.219921,  0.028018, -0.469507,  0.280857,  0.837068,  0.624056,  0.542636,  0.562227, -0.587452, -0.167005,  0.791839, -0.787331, -0.508623,  0.348444, -0.635774, -0.684775,  0.356194,  0.209275,  0.912185,  0.352311, -0.130606,  0.823294,  0.552385,  0.277031, -0.673103,  0.685701,  0.943407,  0.067827,  0.324627,  0.094382,  0.833359,  0.544615,  0.705499,  0.598256,  0.379949, -0.998201, -0.009361,  0.059214,  0.312907,  0.939665,  0.138273, -0.543161,  0.057649,  0.837647,  0.043336,  0.210209,  0.976696, -0.300878, -0.488920,  0.818798, -0.263654, -0.286839,  0.920983,  0.277342, -0.960302, -0.030036,  0.855229, -0.024408,  0.517676,  0.531294, -0.188947,  0.825848, -0.576105, -0.800595,  0.164776,  0.061156,  0.570615,  0.818938, -0.610805,  0.785580, -0.098900, -0.351798,  0.907663,  0.228879, -0.093350, -0.434185,  0.895974, -0.634033, -0.367831,  0.680222, -0.964476,  0.140928,  0.223441,  0.730715, -0.092204,  0.676427,  0.376613, -0.083409,  0.922608,  0.599325, -0.651056,  0.465764, -0.456352, -0.825809,  0.331334,  0.491218, -0.868598, -0.065125,  0.413680, -0.899682,  0.139430, -0.959952, -0.093432,  0.264128, -0.495619, -0.691821,  0.525115, -0.654970, -0.538841,  0.529778,  0.492553, -0.409067,  0.768150,  0.754119, -0.507621,  0.416683, -0.141648, -0.906519,  0.397693, -0.357552,  0.109315,  0.927473,  0.739582,  0.127132,  0.660951, -0.411409, -0.118175,  0.903758,  0.654333, -0.468390,  0.593683, -0.510271, -0.855263, -0.090273, -0.749193,  0.338889,  0.569091, -0.770175,  0.636470, -0.041667, -0.982403, -0.149632, -0.111779,  0.124624, -0.378353,  0.917234, -0.808083,  0.120799,  0.576549,  0.849568,  0.198834,  0.488569, -0.359153,  0.793337,  0.491554, -0.316218, -0.814557,  0.486316, -0.064441,  0.392328,  0.917565, -0.541129,  0.698109,  0.468852,  0.523474, -0.794890,  0.306799, -0.155255, -0.768293,  0.620985, -0.750054, -0.637402,  0.176457,  0.195614,  0.701338,  0.685464, -0.930548, -0.360481, -0.064300, -0.052547, -0.977057,  0.206396, -0.166249,  0.554338,  0.815519,  0.103357, -0.930690,  0.350903, -0.875029,  0.269442,  0.402150, -0.966567, -0.234101,  0.104615, -0.689874,  0.649518,  0.319688, -0.662030,  0.739550,  0.121584,  0.519621,  0.459322,  0.720429, -0.570635,  0.426231,  0.701928, -0.747142, -0.187343,  0.637716,  0.986311, -0.071331,  0.148669,  0.012485, -0.850867,  0.525232, -0.331118, -0.661059,  0.673322,  0.700264, -0.664601,  0.260645,  0.418782, -0.753100,  0.507406, -0.660175,  0.531309,  0.530923, -0.594755, -0.760404, -0.260868,  0.396460,  0.351493,  0.848099, -0.812040,  0.560915,  0.161137, -0.785300, -0.357898,  0.505187, -0.801040,  0.469113,  0.371844,  0.475067, -0.590045,  0.652808, -0.518066,  0.804843,  0.289544,  0.257054,  0.224116,  0.940051,  0.064344, -0.721627,  0.689286,  0.703561,  0.342334,  0.622743, -0.035408,  0.712102,  0.701183, -0.835777, -0.549061, -0.002780, -0.218031,  0.899578,  0.378446,  0.417935,  0.642926,  0.641854, -0.216299, -0.066826,  0.974037, -0.864923, -0.129826,  0.484824,  0.519357,  0.717198,  0.464646, -0.457077, -0.338103,  0.822659,  0.923928, -0.145010,  0.354019,  0.192538, -0.967486,  0.164011,  0.313334,  0.797060,  0.516253, -0.281968, -0.927006,  0.247293,  0.098371,  0.980726,  0.168820, -0.268672,  0.335665,  0.902854, -0.865695,  0.348125, -0.359697, -0.708989,  0.003177,  0.705212, -0.054062, -0.230996,  0.971452, -0.898705,  0.438550, -0.001571, -0.009875, -0.013440,  0.999861, -0.910727,  0.360066,  0.202307,  0.221249, -0.819510,  0.528632,  0.917389,  0.275999,  0.286744, -0.459806,  0.612937,  0.642563,  0.798171,  0.407995,  0.443241,  0.201393,  0.029833,  0.979056, -0.652327,  0.220488,  0.725158, -0.672759, -0.739837, -0.006106, -0.897309, -0.292915,  0.330207, -0.909869,  0.055356,  0.411186, -0.124788,  0.974241,  0.187836,  0.162465, -0.177647,  0.970591, -0.124963, -0.616758,  0.777170, -0.369279,  0.488879,  0.790336, -0.002669,  0.928086,  0.372355,  0.037862, -0.999234,  0.009887,  0.406116,  0.910114, -0.082226, -0.850228, -0.470288, -0.236522,  0.192664,  0.980463, -0.039660,  0.322359, -0.883833,  0.339004,  0.746830,  0.642443,  0.171791,  0.411262,  0.854856,  0.316362,  0.300360, -0.501894,  0.811102,  0.660945, -0.289516,  0.692338,  0.941738, -0.291257,  0.168223,  0.582622,  0.243265,  0.775483,  0.435970,  0.120009,  0.891924, -0.886210, -0.432370,  0.166394,  0.840684, -0.503016,  0.200564,  0.595542,  0.015152,  0.803181};
static float ps144 [432] = { -0.467369,  0.457170,  0.756678, -0.569659, -0.492344,  0.658093,  0.152156, -0.067096,  0.986076,  0.688051, -0.564129,  0.456447, -0.982174, -0.139234,  0.126286, -0.259035, -0.959295, -0.112493,  0.424431, -0.870432,  0.249412,  0.178838, -0.983749,  0.015975, -0.638325,  0.323059,  0.698694,  0.119636,  0.571446,  0.811873, -0.739109, -0.582195,  0.338772,  0.495905,  0.661637,  0.562418, -0.445090,  0.632058,  0.634347, -0.867854, -0.013075,  0.496648, -0.854398, -0.394446,  0.338254, -0.806210,  0.567833,  0.166104, -0.459197,  0.047070,  0.887087, -0.257242,  0.892231,  0.371147, -0.929414, -0.342732,  0.136836, -0.601580, -0.798638, -0.016687, -0.991552,  0.072122,  0.107813, -0.310184, -0.828305,  0.466579, -0.914151,  0.397712,  0.078442, -0.229673, -0.716861,  0.658301,  0.010232, -0.984512,  0.175017, -0.433171, -0.691955,  0.577548, -0.079167,  0.486108,  0.870305,  0.522155, -0.634693,  0.569666, -0.631334,  0.109547,  0.767734,  0.764756,  0.293386,  0.573650, -0.150333, -0.931034,  0.332529,  0.298550,  0.644503,  0.703906,  0.605724, -0.722148,  0.334066, -0.747792, -0.060972,  0.661127, -0.306436, -0.893005, -0.329605,  0.608921,  0.359843,  0.706914, -0.646973, -0.732226, -0.212769, -0.473216,  0.260106,  0.841672, -0.824732, -0.229408,  0.516903, -0.622291,  0.516437,  0.588257,  0.463878,  0.514313,  0.721318,  0.316280, -0.727205,  0.609212, -0.098179, -0.838519,  0.535953, -0.768815,  0.372618,  0.519691,  0.111846,  0.369405,  0.922513, -0.388925,  0.920068, -0.047031, -0.595692, -0.115175,  0.794912, -0.888420, -0.455382, -0.057770, -0.075540,  0.821825,  0.564710,  0.882816, -0.262463,  0.389549,  0.295858,  0.225463,  0.928243, -0.309670, -0.404625,  0.860455, -0.741570,  0.638294, -0.206532, -0.773107,  0.163156,  0.612933,  0.104586,  0.961311,  0.254838, -0.266666,  0.433371, -0.860859,  0.879917, -0.206030, -0.428132,  0.511489,  0.084822,  0.855093, -0.017252, -0.697334,  0.716538, -0.122030,  0.974309,  0.189289, -0.880940, -0.205895, -0.426088, -0.281749,  0.187622,  0.940965,  0.507127, -0.134744,  0.851273, -0.052540,  0.920384, -0.387469, -0.652070, -0.506827, -0.563854, -0.333814,  0.928829,  0.160765,  0.598292, -0.457712,  0.657683, -0.402663, -0.914678,  0.035034, -0.378605, -0.560192,  0.736779,  0.807263,  0.064346,  0.586673,  0.462745,  0.868735,  0.176540, -0.730454,  0.561451,  0.388857, -0.065481,  0.667211,  0.741985, -0.974118,  0.223362, -0.034706,  0.967130,  0.244753,  0.068957,  0.924180,  0.191499, -0.330482,  0.043180,  0.998234,  0.040807, -0.762401, -0.646314, -0.031983, -0.529458, -0.828146,  0.183979, -0.681052, -0.282910,  0.675374, -0.737804, -0.430877,  0.519606, -0.223715, -0.230960,  0.946894,  0.904587,  0.344561,  0.250998,  0.137055,  0.871298,  0.471228, -0.475542,  0.829865,  0.291863,  0.322019,  0.783411,  0.531573,  0.112038, -0.803526,  0.584630, -0.797718, -0.557202, -0.230593, -0.074120,  0.067432,  0.994967, -0.084461,  0.283540,  0.955234,  0.222260, -0.949133,  0.223041, -0.587160,  0.799690, -0.125454, -0.081016, -0.382338,  0.920464, -0.958999, -0.113263, -0.259792,  0.664197,  0.633218,  0.397338, -0.833541, -0.533159,  0.144743,  0.676079, -0.053202,  0.734906,  0.337718,  0.010824,  0.941185, -0.996604, -0.019999, -0.079880, -0.845717,  0.529402, -0.067060,  0.201800,  0.972967, -0.112306, -0.119229,  0.298508, -0.946931,  0.251952, -0.869185,  0.425485, -0.349247, -0.898402,  0.266272,  0.114648,  0.148941,  0.982177, -0.916313,  0.357055, -0.181333,  0.759958, -0.371518,  0.533327, -0.906564,  0.033647, -0.420726, -0.571669,  0.688445,  0.446360,  0.441165, -0.781372,  0.441398, -0.245292,  0.737685,  0.629009,  0.056658, -0.920227, -0.387262,  0.468014, -0.348254,  0.812208, -0.613524, -0.617396,  0.492352,  0.060332, -0.517743,  0.853406, -0.505506, -0.343571,  0.791468, -0.161279, -0.563832,  0.809989,  0.689690,  0.701426, -0.179802,  0.820017, -0.471730,  0.324102,  0.297678,  0.446788,  0.843664, -0.282524,  0.389202,  0.876756,  0.494729,  0.782495,  0.378080,  0.962988, -0.130972,  0.235585, -0.856304,  0.409953,  0.314137, -0.262650, -0.022587,  0.964627,  0.401427, -0.547815,  0.734000, -0.419115, -0.175559,  0.890798,  0.651692, -0.261181,  0.712097,  0.194229, -0.634373,  0.748228, -0.522523, -0.764420,  0.377666, -0.382809,  0.785828,  0.485728,  0.665643,  0.157900,  0.729375,  0.798782,  0.437871,  0.412573,  0.542926, -0.835457, -0.085105, -0.037913, -0.146923,  0.988421,  0.318557, -0.215874,  0.922995,  0.802163, -0.159361,  0.575447, -0.950551,  0.023166,  0.309704, -0.715919,  0.697995,  0.016222,  0.117874,  0.748080,  0.653055, -0.658348,  0.714720,  0.236121,  0.463156,  0.294999,  0.835741, -0.944688,  0.241865,  0.221508, -0.269773,  0.574368,  0.772867};
static float ps145 [435] = {  0.617253,  0.466325, -0.633671, -0.768211,  0.611547,  0.189375,  0.846862,  0.517204,  0.123790, -0.507863, -0.663926,  0.548888,  0.798558,  0.379506,  0.467205,  0.226831, -0.331363,  0.915831, -0.524935, -0.335280,  0.782324,  0.959816, -0.275098,  0.055443,  0.375658, -0.011654,  0.926685, -0.309610, -0.750733,  0.583560,  0.658867,  0.413900,  0.628157, -0.422115,  0.828494,  0.367990,  0.850779, -0.223011,  0.475859, -0.102744, -0.987144, -0.122435,  0.154796, -0.728035,  0.667834,  0.238874,  0.816736,  0.525244, -0.887896,  0.240678,  0.392066, -0.688549, -0.565311,  0.454228,  0.447376,  0.714862,  0.537426,  0.934192, -0.040915,  0.354416,  0.004480, -0.948544,  0.316615, -0.473180,  0.865165,  0.166103,  0.567428, -0.111034,  0.815903,  0.708634,  0.678123,  0.194904,  0.840073, -0.014802,  0.542271, -0.715877,  0.566192,  0.408590, -0.198538, -0.098159,  0.975165, -0.564165, -0.739950,  0.366322,  0.791465,  0.197229,  0.578518, -0.600220, -0.781920,  0.168338,  0.005726, -0.627124,  0.778898,  0.894362,  0.356832,  0.269792, -0.176439, -0.302423,  0.936702, -0.381813, -0.206257,  0.900931, -0.205238,  0.878441,  0.431531,  0.405018, -0.227689,  0.885505,  0.487157,  0.549603,  0.678686, -0.206148,  0.108604,  0.972475,  0.325296,  0.469604,  0.820765, -0.815163, -0.137712,  0.562622, -0.377357, -0.847728, -0.372771,  0.071696, -0.857384,  0.509659, -0.050416,  0.956824,  0.286263, -0.736470, -0.627782,  0.251995,  0.352811, -0.632160,  0.689853,  0.353361, -0.754819, -0.552616,  0.248528,  0.967257, -0.051450, -0.848684,  0.438635,  0.295525,  0.040701, -0.421762,  0.905792, -0.834487, -0.446854,  0.322416, -0.759341, -0.649731,  0.035356, -0.555046,  0.344921,  0.756937, -0.637449,  0.478790,  0.603671,  0.929108, -0.251551,  0.271073,  0.495134,  0.356110,  0.792482, -0.705249,  0.235754,  0.668613,  0.015147, -0.216278,  0.976214,  0.002787,  0.428992,  0.903304, -0.433641, -0.556102,  0.709018, -0.572295,  0.120292,  0.811177, -0.960895,  0.107249,  0.255300, -0.185579,  0.501823,  0.844828, -0.080530, -0.766604,  0.637050,  0.769605,  0.541279,  0.338710, -0.562631, -0.108124,  0.819607, -0.941557, -0.332326, -0.055036, -0.274094,  0.644256,  0.714008, -0.776702,  0.628120, -0.046900, -0.585822, -0.722778, -0.366613, -0.413923, -0.885017,  0.213103,  0.296971,  0.661776,  0.688375, -0.938401, -0.303127,  0.165886, -0.778899,  0.366540,  0.508886,  0.201557, -0.118609,  0.972269, -0.502473,  0.863196, -0.049129,  0.727848, -0.182970,  0.660877, -0.234553, -0.636850,  0.734444, -0.712882, -0.003151,  0.701276,  0.899615,  0.173279,  0.400832, -0.342210, -0.420110,  0.840476,  0.395979, -0.439721,  0.806131,  0.363157, -0.912368,  0.188949, -0.399938,  0.011160,  0.916474,  0.637414,  0.579766,  0.507518, -0.650342,  0.755847,  0.075829, -0.548895,  0.673852,  0.494608,  0.772719,  0.363192, -0.520574,  0.995030, -0.084929, -0.051989, -0.909306, -0.021617,  0.415567,  0.678347, -0.555944,  0.480387, -0.631090,  0.759083, -0.159744, -0.399603,  0.226801,  0.888188, -0.380300,  0.435842,  0.815729,  0.866932,  0.487351, -0.104487, -0.892966, -0.243681,  0.378459, -0.684895, -0.241736,  0.687373, -0.050545,  0.641103,  0.765788,  0.093614,  0.747580,  0.657541,  0.547659, -0.515289,  0.659202,  0.440786, -0.826251,  0.350737, -0.310604,  0.950219,  0.024690,  0.967185,  0.151622,  0.203870,  0.018792,  0.875156,  0.483475,  0.607135, -0.710925,  0.354928, -0.105492,  0.991783,  0.072374, -0.370433, -0.834525,  0.407857, -0.145448, -0.497737,  0.855046,  0.159344, -0.975934,  0.148867,  0.704242,  0.022467,  0.709605,  0.880679, -0.445848,  0.160076, -0.201479,  0.310606,  0.928940,  0.762179, -0.591829,  0.262339, -0.002338,  0.217476,  0.976063,  0.505621,  0.837404,  0.207610, -0.468107,  0.573702,  0.672118,  0.166593,  0.927323,  0.335140, -0.612477,  0.733949,  0.293582, -0.941390,  0.296878,  0.160151,  0.191172,  0.108639,  0.975526,  0.363616,  0.225991,  0.903721,  0.219099, -0.908056,  0.356973,  0.142080,  0.566158,  0.811960,  0.202647, -0.542343,  0.815351,  0.490108, -0.687857,  0.535394,  0.647677,  0.230211,  0.726304,  0.574503, -0.320524,  0.753134,  0.986058, -0.063392,  0.153857,  0.046044,  0.993732, -0.101862,  0.441999,  0.897003, -0.004676,  0.528341,  0.104359,  0.842594,  0.991714,  0.128405, -0.003885,  0.187272,  0.335743,  0.923150, -0.824907,  0.111893,  0.554084, -0.152962, -0.872852,  0.463392,  0.300176, -0.796312,  0.525150, -0.130878,  0.781012,  0.610648, -0.000000, -0.000000,  1.000000, -0.266786,  0.934854,  0.234251,  0.825168, -0.417341,  0.380689, -0.971189, -0.102636,  0.215078,  0.879449, -0.472267, -0.059441,  0.718851, -0.378293,  0.583221, -0.208629, -0.941223,  0.265656, -0.617811, -0.785382, -0.038531,  0.308252,  0.936005,  0.169929};
static float ps146 [438] = { -0.519152,  0.284120,  0.806075, -0.480134,  0.873235,  0.083264, -0.016287, -0.617812,  0.786157, -0.328461, -0.452560,  0.829037,  0.504740, -0.720432,  0.475620,  0.037913, -0.888072, -0.458138,  0.353677,  0.844974,  0.401164, -0.705503, -0.574626,  0.414814,  0.295690, -0.936546,  0.188280, -0.877781,  0.467818, -0.103187,  0.204109, -0.226657,  0.952348, -0.780033,  0.624665,  0.036641, -0.756142, -0.620985,  0.206462, -0.858011, -0.495855, -0.133959, -0.592501, -0.741103,  0.315768, -0.779267, -0.376571,  0.500937,  0.851080, -0.037754,  0.523678, -0.253120,  0.573423,  0.779177,  0.118771,  0.529459,  0.839980,  0.691866, -0.600865,  0.400353, -0.384534, -0.263125,  0.884816,  0.497751,  0.409033,  0.764811,  0.120253, -0.741083,  0.660557,  0.221053, -0.594028,  0.773477,  0.173436, -0.022443,  0.984589,  0.582266, -0.562075,  0.587400, -0.026908, -0.083642,  0.996132,  0.994077,  0.107837, -0.013520, -0.276196,  0.934736,  0.223574,  0.260668, -0.817877,  0.512961,  0.523197,  0.576063,  0.628025, -0.381143,  0.659714,  0.647694, -0.628580, -0.770438,  0.106359,  0.009939,  0.133584,  0.990988, -0.972725, -0.115638, -0.201084,  0.149197,  0.930533,  0.334438,  0.324524,  0.925687,  0.194390, -0.320210,  0.380342,  0.867644,  0.286260,  0.394208,  0.873301,  0.695832,  0.661230,  0.280344,  0.958181, -0.285501,  0.019458, -0.558936, -0.156855,  0.814240, -0.468319,  0.481613,  0.740761, -0.052831, -0.948945,  0.310987, -0.816963,  0.086280,  0.570200, -0.685304,  0.190468,  0.702908,  0.220601,  0.203248,  0.953953, -0.225708, -0.619712,  0.751674, -0.703531, -0.037771,  0.709660, -0.161779,  0.253721,  0.953653, -0.096649,  0.984476, -0.146510,  0.481063,  0.875787,  0.039696,  0.292588, -0.956059, -0.018505, -0.567404,  0.595051,  0.569181, -0.612055,  0.693632,  0.379821,  0.626433, -0.375862,  0.682869, -0.365862,  0.169021,  0.915192,  0.399736, -0.153112,  0.903752,  0.627471,  0.213009,  0.748937, -0.906718, -0.031953,  0.420525, -0.460048,  0.839838,  0.288146,  0.085560, -0.994274, -0.064022, -0.001284,  0.778068,  0.628179,  0.549687,  0.029874,  0.834836,  0.597405, -0.753241,  0.275201, -0.065428,  0.961333,  0.267505, -0.949756, -0.307407, -0.058867,  0.061978,  0.342469,  0.937483, -0.259938, -0.932289,  0.251536,  0.433403, -0.518657,  0.736992, -0.799612,  0.294553,  0.523315, -0.206670,  0.764424,  0.610691, -0.402943, -0.825843,  0.394486,  0.481407, -0.866721,  0.130541,  0.470394, -0.326543,  0.819817, -0.896882, -0.244251,  0.368706,  0.340149,  0.563574,  0.752784,  0.002432, -0.285167,  0.958475,  0.367221,  0.721920,  0.586499,  0.645086, -0.762545,  0.048875,  0.756599, -0.211669,  0.618671, -0.439151, -0.581095,  0.685182,  0.511172,  0.818890,  0.261001, -0.054807,  0.638331,  0.767808,  0.400083, -0.846921,  0.350225,  0.439391,  0.235228,  0.866950,  0.541415,  0.713410,  0.444876, -0.137686, -0.435607,  0.889544,  0.283893, -0.401682,  0.870664, -0.773605, -0.633520, -0.013716, -0.900789, -0.140172, -0.411012,  0.670423,  0.388438,  0.632178, -0.726130,  0.506978,  0.464443,  0.181047,  0.831238,  0.525609,  0.750432, -0.418144,  0.511867,  0.835241, -0.450400,  0.315456,  0.362587,  0.046128,  0.930808,  0.989940, -0.092722, -0.106871,  0.083364, -0.461319,  0.883309, -0.203084, -0.211986,  0.955939, -0.104328, -0.990274,  0.092054, -0.107904, -0.761721,  0.638857, -0.845712, -0.442138,  0.298807,  0.679245,  0.549527,  0.486463, -0.424967,  0.768263,  0.478722, -0.948133, -0.273850,  0.161403, -0.943404,  0.267275,  0.196346,  0.942731, -0.067772,  0.326597,  0.756685, -0.625759,  0.189350,  0.811316,  0.341937,  0.474179, -0.195982, -0.866828,  0.458476,  0.605430, -0.167314,  0.778113, -0.299071, -0.953816,  0.028143, -0.880141, -0.466615,  0.087307, -0.639923,  0.749080,  0.171398,  0.866214, -0.248533,  0.433480, -0.453402, -0.872267,  0.183238, -0.200772,  0.032437,  0.979101, -0.549335,  0.066090,  0.832985,  0.782247,  0.175453,  0.597751,  0.362564, -0.685212,  0.631690,  0.933654, -0.273715,  0.231021,  0.989820, -0.086202,  0.113253,  0.880914, -0.458007, -0.119251, -0.104769,  0.455484,  0.884058,  0.718578,  0.000135,  0.695446, -0.627070, -0.494438,  0.601925,  0.177491,  0.688147,  0.703527, -0.683723, -0.275878,  0.675584, -0.247085,  0.871974,  0.422623,  0.805163,  0.494315,  0.327665, -0.524725, -0.383748,  0.759869, -0.812064, -0.156943,  0.562068, -0.123819, -0.984275, -0.125984,  0.911000,  0.310555,  0.271356, -0.384113, -0.052957,  0.921766, -0.970480, -0.063658,  0.232629, -0.853649,  0.395149,  0.339323,  0.024770, -0.862535,  0.505391, -0.650000,  0.398188,  0.647261, -0.325134, -0.741775,  0.586565, -0.645052, -0.756198, -0.109874, -0.528016, -0.677825,  0.511618,  0.163950, -0.920417,  0.354899, -0.765757,  0.588473,  0.259455, -0.914713,  0.165370,  0.368718};
static float ps147 [441] = { -0.546938, -0.188325,  0.815716, -0.454043, -0.686262,  0.568233, -0.241661, -0.524129,  0.816633, -0.467251,  0.398443,  0.789253, -0.938217, -0.329360,  0.106169, -0.717282, -0.202919,  0.666581,  0.810314,  0.189673,  0.554450,  0.614279,  0.675070, -0.408584, -0.318963,  0.562487,  0.762805,  0.996431, -0.061365,  0.057957,  0.175633, -0.424555,  0.888204,  0.302431, -0.014536,  0.953061,  0.161948,  0.881847,  0.442853,  0.251770, -0.230307,  0.939984,  0.361431,  0.803885,  0.472374, -0.796962,  0.349835,  0.492409, -0.266223, -0.687582,  0.675542,  0.220425,  0.750675,  0.622816, -0.809810,  0.579042,  0.094438,  0.955814, -0.128870,  0.264221, -0.695060,  0.526696,  0.489370,  0.399981, -0.362799,  0.841661,  0.175626, -0.981124,  0.080936,  0.724946,  0.577145,  0.375974,  0.117688,  0.502271,  0.856664, -0.282355,  0.023094,  0.959032, -0.745206,  0.656281, -0.118166,  0.298217, -0.826905,  0.476754,  0.008317,  0.810664,  0.585453,  0.438090,  0.656505,  0.614066, -0.425165,  0.873515,  0.237078, -0.734345,  0.611684,  0.294247,  0.602394,  0.798185,  0.004679, -0.028684,  0.338277,  0.940609, -0.647994,  0.381452,  0.659240,  0.478216,  0.826653,  0.296570, -0.365026,  0.701500,  0.612090, -0.698841, -0.681912,  0.215909, -0.872811, -0.359485,  0.330109, -0.909910,  0.401409,  0.104568, -0.197912,  0.826616,  0.526817,  0.456414,  0.803170, -0.382890,  0.796298, -0.018973,  0.604606,  0.442721,  0.890739,  0.102874, -0.807543, -0.024319,  0.589307, -0.577865,  0.203719,  0.790298,  0.326254, -0.694854,  0.640887,  0.754608,  0.655921, -0.018285, -0.924855, -0.005604,  0.380278,  0.344633,  0.407239,  0.845804, -0.950505, -0.282768, -0.128773,  0.058341,  0.667923,  0.741940,  0.677248,  0.127780,  0.724574,  0.493446, -0.707918,  0.505335, -0.439710, -0.537405,  0.719619, -0.196017, -0.337966,  0.920520, -0.522500,  0.556270,  0.646187,  0.782836,  0.596391,  0.177442, -0.540607, -0.818760,  0.193328, -0.861263,  0.156877,  0.483338,  0.870312, -0.481614,  0.102986, -0.387731,  0.912626, -0.129535, -0.132958, -0.137450,  0.981545,  0.325768, -0.537663,  0.777685, -0.846929,  0.435905,  0.304465,  0.746440, -0.225425,  0.626108, -0.924913,  0.232511,  0.300790, -0.735277,  0.182782,  0.652655, -0.950744, -0.177872,  0.253865,  0.573217, -0.295320,  0.764335,  0.871192, -0.184556,  0.454932, -0.972990,  0.207829,  0.100489, -0.510333,  0.858694,  0.046968,  0.630145,  0.530681,  0.566829,  0.023021,  0.999735,  0.000721, -0.482699,  0.011976,  0.875704,  0.087921,  0.970186,  0.225850, -0.235972,  0.916991,  0.321629, -0.039766,  0.919212,  0.391749,  0.352680,  0.191422,  0.915955, -0.592635, -0.374071,  0.713340, -0.859580, -0.195519,  0.472118,  0.785665, -0.540359,  0.301236,  0.658121, -0.575109,  0.485928, -0.077879, -0.810389,  0.580694, -0.306477, -0.906363,  0.290823,  0.286153,  0.592340,  0.753160,  0.583785, -0.799682,  0.140375,  0.510371,  0.058030,  0.857994,  0.020649, -0.291988,  0.956199, -0.113589, -0.916369,  0.383883,  0.906919,  0.027412,  0.420412, -0.755910, -0.371854,  0.538818, -0.758615, -0.526541,  0.383741, -0.154917,  0.696371,  0.700763, -0.881983, -0.469590, -0.039888,  0.786902,  0.391420,  0.477049,  0.134531,  0.126923,  0.982747,  0.643009, -0.086025,  0.761012,  0.792528, -0.391281,  0.467759, -0.097887,  0.532859,  0.840523,  0.033838, -0.970881,  0.237159,  0.635301, -0.702964,  0.319742, -0.404082, -0.364156,  0.839112, -0.659350, -0.004509,  0.751822,  0.525580,  0.273643,  0.805534, -0.618864, -0.534799,  0.575324,  0.130865, -0.616922,  0.776068, -0.305701,  0.948853,  0.078904,  0.954491, -0.281674,  0.098016,  0.454176, -0.148833,  0.878392, -0.057684,  0.069151,  0.995937,  0.654153, -0.420676,  0.628582,  0.864762,  0.422281,  0.271781,  0.903882,  0.233704,  0.358302,  0.893181, -0.344724,  0.288780,  0.983646, -0.021238, -0.178855,  0.090893, -0.890736,  0.445340, -0.592512,  0.758421,  0.271526, -0.384333, -0.918367,  0.094287,  0.498360,  0.478242,  0.723133, -0.561228,  0.681377,  0.469839, -0.398451,  0.807983,  0.434052,  0.174943,  0.316604,  0.932286, -0.113587,  0.980370,  0.161162,  0.125229, -0.769934,  0.625715,  0.973342,  0.086840,  0.212283, -0.068507, -0.666700,  0.742171, -0.234919, -0.970120, -0.060666,  0.502015, -0.545108,  0.671445,  0.673042,  0.336694,  0.658523, -0.188787,  0.217651,  0.957595, -0.838817, -0.515644,  0.174634,  0.245151, -0.925237,  0.289546,  0.453484, -0.830362,  0.323807, -0.350693, -0.174454,  0.920098,  0.287765,  0.918284,  0.271930, -0.675605,  0.733421,  0.075181, -0.164819, -0.974433,  0.152690,  0.640125,  0.739443,  0.208482,  0.275389,  0.818949, -0.503471, -0.394241,  0.215257,  0.893442,  0.565324,  0.698264,  0.439131, -0.258769,  0.399102,  0.879634,  0.093069, -0.092597,  0.991345,  0.990461,  0.135750, -0.023648, -0.034558, -0.489573,  0.871277};
static float ps148 [444] = {  0.239476, -0.697006,  0.675895,  0.902823, -0.413898,  0.116617,  0.424033, -0.440315,  0.791403, -0.053212,  0.635067,  0.770622, -0.794566,  0.296971,  0.529597,  0.570766, -0.116612,  0.812790, -0.412944,  0.140420,  0.899866, -0.579184,  0.598154,  0.553857,  0.573503,  0.471068,  0.670216,  0.859844, -0.229739,  0.455948,  0.599474, -0.503351,  0.622308,  0.790127,  0.517550,  0.328393, -0.384183,  0.693515,  0.609459,  0.043654, -0.668307,  0.742603, -0.897715, -0.246054,  0.365464, -0.576860,  0.042375,  0.815743, -0.974159,  0.225340, -0.015349,  0.781260,  0.181542,  0.597223,  0.057793,  0.764601,  0.641908, -0.550315,  0.796948,  0.249050, -0.868939, -0.073090,  0.489493, -0.275221,  0.594522,  0.755511, -0.660469, -0.261233,  0.703945, -0.665225, -0.733409,  0.139956, -0.720381, -0.063255,  0.690689,  0.195803,  0.047549,  0.979490, -0.904642,  0.144745,  0.400839, -0.147634,  0.937264,  0.315818, -0.689650,  0.719489, -0.081964,  0.936405, -0.061992,  0.345402,  0.634031, -0.714811,  0.295042,  0.392779, -0.032632,  0.919054, -0.202282,  0.022325,  0.979073, -0.708293,  0.692553,  0.136720,  0.466161,  0.344174,  0.815008,  0.281808,  0.918179,  0.278445,  0.682617,  0.686796,  0.249691, -0.349817,  0.385446,  0.853850, -0.990518,  0.046462, -0.129286,  0.347484, -0.779566,  0.521087, -0.377546,  0.925990,  0.000923, -0.533192, -0.164790,  0.829789,  0.824657, -0.565549, -0.009782,  0.429109, -0.611271,  0.664991,  0.174769,  0.617762,  0.766698, -0.360141, -0.295272,  0.884937, -0.877507,  0.349824,  0.328031,  0.064339, -0.962422,  0.263827, -0.960738, -0.137426, -0.241033,  0.544455, -0.680667,  0.490164, -0.811038,  0.533770,  0.239391, -0.213420,  0.893610, -0.394860, -0.299902, -0.693840,  0.654710, -0.070392, -0.993147,  0.093292, -0.170485, -0.405870,  0.897889, -0.157689,  0.467373,  0.869883, -0.020379,  0.993435,  0.112567, -0.563126, -0.761406,  0.321171,  0.745042, -0.375103,  0.551552, -0.963446, -0.208743,  0.167923, -0.624644, -0.619948,  0.474851, -0.665622,  0.654469,  0.358631,  0.109197, -0.816593,  0.566791, -0.579959, -0.812201, -0.063066, -0.913963,  0.390558,  0.110161, -0.379747, -0.550393, -0.743545, -0.166021, -0.942976,  0.288502,  0.273458,  0.417331,  0.866635, -0.984682, -0.170464, -0.036665,  0.922108,  0.337518,  0.189203,  0.411207, -0.245912,  0.877745,  0.187429,  0.977787,  0.093825,  0.276561,  0.732306,  0.622287, -0.515610,  0.842604, -0.155450, -0.440505, -0.747547,  0.497120, -0.045614,  0.312693,  0.948758, -0.374182, -0.873010,  0.312796, -0.524993,  0.293304,  0.798972,  0.225070, -0.360521,  0.905190,  0.940829, -0.243033,  0.236168, -0.280263,  0.843013,  0.459109,  0.356366,  0.190490,  0.914722,  0.822250,  0.560176, -0.100538, -0.487418, -0.586709,  0.646682, -0.377581, -0.070893,  0.923259,  0.787935, -0.577181,  0.214523,  0.734941,  0.676074,  0.052783, -0.746675,  0.486150,  0.454021,  0.018439, -0.297336,  0.954595, -0.227565,  0.965409,  0.127275, -0.891459, -0.131290, -0.433663,  0.845411,  0.518878,  0.126670,  0.859260,  0.331238,  0.389812,  0.832346, -0.039387,  0.552855, -0.102067, -0.773071,  0.626054,  0.033258,  0.895636, -0.443543, -0.471722,  0.507687,  0.720925, -0.389969, -0.918027, -0.071766,  0.664949,  0.565589,  0.487804,  0.739329,  0.383800,  0.553254,  0.068182,  0.477146,  0.876175, -0.784532, -0.443731,  0.433142,  0.069074,  0.950718,  0.302266, -0.012545,  0.113746,  0.993431,  0.163938,  0.864900,  0.474417,  0.877553,  0.414785, -0.240527, -0.476353,  0.762490,  0.437833,  0.030318, -0.491493,  0.870353,  0.152362, -0.985640,  0.072802, -0.245315, -0.841422,  0.481487, -0.550925,  0.832848,  0.053350, -0.234406,  0.230809,  0.944341, -0.053673,  0.870184,  0.489795, -0.149567, -0.601486,  0.784757,  0.712034, -0.556036,  0.428756, -0.173158,  0.755849,  0.631434,  0.696000,  0.027196,  0.717526, -0.804953,  0.092886,  0.586023, -0.996432,  0.011193,  0.083649, -0.487617, -0.863472,  0.129019, -0.185534, -0.195178,  0.963059,  0.441780, -0.827312,  0.346966,  0.845593, -0.411315,  0.340283,  0.370092,  0.814445,  0.446890,  0.486194,  0.837627,  0.248989,  0.306138, -0.929277,  0.206697,  0.536945,  0.110072,  0.836406,  0.590324, -0.323025,  0.739711, -0.336220, -0.509051,  0.792353, -0.646809,  0.406911,  0.645029, -0.371733,  0.892206,  0.256482,  0.159931,  0.262348,  0.951628,  0.006290, -0.092908,  0.995655, -0.657009, -0.452260,  0.603158,  0.926020,  0.376433, -0.028005,  0.558741,  0.722765,  0.406719, -0.961680,  0.189118,  0.198511, -0.792597, -0.259309,  0.551859,  0.475785,  0.653826,  0.588336, -0.956241, -0.030663,  0.290970,  0.214027, -0.162149,  0.963276, -0.678257,  0.193364,  0.708927,  0.637372,  0.263007,  0.724282, -0.740343, -0.603306,  0.296503,  0.234726, -0.540290,  0.808078,  0.728387, -0.188349,  0.658769,  0.286809,  0.951215, -0.113715, -0.508825, -0.395245,  0.764773};
static float ps149 [447] = {  0.585047, -0.203522,  0.785047,  0.025353,  0.238936,  0.970704, -0.474544, -0.857743,  0.197697, -0.848727,  0.444133,  0.287068, -0.737135, -0.467882,  0.487563, -0.154833, -0.702164,  0.694977,  0.783319,  0.172777,  0.597126, -0.410308,  0.584956,  0.699624, -0.729473,  0.514700,  0.450504,  0.550730,  0.003220,  0.834677,  0.220259,  0.972683,  0.073300,  0.714163, -0.586026,  0.382811,  0.056254, -0.971485,  0.230330, -0.214319,  0.960510,  0.177451, -0.714506,  0.699627,  0.001855,  0.967268,  0.149948,  0.204714,  0.897409, -0.419436,  0.136861,  0.788960,  0.548340, -0.277247, -0.542569,  0.644559,  0.538667, -0.890701, -0.437667,  0.122880, -0.952838, -0.242066,  0.183039, -0.001546, -0.886881,  0.461995, -0.902531,  0.065583,  0.425602,  0.279376, -0.940590,  0.192974, -0.015404,  0.732237,  0.680876,  0.000000,  0.000000,  1.000000, -0.867583, -0.360378,  0.342677,  0.225760, -0.315805,  0.921574,  0.382229, -0.835462,  0.394846,  0.559937, -0.726040,  0.399169, -0.274915,  0.486736,  0.829162,  0.471480, -0.857424,  0.206230,  0.898129,  0.158699,  0.410096,  0.738206, -0.413525,  0.532963,  0.355602, -0.934637, -0.001400,  0.787163, -0.589822,  0.180235,  0.839918, -0.420003,  0.343708, -0.158524, -0.321030,  0.933708,  0.992113, -0.096635, -0.079831, -0.647526, -0.727440,  0.227025,  0.030756, -0.770075,  0.637211,  0.820016,  0.346231,  0.455740, -0.323195, -0.873418,  0.364261, -0.542477, -0.602441,  0.585478, -0.471405,  0.795447,  0.380844, -0.137415, -0.941785,  0.306853,  0.310626,  0.454348,  0.834913,  0.202204, -0.113702,  0.972721,  0.367506,  0.031713,  0.929480, -0.149794,  0.828157,  0.540110, -0.495818, -0.769808,  0.401946, -0.738665,  0.353571,  0.573900, -0.649974, -0.636031,  0.415931, -0.862338, -0.498804, -0.086992, -0.859997,  0.286442,  0.422323,  0.020717, -0.221973,  0.974833,  0.800423,  0.524248,  0.290666,  0.934345, -0.235455,  0.267506,  0.841665, -0.029250,  0.539207,  0.629923,  0.190239,  0.752998,  0.091159,  0.432766,  0.896885, -0.176733, -0.101766,  0.978984, -0.496512,  0.157225,  0.853672,  0.136876,  0.948360,  0.286142, -0.425453, -0.904894, -0.012523,  0.260387,  0.864212,  0.430507,  0.501880,  0.381949,  0.776036,  0.643338, -0.738265,  0.202685, -0.278059,  0.883918,  0.375995, -0.127275,  0.370018,  0.920265,  0.236331, -0.666898,  0.706678, -0.545374,  0.838116, -0.011317,  0.854866, -0.231155,  0.464512, -0.519398, -0.274077,  0.809387,  0.711147, -0.015055,  0.702882, -0.350930,  0.753757,  0.555606, -0.082238, -0.993489,  0.078845,  0.423458, -0.371305,  0.826327, -0.187842, -0.829703,  0.525650,  0.035874, -0.429636,  0.902289,  0.379701,  0.739632,  0.555673,  0.140960, -0.989638,  0.027312, -0.216388,  0.674592,  0.705763, -0.488948, -0.473008,  0.732935,  0.540026,  0.534303,  0.650302, -0.151254, -0.526909,  0.836355,  0.349424,  0.910130,  0.222635,  0.613736,  0.788884, -0.031467,  0.775806,  0.626035, -0.078778, -0.679174, -0.169457,  0.714148,  0.185186,  0.755643,  0.628260, -0.553357, -0.816032, -0.166994, -0.904626, -0.154805,  0.397099, -0.164086,  0.138870,  0.976622,  0.681364,  0.363094,  0.635535, -0.927695,  0.354258,  0.117829,  0.404595, -0.175479,  0.897502,  0.936195, -0.038731,  0.349339,  0.988788, -0.048839,  0.141115,  0.903620,  0.343545,  0.255828,  0.417584, -0.706642,  0.571210, -0.330841, -0.602514,  0.726307, -0.463254,  0.381789,  0.799771,  0.143732,  0.606333,  0.782114, -0.789317,  0.150016,  0.595377, -0.644341,  0.674968,  0.359504,  0.040169, -0.614613,  0.787805,  0.445011,  0.212801,  0.869874, -0.993854, -0.109356,  0.017215, -0.591336,  0.475254,  0.651503, -0.669628,  0.045299,  0.741314, -0.810136, -0.058903,  0.583275,  0.706582,  0.517242,  0.482910,  0.658562,  0.684698,  0.312226, -0.945775,  0.199277,  0.256512, -0.072938,  0.936669,  0.342536, -0.652307, -0.382517,  0.654352, -0.762905,  0.613677,  0.203410,  0.967466, -0.247109,  0.054286, -0.796792, -0.267484,  0.541826,  0.185253, -0.907086,  0.377989,  0.220448, -0.803562,  0.552894,  0.253911,  0.283582,  0.924722, -0.603123,  0.777102,  0.179872,  0.591716, -0.574700,  0.565324, -0.521891, -0.061833,  0.850768,  0.597220, -0.398712,  0.695958,  0.950657,  0.307340,  0.042343, -0.419495,  0.886576,  0.194955, -0.631253,  0.259356,  0.730926, -0.369551, -0.735626,  0.567703, -0.970285, -0.033710,  0.239605,  0.730372,  0.672568,  0.119203, -0.343989,  0.033653,  0.938370,  0.847927, -0.528908, -0.035725, -0.068250,  0.564289,  0.822752, -0.284327, -0.947333,  0.147371, -0.318790,  0.268441,  0.909017,  0.352109,  0.608170,  0.711441,  0.055267,  0.866287,  0.496480, -0.352701, -0.193036,  0.915608, -0.330939, -0.404101,  0.852750,  0.428720, -0.550420,  0.716405, -0.001143, -0.991242, -0.132057,  0.180746,  0.100830,  0.978348,  0.236839, -0.502445,  0.831539,  0.560302,  0.664561,  0.494388,  0.736773, -0.220496,  0.639177,  0.466801,  0.807250,  0.361171};
static float ps150 [450] = {  0.374279, -0.424123,  0.824642, -0.791503,  0.045115,  0.609498,  0.525612, -0.692064,  0.494751,  0.844951, -0.487953,  0.218996, -0.031644,  0.612132,  0.790123,  0.282933, -0.224262,  0.932553, -0.232227,  0.018381,  0.972488,  0.716004,  0.586129,  0.379197, -0.826546, -0.399099, -0.396916, -0.394386,  0.393754,  0.830311,  0.469957, -0.234236,  0.851043,  0.572009, -0.762428,  0.302504,  0.523251,  0.502874,  0.687988, -0.130149, -0.543965,  0.828953, -0.754087, -0.532113,  0.384979, -0.098429,  0.185759,  0.977653, -0.495041, -0.517748,  0.697762, -0.530911,  0.224519,  0.817144, -0.085251, -0.712047,  0.696937,  0.530782,  0.159277,  0.832407, -0.211466,  0.531823,  0.820028, -0.555591, -0.145110,  0.818695, -0.621193, -0.344665,  0.703793,  0.326207, -0.738114,  0.590573, -0.326282, -0.486708,  0.810343, -0.972887,  0.058169,  0.223845, -0.801453,  0.561822,  0.205012,  0.705384, -0.608003,  0.364369,  0.210000, -0.015507,  0.977578,  0.976200, -0.150784,  0.155876, -0.248218, -0.795307,  0.553059, -0.559189, -0.639677, -0.527372,  0.911774, -0.287568,  0.293212,  0.032950,  0.844252, -0.534933, -0.263612, -0.963448,  0.047714, -0.596185, -0.696626,  0.399093, -0.116482, -0.973528,  0.196661,  0.230071, -0.933261,  0.275848,  0.254717,  0.954932,  0.152395, -0.293831, -0.655187,  0.695984,  0.905612,  0.186759,  0.380772, -0.831550,  0.239292,  0.501263, -0.698462,  0.235613,  0.675749,  0.972112,  0.230809,  0.041541, -0.185489,  0.837299,  0.514320, -0.396421,  0.576248,  0.714695, -0.976240,  0.205487,  0.068778, -0.205327,  0.700458,  0.683520,  0.239706,  0.347733,  0.906434, -0.700405,  0.577321,  0.419683,  0.735755, -0.658492,  0.158281, -0.028410,  0.937283, -0.347409, -0.568084,  0.415816,  0.710195, -0.999894,  0.006905, -0.012841,  0.457042, -0.586145,  0.668990,  0.571526,  0.745212,  0.343535,  0.404299,  0.877308,  0.258600,  0.691386,  0.459969,  0.557148,  0.334296,  0.526631,  0.781604,  0.929749,  0.327039, -0.169149,  0.396209,  0.794837,  0.459622,  0.135107,  0.515845,  0.845961, -0.832315,  0.417886,  0.364175, -0.715790, -0.153378,  0.681263,  0.354076, -0.855640, -0.377506,  0.661130, -0.704579, -0.257827, -0.640600, -0.766817,  0.040293, -0.279755,  0.958891, -0.047590,  0.179566, -0.857709,  0.481758, -0.190308, -0.903066,  0.385038, -0.637619, -0.536603,  0.552720,  0.836079, -0.548599,  0.003435,  0.788476,  0.260197,  0.557318, -0.460422, -0.684627,  0.565064, -0.454600,  0.030883,  0.890160,  0.172978,  0.678667,  0.713785,  0.375523,  0.676276,  0.633746,  0.831212,  0.517080,  0.204242,  0.412999, -0.034420,  0.910081,  0.936315, -0.053213,  0.347106, -0.532860, -0.817359,  0.219056, -0.194987,  0.358050,  0.913116, -0.448909, -0.893392,  0.018194, -0.408880, -0.816595,  0.407419,  0.116265,  0.189934,  0.974888, -0.902411,  0.049355,  0.428039,  0.647362, -0.524953,  0.552582, -0.560885,  0.588764,  0.582035,  0.597241, -0.032462,  0.801405,  0.067568,  0.997302,  0.028683, -0.705305, -0.674230,  0.218996,  0.900328,  0.434824,  0.018365,  0.639381, -0.223325,  0.735743,  0.441595,  0.346941,  0.827421,  0.420482, -0.886204,  0.194517, -0.981729, -0.129669,  0.139261, -0.922820,  0.242337,  0.299458, -0.333148,  0.199180,  0.921596,  0.083952, -0.206037,  0.974936, -0.126146, -0.170108,  0.977318,  0.122580, -0.740029,  0.661311,  0.788441,  0.615061, -0.007782, -0.511278,  0.831473,  0.217364,  0.838516,  0.510511, -0.190447, -0.468262,  0.883086,  0.029840, -0.920931, -0.322138, -0.219349,  0.179468, -0.412464,  0.893121,  0.856941, -0.194004,  0.477508, -0.451101, -0.322527,  0.832156, -0.377443,  0.733599,  0.565128,  0.068772, -0.580535,  0.811326, -0.343067, -0.910370,  0.231368,  0.737852, -0.329679,  0.588970, -0.594916,  0.798024, -0.096087, -0.317021,  0.931610,  0.177763,  0.804103, -0.423027,  0.417692, -0.929921, -0.147241,  0.336997,  0.711821,  0.115269,  0.692838,  0.199858,  0.808773,  0.553121, -0.530205,  0.734664,  0.423264, -0.861782, -0.348066,  0.369029, -0.023917, -0.383307,  0.923311,  0.335085,  0.161860,  0.928181,  0.085401, -0.988335,  0.126096,  0.217496,  0.905007,  0.365594,  0.004586,  0.384883,  0.922954,  0.693369, -0.718029, -0.060611,  0.852506,  0.047150,  0.520586,  0.975166,  0.089533,  0.202569,  0.908696, -0.392493, -0.142200, -0.000000, -0.000000,  1.000000,  0.559733, -0.413976,  0.717860, -0.152095,  0.934762,  0.321071,  0.552228,  0.821940,  0.139497, -0.349857, -0.153418,  0.924155,  0.265503, -0.596463,  0.757456,  0.382462, -0.831923,  0.402028,  0.017533,  0.891512,  0.452657, -0.126277,  0.986461,  0.104641, -0.230456, -0.335819,  0.913299, -0.840467, -0.155538,  0.519060,  0.930936, -0.357095,  0.076427,  0.762964, -0.090396,  0.640090,  0.630645,  0.314236,  0.709608, -0.757463, -0.354123,  0.548495, -0.717977,  0.416861,  0.557437, -0.007117,  0.773287,  0.634016,  0.707381,  0.684348,  0.176858, -0.642336,  0.044540,  0.765127, -0.049101, -0.968407, -0.244494};
//static float ps246 [738] = { -0.996801000000, 0.000000000000, 0.079922200000, 0.992869000000, 0.090802700000, 0.077241400000, 0.992869000000, -0.090802700000, 0.077241400000, -0.979656000000, -0.184676000000, 0.078547500000, -0.979656000000, 0.184676000000, 0.078547500000, 0.975263000000, -0.000000000000, 0.221047000000, -0.966274000000, -0.093954200000, 0.239767000000, -0.966274000000, 0.093954200000, 0.239767000000, 0.961659000000, 0.263845000000, 0.074813400000, 0.961659000000, -0.263845000000, 0.074813400000, 0.960950000000, 0.170695000000, 0.217803000000, 0.960950000000, -0.170695000000, 0.217803000000, 0.935659000000, 0.080770400000, 0.343537000000, 0.935659000000, -0.080770400000, 0.343537000000, -0.933862000000, -0.272408000000, 0.231724000000, -0.933862000000, 0.272408000000, 0.231724000000, -0.933083000000, -0.351794000000, 0.074813400000, -0.933083000000, 0.351794000000, 0.074813400000, -0.919651000000, 0.000000000000, 0.392738000000, 0.915627000000, 0.328527000000, 0.231724000000, 0.915627000000, -0.328527000000, 0.231724000000, -0.904358000000, -0.181605000000, 0.386207000000, -0.904358000000, 0.181605000000, 0.386207000000, 0.901581000000, 0.226771000000, 0.368411000000, 0.901581000000, -0.226771000000, 0.368411000000, 0.901108000000, 0.426421000000, 0.078547500000, 0.901108000000, -0.426421000000, 0.078547500000, 0.894427000000, -0.000000000000, 0.447214000000, -0.877757000000, -0.426737000000, 0.217803000000, -0.877757000000, 0.426737000000, 0.217803000000, 0.866952000000, 0.130689000000, 0.480952000000, 0.866952000000, -0.130689000000, 0.480952000000, -0.862687000000, -0.346475000000, 0.368411000000, -0.862687000000, 0.346475000000, 0.368411000000, -0.856620000000, -0.510133000000, 0.077241400000, -0.856620000000, 0.510133000000, 0.077241400000, -0.847355000000, -0.087948400000, 0.523694000000, -0.847355000000, 0.087948400000, 0.523694000000, 0.838386000000, 0.384646000000, 0.386207000000, 0.838386000000, -0.384646000000, 0.386207000000, 0.836957000000, 0.491951000000, 0.239767000000, 0.836957000000, -0.491951000000, 0.239767000000, 0.824489000000, -0.000000000000, 0.565879000000, -0.822295000000, -0.256042000000, 0.508206000000, -0.822295000000, 0.256042000000, 0.508206000000, 0.815748000000, 0.276190000000, 0.508206000000, 0.815748000000, -0.276190000000, 0.508206000000, 0.806429000000, 0.585905000000, 0.079922200000, 0.806429000000, -0.585905000000, 0.079922200000, -0.804440000000, -0.484622000000, 0.343537000000, -0.804440000000, 0.484622000000, 0.343537000000, -0.789005000000, -0.573245000000, 0.221047000000, -0.789005000000, 0.573245000000, 0.221047000000, 0.782361000000, 0.140152000000, 0.606852000000, 0.782361000000, -0.140152000000, 0.606852000000, -0.778196000000, -0.403852000000, 0.480952000000, -0.778196000000, 0.403852000000, 0.480952000000, -0.762555000000, 0.000000000000, 0.646923000000, -0.749875000000, -0.657055000000, 0.077241400000, -0.749875000000, -0.181605000000, 0.636165000000, -0.749875000000, 0.181605000000, 0.636165000000, -0.749875000000, 0.657055000000, 0.077241400000, 0.744013000000, 0.540557000000, 0.392738000000, 0.744013000000, -0.540557000000, 0.392738000000, 0.737219000000, 0.426911000000, 0.523694000000, 0.737219000000, -0.426911000000, 0.523694000000, 0.726507000000, 0.643972000000, 0.239767000000, 0.726507000000, -0.643972000000, 0.239767000000, 0.726009000000, -0.000000000000, 0.687685000000, -0.723607000000, -0.525731000000, 0.447214000000, -0.723607000000, 0.525731000000, 0.447214000000, -0.715322000000, -0.346475000000, 0.606852000000, -0.715322000000, 0.346475000000, 0.606852000000, 0.713407000000, 0.293844000000, 0.636165000000, 0.713407000000, -0.293844000000, 0.636165000000, -0.709489000000, -0.615311000000, 0.343537000000, -0.709489000000, 0.615311000000, 0.343537000000, 0.684008000000, 0.725233000000, 0.078547500000, 0.684008000000, -0.725233000000, 0.078547500000, -0.677093000000, -0.702927000000, 0.217803000000, -0.677093000000, 0.702927000000, 0.217803000000, -0.667025000000, -0.484622000000, 0.565879000000, -0.667025000000, 0.484622000000, 0.565879000000, 0.665669000000, 0.146922000000, 0.731641000000, 0.665669000000, -0.146922000000, 0.731641000000, -0.646585000000, -0.093954200000, 0.757034000000, -0.646585000000, 0.093954200000, 0.757034000000, 0.633829000000, 0.569214000000, 0.523694000000, 0.633829000000, -0.569214000000, 0.523694000000, 0.624896000000, 0.678490000000, 0.386207000000, -0.624896000000, -0.272408000000, 0.731641000000, -0.624896000000, 0.272408000000, 0.731641000000, 0.624896000000, -0.678490000000, 0.386207000000, -0.624562000000, -0.615311000000, 0.480952000000, -0.624562000000, 0.615311000000, 0.480952000000, -0.622914000000, -0.778705000000, 0.074813400000, -0.622914000000, 0.778705000000, 0.074813400000, 0.616920000000, 0.448219000000, 0.646923000000, 0.616920000000, -0.448219000000, 0.646923000000, 0.598507000000, -0.000000000000, 0.801117000000, -0.596102000000, -0.713397000000, 0.368411000000, -0.596102000000, 0.713397000000, 0.368411000000, 0.595392000000, 0.769293000000, 0.231724000000, 0.595392000000, -0.769293000000, 0.231724000000, -0.587353000000, -0.426737000000, 0.687685000000, -0.587353000000, 0.426737000000, 0.687685000000, 0.578323000000, 0.304042000000, 0.757034000000, 0.578323000000, -0.304042000000, 0.757034000000, -0.550564000000, -0.573245000000, 0.606852000000, -0.550564000000, 0.573245000000, 0.606852000000, 0.548101000000, 0.833060000000, 0.074813400000, 0.548101000000, -0.833060000000, 0.074813400000, -0.547655000000, -0.803977000000, 0.231724000000, -0.547655000000, 0.803977000000, 0.231724000000, 0.519830000000, 0.149406000000, 0.841103000000, 0.519830000000, -0.149406000000, 0.841103000000, -0.517268000000, 0.000000000000, 0.855824000000, 0.514753000000, 0.690475000000, 0.508206000000, 0.514753000000, -0.690475000000, 0.508206000000, -0.508370000000, -0.184676000000, 0.841103000000, -0.508370000000, 0.184676000000, 0.841103000000, 0.499917000000, 0.587688000000, 0.636165000000, 0.499917000000, -0.587688000000, 0.636165000000, -0.497614000000, -0.702927000000, 0.508206000000, -0.497614000000, 0.702927000000, 0.508206000000, 0.494276000000, 0.787378000000, 0.368411000000, 0.494276000000, -0.787378000000, 0.368411000000, -0.484203000000, -0.351794000000, 0.801117000000, -0.484203000000, 0.351794000000, 0.801117000000, -0.478368000000, -0.874640000000, 0.078547500000, -0.478368000000, 0.874640000000, 0.078547500000, 0.467873000000, 0.456063000000, 0.757034000000, 0.467873000000, -0.456063000000, 0.757034000000, 0.459290000000, 0.861170000000, 0.217803000000, 0.459290000000, -0.861170000000, 0.217803000000, -0.452179000000, -0.803977000000, 0.386207000000, -0.452179000000, -0.510133000000, 0.731641000000, -0.452179000000, 0.510133000000, 0.731641000000, -0.452179000000, 0.803977000000, 0.386207000000, 0.448881000000, -0.000000000000, 0.893592000000, 0.418478000000, 0.304042000000, 0.855824000000, 0.418478000000, -0.304042000000, 0.855824000000, -0.404441000000, -0.657055000000, 0.636165000000, -0.404441000000, 0.657055000000, 0.636165000000, 0.393172000000, 0.916215000000, 0.077241400000, 0.393172000000, -0.916215000000, 0.077241400000, 0.392196000000, 0.784135000000, 0.480952000000, 0.392196000000, -0.784135000000, 0.480952000000, -0.387951000000, -0.889947000000, 0.239767000000, -0.387951000000, 0.889947000000, 0.239767000000, 0.375055000000, 0.700760000000, 0.606852000000, 0.375055000000, -0.700760000000, 0.606852000000, -0.374938000000, -0.090802700000, 0.922592000000, -0.374938000000, 0.090802700000, 0.922592000000, 0.365952000000, 0.864906000000, 0.343537000000, 0.365952000000, -0.864906000000, 0.343537000000, -0.363152000000, -0.263845000000, 0.893592000000, -0.363152000000, 0.263845000000, 0.893592000000, 0.356703000000, 0.146922000000, 0.922592000000, 0.356703000000, -0.146922000000, 0.922592000000, -0.345491000000, -0.778705000000, 0.523694000000, -0.345491000000, 0.778705000000, 0.523694000000, 0.345434000000, 0.587688000000, 0.731641000000, 0.345434000000, -0.587688000000, 0.731641000000, -0.332733000000, -0.426421000000, 0.841103000000, -0.332733000000, 0.426421000000, 0.841103000000, -0.308028000000, -0.948014000000, 0.079922200000, -0.308028000000, 0.948014000000, 0.079922200000, 0.302730000000, 0.448219000000, 0.841103000000, 0.302730000000, -0.448219000000, 0.841103000000, 0.301373000000, 0.927530000000, 0.221047000000, 0.301373000000, -0.927530000000, 0.221047000000, 0.290404000000, -0.000000000000, 0.956904000000, -0.289161000000, -0.585905000000, 0.757034000000, -0.289161000000, 0.585905000000, 0.757034000000, -0.284188000000, -0.874640000000, 0.392738000000, -0.284188000000, 0.874640000000, 0.392738000000, 0.276393000000, 0.850651000000, 0.447214000000, 0.276393000000, -0.850651000000, 0.447214000000, 0.254781000000, 0.784135000000, 0.565879000000, 0.254781000000, -0.784135000000, 0.565879000000, 0.249958000000, 0.293844000000, 0.922592000000, 0.249958000000, -0.293844000000, 0.922592000000, -0.238441000000, 0.000000000000, 0.971157000000, -0.235643000000, -0.725233000000, 0.646923000000, -0.235643000000, 0.725233000000, 0.646923000000, -0.234941000000, -0.170695000000, 0.956904000000, -0.234941000000, 0.170695000000, 0.956904000000, 0.224349000000, 0.690475000000, 0.687685000000, 0.224349000000, -0.690475000000, 0.687685000000, 0.220455000000, 0.972334000000, 0.077241400000, 0.220455000000, -0.972334000000, 0.077241400000, 0.212317000000, 0.914824000000, 0.343537000000, 0.212317000000, -0.914824000000, 0.343537000000, -0.209239000000, -0.948014000000, 0.239767000000, -0.209239000000, 0.948014000000, 0.239767000000, -0.202221000000, -0.328527000000, 0.922592000000, -0.202221000000, 0.328527000000, 0.922592000000, 0.192903000000, 0.140152000000, 0.971157000000, 0.192903000000, -0.140152000000, 0.971157000000, 0.184949000000, 0.569214000000, 0.801117000000, 0.184949000000, -0.569214000000, 0.801117000000, -0.178203000000, -0.833060000000, 0.523694000000, -0.178203000000, 0.833060000000, 0.523694000000, -0.159844000000, -0.491951000000, 0.855824000000, -0.159844000000, 0.491951000000, 0.855824000000, 0.143610000000, 0.864906000000, 0.480952000000, 0.143610000000, -0.864906000000, 0.480952000000, 0.138712000000, 0.426911000000, 0.893592000000, 0.138712000000, -0.426911000000, 0.893592000000, 0.137415000000, -0.000000000000, 0.990514000000, 0.134609000000, 0.966666000000, 0.217803000000, 0.134609000000, -0.966666000000, 0.217803000000, -0.127093000000, -0.988776000000, 0.078547500000, -0.127093000000, 0.988776000000, 0.078547500000, -0.111171000000, -0.080770400000, 0.990514000000, -0.111171000000, 0.080770400000, 0.990514000000, -0.110450000000, -0.643972000000, 0.757034000000, -0.110450000000, 0.643972000000, 0.757034000000, 0.108470000000, 0.787378000000, 0.606852000000, 0.108470000000, -0.787378000000, 0.606852000000, -0.106745000000, -0.916215000000, 0.386207000000, -0.106745000000, 0.916215000000, 0.386207000000, 0.089739600000, 0.276190000000, 0.956904000000, 0.089739600000, -0.276190000000, 0.956904000000, -0.073682200000, -0.226771000000, 0.971157000000, -0.073682200000, 0.226771000000, 0.971157000000, 0.065972000000, 0.678490000000, 0.731641000000, 0.065972000000, -0.678490000000, 0.731641000000, 0.062932200000, 0.927530000000, 0.368411000000, 0.062932200000, -0.927530000000, 0.368411000000, -0.059007200000, -0.769293000000, 0.636165000000, -0.059007200000, 0.769293000000, 0.636165000000, 0.046237200000, 0.996125000000, 0.074813400000, 0.046237200000, -0.996125000000, 0.074813400000, 0.042463500000, 0.130689000000, 0.990514000000, 0.042463500000, -0.130689000000, 0.990514000000, -0.029503600000, -0.972334000000, 0.231724000000, -0.029503600000, -0.384646000000, 0.922592000000, -0.029503600000, 0.384646000000, 0.922592000000, -0.029503600000, 0.972334000000, 0.231724000000, 0.018542600000, 0.540557000000, 0.841103000000, 0.018542600000, -0.540557000000, 0.841103000000, -0.010592300000, -0.861170000000, 0.508206000000, -0.010592300000, 0.861170000000, 0.508206000000, -0.000000000000, -0.000000000000, 1.000000000000};
//static float ps256 [768] = { -0.620130, 0.642380, 0.450318, -0.718782, 0.617632, 0.319191, -0.248402, 0.840801, 0.480988, -0.570135, -0.166209, 0.804562, 0.950801, 0.065535, 0.302791, -0.654486, -0.600488, 0.459416, 0.659421, 0.308655, 0.685489, 0.794086, 0.087188, 0.601519, 0.627976, -0.125828, 0.767993, 0.663437, -0.414870, 0.622683, -0.467373, 0.236486, 0.851843, -0.649163, -0.026427, 0.760190, -0.764200, -0.037654, 0.643879, 0.645691, 0.716458, 0.264142, 0.311288, -0.707015, 0.635004, -0.068251, -0.913618, 0.400803, 0.921930, -0.207222, 0.327268, -0.375380, -0.077486, 0.923627, -0.558133, -0.829707, -0.008603, -0.837962, -0.482703, 0.254591, -0.036415, -0.973851, 0.224251, -0.894816, 0.088757, 0.437524, 0.529977, 0.365359, 0.765269, 0.128541, -0.547209, 0.827067, 0.038387, 0.858153, 0.511957, -0.256205, 0.730722, 0.632775, 0.313154, 0.146255, 0.938373, 0.920560, 0.365562, 0.137601, 0.016146, -0.758089, 0.651951, -0.257125, -0.765503, 0.589824, 0.901740, -0.352943, 0.249594, -0.091927, -0.005200, 0.995752, -0.498853, 0.382347, 0.777789, -0.957324, 0.079378, 0.277903, -0.101911, -0.822626, 0.559375, -0.215483, -0.864219, 0.454634, 0.119626, -0.399029, 0.909102, 0.835600, -0.053750, 0.546702, 0.656971, -0.271980, 0.703147, 0.962514, -0.218925, 0.160118, -0.794647, -0.193686, 0.575345, 0.543597, -0.510360, 0.666359, 0.183203, 0.579642, 0.794010, 0.239699, 0.921520, 0.305523, 0.429131, 0.648634, 0.628587, 0.272929, 0.302174, 0.913346, 0.564022, 0.207258, 0.799327, 0.347074, -0.014907, 0.937719, 0.171548, 0.793770, 0.583524, 0.854685, -0.197055, 0.480295, 0.836728, 0.494048, 0.236226, 0.672090, -0.729131, 0.129089, 0.582074, -0.687731, 0.433838, -0.607903, 0.255359, 0.751829, -0.832957, 0.377875, 0.404220, 0.019866, -0.255952, -0.966485, 0.222679, -0.124972, 0.966848, 0.847158, -0.341919, 0.406711, 0.734568, 0.590964, 0.333423, -0.793510, 0.598167, 0.111973, 0.156910, -0.681296, 0.714993, 0.155387, -0.987852, -0.002013, -0.126756, 0.688721, 0.713861, -0.169980, -0.440767, 0.881380, 0.066832, -0.067507, 0.995478, -0.028045, -0.346395, 0.937669, 0.688272, 0.537729, 0.486959, 0.425722, 0.257792, 0.867355, -0.317481, -0.219554, 0.922498, 0.977462, -0.210963, -0.007885, -0.262644, 0.042262, 0.963967, 0.987630, 0.071435, 0.139585, -0.295400, -0.939523, 0.173307, 0.429713, -0.900955, 0.060230, -0.780770, -0.606777, 0.149066, -0.699430, 0.116328, 0.705170, -0.871481, 0.490374, -0.007319, 0.784918, -0.616502, 0.061882, -0.628289, 0.395222, 0.670114, -0.693675, 0.719711, 0.028833, 0.686486, 0.148024, 0.711917, -0.516393, -0.706505, 0.483932, -0.419656, -0.902843, -0.093611, -0.452757, -0.250257, 0.855794, -0.677330, -0.733726, 0.053578, -0.174263, -0.286354, 0.942143, 0.295657, -0.804905, 0.514505, -0.869061, -0.348774, 0.350842, -0.180734, -0.939832, 0.289916, -0.685062, -0.472675, 0.554318, -0.483363, -0.803152, 0.348291, -0.877589, -0.203823, 0.433928, -0.306188, -0.514540, 0.800936, 0.248074, -0.287721, 0.925028, 0.076252, -0.940511, 0.331095, 0.611572, 0.671740, 0.418025, -0.152410, -0.584055, 0.797277, 0.437590, -0.758547, 0.482826, -0.065740, 0.406353, 0.911348, -0.253187, 0.598285, 0.760231, -0.598244, 0.738975, 0.309872, 0.735456, -0.511846, -0.443980, -0.736505, -0.598332, 0.315531, 0.720077, -0.020782, 0.693583, 0.735547, 0.394694, 0.550624, 0.617284, 0.461182, 0.637394, -0.831349, 0.244845, 0.498908, -0.692060, -0.182096, 0.698494, 0.953801, 0.218666, 0.206029, 0.780316, 0.625281, 0.011467, -0.353865, 0.349331, 0.867610, -0.325677, -0.881558, 0.341745, 0.395261, -0.892361, 0.217855, 0.119676, -0.979011, 0.164973, -0.105177, -0.971084, -0.214321, 0.277615, -0.954320, 0.110473, 0.774029, 0.237439, 0.586943, -0.741556, 0.388644, 0.546855, -0.935339, 0.344423, 0.080708, -0.730238, 0.255546, 0.633599, 0.929945, -0.357765, 0.084895, -0.373247, -0.795324, 0.477647, -0.567371, 0.110262, 0.816047, 0.854176, -0.492195, 0.167712, 0.340151, 0.551967, 0.761334, -0.808702, 0.105634, 0.578656, 0.861798, 0.501775, 0.074331, -0.319296, -0.370765, 0.872114, -0.569215, 0.821996, -0.017790, 0.837072, 0.296740, 0.459626, -0.584706, 0.798355, 0.144041, -0.134900, -0.711034, 0.690096, 0.266450, -0.441901, 0.856579, -0.366202, 0.849992, 0.378693, -0.019791, -0.499101, 0.866318, 0.386699, 0.410577, 0.825767, -0.698640, 0.690583, 0.187076, 0.484836, 0.513333, 0.708113, 0.044640, -0.863416, 0.502513, -0.916026, -0.353738, 0.189120, -0.453680, 0.885409, 0.101122, 0.374137, -0.175619, 0.910593, -0.286118, -0.646122, 0.707576, 0.028534, 0.099760, 0.994602, 0.412101, -0.469382, 0.780931, -0.287550, -0.957731, 0.008113, -0.416431, 0.088875, 0.904813, -0.933220, -0.358525, 0.023689, -0.814096, 0.502744, 0.290682, 0.229816, 0.446843, 0.864590, -0.054564, -0.177204, 0.982660, 0.598571, 0.038613, 0.800138, -0.580704, -0.315567, 0.750467, 0.130867, 0.698714, 0.703329, -0.336993, 0.915939, 0.217923, -0.493934, 0.756214, 0.429150, -0.666933, -0.737672, -0.105070, -0.476896, 0.838116, 0.264823, 0.508576, 0.038874, -0.860139, -0.188440, 0.307793, 0.932606, -0.058191, 0.960034, 0.273766, 0.476758, 0.731455, 0.487519, 0.190810, 0.040014, 0.980811, 0.027218, -0.993250, -0.112758, -0.132347, -0.989745, -0.053756, -0.233702, 0.452517, 0.860588, -0.977785, -0.067101, 0.198579, 0.010295, 0.629172, 0.777198, -0.206855, -0.112640, 0.971866, 0.755718, -0.166569, 0.633360, -0.867686, -0.489340, 0.087566, -0.540484, -0.822680, -0.176279, -0.552383, -0.588815, 0.590060, 0.408374, -0.600034, 0.687887, 0.151396, 0.201806, 0.967654, -0.908588, 0.223984, 0.352561, 0.284267, 0.680542, 0.675318, 0.756288, -0.614133, 0.225541, -0.673174, -0.709401, 0.208774, 0.997067, 0.070525, -0.029721, 0.512916, 0.789942, 0.336018, 0.518916, -0.227678, 0.823947, -0.385114, 0.633154, 0.671419, 0.876850, 0.101736, 0.469876, 0.540608, -0.822439, 0.177023, 0.236040, -0.933091, 0.271341, 0.762904, 0.623095, 0.172424, 0.498306, -0.800809, 0.332257, 0.707924, -0.593135, 0.383451, 0.541684, -0.371693, 0.753938, -0.634193, 0.560004, -0.533099, 0.058866, 0.497286, 0.865587, 0.564270, 0.600864, 0.566181, 0.329542, 0.771485, 0.544254, 0.635964, -0.717502, 0.284149, 0.366066, 0.839325, 0.401906, -0.407014, -0.684778, 0.604499, 0.001305, 0.764224, 0.644950, 0.159817, -0.790250, 0.591578, -0.562474, -0.810067, 0.165573, -0.143285, 0.152241, 0.977902, 0.153335, 0.981772, -0.112299, -0.608221, -0.714282, 0.346220, -0.775181, -0.476385, 0.414912, -0.991473, 0.073673, 0.107490, 0.308573, -0.949687, -0.053640, 0.497279, -0.650830, 0.573702, -0.306653, 0.202320, 0.930070, 0.798532, 0.450359, 0.399405, -0.379113, 0.753469, 0.537176, -0.125276, 0.811828, 0.570299, 0.396121, 0.883338, 0.250605, 0.186984, -0.884024, 0.428414, -0.091922, 0.899655, 0.426816, -0.433498, -0.898776, 0.065433, 0.217505, -0.919248, -0.328139, 0.097447, -0.236298, 0.966782, -0.452536, -0.412501, 0.790604, -0.869251, 0.470555, 0.151593, -0.900643, 0.350970, 0.256247, -0.349022, 0.861353, -0.369127, -0.936828, -0.210371, 0.279459, 0.073639, 0.926939, 0.367916, 0.207787, 0.868368, 0.450290, 0.749579, -0.456767, 0.479057, -0.858236, -0.051870, 0.510628, -0.000777, -0.998588, 0.053116, -0.436413, -0.556048, 0.707357, 0.266925, 0.951675, 0.151873, -0.111266, 0.546236, 0.830209, -0.508059, 0.648008, 0.567416, 0.001856, -0.639340, 0.768922, 0.914045, -0.055527, 0.401794, -0.630529, 0.525988, 0.570763, 0.969196, -0.076957, 0.233960, 0.909922, 0.205559, 0.360260, 0.974964, 0.217320, 0.047096, -0.795091, -0.340784, 0.501694, -0.185220, 0.968324, 0.167458, 0.272552, -0.582122, 0.766061, -0.510696, 0.521459, 0.683572, -0.959285, 0.214491, 0.183757, 0.886393, 0.354224, 0.298047, -0.930737, -0.061497, 0.360481, 0.463009, 0.095403, 0.881204, -0.697735, -0.332759, 0.634379, 0.995219, -0.071685, 0.066331, -0.574963, -0.457503, 0.678313, 0.105025, 0.354067, 0.929304, 0.396109, -0.329252, 0.857141, -0.768114, 0.310746, -0.559855, -0.433437, -0.873681, 0.220939, -0.378453, 0.496000, 0.781510, 0.493624, -0.069092, 0.866927, 0.813815, -0.482045, 0.324556, -0.969788, -0.214033, 0.117055};
//static float ps755 [2265] = { -0.9994266714825543, 0, -0.033857470804872, -0.9963958764033671, 0, 0.08482486360947544, -0.9916866386330564, -0.12860074160506021, 0.004411350734000926, -0.9916866386330564, 0.12860074160506021, 0.004411350734000926, -0.98607357578129, 0, -0.16630966041063486, -0.9857439588996775, -0.12048320108981783, 0.11744209529781555, -0.9857439588996775, 0.12048320108981783, 0.11744209529781555, -0.9830738669028043, -0.13576245663339054, -0.1230216549296677, -0.9830738669028043, 0.13576245663339054, -0.1230216549296677, -0.9822469463768458, 0, 0.1875924740850802, -0.9683162594941432, -0.24649537802113122, 0.04004560167472076, -0.9683162594941432, 0.24649537802113122, 0.04004560167472076, -0.967244736312809, -0.08971798394659967, 0.2374622147452561, -0.967244736312809, 0.08971798394659967, 0.2374622147452561, -0.9621351351385714, -0.2605138798701924, -0.0801779279282143, -0.9621351351385714, 0.2605138798701924, -0.0801779279282143, -0.9614507908933246, -0.22165711242617028, 0.16272830485656992, -0.9614507908933246, 0.22165711242617028, 0.16272830485656992, -0.9551169259871324, -0.14106195520847561, -0.2604864343601271, -0.9551169259871324, 0.14106195520847561, -0.2604864343601271, -0.9517112057440018, 0, -0.3069947570583225, -0.9465965938004058, 0, 0.32242036009760555, -0.9389619707336497, -0.271524913266781, -0.21124544726797656, -0.9389619707336497, 0.271524913266781, -0.21124544726797656, -0.9373125561577, -0.19116759016150822, 0.2913762593983249, -0.9373125561577, 0.19116759016150822, 0.2913762593983249, -0.931509891817533, -0.35476868706133874, 0.0801779279282143, -0.931509891817533, 0.35476868706133874, 0.0801779279282143, -0.9282706578194226, -0.36974306703169696, -0.040045601674720765, -0.9282706578194226, 0.36974306703169696, -0.040045601674720765, -0.9196658769956819, -0.0930563708211101, 0.38151695445860245, -0.9196658769956819, 0.0930563708211101, 0.38151695445860245, -0.9192345310435216, -0.3322397296317068, 0.21124544726797656, -0.9192345310435215, 0.3322397296317067, 0.21124544726797653, -0.9081168108377711, -0.38580222481518056, -0.1627283048565699, -0.9081168108377711, 0.38580222481518056, -0.1627283048565699, -0.905039215834912, -0.14364161499041464, -0.4003387368765019, -0.905039215834912, 0.14364161499041464, -0.4003387368765019, -0.895768418755139, -0.2779306951014105, -0.3469199744629367, -0.895768418755139, 0.2779306951014105, -0.3469199744629367, -0.8944271909999159, 0, -0.447213595499958, -0.8882250561447018, 0, 0.45940858681215463, -0.8882250561447018, 0, 0.45940858681215474, -0.8880554375372868, -0.3016688104181361, 0.34691997446293665, -0.8880554375372868, 0.3016688104181361, 0.34691997446293665, -0.8778809630980393, -0.47885859563629063, -0.004411350734000925, -0.8778809630980393, 0.47885859563629063, -0.004411350734000925, -0.8751226348743671, 0.19647727299831622, 0.4422183342260814, -0.8751226348743671, 0.4680021862650972, 0.12302165492966767, -0.875122634874367, -0.4680021862650972, 0.12302165492966767, -0.875122634874367, -0.19647727299831622, 0.4422183342260814, -0.8706672771858278, -0.39628066808369156, -0.2913762593983249, -0.8706672771858278, 0.39628066808369156, -0.2913762593983249, -0.8683018636018623, -0.48193280435927155, -0.1174420952978156, -0.8683018636018623, 0.48193280435927155, -0.1174420952978156, -0.8556199616698325, 0.44728212428674363, 0.2604864343601271, -0.8556199616698322, -0.44728212428674363, 0.2604864343601271, -0.8468638011056542, -0.09425480719114633, 0.5233906129252252, -0.8468638011056542, 0.09425480719114633, 0.5233906129252252, -0.8352523372260013, -0.4959488176483333, -0.2374622147452561, -0.8352523372260013, 0.4959488176483333, -0.2374622147452561, -0.8351224597613233, -0.22824303804682877, -0.5004753668117452, -0.8351224597613233, 0.22824303804682877, -0.5004753668117452, -0.8295223498394129, -0.08877540026300262, -0.5513724688674999, -0.8295223498394129, 0.08877540026300262, -0.5513724688674999, -0.8234755272545975, -0.35543072586240004, -0.4422183342260814, -0.8234755272545975, 0.35543072586240004, -0.4422183342260814, -0.8166225290930591, -0.41576019618739724, 0.4003387368765019, -0.8166225290930591, 0.41576019618739724, 0.4003387368765019, -0.8097861540334746, 0.306220169078268, 0.5004753668117451, -0.8097861540334745, -0.306220169078268, 0.5004753668117451, -0.8085531618609739, -0.5874482582451998, 0.033857470804871996, -0.8085531618609739, 0.5874482582451998, 0.033857470804871996, -0.8081338527952605, 0, 0.5889988760314302, -0.8061011971354436, -0.585666801594933, -0.08482486360947541, -0.8061011971354436, 0.585666801594933, -0.08482486360947541, -0.7987224860367548, -0.46528185410555045, -0.38151695445860245, -0.7987224860367548, 0.46528185410555045, -0.38151695445860245, -0.7977502805111363, 0.5795995055195465, 0.16630966041063483, -0.7977502805111361, -0.5795995055195465, 0.16630966041063483, -0.794654472291766, -0.5773502691896257, -0.1875924740850802, -0.794654472291766, 0.5773502691896257, -0.1875924740850802, -0.7868988425360586, 0.19647727299831622, 0.5849674288458774, -0.7868988425360585, -0.19647727299831622, 0.5849674288458774, -0.7699505391839696, 0.5594018111778118, 0.3069947570583225, -0.7699505391839695, -0.5594018111778118, 0.3069947570583225, -0.7658127312019672, -0.5563955177061672, -0.32242036009760555, -0.7658127312019672, 0.5563955177061672, -0.32242036009760555, -0.7609618011718721, 0, -0.6487966839906476, -0.7525262214051114, -0.0930563708211101, 0.6519545980720968, -0.7525262214051114, 0.0930563708211101, 0.6519545980720968, -0.7521009799446995, -0.3035740818246284, -0.5849674288458774, -0.7521009799446995, 0.3035740818246284, -0.5849674288458774, -0.7496516053274307, -0.17177061608955554, -0.6391536013183533, -0.7496516053274307, 0.17177061608955554, -0.6391536013183533, -0.7405287926400659, -0.4215203121710784, -0.5233906129252252, -0.7405287926400659, 0.4215203121710784, -0.5233906129252252, -0.7297825215675532, -0.641115565076049, -0.2374622147452561, -0.7297825215675532, 0.641115565076049, -0.2374622147452561, -0.7267017243993805, 0.6869389665317207, -0.004411350734000925, -0.7267017243993803, -0.6869389665317207, -0.004411350734000925, -0.7266653661026955, -0.6768787187959852, -0.11744209529781557, -0.7266653661026955, 0.6768787187959852, -0.11744209529781557, -0.7236067977499792, 0.5257311121191337, 0.447213595499958, -0.723606797749979, -0.5257311121191337, 0.447213595499958, -0.72327854927488, 0.41576019618739724, 0.5513724688674999, -0.7232785492748799, -0.41576019618739724, 0.5513724688674999, -0.7185891652507057, -0.5220855887185097, -0.45940858681215474, -0.7185891652507057, 0.5220855887185097, -0.45940858681215474, -0.7155242952261605, 0.6876704554941067, 0.12302165492966767, -0.7155242952261603, -0.6876704554941067, 0.12302165492966767, -0.7117124032047751, 0, 0.7024709639013443, -0.7074451234849853, -0.30166881041813615, 0.6391536013183533, -0.7074451234849853, 0.30166881041813615, 0.6391536013183533, -0.6897916878076654, -0.6755251623335724, 0.26048643436012703, -0.6897916878076654, 0.6755251623335724, 0.26048643436012703, -0.6893281612357385, -0.6158502249638206, -0.38151695445860245, -0.6893281612357385, 0.6158502249638206, -0.38151695445860245, -0.6807264326348014, -0.08390581259561895, -0.7277165234656735, -0.6807264326348014, 0.08390581259561895, -0.7277165234656735, -0.6797937675642477, 0.19116759016150822, 0.7080504120842295, -0.6797937675642476, -0.19116759016150822, 0.7080504120842295, -0.6635036642300076, -0.36703962947616986, -0.6519545980720968, -0.6635036642300076, 0.36703962947616986, -0.6519545980720968, -0.6623302008428754, -0.24491492196023698, -0.7080504120842295, -0.6623302008428754, 0.24491492196023698, -0.7080504120842295, -0.6537940206410678, -0.4750091605513505, -0.5889988760314302, -0.6537940206410678, 0.4750091605513505, -0.5889988760314302, -0.6477616832793809, 0.6481772114408144, 0.4003387368765019, -0.6477616832793808, -0.6481772114408144, 0.4003387368765019, -0.647543247338096, -0.7444509665688808, -0.1627283048565699, -0.647543247338096, 0.7444509665688808, -0.1627283048565699, -0.6459362967593755, -0.7055963265124218, -0.2913762593983249, -0.6459362967593755, 0.7055963265124218, -0.2913762593983249, -0.644957657958078, -0.08971798394659967, 0.7589336616581112, -0.644957657958078, 0.08971798394659967, 0.7589336616581112, -0.6384979619012642, -0.7685809667396409, -0.04004560167472076, -0.6384979619012642, 0.7685809667396409, -0.04004560167472076, -0.6297256213908132, -0.5740277938094211, -0.5233906129252252, -0.6297256213908132, 0.5740277938094211, -0.5233906129252252, -0.6252574586071488, -0.776288999232417, 0.08017792792821428, -0.6252574586071488, 0.776288999232417, 0.08017792792821428, -0.6189168071929715, 0.5594018111778118, 0.5513724688674999, -0.6189168071929714, -0.5594018111778118, 0.5513724688674999, -0.6156310292182143, 0.4472821242867436, 0.6487966839906475, -0.6156310292182142, -0.44728212428674363, 0.6487966839906476, -0.6070619982066863, 0, 0.7946544722917661, -0.6035685408534394, 0, -0.7973111164984783, -0.6000378517471081, 0.7715762680897256, 0.21124544726797656, -0.600037851747108, -0.7715762680897256, 0.21124544726797656, -0.600037851747108, -0.3322397296317067, 0.7277165234656734, -0.600037851747108, 0.3322397296317067, 0.7277165234656734, -0.595894253326634, -0.1589592585807398, -0.787173419878844, -0.595894253326634, 0.1589592585807398, -0.787173419878844, -0.5925026402964927, -0.6733376315905675, -0.4422183342260814, -0.5925026402964927, 0.6733376315905675, -0.4422183342260814, -0.5757874293000979, -0.418334054477401, -0.7024709639013442, -0.5757874293000979, 0.418334054477401, -0.7024709639013442, -0.5755224857007216, 0.22165711242617026, 0.7871734198788439, -0.5755224857007215, -0.22165711242617026, 0.7871734198788439, -0.574516613769574, -0.3065132259869937, -0.7589336616581113, -0.574516613769574, 0.3065132259869937, -0.7589336616581113, -0.5613283100572769, -0.7513701216091022, 0.3469199744629367, -0.5613283100572769, 0.751370121609102, 0.34691997446293665, -0.5541093394289913, -0.5176080003344399, -0.6519545980720968, -0.5541093394289913, 0.5176080003344399, -0.6519545980720968, -0.5483907441273382, -0.809100088223958, -0.21124544726797653, -0.5483907441273382, 0.809100088223958, -0.21124544726797653, -0.545881503504257, -0.12048320108981789, 0.8291544985025905, -0.545881503504257, 0.12048320108981789, 0.8291544985025905, -0.5450795306789344, -0.8345416736796134, -0.08017792792821428, -0.5450795306789344, 0.8345416736796134, -0.08017792792821428, -0.5414703706287631, 0.6755251623335724, 0.5004753668117451, -0.541470370628763, -0.6755251623335724, 0.5004753668117451, -0.5411354630743502, -0.7660410837037032, -0.34691997446293676, -0.5411354630743502, 0.7660410837037032, -0.34691997446293676, -0.5336572156169057, -0.8447522275830474, 0.04004560167472076, -0.5336572156169057, 0.8447522275830474, 0.04004560167472076, -0.5242037314217923, -0.07617126084340645, -0.8481794544699813, -0.5242037314217923, 0.07617126084340645, -0.8481794544699813, -0.5214714469128553, 0, 0.853268732612776, -0.5211280929865946, -0.6214809875527959, -0.5849674288458774, -0.5211280929865946, 0.6214809875527959, -0.5849674288458774, -0.5124456619994747, -0.22338814004068744, -0.8291544985025905, -0.5124456619994747, 0.22338814004068744, -0.8291544985025905, -0.5079130747973478, -0.8458982251124606, 0.1627283048565699, -0.5079130747973478, 0.8458982251124606, 0.1627283048565699, -0.5055166536557195, -0.5795995055195465, 0.6391536013183533, -0.5055166536557195, 0.5795995055195464, 0.6391536013183532, -0.5014006532964664, -0.4680021862650972, 0.7277165234656735, -0.5014006532964664, 0.4680021862650971, 0.7277165234656734, -0.49112347318842303, -0.35682208977308993, -0.7946544722917661, -0.49112347318842303, 0.35682208977308993, -0.7946544722917661, -0.4882972068205223, -0.35476868706133874, 0.7973111164984783, -0.4882972068205223, 0.35476868706133874, 0.7973111164984783, -0.47513906108389614, -0.7237176796562059, -0.5004753668117452, -0.47513906108389614, 0.7237176796562059, -0.5004753668117452, -0.4714566912212953, -0.8323631802054368, 0.2913762593983249, -0.4714566912212953, 0.8323631802054368, 0.2913762593983249, -0.46904679811112565, -0.45167997341470945, -0.7589336616581113, -0.46904679811112565, 0.45167997341470945, -0.7589336616581113, -0.46886207100726823, -0.24649537802113128, 0.8481794544699813, -0.46886207100726823, 0.24649537802113128, 0.8481794544699813, -0.45728875712731204, 0.7715762680897255, 0.4422183342260813, -0.457288757127312, -0.7715762680897256, 0.4422183342260814, -0.44744137931788447, -0.12860074160506021, 0.8850186785219495, -0.44744137931788447, 0.12860074160506021, 0.8850186785219495, -0.4432126849970109, 0, -0.8964164857128302, -0.4375992204164231, -0.5542305803889672, -0.7080504120842295, -0.4375992204164231, 0.5542305803889672, -0.7080504120842295, -0.4375772992040651, -0.1589592585807398, -0.8850186785219495, -0.4375772992040651, 0.1589592585807398, -0.8850186785219495, -0.4329043006482858, -0.893005900819577, -0.12302165492966766, -0.4329043006482858, 0.893005900819577, -0.12302165492966766, -0.4300261059865685, -0.6876704554941067, 0.5849674288458774, -0.4300261059865685, 0.6876704554941067, 0.5849674288458774, -0.42930525344753856, -0.8647796348646817, -0.26048643436012714, -0.42930525344753856, 0.8647796348646817, -0.26048643436012714, -0.4287545977360634, -0.9034102251496204, 0.004411350734000925, -0.4287545977360634, 0.9034102251496204, 0.004411350734000925, -0.4218792626337932, -0.3065132259869937, -0.853268732612776, -0.4218792626337932, 0.3065132259869937, -0.853268732612776, -0.4191979689030107, -0.9002668588366725, 0.11744209529781557, -0.4191979689030107, 0.9002668588366725, 0.11744209529781557, -0.41667415268590485, 0, 0.9090559116377184, -0.4162837922165572, -0.8163557435909476, -0.4003387368765019, -0.4162837922165572, 0.8163557435909476, -0.4003387368765019, -0.39501864964664196, -0.6598810046918482, -0.6391536013183534, -0.39501864964664196, 0.6598810046918482, -0.6391536013183534, -0.39187900917506946, 0.5874482582451998, 0.7080504120842295, -0.39187900917506935, -0.5874482582451998, 0.7080504120842295, -0.38865466988251257, 0.47885859563629063, 0.7871734198788439, -0.38865466988251246, -0.47885859563629063, 0.7871734198788439, -0.38422193450165043, 0.8921800275819267, 0.2374622147452561, -0.3842219345016504, -0.8921800275819267, 0.2374622147452561, -0.3793173834627134, -0.36974306703169685, 0.8481794544699811, -0.3793173834627134, 0.36974306703169685, 0.8481794544699811, -0.37269425299060066, -0.8458982251124604, 0.3815169544586024, -0.37269425299060066, 0.8458982251124604, 0.3815169544586024, -0.370809164500308, -0.41833405447740113, -0.8291544985025905, -0.370809164500308, 0.41833405447740113, -0.8291544985025905, -0.35856659428513216, -0.2605138798701924, 0.89641648571283, -0.35856659428513216, 0.2605138798701924, 0.89641648571283, -0.35133695503389567, 0.776288999232417, 0.523390612925225, -0.3513369550338956, -0.776288999232417, 0.523390612925225, -0.3464593304371572, -0.08390581259561895, -0.9343049539447521, -0.3464593304371572, 0.08390581259561895, -0.9343049539447521, -0.34076692622105814, -0.7614895288635357, -0.5513724688674999, -0.34076692622105814, 0.7614895288635357, -0.5513724688674999, -0.3370964706396787, -0.24491492196023704, -0.9090559116377184, -0.3370964706396787, 0.24491492196023704, -0.9090559116377184, -0.33532068982695884, -0.51760800033444, -0.787173419878844, -0.3353206898269588, 0.51760800033444, -0.787173419878844, -0.32961008540874664, -0.1357624566333905, 0.9343049539447521, -0.32961008540874664, 0.1357624566333905, 0.9343049539447521, -0.32104525897913894, -0.6869389665317207, 0.6519545980720968, -0.32104525897913894, 0.6869389665317207, 0.6519545980720968, -0.3088398261196968, -0.9505112484726589, -0.033857470804871996, -0.3088398261196968, 0.9505112484726589, -0.033857470804871996, -0.30790325893376, -0.9476287910630427, 0.08482486360947541, -0.30790325893376, 0.9476287910630427, 0.08482486360947541, -0.30471349262049113, -0.9378116997932586, -0.16630966041063486, -0.30471349262049113, 0.9378116997932586, -0.16630966041063486, -0.3035309991033431, -0.9341723589627156, 0.1875924740850802, -0.3035309991033431, 0.9341723589627156, 0.1875924740850802, -0.29409493631196854, -0.9051311438539504, -0.3069947570583225, -0.29409493631196854, 0.9051311438539504, -0.3069947570583225, -0.29251443430176444, -0.9002668588366725, 0.32242036009760555, -0.29251443430176444, 0.9002668588366725, 0.32242036009760555, -0.2922336268554169, 0, 0.9563469597039189, -0.2901552060284897, -0.6214809875527959, -0.7277165234656735, -0.2901552060284897, 0.6214809875527959, -0.7277165234656735, -0.28639806050540606, -0.3670396294761699, -0.8850186785219495, -0.28639806050540606, 0.3670396294761699, -0.8850186785219495, -0.28462975022258863, 0.585666801594933, 0.7589336616581112, -0.2846297502225886, -0.585666801594933, 0.7589336616581112, -0.2832729949983337, -0.48193280435927144, 0.8291544985025904, -0.2832729949983337, 0.48193280435927144, 0.8291544985025904, -0.27639320225002095, -0.85065080835204, -0.447213595499958, -0.27639320225002095, 0.85065080835204, -0.447213595499958, -0.27447663717835485, -0.8447522275830474, 0.45940858681215474, -0.27447663717835485, 0.8447522275830474, 0.45940858681215474, -0.26057356349967536, -0.38580222481518056, 0.8850186785219493, -0.26057356349967536, 0.38580222481518056, 0.8850186785219493, -0.24972709424343778, -0.7685809667396409, 0.5889988760314302, -0.24972709424343778, 0.7685809667396409, 0.5889988760314302, -0.23998893245161798, 0, -0.9707756240763016, -0.2364219704538591, -0.17177061608955557, -0.9563469597039189, -0.2364219704538591, 0.17177061608955557, -0.9563469597039189, -0.2351501286322781, -0.7237176796562059, -0.6487966839906476, -0.2351501286322781, 0.7237176796562059, -0.6487966839906476, -0.23443103550363395, -0.4750091605513506, -0.8481794544699813, -0.2344310355036339, -0.4750091605513506, -0.8481794544699813, -0.2344310355036339, 0.4750091605513506, -0.8481794544699813, -0.230972886958105, -0.271524913266781, 0.9343049539447521, -0.230972886958105, 0.271524913266781, 0.9343049539447521, -0.2199312276977106, -0.6768787187959852, 0.7024709639013442, -0.2199312276977106, 0.6768787187959852, 0.7024709639013442, -0.21356818797909516, -0.9476287910630424, 0.23746221474525606, -0.21356818797909516, 0.9476287910630424, 0.23746221474525606, -0.19569051728621087, -0.9034102251496204, 0.38151695445860245, -0.19569051728621087, 0.9034102251496204, 0.38151695445860245, -0.1941551248152605, -0.14106195520847564, 0.9707756240763016, -0.1941551248152605, 0.14106195520847564, 0.9707756240763016, -0.19002530190186945, -0.9747295721835683, 0.11744209529781557, -0.19002530190186945, 0.9747295721835683, 0.11744209529781557, -0.1875924740850802, -0.5773502691896257, 0.794654472291766, -0.1875924740850802, 0.5773502691896257, 0.794654472291766, -0.18686099078895035, -0.30357408182462847, -0.9343049539447523, -0.18686099078895035, 0.30357408182462847, -0.9343049539447523, -0.18651293639380231, -0.5740277938094211, -0.7973111164984783, -0.18651293639380231, 0.5740277938094211, -0.7973111164984783, -0.1841414511282999, -0.9828898544399904, 0.004411350734000925, -0.1841414511282999, 0.9828898544399904, 0.004411350734000925, -0.1746687625494374, -0.9769117134151961, -0.12302165492966767, -0.1746687625494374, 0.9769117134151961, -0.12302165492966767, -0.17205365789132968, -0.8345416736796134, 0.523390612925225, -0.17205365789132968, 0.8345416736796134, 0.523390612925225, -0.17190608040737973, -0.8163557435909476, -0.5513724688674999, -0.17190608040737973, 0.8163557435909476, -0.5513724688674999, -0.16114353917736582, -0.4959488176483333, 0.8532687326127759, -0.16114353917736582, 0.4959488176483333, 0.8532687326127759, -0.16098947004282693, -0.9519607177030348, -0.26048643436012714, -0.16098947004282693, 0.9519607177030348, -0.26048643436012714, -0.1510337319909983, 0, 0.9885286095004389, -0.14404152327474903, -0.7444509665688808, 0.6519545980720968, -0.14404152327474903, 0.7444509665688808, 0.6519545980720968, -0.14306120432097033, -0.9051311438539504, -0.4003387368765019, -0.14306120432097033, 0.9051311438539504, -0.4003387368765019, -0.13696025178662646, -0.42152031217107844, -0.8964164857128302, -0.13696025178662646, 0.42152031217107844, -0.8964164857128302, -0.13055686638028288, -0.6733376315905675, -0.7277165234656735, -0.13055686638028288, 0.6733376315905675, -0.7277165234656735, -0.12875939429672653, -0.39628066808369167, 0.9090559116377184, -0.12875939429672653, 0.39628066808369167, 0.9090559116377184, -0.12218885590458858, -0.08877540026300264, -0.988528609500439, -0.12218885590458858, 0.08877540026300264, -0.988528609500439, -0.11397600370003338, -0.641115565076049, 0.7589336616581112, -0.11397600370003338, 0.641115565076049, 0.7589336616581112, -0.10783432656620794, -0.9505112484726589, 0.2913762593983249, -0.10783432656620794, 0.9505112484726589, 0.2913762593983249, -0.09030515702615106, -0.27793069510141055, 0.9563469597039189, -0.09030515702615106, 0.27793069510141055, 0.9563469597039189, -0.08954468754455468, -0.5220855887185099, -0.8481794544699813, -0.08954468754455468, 0.5220855887185099, -0.8481794544699813, -0.08629619248519454, -0.9828898544399904, 0.1627283048565699, -0.08629619248519454, 0.9828898544399904, 0.1627283048565699, -0.08356677554941112, -0.893005900819577, 0.4422183342260813, -0.08356677554941112, 0.893005900819577, 0.4422183342260813, -0.07416065858945113, -0.22824303804682883, -0.9707756240763016, -0.07416065858945113, 0.22824303804682883, -0.9707756240763016, -0.06829152216663177, -0.7660410837037032, -0.6391536013183534, -0.06829152216663177, 0.7660410837037032, -0.6391536013183534, -0.06479514460963778, -0.9970947492698604, 0.040045601674720765, -0.06479514460963778, 0.9970947492698604, 0.040045601674720765, -0.056304124408667555, -0.8091000882239582, 0.5849674288458774, -0.056304124408667555, 0.8091000882239582, 0.5849674288458774, -0.0541003279971925, -0.5563955177061674, 0.8291544985025905, -0.0541003279971925, 0.5563955177061674, 0.8291544985025905, -0.049552684607175844, -0.9955481059804994, -0.08017792792821428, -0.049552684607175844, 0.9955481059804994, -0.08017792792821428, -0.046671989909089814, -0.14364161499041467, 0.988528609500439, -0.046671989909089814, 0.14364161499041467, 0.988528609500439, -0.04099500381701781, -0.8647796348646815, -0.5004753668117452, -0.04099500381701781, 0.8647796348646815, -0.5004753668117452, -0.032962212429640814, -0.6158502249638206, -0.7871734198788439, -0.032962212429640814, 0.6158502249638206, -0.7871734198788439, -0.03191966792964132, -0.9769117134151959, -0.21124544726797653, -0.03191966792964132, 0.9769117134151959, -0.21124544726797653, -0.028256644519982034, -0.7055963265124218, 0.7080504120842295, -0.028256644519982034, 0.7055963265124218, 0.7080504120842295, -0.027262651140743457, -0.35543072586240004, -0.9343049539447521, -0.027262651140743457, 0.35543072586240004, -0.9343049539447521, -0.0159604168919119, -0.4652818541055505, 0.8850186785219495, -0.0159604168919119, 0.4652818541055505, 0.8850186785219495, -0.012479865765074368, -0.9378116997932586, -0.34691997446293676, -0.012479865765074368, 0.9378116997932586, -0.34691997446293676, 0, 0, -1., 0, 0, 1., 0.012479865765074368, -0.9378116997932586, 0.34691997446293676, 0.012479865765074368, 0.9378116997932586, 0.34691997446293676, 0.0159604168919119, -0.4652818541055505, -0.8850186785219495, 0.0159604168919119, 0.4652818541055505, -0.8850186785219495, 0.027262651140743457, -0.35543072586240004, 0.9343049539447521, 0.027262651140743457, 0.35543072586240004, 0.9343049539447521, 0.028256644519982034, -0.7055963265124218, -0.7080504120842295, 0.028256644519982034, 0.7055963265124218, -0.7080504120842295, 0.03191966792964132, -0.9769117134151959, 0.21124544726797653, 0.03191966792964132, 0.9769117134151959, 0.21124544726797653, 0.032962212429640814, -0.6158502249638206, 0.7871734198788439, 0.032962212429640814, 0.6158502249638206, 0.7871734198788439, 0.04099500381701781, -0.8647796348646815, 0.5004753668117452, 0.04099500381701781, 0.8647796348646815, 0.5004753668117452, 0.046671989909089814, -0.14364161499041467, -0.988528609500439, 0.046671989909089814, 0.14364161499041467, -0.988528609500439, 0.049552684607175844, -0.9955481059804994, 0.08017792792821428, 0.049552684607175844, 0.9955481059804994, 0.08017792792821428, 0.0541003279971925, -0.5563955177061674, -0.8291544985025905, 0.0541003279971925, 0.5563955177061674, -0.8291544985025905, 0.056304124408667555, -0.8091000882239582, -0.5849674288458774, 0.056304124408667555, 0.8091000882239582, -0.5849674288458774, 0.06479514460963778, -0.9970947492698604, -0.040045601674720765, 0.06479514460963778, 0.9970947492698604, -0.040045601674720765, 0.06829152216663177, -0.7660410837037032, 0.6391536013183534, 0.06829152216663177, 0.7660410837037032, 0.6391536013183534, 0.07416065858945113, -0.22824303804682883, 0.9707756240763016, 0.07416065858945113, 0.22824303804682883, 0.9707756240763016, 0.08356677554941112, -0.893005900819577, -0.4422183342260813, 0.08356677554941112, 0.893005900819577, -0.4422183342260813, 0.08629619248519454, -0.9828898544399904, -0.1627283048565699, 0.08629619248519454, 0.9828898544399904, -0.1627283048565699, 0.08954468754455468, -0.5220855887185099, 0.8481794544699813, 0.08954468754455468, 0.5220855887185099, 0.8481794544699813, 0.09030515702615106, -0.27793069510141055, -0.9563469597039189, 0.09030515702615106, 0.27793069510141055, -0.9563469597039189, 0.10783432656620794, -0.9505112484726589, -0.2913762593983249, 0.10783432656620794, 0.9505112484726589, -0.2913762593983249, 0.11397600370003338, -0.641115565076049, -0.7589336616581112, 0.11397600370003338, 0.641115565076049, -0.7589336616581112, 0.12218885590458858, -0.08877540026300264, 0.988528609500439, 0.12218885590458858, 0.08877540026300264, 0.988528609500439, 0.12875939429672653, -0.39628066808369167, -0.9090559116377184, 0.12875939429672653, 0.39628066808369167, -0.9090559116377184, 0.13055686638028288, -0.6733376315905675, 0.7277165234656735, 0.13055686638028288, 0.6733376315905675, 0.7277165234656735, 0.13696025178662646, -0.42152031217107844, 0.8964164857128302, 0.13696025178662646, 0.42152031217107844, 0.8964164857128302, 0.14306120432097033, -0.9051311438539504, 0.4003387368765019, 0.14306120432097033, 0.9051311438539504, 0.4003387368765019, 0.14404152327474903, -0.7444509665688808, -0.6519545980720968, 0.14404152327474903, 0.7444509665688808, -0.6519545980720968, 0.1510337319909983, 0, -0.9885286095004389, 0.16098947004282693, -0.9519607177030348, 0.26048643436012714, 0.16098947004282693, 0.9519607177030348, 0.26048643436012714, 0.16114353917736582, -0.4959488176483333, -0.8532687326127759, 0.16114353917736582, 0.4959488176483333, -0.8532687326127759, 0.17190608040737973, -0.8163557435909476, 0.5513724688674999, 0.17190608040737973, 0.8163557435909476, 0.5513724688674999, 0.17205365789132968, -0.8345416736796134, -0.523390612925225, 0.17205365789132968, 0.8345416736796134, -0.523390612925225, 0.1746687625494374, -0.9769117134151961, 0.12302165492966767, 0.1746687625494374, 0.9769117134151961, 0.12302165492966767, 0.1841414511282999, -0.9828898544399904, -0.004411350734000925, 0.1841414511282999, 0.9828898544399904, -0.004411350734000925, 0.18651293639380231, -0.5740277938094211, 0.7973111164984783, 0.18651293639380231, 0.5740277938094211, 0.7973111164984783, 0.18686099078895035, -0.30357408182462847, 0.9343049539447523, 0.18686099078895035, 0.30357408182462847, 0.9343049539447523, 0.1875924740850802, -0.5773502691896257, -0.794654472291766, 0.1875924740850802, 0.5773502691896257, -0.794654472291766, 0.19002530190186945, -0.9747295721835683, -0.11744209529781557, 0.19002530190186945, 0.9747295721835683, -0.11744209529781557, 0.1941551248152605, -0.14106195520847564, -0.9707756240763016, 0.1941551248152605, 0.14106195520847564, -0.9707756240763016, 0.19569051728621087, -0.9034102251496204, -0.38151695445860245, 0.19569051728621087, 0.9034102251496204, -0.38151695445860245, 0.21356818797909516, -0.9476287910630424, -0.23746221474525606, 0.21356818797909516, 0.9476287910630424, -0.23746221474525606, 0.2199312276977106, -0.6768787187959852, -0.7024709639013442, 0.2199312276977106, 0.6768787187959852, -0.7024709639013442, 0.230972886958105, -0.271524913266781, -0.9343049539447521, 0.230972886958105, 0.271524913266781, -0.9343049539447521, 0.2344310355036339, -0.4750091605513506, 0.8481794544699813, 0.23443103550363395, 0.4750091605513506, 0.8481794544699813, 0.2351501286322781, -0.7237176796562059, 0.6487966839906476, 0.2351501286322781, 0.7237176796562059, 0.6487966839906476, 0.2364219704538591, -0.17177061608955557, 0.9563469597039189, 0.2364219704538591, 0.17177061608955557, 0.9563469597039189, 0.23998893245161798, 0, 0.9707756240763016, 0.24972709424343778, -0.7685809667396409, -0.5889988760314302, 0.24972709424343778, 0.7685809667396409, -0.5889988760314302, 0.26057356349967536, -0.38580222481518056, -0.8850186785219493, 0.26057356349967536, 0.38580222481518056, -0.8850186785219493, 0.27447663717835485, -0.8447522275830474, -0.45940858681215474, 0.27447663717835485, 0.8447522275830474, -0.45940858681215474, 0.27639320225002095, -0.85065080835204, 0.447213595499958, 0.27639320225002095, 0.85065080835204, 0.447213595499958, 0.2832729949983337, -0.48193280435927144, -0.8291544985025904, 0.2832729949983337, 0.48193280435927144, -0.8291544985025904, 0.2846297502225886, -0.585666801594933, -0.7589336616581112, 0.2846297502225886, 0.585666801594933, -0.7589336616581112, 0.28639806050540606, -0.3670396294761699, 0.8850186785219495, 0.2863980605054061, 0.3670396294761699, 0.8850186785219495, 0.2901552060284897, -0.6214809875527959, 0.7277165234656735, 0.2901552060284897, 0.6214809875527959, 0.7277165234656735, 0.2922336268554169, 0, -0.9563469597039189, 0.29251443430176444, -0.9002668588366725, -0.32242036009760555, 0.29251443430176444, 0.9002668588366725, -0.32242036009760555, 0.29409493631196854, -0.9051311438539504, 0.3069947570583225, 0.29409493631196854, 0.9051311438539504, 0.3069947570583225, 0.3035309991033431, -0.9341723589627156, -0.1875924740850802, 0.3035309991033431, 0.9341723589627156, -0.1875924740850802, 0.30471349262049113, -0.9378116997932586, 0.16630966041063486, 0.30471349262049113, 0.9378116997932586, 0.16630966041063486, 0.30790325893376, -0.9476287910630427, -0.08482486360947541, 0.30790325893376, 0.9476287910630427, -0.08482486360947541, 0.3088398261196968, -0.9505112484726589, 0.033857470804871996, 0.3088398261196968, 0.9505112484726589, 0.033857470804871996, 0.3210452589791389, -0.6869389665317207, -0.6519545980720968, 0.32104525897913894, 0.6869389665317207, -0.6519545980720968, 0.32961008540874664, -0.1357624566333905, -0.9343049539447521, 0.32961008540874664, 0.1357624566333905, -0.9343049539447521, 0.3353206898269588, -0.51760800033444, 0.787173419878844, 0.33532068982695884, 0.51760800033444, 0.787173419878844, 0.3370964706396787, -0.24491492196023704, 0.9090559116377184, 0.3370964706396787, 0.24491492196023704, 0.9090559116377184, 0.34076692622105814, -0.7614895288635357, 0.5513724688674999, 0.34076692622105814, 0.7614895288635357, 0.5513724688674999, 0.3464593304371572, -0.08390581259561895, 0.9343049539447521, 0.3464593304371572, 0.08390581259561895, 0.9343049539447521, 0.3513369550338956, -0.776288999232417, -0.523390612925225, 0.3513369550338956, 0.776288999232417, -0.523390612925225, 0.35856659428513216, -0.2605138798701924, -0.89641648571283, 0.35856659428513216, 0.2605138798701924, -0.89641648571283, 0.370809164500308, -0.41833405447740113, 0.8291544985025905, 0.370809164500308, 0.41833405447740113, 0.8291544985025905, 0.3726942529906006, -0.8458982251124604, -0.3815169544586024, 0.37269425299060066, 0.8458982251124604, -0.3815169544586024, 0.3793173834627134, -0.36974306703169685, -0.8481794544699811, 0.3793173834627134, 0.36974306703169685, -0.8481794544699811, 0.3842219345016504, -0.8921800275819267, -0.2374622147452561, 0.3842219345016504, 0.8921800275819267, -0.2374622147452561, 0.38865466988251246, -0.47885859563629063, -0.7871734198788439, 0.38865466988251246, 0.47885859563629063, -0.7871734198788439, 0.39187900917506935, -0.5874482582451998, -0.7080504120842295, 0.39187900917506935, 0.5874482582451998, -0.7080504120842295, 0.39501864964664196, -0.6598810046918482, 0.6391536013183534, 0.39501864964664196, 0.6598810046918482, 0.6391536013183534, 0.4162837922165572, -0.8163557435909476, 0.4003387368765019, 0.4162837922165572, 0.8163557435909476, 0.4003387368765019, 0.41667415268590485, 0, -0.9090559116377184, 0.4191979689030107, -0.9002668588366725, -0.11744209529781557, 0.4191979689030107, 0.9002668588366725, -0.11744209529781557, 0.4218792626337932, -0.3065132259869937, 0.853268732612776, 0.4218792626337932, 0.3065132259869937, 0.853268732612776, 0.4287545977360634, -0.9034102251496204, -0.004411350734000925, 0.4287545977360634, 0.9034102251496204, -0.004411350734000925, 0.42930525344753856, -0.8647796348646817, 0.26048643436012714, 0.42930525344753856, 0.8647796348646817, 0.26048643436012714, 0.4300261059865684, -0.6876704554941067, -0.5849674288458774, 0.4300261059865685, 0.6876704554941067, -0.5849674288458774, 0.4329043006482858, -0.893005900819577, 0.12302165492966766, 0.4329043006482858, 0.893005900819577, 0.12302165492966766, 0.4375772992040651, -0.1589592585807398, 0.8850186785219495, 0.4375772992040651, 0.1589592585807398, 0.8850186785219495, 0.4375992204164231, -0.5542305803889672, 0.7080504120842295, 0.4375992204164231, 0.5542305803889672, 0.7080504120842295, 0.4432126849970109, 0, 0.8964164857128302, 0.44744137931788447, -0.12860074160506021, -0.8850186785219495, 0.44744137931788447, 0.12860074160506021, -0.8850186785219495, 0.457288757127312, -0.7715762680897256, -0.4422183342260814, 0.457288757127312, 0.7715762680897256, -0.4422183342260814, 0.46886207100726823, -0.24649537802113128, -0.8481794544699813, 0.46886207100726823, 0.24649537802113128, -0.8481794544699813, 0.46904679811112565, -0.45167997341470945, 0.7589336616581113, 0.46904679811112565, 0.45167997341470945, 0.7589336616581113, 0.4714566912212952, -0.8323631802054368, -0.2913762593983249, 0.4714566912212953, 0.8323631802054368, -0.2913762593983249, 0.47513906108389614, -0.7237176796562059, 0.5004753668117452, 0.47513906108389614, 0.7237176796562059, 0.5004753668117452, 0.4882972068205223, -0.35476868706133874, -0.7973111164984783, 0.4882972068205223, 0.35476868706133874, -0.7973111164984783, 0.49112347318842303, -0.35682208977308993, 0.7946544722917661, 0.49112347318842303, 0.35682208977308993, 0.7946544722917661, 0.5014006532964664, -0.4680021862650972, -0.7277165234656735, 0.5014006532964664, 0.4680021862650972, -0.7277165234656735, 0.5055166536557194, -0.5795995055195465, -0.6391536013183533, 0.5055166536557195, 0.5795995055195465, -0.6391536013183533, 0.5079130747973478, -0.8458982251124606, -0.1627283048565699, 0.5079130747973478, 0.8458982251124606, -0.1627283048565699, 0.5124456619994747, -0.22338814004068744, 0.8291544985025905, 0.5124456619994747, 0.22338814004068744, 0.8291544985025905, 0.5211280929865946, -0.6214809875527959, 0.5849674288458774, 0.5211280929865946, 0.6214809875527959, 0.5849674288458774, 0.5214714469128553, 0, -0.853268732612776, 0.5242037314217923, -0.07617126084340645, 0.8481794544699813, 0.5242037314217923, 0.07617126084340645, 0.8481794544699813, 0.5336572156169057, -0.8447522275830474, -0.04004560167472076, 0.5336572156169057, 0.8447522275830474, -0.04004560167472076, 0.5411354630743502, -0.7660410837037032, 0.34691997446293676, 0.5411354630743502, 0.7660410837037032, 0.34691997446293676, 0.5414703706287629, -0.6755251623335724, -0.5004753668117451, 0.541470370628763, 0.6755251623335724, -0.5004753668117451, 0.5450795306789344, -0.8345416736796134, 0.08017792792821428, 0.5450795306789344, 0.8345416736796134, 0.08017792792821428, 0.545881503504257, -0.12048320108981789, -0.8291544985025905, 0.545881503504257, 0.12048320108981789, -0.8291544985025905, 0.5483907441273382, -0.809100088223958, 0.21124544726797653, 0.5483907441273382, 0.809100088223958, 0.21124544726797653, 0.5541093394289913, -0.5176080003344399, 0.6519545980720968, 0.5541093394289913, 0.5176080003344399, 0.6519545980720968, 0.5613283100572769, -0.7513701216091022, -0.3469199744629367, 0.5613283100572769, 0.7513701216091022, -0.3469199744629367, 0.574516613769574, -0.3065132259869937, 0.7589336616581113, 0.574516613769574, 0.3065132259869937, 0.7589336616581113, 0.5755224857007215, -0.22165711242617026, -0.7871734198788439, 0.5755224857007215, 0.22165711242617026, -0.7871734198788439, 0.5757874293000979, -0.418334054477401, 0.7024709639013442, 0.5757874293000979, 0.418334054477401, 0.7024709639013442, 0.5925026402964927, -0.6733376315905675, 0.4422183342260814, 0.5925026402964927, 0.6733376315905675, 0.4422183342260814, 0.595894253326634, -0.1589592585807398, 0.787173419878844, 0.595894253326634, 0.1589592585807398, 0.787173419878844, 0.6000378517471079, -0.7715762680897256, -0.21124544726797656, 0.600037851747108, -0.3322397296317068, -0.7277165234656735, 0.600037851747108, 0.3322397296317067, -0.7277165234656734, 0.600037851747108, 0.7715762680897256, -0.21124544726797656, 0.6035685408534394, 0, 0.7973111164984783, 0.6070619982066863, 0, -0.7946544722917661, 0.6156310292182142, -0.44728212428674363, -0.6487966839906476, 0.6156310292182142, 0.44728212428674363, -0.6487966839906476, 0.6189168071929713, -0.5594018111778118, -0.5513724688674999, 0.6189168071929714, 0.5594018111778118, -0.5513724688674999, 0.6252574586071488, -0.776288999232417, -0.08017792792821428, 0.6252574586071488, 0.776288999232417, -0.08017792792821428, 0.6297256213908132, -0.5740277938094211, 0.5233906129252252, 0.6297256213908132, 0.5740277938094211, 0.5233906129252252, 0.6384979619012642, -0.7685809667396409, 0.04004560167472076, 0.6384979619012642, 0.7685809667396409, 0.04004560167472076, 0.644957657958078, -0.08971798394659967, -0.7589336616581112, 0.644957657958078, 0.08971798394659967, -0.7589336616581112, 0.6459362967593755, -0.7055963265124218, 0.2913762593983249, 0.6459362967593755, 0.7055963265124218, 0.2913762593983249, 0.647543247338096, -0.7444509665688808, 0.1627283048565699, 0.647543247338096, 0.7444509665688808, 0.1627283048565699, 0.6477616832793808, -0.6481772114408144, -0.4003387368765019, 0.6477616832793808, 0.6481772114408144, -0.4003387368765019, 0.6537940206410678, -0.4750091605513505, 0.5889988760314302, 0.6537940206410678, 0.4750091605513505, 0.5889988760314302, 0.6623302008428754, -0.24491492196023698, 0.7080504120842295, 0.6623302008428754, 0.24491492196023698, 0.7080504120842295, 0.6635036642300076, -0.36703962947616986, 0.6519545980720968, 0.6635036642300076, 0.36703962947616986, 0.6519545980720968, 0.6797937675642476, -0.19116759016150825, -0.7080504120842296, 0.6797937675642476, 0.19116759016150822, -0.7080504120842295, 0.6807264326348014, -0.08390581259561895, 0.7277165234656735, 0.6807264326348014, 0.08390581259561895, 0.7277165234656735, 0.6893281612357385, -0.6158502249638206, 0.38151695445860245, 0.6893281612357385, 0.6158502249638206, 0.38151695445860245, 0.6897916878076654, -0.6755251623335724, -0.26048643436012703, 0.6897916878076654, 0.6755251623335724, -0.26048643436012703, 0.7074451234849853, -0.3016688104181362, -0.6391536013183534, 0.7074451234849853, 0.30166881041813615, -0.6391536013183533, 0.7117124032047751, 0, -0.7024709639013443, 0.7155242952261603, -0.6876704554941067, -0.12302165492966767, 0.7155242952261603, 0.6876704554941067, -0.12302165492966767, 0.7185891652507057, -0.5220855887185097, 0.45940858681215474, 0.7185891652507057, 0.5220855887185097, 0.45940858681215474, 0.7232785492748799, -0.41576019618739724, -0.5513724688674999, 0.7232785492748799, 0.41576019618739724, -0.5513724688674999, 0.7236067977499789, -0.5257311121191337, -0.447213595499958, 0.723606797749979, 0.5257311121191337, -0.447213595499958, 0.7266653661026955, -0.6768787187959852, 0.11744209529781557, 0.7266653661026955, 0.6768787187959852, 0.11744209529781557, 0.7267017243993803, -0.6869389665317207, 0.004411350734000925, 0.7267017243993803, 0.6869389665317207, 0.004411350734000925, 0.7297825215675532, -0.641115565076049, 0.2374622147452561, 0.7297825215675532, 0.641115565076049, 0.2374622147452561, 0.7405287926400659, -0.4215203121710784, 0.5233906129252252, 0.7405287926400659, 0.4215203121710784, 0.5233906129252252, 0.7496516053274307, -0.17177061608955554, 0.6391536013183533, 0.7496516053274307, 0.17177061608955554, 0.6391536013183533, 0.7521009799446995, -0.3035740818246284, 0.5849674288458774, 0.7521009799446995, 0.3035740818246284, 0.5849674288458774, 0.7525262214051114, -0.0930563708211101, -0.6519545980720968, 0.7525262214051114, 0.0930563708211101, -0.6519545980720968, 0.7609618011718721, 0, 0.6487966839906476, 0.7658127312019672, -0.5563955177061672, 0.32242036009760555, 0.7658127312019672, 0.5563955177061672, 0.32242036009760555, 0.7699505391839694, -0.5594018111778118, -0.3069947570583225, 0.7699505391839695, 0.5594018111778118, -0.3069947570583225, 0.7868988425360585, -0.19647727299831622, -0.5849674288458774, 0.7868988425360585, 0.19647727299831622, -0.5849674288458774, 0.794654472291766, -0.5773502691896257, 0.1875924740850802, 0.794654472291766, 0.5773502691896257, 0.1875924740850802, 0.7977502805111361, 0.5795995055195465, -0.16630966041063483, 0.7977502805111362, -0.5795995055195466, -0.16630966041063486, 0.7987224860367548, -0.46528185410555045, 0.38151695445860245, 0.7987224860367548, 0.46528185410555045, 0.38151695445860245, 0.8061011971354436, -0.585666801594933, 0.08482486360947541, 0.8061011971354436, 0.585666801594933, 0.08482486360947541, 0.8081338527952605, 0, -0.5889988760314302, 0.8085531618609739, 0.5874482582451998, -0.033857470804871996, 0.808553161860974, -0.5874482582451999, -0.033857470804872, 0.8097861540334745, -0.30622016907826805, -0.5004753668117452, 0.8097861540334745, 0.306220169078268, -0.5004753668117451, 0.816622529093059, -0.41576019618739724, -0.4003387368765019, 0.8166225290930591, 0.41576019618739724, -0.4003387368765019, 0.8234755272545975, -0.35543072586240004, 0.4422183342260814, 0.8234755272545975, 0.35543072586240004, 0.4422183342260814, 0.8295223498394129, -0.08877540026300262, 0.5513724688674999, 0.8295223498394129, 0.08877540026300262, 0.5513724688674999, 0.8351224597613233, -0.22824303804682877, 0.5004753668117452, 0.8351224597613233, 0.22824303804682877, 0.5004753668117452, 0.8352523372260013, -0.4959488176483333, 0.2374622147452561, 0.8352523372260013, 0.4959488176483333, 0.2374622147452561, 0.8468638011056542, -0.09425480719114633, -0.5233906129252252, 0.8468638011056542, 0.09425480719114633, -0.5233906129252252, 0.8556199616698322, -0.44728212428674363, -0.2604864343601271, 0.8556199616698322, 0.44728212428674363, -0.2604864343601271, 0.8683018636018623, -0.48193280435927155, 0.1174420952978156, 0.8683018636018623, 0.48193280435927155, 0.1174420952978156, 0.8706672771858278, -0.39628066808369156, 0.2913762593983249, 0.8706672771858278, 0.39628066808369156, 0.2913762593983249, 0.875122634874367, -0.4680021862650972, -0.12302165492966767, 0.875122634874367, -0.19647727299831622, -0.4422183342260814, 0.875122634874367, 0.19647727299831622, -0.4422183342260814, 0.875122634874367, 0.4680021862650972, -0.12302165492966767, 0.8778809630980393, -0.47885859563629063, 0.004411350734000925, 0.8778809630980393, 0.47885859563629063, 0.004411350734000925, 0.8880554375372868, -0.30166881041813615, -0.3469199744629367, 0.8880554375372868, 0.3016688104181361, -0.34691997446293665, 0.8882250561447018, 0, -0.45940858681215474, 0.8882250561447018, 0, -0.45940858681215463, 0.8944271909999159, 0, 0.447213595499958, 0.895768418755139, -0.2779306951014105, 0.3469199744629367, 0.895768418755139, 0.2779306951014105, 0.3469199744629367, 0.905039215834912, -0.14364161499041464, 0.4003387368765019, 0.905039215834912, 0.14364161499041464, 0.4003387368765019, 0.9081168108377711, -0.38580222481518056, 0.1627283048565699, 0.9081168108377711, 0.38580222481518056, 0.1627283048565699, 0.9192345310435216, -0.3322397296317068, -0.21124544726797656, 0.9192345310435216, 0.3322397296317068, -0.21124544726797656, 0.9196658769956818, -0.0930563708211101, -0.38151695445860245, 0.9196658769956819, 0.0930563708211101, -0.38151695445860245, 0.9282706578194226, -0.36974306703169696, 0.040045601674720765, 0.9282706578194226, 0.36974306703169696, 0.040045601674720765, 0.931509891817533, 0.35476868706133874, -0.0801779279282143, 0.9315098918175331, -0.3547686870613388, -0.08017792792821432, 0.9373125561577, -0.19116759016150822, -0.2913762593983249, 0.9373125561577, 0.19116759016150822, -0.2913762593983249, 0.9389619707336497, -0.271524913266781, 0.21124544726797656, 0.9389619707336497, 0.271524913266781, 0.21124544726797656, 0.9465965938004058, 0, -0.32242036009760555, 0.9517112057440018, 0, 0.3069947570583225, 0.9551169259871324, -0.14106195520847561, 0.2604864343601271, 0.9551169259871324, 0.14106195520847561, 0.2604864343601271, 0.9614507908933246, -0.22165711242617028, -0.16272830485656992, 0.9614507908933246, 0.22165711242617028, -0.16272830485656992, 0.9621351351385714, -0.2605138798701924, 0.0801779279282143, 0.9621351351385714, 0.2605138798701924, 0.0801779279282143, 0.967244736312809, -0.08971798394659967, -0.2374622147452561, 0.967244736312809, 0.08971798394659967, -0.2374622147452561, 0.9683162594941432, -0.24649537802113122, -0.04004560167472076, 0.9683162594941432, 0.24649537802113122, -0.04004560167472076, 0.9822469463768458, 0, -0.1875924740850802, 0.9830738669028043, -0.13576245663339054, 0.1230216549296677, 0.9830738669028043, 0.13576245663339054, 0.1230216549296677, 0.9857439588996775, -0.12048320108981783, -0.11744209529781555, 0.9857439588996775, 0.12048320108981783, -0.11744209529781555, 0.98607357578129, 0, 0.16630966041063486, 0.9916866386330564, -0.12860074160506021, -0.004411350734000926, 0.9916866386330564, 0.12860074160506021, -0.004411350734000926, 0.9963958764033671, 0, -0.08482486360947544, 0.9994266714825543, 0, 0.033857470804872};


float* get_dti_dirarr(unsigned int ndir) {
  if(ndir==3) return ps3;
  if(ndir==4) return ps4;
  if(ndir==5) return ps5;
  if(ndir==6) return ps6;
  if(ndir==7) return ps7;
  if(ndir==8) return ps8;
  if(ndir==9) return ps9;
  if(ndir==10) return ps10;
  if(ndir==11) return ps11;
  if(ndir==12) return ps12;
  if(ndir==13) return ps13;
  if(ndir==14) return ps14;
  if(ndir==15) return ps15;
  if(ndir==16) return ps16;
  if(ndir==17) return ps17;
  if(ndir==18) return ps18;
  if(ndir==19) return ps19;
  if(ndir==20) return ps20;
  if(ndir==21) return ps21;
  if(ndir==22) return ps22;
  if(ndir==23) return ps23;
  if(ndir==24) return ps24;
  if(ndir==25) return ps25;
  if(ndir==26) return ps26;
  if(ndir==27) return ps27;
  if(ndir==28) return ps28;
  if(ndir==29) return ps29;
  if(ndir==30) return ps30;
  if(ndir==31) return ps31;
  if(ndir==32) return ps32;
  if(ndir==33) return ps33;
  if(ndir==34) return ps34;
  if(ndir==35) return ps35;
  if(ndir==36) return ps36;
  if(ndir==37) return ps37;
  if(ndir==38) return ps38;
  if(ndir==39) return ps39;
  if(ndir==40) return ps40;
  if(ndir==41) return ps41;
  if(ndir==42) return ps42;
  if(ndir==43) return ps43;
  if(ndir==44) return ps44;
  if(ndir==45) return ps45;
  if(ndir==46) return ps46;
  if(ndir==47) return ps47;
  if(ndir==48) return ps48;
  if(ndir==49) return ps49;
  if(ndir==50) return ps50;
  if(ndir==51) return ps51;
  if(ndir==52) return ps52;
  if(ndir==53) return ps53;
  if(ndir==54) return ps54;
  if(ndir==55) return ps55;
  if(ndir==56) return ps56;
  if(ndir==57) return ps57;
  if(ndir==58) return ps58;
  if(ndir==59) return ps59;
  if(ndir==60) return ps60;
  if(ndir==61) return ps61;
  if(ndir==62) return ps62;
  if(ndir==63) return ps63;
  if(ndir==64) return ps64;
  if(ndir==65) return ps65;
  if(ndir==66) return ps66;
  if(ndir==67) return ps67;
  if(ndir==68) return ps68;
  if(ndir==69) return ps69;
  if(ndir==70) return ps70;
  if(ndir==71) return ps71;
  if(ndir==72) return ps72;
  if(ndir==73) return ps73;
  if(ndir==74) return ps74;
  if(ndir==75) return ps75;
  if(ndir==76) return ps76;
  if(ndir==77) return ps77;
  if(ndir==78) return ps78;
  if(ndir==79) return ps79;
  if(ndir==80) return ps80;
  if(ndir==81) return ps81;
  if(ndir==82) return ps82;
  if(ndir==83) return ps83;
  if(ndir==84) return ps84;
  if(ndir==85) return ps85;
  if(ndir==86) return ps86;
  if(ndir==87) return ps87;
  if(ndir==88) return ps88;
  if(ndir==89) return ps89;
  if(ndir==90) return ps90;
  if(ndir==91) return ps91;
  if(ndir==92) return ps92;
  if(ndir==93) return ps93;
  if(ndir==94) return ps94;
  if(ndir==95) return ps95;
  if(ndir==96) return ps96;
  if(ndir==97) return ps97;
  if(ndir==98) return ps98;
  if(ndir==99) return ps99;
  if(ndir==100) return ps100;
  if(ndir==101) return ps101;
  if(ndir==102) return ps102;
  if(ndir==103) return ps103;
  if(ndir==104) return ps104;
  if(ndir==105) return ps105;
  if(ndir==106) return ps106;
  if(ndir==107) return ps107;
  if(ndir==108) return ps108;
  if(ndir==109) return ps109;
  if(ndir==110) return ps110;
  if(ndir==111) return ps111;
  if(ndir==112) return ps112;
  if(ndir==113) return ps113;
  if(ndir==114) return ps114;
  if(ndir==115) return ps115;
  if(ndir==116) return ps116;
  if(ndir==117) return ps117;
  if(ndir==118) return ps118;
  if(ndir==119) return ps119;
  if(ndir==120) return ps120;
  if(ndir==121) return ps121;
  if(ndir==122) return ps122;
  if(ndir==123) return ps123;
  if(ndir==124) return ps124;
  if(ndir==125) return ps125;
  if(ndir==126) return ps126;
  if(ndir==127) return ps127;
  if(ndir==128) return ps128;
  if(ndir==129) return ps129;
  if(ndir==130) return ps130;
  if(ndir==131) return ps131;
  if(ndir==132) return ps132;
  if(ndir==133) return ps133;
  if(ndir==134) return ps134;
  if(ndir==135) return ps135;
  if(ndir==136) return ps136;
  if(ndir==137) return ps137;
  if(ndir==138) return ps138;
  if(ndir==139) return ps139;
  if(ndir==140) return ps140;
  if(ndir==141) return ps141;
  if(ndir==142) return ps142;
  if(ndir==143) return ps143;
  if(ndir==144) return ps144;
  if(ndir==145) return ps145;
  if(ndir==146) return ps146;
  if(ndir==147) return ps147;
  if(ndir==148) return ps148;
  if(ndir==149) return ps149;
  if(ndir==150) return ps150;
//  if(ndir==246) return ps246;
//  if(ndir==256) return ps256;
//  if(ndir==755) return ps755;
  return 0;
}


