/*	Conductor_Table_Model

PIRL CVS ID: Conductor_Table_Model.java,v 1.42 2012/04/16 06:04:11 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.Conductor.Maestro;

import	PIRL.Conductor.Conductor;
import	PIRL.Conductor.Management;
import	PIRL.Conductor.Manager;
import	PIRL.Conductor.Processing_Listener;
import	PIRL.Conductor.Processing_Event;
import	PIRL.Messenger.Message;

import	javax.swing.table.AbstractTableModel;
import	javax.swing.event.TableModelListener;
import	javax.swing.event.TableModelEvent;
import	javax.swing.SwingUtilities;
import	java.io.IOException;
import	java.util.List;
import	java.util.Vector;


/**	A <i>Conductor_Table_Model<i> contains a table of Conductor
	identities each with its associated Remote_Theater Management and
	Manager.
<p>
	A Conductor is represented by its {@link Conductor#Identity()
	identity Message} which is supplemented by its Theater location plus
	any additional parameters that the application might find useful. The
	data model maintains a list of all known Conductor identities.
<p>
	When a Conductor identity is added to the data model Remote_Theater
	Management is provided for it that uses a specified Theater. The data
	model will register itself with the Remote_Theater Management as a
	Processing_Listener. This is used to maintain the current Conductor
	processing state in each identity and to notify the listeners of this
	TableModel of state changes in addition to the usual data model
	content changes. Also, any TableModelListener may register with this
	data model to only receive notices of Conductor processing state
	changes.
<p>
	A list of Managers associate with each Conductor is also maintained.
	A Condcutor is not required to have a Manager.
<p>
	The TableModel interface that is implemented presents Theater
	locations in the first column and Conductor names in the second
	column.
<p>
	@author		Bradford Castalia, UA/PIRL
	@version	1.42
*/
public class Conductor_Table_Model
	extends AbstractTableModel
	implements Processing_Listener
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Conductor.Maestro.Conductor_Table_Model (1.42 2012/04/16 06:04:11)";

/**	The names of the table columns.
*/
public static String[]
	Column_Names					= {"Theaters", "Conductors"};

/**	The index of the Theaters column in the data model.
*/
public static final int
	THEATERS_COLUMN					= 0;

/**	The index of the Conductos column in the data model.
*/
public static final int
	CONDUCTORS_COLUMN				= 1;

/**	Conductor identity parameters of interest to the data model.
*/
public static final String

	//	Theater location.
	THEATER_LOCATION_PARAMETER_NAME
		= Kapellmeister.THEATER_LOCATION_PARAMETER_NAME,
	ADDRESS_PARAMETER_NAME
		= Kapellmeister.ADDRESS_PARAMETER_NAME,
	CONDUCTOR_ID_PARAMETER_NAME
		= Conductor.CONDUCTOR_ID_PARAMETER,

	//	Pipeline name.
	PIPELINE_PARAMETER_NAME
		= Conductor_Definition.PIPELINE_PARAMETER_NAME,

	//	Processing state.
	PROCESSING_STATE_PARAMETER_NAME
		= Theater.PROCESSING_STATE_ACTION;

private Vector<Message>
	Identities						= new Vector<Message> ();
private Vector<Remote_Theater>
	Managements						= new Vector<Remote_Theater> ();
private Vector<Manager>
	Managers						= new Vector<Manager> ();

private Vector<TableModelListener>
	State_Change_Listeners			= new Vector<TableModelListener> ();

private static final String
	NL								= System.getProperty ("line.separator");


//  DEBUG control.
private static final int
	DEBUG_OFF					= 0,
	DEBUG_CONSTRUCTOR			= 1 << 0,
	DEBUG_ACCESSORS				= 1 << 1,
	DEBUG_SET_IDENTITY			= 1 << 2,
	DEBUG_PROFILE				= 1 << 3,
	DEBUG_REPLACE				= 1 << 4,
	DEBUG_OPEN_THEATER			= 1 << 5,
	DEBUG_MANAGEMENT			= 1 << 6,
	DEBUG_LISTENER				= 1 << 7,
	DEBUG_ALL					= -1,

	DEBUG						= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Construct an empty Conductor_Table_Model.
*/
public Conductor_Table_Model ()
{}

/*==============================================================================
	AbstractTableModel
*/
/**	Find the data model column index of a name.
<p>
	@param	name	A column name String. May be null.
	@return	The index of the named column in the data model. This will be
		-1 if the name does not match (case insensitive) a column name.
*/
public int findColumn
	(
	String	name
	)
{
if (name != null)
	{
	for (int
			column = 0;
			column < Column_Names.length;
			column++)
		if (name.equalsIgnoreCase (Column_Names[column]))
			return column;
	}
return -1;
}

/*==============================================================================
	TableModel
*/
/**	Get the Class for a data model column.
<p>
	@param	column	A data model column index.
	@return	If the column index is valid String.class is
		returned; otherwise Object.class is returned.
*/
public Class getColumnClass
	(
	int		column
	)
{
if (column >= 0 &&
	column < Column_Names.length)
	return String.class;
return Object.class;
}

/**	Get the number of columns in the data model.
<p>
	@return	The size of the {@link #Column_Names} array.
*/
public int getColumnCount ()
{return Column_Names.length;}

/**	Get the name of a data model column.
<p>
	@param	column	A data model column index.
	@return	The column name String. This will be null.if the column index
		is not valid.
*/
public String getColumnName
	(
	int		column
	)
{
if (column >= 0 &&
	column < Column_Names.length)
	return Column_Names[column];
return null;
}

/**	Get the number of rows in the data model.
<p>
	@return	The number of Conductor identities contained in the model.
*/
public int getRowCount ()
{return Identities.size ();}

/**	Get the value of a data model cell.
<p>
	For the {@link #THEATERS_COLUMN} the value will be the
	{@link #Location(int) Theater location}. For the {@link #CONDUCTORS_COLUMN}
	the value will be the {@link #Conductor_Name(int) Conductor name}.
<p>
	@param	row	A data model row index.
	@param	column	A data model column index.
	@return	The String value of the data model cell.  For the {@link
		#THEATERS_COLUMN} the value will be the {@link
		Theater#Location(String) abbreviated location} of the {@link
		#Location(int) Theater location} for the row. For the {@link
		#CONDUCTORS_COLUMN} the value will be the {@link
		#Conductor_Name(int) Conductor name} of the row. This will be
		null if either of the data model index values are invalid.
*/
public Object getValueAt
	(
	int		row,
	int		column
	)
{
if (column >= 0 &&
	column < getColumnCount ())
	{
	switch (column)
		{
		case THEATERS_COLUMN:
			return Theater.Location (Location (row));
		case CONDUCTORS_COLUMN:
			return Conductor_Name (row);
		}
	}
return null;
}

/**	Test if a data model cell is editable.
<p>
	@param	row	A data model row index.
	@param	column	A data model column index.
	@return	Always returns false.
*/
public boolean isCellEditable
	(
	int		row,
	int		column
	)
{return false;}

/*==============================================================================
	Accessors
*/
/**	Get a Profile of the Conductors.
<p>
	@return	A Profile representing all the Conductors on all the Theaters
		listed in this data model.
	@throws	If a Conductor identity can not be used to construct a valid
		Conductor_Definition or the definition is in conflict with another
		of the same name.
*/
public Profile Profile ()
	throws IllegalArgumentException
{
if ((DEBUG & DEBUG_PROFILE) != 0)
	System.out.println
		(">>> Conductor_Table_Model.Profile");
Profile
	profile = new Profile ();

int
	index = -1,
	size = Identities.size ();
while (++index < size)
	{
	Message
		identity = Identities.get (index);
	if ((DEBUG & DEBUG_PROFILE) != 0)
		System.out.println
			("    Adding Conductor identity for Theater "
				+ Theater_Location (identity) + NL
			+ identity);
	profile.Add (Theater_Location (identity), identity);
	}
if ((DEBUG & DEBUG_PROFILE) != 0)
	System.out.println
		(profile + NL
		+"<<< Conductor_Table_Model.Profile");
return profile;
}

/*------------------------------------------------------------------------------
	Identities
*/
/**	Add a Conductor identity to the data model.
<p>
	The data model listeners are notified that a new row has been
	inserted.
<p>
	<b>N.B.</b>: If {@link #Management(Message, Theater, Message)
	Remote_Theater Management} could not be obtained the identity is not
	added to the the data model and no listener notification is sent; an
	IOException is thrown in this case.
<p>
	@param	identity	A Conductor identity Message. If null, or the
		identity is already {@link #Index(Message) present} in the data
		model, nothing is done.
	@param	theater	The Theater where the Stage_Manager is located from
		which the Conductor identities were obtained. If null, nothing
		is done.
	@param	client_identity	The Message containing client authentication
		information for connecting to the Stage_Manager. If null, nothing
		is done.
	@return	This Conductor_Table_Model.
	@throws	IOException	If {@link #Management(Message, Theater, Message)
		Remote_Theater Management} could not be obtained.
	@see	#Set_Identity(Message, Theater, Message)
*/
public Conductor_Table_Model Identity
	(
	Message			identity,
	Theater			theater,
	Message			client_identity
	)
	throws IOException
{
if ((DEBUG & DEBUG_SET_IDENTITY) != 0)
	System.out.println
		(">>> Conductor_Table_Model.Identity");
if (theater != null &&
	identity != null &&
	Index (identity) < 0)
	{
	if (Set_Identity (identity, theater, client_identity))
		Notify_Inserted (getRowCount () - 1);
	}
if ((DEBUG & DEBUG_SET_IDENTITY) != 0)
	System.out.println
		("<<< Conductor_Table_Model.Identity");
return this;
}

/**	Set a Conductor identity.
<p>
	<b>N.B.</b>: This method should only be used to set a new Conductor
	identity by a method that does the neccessary state checks and table
	view update notification.
<p>
	The {@link #THEATER_LOCATION_PARAMETER_NAME} is set at the beginning
	of the identity with the {@link Theater#Location() Remote_Theater
	location} before the identity is added to the data model along with
	null Managements and Managers entries. The Theater location in the
	identity is necessary to correctly {@link #Matching_Index(Message)
	match} identities. This is particularly critical when identities are
	{@link #Replace(List, Theater, Message) replaced} which
	depends on recognizing previously added identities by matching
	Theater locations.
<p>
	The {@link Message#Name(String) name} of the identity is set to the
	identity's {@link #Conductor_Name(Message) Conductor name}.
<p>
	The identity is added to the data model along with null place holders
	for the Remote_Theater Management and Conductor Manager. Then the new
	{@link #Management(Message, Theater, Message) Management} is obtained
	and {@link #Set_Management(int, Remote_Theater) set} in the data
	model.
<p>
	<b>N.B.</b>: If {@link #Management(Message, Theater, Message)
	Remote_Theater Management} could not be obtained the tentative new
	data model row is removed and an IOException is thrown.
<p>
	<b>N.B.</b>: No data model listener notification is done.
<p>
	@param	identity	A Conductor identity Message.<b>N.B.</b>:  Must
		not be null; there should be no null identities in the data
		model. The identity must also be unique (not already present) in
		the data model.
	@param	theater	The Theater where the Stage_Manager is located from
		which the Conductor identities were obtained. Must not be null.
	@param	client_identity	The Message containing client authentication
		information for connecting to the Stage_Manager. Should not be
		null.
	@return	true if the identity was set; false if the Theater location
		is invalid and not entered in the data model.
	@throws	IOException	If {@link #Management(Message, Theater, Message)
		Remote_Theater Management} could not be obtained.
	@see	#Management(Message, Theater, Message)
*/
protected boolean Set_Identity
	(
	Message			identity,
	Theater			theater,
	Message			client_identity
	)
	throws IOException
{
if ((DEBUG & DEBUG_SET_IDENTITY) != 0)
	System.out.println
		(">>> Conductor_Table_Model.Set_Identity:" + NL
		+"    identity -" + NL
		+ identity + NL
		+"    theater -" + NL
		+ theater + NL
		+"    client_identity -" + NL
		+ client_identity);
/*	>>> WARNING <<<

	The Theater location in the identity is necessary to associate
	identities with subsequent searches by Theater. This is particularly
	critical for the Replace method which will not be able to recognize
	previously added identities without a matching Theater location in
	their identities.
*/
String
	location = theater.Location ();
if (Theater.Port (location) <= 0)
	{
	//	Invalid Theater location.
	if ((DEBUG & DEBUG_SET_IDENTITY) != 0)
		System.out.println
			("    Invalid Theater location: " + location + NL
			+"<<< Conductor_Table_Model.Set_Identity: true");
	return false;
	}

identity
	.Set (THEATER_LOCATION_PARAMETER_NAME, location, 0);
Identities.add (identity);

/*
	The Conductor identity name is used a reference.

	N.B.: It should uniquely identify all match Conductor_Definitions.
*/
identity.Name (Conductor_Name (identity));

//	Place holders.
Managements.add (null);
Managers.add (null);

if ((DEBUG & DEBUG_SET_IDENTITY) != 0)
	System.out.println
		("    Setting the Management for the Identity");
IOException
	exception = null;
try {Set_Management (Identities.size () - 1,
		Management (identity, theater, client_identity));}
catch (IOException except)
	{exception = except;}
catch (IllegalArgumentException except)
	{
	exception = new IOException (except.getMessage ());
	exception.initCause (except);
	}
if (exception != null)
	{
	//	Backout the identity that failed to obtain a Management.
	if ((DEBUG & DEBUG_SET_IDENTITY) != 0)
		System.out.println
			("   Set_Identity failed to obtain Management -" + NL
			+ exception); 
	int
		row = Identities.size () - 1;
	Identities.remove (row);
	Managements.remove (row);
	Managers.remove (row);

	throw exception;
	}
if ((DEBUG & DEBUG_SET_IDENTITY) != 0)
	System.out.println
		("<<< Conductor_Table_Model.Set_Identity: true");
return true;
}

/**	Get the Identity of a Conductor at a row index.
<p>
	@param	row	A data model row index.
	@return	A Conductor identity Message. This will be null if the row
		index is less than zero or greater than or equal to the {@link
		#getRowCount() total data model rows}.
*/
public Message Identity
	(
	int		row
	)
{
if (row >= 0 &&
	row < getRowCount ())
	return Identities.get (row);
return null;
}

/**	Get the Identity of a Conductor associated with a Remote_Theater
	Management.
<p>
	@param	management	A Remote_Theater Management reference.
	@return	A Conductor identity Message. This will be null if the
		management could not be {@link #Index(Remote_Theater) found} in
		the data model.
*/
public Message Identity
	(
	Remote_Theater	management
	)
{return Identity (Index (management));}

/**	Get the Identity of a Conductor associated with a Manager.
<p>
	@param	manager	A Manager reference.
	@return	A Conductor identity Message. This will be null if the
		manager could not be {@link #Index(Manager) found} in the
		data model.
*/
public Message Identity
	(
	Manager		manager
	)
{return Identity (Index (manager));}

/**	Get the data model row index of a Conductor identity.
<p>
	@param	identity	A Conductor identity Message. May be null.
	@return	The data model index of the Conductor identity, or -1 if the
		identity is not present.
*/
public int Index
	(
	Message	identity
	)
{
int
	row = -1,
	rows = getRowCount ();
while (++row < rows)
	if (identity == Identities.get (row))
		return row;
return -1;
}

/**	Get the data model row index of a Conductor with a matching identity.
<p>
	<b>N.B.</b>: This method searches the data model for a Conductor
	identity that {@link #Matching_Index(Message, List) matches} the
	specified identity. The matching identity need not be the
	{@link #Index(Message) same} object.
<p>
	@param	identity	A Conductor identity Message.
	@return	The data model row index of the matching identity. This will
		be -1 if a matching identity could not be  found in the data
		model.
	@see	#Matching_Index(Message, List)
*/
public int Matching_Index
	(
	Message	identity
	)
{return Matching_Index (identity, Identities);}

/**	Get the index of an identity Message in a list of identities.
<p>
	The index of the first identities entry that matches the specified
	identity is returned. An identity matches if either it has the same
	{@link #ADDRESS_PARAMETER_NAME}, or all the values of the {@link
	#THEATER_LOCATION_PARAMETER_NAME}, {@link
	#CONDUCTOR_ID_PARAMETER_NAME} and {@link #PIPELINE_PARAMETER_NAME}
	parameters are the same.
<p>
	@param	identity	A Conductor {@link Conductor#Identity() identity}
		Message supplemented with a {@link #THEATER_LOCATION_PARAMETER_NAME}
		parameter. If null, -1 is returned.
	@param	identities	A List of Conductor identities. If null, -1 is
		returned.
	@return	The index of the entry in the identities list that matches
		the identity, or -1 if no match is found.
*/
protected static int Matching_Index
	(
	Message			identity,
	List<Message>	identities
	)
{
if (identity == null ||
	identities == null ||
	identities.isEmpty ())
	return -1;
String
	address = identity.Get (ADDRESS_PARAMETER_NAME);
if (address != null)
	for (int
			index = 0;
			index < identities.size ();
			index++)
		if (address.equals (((Message)identities.get (index))
				.Get (ADDRESS_PARAMETER_NAME)))
			return index;

String
	theater_location = identity.Get (THEATER_LOCATION_PARAMETER_NAME),
	conductor_ID     = identity.Get (CONDUCTOR_ID_PARAMETER_NAME),
	pipeline         = identity.Get (PIPELINE_PARAMETER_NAME);
if (theater_location == null ||
	conductor_ID == null ||
	pipeline == null)
	return -1;

int
	index = identities.size ();
while (--index >= 0)
	{
	identity = identities.get (index);
	if (theater_location
			.equals (identity.Get (THEATER_LOCATION_PARAMETER_NAME)) &&
		conductor_ID
			.equals (identity.Get (CONDUCTOR_ID_PARAMETER_NAME)) &&
		pipeline
			.equals (identity.Get (PIPELINE_PARAMETER_NAME)))
		return index;
	}
return -1;
}

/**	Replace all identities for a specified Remote_Theater Management
	with new identities.
<p>
	Each current identity in the data model that has a {@link
	#THEATER_LOCATION_PARAMETER_NAME} value that matches the {@link
	Theater#Location() Theater location} is compared against
	the new identities list. An attempt is made to get the {@link
	#Matching_Index(Message, List) index} of the existing
	identity in the new identities list. If a matching identity can not
	be found the data model row for the current identity is removed; the
	current identity is defunct. If a matching identity is found the
	identity is removed from the list of new identities; the identity is
	redundant. This checking removes all current identities that are not
	in the new identities list, and removes from the new identities list
	all identities that are already in the current identitites list.
	After all current identities have been checked all the remaining new
	{@link #Set_Identity(Message, Theater, Message) identities are
	added} to the data model along with the Remote_Theater Management
	and an equal number of null Manager entries.
<p>
	A {@link #Notify_Table_Changed() table data changed notification} is
	sent to the data model listeners only if any changes occured to the
	data model.
<p>
	<b>N.B.</b>: If {@link #Management(Message, Theater, Message)
	Remote_Theater Management} could not be obtained for an identity
	it is not added to the the data model. In this case, after an attempt
	has been made to add all the new identities to the model an IOException
	will thrown. Because multiple exceptions might have occured a generic
	IOException is thrown containing a message that lists each specific
	exception message, including its specific type, with each exception
	message separated from the one that prceeded it by a single line
	containing seven dash ('-') characters.
<p>
	@param	identities	A List of Conductor {@link
		Conductor#Identity() identity} Messages. If null, nothing is
		done.
	@param	theater	The Theater where the Stage_Manager is located from
		which the Conductor identities were obtained. If null, nothing
		is done.
	@param	client_identity	The Message containing client authentication
		information for connecting to the Stage_Manager. If null, nothing
		is done.
	@throws	IOException	If {@link #Management(Message, Theater, Message)
		Remote_Theater Management} could not be obtained.
	@see	#Matching_Index(Message, List)
	@see	#Set_Identity(Message, Theater, Message)
*/
public Conductor_Table_Model Replace
	(
	List<Message>	identities,
	Theater			theater,
	Message			client_identity
	)
	throws IOException
{
if (identities == null ||
	theater == null)
	return this;
if ((DEBUG & DEBUG_REPLACE) != 0)
	System.out.println
		(">>> Conductor_Table_Model.Replace:" + NL
		+"    " + identities.size () + " Identities -" + NL
		+ identities + NL
		+"    theater -" + NL
		+ theater + NL
		+"    client_identity -" + NL
		+ client_identity);

String
	theater_location = theater.Location ();
boolean
	changed = false;
int
	index,
	row = getRowCount ();
Message
	identity;
while (--row >= 0)
	{
	identity = Identities.get (row);
	if (! theater_location
			.equals (identity.Get (THEATER_LOCATION_PARAMETER_NAME)))
		continue;
	if ((DEBUG & DEBUG_REPLACE) != 0)
		System.out.println
			(NL
			+"    Identity at row " + row + " -" + NL
			+ identity);

	//	Find the matching current identity in the new identities list.
	index = Matching_Index (identity, identities);
	if (index < 0)
		{
		//	Defunct: Current identity is not in the new list.
		if ((DEBUG & DEBUG_REPLACE) != 0)
			System.out.println
				("    Obsolete");
		Remove_at (row);
		changed = true;
		}
	else
		{
		//	Redundant: Current identity is in the new list.
		if ((DEBUG & DEBUG_REPLACE) != 0)
			System.out.println
				("    Redundant");
		identities.remove (index);
		while ((index = Matching_Index (identity, identities)) >= 0)
			identities.remove (index);
		}
	}

//	What's left are new identities.
String
	report = null;

if (identities.size () > 0)
	{
	if ((DEBUG & DEBUG_REPLACE) != 0)
		System.out.println
			(NL + "    Adding " + identities.size () + " Identities");
	index = -1;
	row = identities.size ();
	while (++index < row)
		{
		if ((DEBUG & DEBUG_REPLACE) != 0)
			System.out.println
				(NL
				+"    Conductor_Table_Model.Replace: Identity -" + NL
				+ identities.get (index));
		try
			{
			if (Set_Identity
					(identities.get (index), theater, client_identity))
				changed = true;
			}
		catch (IOException exception)
			{
			//	Accumulate exception reports
			if ((DEBUG & DEBUG_REPLACE) != 0)
				System.out.println
					("    IOException -" + NL
					+ exception);
			if (report == null)
				report = exception.toString ();
			else
				report += NL
					+ "--------" + NL
					+ exception;
			report += NL
				+ "For identity -" + NL
				+ identities.get (index);
			}
		}
	}

if ((DEBUG & DEBUG_REPLACE) != 0)
	System.out.println
		("    Conductor_Table_Model.Replace: changed = " + changed);
if (changed)
	Notify_Table_Changed ();

if (report != null)
	throw new IOException (report);

if ((DEBUG & DEBUG_REPLACE) != 0)
	System.out.println
		("<<< Conductor_Table_Model.Replace");
return this;
}

/**	Get the name of a Conductor at a data model index.
<p>
	@param	row	A data model row index.
	@return	The {@link #Conductor_Name(Message) Conductor name} from
		the {@link #Identity(int) identity at the row index}. This will
		be null if the row index is invalid or the identity does not
		contain the required information.
*/
public String Conductor_Name
	(
	int		row
	)
{return Conductor_Name (Identity (row));}

/**	Get the Theater location of a Conductor at a data model index.
<p>
	The fully qualified Theater location is returned.
<p>
	@param	row	A data model row index.
	@return	The {@link #Theater_Location(Message) Theater location} from
		the {@link #Identity(int) identity at the row index}. This will
		be null if the row index is invalid or the identity does not
		contain the required information.
*/
public String Location
	(
	int		row
	)
{return Theater_Location (Identity (row));}

/**	Get the processing state of a Conductor at a data model index.
<p>
	If the {@link #Identity(int) row's Conductor identity} has a {@link
	#Processing_State(Message) procesing state} of zero an attempt is
	made to use the {@link #Management(int) row's Management} to {@link
	Management#Processing_State() obtain the current procesing state}. If
	this is successful the identity is updated with the current
	processing state.
<p>
	@param	row	A Conductor row index.
	@return	The Conductor {@link #Processing_State(Message) processing
		state}. This will be zero if the row index is outside the range
		of available Conductors or the processing state has not yet been
		obtained from the Conductor.
*/
public int Processing_State
	(
	int		row
	)
{
int
	processing_state = 0;
Message
	identity = Identity (row);
if (identity != null &&
	(processing_state = Processing_State (identity)) == 0)
	{
	Remote_Theater
		management = Management (row);
	if (management != null &&
		management.Opened ())
		{
		try
			{
			processing_state = management.Processing_State ();
			Processing_State (identity, processing_state);
			}
		catch (Remote_Management_Exception exception) {}
		}
	}
return processing_state;
}

/*------------------------------------------------------------------------------
	Managements
*/
/**	Get the Remote_Theater Management at a data model row index.
<p>
	<b>N.B.</b>: The current Remote_Theater Management data model row is
	obtained as-is; Remote_Theater Management will not be {@link
	#Open_Management(Message, Theater, Message) opened or re-opened}.
<p>
	@param	row	A data model row index.
	@return	The Remote_Theater Management for the data model row. This
		will be null if the row index is not valid or no Remote_Theater
		Management is present at the index.
*/
public Remote_Theater Management
	(
	int		row
	)
{
if (row >= 0 &&
	row < getRowCount ())
	return Managements.get (row);
return null;
}

/**	Get the Remote_Theater Management for a Conductor.
<p>
	<b>N.B.</b>: The current Remote_Theater Management associated with
	the Conductor identity is obtained as-is; Remote_Theater Management
	will not be {@link #Open_Management(Message, Theater, Message) opened
	or re-opened}.
<p>
	@param	identity	A Conductor identity Message.
	@return	The Remote_Theater Management associated with the Conductor
		identity. This will be null if the Conductor identity is null or
		not found in the data model.
	@see	#Index(Message)
	@see	#Management(int)
*/
public Remote_Theater Management
	(
	Message	identity
	)
{return Management (Index (identity));}

/**	Get Remote_Theater Management for a Conductor.
<p>
	If the Conductor already has Remote_Theater Management and it is
	{@link Remote_Theater#Opened() open}, that is returned. Otherwise an
	attempt is made to obtain it.
<p>
	Remote_Theater Management can be obtained when the Theater and a
	client identity containing any authentication information required
	by the Stage_Manager are provided. If Remote_Theater Management was
	found for the Conductor but it is closed, it is re-opened at the
	same location. Otherwise a {@link #Management(Message, Theater,
	Message) new Remote_Theater Management} is constructed and {@link
	#Management(int, Remote_Theater) registered}.
<p>
	@param	conductor_identity	The identity Message of the Conductor to
		be opened. If Remote_Theater Management is to be constructed or
		re-opened this must not be null and must contain an {@link
		#ADDRESS_PARAMETER_NAME} parameter.
	@param	theater	The Theater where the Stage_Manager is located from
		which the Conductor identity was obtained. If Remote_Theater
		Management is to be constructed or re-opened this must not be
		null.
	@param	client_identity	The Message containing the {@link
		Theater#KEY_PARAMETER_NAME} that may be needed by the
		Stage_Manager to construct or re-open Remote_Theater Management.
		<b>N.B.</b>: This may be null if the Stage_Manager allows
		unauthenticated connections; otherwise the Stage_Manager will
		reject the connection.
	@return	An open Remote_Theater Management for the Conductor. This
		will be null if the identity could not be found in the data
		model or the theater or client_identity is null and a
		Remote_Theater Management was needed to be constructed or
		re-opened
	@throws	IOException	If a connection could not be established to the
		Stage_Manager. This will be a Theater_Protocol_Exception if
		there were any problems in the content of the Messages used to
		establish the connection.
*/
public Remote_Theater Open_Management
	(
	Message		conductor_identity,
	Theater		theater,
	Message		client_identity
	)
	throws IOException
{
if ((DEBUG & DEBUG_MANAGEMENT) != 0)
	System.out.println
		(">>> Conductor_Table_Model.Open_Management:" + NL
		+"    Identity -" + NL
		+ conductor_identity + NL
		+"    Theater -" + NL
		+ theater);
Remote_Theater
	management = null;
int
	row = Index (conductor_identity);
if (row >= 0)
	{
	management = Managements.get (row);
	if (management == null ||
		! management.Opened ())
		{
		if (theater == null ||
			client_identity == null)
			management = null;
		else
			{
			if (management == null)
				{
				//	Open with new Managment.
				if ((DEBUG & DEBUG_MANAGEMENT) != 0)
					System.out.println
						("    Opening new management");
				management = Management
					(conductor_identity, theater, client_identity);

				//	Apply the new Management.
				Management (row, management);
				}
			else
				{
				//	Re-open with existing Managment.
				if ((DEBUG & DEBUG_MANAGEMENT) != 0)
					System.out.println
						("    Re-opening management:" + NL
						+ management);
				management.Open
					(
					theater.Messenger ().Client_Hostname (),
					theater.Messenger ().Client_Port (),
					client_identity,
					conductor_identity.Get (ADDRESS_PARAMETER_NAME)
					);
				}
			}
		}
	}
if ((DEBUG & DEBUG_MANAGEMENT) != 0)
	System.out.println
		("    Remote_Theater Management -" + NL
		+ management + NL
		+"<<< Conductor_Table_Model.Open_Management");
return management;
}

/**	Get Remote_Theater Management for a Conductor.
<p>
	Remote_Theater Management services are obtained for a Theater
	associated with a Stage_Manager. The client identity {@link
	Message#NAME_PARAMETER_NAME} is qualified with the {@link
	Remote_Theater#REMOTE_THEATER_NAME} following a '/' delimiter if it
	does not already contain this qualifier name. A Remote_Theater
	connection is then established with the Stage_Mangager at the
	specified Theater using the authentication information in the client
	identity, and the Remote_Theater Messenger is linked to the Conductor
	Messenger at the {@link #ADDRESS_PARAMETER_NAME} found in the
	Conductor's identity information. This is done by constructing a new
	Remote_Theater that will provide Management communication directly
	with the Conductor.
<p>
	@param	conductor_identity	The identity Message of the Conductor to
		be opened. Must not be null and must contain an {@link
		#ADDRESS_PARAMETER_NAME} parameter.
	@param	theater	The Theater where the Stage_Manager is located from
		which the Conductor identity was obtained. Must not be null.
	@param	client_identity	The Message containing the {@link
		Theater#KEY_PARAMETER_NAME}.
	@throws	IOException	If a connection could not be established to
		the Stage_Manager. This will be a Theater_Protocol_Exception
		if there were any problems in the content of the Messages
		used to establish the connection.
	@throws	IllegalArgumentException	If the argument requirements are
		not satsified, or the Conductor identity does not have an {@link
		#ADDRESS_PARAMETER_NAME} parameter.
	@see	Remote_Theater#Open(String, int, Message, String)
*/
public static Remote_Theater Management
	(
	Message			conductor_identity,
	Theater			theater,
	Message			client_identity
	)
	throws IOException
{
if ((DEBUG & DEBUG_OPEN_THEATER) != 0)
	System.out.println
		(">>> Management:" + NL
		+"    conductor_identity -" + NL
		+ conductor_identity + NL
		+"    theater -" + NL
		+ theater + NL
		+"    client_identity -" + NL
		+ client_identity);
if (conductor_identity == null)
	throw new IllegalArgumentException (ID + NL
		+ "Can't get remote Theater management without a Conductor identity.");
if (theater == null)
	throw new IllegalArgumentException (ID + NL
		+ "Can't get remote Theater Management without a Theater.");

//	Qualify the client name.
String
	client_name = client_identity.Get (Message.NAME_PARAMETER_NAME);
if (client_name == null ||
	client_name.indexOf (Remote_Theater.REMOTE_THEATER_NAME) < 0)
	{
	if (client_name == null)
		client_name = Remote_Theater.REMOTE_THEATER_NAME;
	else
		client_name += "/" + Remote_Theater.REMOTE_THEATER_NAME;
	client_identity.Set (Message.NAME_PARAMETER_NAME, client_name);
	}

if ((DEBUG & DEBUG_OPEN_THEATER) != 0)
	System.out.println
		("    Constructing Remote_Theater -" + NL
		+"    Hostname - " + theater.Messenger ().Client_Hostname () + NL
		+"        Port - " + theater.Messenger ().Client_Port () + NL
		+"    Client Identity -" + NL
		+ client_identity + NL
		+"    Conductor Address - "
			+ conductor_identity.Get (ADDRESS_PARAMETER_NAME));
/*
	N.B.: Can throw IllegalArgumentException if 
*/
Remote_Theater
	management = new Remote_Theater
		(
		theater.Messenger ().Client_Hostname (),
		theater.Messenger ().Client_Port (),
		client_identity,
		conductor_identity.Get (ADDRESS_PARAMETER_NAME)
		);
if ((DEBUG & DEBUG_OPEN_THEATER) != 0)
	System.out.println
		(management + NL
		+"<<< Management");
return management;
}

/**	Get the data model row index for a Remote_Theater Management.
<p>
	@param	management	A Remote_Theater Management object.
	@return	The data model row index containing the Remote_Theater
		Management, or -1 if the Management was not found in the
		data model.
*/
public int Index
	(
	Remote_Theater	management
	)
{return Managements.indexOf (management);}

/**	Get the data model row index of the next Conductor entry at a Theater
	location starting after a given index.
<p>
	Each Conductor identity, beginning with the data model entry
	immediately following the specified index, has the value of its
	{@link #THEATER_LOCATION_PARAMETER_NAME} parameter compared with the
	specified Theater location. The index of the first one that matches
	is returned.
<p>
	@param	theater_location	A String specifying a Theater location.
		The Theater location is {@link Theater#Full_Location(String)
		fully qualified} to ensure it will correctly match the standard
		locations stored with the Conductor identities.
	@param	index	The data model row index after which the search for
		the next matching Conductor identity will begin. If less than
		zero the search begins with the first row.
	@return	The index of the next Conductor identity that contains a
		matching Theater location, or -1 if no match is found or the
		Theater location is null or the empty String.
	@see	Theater#Location()
*/
public int Next_Index
	(
	String	theater_location,
	int		index
	)
{
//	Fully qualify the Theater location.
theater_location = Theater.Full_Location (theater_location);
if (theater_location != null)
	{
	if (index < -1)
		index = -1;
	int
		size = getRowCount ();
	while (++index < size)
		if (theater_location.equals (Identities.get (index)
				.Get (THEATER_LOCATION_PARAMETER_NAME)))
			return index;
	}
return -1;
}

/**	Get the count of Conductors identities on a Theater that match
	a Conductor_Definition.
<p>
	@param	theater_location	The String specifying a Theater location
		(does not need to be fully qualified). If null, the count will
		be zero.
	@param	conductor_definition	A Conductor_Definition to {@link
		Conductor_Definition#Matches(Message) match} against Conductor
		identities.
	@return	The count of Conductor identities associated with the Theater
		location that match the Conductor_Definition.
	@see	#Next_Index(String, int)
*/
public int Count
	(
	String					theater_location,
	Conductor_Definition	conductor_definition
	)
{
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		(">>> Conductor_Table_Model.Count:" + NL
		+"    theater_location - " + theater_location + NL
		+"    conductor_definition -" + NL
		+ conductor_definition);
int
	count = 0;
if (conductor_definition != null)
	{
	int
		index = -1;
	while ((index = Next_Index (theater_location, index)) >= 0)
		{
		if ((DEBUG & DEBUG_ACCESSORS) != 0)
			System.out.println
				("    Identity " + index + " -" + NL
				+ Identity (index));
		if (conductor_definition.Matches (Identity (index)))
			{
			if ((DEBUG & DEBUG_ACCESSORS) != 0)
				System.out.println
					("    Matches");
			++count;
			}
		else if ((DEBUG & DEBUG_ACCESSORS) != 0)
			System.out.println
				("    No match");
		}
	}
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		("<<< Conductor_Table_Model.Count: " + count);
return count;
}

/**	Set the Remote_Theater Management for a data model row index.
<p>
	If the Remote_Theater Management is identical to the current entry,
	or the row does not have a Conductor identity, nothing is done.
<p>
	If the current Remote_Theater Management entry for the row is
	non-null the row is closed before the entry is replaced. The {@link
	#Processing_State(Message) current Conductor processing state} is
	replaced with the {@link Management#Processing_State() state
	obtained from the new Management}, or zero if the new Management is
	null. This Conductor_Table_Model is registered with the new
	Managment, if non-null, {@link
	Management#Add_Processing_Listener(Processing_Listener) to receive}
	{@link #Processing_Event_Occurred(Processing_Event) Conductor
	processing state change notifications}. Finally, if the the
	processing state changed all table listeners are {@link
	#Notify_State_Changed(int, int) notified of the new state}.
<p>
	@param	row	A data model row index. If not a valid data model row
		index nothing is done.
	@param	management	A Remote_Theater Management object. If this is
		identical to the {@link #Management(int) existing Management}
		nothing is done.
	@return	This Conductor_Table_Model.
*/
public Conductor_Table_Model Management
	(
	int				row,
	Remote_Theater	management
	)
{
if (row >= 0 &&
	row < getRowCount ())
	{
	int
		processing_state = Set_Management (row, management);
	if (processing_state != UNCHANGED_STATE)
		Notify_State_Changed (row, processing_state);
	}
return this;
}

/**	Set the Remote_Theater Management for a Conductor.
<>
	@param	identity	A Conductor identity Message.
	@param	management	A Remote_Theater Management object. If this is
		identical to the {@link #Management(int) existing Management}
		nothing is done.
	@return	This Conductor_Table_Model.
	@see	#Management(int, Remote_Theater)
*/
public Conductor_Table_Model Management
	(
	Message			identity,
	Remote_Theater	management
	)
{return Management (Index (identity), management);}


/**	Special {@link #Processing_State(Message) Conductor processing state}
	value that indicates no change from the current state.
<p>
	When Remote_Theater Management is {@link #Close_Management(int)
	closed} or {@link #Set_Management(int, Remote_Theater) set} the
	previous processing state is returned for use in notifying processing
	state change listeners. If the state was unchanged this special
	unchanged value is returned instead. This enables the caller to avoid
	the cost of the notification and consequent table view updates.
*/
protected static final int
	UNCHANGED_STATE	= -99999;

/**	Set a Remote_Theater Management.
<p>
	<b>N.B.</b>: This method should only be used to set a new
	Remote_Theater Management by a method that does the neccessary state
	checks and table view update notification.
<p>
	If the row does not have a Conductor identity or the Remote_Theater
	Management is identical to the current entry, nothing is done.
<p>
	Management for the row is {@link #Close_Management(int) closed}.
	This will obtain the previous {@link #Processing_State(Message)
	Conductor processing state} or the {@link #UNCHANGED_STATE} value.
<p>
	If the new Management is non-null it is set in the data model. The
	{@link #Processing_State(Message) current Conductor processing state}
	is obtained from the new Management and compared with the previous
	state (or zero if the previous state before closure was unchanged).
	If the state changed the Conductor identity is updated with the
	{@link #Processing_State(Message, int) new processing state}.
<p>
	<b>N.B.</b>: No data model listener notification is done.
<p>
	@param	row	A data model row index.
	@param	management	A Remote_Theater Management object. May be null.
	@return	The previous processing state from the row's identity. This
		will be {@link #UNCHANGED_STATE} if the row has no identity
		(shouldn't happen), the existing Management is identical to the
		new Management, or the processing state is unchanged.
	@see	#Close_Management(int)
*/
protected int Set_Management
	(
	int				row,
	Remote_Theater	management
	)
{
if ((DEBUG & DEBUG_MANAGEMENT) != 0)
	System.out.println
		(">>> Set_Management: " + row + NL
		+ management);
int
	old_processing_state = UNCHANGED_STATE;
Message
	identity = Identity (row);
if (identity != null &&
	management != Managements.get (row))
	{
	//	Close any previous management.
	if ((DEBUG & DEBUG_MANAGEMENT) != 0)
		System.out.println
			("    Closing any previous Management");
	int
		new_processing_state =
		old_processing_state = Close_Management (row);

	if (management != null)
		{
		//	Set the new management.
		Managements.set (row, management);

		//	Update the processing state.
		if (old_processing_state == UNCHANGED_STATE)
			old_processing_state = 0;
		if ((DEBUG & DEBUG_MANAGEMENT) != 0)
			System.out.println
				("    Old processing state: " + old_processing_state);
		try {new_processing_state = management.Processing_State ();}
		catch (Remote_Management_Exception exception)
			{
			if ((DEBUG & DEBUG_MANAGEMENT) != 0)
				System.out.println
					("    Failed to get Management.Processing_State -" + NL
					+ exception);
			}
		if ((DEBUG & DEBUG_MANAGEMENT) != 0)
			System.out.println
				("    New processing state: " + new_processing_state);
		if (new_processing_state != old_processing_state)
			Processing_State (identity, new_processing_state);
		else
			old_processing_state = UNCHANGED_STATE;

		//	Register for processing state updates.
		management.Add_Processing_Listener (this);
		}
	}
if ((DEBUG & DEBUG_MANAGEMENT) != 0)
	System.out.println
		("<<< Set_Management: " + old_processing_state);
return old_processing_state;
}

/**	Test if the Remote_Theater Management at a data model index is open.
<p>
	@param	row	A data model row index.
	@return	true if the Conductor {@link #Management(int) Remote_Theater
		Management at the row index} has been {@link
		Remote_Theater#Opened() opened}; false if the row index is
		invalid, no Remote_Theater Management is present at the row
		index, or the Remote_Theater is not open.
*/
public boolean Opened
	(
	int		row
	)
{
boolean
	opened = false;
if (row >= 0 &&
	row < getRowCount ())
	{
	Remote_Theater
		management = Managements.get (row);
	if (management != null)
		opened = management.Opened ();
	}
return opened;
}

/**	Test if the Remote_Theater associated with a Conductor identity is open.
<p>
	@param	identity	A Conductor identity Message.
	@return	true if the Conductor identity is associated with an open
		Remote_Theater Management; false if the identity is not present,
		has no associated Remote_Theater Management, or the
		Remote_Theater is not open.
	@see	#Opened(int)
*/
public boolean Opened
	(
	Message	identity
	)
{return Opened (Index (identity));}

/**	Close the Remote_Theater Management at a data model index.
<p>
	<b>N.B.</b>: This method should only be used to close a
	Remote_Theater Management by a method that does the neccessary state
	checks and table view update notification.
<p>
	If a {@link #Manager(int) Manager} is present for the row it is
	{@link Manager#Disable() disabled}.
<p>
	This Conductor_Table_Model is {@link
	Remote_Theater#Remove_Processing_Listener(Processing_Listener)
	unregisterd} as a processing listener of the Remote_Theater
	Management which is removed from (nullified in) the data model.
<p.
	The current Conductor {@link #Processing_State(Message) processing
	state} is obtained and then {@link #Processing_State(Message, int)
	reset} to zero (unknown state).
<p>
	<b>N.B.</b>: No data model listener notification is done.
<p>
	@param	row	A data model row index.
	@return	The previous processing state from the row's identity. This
		will be {@link #UNCHANGED_STATE} if the row has no identity
		(shouldn't happen) or the processing state is unchanged.
*/
protected int Close_Management
	(
	int		row
	)
{
if ((DEBUG & DEBUG_MANAGEMENT) != 0)
	System.out.println
		(">>> Close_Management: " + row);
int
	previous_state = UNCHANGED_STATE;
Message
	identity = Identity (row);
if (identity != null)
	{
	Manager
		manager = Manager (row);
	if (manager != null)
		{
		if ((DEBUG & DEBUG_MANAGEMENT) != 0)
			System.out.println
				("    Disabling the Manager");
		manager.Disable ();
		}

	Remote_Theater
		management = Managements.get (row);
	if (management != null)
		{
		if ((DEBUG & DEBUG_MANAGEMENT) != 0)
			System.out.println
				("    Removing Remote_Theater Management -"
				+ management);
		//	Remove from the data model.
		Managements.set (row, null);

		//	Unregister for processing state updates.
		if ((DEBUG & DEBUG_MANAGEMENT) != 0)
			System.out.println
				("    Remove_Processing_Listener");
		management.Remove_Processing_Listener (this);

		//	Close the management.
		if ((DEBUG & DEBUG_MANAGEMENT) != 0)
			System.out.println
				("    Close the Management");
		management.Close ();

		//	Update the processing state.
		previous_state = Processing_State (identity);
		Processing_State (identity, 0);
		if ((DEBUG & DEBUG_MANAGEMENT) != 0)
			System.out.println
				("    Old processing state: " + previous_state);

		if (previous_state == 0)
			previous_state = UNCHANGED_STATE;
		}
	}
if ((DEBUG & DEBUG_MANAGEMENT) != 0)
	System.out.println
		("<<< Close_Management: " + previous_state);
return previous_state;
}

/*------------------------------------------------------------------------------
	Managers
*/
/**	Get the Conductor Manager for a data model row index.
<p>
	@param	row	A data model row index.
	@return	A Conductor Manager. This will be null if the row index is
		not valid or no Manager is present at the index.
*/
public Manager Manager
	(
	int		row
	)
{
if (row >= 0 &&
	row < getRowCount ())
	return Managers.get (row);
return null;
}

/**	Get the Manager for a Conductor.
<p>
	@param	identity	A Conductor identity Message.
	@return	A Conductor Manager. This will be null if the Conductor
		identity was null or not found.
*/
public Manager Manager
	(
	Message	identity
	)
{return Manager (Index (identity));}

/**	Set the Manager for a data model row index.
<p>
	If the row has a non-null Manager entry that Manager is {@link
	Manager#Close() closed} before the entry is replaced. If the new
	Manager is not null the {@link #Management(int, Remote_Theater)
	Management is set} to new {@link Manager#Management() Manager's
	Management}.
<p>
	@param	row	A data model row index. If not a valid data model row
		index nothing is done.
	@param	manager	A Conductor Manager.
	@return	This Conductor_Table_Model.
*/
public Conductor_Table_Model Manager
	(
	int			row,
	Manager		manager
	)
{
if (row >= 0 &&
	row < getRowCount ())
	{
	Manager
		current_manager = Managers.get (row);
	if (current_manager != null &&
		current_manager != manager)
		current_manager.Close ();
	Managers.set (row, manager);

	if (manager != null)
		Management (row, (Remote_Theater)manager.Management ());
	}
return this;
}

/**	Get the data model row index for a Conductor Manager.
<p>
	@param	manager	A Conductor Manager.
	@return	The data model row index containing the Manager, or -1
		if the Manager was not found.
*/
public int Index
	(
	Manager	manager
	)
{return Managers.indexOf (manager);}

/**	Open a Conductor Manager at a data model row index.
<p>
	If a {@link #Manager(int) Manager is present} at the data model row it
	is returned. Otherwise, if {@link #Management(int) Remote_Theater
	Management is present} at the data model row and it is {@link
	Remote_Theater#Opened() opened}, it is used to construct a new
	Manager which is set for the data model row and then returned.
<p>
	@param	row	A data model row index.
	@return	A Conductor Manager. This will be null if the row index is
		invalid, or an opened Remote_Theater Management is not available
		for the row.
	@throws	Remote_Management_Exception	If a problem was encountered
		while constructing a new Manager.
*/
public Manager Open_Manager
	(
	int		row
	)
	throws Remote_Management_Exception
{
Manager
	manager = null;
if (row >= 0 &&
	row < getRowCount ())
	{
	manager = Managers.get (row);
	if (manager == null)
		{
		Remote_Theater
			management = Managements.get (row);
		if (management != null &&
			management.Opened ())
			{
			manager = new Manager (management);
			Managers.set (row, manager);
			}
		}
	}
return manager;
}

/**	Close a Conductor Manager at a data model row index.
<p>
	If a {@link #Manager(int) Manager} is present at the data model row
	index it is {@link Manager#Close() closed} and it's entry removed
	from the data model.
<p>
	@param	row	A data model row index.
	@return	true if the Manager was present in the data model and has
		been closed; false if the Manager was not present.
*/
public boolean Close_Manager
	(
	int		row
	)
{
Manager
	manager = Manager (row);
if (manager != null)
	{
	manager.Close ();
	Managers.set (row, null);
	return true;
	}
return false;
}

/**	Close a Conductor Manager.
<p>
	If the {@link #Index(Manager) Manager is present} in the data model
	it is {@link Manager#Close() closed} and it's entry removed from the
	data model.
<p>
	<b>N.B.</b>: If the Manager is not present in data model it is not
	closed.
<p>
	@param	manager	A Conductor Manager.
	@return	true if the Manager was present in the data model and has
		been closed; false if the Manager was not present.
*/
public boolean Close
	(
	Manager	manager
	)
{
int
	row = Index (manager);
if (row >= 0)
	{
	manager.Close ();
	Managers.set (row, null);
	return true;
	}
return false;
}

/*==============================================================================
	Manipulators
*/
/**	Remove a data model row entry.
<p>
	@param	row	A data model row index.
	@return	The Conductor identity that was removed. This will be
		null if the row index is invalid.
*/
public Message Remove
	(
	int		row
	)
{
Message
	identity = Identity (row);
if (identity != null)
	{
	//	Notify state change listeners before the row is removed.
	Conductor_Table_Model_Event
		event = Notify_Deleting (row);

	//	Remove the row.
	Remove_at (row);

	//	Notify table view listeners after the row is removed.
	Notify_Table_Change_Listeners (event);
	}
return identity;
}

/**	Remove a data model entry associated with a Conductor Identity.
<p>
	@param	identity	A Conductor identity Message. If null nothing
		is done and false is returned.
	@return	true if the identity was present and removed; false otherwise.
*/
public boolean Remove
	(
	Message	identity
	)
{return Remove (Index (identity)) != null;}

/**	Remove all Conductors at a Theater location.
<p>
	Each data model entry {@link #Next_Index(String, int) found for the
	Theater location} is {@link #Remove_at(int) removed}. If any row was
	removed a change notification event for the entire table will be sent
	to the state change listeners. This event will also be sent to the
	table change listeners if non-contiguous ranges of rows were deleted;
	otherwise a {@link #Notify_Deleted(int, int) deleted} event will be
	delivered for only the rows removed.
<p>
	@param	theater_location	The String specifying a Theater location
		(does not need to be fully qualified).
	@return	true if at least one row was removed; false otherwise.
	@see	#Next_Index(String, int)
*/
public boolean Remove
	(
	String	theater_location
	)
{
int
	first_row = -1,
	last_row = -1,
	index = -1;
while((index = Next_Index (theater_location, index)) >= 0)
	{
	Remove_at (index);
	if (first_row < 0)
		//	Initialize the delete row indices.
		first_row = last_row = index;
	else
	if (last_row >= 0)
		{
		//	Contigous range so far.
		if (first_row == index)
			//	Contiguous row removal.
			++last_row;
		else
			//	Non-contiguous removal.
			last_row = -1;
		}
	/*
		Decrement the index so the Next_Index search will begin
		with the row following the one just removed.
	*/
	--index;
	}
if (first_row >= 0)
	{
	//	At least one row was removed.
	Conductor_Table_Model_Event
		event = new Conductor_Table_Model_Event (this);
	if (last_row >= 0)
		//	Contiguous range removed. Notify the table listeners.
		Notify_Deleted (first_row, last_row);
	else
		//	Non-contiguous ranges removed. Notify the table listeners.
		Notify_Table_Change_Listeners (event);

	//	Notify the state change listeners.
	Notify_State_Change_Listeners (event);

	return true;
	}
return false;
}

/**	Clear the data model of all entries.
<p>
	If the data model contians any {@link #getRowCount() rows} all are
	(@link #Remove_at(int) removed} and a {#link Notify_Table_Changed()
	table changed notification} is sent to all listeners.
*/
public void Clear ()
{
int
	row = getRowCount ();
if (row > 0)
	{
	while (--row >= 0)
		Remove_at (row);
	Notify_Table_Changed ();
	}
}

/**	Remove a data model row entry.
<p>
	<b>N.B.</b>: This method should only be used to remove a data model
	row by a method that will do the expected table change notification.
<p>
	Any Manager and Remote_Theater Management for the row is
	{@link #Close_Management(int) closed}. Then the Manager, Mangement
	and Identity entries are removed from the data model.
<p>
	<b>N.B.</b>: No data model listener notification is done.
<p>
	@param	row	A data model row index.
	@return	The Conductor identity Message for the row that was
		removed. This will be null if the row index is not valid.
*/
protected Message Remove_at
	(
	int		row
	)
{
Message
	identity = Identity (row);
if (identity != null)
	{
	Close_Management (row);
	Identities.remove (row);
	Managements.remove (row);
	Managers.remove (row);
	}
return identity;
}

/**	Close the Remote_Theater and Manager at a row index.
<p>
	The Remote_Theater Management for the row is {@link
	#Close_Management(int) closed}. If this resulted in a Conductor
	{@link #Processing_State(int) processing state} change a {@link
	#Notify_State_Changed(int, int) state change notification} is sent to
	all data model listeners.
<p>
	@param	row	A data model row index.
*/
public void Close
	(
	int		row
	)
{
int
	previous_state = Close_Management (row);
if (previous_state != UNCHANGED_STATE)
	Notify_State_Changed (row, previous_state);
}

/**	Close a Theater.
<p>
	The Remote_Theater Management {@link #Next_Index(String, int) found
	at the Theater location} is {@link #Close_Management(int) closed}. If
	any of these closures resulted in a Conductor {@link
	#Processing_State(int) processing state} change a {@link
	#Notify_Table_Changed() state change notification} is sent to all
	data model listeners.
<p>
	@param	theater_location	A String specifying the Theater location
		(does not need to be fully qualified). If null or the emtpy
		String nothing is done.
	@return	true if any closures resulted in a Conductor processing state
		change; false otherwise.
	@see	#Next_Index(String, int)
*/
public boolean Close
	(
	String	theater_location
	)
{
boolean
	changed = false;
int
	index = -1;
while ((index = Next_Index (theater_location, index)) >= 0)
	if (Close_Management (index) != UNCHANGED_STATE)
		changed = true;
if (changed)
	Notify_Table_Changed ();
return changed;
}

/**	Close all Managers and Remote_Theater Managements.
<p>
	Each {@link #Close_Manager(int) Manager is closed} and each {@link
	#Close_Management(int) Remote_Theater Management is closed}. If any
	of these closures resulted in a Conductor {@link
	#Processing_State(int) processing state} change a {@link
	#Notify_Table_Changed() state change notification} is sent to all
	data model listeners.
<p>
	@return	true if any closures resulted in a Conductor processing state
		change; false otherwise.
*/
public boolean Close_All ()
{
boolean
	changed = false;
int
	index = Managers.size ();
while (--index >= 0)
	{
	Close_Manager (index);
	if (Close_Management (index) != UNCHANGED_STATE)
		changed = true;
	}
if (changed)
	Notify_Table_Changed ();
return changed;
}

/*==============================================================================
	Event Notification
*/
/**	Notify all listeners that a data model row has been inserted.
<p>
	Listeners are sent an {@link TableModelEvent#INSERT}
	Conductor_Table_Model_Event that specifies the row.
<p>
	@param	row	The data model row that was inserted.
*/
protected void Notify_Inserted
	(
	int		row
	)
{
Conductor_Table_Model_Event
	event = new Conductor_Table_Model_Event (this,
		row, row, TableModelEvent.ALL_COLUMNS, TableModelEvent.INSERT);
Notify_State_Change_Listeners (event);
fireTableChanged (event);
}

/**	Notify all listeners that a data model row has been updated.
<p>
	Listeners are sent an {@link TableModelEvent#UPDATE}
	Conductor_Table_Model_Event that specifies the row.
<p>
	@param	row	The data model row that was updated.
*/
protected void Notify_Updated
	(
	int		row
	)
{
Conductor_Table_Model_Event
	event = new Conductor_Table_Model_Event (this,
		row, row, TableModelEvent.ALL_COLUMNS, TableModelEvent.UPDATE);
Notify_State_Change_Listeners (event);
fireTableChanged (event);
}

/**	Notify all listeners that a continguous range of data model rows
	have been updated.
<p>
	Listeners are sent an {@link TableModelEvent#UPDATE}
	Conductor_Table_Model_Event that specifies the range from first_row
	to last_row, inclusive.
<p>
	@param	first_row	The first data model row that was updated.
	@param	last_row	The last data model row that was updated.
*/
protected void Notify_Updated
	(
	int		first_row,
	int		last_row
	)
{
Conductor_Table_Model_Event
	event = new Conductor_Table_Model_Event (this,
		first_row, last_row, TableModelEvent.ALL_COLUMNS,
		TableModelEvent.UPDATE);
Notify_State_Change_Listeners (event);
fireTableChanged (event);
}

/**	Notify all listeners that a Conductor processing state changed.
<p>
	Listeners are sent an {@link TableModelEvent#UPDATE}
	Conductor_Table_Model_Event that specifies the row and the
	processing state.
<p>
	@param	row	The data model row that has changed.
	@param	processing_state	The previous processing state. The {@link
		#Processing_State(Message) current processing state} is contained
		in the Conductor {@link #Identity(int) identity} for the row.
*/
protected void Notify_State_Changed
	(
	int		row,
	int		processing_state
	)
{
Conductor_Table_Model_Event
	event = new Conductor_Table_Model_Event (this,
		row, row, CONDUCTORS_COLUMN, TableModelEvent.UPDATE)
		.Processing_State (processing_state);
Notify_State_Change_Listeners (event);
fireTableChanged (event);
}

/**	Notify table change listeners that a data model row has been deleted.
<p>
	Listeners are sent a {@link TableModelEvent#DELETE}
	Conductor_Table_Model_Event that specifies the row. <b>N.B.</b>: The
	{@link #Add_State_Change_Listener(TableModelListener) processing
	state change listeners} are not notified. They should have been sent
	a pre-deletion {@link #Notify_Deleting(int) deleting} notification of
	the same event.
<p>
	@param	row	The data model row that has been deleted.
*/
protected void Notify_Deleted
	(
	int		row
	)
{fireTableChanged (new Conductor_Table_Model_Event (this,
	row, row, TableModelEvent.ALL_COLUMNS, TableModelEvent.DELETE));}

/**	Notify table change listeners that a data model row range has been
	deleted.
<p>
	Listeners are sent a {@link TableModelEvent#DELETE}
	Conductor_Table_Model_Event that specifies the range from first_row
	to last_row, inclusive. <b>N.B.</b>: The {@link
	#Add_State_Change_Listener(TableModelListener) processing state
	change listeners} are not notified. They should have been sent a
	pre-deletion {@link #Notify_Deleting(int) deleting} notification of
	the same event.
<p>
	@param	first_row	The first data model row that has been deleted.
	@param	last_row	The last data model row that has been deleted.
*/
protected void Notify_Deleted
	(
	int		first_row,
	int		last_row
	)
{fireTableChanged (new Conductor_Table_Model_Event (this,
	first_row, last_row, TableModelEvent.ALL_COLUMNS, TableModelEvent.DELETE));}

/**	Notify all listeners that the data model content has changed.
<p>
	This notification is used when the change to the data model can
	not be localized to a contiguous range of rows. All listeners are
	sent an {@link TableModelEvent#UPDATE} Conductor_Table_Model_Event
	that specifies the maximum row value range (0 - Integer.MAX_VALUE}.
*/
protected void Notify_Table_Changed ()
{
Conductor_Table_Model_Event
	event = new Conductor_Table_Model_Event (this);
Notify_State_Change_Listeners (event);
fireTableChanged (event);
}

/**	Send a change notification to the table change listeners.
<p>
	All  {@link
	AbstractTableModel#addTableModelListener(TableModelListener) table
	model listeners} are notified of the event. <b>N.B.</b>: Only table
	model listeners are notified; the {@link
	#Add_State_Change_Listener(TableModelListener) processing state
	change listeners} are not notified. Use the {@link
	#Notify_State_Change_Listeners(Conductor_Table_Model_Event)} to notify
	only the processing state change listeners.
<p>
	@param	event	The Conductor_Table_Model_Event to be delivered.
*/
protected void Notify_Table_Change_Listeners
	(
	Conductor_Table_Model_Event	event
	)
{fireTableChanged (event);}

/*------------------------------------------------------------------------------
	State Change Listeners
*/
/**	Add a listener for Conductor processing state change events.
<p>
	These listeners receive notification whenever one or more
	Conductor identities in the data model may have a changed
	{@link #Processing_State(Message) processing state}.
<p>
	State change listeners are distinct from {@link
	AbstractTableModel#addTableModelListener(TableModelListener) table
	model listeners}. When appropriate, both sets of listeners will be
	notified of a data model change event. At other times only one or the
	other set of listeners will be sent a notification. A listener should
	register to receive processing state change notifications when that
	is all it is interested in knowing aobut. A listener should register
	to receive table change notifications whien it is interested in any
	changes, including processing state changes.
<p>
	@param	listener	A TableModelListener.
	@return	This Conductor_Table_Model.
*/
public Conductor_Table_Model Add_State_Change_Listener
	(
	TableModelListener	listener
	)
{
if (listener != null &&
	! State_Change_Listeners.contains (listener))
	State_Change_Listeners.add (listener);
return this;
}

/**	Remove a listener for Conductor processing state change events.
<p>
	@param	listener	A TableModelListener.
	@return	true if the listener was removed; false if it was not
		registered.
	@see	#Add_State_Change_Listener(TableModelListener)
*/
public boolean Remove_State_Change_Listener
	(
	TableModelListener	listener
	)
{return State_Change_Listeners.remove (listener);}

/**	Send a change notification to the processing state change listeners.
<p>
	All {@link #Add_State_Change_Listener(TableModelListener) state
	change listeners} are notified of the event. <b>N.B.</b>: Only
	processing state change listeners are notified; the {@link
	AbstractTableModel#addTableModelListener(TableModelListener) table
	model listeners} are not notified. Use the {@link
	#Notify_Table_Change_Listeners(Conductor_Table_Model_Event)} to notify
	only the table model listeners of an event.
<p>
	@param	event	The Conductor_Table_Model_Event to be delivered.
*/
protected void Notify_State_Change_Listeners
	(
	Conductor_Table_Model_Event	event
	)
{
int
	index = -1,
	size = State_Change_Listeners.size ();
while (++index < size)
	State_Change_Listeners.get (index).tableChanged (event);
}

/**	Notify processing state change listeners that a data model is about
	to be deleted.
<p>
	Listeners are sent a {@link TableModelEvent#DELETE}
	Conductor_Table_Model_Event that specifies the row. <b>N.B.</b>: The
	{@link AbstractTableModel#addTableModelListener(TableModelListener)
	table change listeners} are not notified. They should be
	sent a post-deletion {@link #Notify_Deleted(int) deleted}
	notification of the same event.
<p>
	@param	row	The data model row that will be deleted.
	@return	The Conductor_Table_Model_Event that was delivered.
*/
protected Conductor_Table_Model_Event Notify_Deleting
	(
	int		row
	)
{
Conductor_Table_Model_Event
	event = new Conductor_Table_Model_Event (this,
		row, row, TableModelEvent.ALL_COLUMNS, TableModelEvent.DELETE);
Notify_State_Change_Listeners (event);
return event;
}

/**	Notify table change listeners that a data model row range is about
	to be deleted.
<p>
	Listeners are sent a {@link TableModelEvent#DELETE}
	Conductor_Table_Model_Event that specifies the range from first_row
	to last_row, inclusive. <b>N.B.</b>: The {@link
	AbstractTableModel#addTableModelListener(TableModelListener) table
	change listeners} are not notified. They should be sent a
	post-deletion {@link #Notify_Deleted(int) deleted} notification of
	the same event.
<p>
	@param	first_row	The first data model row that is about to be
		deleted.
	@param	last_row	The last data model row that is about to be
		deleted.
	@return	The Conductor_Table_Model_Event that was delivered.
*/
protected Conductor_Table_Model_Event Notify_Deleting
	(
	int		first_row,
	int		last_row
	)
{
Conductor_Table_Model_Event
	event = new Conductor_Table_Model_Event (this,
		first_row, last_row, TableModelEvent.ALL_COLUMNS,
			TableModelEvent.DELETE);
Notify_State_Change_Listeners (event);
return event;
}

/*==============================================================================
	Processing_Listener
*/
private Vector<Processing_Event>
	Processing_Event_Queue		= new Vector<Processing_Event> ();

/**	Event handler for Conductor Processing_Events.
<p>
	<b>N.B.</b>: This method is public as a side effect of implementing
	the Processing_Listener interface. It should not be used directly.
<p>
	If the incoming event contains a Conductor processing event change
	the event is enqueued and a Runnable placed  on the Swing event
	handling queue for disposition of the event.
<p>
	The event disposition method pulls the next event from the front of
	the event queue. The data model index for the Remote_Theater
	Management source of the event is obtained and used to obtain the
	associated Conductor identity. The identity provides the previous
	processing state and is set with the new state. Then all {@link
	#Add_State_Change_Listener(TableModelListener) processing state change
	listeners} are {@link #Notify_State_Changed(int, int) notified of the
	state change}.
<p>
	@param	event	A Processing_Event.
	@see	Processing_Event
*/
public void Processing_Event_Occurred
	(
	Processing_Event	event
	)
{
if (event.Changes.Processing_State () == 0)
	return;
if ((DEBUG & DEBUG_LISTENER) != 0)
	System.out.println
		(">-< Processing_Event_Occurred");

synchronized (Processing_Event_Queue)
	{Processing_Event_Queue.add (event);}
SwingUtilities.invokeLater (new Runnable ()
{public void run () {Event_Disposition ();}});
}

private void Event_Disposition ()
{
if (Processing_Event_Queue.isEmpty ())
	return;
if ((DEBUG & DEBUG_LISTENER) != 0)
	System.out.println
		(">>> Event_Disposition");

Processing_Event
	event;
synchronized (Processing_Event_Queue)
	/*
		>>> WARNING <<< No check is being done for an empty queue
		because Message_Disposition is run from the event queue
		after a Message_Delivered enqueues the delivered Message,
		and only one Message is removed from the queue each time
		Message_Disposition is run. If this one-to-one Message queuing
		relationship between Message_Delivered and Message_Disposition
		is ever changed an empty queue check must be added.
	*/
	{event = (Processing_Event)Processing_Event_Queue.remove (0);}

int
	index = Index ((Remote_Theater)event.getSource ());
if (index < 0)
	//	Shouldn't happen if properly deregistered on Management removal.
	return;

//	Update the processing state.
Message
	identity = Identities.get (index);
if ((DEBUG & DEBUG_LISTENER) != 0)
	System.out.println
		("    Notify_State_Changed to "
			+ event.Changes.Processing_State () + NL
		+"for" + NL
		+ identity);
int
	processing_state = Processing_State (identity);
Processing_State (identity, event.Changes.Processing_State ());
Notify_State_Changed (index, processing_state);
if ((DEBUG & DEBUG_LISTENER) != 0)
	System.out.println
		("<<< Event_Disposition");
}

/*==============================================================================
	Utilities
*/
/**	Get the location of a Conductor from its identity.
<p>
	If the {@link #THEATER_LOCATION_PARAMETER_NAME} is not present in
	the identity the value of the {@link #CONDUCTOR_ID_PARAMETER_NAME}
	will be used, without any PID suffix, to produce a {@link
	Theater#Full_Location(String) fully qualified location}.
<p>
	@param	identity	A Conductor {@link Conductor#Identity() identity}
		Message.
	@return	A String providing the Theater location of the Conductor. This
		will be null if the identity is null or neither of the expected
		parameters is present in the identity.
*/
public static String Theater_Location
	(
	Message	identity
	)
{
String
	location = null;
if (identity != null)
	{
	if ((location = identity.Get (THEATER_LOCATION_PARAMETER_NAME)) == null)
		{
		location = identity.Get (CONDUCTOR_ID_PARAMETER_NAME);
		int
			index = location.indexOf (':');
		if (index > 0)
			location = location.substring (0, index);
		location = Theater.Full_Location (location);
		}
	}
return location;
}

/**	Get the name of a Conductor from its identity.
<p>
	The value of the {@link #PIPELINE_PARAMETER_NAME} will be returned.
	<b>N.B.</b>: A Conductor's name is not necessarily the same as its
	{@link #Pipeline(Message) pipeline name}.
<p>
	@param	identity	A Conductor {@link Conductor#Identity() identity}
		Message.
	@return	A String providing the name of the Conductor. This will be
		null if the identity is null or neither of the expected
		parameters is present in the identity.
*/
public static String Conductor_Name
	(
	Message	identity
	)
{
if (identity != null)
	{
	if (identity.Name ().equals (PIRL.PVL.Parser.CONTAINER_NAME) ||
		identity.Name ().equals (Theater.IDENTITY_ACTION))
		return identity.Get (PIPELINE_PARAMETER_NAME);
	return identity.Name ();
	}
return null;
}

/**	Get the name of a Conductor pipeline from its identity.
<p>
	The {@link #PIPELINE_PARAMETER_NAME} value is obtained.
<p>
	@param	identity	A Conductor {@link Conductor#Identity() identity}
		Message.
	@return	A String providing the name of the Conductor pipeline. This
		will be null if the identity is null or the required parameter
		is not present in the identity.
*/
public static String Pipeline
	(
	Message	identity
	)
{
String
	name = null;
if (identity != null)
	name = identity.Get (PIPELINE_PARAMETER_NAME);
return name;
}

/**	Get a Conductor identification from its identity.
<p>
	The {@link #CONDUCTOR_ID_PARAMETER_NAME} value is obtained.
<p>
	@param	identity	A Conductor {@link Conductor#Identity() identity}
		Message.
	@return	A String providing the identification of the Conductor
		pipeline. This will be null if the identity is null or the
		required parameter is not present in the identity.
*/
public static String Identification
	(
	Message	identity
	)
{
String
	id = null;
if (identity != null)
	id = identity.Get (CONDUCTOR_ID_PARAMETER_NAME);
return id;
}

/**	Get the processing state of a Conductor.
<p>
	The processing state is the value of the {@link
	#PROCESSING_STATE_PARAMETER_NAME} from the identity. If this
	parameter is not present zero will be returned. The expected
	processing state values are:
<p>
<dl>
<dt>{@link Conductor#RUNNING}
	<dd>Source records are being processing.
<dt>{@link Conductor#POLLING}
	<dd>No source records are currently available for processing; the
	Conductor is polling for new source records to process.
<dt>{@link Conductor#RUN_TO_WAIT}
	<dd>When processing of the current source record completes Conductor
	will go into the waiting state.
<dt>{@link Conductor#WAITING}
	<dd>The Conductor is waiting to be told to being processing.
<dt>{@link Conductor#HALTED}
	<dd>A problem condition caused the Conductor to halt processing.
	The problem may be the result of the maximum number of sequential
	source records processing procedure failures having occured, a
	database access failure, or some other system error.
<dt>0 - Unknown
	<dd>A processing state has not yet been obtained from the Conductor.
</dl>
	The WAITING and HALTED state codes are negative; all others are positive.
<p>
	@param	identity	A Conductor {@link Conductor#Identity() identity}
		Message. If null, zero is returned.
	@return	A Conductor processing state value, or zero if none is
		available.
*/
public static int Processing_State
	(
	Message	identity
	)
{
int
	processing_state = 0;
if (identity != null)
	{
	try {processing_state = Integer.parseInt
			(identity.Get (PROCESSING_STATE_PARAMETER_NAME));}
	catch (NumberFormatException exception) {}
	}
return processing_state;
}

/**	Set the processing state of a Conductor.
<p>
	<b>N.B.</b>: This method should only be used to set a new processing
	state by a method that does the neccessary state checks and table
	view update notification.
<p>
	The Conductor identity has the {@link #PROCESSING_STATE_PARAMETER_NAME}
	set to the processing state value.
<p>
	@param	identity	A Conductor {@link Conductor#Identity() identity}
		Message. Must not be null.
	@param	processing_state	A processing state value.
	@see	#Processing_State(Message)
*/
protected static void Processing_State
	(
	Message	identity,
	int		processing_state
	)
{identity.Set (PROCESSING_STATE_PARAMETER_NAME, processing_state, 0);}



}
