/*	Database_Exception

PIRL CVS ID: Database_Exception.java,v 1.18 2012/04/16 06:08:57 castalia Exp

Copyright (C) 2001-2007  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package	PIRL.Database;

import PIRL.Strings.Words;

import java.sql.*;
import java.util.Vector;

import PIRL.Configuration.Configuration_Exception;

/**
The excpetion thrown from the PIRL.Database package.
<P>
@author		Bradford Castalia - UA/PIRL
@version	1.18
*/
public class Database_Exception
	extends java.lang.Exception
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Database.Database_Exception (1.18 2012/04/16 06:08:57)";

/**	Delimiter characters for message words.
*/
private static final String
	DELIMITERS = " \t\n\r&:=";

/**	System new-line sequence.
*/
private static final String
	NL = Database.NL;

/**	The list of words that may preceed words in the message to be masked out.
*/
private static final Vector
	MASKED_WORDS = new Vector (1);
static
	{
	MASKED_WORDS.add ("password");
	MASKED_WORDS.add ("Password");
	MASKED_WORDS.add ("PASSWORD");
	}

/**	The two character "class" prefix of a five character SQLState value
	defined by the X/Open and SQL Access Group (Open Group) SQL CAE
	specification (1992) that indicates a failure to connect to the
	database.
*/
public static final String
	DISCONNECTED_STATE = "08";


// Debug control.
private static final int
	DEBUG_OFF					= 0,
	DEBUG_CONSTRUCTOR			= 1 << 0,
	DEBUG_ACCESSORS				= 1 << 1,
	DEBUG_HELPERS				= 1 << 2,
	DEBUG_ALL					= -1,

	DEBUG						= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Constructs a Database_Exception with a message and a cause.
<p>
	The ID of this class will preceed the message. If a none-null message
	is specified it will be preceed with a new-line character. If a
	non-null cause is specified its message will be appended to the
	specified message after a new-line character.
<p>
	Appending the cause's message to the specified message is done for
	backwards compatibility with the exceptions that did not support a
	chained cause. This can result in applications that expect to chain
	the cause messages themselves to get unexpected redundant messages.
<p>
	<b>N.B.</b>: Once a cause is set it can not be changed. Therefore,
	though it is allowed to set the cause to null, the Database_Exception
	will not do this to avoid the situation of not being able to replace
	a null cause with a valid cause. Use the {@link
	Throwable#initCause(Throwable)} method to provide a cause after the
	Configuration_Exception has been constructed without one.
<p>
	@param	message	The exception's message String (may be null).
	@param	cause	The exception's Throwable cause (may be null).
		If the cause is null, no cause is set in the exception object.
*/
public Database_Exception
	(
	String		message,
	Throwable	cause
	)
{
super
	(ID +
	((message == null) ? "" : (NL + masked_String (message))) +
	((cause == null)   ? "" : (NL + exception_String (cause))));
if (cause != null)
	initCause (cause);
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		(">-< Database_Exception: cause - " +
			((cause instanceof SQLException) ? "SQLException" :
			((cause instanceof Database_Exception) ? "Database_Exception" :
			((cause == null) ? "none" : "Throwable"))));
}

/**	Constructs a Database_Exception having a cause.
<p>
	The exception will have the ID of this class as its message.
<p>
	@param	cause	The exception's Throwable cause (may be null).
*/
public Database_Exception
	(
	Throwable	cause
	)
{this ((String)null, cause);}

/**	Constructs a Database_Exception with a message.
<p>
	@param	message	The exception's message String (may be null).
	@see	#Database_Exception(String, Throwable)
*/
public Database_Exception
	(
	String		message
	)
{this (message, null);}

/**	Constructs a Database_Exception.
<p>
	The exception will have the ID of this class as its message.
<p>
	@see	#Database_Exception(String, Throwable)
*/
public Database_Exception ()
{this (null, null);}

/*==============================================================================
	Methods
*/
/**	Tests if this Database_Exception was caused by an SQLException indicating
	loss of database connection.
<p>
	If the cause of the Database_Exception was itself a Database_Exception
	it's cause is recursivly checked.
<p>
	@return	true if this exception was caused by an SQLException which
		has an SQL state class (first two characters) of
		{@link #DISCONNECTED_STATE}.
*/
public boolean Disconnected ()
{
Throwable
	cause = getCause ();
while (cause != null &&
		cause instanceof Database_Exception)
	cause = cause.getCause ();
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		(">-< Database_Exception.Disconnected:" +
		((cause instanceof SQLException) ? (" " +
		((SQLException)cause).getSQLState ().startsWith (DISCONNECTED_STATE) +
			" (" + ((SQLException)cause).getSQLState () + ')') : " false"));
return
	cause instanceof SQLException &&
	((SQLException)cause).getSQLState ().startsWith (DISCONNECTED_STATE);
}

/*==============================================================================
	Helpers
*/
/**	Generates a message String for a Throwable.
<p>
	The String returned by the {@link Throwable#getMessage()} method is
	used. If the throwable is null or its message is null, the empty
	String is used. This String has words that should be hidden {@link
	#masked_String(String) masked} out.
<p>
	If the throwable is a SQLException its message is preceed with a "SQL
	exception -" line and appended with two lines listing the SQL state
	and error codes of the exception.
<p>
	@param	throwable	The Throwable from which to generate a message.
*/
static public String exception_String
	(
	Throwable	throwable
	)
{
String
	message = "";
while (throwable != null)
	{
	if ((throwable instanceof SQLException) &&
		message.length () == 0)
		message = "SQL exception -" + NL;
	message += masked_String (throwable.getMessage ());
	if (throwable instanceof SQLException)
		{
		message +=
		"    State Code: " + ((SQLException)throwable).getSQLState () + NL +
		"    Error Code: " + ((SQLException)throwable).getErrorCode ();
		if (((SQLException)throwable).getNextException () != null)
			throwable = ((SQLException)throwable).getNextException ();
		else
			throwable = ((SQLException)throwable).getCause ();
		}
	else
		throwable = throwable.getCause ();
	}
if ((DEBUG & DEBUG_HELPERS) != 0)
	System.out.println
		(">-< Database_Exception.exception_String:" + NL
		+ message);
return message;
}

/**	Generates a message String for an Exception.
<p>
	@param	exception	The Exception from which to generate a message.
	@see	#exception_String(Throwable)
*/
static public String exception_String
	(
	Exception	exception
	)
{return exception_String ((Throwable)exception);}

/**	Masks out words that should be hidden in a String.
<p>
	When the word "password", "Password" or "PASSWORD" is encountered in
	the string the following word is replaced with "*******". The
	delimiters used to identify word characters are any of the
	" \t\n\r&:=" characters.
<p>
	@param	string	The String to be masked.
	@return	The masked String.
	@see	Words#Mask(Vector)
	@see	Words#Delimiters(String)
*/
static public String masked_String
	(
	String	string
	)
{
if (string == null)
	return "";
Words
	words = new Words (string);
return words
	.Delimiters (DELIMITERS)
	.Mask (MASKED_WORDS)
	.toString ();
}

/**	Copy an SQLException with any passwords masked out.
<P>
	A new SQLException is constructed containing a copy of the message
	and state description from the specified SQLException with any
	passwords {@link #masked_String(String) masked out} in the message.
	The stack track is also copied. If the SQLException has a chained
	cause, that is copied over.
<P>
	This method is applied recursively to any SQLException chained to
	the specified exception, and the resulting new SQLException used
	to replace the chained exception.
<P>
	In effect, a deep copy of the specified SQLException is made
	with the message being masked in the process.
<P>
	@param	SQL_exception	The SQLException to be masked and copied.
	@return	A SQLExcpetion that is a deep copy of the original.
*/
public static SQLException masked_SQLException
	(
	SQLException	SQL_exception
	)
{
if (SQL_exception == null)
	return null;

//	Message.
String
	message = SQL_exception.getMessage ();
if (message != null)
	message = Database_Exception.masked_String (message);
SQLException
	exception = new SQLException
		(message, SQL_exception.getSQLState ());

//	Stack trace.
exception.setStackTrace (SQL_exception.getStackTrace ());

//	Chained Throwable.
Throwable
	throwable = SQL_exception.getCause ();
if (throwable != null)
	exception.initCause (throwable);

//	Chained SQLException.
SQLException
	chained_exception = SQL_exception.getNextException ();
if (chained_exception != null)
	SQL_exception.setNextException (masked_SQLException (chained_exception));

return exception;
}

}	//	End of class

	
