/*	Lister

PIRL CVS ID: Lister.java,v 1.20 2012/04/16 06:14:23 castalia Exp

Copyright (C) 2002-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package PIRL.PVL;

import	java.io.Writer;
import	java.io.OutputStream;
import	java.io.OutputStreamWriter;
import	java.io.BufferedWriter;
import	java.io.File;
import	java.io.FileWriter;
import	java.io.StringWriter;
import	java.io.IOException;
import	java.io.UnsupportedEncodingException;
import	java.util.ListIterator;

import	PIRL.Strings.String_Buffer;

/**	A <I>Lister</I> writes PVL syxtax.
<P>
	A <I>Lister</I>  complements a <I>Parser</I> in that the latter
	interprets Parameter Value Language (PVL) syntax from a Reader into
	<I>Parameter</I> and <I>Value</I> objects, while the former
	generates PVL syntax to a Writer.
<P>
@see		Parser
@see		Parameter
@see		Value

@author		Bradford Castalia, UA/PIRL
@version	1.20
*/
public class Lister
{
/**	Class name and version identification.
*/
public static final String
	ID = "PIRL.PVL.Lister (1.20 2012/04/16 06:14:23)";

/*------------------------------------------------------------------------------
	PVL syntax modes:
*/
/**	The default for enforcing {@link #Strict(boolean) strict} PVL syntax rules.
*/
public static boolean
	Strict_Default				= false;
private boolean
	_Strict_					= Strict_Default;

/**	{@link #Verbatim_Strings(boolean) Verbatim strings} default.
*/
public static boolean
	Verbatim_Strings_Default	= false;
private boolean
	_Verbatim_Strings_			= Verbatim_Strings_Default;

/**	The default {@link #Indent_Level(int) statement base indent level}
	(none if negative).
*/
public static int
	Indent_Level_Default		= 0;
private int
	_Aggregate_Indent_Level_	= Indent_Level_Default;

/**	The default for applying {@link #Indent_Arrays(boolean) array indenting}.
*/
public static boolean
	Indent_Arrays_Default		= true;
private boolean
	_Array_Indent_				= Indent_Arrays_Default;

//	Output destination.
private Writer
	_Writer_					= null;

/**	The indent character(s).
<P>
	This String is written once for each {@link #Indent_Level(int)
	indent level}. The default value is the tab character. A
	suitable alternative could be four space characters.
*/
public static String
	INDENT						= "\t";

/**	The platform's native line separator (newline) String.
<P>
	This String is obtained from System.getProperty ("line.separator"),
	or "\n" is used if the System property can't be obtained.
*/
public static String
	NEW_LINE;
static
	{
	NEW_LINE = System.getProperty ("line.separator");
	if (NEW_LINE == null)
		NEW_LINE = "\n";
	}

//	When throwing an exception is inappropriate.
private PVL_Exception
	_First_Warning_				= null,
	_Last_Warning_				= null;

/**	The default for returning the {@link #First_Warning(boolean) first
	warning} (or {@link #Last_Warning(boolean) last warning}).
*/
public static boolean
	First_Warning_Default		= true;
private boolean
	_Use_First_Warning_			= First_Warning_Default;


//	DEBUG control.
private static final int
	DEBUG_OFF			= 0,
	DEBUG_SETUP			= 1 << 0,
	DEBUG_PARAMETER		= 1 << 1,
	DEBUG_VALUE			= 1 << 2,
	DEBUG_COMMENTS		= 1 << 3,
	DEBUG_WRITE			= 1 << 4,
	DEBUG_ALL			= -1,

	DEBUG				= DEBUG_OFF;


/*==============================================================================
	Constructors
*/
/**	Creates a Lister using a Writer as the destination for PVL statements.
<P>
	@param	writer	The Writer destination for characters.
	@see	#Set_Writer(Writer)
*/
public Lister
	(
	Writer		writer
	)
{Set_Writer (writer);}

/**	Creates a Lister using an OutputStream as the destination for PVL
	statements.
<P>
	@param	output_stream	The OutputStream destination for characters.
	@throws	PVL_Exception	From <CODE>Set_Writer</CODE>
	@see	#Set_Writer(OutputStream)
*/
public Lister
	(
	OutputStream	output_stream
	)
	throws PVL_Exception
{Set_Writer (output_stream);}

/**	Creates a Lister using a File as the destination for PVL
	statements.
<P>
	@param	file	The File destination for characters.
	@throws	PVL_Exception	From <CODE>Set_Writer</CODE>
	@see	#Set_Writer(File)
*/
public Lister
	(
	File		file
	)
	throws PVL_Exception
{Set_Writer (file);}

/**	Creates a Lister with the default (System.out) destination for PVL
	statements.
*/
public Lister ()
{Set_Writer ((Writer)null);}


/*------------------------------------------------------------------------------
*/
/**	Sets the Writer where the Lister will send characters.
<P>
	<b>N.B.</b>: The US-ASCII character encoding is used.
<P>
	@param	writer	The Writer destination for Lister output. If null,
		then a BufferedWriter constructed from an OutputStreamWriter
		using System.out is provided.
	@return	This Lister.
	@throws	IllegalStateException	If the {@link Parser#CHARACTER_ENCODING}
		(US-ASCII) is not supported.
*/
public Lister Set_Writer
	(
	Writer		writer
	)
{
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println (">-< Lister.Set_Writer");
_Writer_ = writer;
if (_Writer_ == null)
	{
	if ((DEBUG & DEBUG_SETUP) != 0)
		System.out.println ("    Using default Writer");
	try {_Writer_ = new BufferedWriter
		(new OutputStreamWriter (System.out, Parser.CHARACTER_ENCODING));}
	catch (UnsupportedEncodingException exception)
		{
		IllegalStateException
			throwable = new IllegalStateException (ID + NEW_LINE
			+ "Unable to use the " + Parser.CHARACTER_ENCODING
				+ " character encoding.");
		throwable.initCause (exception);
		throw throwable;
		}
	}
return this;
}

/**	Gets the current destination for Lister characters.
<P>
	@return	The current Writer for the Lister.
	@see	#Set_Writer(Writer)
*/
public Writer Get_Writer ()
{return _Writer_;}

/**	Sets the Writer where the Lister will send characters by
	constructing an OutputStreamWriter from the specified OutputStream
	and wrapping this in a BufferedWriter for efficiency.
<P>
	<b>N.B.</b>: The US-ASCII character encoding is used.
<P>
	@param	output_stream	The OutputStream destination for Lister
		output. If null, System.out is used.
	@return	This Lister.
	@throws	IllegalStateException	If the {@link Parser#CHARACTER_ENCODING}
		(US-ASCII) is not supported.
*/
public Lister Set_Writer
	(
	OutputStream	output_stream
	)
{
if (output_stream == null)
	output_stream = System.out;
try {Set_Writer (new BufferedWriter
		(new OutputStreamWriter (output_stream, Parser.CHARACTER_ENCODING)));}
catch (UnsupportedEncodingException exception)
	{
	IllegalStateException
		throwable = new IllegalStateException (ID + NEW_LINE
		+ "Unable to use the " + Parser.CHARACTER_ENCODING
			+ " character encoding.");
	throwable.initCause (exception);
	throw throwable;
	}
return this;
}

/**	Sets the Writer where the Lister will send characters by constructing
	a FileWriter from the specified File and wrapping this in a BufferedWriter
	for efficiency.
<P>
	@param	file	The File destination for Lister output.
	@return	This Lister.
	@throws	PVL_Exception	A FILE_IO error condition will occur if the
		File refers to a directory or a file for which write permission
		is not provided. Any other IOException that occurs while
		constructing the FileWriter will also be converted into a
		PVL_Exception.FILE_IO exception.
*/
public Lister Set_Writer
	(
	File		file
	)
	throws PVL_Exception
{
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println
		(">>> Lister.Set_Writer: File " + file.getName ());
if (file.isDirectory ())
	throw new PVL_Exception
		(
		ID, PVL_Exception.FILE_IO,
		"The File pathname is for a directory - "
			+ file.getAbsolutePath ()
		);
if (! file.canWrite ())
	throw new PVL_Exception
		(
		ID, PVL_Exception.FILE_IO,
		"Can't write the file - "
			+ file.getAbsolutePath ()
		);
try {Set_Writer (new BufferedWriter (new FileWriter (file)));}
catch (IOException exception)
	{
	throw new PVL_Exception
		(
		ID, PVL_Exception.FILE_IO,
		"Can't access the file - "
			+ file.getAbsolutePath ()
		+ ((exception.getMessage () == null) ?
			"" : "\n" + exception.getMessage ())
		);
	}
return this;
}


/*==============================================================================
	Accessors:
*/
/**	Enables or disables strict PVL syntax rules in the Lister.
<P>
	The default is controlled by the {@link #Strict_Default} value.
<P>
	@param	strict	true if strict rules are applied; false otherwise.
	@return	This Lister.
*/
public Lister Strict
	(
	boolean		strict
	)
{
_Strict_ = strict;
return this;
}

/**	Tests if the Lister will enforce strict PVL syntax rules.
<P>
	@return	true if the Lister will enforce strict syntax rules;
		false otherwise.
*/
public boolean Strict ()
{return _Strict_;}


/**	Enable or disable verbatim quoted strings.
<P>
	Verbatim strings are listed exactly as they occur.
<P>
	With verbatim strings disabled Parameter names, <CODE>STRING</CODE>
	Values, and units description strings will have special characters
	translated to escape sequences. However, in this case those
	sections of these strings that are bracketed by {@link
	Parser#VERBATIM_STRING_DELIMITERS
	<CODE>VERBATIM_STRING_DELIMITERS</CODE>} will not be translated,
	though the <CODE>VERBATIM_STRING_DELIMITERS</CODE> will be removed.
<P>
	The default is controlled by the {@link #Verbatim_Strings_Default}
	value.
<P>
	@param	verbatim	true if strings are to be written verbatim;
		false if special character translation is to be applied.
	@return	This Lister.
	@see	#translate_to_escape_sequences(String_Buffer)
*/
public Lister Verbatim_Strings
	(
	boolean		verbatim
	)
{
_Verbatim_Strings_ = verbatim;
return this;
}

/**	Tests if verbatim strings mode is enabled.
<P>
	@return	true if verbatim strings mode is enabled; false otherwise.
	@see	#Verbatim_Strings(boolean)
*/
public boolean Verbatim_Strings ()
{return _Verbatim_Strings_;}


/**	Sets the base indenting level when listing PVL statements.
<P>
	The default base level is 0. However, the default is controlled by
	the {@link #Indent_Level_Default} value. Aggregate Parameter
	contents and subarrays of array Values are idented an additional
	level from their containing statement listing. If the base level is
	negative, indenting is disabled.
<P>
	The default base level is 0. However, the default is controlled by the
	{@link #Indent_Level_Default} value.
<P>
	@param	level	The base indenting level.
	@return	This Lister.
	@see	#Indent(boolean)
	@see	#Indent_Arrays(boolean)
*/
public Lister Indent_Level
	(
	int			level
	)
{
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println (">-< Lister.Indent_Level: "+ level);
_Aggregate_Indent_Level_ = level;
return this;
}

/**	Gets the current base indent level.
<P>
	@return	The current base indent level.
	@see	#Indent_Level(int)
*/
public int Indent_Level ()
{return _Aggregate_Indent_Level_;}

/**	Enables or disables indenting of PVL statement listings.
<P>
	When enabling indenting, if indenting was previously disabled
	the default (0) base level is set. However, if indenting was
	previously enabled the current level remains unchanged.
<P>
	@param	indent	true to enable indenting; false to disable it.
	@return	This Lister.
	@see	#Indent_Level(int)
*/
public Lister Indent
	(
	boolean		indent
	)
{
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println (">-< Lister.Indenting: " + indent);
if (indent)
	{
	if (_Aggregate_Indent_Level_ < 0)
		_Aggregate_Indent_Level_ = 0;
	}
else
	_Aggregate_Indent_Level_ = -1;
return this;
}

/**	Enables or disables indenting of Array Value listings.
<P>
	Normally indenting of Array Values is controlled by the indent
	level of PVL statement listings; with the listing of the Array, and
	each Array Value within the Array, starting on the next line at the
	next indent level. If indenting for PVL statements (i.e. Aggregate
	Parameters) is disabled, Arrays will be indented from level 0.
	However, Array indenting may be disabled even though PVL statement
	indenting remains enabled, in which case all Values of an Array
	will be written on one line.
<P>
	The default is controlled by the {@link #Indent_Arrays_Default} value.
<P>
	@param	indent	true to enable Array indenting; false to disable
		it.
	@return	This Lister.
	@see	#Indent(boolean)
	@see	#Indent_Level(int)
*/
public Lister Indent_Arrays
	(
	boolean		indent
	)
{
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println (">-< Lister.Indent_Arrays: " + indent);
_Array_Indent_ = indent;
return this;
}

/**	Tests if Array Values will be indented.
<P>
	@return	true if Arrays Values will be indented; false otherwise.
	@see	#Indent_Arrays(boolean)
*/
public boolean Indent_Arrays ()
{return _Array_Indent_;}

/**	Gets the current warning status.
<P>
	When conditions are encountered that are unusual enough to warrant
	attention, but not an error condition that would prevent successful
	processing which would cause an exception to be thrown, a warning
	condition is registered. The warning is in the form of a
	PVL_Exception that was not thrown. The current warning status is
	either the {@link #First_Warning(boolean)
	<CODE>First_Warning</CODE>} or the {@link #Last_Warning(boolean)
	<CODE>Last_Warning</CODE>} since a {@link #Reset_Warning()
	<CODE>Reset_Warning</CODE>}.
<P>
	@return	The current warning status as a PVL_Exception object,
		or null if no warning condition is registered.
	@see	PVL_Exception
	@see	#First_Warning(boolean)
	@see	#Last_Warning(boolean)
	@see	#Reset_Warning()
*/
public PVL_Exception Warning ()
{
if (_Use_First_Warning_)
	return _First_Warning_;
return _Last_Warning_;
}

/**	Clears any warning status so that the <CODE>Warning</CODE> method
	will return null until the next warning condition occurs.
<P>
	@return	This Lister.
	@see	#Warning()
*/
public Lister Reset_Warning ()
{
_First_Warning_ = null;
_Last_Warning_ = null;
return this;
}

/**	Enables or disables returning the first warning that occurs as the
	current warning status.
<P>
	The first warning is one that occurs when the current warning
	status is null. If the use of the first warning is disabled the
	last warning that occurred will be returned instead. The default is
	to use the first warning. However, the default is controlled by the
	{@link #First_Warning_Default} value.
<P>
	<B>N.B.</B>: Enabling the return of the first warning is the same
	as disabling the {@link #Last_Warning(boolean) return of the last
	warning}.
<P>
	@param	first	true to enable returning the first warning status;
		false to return the last warning that occurred as the current
		warning status.
	@return	This Lister.
	@see	#Last_Warning(boolean)
	@see	#Warning()
	@see	#First_Warning()
	@see	#Reset_Warning()
*/
public Lister First_Warning
	(
	boolean		first
	)
{
_Use_First_Warning_ = first;
return this;
}

/**	Returns the first warning since the last <CODE>Reset_Warning</CODE>.
<P>
	@return	The first warning status as a PVL_Exception object,
		or null if no warning condition is registered.
	@see	PVL_Exception
	@see	#Warning()
	@see	#First_Warning(boolean)
	@see	#Reset_Warning()
*/
public PVL_Exception First_Warning ()
{return _First_Warning_;}

/**	Enables or disables returning the last warning that occurs as the
	current warning status.
<P>
	The last warning is the most recent one that ocurred, regardless of
	any previous warning conditions that may have occured, without an
	intervening <CODE>Reset_Warning</CODE>.
<P>
	<B>N.B.</B>: Enabling the return of the last warning is the same
	as disabling the {@link #First_Warning(boolean) return of the first
	warning}.
<P>
	@param	last	true to enable returning the last warning status;
		false to return the first warning condition that occurred as
		the current warning status.
	@return	This Lister.
	@see	#First_Warning(boolean)
	@see	#Warning()
	@see	#Last_Warning()
	@see	#Reset_Warning()
*/
public Lister Last_Warning
	(
	boolean		last
	)
{
_Use_First_Warning_ = ! last;
return this;
}

/**	Returns the last warning since a <CODE>Reset_Warning</CODE>.
<P>
	@return	The last warning status as a PVL_Exception object,
		or null if no warning condition is registered.
	@see	PVL_Exception
	@see	#Warning()
	@see	#Last_Warning(boolean)
	@see	#Reset_Warning()
*/
public PVL_Exception Last_Warning ()
{return _Last_Warning_;}


/*==============================================================================
	Write:
*/
/**	Writes the Parameter in PVL syntax.
<P>
	The PVL syntax that is written follows this pattern:
<P>
	<BLOCKQUOTE><B>
	<I>Comments</I><BR>
	<I>Name</I> = <I>Value</I>
	</B></BLOCKQUOTE>
<P>
	The comments string, if there is one, is written by the
	<CODE>Write_Comments</CODE> method.
<P>
	The Parameter's name is written next beginning on a new line. If
	indenting is enabled, the name will be preceeded by the current
	indent count number of horizontal tab characters. The name String
	itself is usually provided by the Parameter's <CODE>Name</CODE>
	method. However, if the Parser's <CODE>Special_Name</CODE> method
	determines that the Parameter's classification is associated with a
	special name (as it is for all aggregate and <CODE>END</CODE>
	Parameters), then that will be written instead. Unless {@link
	#Verbatim_Strings <CODE>Verbatim_Strings</CODE>} are enabled,
	escape sequences are substituted for special (format control)
	characters and {@link Parser#TEXT_DELIMITER
	<CODE>Parser.TEXT_DELIMITER</CODE>}, {@link Parser#SYMBOL_DELIMITER
	<CODE>Parser.SYMBOL_DELIMITER</CODE>} and backslash ('\')
	characters are escaped (with a preceding backslash character). If
	the name contains {@link Parser#WHITESPACE whitespace} or {@link
	Parser#RESERVED_CHARACTERS reserved} characters or is empty then it
	will be quoted (with {@link Parser#TEXT_DELIMITER
	<CODE>Parser.TEXT_DELIMITER</CODE>} characters). <B>Note</B>: When
	<CODE>Strict</CODE> PVL has been specified, Parameter's that begin
	an aggregate list will have the special name
	<CODE>BEGIN_GROUP</CODE> or <CODE>BEGIN_OBJECT</CODE>, otherwise
	the <CODE>BEGIN_</CODE> portion will not be included.
<P>
	The Parameter's value is written next, if the Parameter is not a
	token or classified as <CODE>UNKNOWN</CODE>. It is preceeded by the
	{@link Parser#PARAMETER_NAME_DELIMITER equals} character surrounded
	by space characters. For an assignment, the <CODE>Write
	Value)</CODE> method completes the output of a parameter, including
	any line termination, and the method returns. If there is no Value,
	an empty quoted String is written by default. For aggregates, the
	name of the Parameter is its value; i.e. in this case the special
	name for the specific classification substitutes for the usual
	name, and the usual name appears as the value (yes, this seems
	strange, but take it up with the committee that wrote the
	specification for the PVL syntax).
<P>
	The Parameter line is completed by writing a new-line character.
	However, if <CODE>Strict</CODE> PVL has been specified, then the
	{@link Parser#STATEMENT_END_DELIMITER
	<CODE>Parser.STATEMENT_END_DELIMITER</CODE>} character followed by
	the {@link Parser#LINE_BREAK <CODE>Parser.LINE_BREAK</CODE>}
	sequence is written instead.
<P>
	When the Parameter being written is an aggregate, the indent level
	(if indenting is enabled) is incremented and then each Parameter in
	the aggregate list is recursively written. At the end of the list
	the indent level is decremented (if indenting is enabled) and an
	<CODE>END_GROUP</CODE> or <CODE>END_OBJECT</CODE> Parameter,
	corresponding to the classification of the aggregate containing the
	list, is written. If an <CODE>END</CODE> Parameter is encountered
	in the aggregate list, then the list is terminated at that point.
	If an <CODE>END_PVL</CODE> Parameter is encountered, then the
	current list is terminated as well as the lists of all containing
	aggregates; i.e. all Parameter lists end at that point.
<P>
	As a special case, when the Parameter being written is an aggregate
	with the name {@link Parser#CONTAINER_NAME
	<CODE>Parser.CONTAINER_NAME</CODE>} this "container" Parameter is
	not written, but the Parameters in it's aggregate list are written
	normally. This special case only applies to the first Parameter
	inspected by the <CODE>Write (Parameter)</CODE> method, not
	aggregate list entries; i.e. only the top of the hierarchy is
	considered as a potential special case.
<P>
	Normally the special <CODE>END_PVL</CODE> Parameter is not written
	at the end of the listing. This is so that several Parameters can
	be written to a file before the application provides the
	<CODE>END_PVL</CODE> Parameter formally expected at the end of a
	PVL sequence. However, when the Parameter being written is the top
	of the output hierarchy and has the special
	<CODE>Parser.CONTAINER_NAME</CODE>, then the <CODE>END_PVL</CODE>
	Parameter is automatically provided to reflect the end of the
	container's parameters; i.e. what is written should be logically
	the equivalent to what was read by the Parser.
<P>
	@param	parameter	The Parameter to write.
	@return	The total number of bytes written.
	@throws	PVL_Exception
		<DL>
		<DT><CODE>BAD_ARGUMENT</CODE>
			<DD>If an element in an aggregate list is not a Parameter.
		</DL>
	@throws	IOException	From the Writer's <CODE>write</CODE> method.
	@see	#Write_Comments(String, int)
	@see	Parameter#Name()
	@see	Parser#Special_Name(int)
	@see	#translate_to_escape_sequences(String_Buffer)
	@see	#Write(Value)
*/
public int Write
	(
	Parameter	parameter
	)
	throws PVL_Exception, IOException
{
if ((DEBUG & DEBUG_PARAMETER) != 0)
	System.out.println (">>> Lister.Write (Parameter)");

/*	Flag when an END_XXX Parameter is encountered.

	Normally END Parameters are not included in Parameter lists; they
	are implicit at the end of the list. However, a user might decide
	to provide them; for example to manually mark the end of an
	existing list (or because it is considered formally correct for
	some application dependent reason). So when they are encountered
	processing of a Parameter list stops.

	There are two kinds of END Parameters: One that ends an aggregate
	Parameter list (Is_End_Aggregate) ends the output of Parameters in
	the current aggregate. One that ends all PVL statements (Is_End_PVL)
	for all nested aggregates.

	An array is used to provide a pass-back value:

		 0 - No END
		 1 - END_AGGREGATE
		-1 = END_PVL
*/
int[] end = {0};
int total = write (parameter, _Aggregate_Indent_Level_, true, end);

if ((DEBUG & DEBUG_PARAMETER) != 0)
	System.out.println
		("<<< Lister.Write (Parameter): total written = " + total);
return total;
}

/*	The method that controls Parameter writing.
	The interface is designed to provide for recursion
	with start and end information.
*/
private int write
	(
	Parameter	parameter,
	int			level,
	boolean		start,
	int[]		end
	)
	throws PVL_Exception, IOException
{
if ((DEBUG & DEBUG_PARAMETER) != 0)
	System.out.println (">>> Lister.write (Parameter): level " + level);
int
	total_written = 0;
boolean
	top_level_container =
		(
		start &&
		parameter.Is_Aggregate () &&

		/*	The aggregate a Parser may create to hold all input Parameters.

			This is an "artificial" Parameter that did not exist in the
			input stream, so only its contents are written. If,
			however, the user renames this Parameter, then it is
			written.
		*/
		parameter.Name ().equals (Parser.CONTAINER_NAME)
		);
if ((DEBUG & DEBUG_PARAMETER) != 0)
	System.out.println
		("    Lister.write: top_level_container = " + top_level_container);

//	Comments:
Write_Comments (parameter.Comments (), level);

if (! top_level_container)
	{
	/*	Name:
	*/
	String
		name;

	total_written += indent (level);

	//	Check for special names for special classifications.
	if ((DEBUG & DEBUG_PARAMETER) != 0)
		System.out.println
			("    Lister.write: Classification "
				+ parameter.Classification_Name ());
	if ((name = Parser.Special_Name (parameter.Classification ())) == null)
		name = parameter.Name ();
	else if (_Strict_)
		{
		if (! name.toUpperCase ().startsWith ("BEGIN_"))
			name = "BEGIN_" + name;
		}
	else if (name.toUpperCase ().startsWith ("BEGIN_"))
			name = name.substring (name.indexOf ('_') + 1);

	total_written += write_name (name);

	/*	Data:
	*/
	if (! parameter.Is_Token () && ! parameter.Is_Unknown ())
		{
		total_written += write (' ');
		total_written += write (Parser.PARAMETER_NAME_DELIMITER);
		total_written += write (' ');

		if (parameter.Is_Aggregate ())
			//	The Parameter name is the value of aggregates.
			total_written += write_name (parameter.Name ());
		else if (parameter.Value () != null)
			/*
				The output of the Value completes the Parameter output.
				This includes the PVL statement termination.
			*/
			return total_written +=
				write
					(parameter.Value (),
					(level < 0 || top_level_container) ? level : level + 1,
					true
					);
		else
			total_written += write_name (null);
		}

	/*	PVL statement termination.
	*/
	if (_Strict_)
		total_written += write (Parser.STATEMENT_END_DELIMITER);
	total_written += new_line ();
	}

if (parameter.Is_Begin_Aggregate ())
	{
	//	Recursively write each Parameter in the aggregate.
	if ((DEBUG & DEBUG_PARAMETER) != 0)
		System.out.println
			("    Lister.write: writing aggregate parameters.");
	if (parameter.List () != null)
		{
		ListIterator
			list = parameter.List ().listIterator ();
		while (list.hasNext ())
			{
			Object object = list.next ();
			if (! Parameter.Is_Parameter (object))
				throw new PVL_Exception
					(
					ID, PVL_Exception.BAD_ARGUMENT,
					"In the AGGREGATE Parameter named \""
						+ parameter.Name () + "\"\n"
					+ "  element " + list.previousIndex ()
						+ " is an object of class "
						+ object.getClass ().getName () + "."
					);

			//	User specified end.
			if (((Parameter)object).Is_End ())
				{
				if (((Parameter)object).Is_End_Aggregate ())
					end[0] = 1;
				else
					end[0] = -1;
				break;
				}

			total_written +=
				write
					(
					(Parameter)object,
					(level < 0 || top_level_container) ? level : level + 1,
					false,
					end
					);

			if (end[0] < 0)
				break;
			}
		}

	total_written += indent (level);
	if (! start ||
		! top_level_container)
		{
		//	Add the end aggregate Parameter.
		total_written += write_name
			(Parser.Special_Name (parameter.Classification () | Parameter.END));
		if (_Strict_)
			total_written += write (Parser.STATEMENT_END_DELIMITER);
		}
	if (top_level_container ||
		(start && end[0] < 0))
		{
		//	The END.
		if (! start ||
			! top_level_container)
			//	Add the NL after the preceeding end of aggregate token.
			total_written += new_line ();

		total_written += write_name
			(Parser.Special_Name (Parameter.END_PVL));
		if (_Strict_)
			total_written += write (Parser.STATEMENT_END_DELIMITER);
		}
	total_written += new_line ();
	}

if ((DEBUG & DEBUG_PARAMETER) != 0)
	System.out.println
		("<<< Lister.write (Parameter): total_written = " + total_written);
return total_written;
}


/**	Writes a PVL comments sequence.
<P>
	Normally comments are attached to a Parameter with the
	<CODE>Comments</CODE> method and they are written as part of the
	Parameter. Comments may be inserted virtually anywhere in a PVL
	listing, and this method is provided for such ad hoc commenting.
<P>
	Comments are delimited by {@link Parser#COMMENT_START_DELIMITERS
	Parser.COMMENT_START_DELIMITERS} and {@link
	Parser#COMMENT_END_DELIMITERS Parser.COMMENT_END_DELIMITERS}
	(C-style) sequences. When <CODE>Strict</CODE> PVL syntax is
	specified each line of the comments String (a line being delmited
	by a new-line character) is separated into individual single line
	comments. Otherwise the comments String is simply output between
	the start and end sequences as-is.
<P>
	If indenting is enabled each comment start sequence will be
	preceeded by the current indent count number of horizontal tab
	characters. Unless <CODE>Strict</CODE> syntax has been specified, 
	the comment end sequence is preceeded by a new-line character and
	any indenting. If indenting is enabled the end of the comments
	sequence is completed by writing a new-line character, or the
	{@link Parser#LINE_BREAK <CODE>Parser.LINE_BREAK</CODE>} sequence
	for <CODE>Strict</CODE> PVL.
<P>
	@param	comments	The String to be written in a comments sequence.
		If comments is null, nothing is written.
	@param	level		The indent level to use.
	@return	The total number of bytes written.
	@throws	IOException	From the OutputStream <CODE>write</CODE> method.
*/
public int Write_Comments
	(
	String		comments,
	int			level
	)
	throws IOException
{
if ((DEBUG & DEBUG_COMMENTS) != 0)
	System.out.println (">>> Lister.Write_Comments: level " + level);
int
	total_written = 0;

if (comments != null)
	{
	if ((DEBUG & DEBUG_COMMENTS) != 0)
		System.out.println ("    Lister.Write_Comments: " + comments);
	int
		count = 0,
		length = comments.length ();
	do
		{
		//	Prefix.
		total_written += indent (level);
		total_written += write (Parser.COMMENT_START_DELIMITERS);

		if (count < length &&
			comments.charAt (count) != '\n')
			total_written += write ("  ");

		//	Body.
		char
			character = 0;
		while (count < length)
			{
			if ((character = comments.charAt (count++)) == '\n' &&
				_Strict_)
				break;
			if (character == '\n')
				{
				total_written += new_line ();
				if (count < length &&
					comments.charAt (count) != '\n')
					{
					total_written += indent (level);
					for (int shift =
								Parser.COMMENT_START_DELIMITERS.length () + 2;
							 shift > 0;
						   --shift)
						total_written += write (' ');
					}
				}
			else
				total_written += write (character);
			}

		//	Postfix.
		if (level >= 0 && ! _Strict_)
			{
			//	Must be at the end of the comments string.
			total_written += new_line ();
			total_written += indent (level);
			}
		total_written += write (Parser.COMMENT_END_DELIMITERS);
		if (level >= 0)
			/*	Only when indenting.
				Turn off indenting for in-line comments.
			*/
			total_written += new_line ();
		}
		while (count < length);
	}
if ((DEBUG & DEBUG_COMMENTS) != 0)
	System.out.println
		("<<< Lister.Write_Comments: total written = " + total_written);
return total_written;
}

/**	Writes a PVL comments sequence at the current indent level.
<P>
	@param	comments	The String to be written in a comments sequence.
		If comments is null, nothing is written.
	@return	The total number of bytes written.
	@throws	IOException	From the Writer's <CODE>write</CODE> method.
	@see	#Write_Comments(String, int)
*/
public int Write_Comments
	(
	String		comments
	)
	throws IOException
{return Write_Comments (comments, _Aggregate_Indent_Level_);}


/*..............................................................................
*/
private int write_name
	(
	String		name
	)
	throws PVL_Exception, IOException
{
if (name == null)
	name = "";
int
	index;
String_Buffer
	representation = new String_Buffer (name);
if ((DEBUG & DEBUG_PARAMETER) != 0)
	System.out.println (">>> Lister.write_name: " + name);

if (! _Verbatim_Strings_)
	translate_to_escape_sequences (representation);

if (representation.length () == 0)
	{
	//	An empty name.
	Warning
		(
		PVL_Exception.ILLEGAL_SYNTAX,
		"Empty Parameter name."
		);
	if (_Strict_)
		throw _Last_Warning_;
	//	Add quotes.
	representation
		.append (Parser.TEXT_DELIMITER)
		.append (Parser.TEXT_DELIMITER);
	}
else if ((index = representation.skip_until
		(0, Parser.RESERVED_CHARACTERS)) >= 0)
	{
	if ((DEBUG & DEBUG_PARAMETER) != 0)
		System.out.println
			("    Lister.write_name: reserved character in name.");
	//	There's a reserved character in the String.
	Warning
		(
		PVL_Exception.ILLEGAL_SYNTAX,
		"Reserved character at index " + index
			+ " of the Parameter named \"" + representation.toString () + "\""
		);
	if (_Strict_)
		throw _Last_Warning_;
	//	Add quotes to protect the reserved characters.
	representation
		.insert (0, Parser.TEXT_DELIMITER)
		.append (Parser.TEXT_DELIMITER);
	}

write (representation.toString ());
if ((DEBUG & DEBUG_PARAMETER) != 0)
	System.out.println
		("<<< Lister.write_name: total written = "
			+ representation.length () + " - " + representation);
return representation.length ();
}


/*------------------------------------------------------------------------------
*/
/**	Writes the Value in PVL syntax.
<P>
	The PVL syntax that is written follows this pattern:
<P>
	<BLOCKQUOTE>
	[ <B>(</B> | <B>{</B> ] <B><I>Datum</I></B> [ <B>< <I>Units</I> ></B> ] [ <B>, <I>Datum</I></B> [...] ] [ <B>)</B> | <B>}</B> [ <B>< <I>Units</I> ></B> ] ]
	</BLOCKQUOTE>
<P>
	The Value datum is always written. If the Value is the
	<CODE>UNKNOWN</CODE> type then an empty <CODE>SYMBOL</CODE> type is
	provided by default. If the datum is a <CODE>STRING</CODE> type and
	{@link #Verbatim_Strings <CODE>Verbatim_Strings</CODE>} are
	disabled, escape sequences are substituted for special (format
	control) characters and {@link Parser#TEXT_DELIMITER
	<CODE>Parser.TEXT_DELIMITER</CODE>}, {@link Parser#SYMBOL_DELIMITER
	<CODE>Parser.SYMBOL_DELIMITER</CODE>} and backslash ('\')
	characters are escaped (with a preceding backslash character). If
	the string contains {@link Parser#WHITESPACE whitespace} or {@link
	Parser#RESERVED_CHARACTERS reserved} characters or is empty then it
	will be quoted (with {@link Parser#TEXT_DELIMITER
	<CODE>Parser.TEXT_DELIMITER</CODE>} characters).
<P>
	If the Value is an <CODE>ARRAY</CODE> type then the appropriate
	enclosure start character ({@link Parser#SET_START_DELIMITER
	<CODE>Parser.SET_START_DELIMITER</CODE>} or {@link
	Parser#SEQUENCE_START_DELIMITER
	<CODE>Parser.SEQUENCE_START_DELIMITER</CODE>}) is written before
	each Value in the array Vector is recursively written by this
	method with a space and {@link Parser#PARAMETER_VALUE_DELIMITER
	<CODE>Parser.PARAMETER_VALUE_DELIMITER</CODE>} separation. After
	all the array Vector Values have been written the appropriate
	enclosure end character ({@link Parser#SET_END_DELIMITER
	<CODE>Parser.SET_END_DELIMITER</CODE>} or {@link
	Parser#SEQUENCE_END_DELIMITER
	<CODE>Parser.SEQUENCE_END_DELIMITER</CODE>}) is written.
<P>
	After each Value is written, whether a single datum or an array, it
	is followed by any units description String (preceeded by a space
	character and enclosed in PVL syntax characters ({@link
	Parser#UNITS_START_DELIMITER
	<CODE>Parser.UNITS_START_DELIMITER</CODE>} and {@link
	Parser#UNITS_END_DELIMITER
	<CODE>Parser.UNITS_END_DELIMITER</CODE>}).
<P>
	Array indenting may be enabled. In this mode each nested array is
	written on separate lines indented by tab characters to its nesting
	level. Each subarray starts a new line and increments the nesting
	level. The end of a subarray is followed by a new-line and the
	nesting level is decremented before continuing a previous array
	level (but not if there is no previous array level or the previous
	level has no more Values). The start of an array at level 0 is not
	given indent treatment.
<P>
	In <CODE>Strict</CODE> mode new-line sequences are the {@link
	Parser#LINE_BREAK <CODE>Parser.LINE_BREAK</CODE>} characters;
	normally the native new-line (the line.separator System property)
	is used. Also, <CODE>Strict</CODE> mode results in the {@link
	Parser#STATEMENT_END_DELIMITER
	<CODE>Parser.STATEMENT_END_DELIMITER</CODE>} character preceeding
	the usual new-line that completes the Value output.
<P>
	If an unenclosed <CODE>STRING</CODE> type datum contains any of the
	{@link Parser#RESERVED_CHARACTERS
	<CODE>Parser.RESERVED_CHARACTERS</CODE>} then a warning status is
	set (and thrown in <CODE>Strict</CODE> mode). These characters have
	special meaning in the PVL syntax so writing values containing them
	could confuse a PVL parser that reads them. Unless strict mode is
	in effect, the string is enclosed in {@link Parser#TEXT_DELIMITER
	<CODE>Parser.TEXT_DELIMITER</CODE>} characters to protect them from
	inappropriate interpretation.
<P>
	@param	value	The Value to write.
	@return	The total number of bytes written.
	@throws	PVL_Exception
		<DL>
		<DT><CODE>BAD_ARGUMENT</CODE>
			<DD>If an element in an array is not a Value.
		<DT><CODE>ILLEGAL_SYNTAX</CODE>
			<DD>If a PVL reserved character was found in an unenclosed
				<CODE>STRING</CODE> datum during <CODE>Strict</CODE> writing.
		</DL>
	@throws	IOException	From the OutputStream <CODE>write</CODE> method.
	@see	#translate_to_escape_sequences(String_Buffer)
*/
public int Write
	(
	Value		value
	)
	throws PVL_Exception, IOException
{
if ((DEBUG & DEBUG_VALUE) != 0)
	System.out.println ("\n>>> Lister.Write (Value)");
int total = write (value, _Aggregate_Indent_Level_, true);
if ((DEBUG & DEBUG_VALUE) != 0)
	System.out.println ("<<< Lister.Write (Value): total written = " + total);
return total;
}


private int write
	(
	Value		value,
	int			level,
	boolean		start
	)
	throws PVL_Exception, IOException
{
int
	total_written = 0;

if (value.Is_Array ())
	{
	//	Array listing:
	if (! _Array_Indent_)
		level = -1;	//	Indenting disabled.
	else if (level < 0)
		level = 0;
	if ((DEBUG & DEBUG_VALUE) != 0)
		System.out.println
			("\n    Lister.write (Value): writing array values at level "
			+ level + ".");
	if (level > 0)
		{
		//	Start an indented array on the next line.
		total_written += new_line ();
		total_written += indent (level);
		}
	if (value.Is_Set ())
		total_written += write (Parser.SET_START_DELIMITER);
	else
		total_written += write (Parser.SEQUENCE_START_DELIMITER);

	//	Recursively output each value in the array.
	boolean
		previous_array = false;
	ListIterator
		values = value.listIterator ();
	while (values.hasNext ())
		{
		Object object = values.next ();
		if (! Value.Is_Value (object))
			throw new PVL_Exception
				(
				ID, PVL_Exception.BAD_ARGUMENT,
				"While Writing element " + values.previousIndex ()
					+ " of a " + value.Type_Name () + " Value\n"
				+ "  an object of class "
					+ object.getClass ().getName () + " was found."
				);

		if (previous_array &&
			level >= 0 &&
			! ((Value)object).Is_Array ())
			{
			/*
				The previous value was an array
				and the current value is not an array
				so reposition back to the current indent level.
			*/
			total_written += new_line ();
			total_written += indent (level);
			}

		//	Write the array Value.
		total_written +=
			write
				(
				(Value)object,
				(level < 0) ? level : (level + 1),
				false	//	Within an array.
				);

		if (values.hasNext ())
			{
			total_written += write (Parser.PARAMETER_VALUE_DELIMITER);
			previous_array = ((Value)object).Is_Array ();
			if (! previous_array ||
				level < 0)
				total_written += write (' ');
			}
		}

	if (value.Is_Set ())
		total_written += write (Parser.SET_END_DELIMITER);
	else
		total_written += write (Parser.SEQUENCE_END_DELIMITER);
	}
else
	{
	//	Fundamental Value listing:
	String_Buffer
		representation;
	int
		index;
	if (value.Is_String ())
		{
		representation = new String_Buffer (value.String_Data ());
		if (! _Verbatim_Strings_)
			translate_to_escape_sequences (representation);

		if (value.Is_Text ())
			representation
				.insert (0, Parser.TEXT_DELIMITER)
				.append (Parser.TEXT_DELIMITER);
		else if (value.Is_Symbol ())
			representation
				.insert (0, Parser.SYMBOL_DELIMITER)
				.append (Parser.SYMBOL_DELIMITER);
		else if ((index = representation.skip_until (0,
				Parser.RESERVED_CHARACTERS)) >= 0)
			{
			//	There's a reserved character in the String.
			Warning
				(
				PVL_Exception.ILLEGAL_SYNTAX,
				"Reserved character at index " + index
					+ " of the " + value.Type_Name () + " value \""
					+ representation + "\""
				);
			if (_Strict_)
				throw _Last_Warning_;
			//	Add quotes to protect the reserved characters.
			representation
				.insert (0, Parser.TEXT_DELIMITER)
				.append (Parser.TEXT_DELIMITER);
			}
		}
	else if (value.Is_Unknown ())
		representation =
			new String_Buffer ("");
	else
		representation = new String_Buffer (value.toString ());
	if (representation.length () == 0)
		//	An empty representation needs to be quoted.
		representation
			.insert (0, Parser.TEXT_DELIMITER)
			.append (Parser.TEXT_DELIMITER);

	total_written += write (representation.toString ());
	}

//	Units:
total_written += write_units (value.Units ());

if (start)
	{
	/*	PVL statement termination.
	*/
	if (_Strict_)
		total_written += write (Parser.STATEMENT_END_DELIMITER);
	total_written += new_line ();
	}

if ((DEBUG & DEBUG_VALUE) != 0)
	System.out.println
		("\n<<< Lister.write (Value): total_written = " + total_written);
return total_written;
}


/*..............................................................................
*/
private int write_units
	(
	String		units
	)
	throws PVL_Exception, IOException
{
if (units == null)
	return 0;
int
	index;
String_Buffer
	representation = new String_Buffer (units);

if (! _Verbatim_Strings_)
	translate_to_escape_sequences (representation);

if ((index = representation.skip_until (0, Parser.RESERVED_CHARACTERS)) >= 0)
	{
	//	There's a reserved character in the String.
	Warning
		(
		PVL_Exception.ILLEGAL_SYNTAX,
		"Reserved character at index " + index
			+ " of units \"" + representation + "\""
		);
	if (_Strict_)
		throw _Last_Warning_;
	}

return write
	(
	representation
		.insert (0, ' ')
		.insert (1, Parser.UNITS_START_DELIMITER)
		.append (Parser.UNITS_END_DELIMITER)
		.toString ()
	);
}


/*------------------------------------------------------------------------------
	Primitive writers
*/
private int indent
	(
	int			level
	)
	throws IOException
{
String
	string = "";
while (level-- > 0)
	 string += INDENT;
return write (string);
}

private int new_line ()
	throws IOException
{
String
	EOL;
if (_Strict_)
	EOL = Parser.LINE_BREAK;
else
	EOL = NEW_LINE;
write (EOL);
_Writer_.flush ();
return EOL.length ();
}


private int write
	(
	String	string
	)
	throws IOException
{
if (string == null)
	return 0;
if ((DEBUG & DEBUG_WRITE) != 0)
	System.out.println
		(">-< Lister.write: " + string.length () + " character string -\n"
		 + ">|" + string + "|<");
_Writer_.write (string);
return string.length ();
}


private int write
	(
	int		character
	)
	throws IOException
{
if ((DEBUG & DEBUG_WRITE) != 0)
	System.out.println
		(">-< Lister.write: character >|" + (char)character + "|<");
_Writer_.write (character);
return 1;
}


/*------------------------------------------------------------------------------
*/
/**	Translates special characters in a String_Buffer to their
	corresponding escape sequences.
<P>
	In addition to the special characters translated by the {@link
	String_Buffer#special_to_escape() <CODE>special_to_escape</CODE>}
	method, the {@link Parser#TEXT_DELIMITER
	<CODE>TEXT_DELIMITER</CODE>} and {@link Parser#SYMBOL_DELIMITER
	<CODE>SYMBOL_DELIMITER</CODE>} characters are also escaped. The
	occurance of {@link Parser#VERBATIM_STRING_DELIMITERS
	<CODE>VERBATIM_STRING_DELIMITERS</CODE>} starts a sequence of
	characters that are taken verbatim (they are not translated) up to
	and including the next <CODE>VERBATIM_STRING_DELIMITERS</CODE> or
	the end of the string.
<P>
	@param	string	The String_Buffer to be translated.
	@return	The translated String_Buffer.
	@see	String_Buffer#special_to_escape()
*/
public static String_Buffer translate_to_escape_sequences
	(
	String_Buffer	string
	)
{
if (string == null)
	return string;

int
	first = 0,
	last,
	length,
	delimiters_length = Parser.VERBATIM_STRING_DELIMITERS.length ();
String_Buffer
	section;

//	Find sections delimited by VERBATIM_STRING_DELIMITERS.
for (last = string.index_of (first, Parser.VERBATIM_STRING_DELIMITERS);
	 first < string.length ();
	 last = string.index_of (first = last, Parser.VERBATIM_STRING_DELIMITERS))
	{
	if (last < 0)
		//	This section extends to the end of the string.
		last = string.length ();
	section = new String_Buffer (string.substring (first, last));
	length = section.length ();

	//	Convert special characters to escape sequences.
	section
		.special_to_escape ()
		.replace (0, new Character (Parser.TEXT_DELIMITER).toString (),
			"\\" + Parser.TEXT_DELIMITER)
		.replace (0, new Character (Parser.SYMBOL_DELIMITER).toString (),
			"\\" + Parser.SYMBOL_DELIMITER);
	if (section.length () != length)
		{
		//	The section changed; replace it in the string.
		string.replace (first, last, section.toString ());
		last = first + section.length ();
		}

	//	Skip the next, verbatim, section.
	if ((last += delimiters_length) >= string.length () ||
		(last = string.index_of (last, Parser.VERBATIM_STRING_DELIMITERS)) < 0 ||
		(last += delimiters_length) == string.length ())
		//	No more sections.
		break;
	}
return string;
}


/*------------------------------------------------------------------------------
*/
private void Warning
	(
	String		description,
	String		explanation
	)
{
_Last_Warning_ =
	new PVL_Exception
		(
		ID,
		description,
		explanation
		);
if (_First_Warning_ == null)
	_First_Warning_ = _Last_Warning_;
}



}	//	class Lister
