/*
 * Copyright 2013 Canonical Ltd.
 *
 * Authors:
 * Michael Frey: michael.frey@canonical.com
 * Matthew Fischer: matthew.fischer@canonical.com
 * Seth Forshee: seth.forshee@canonical.com
 *
 * This file is part of powerd.
 *
 * powerd is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * powerd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <glib.h>
#include <glib-object.h>

#include "powerd.h"
#include "powerd-internal.h"
#include "log.h"

enum bl_state {
    BL_OFF,
    BL_DIM,
    BL_BRIGHT,
    BL_AUTO,

    NUM_BL_STATES
};
static enum bl_state target_bl_state = BL_BRIGHT;

/* Treat "don't care" display state as off */
#define DISPLAY_STATE_OFF POWERD_DISPLAY_STATE_DONT_CARE

/* Autobrightness only enabled when bright && !disabled */
#define AB_ENABLED_MASK (POWERD_DISPLAY_FLAG_BRIGHT | \
                         POWERD_DISPLAY_FLAG_DISABLE_AUTOBRIGHTNESS)
#define AB_ENABLED      POWERD_DISPLAY_FLAG_BRIGHT

static int user_brightness;
static gboolean user_ab_enabled;

/* Assume screen is off to start; we will force it on */
static uuid_t internal_request_cookie;

/* Currently requested state of the display */
struct powerd_display_request internal_state = {
    .state = DISPLAY_STATE_OFF,
    .flags = 0,
};

/*
 * When using the proximity sensor, our actual state of the screen
 * may not match the requested state in internal_state. Therefore
 * this variable keeps track of the actual state of the screen.
 */
enum powerd_display_state actual_screen_state = DISPLAY_STATE_OFF;

/*
 * Screen state overrides. These are used when powerd needs to force
 * the screen off regardless of the requested display state.
 */
static unsigned screen_off_overrides;

static const char *display_state_strs[POWERD_NUM_DISPLAY_STATES] = {
    [POWERD_DISPLAY_STATE_DONT_CARE]    = "off",
    [POWERD_DISPLAY_STATE_ON]           = "on",
};

static const char *display_state_to_str(enum powerd_display_state state)
{
    if (state >= POWERD_NUM_DISPLAY_STATES)
        return "invalid";
    return display_state_strs[state];
}

static gboolean ab_enabled(guint32 flags)
{
    if (!powerd_autobrightness_available())
        return FALSE;
    if (!user_ab_enabled)
        return FALSE;
    return (flags & AB_ENABLED_MASK) == AB_ENABLED;
}

static void set_backlight(enum bl_state state)
{
    static enum bl_state current_state = BL_BRIGHT;
    static int restore_brightness = 0;

    if (state >= NUM_BL_STATES) {
        powerd_warn("Unknown backlight state %d\n", state);
        return;
    }
    if (state != BL_AUTO)
        powerd_autobrightness_disable();

    /*
     * When we're enabling autobrightness it takes a second to
     * settle on a brightness level after enabling. This delay
     * becomes very noticible when going from the dim to bright
     * state. To avoid this lag, save off the current brightness
     * any time the state goes from BRIGHT or AUTO and restore
     * it if transitioning back to AUTO.
     */
    if (current_state >= BL_BRIGHT && state < BL_BRIGHT)
        restore_brightness = powerd_get_brightness();

    switch (state) {
    case BL_OFF:
        powerd_set_brightness(0);
        break;
    case BL_DIM:
        powerd_dim_screen();
        break;
    case BL_BRIGHT:
        powerd_set_user_brightness(user_brightness);
        break;
    case BL_AUTO:
        if (current_state < BL_BRIGHT && restore_brightness > 0)
            powerd_set_brightness(restore_brightness);
        powerd_autobrightness_enable();
        break;
    default:
        /* default case is to satisfy static analysis tools, should
         * never actually get here */
        powerd_error("Unknwown backlight state %d\n", state);
        return;
    }
    current_state = state;
}

gboolean display_set_power_mode(int display, const char *power_mode)
{
    GError *error = NULL;
    GDBusProxy *unity_proxy = NULL;

    powerd_debug("display_set_power_mode(%s)", power_mode);

    if (strcmp(power_mode, "on") == 0)
        powerd_hal_signal_activity();

    unity_proxy = g_dbus_proxy_new_for_bus_sync(G_BUS_TYPE_SYSTEM,
                G_DBUS_PROXY_FLAGS_NONE,
                NULL,
                "com.canonical.Unity.Screen",
                "/com/canonical/Unity/Screen",
                "com.canonical.Unity.Screen",
                NULL,
                &error);

    if (error != NULL) {
        powerd_warn("could not connect to Unity: %s", error->message);
        g_error_free(error);
        return FALSE;
    }

    GVariant *ret = g_dbus_proxy_call_sync(unity_proxy,
            "setScreenPowerMode",
            g_variant_new("(s)", power_mode),
            G_DBUS_CALL_FLAGS_NONE,
            -1,
            NULL,
            &error);

    if (ret == NULL) {
        powerd_warn("screen power setting failed: %s", error->message);
        g_error_free(error);
        g_object_unref(unity_proxy);
        return FALSE;
    }

    g_object_unref(unity_proxy);

    return TRUE;
}

static void turn_on_display(void)
{
    powerd_debug("turning on display");
    display_set_power_mode(0, "on");
    set_backlight(target_bl_state);
}

gboolean powerd_display_enabled(void)
{
    return actual_screen_state == POWERD_DISPLAY_STATE_ON;
}

static void update_display_state(struct powerd_display_request *req)
{
    enum powerd_display_state state = req->state;
    int applied_state;
    int ret;

    applied_state = screen_off_overrides ? DISPLAY_STATE_OFF : state;

    switch (applied_state) {
    case DISPLAY_STATE_OFF:
        if (actual_screen_state == DISPLAY_STATE_OFF) {
            /* Nothing to do */
            return;
        }

        powerd_debug("turning off display");
        set_backlight(BL_OFF);
        if (!display_set_power_mode(0, "off")) {
            powerd_warn("failed to set display power mode, not clearing state");
            return;
        }

        powerd_debug("Releasing internal active state request");
        ret = clear_sys_state_internal(internal_request_cookie);
        if (!ret) {
            char cookie_str[UUID_STR_LEN];
            uuid_unparse(internal_request_cookie, cookie_str);
            powerd_warn("Internal system state request cookie invalid: %s",
                        cookie_str);
        }
        break;
    case POWERD_DISPLAY_STATE_ON:
        if (ab_enabled(req->flags))
            target_bl_state = BL_AUTO;
        else if (req->flags & POWERD_DISPLAY_FLAG_BRIGHT)
            target_bl_state = BL_BRIGHT;
        else
            target_bl_state = BL_DIM;

        if (actual_screen_state != POWERD_DISPLAY_STATE_ON) {
            powerd_debug("Requesting active state internally");
            ret = request_sys_state_internal("display-request",
                                             POWERD_SYS_STATE_ACTIVE,
                                             &internal_request_cookie, NULL);
            if (!ret)
                powerd_warn("Request for active state failed");

            /*
             * If currently in the suspend state we need to wait for
             * notification of entering the active state, otherwise the
             * screen may fail to turn on for devices using earlysuspend.
             * Otherwise we can turn on the screen right away.
             */
            if (!powerd_suspend_active())
                turn_on_display();
        } else {
            /* Only updating backlight state */
            set_backlight(target_bl_state);
        }
        break;
    default:
        powerd_warn("Invalid display state %d", applied_state);
        return;
    }

    actual_screen_state = applied_state;

    powerd_display_state_signal_emit(actual_screen_state, req->flags);
}

static void update_flags(guint32 flags)
{
    int prox_enabled, internal_prox_enabled;

    prox_enabled = (flags & POWERD_DISPLAY_FLAG_USE_PROXIMITY);
    internal_prox_enabled = (internal_state.flags & POWERD_DISPLAY_FLAG_USE_PROXIMITY);

    if (prox_enabled != internal_prox_enabled) {
        if (prox_enabled) {
            powerd_sensors_proximity_enable();
        } else {
            powerd_sensors_proximity_disable();
            powerd_display_clear_override(POWERD_OVERRIDE_REASON_PROXIMITY);
        }
    }
}

/* 
 * *** WARNING ***
 *
 * This should now only be called by display request code. Everyone
 * else must use powerd_{add,remove}_display_request()!!!
 */
void powerd_set_display_state(struct powerd_display_request *req)
{
    powerd_debug("powerd_set_display_state: %s -> %s, 0x%x -> 0x%x",
                 display_state_to_str(internal_state.state),
                 display_state_to_str(req->state),
                 internal_state.flags, req->flags);

    /* Update flags before display state to ensure proximity flag is
     * taken into account */
    update_flags(req->flags);
    update_display_state(req);

    internal_state = *req;
}

static void user_ab_enable(gboolean enable)
{
    user_ab_enabled = enable;
    update_display_state(&internal_state);
}

static gboolean prox_update_worker(gpointer data)
{
    unsigned long near = (unsigned long)data;
    if (near)
        powerd_display_set_override(POWERD_OVERRIDE_REASON_PROXIMITY);
    else
        powerd_display_clear_override(POWERD_OVERRIDE_REASON_PROXIMITY);
    return FALSE;
}

void powerd_proximity_event(gboolean near)
{
    unsigned long data = (unsigned long)near;
    g_timeout_add(0, prox_update_worker, (gpointer)data);
}

struct apply_override_data {
    enum powerd_override_reason reason;
    int set;
};

static int apply_override_worker(gpointer data)
{
    struct apply_override_data *ovr_data = data;
    unsigned orig_overrides = screen_off_overrides;
    unsigned mask;

    mask = 1 << ovr_data->reason;
    if (ovr_data->set)
        screen_off_overrides |= mask;
    else
        screen_off_overrides &= ~mask;

    if (orig_overrides != screen_off_overrides)
        update_display_state(&internal_state);
    return 0;
}

void powerd_display_set_override(enum powerd_override_reason reason)
{
    struct apply_override_data ovr_data;

    if (reason >= POWERD_NUM_OVERRIDE_REASONS) {
        powerd_debug("Refusing to set invalid override reason (%d)", reason);
        return;
    }

    ovr_data.reason = reason;
    ovr_data.set = 1;
    powerd_run_mainloop_sync(apply_override_worker, &ovr_data);
}

void powerd_display_clear_override(enum powerd_override_reason reason)
{
    struct apply_override_data ovr_data;

    if (reason >= POWERD_NUM_OVERRIDE_REASONS) {
        powerd_debug("Refusing to clear invalid override reason (%d)", reason);
        return;
    }

    ovr_data.reason = reason;
    ovr_data.set = 0;
    powerd_run_mainloop_sync(apply_override_worker, &ovr_data);
}

static void set_user_brightness(int brightness)
{
    int max = powerd_get_max_brightness();
    if (brightness > max)
        brightness = max;
    user_brightness = brightness;
    update_display_state(&internal_state);
}

static gboolean exit_suspend_worker(gpointer unused)
{
    /* Make sure suspend is still disabled */
    if (!powerd_suspend_active() && powerd_display_enabled())
        turn_on_display();
    return FALSE;
}

void powerd_display_exit_suspend(void)
{
    g_timeout_add(0, exit_suspend_worker, NULL);
}

int powerd_display_init(void)
{
    struct stat stats;

    /* Use the current brightness until we're told another value */
    user_brightness = powerd_get_brightness();

    return 0;
}


/** dbus interfaces **/

gboolean handle_user_autobrightness_enable(PowerdSource *obj,
                                           GDBusMethodInvocation *invocation,
                                           gboolean enable)
{
    user_ab_enable(enable);
    g_dbus_method_invocation_return_value(invocation, NULL);
    return TRUE;
}

gboolean handle_set_user_brightness(PowerdSource *obj,
                                    GDBusMethodInvocation *invocation,
                                    gint brightness)
{
    set_user_brightness(brightness);
    g_dbus_method_invocation_return_value(invocation, NULL);
    return TRUE;
}
