/***************************************************************************
 *   Copyright (C) 2005 by Johan Maes   *
 *   on4qz@telenet.be   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 *                                                                         *
 *   In addition, as a special exception, the copyright holders give       *
 *   permission to link the code of this program with any edition of       *
 *   the Qt library by Trolltech AS, Norway (or with modified versions     *
 *   of Qt that use the same license as Qt), and distribute linked         *
 *   combinations including the two.  You must obey the GNU General        *
 *   Public License in all respects for all of the code used other than    *
 *   Qt.  If you modify this file, you may extend this exception to        *
 *   your version of the file, but you are not obligated to do so.  If     *
 *   you do not wish to do so, delete this exception statement from        *
 *   your version.                                                         *
 ***************************************************************************/

#include "wavio.h"
#include "global.h"
#include <qfiledialog.h>
#include "configparams.h"
#include "supportfunctions.h"

//#include "qfilterdesignglobal.h"


/**
	constructor: creates a waveIO instance
	\param samplingRate wave file samplingrate (e.g. 8000, 11025 ...)
*/

wavIO::wavIO(unsigned int samplingRate)
{
	samplingrate=samplingRate;
	reading=FALSE,
	writing=FALSE;
}


wavIO::~wavIO()
{
}

void wavIO::closeFile()
{
	inopf.close();
	reading=FALSE;
	writing=FALSE;
}

/**
	 opens a wave file for reading
	\param fname the name of the file to open
	\param ask if ask==TRUE, a filedialog will be opened
	\return TRUE if the file is succcesfully opened. The file is also checked if it is a supported format.
	\sa read
*/ 

bool  wavIO::openFileForRead(QString fname,bool ask)
{
  QString tmp;
	
  if (ask)
    {
			 	dirDialog d;
			 	QString s=d.openFileName(audioPath,"*.wav");
				if (s==QString::null) return FALSE;
				if (s.isEmpty()) return FALSE;
       	inopf.setFileName(s);
    }
  else
    {
      inopf.setFileName(fname);
    }
  if(!inopf.open(QIODevice::ReadOnly))
    {
    return FALSE;
    }
	reading=TRUE;
  if(inopf.read(&waveHeader.chunkID[0],sizeof(sWave))!=sizeof(sWave))
    {
			closeFile();
      return FALSE;
    }


// check the header
  if(  (!checkString(waveHeader.chunkID,"RIFF"))
     ||(!checkString(waveHeader.format,"WAVE"))
     ||(!checkString(waveHeader.subChunk1ID,"fmt "))
     ||(!checkString(waveHeader.subChunk2ID,"data")))
     {
      addToLog("wavio read header error",DBALL);
			closeFile();
      return FALSE;
     }

  if( (waveHeader.subChunk1Size!=16)
      ||(waveHeader.audioFormat!=1)
      ||(waveHeader.numChannels!=1)
      ||(waveHeader.sampleRate!=samplingrate)
      ||(waveHeader.byteRate!=sizeof(short int)*samplingrate)
      ||(waveHeader.blockAlign!=2)
      ||(waveHeader.bitsPerSample!=16))
      {
        addToLog("wavio read header error, not supported",DBALL);
				closeFile();
        return FALSE;
      }

  numberOfSamples=waveHeader.subChunk2Size/sizeof(short int);
	samplesRead=0;
	return TRUE;
}

/**
	read data from wave file

	\param dPtr pointer to buffer for 16 bit samples
	\param numSamples  number of samples to read
	\return returns the nummber of samples read. -1 is returned if the end of the file is reached. The file is then automatically closed.
*/

int  wavIO::read(short int *dPtr ,uint numSamples)
{
	int llen,result;
	if(!inopf.isOpen())
		{
      addToLog("wavio not open during read",DBALL);
	 		return TRUE;
		}
	llen=numSamples*sizeof(short int);
	if(numberOfSamples<=samplesRead)
		{
			closeFile();
			return -1;
		}
	result=inopf.read((char*)dPtr,llen);
	if(result==0) inopf.close();
	samplesRead+=result/sizeof(short int);
	return result/sizeof(short int);
}

/**
	 opens a wave file for writing
	\param fname the name of the file to open
	\param ask if ask==TRUE, a filedialog will be opened
	\return TRUE if the file is succcesfully opened, and the header written, FALSE otherwise?
	\sa write
*/

bool  wavIO::openFileForWrite(QString fname,bool ask)
{
  QFileInfo fin;
  if (ask)
    {
			dirDialog d;
			QString fn=d.saveFileName(audioPath,"*.wav","wav");
/*.       QString fn(QFileDialog::getSaveFileName(audioPath,"*.wav"));
       fin.setFile(fn);
       if(fin.extension().isEmpty())
        {
          fn+=".wav"; 
        }*/
       inopf.setFileName(fn);
    }
  else
    {
      inopf.setFileName(fname);
    }
  if(!inopf.open(QIODevice::WriteOnly|QIODevice::Truncate))
    {
      return FALSE;
    }
  numberOfSamples=0;
	initHeader();
	writing=TRUE;
	if(!writeHeader()) return FALSE;
	numberOfSamples=0;
	return TRUE;
}

/**
	\brief write data to wave file

	To signal the end, call this function with numSamples=0. The file will automatically be closed.
	\param dPtr pointer to buffer for 16 bit samples
	\param numSamples  number of samples to read
	\return returns TRUE the correct number of samples are written. FALSE otherwise.
*/

bool  wavIO::write(short int *dPtr,uint numSamples)
{
	int len;
	len=numSamples*sizeof(short int);
	if((!writing)&&(numSamples!=0))
		{
      addToLog("wavio not open during write",DBALL);
	 		return TRUE;
		}
	if((!writing)&&(numSamples==0)) return TRUE;
	if(numSamples==0)
		{
      addToLog(QString("wavio write close samples=%1").arg(numberOfSamples),DBWAVIO);
			inopf.flush();
			writeHeader();
			closeFile();
			return TRUE;
		}
   if(inopf.write((char *)dPtr,len)!=len)
     {
      addToLog("wavio write error",DBALL);
      closeFile();
      return FALSE;
     }
	numberOfSamples+=numSamples;
  addToLog(QString("wavio write samp0les=%1").arg(numberOfSamples),DBWAVIO);
  return TRUE;
}


/** setup the defaults in the wave header */

void wavIO::initHeader()
{
    waveHeader.chunkID[0]='R';
    waveHeader.chunkID[1]='I';
    waveHeader.chunkID[2]='F';
    waveHeader.chunkID[3]='F';
    
    waveHeader.format[0]='W';
    waveHeader.format[1]='A';
    waveHeader.format[2]='V';
    waveHeader.format[3]='E';

 
    waveHeader.subChunk1ID[0]='f';
    waveHeader.subChunk1ID[1]='m';
    waveHeader.subChunk1ID[2]='t';
    waveHeader.subChunk1ID[3]=' ';

    waveHeader.subChunk2ID[0]='d';
    waveHeader.subChunk2ID[1]='a';
    waveHeader.subChunk2ID[2]='t';
    waveHeader.subChunk2ID[3]='a';

    waveHeader.subChunk1Size=16;      // always 16 for PCM
    waveHeader.audioFormat=1;         // PCM
    waveHeader.numChannels=1;         // Mono
    waveHeader.sampleRate=samplingrate;
    waveHeader.byteRate=sizeof(short int)*samplingrate;    // 16 bit samples
    waveHeader.blockAlign=2;
    waveHeader.bitsPerSample=16;
    waveHeader.chunkSize=36+numberOfSamples*sizeof(short int);
    waveHeader.subChunk2Size=numberOfSamples*sizeof(short int);
}

bool  wavIO::checkString(char *str,const char *cstr)
{
  for (int i=0;i<4;i++)
    {
      if (str[i]!=cstr[i]) return FALSE;
    }
  return TRUE;
}

bool  wavIO::writeHeader()
{
	int err;
	waveHeader.subChunk2Size=numberOfSamples*sizeof(short int);
	lseek(inopf.handle(),0,SEEK_SET); //position at beginning
  if((err=inopf.write(&waveHeader.chunkID[0],sizeof(sWave)))!=sizeof(sWave))
     {
		 	
      addToLog(QString("wavio write header error %1").arg(err),DBWAVIO);
      closeFile();
      return FALSE;
     }
	inopf.flush();
	lseek(inopf.handle(),0,SEEK_END); //position at beginning
  addToLog(QString("wavio write header %1 %2 %3 %4").arg(waveHeader.chunkID[0]).arg(waveHeader.chunkID[1]).arg(waveHeader.chunkID[2]).arg(waveHeader.chunkID[3]),DBWAVIO);
  addToLog(QString("wavio write header samples=%1").arg(numberOfSamples),DBWAVIO);
  addToLog(QString("wavio write header total bytes=%1").arg(numberOfSamples*2+sizeof(sWave)),DBWAVIO);
	return TRUE;
}
