/*****************************************************************************
 * dictbrowser.h - QStarDict, a dictionary application for learning foreign  *
 *                 languages                                                 *
 * Copyright (C) 2007-2023 Alexander Rodin                                   *
 *                                                                           *
 * This program is free software; you can redistribute it and/or modify      *
 * it under the terms of the GNU General Public License as published by      *
 * the Free Software Foundation; either version 2 of the License, or         *
 * (at your option) any later version.                                       *
 *                                                                           *
 * This program is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 * GNU General Public License for more details.                              *
 *                                                                           *
 * You should have received a copy of the GNU General Public License along   *
 * with this program; if not, write to the Free Software Foundation, Inc.,   *
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.               *
 *****************************************************************************/

#ifndef DICTBROWSER_H
#define DICTBROWSER_H

#include <QTextBrowser>

#include <QTextCursor>
#include <QTextCharFormat>
#include "dictcore.h"

namespace QStarDict
{

/**
 * The DictBrowser widget provides view of translations from given dictionary.
 */
class DictBrowser: public QTextBrowser
{
    Q_OBJECT
    public:
        /**
         * Construct empty DictWidget.
         */
        DictBrowser(QWidget *parent = 0);

        /**
         * Set source dictionary.
         * Warning: DictBrowser will copy only a pointer to dict. So set dictionaries
         * allocated from heap and don't destroy it befor DictWidget.
         */
        void setDict(DictCore *dict)
        { m_dict = dict; }
        /**
         * Return pointer to dictionary.
         */
        const DictCore* dict() const
        { return m_dict; }

        void setShowLinks(bool showLinks)
        { m_showLinks = showLinks; }
        bool showLinks() const
        { return m_showLinks; }

        void setShowLinksModifierKey(int key)
        { m_showLinksModifierKey = key; }
        int showLinksModifierKey() const
        { return m_showLinksModifierKey; }

        /**
          * Set whether to show buttons near IPA transcriptions which allow
          * the user to listen the pronunciation of these transcriptions.
          */
        void setShowIpaPronouncers(bool showIpaPronouncers)
        { m_showIpaPronouncers = showIpaPronouncers; }
        /**
          * Return true the IPA pronouncer is enabled.
          */
        bool showIpaPronouncers() const
        { return m_showIpaPronouncers; }

        QVariant loadResource(int type, const QUrl &name);

        bool eventFilter(QObject *object, QEvent *event);

        /**
          * Show an introductory text about QStarDict.
          */
        void showIntro();

    signals:
        void searchResult(bool success);

    public slots:
        void search(const QString & exp, QTextDocument::FindFlags options);
        void searchActive(bool active);

    protected:
        void mouseMoveEvent(QMouseEvent *event);
        void mouseReleaseEvent(QMouseEvent *event);
        void timerEvent(QTimerEvent *event);

    private slots:
        void on_anchorClicked(const QUrl &link);
        void on_sourceChanged(const QUrl &link);
        bool areLinksActive();
        QRect wordRect(const QPoint &mousePosition);

    private:
        DictCore *m_dict;

        QTextCursor m_oldCursor;
        QTextCharFormat m_oldFormat;
        bool m_highlighted;
        int m_highlightTimerId;
        QString m_highlightedWord;

        bool m_showLinks;
        int m_showLinksModifierKey;
        bool m_highlightInCurrentTranslation;

        bool m_showIpaPronouncers;

        void invalidateHighlight();
        void addIpaPronouncers();
};

}

#endif // DICTBROWSER_H

// vim: tabstop=4 softtabstop=4 shiftwidth=4 expandtab cindent textwidth=120 formatoptions=tc

