% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/combineRecomputedResults.R
\name{combineRecomputedResults}
\alias{combineRecomputedResults}
\title{Combine SingleR results with recomputation}
\usage{
combineRecomputedResults(
  results,
  test,
  trained,
  quantile = 0.8,
  assay.type.test = "logcounts",
  check.missing = TRUE,
  allow.lost = FALSE,
  warn.lost = TRUE,
  BNPARAM = KmknnParam(),
  BPPARAM = SerialParam()
)
}
\arguments{
\item{results}{A list of \linkS4class{DataFrame} prediction results as returned by \code{\link{classifySingleR}} when run on each reference separately.}

\item{test}{A numeric matrix of single-cell expression values where rows are genes and columns are cells.
Alternatively, a \linkS4class{SummarizedExperiment} object containing such a matrix.}

\item{trained}{A list of \linkS4class{List}s containing the trained outputs of multiple references,
equivalent to either (i) the output of \code{\link{trainSingleR}} on multiple references with \code{recompute=TRUE},
or (ii) running \code{trainSingleR} on each reference separately and manually making a list of the trained outputs.}

\item{quantile}{Further arguments to pass to \code{\link{classifySingleR}}.}

\item{assay.type.test}{An integer scalar or string specifying the assay of \code{test} containing the relevant expression matrix,
if \code{test} is a \linkS4class{SummarizedExperiment} object.}

\item{check.missing}{Logical scalar indicating whether rows should be checked for missing values (and if found, removed).}

\item{allow.lost}{Logical scalar indicating whether to use lost markers in references where they are available.}

\item{warn.lost}{Logical scalar indicating whether to emit a warning if markers from one reference in \code{trained} are \dQuote{lost} in other references.}

\item{BNPARAM}{A \linkS4class{BiocNeighborParam} object specifying the algorithm to use for building nearest neighbor indices.}

\item{BPPARAM}{A \linkS4class{BiocParallelParam} object specifying how parallelization should be performed, if any.}
}
\value{
A \linkS4class{DataFrame} is returned containing the annotation statistics for each cell or cluster (row).
This mimics the output of \code{\link{classifySingleR}} and contains the following fields:
\itemize{
\item \code{scores}, a numeric matrix of correlations containing the \emph{recomputed} scores.
For any given cell, entries of this matrix are only non-\code{NA} for the assigned label in each reference;
scores are not recomputed for the other labels.
\item \code{labels}, a character vector containing the per-cell combined label across references.
\item \code{references}, an integer vector specifying the reference from which the combined label was derived.
\item \code{orig.results}, a DataFrame containing \code{results}.
}
It may also contain \code{first.labels} and \code{pruned.labels} if these were also present in \code{results}.

The \code{\link{metadata}} contains \code{label.origin}, 
a DataFrame specifying the reference of origin for each label in \code{scores}.
Note that, unlike \code{\link{combineCommonResults}}, no \code{common.genes} is reported
as this function does not use a common set of genes across all references.
}
\description{
Combine results from multiple runs of \code{\link{classifySingleR}} (usually against different references) into a single \linkS4class{DataFrame}.
The label from the results with the highest score for each cell is retained.
Unlike \code{\link{combineCommonResults}}, this does not assume that each run of \code{\link{classifySingleR}} was performed using the same set of common genes, instead recomputing the scores for comparison across references.
}
\details{
Here, the strategy is to perform classification separately within each reference, 
then collate the results to choose the label with the highest score across references.
For a given cell in \code{test}, we extract its assigned label from \code{results} for each reference.
We also retrieve the marker genes associated with that label and take the union of markers across all references.
This defines a common feature space in which the score for each reference's assigned label is recomputed using \code{ref};
the label from the reference with the top recomputed score is then reported as the combined annotation for that cell.

Unlike \code{\link{combineCommonResults}}, the union of markers is not used for the within-reference calls.
This avoids the inclusion of noise from irrelevant genes in the within-reference assignments.
Obviously, \code{combineRecomputedResults} is slower as it does require recomputation of the scores,
but the within-reference calls are faster as there are fewer genes in the union of markers for assigned labels
(compared to the union of markers across all labels, as required by \code{\link{combineCommonResults}}),
so it is likely that the net compute time should be lower.
}
\section{Dealing with mismatching gene availabilities}{

It is strongly recommended that the universe of genes be the same across all references in \code{trained}.
If this is not the case, the intersection of genes across all \code{trained} will be used in the recomputation.
This at least provides a common feature space for comparing correlations, 
though differences in the availability of markers between references may have unpredictable effects on the results
(and so a warning will be emitted by default, when when \code{warn.lost=TRUE}).

That said, the intersection may be too string when dealing with many references with diverse feature annotations. 
In such cases, we can set \code{allow.lost=TRUE} so that the recomputation for each reference will use all available markers in that reference.
The idea here is to avoid penalizing all references by removing an informative marker when it is only absent in a single reference.
We hope that the recomputed scores are still roughly comparable if the number of lost markers is relatively low,
coupled with the use of ranks in the calculation of the Spearman-based scores to reduce the influence of individual markers.
This is perhaps as reliable as one might imagine, so setting \code{allow.lost=TRUE} should be considered a last resort.
}

\examples{
# Making up data.
ref <- .mockRefData(nreps=8)
ref1 <- ref[,1:2\%\%2==0]
ref2 <- ref[,1:2\%\%2==1]
ref2$label <- tolower(ref2$label)

test <- .mockTestData(ref)

# Performing classification within each reference.
test <- scuttle::logNormCounts(test)

ref1 <- scuttle::logNormCounts(ref1)
train1 <- trainSingleR(ref1, labels=ref1$label)
pred1 <- classifySingleR(test, train1)

ref2 <- scuttle::logNormCounts(ref2)
train2 <- trainSingleR(ref2, labels=ref2$label)
pred2 <- classifySingleR(test, train2)

# Combining results with recomputation of scores.
combined <- combineRecomputedResults(
    results=list(pred1, pred2), 
    test=test,
    trained=list(train1, train2))

combined[,1:5]

}
\references{
Lun A, Bunis D, Andrews J (2020).
Thoughts on a more scalable algorithm for multiple references.
\url{https://github.com/LTLA/SingleR/issues/94}
}
\seealso{
\code{\link{SingleR}} and \code{\link{classifySingleR}}, for generating predictions to use in \code{results}.

\code{\link{combineCommonResults}}, for another approach to combining predictions.
}
\author{
Aaron Lun
}
