% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/trainSingleR.R
\name{trainSingleR}
\alias{trainSingleR}
\title{Train the SingleR classifier}
\usage{
trainSingleR(
  ref,
  labels,
  test.genes = NULL,
  genes = "de",
  sd.thresh = NULL,
  de.method = c("classic", "wilcox", "t"),
  de.n = NULL,
  de.args = list(),
  aggr.ref = FALSE,
  aggr.args = list(),
  recompute = TRUE,
  restrict = NULL,
  assay.type = "logcounts",
  check.missing = TRUE,
  approximate = FALSE,
  num.threads = bpnworkers(BPPARAM),
  BNPARAM = NULL,
  BPPARAM = SerialParam()
)
}
\arguments{
\item{ref}{A numeric matrix of expression values where rows are genes and columns are reference samples (individual cells or bulk samples).
Each row should be named with the gene name.
In general, the expression values are expected to be log-transformed, see Details.

Alternatively, a \linkS4class{SummarizedExperiment} object containing such a matrix.

Alternatively, a list or \linkS4class{List} of SummarizedExperiment objects or numeric matrices containing multiple references.}

\item{labels}{A character vector or factor of known labels for all samples in \code{ref}.

Alternatively, if \code{ref} is a list, \code{labels} should be a list of the same length.
Each element should contain a character vector or factor specifying the labels for the columns of the corresponding element of \code{ref}.}

\item{test.genes}{Character vector of the names of the genes in the test dataset, i.e., the row names of \code{test} in \code{\link{classifySingleR}}.
If \code{NULL}, it is assumed that the test dataset and \code{ref} have the same genes in the same row order.}

\item{genes}{A string containing \code{"de"}, indicating that markers should be calculated from \code{ref}.
For back compatibility, other string values are allowed but will be ignored with a deprecation warning.

Alternatively, if \code{ref} is \emph{not} a list, \code{genes} can be either:
\itemize{
\item A list of lists of character vectors containing DE genes between pairs of labels.
\item A list of character vectors containing marker genes for each label.
}

If \code{ref} \emph{is} a list, \code{genes} can be a list of length equal to \code{ref}.
Each element of the list should be one of the two above choices described for non-list \code{ref},
containing markers for labels in the corresponding entry of \code{ref}.}

\item{sd.thresh}{Deprecated and ignored.}

\item{de.method}{String specifying how DE genes should be detected between pairs of labels.
Defaults to \code{"classic"}, which sorts genes by the log-fold changes and takes the top \code{de.n}.
Setting to \code{"wilcox"} or \code{"t"} will use Wilcoxon ranked sum test or Welch t-test between labels, respectively,
and take the top \code{de.n} upregulated genes per comparison.
Ignored if \code{genes} is a list of markers/DE genes.}

\item{de.n}{An integer scalar specifying the number of DE genes to use when \code{genes="de"}.
If \code{de.method="classic"}, defaults to \code{500 * (2/3) ^ log2(N)} where \code{N} is the number of unique labels.
Otherwise, defaults to 10.
Ignored if \code{genes} is a list of markers/DE genes.}

\item{de.args}{Named list of additional arguments to pass to \code{\link[scran]{pairwiseTTests}} or \code{\link[scran]{pairwiseWilcox}} when \code{de.method="wilcox"} or \code{"t"}.
Ignored if \code{genes} is a list of markers/DE genes.}

\item{aggr.ref}{Logical scalar indicating whether references should be aggregated to pseudo-bulk samples for speed, see \code{\link{aggregateReference}}.}

\item{aggr.args}{Further arguments to pass to \code{\link{aggregateReference}} when \code{aggr.ref=TRUE}.}

\item{recompute}{Deprecated and ignored.}

\item{restrict}{A character vector of gene names to use for marker selection.
By default, all genes in \code{ref} are used.}

\item{assay.type}{An integer scalar or string specifying the assay of \code{ref} containing the relevant expression matrix,
if \code{ref} is a \linkS4class{SummarizedExperiment} object (or is a list that contains one or more such objects).}

\item{check.missing}{Logical scalar indicating whether rows should be checked for missing values.
If true and any missing values are found, the rows containing these values are silently removed.}

\item{approximate}{Deprecated, use \code{BNPARAM} instead.}

\item{num.threads}{Integer scalar specifying the number of threads to use for index building.}

\item{BNPARAM}{A \linkS4class{BiocNeighborParam} object specifying how the neighbor search index should be constructed.}

\item{BPPARAM}{A \linkS4class{BiocParallelParam} object specifying how parallelization should be performed.
Relevant for marker detection if \code{genes = NULL}, aggregation if \code{aggr.ref = TRUE}, and \code{NA} checking if \code{check.missing = TRUE}.}
}
\value{
For a single reference, a \linkS4class{List} is returned containing:
\describe{
\item{\code{built}:}{An external pointer to various indices in C++ space.
Note that this cannot be serialized and should be removed prior to any \code{\link{saveRDS}} step.}
\item{\code{ref}:}{The reference expression matrix.
This may have fewer columns than the input \code{ref} if \code{aggr.ref = TRUE}.}
\item{\code{markers}:}{A list containing \code{unique}, a character vector of all marker genes used in training;
and \code{full}, a list of list of character vectors containing the markers for each pairwise comparison between labels.}
\item{\code{labels}:}{A list containing \code{unique}, a character vector of all unique reference labels;
and \code{full}, a character vector containing the assigned label for each column in \code{ref}.}
}

For multiple references, a List of Lists is returned where each internal List corresponds to a reference in \code{ref} and has the same structure as described above.
}
\description{
Train the SingleR classifier on one or more reference datasets with known labels.
}
\details{
This function uses a training data set to select interesting features and construct nearest neighbor indices in rank space.
The resulting objects can be re-used across multiple classification steps with different test data sets via \code{\link{classifySingleR}}.
This improves efficiency by avoiding unnecessary repetition of steps during the downstream analysis.

The automatic marker detection (\code{genes="de"}) identifies genes that are differentially expressed between labels.
This is done by identifying the median expression within each label, and computing differences between medians for each pair of labels.
For each label, the top \code{de.n} genes with the largest differences compared to another label are chosen as markers to distinguish the two labels.
The expression values are expected to be log-transformed and normalized.

Classification with \code{classifySingleR} assumes that the test dataset contains all marker genes that were detected from the reference.
If the test and reference datasets do not have the same genes in the same order, we can set \code{test.genes} to the row names of the test dataset.
This will instruct \code{trainSingleR} to only consider markers that are present in the test dataset.
Any subsequent call to \code{classifySingleR} will also check that \code{test.genes} is consistent with \code{rownames(test)}.

On a similar note, if \code{restrict} is specified, marker selection will only be performed using the specified subset of genes.
This can be convenient for ignoring inappropriate genes like pseudogenes or predicted genes.
It has the same effect as filtering out undesirable rows from \code{ref} prior to calling \code{trainSingleR}.
Unlike \code{test.genes}, setting \code{restrict} does not introduce further checks on \code{rownames(test)} in \code{classifySingleR}.
}
\section{Custom feature specification}{

Rather than relying on the in-built feature selection, users can pass in their own features of interest to \code{genes}.
The function expects a named list of named lists of character vectors, with each vector containing the DE genes between a pair of labels.
For example:
\preformatted{genes <- list(
   A = list(A = character(0), B = "GENE_1", C = c("GENE_2", "GENE_3")),
   B = list(A = "GENE_100", B = character(0), C = "GENE_200"),
   C = list(A = c("GENE_4", "GENE_5"), B = "GENE_5", C = character(0))
)
}
If we consider the entry \code{genes$A$B}, this contains marker genes for label \code{"A"} against label \code{"B"}.
That is, these genes are upregulated in \code{"A"} compared to \code{"B"}.
The outer list should have one list per label, and each inner list should have one character vector per label.
(Obviously, a label cannot have markers against itself, so this is just set to \code{character(0)}.)

Alternatively, \code{genes} can be a named list of character vectors containing per-label markers.
For example:
\preformatted{genes <- list(
     A = c("GENE_1", "GENE_2", "GENE_3"),
     B = c("GENE_100", "GENE_200"),
     C = c("GENE_4", "GENE_5")
)
}
The entry \code{genes$A} represent the genes that are upregulated in \code{A} compared to some or all other labels. 
This allows the function to handle pre-defined marker lists for specific cell populations.
However, it obviously captures less information than marker sets for the pairwise comparisons.

If \code{genes} is manually passed, \code{ref} can be the raw counts or any monotonic transformation thereof.
There is no need to supply (log-)normalized expression values for the benefit of the automatic marker detection.
Similarly, for manual \code{genes}, the values of \code{de.method}, \code{de.n} and \code{sd.thresh} have no effect.
}

\section{Dealing with multiple references}{

The default \pkg{SingleR} policy for dealing with multiple references is to perform the classification for each reference separately and combine the results 
(see \code{?\link{combineRecomputedResults}} for an explanation).
To this end, if \code{ref} is a list with multiple references, marker genes are identified separately within each reference if \code{genes = NULL}.
Rank calculation and index construction is then performed within each reference separately.
The result is identical to \code{lapply}ing over a list of references and runing \code{trainSingleR} on each reference.

Alternatively, \code{genes} can still be used to explicitly specify marker genes for each label in each of multiple references.
This is achieved by passing a list of lists to \code{genes},
where each inner list corresponds to a reference in \code{ref} and can be of any format described in \dQuote{Custom feature specification}.
Thus, it is possible for \code{genes} to be - wait for it - a list (per reference) of lists (per label) of lists (per label) of character vectors.
}

\section{Note on single-cell references}{

The default marker selection is based on log-fold changes between the per-label medians and is very much designed with bulk references in mind.
It may not be effective for single-cell reference data where it is not uncommon to have more than 50\% zero counts for a given gene such that the median is also zero for each group.
Users are recommended to either set \code{de.method} to another DE ranking method, or detect markers externally and pass a list of markers to \code{genes} (see Examples).

In addition, it is generally unnecessary to have single-cell resolution on the reference profiles.
We can instead set \code{aggr.ref=TRUE} to aggregate per-cell references into a set of pseudo-bulk profiles using \code{\link{aggregateReference}}.
This improves classification speed while using vector quantization to preserve within-label heterogeneity and mitigate the loss of information.
Note that any aggregation is done \emph{after} marker gene detection; this ensures that the relevant tests can appropriately penalize within-label variation.
Users should also be sure to set the seed as the aggregation involves randomization.
}

\examples{
# Making up some data for a quick demonstration.
ref <- .mockRefData()

# Normalizing and log-transforming for automated marker detection.
ref <- scuttle::logNormCounts(ref)

trained <- trainSingleR(ref, ref$label)
trained
length(trained$markers$unique)

# Alternatively, computing and supplying a set of label-specific markers.
by.t <- scran::pairwiseTTests(assay(ref, 2), ref$label, direction="up")
markers <- scran::getTopMarkers(by.t[[1]], by.t[[2]], n=10)
trained <- trainSingleR(ref, ref$label, genes=markers)
length(trained$markers$unique)

}
\seealso{
\code{\link{classifySingleR}}, where the output of this function gets used.

\code{\link{combineRecomputedResults}}, to combine results from multiple references.

\code{\link{rebuildIndex}}, to rebuild the index after external memory is invalidated.
}
\author{
Aaron Lun, based on the original \code{SingleR} code by Dvir Aran.
}
