% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/riskRegression-package.R, R/riskRegression.R
\docType{package}
\name{riskRegression}
\alias{riskRegression}
\alias{ARR}
\alias{LRR}
\title{Risk Regression
Fits a regression model for the risk of an event -- allowing for competing
risks.}
\usage{
riskRegression(
  formula,
  data,
  times,
  link = "relative",
  cause,
  conf.int = TRUE,
  cens.model,
  cens.formula,
  max.iter = 50,
  conservative = TRUE,
  ...
)
}
\arguments{
\item{formula}{Formula where the left hand side specifies the event
history event.history and the right hand side the linear predictor.  See
examples.}

\item{data}{The data for fitting the model in which includes all
the variables included in formula.}

\item{times}{Vector of times. For each time point in \code{times}
estimate the baseline risk and the timevarying coefficients.}

\item{link}{\code{"relative"} for the absolute risk regression
model.  \code{"logistic"} for the logistic risk regression model.
\code{"prop"} for the Fine-Gray regression model.}

\item{cause}{The cause of interest.}

\item{conf.int}{If \code{TRUE} return the iid decomposition, that
can be used to construct confidence bands for predictions.}

\item{cens.model}{Specified the model for the (conditional)
censoring distribution used for deriving weights (IFPW). Defaults
to "KM" (the Kaplan-Meier method ignoring covariates) alternatively
it may be "Cox" (Cox regression).}

\item{cens.formula}{Right hand side of the formula used for fitting
the censoring model.  If not specified the right hand side of
\code{formula} is used.}

\item{max.iter}{Maximal number of iterations.}

\item{conservative}{If \code{TRUE} use variance formula that ignores the contribution
by the estimate of the inverse of the probability of censoring weights}

\item{...}{Further arguments passed to \code{comp.risk}}
}
\description{
This is a wrapper for the function \code{comp.risk} from the timereg package.
The main difference is one marks variables in the formula that should have a
time-dependent effect whereas in \code{comp.risk} one marks variables that
should have a time constant (proportional) effect.
}
\examples{

library(prodlim)
data(Melanoma,package="riskRegression")
## tumor thickness on the log-scale
Melanoma$logthick <- log(Melanoma$thick)

# Single binary factor

## absolute risk regression
library(survival)
library(prodlim)
fit.arr <- ARR(Hist(time,status)~sex,data=Melanoma,cause=1)
print(fit.arr)
# show predicted cumulative incidences
plot(fit.arr,col=3:4,newdata=data.frame(sex=c("Female","Male")))

## compare with non-parametric Aalen-Johansen estimate
library(prodlim)
fit.aj <- prodlim(Hist(time,status)~sex,data=Melanoma)
plot(fit.aj,conf.int=FALSE)
plot(fit.arr,add=TRUE,col=3:4,newdata=data.frame(sex=c("Female","Male")))

## with time-dependent effect
fit.tarr <- ARR(Hist(time,status)~strata(sex),data=Melanoma,cause=1)
plot(fit.tarr,newdata=data.frame(sex=c("Female","Male")))

## logistic risk regression
fit.lrr <- LRR(Hist(time,status)~sex,data=Melanoma,cause=1)
summary(fit.lrr)


# Single continuous factor

## tumor thickness on the log-scale
Melanoma$logthick <- log(Melanoma$thick)

## absolute risk regression 
fit2.arr <- ARR(Hist(time,status)~logthick,data=Melanoma,cause=1)
print(fit2.arr)
# show predicted cumulative incidences
plot(fit2.arr,col=1:5,newdata=data.frame(logthick=quantile(Melanoma$logthick)))

## comparison with nearest neighbor non-parametric Aalen-Johansen estimate
library(prodlim)
fit2.aj <- prodlim(Hist(time,status)~logthick,data=Melanoma)
plot(fit2.aj,conf.int=FALSE,newdata=data.frame(logthick=quantile(Melanoma$logthick)))
plot(fit2.arr,add=TRUE,col=1:5,lty=3,newdata=data.frame(logthick=quantile(Melanoma$logthick)))

## logistic risk regression
fit2.lrr <- LRR(Hist(time,status)~logthick,data=Melanoma,cause=1)
summary(fit2.lrr)

## change model for censoring weights
library(rms)
fit2a.lrr <- LRR(Hist(time,status)~logthick,
                 data=Melanoma,
                 cause=1,
                 cens.model="cox",
                 cens.formula=~sex+epicel+ulcer+age+logthick)
summary(fit2a.lrr)

##  compare prediction performance
Score(list(ARR=fit2.arr,AJ=fit2.aj,LRR=fit2.lrr),formula=Hist(time,status)~1,data=Melanoma)


# multiple regression
library(riskRegression)
library(prodlim)
# absolute risk model
multi.arr <- ARR(Hist(time,status)~logthick+sex+age+ulcer,data=Melanoma,cause=1)

# stratified model allowing different baseline risk for the two gender
multi.arr <- ARR(Hist(time,status)~thick+strata(sex)+age+ulcer,data=Melanoma,cause=1)

# stratify by a continuous variable: strata(age)
multi.arr <- ARR(Hist(time,status)~tp(thick,power=0)+strata(age)+sex+ulcer,
                 data=Melanoma,
                 cause=1)

fit.arr2a <- ARR(Hist(time,status)~tp(thick,power=1),data=Melanoma,cause=1)
summary(fit.arr2a)
fit.arr2b <- ARR(Hist(time,status)~timevar(thick),data=Melanoma,cause=1)
summary(fit.arr2b)

## logistic risk model
fit.lrr <- LRR(Hist(time,status)~thick,data=Melanoma,cause=1)
summary(fit.lrr)





## nearest neighbor non-parametric Aalen-Johansen estimate
library(prodlim)
fit.aj <- prodlim(Hist(time,status)~thick,data=Melanoma)
plot(fit.aj,conf.int=FALSE)

# prediction performance
x <- Score(list(fit.arr2a,fit.arr2b,fit.lrr),
             data=Melanoma,
             formula=Hist(time,status)~1,
             cause=1,
             split.method="none")


}
\references{
Thomas A Gerds, Thomas H Scheike, and Per K Andersen. Absolute risk
regression for competing risks: interpretation, link functions, and
prediction. Statistics in medicine, 31(29):3921--3930, 2012.

Scheike, Zhang and Gerds (2008), Predicting cumulative incidence probability
by direct binomial regression, Biometrika, 95, 205-220.

Scheike and Zhang (2007), Flexible competing risks regression modelling and
goodness of fit, LIDA, 14, 464-483.

Martinussen and Scheike (2006), Dynamic regression models for survival data,
Springer.
}
\author{
Thomas A. Gerds \email{tag@biostat.ku.dk}, Thomas H. Scheike \email{ts@biostat.ku.dk}
}
\keyword{survival}
