
#' Operate on handleables by chunk
#'
#' It is often impractical, inefficient, or impossible to perform
#' an operation on a vector of geometries with all the geometries loaded
#' into memory at the same time. These functions generalize the pattern of
#' split-apply-combine to one or more handlers recycled along a common
#' length. These functions are designed for developers rather than users
#' and should be considered experimental.
#'
#' @param handleables A single handleable or a `list()` of handleables
#'   recycleable along a common length.
#' @param fun A function called like
#'   `fun(!!! transformed_handleables, !!! vector_args, !!! args)` for each chunk.
#'   For [wk_chunk_map_feature()] this must be length-stable (i.e., return
#'   a value whose size is the recycled length of handleables and vector_args
#'   for that chunk).
#' @param vector_args Vectorized arguments to `fun`.
#' @param args Non-vectorized arguments to `fun`.
#' @param input_handler_factory A function of `handleable` applied to
#'   handleable inputs. The default, [wk_writer()], will result in `fun`
#'   getting called with a clone of the handleables for each chunk. Another
#'   useful pattern is to return a single type of handler so that all
#'   `handleables` have a common type.
#' @param output_template A vector whose subset-assign method will
#'   get called for every chunk or `NULL` to ignore the output of `fun`.
#' @param strategy A function of `handleables` and `n_features`
#'   such as that returned by [wk_chunk_strategy_feature()].
#'
#' @return `output_template` of the recycled common length of `handleables`
#'   and `vector_args` filled with values generated by `fun`.
#' @export
#'
#' @examples
#' # apply a transformation or calculate a value using the data frame version
#' # of the geometries (but without resolving all of them at once)
#' wk_chunk_map_feature(
#'   wk_linestring(xy(1:10, 1:10), rep(1:5, each = 2)),
#'   function(features) {
#'     coords <- wk_coords(features)
#'     vapply(split(coords, coords$feature_id), nrow, integer(1))
#'   },
#'   output_template = integer()
#' )
#'
wk_chunk_map_feature <- function(handleables, fun,
                                 vector_args = NULL,
                                 args = NULL,
                                 input_handler_factory = wk_writer,
                                 output_template = NULL,
                                 strategy = wk_chunk_strategy_feature(chunk_size = 1e4)) {
  if (is_handleable(handleables)) {
    handleables <- list(handleables)
  } else if (!is.list(handleables)) {
    stop("`handleables` must be a list() or a single handleable", call. = FALSE)
  }

  check_is_handleable <- vapply(handleables, is_handleable, logical(1))
  if (any(!check_is_handleable)) {
    stop("All items in `handleables` must be objects with a wk_handle() method", call. = FALSE)
  }

  stopifnot(
    is.function(fun),
    is.null(vector_args) || inherits(vector_args, "data.frame"),
    is.null(args) || is.list(args),
    is.function(input_handler_factory),
    is.function(strategy)
  )

  # get the number of features
  vector_metas <- lapply(handleables, wk_vector_meta)
  sizes <- vapply(vector_metas, "[[", "size", FUN.VALUE = double(1))
  sizes_unknown <- is.na(sizes)
  sizes[is.na(sizes)] <- vapply(
    handleables[is.na(sizes)],
    function(handleable) nrow(wk_meta(handleable)),
    integer(1)
  )

  # consider vector_args as number of features if passed
  if (!is.null(vector_args)) {
    sizes <- c(sizes, nrow(vector_args))
  }

  # calculate the recycled length
  all_sizes <- unique(sizes)
  non_one_sizes <- unique(sizes[sizes != 1L])
  if ((length(all_sizes) > 1L) && (length(non_one_sizes) != 1L)) {
    stop(
      "All items in `handleables` and `vector_meta` must be recycleable to a common length.",
      call. = FALSE
    )
  }

  n_features <- if (length(all_sizes) == 1L) all_sizes else non_one_sizes

  # if the recycled size is 0, don't use the chunking strategy and return early
  if (n_features == 0) {
    return(output_template[integer(0)])
  }

  chunks <- strategy(handleables, n_features)
  output <- output_template[rep(NA_integer_, n_features)]

  for (chunk_i in seq_len(nrow(chunks))) {
    from <- chunks$from[chunk_i]
    to <- chunks$to[chunk_i]

    handleables_chunk <- lapply(seq_along(handleables), function(handleable_i) {
      handleable <- handleables[[handleable_i]]
      if (sizes[handleable_i] == 1L) {
        wk_handle(handleable, input_handler_factory(handleable))
      } else {
        wk_handle_slice(
          handleable,
          from = from, to = to,
          handler =  input_handler_factory(handleable)
        )
      }
    })

    vector_args_chunk <- if (identical(nrow(vector_args), 1L)) {
      vector_args
    } else {
      vector_args[from:to, , drop = FALSE]
    }

    result_i <- do.call(fun, c(handleables_chunk, vector_args_chunk, args))
    if (!is.null(output)) {
      output[from:to] <- result_i
    }
  }

  output
}

#' Chunking strategies
#'
#' @param reduce For [wk_chunk_strategy_coordinates()] this refers to
#'   the function used with [Reduce()] to combine coordinate counts
#'   from more than one handleable.
#' @param n_chunks,chunk_size Exactly one of the number of
#'   chunks or the chunk size. For [wk_chunk_strategy_feature()]
#'   the chunk size refers to the number of features; for
#'   [wk_chunk_strategy_coordinates()] this refers to the number
#'   of coordinates as calculated from multiple handleables
#'   using `reduce`.
#'
#' @return A function that returns a `data.frame` with columns `from` and `to`
#'   when called with a `handleable` and the feature count.
#' @export
#'
#' @examples
#' feat <- c(as_wkt(xy(1:4, 1:4)), wkt("LINESTRING (1 1, 2 2)"))
#' wk_chunk_strategy_single()(list(feat), 5)
#' wk_chunk_strategy_feature(chunk_size = 2)(list(feat), 5)
#' wk_chunk_strategy_coordinates(chunk_size = 2)(list(feat), 5)
#'
wk_chunk_strategy_single <- function() {
  function(handleables, n_features) {
    new_data_frame(list(from = 1, to = n_features))
  }
}

#' @rdname wk_chunk_strategy_single
#' @export
wk_chunk_strategy_feature <- function(n_chunks = NULL, chunk_size = NULL) {
  force(n_chunks)
  force(chunk_size)

  function(handleables, n_features) {
    chunk_info <- chunk_info(n_features, n_chunks = n_chunks, chunk_size = chunk_size)

    from <- (chunk_info$chunk_size * (seq_len(chunk_info$n_chunks) - 1L)) + 1L
    to <- chunk_info$chunk_size * seq_len(chunk_info$n_chunks)
    to[chunk_info$n_chunks] <- n_features
    new_data_frame(list(from = from, to = to))
  }
}

#' @rdname wk_chunk_strategy_single
#' @export
wk_chunk_strategy_coordinates <- function(n_chunks = NULL, chunk_size = NULL, reduce = "*") {
  force(n_chunks)
  force(reduce)

  function(handleables, n_features) {
    coord_count <- lapply(handleables, function(handleable) {
      vm <- wk_vector_meta(handleable)
      if (identical(vm$geometry_type, 1L)) {
        1L
      } else {
        wk_count(handleable)$n_coord
      }
    })

    coord_count <- Reduce(reduce, coord_count)
    if (identical(coord_count, 1L)) {
      return(wk_chunk_strategy_feature(n_chunks, chunk_size)(handleables, n_features))
    }

    coord_count <- rep_len(coord_count, n_features)

    coord_count_total <- sum(coord_count)
    chunk_info <- chunk_info(coord_count_total, n_chunks, chunk_size)

    from <- rep(NA_integer_, chunk_info$n_chunks)
    to <- rep(NA_integer_, chunk_info$n_chunks)
    from[1] <- 1L

    coord_count_chunk <- (coord_count_total / chunk_info$n_chunks)
    coord_count_feat <- cumsum(coord_count)

    for (chunk_id in seq_len(chunk_info$n_chunks - 1L)) {
      next_coord_gt <- coord_count_feat >= coord_count_chunk
      if (!any(next_coord_gt)) {
        to[chunk_id] <- n_features
        break
      }

      i <- max(min(which(next_coord_gt)), from[chunk_id] + 1L)
      to[chunk_id] <- i
      from[chunk_id + 1L] <- i + 1L
      coord_count[1:i] <- 0L
      coord_count_feat <- cumsum(coord_count)
    }

    valid <- !is.na(from)
    from <- from[valid]
    to <- to[valid]

    if (is.na(to[length(to)])) {
      to[length(to)] <- n_features
    }

    new_data_frame(list(from = from, to = to))
  }
}

chunk_info <- function(n_features, n_chunks = NULL, chunk_size = NULL) {
  if (is.null(n_chunks) && is.null(chunk_size)) {
    stop("Must specify exactly one of `n_chunks` or `chunk_size`", call. = FALSE)
  } else if (is.null(n_chunks)) {
    n_chunks <- ((n_features - 1L) %/% chunk_size) + 1L
  } else if (is.null(chunk_size)) {
    if (n_features == 0) {
      n_chunks <- 0L
      chunk_size <- 1L
    } else {
      chunk_size <- ((n_features - 1L) %/% n_chunks) + 1L
    }
  } else {
    stop("Must specify exactly one of `n_chunks` or `chunk_size`", call. = FALSE)
  }

  list(n_chunks = n_chunks, chunk_size = chunk_size)
}
