/*
*  RAL -- Rubrica Addressbook Library
*  file: init.c
*
*  Copyright (C) 2006 Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

// #define GETTEXT_PACKAGE "libral"

#include <glib.h>
#include <glib/gi18n-lib.h>

#include "rlib.h"
#include "manager.h"
#include "group_box.h"
#include "timeout.h"




RPluginManager *manager   = NULL;
RGroupBox      *group_box = NULL;
RTimeout       *timeout   = NULL;
GList          *adrbooks  = NULL;
     
gboolean init             = FALSE;
gchar*   release          = RUBRICA_VERSION;
gchar*   code_name        = LIBRAL_CODE_NAME;
gchar*   info_code_name   = LIBRAL_INFO_CODE_NAME;


static void r_lib_remove(gpointer obj, gpointer userdata);


/*  Private callback
 */
static void 
r_lib_remove(gpointer obj, gpointer userdata)
{
  g_return_if_fail(obj != NULL);

  adrbooks = g_list_remove(adrbooks, obj);
}


/** Public **/


/**
 * r_lib_get_release
 *
 * Get the libral release
 *
 * Returns: a gchar*
 */
gchar*
r_lib_get_release (void)
{
  return release;
}


/**
 * r_lib_get_code_name
 *
 * Get the libral release code name
 *
 * Returns: a gchar*
 */
gchar* 
r_lib_get_code_name(void)
{
  return code_name;
}

/**
 * r_lib_get_ral_info
 *
 * Get the libral release infos
 *
 * Returns: a gchar*
 */
gchar*
r_lib_get_ral_info (void)
{
  return info_code_name;
}


/**
 * r_lib_init
 *
 * initialize the libral library
 *
 * Returns: %TRUE if libral was successfully initialized, %FALSE otherwise
 */
gboolean 
r_lib_init(void)
{
  manager = r_plugin_manager_new();
  if(manager)
    r_plugin_manager_scan_directory(manager, LIBRAL_PLUGINS_DIR);      
  else
    g_error(_("\nCan't initialize r"));
  
  group_box = r_group_box_new();
  if (!group_box)
    g_error(_("\nCan't initialize r"));
 
  timeout = r_timeout_new(0);
  if (!timeout)
    g_error(_("\nCan't initialize r"));

  init = TRUE;
  return init;
}


/**
 * r_lib_admit
 * @abook: a gpointer
 *
 * admit a new addressbook (a #RAbook) to the list of active addressbooks
 * User should not use this function. 
 * It is called during RAbook's object initialization.
 */
void
r_lib_admit(gpointer abook)
{
  g_signal_connect(G_OBJECT(abook), "addressbook_closed", 
		   G_CALLBACK(r_lib_remove), NULL);
  
  adrbooks = g_list_append(adrbooks, abook);
}


/**
 * r_lib_is_running
 * 
 * Check if libral is running
 *
 * Returns: %TRUE if libral is running, %FALSE otherwise
 */
gboolean 
r_lib_is_running(void)
{
  return init;
}


/**
 * r_lib_get_manager
 *
 * Get the plugin manager, a #RPluginManager
 *
 * Returns: a gpointer
 */
gpointer 
r_lib_get_manager(void)
{
  return manager;
}


/**
 * r_lib_get_group_set
 *
 * Get the active group set, a #RGroupSet
 *
 * Returns: a gpointer
 */
gpointer 
r_lib_get_group_box (void)
{
  return group_box;
}


gpointer 
r_lib_get_timeout(void)
{
  return timeout;
}


/**
 * r_lib_get_addressbooks
 * 
 * Get the list of the opened addressbooks
 *
 * Returns: a #GList of #RAbooks
 */
GList* 
r_lib_get_addressbooks (void)
{
  return adrbooks;
}


/**
 * r_lib_get_nth_addressbook
 *
 * Get the nth of the opened addressbooks
 *
 * Returns: a gpointer (#RAbook)
 */
gpointer 
r_lib_get_nth_addressbook(gint n)
{
  return g_list_nth_data(adrbooks, n);
}
