/*
*  RAL -- Rubrica Addressbook Library
*  file: timeout.c
*  
*  Copyright (C) 2000-2001 Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <glib.h>
#include <glib-object.h>

#include "timeout.h"


struct _RTimeoutPrivate {
  gint timeout;
  gboolean have_timeout;

  gboolean dispose_has_run;
};

static void r_timeout_class_init (RTimeoutClass* klass);
static void r_timeout_init       (RTimeout* obj);

static void r_timeout_dispose    (RTimeout* obj);
static void r_timeout_finalize   (RTimeout* obj);

static gboolean r_timeout_emit_signal (RTimeout *timeout);


/*    signals enumeration 
 */
enum {
  TIMEOUT,                 /* signal: timeout, countdown is done */ 
  LAST_SIGNAL
};


static guint r_timeout_signals[LAST_SIGNAL] = {0};



GType
r_timeout_get_type()
{
  static GType r_timeout_type = 0;
  
  if (!r_timeout_type)
    {
      static const GTypeInfo r_timeout_info =
	{
	  sizeof(RTimeoutClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) r_timeout_class_init,
	  NULL,
	  NULL,
	  sizeof(RTimeout),
	  0,
	  (GInstanceInitFunc) r_timeout_init
	};
      
      r_timeout_type = g_type_register_static (G_TYPE_OBJECT,
					       "RTimeout",
					       &r_timeout_info, 0);
    }
  
  return r_timeout_type;
}


static void
r_timeout_class_init(RTimeoutClass* klass)
{
  GObjectClass *class;
  
  class = G_OBJECT_CLASS (klass);
  
  class->dispose  = (GObjectFinalizeFunc) r_timeout_dispose;
  class->finalize = (GObjectFinalizeFunc) r_timeout_finalize;
  
  r_timeout_signals[TIMEOUT] = g_signal_new("timeout", 
	       R_TIMEOUT_TYPE,
	       G_SIGNAL_RUN_LAST,
	       G_STRUCT_OFFSET(RTimeoutClass, timeout_signal),
	       NULL, 
	       NULL,
	       g_cclosure_marshal_VOID__VOID,
	       G_TYPE_NONE,
	       0,
	       G_TYPE_NONE);
}


static void
r_timeout_init(RTimeout* timeout)
{  
  timeout->private = g_new(RTimeoutPrivate, 1);
  
  timeout->private->timeout      = 0;
  timeout->private->have_timeout = FALSE;
  
  timeout->private->dispose_has_run = FALSE;
}


static void 
r_timeout_dispose (RTimeout* timeout)
{
  g_return_if_fail(IS_R_TIMEOUT(timeout));
  
  if (timeout->private->dispose_has_run)
    return;
  
  timeout->private->dispose_has_run = TRUE;
}


static void 
r_timeout_finalize (RTimeout* timeout)
{
  g_return_if_fail(IS_R_TIMEOUT(timeout));
  
  g_free(timeout->private);
  timeout->private = NULL;
}


static gboolean
r_timeout_emit_signal(RTimeout *timeout)
{
  g_return_val_if_fail(IS_R_TIMEOUT(timeout), FALSE);
  
  if (timeout->private->have_timeout)
    {
      g_signal_emit_by_name(timeout, "timeout", G_TYPE_NONE);
      
      return TRUE;
    }
  
  return FALSE;
}



/* Public Functions
 */


/**
 * r_timeout_new
 * @time: time
 *
 * Create a new timer. 
 * Timer will wait "time" seconds, when a "timeout" signal is emitted.
 * If time is 0, then timer will wait 15 minutes, before emit signal.
 *
 * returns: a #RTimeout*
 */
RTimeout* 
r_timeout_new(guint32 time)
{
  RTimeout* timeout;  

  timeout = g_object_new(r_timeout_get_type(), NULL);
  timeout->wait_time = time;

  return (RTimeout*) timeout;
}

/**
 * r_timeout_free
 * @timeout: a #RTimeout
 *
 * free the #RTimeout object
 */
void 
r_timeout_free(RTimeout *timeout)
{
  g_return_if_fail(IS_R_TIMEOUT(timeout));
  
  g_object_unref(timeout);
}


/**
 * r_timeout_start
 * @timeout: #RTimeout
 *
 * start the timeout object
 */
void 
r_timeout_start(RTimeout *timeout)
{
  g_return_if_fail(IS_R_TIMEOUT(timeout));

  /*     default time to wait is 15 minutes 
   */
  if (timeout->wait_time == 0)
    timeout->wait_time = 15;  
  
  /*     wait_time *= 60000 (1 minute = 60 seconds * 1000 microseconds)
  */
  timeout->wait_time *= 60000;
  timeout->private->timeout = 
    g_timeout_add(timeout->wait_time, 
		  (GSourceFunc) r_timeout_emit_signal, timeout);
  
  timeout->private->have_timeout = TRUE;
}


/**
 * r_timeout_enable
 * @timeout: a #RTimeout
 *
 * enable the timeout
 */
void 
r_timeout_enable(RTimeout *timeout)
{
  g_return_if_fail(IS_R_TIMEOUT(timeout));

  r_timeout_start(timeout);  
}


/**
 * r_timeout_disable
 * @timeout: a #RTimeout
 *
 * disable the timeout
 */
void 
r_timeout_disable(RTimeout *timeout)
{
  g_return_if_fail(IS_R_TIMEOUT(timeout));

  if (timeout->private->timeout != 0)
    g_source_remove(timeout->private->timeout);
  else
    return;

  timeout->private->timeout = 0;
  timeout->private->have_timeout = FALSE;
}


/**
 * r_timeout_restart
 * @timeout: a #RTimeout
 * @time: anew time
 *
 * restart the timeout, but waiting "time" seconds
 */
void 
r_timeout_restart(RTimeout *timeout, guint32 time)
{
  g_return_if_fail(IS_R_TIMEOUT(timeout));

  r_timeout_disable(timeout);

  timeout->wait_time = time;
  r_timeout_enable(timeout);
}


/**
 * r_timeout_is_running
 * @timeout: a #RTimeout
 *
 * check if timer is running or is disabled
 *
 * returns: a gboolean, %TRUE if timeout is running, %FALSE otherwise.
 */
gboolean         
r_timeout_is_running(RTimeout* timeout)
{
  g_return_val_if_fail(IS_R_TIMEOUT(timeout), FALSE);

  return timeout->private->have_timeout;
}
