/*
*  RAL -- Rubrica Addressbook Library
*  file: ref.c
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <glib/gi18n-lib.h>
#include <gdk/gdkkeysyms.h>

#include "calendar.h"

#define BUFSIZE 64

enum {
  PROP_0,
  CALENDAR_DAY,
  CALENDAR_MONTH,
  CALENDAR_YEAR
};


/*    signals enumeration 
 */
enum {
  CALENDAR_CHANGE,  
  LAST_SIGNAL
};


struct _RubricaCalendarPrivate {
  GtkWidget* entry;
  GtkWidget* button;
  GtkWidget* popup;
  GtkWidget* calendar;
  
  GDate* date;

  gint day;
  gint month;
  gint year;

  gboolean dispose_has_run;
};


static guint calendar_signals[LAST_SIGNAL] = {0};


#define RUBRICA_CALENDAR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE((o),  \
                                         RUBRICA_CALENDAR_TYPE,            \
                                         RubricaCalendarPrivate))


static GObjectClass *parent_class = NULL;


static void rubrica_calendar_class_init   (RubricaCalendarClass* klass);
static void rubrica_calendar_init         (RubricaCalendar* obj);

static void rubrica_calendar_finalize     (RubricaCalendar* self);
static void rubrica_calendar_dispose      (RubricaCalendar* self);


static void rubrica_calendar_set_property (GObject* obj, guint property_id,
					   GValue* value,  GParamSpec* spec);
static void rubrica_calendar_get_property (GObject* obj, guint property_id,
					   GValue* value, GParamSpec* spec);


static gboolean date_is_valid          (RubricaCalendar* calendar);

static void     update_calendar        (RubricaCalendar *calendar);
static void     hide_popup             (RubricaCalendar *calendar);
static gboolean popup_grab_on_window   (GdkWindow *window, 
					guint32 activate_time);

static void     on_day_selected        (GtkCalendar *calendar, gpointer data);
static void     on_day_selected_double (GtkCalendar *calendar, gpointer data);
static gint     on_popup_key_press     (GtkWidget *widget, GdkEventKey *event,
					gpointer data);
static gint     on_popup_button_press  (GtkWidget *widget, 
					GdkEventButton *event, gpointer data);
static void     on_button_toggled_cb   (GtkToggleButton *togglebutton, 
					gpointer data);


static gboolean
date_is_valid(RubricaCalendar* calendar)
{
  RubricaCalendarPrivate *priv;
    
  g_return_val_if_fail(IS_RUBRICA_CALENDAR(calendar), FALSE);
  
  priv = RUBRICA_CALENDAR_GET_PRIVATE (calendar);  
  
  if (priv->day && (priv->month+1) && priv->year)
    return TRUE;
  
  return FALSE;
}


static void
update_calendar (RubricaCalendar *calendar)
{
  RubricaCalendarPrivate *priv = RUBRICA_CALENDAR_GET_PRIVATE (calendar);  
  gchar buffer[BUFSIZE];

  if (date_is_valid(calendar))
    {
      g_date_set_dmy(priv->date, priv->day, priv->month+1, priv->year);
      
      g_date_strftime(buffer, BUFSIZE, "%Ex", priv->date);
      gtk_entry_set_text(GTK_ENTRY(priv->entry), buffer);    
      
      g_signal_emit_by_name(calendar, "calendar-change", 
			    priv->date, G_TYPE_POINTER);
    }
  else
    {
      gtk_entry_set_text(GTK_ENTRY(priv->entry), _("unknown"));
    }
}


static void
hide_popup (RubricaCalendar *calendar)
{
  RubricaCalendarPrivate *priv;

  priv = RUBRICA_CALENDAR_GET_PRIVATE (calendar);

  gtk_widget_hide (priv->popup);
  gtk_grab_remove (priv->popup);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (priv->button), FALSE);
}


static gboolean
popup_grab_on_window (GdkWindow *window, guint32 activate_time)
{
  if ((gdk_pointer_grab(window, TRUE,
			GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
			GDK_POINTER_MOTION_MASK,
			NULL, NULL, activate_time) == 0))
    {
      if (gdk_keyboard_grab (window, TRUE, activate_time) == 0)
	return TRUE;
      else
	{
	  gdk_pointer_ungrab (activate_time);
	  return FALSE;
	}
    }
  
  return FALSE;
}


static void
on_day_selected (GtkCalendar *cal, gpointer data)
{
  RubricaCalendar *calendar = (RubricaCalendar *) data;
  RubricaCalendarPrivate *priv = RUBRICA_CALENDAR_GET_PRIVATE (data);  
  guint day, month, year;
  
  gtk_calendar_get_date (cal, &year, &month, &day);

  priv->day   = day;
  priv->month = month;
  priv->year  = year;

  update_calendar(RUBRICA_CALENDAR(calendar));
}


static void
on_day_selected_double (GtkCalendar *calendar, gpointer data)
{
  hide_popup(RUBRICA_CALENDAR(data));
}


static gint
on_delete_popup (GtkWidget *widget, gpointer data)
{
  hide_popup(RUBRICA_CALENDAR(data));
  
  return TRUE;
}


static gint
on_popup_key_press (GtkWidget *widget, GdkEventKey *event, gpointer data)
{
  /* evaluate event so calendar is navigable with keyboard */
  switch (event->keyval)
    {
    case GDK_Escape:
      break;
      
    case GDK_Return:
    case GDK_KP_Enter:
      break;
      
    default:
      return FALSE;
    }
  
  g_signal_stop_emission_by_name (widget, "key_press_event");
  hide_popup (RUBRICA_CALENDAR(data));

  return TRUE;
}



/* This function is yanked from gtkcombo.c 
*/
static gint
on_popup_button_press (GtkWidget *widget, GdkEventButton *event, gpointer data)
{
  GtkWidget *child; 
  
  /* We don't ask for button press events on the grab widget, so
   *  if an event is reported directly to the grab widget, it must
   *  be on a window outside the application (and thus we remove
   *  the popup window). Otherwise, we check if the widget is a child
   *  of the grab widget, and only remove the popup window if it
   *  is not.
   */
  child = gtk_get_event_widget ((GdkEvent *)event);

  if (child != widget) 
    {
      while (child) 
	{
	  if (child == widget) return FALSE;
	  child = child->parent;
      }
    }
  
  hide_popup (RUBRICA_CALENDAR(data));
  
  return TRUE;
}


static void
on_button_toggled_cb (GtkToggleButton *togglebutton, gpointer data)
{
  RubricaCalendar *calendar = (RubricaCalendar *)data;
  RubricaCalendarPrivate *priv = RUBRICA_CALENDAR_GET_PRIVATE (calendar);
  
  gint x, y, width, height;
  GtkRequisition req;
    
  if (gtk_toggle_button_get_active (togglebutton))
    {
      if (date_is_valid(calendar))
	{
	  gtk_calendar_select_month (GTK_CALENDAR (priv->calendar),
				     priv->month, priv->year);
	  gtk_calendar_select_day (GTK_CALENDAR (priv->calendar), priv->day);
	}
      
      /* show calendar 
       */
      gtk_widget_size_request (priv->popup, &req);
      gdk_window_get_origin (GDK_WINDOW (priv->button->window), &x, &y);
      x += priv->button->allocation.x;
      y += priv->button->allocation.y;
      width = priv->button->allocation.width;
      height = priv->button->allocation.height;
      x += width - req.width;
      y += height;
      if (x < 0) x = 0;
      if (y < 0) y = 0;
      
      gtk_grab_add (priv->popup);
      gtk_window_move (GTK_WINDOW (priv->popup), x, y);
      gtk_widget_show (priv->popup);
      gtk_widget_grab_focus (priv->calendar);
      popup_grab_on_window (priv->popup->window, 
			    gtk_get_current_event_time ());
    }
}




GType
rubrica_calendar_get_type()
{
  static GType calendar_type = 0;
  
  if (!calendar_type)
    {
      static const GTypeInfo calendar_info =
	{
	  sizeof(RubricaCalendarClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) rubrica_calendar_class_init,
	  NULL,
	  NULL,
	  sizeof(RubricaCalendar),
	  0,
	  (GInstanceInitFunc) rubrica_calendar_init
	};

      calendar_type = g_type_register_static (GTK_TYPE_HBOX, 
					      "RubricaCalendar", 
					      &calendar_info, 0);
    }
  
  return calendar_type;
}


static void 
rubrica_calendar_dispose (RubricaCalendar* self)
{
  RubricaCalendarPrivate* priv;

  g_return_if_fail(IS_RUBRICA_CALENDAR(self));
  
  priv = RUBRICA_CALENDAR_GET_PRIVATE(self);

  if (priv->dispose_has_run)
    return;

  priv->dispose_has_run = TRUE;  
}


static void 
rubrica_calendar_finalize (RubricaCalendar* self)
{
  g_return_if_fail(IS_RUBRICA_CALENDAR(self));  
}


static void
rubrica_calendar_class_init(RubricaCalendarClass* klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  GParamSpec* pspec;
  
  parent_class = g_type_class_peek_parent (klass);

  object_class->finalize     = (GObjectFinalizeFunc) rubrica_calendar_finalize;
  object_class->dispose      = (GObjectFinalizeFunc) rubrica_calendar_dispose;

  object_class->set_property = (gpointer) rubrica_calendar_set_property;
  object_class->get_property = (gpointer) rubrica_calendar_get_property;

  g_type_class_add_private (klass, sizeof(RubricaCalendarPrivate));

  /**
   */
  calendar_signals[CALENDAR_CHANGE] =
    g_signal_new("calendar-change", 
		 RUBRICA_CALENDAR_TYPE,
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(RubricaCalendarClass, calendar_change),
		 NULL,
		 NULL,
		 g_cclosure_marshal_VOID__POINTER,
		 G_TYPE_NONE,            /* return type */
		 1,                      /* params      */
		 G_TYPE_POINTER);        /* params type: error code */

  pspec = g_param_spec_int("calendar-day",
			   "day",
			   "day",
			   0,      // min
			   31,     // max
			   0,      // default, invalid day
			   G_PARAM_READWRITE);
  g_object_class_install_property(object_class, CALENDAR_DAY, pspec);

  pspec = g_param_spec_int("calendar-month",
			   "month",
			   "month",
			   0,       // min
			   12,      // max
			   0,       // default, invalid month
			   G_PARAM_READWRITE);
  g_object_class_install_property(object_class, CALENDAR_MONTH, pspec);

  pspec = g_param_spec_int("calendar-year",
			   "year",
			   "year",
			   0,        // min
			   10000,    // max
			   0,        // default, invalid year
			   G_PARAM_READWRITE);
  g_object_class_install_property(object_class, CALENDAR_YEAR, pspec);  
}


static void
rubrica_calendar_init(RubricaCalendar* self)
{
  GtkWidget* frame;
  GtkWidget* arrow;
  GtkIconTheme* theme;
  GtkWidget* image;
  GdkPixbuf* pixbuf;
  RubricaCalendarPrivate* priv;

  priv = RUBRICA_CALENDAR_GET_PRIVATE(self);
  
  gtk_box_set_homogeneous(GTK_BOX(self), FALSE);

  theme  = gtk_icon_theme_get_default();
  pixbuf = gtk_icon_theme_load_icon(theme, "stock_calendar", 
				    GTK_ICON_SIZE_BUTTON,
				    GTK_ICON_LOOKUP_USE_BUILTIN, NULL);

  priv->date = g_date_new();
  g_date_clear (priv->date, 1);
  g_date_set_time_t(priv->date, time(NULL));

  priv->day      = 0;
  priv->month    = 0;
  priv->year     = 0;
  priv->entry    = gtk_entry_new();
  priv->button   = gtk_toggle_button_new ();
  priv->popup    = gtk_window_new (GTK_WINDOW_POPUP);
  priv->calendar = gtk_calendar_new ();
    
  gtk_entry_set_editable(GTK_ENTRY(priv->entry), FALSE);
  gtk_entry_set_text(GTK_ENTRY(priv->entry), _("unknown"));
  gtk_box_pack_start (GTK_BOX (self), priv->entry, TRUE, TRUE, 0);
  gtk_widget_show (priv->entry);
  
  gtk_box_pack_start (GTK_BOX (self), priv->button, FALSE, FALSE, 0);
  if (pixbuf)
    {
      image = gtk_image_new_from_pixbuf(pixbuf);
      gtk_button_set_image(GTK_BUTTON(priv->button), image);
      gtk_button_set_relief(GTK_BUTTON(priv->button), GTK_RELIEF_NONE);
      gtk_widget_show (image);
      
      gdk_pixbuf_unref (pixbuf);
    }
  else
    {
      arrow = (GtkWidget *) gtk_arrow_new (GTK_ARROW_DOWN, GTK_SHADOW_NONE);
      gtk_container_add (GTK_CONTAINER (priv->button), arrow);
      gtk_widget_show (arrow);
    }
  gtk_widget_show (priv->button);

  frame = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_ETCHED_IN);
  gtk_widget_show(frame);
  
  gtk_window_set_resizable (GTK_WINDOW (priv->popup), FALSE);
  gtk_container_add(GTK_CONTAINER(priv->popup), frame);
  gtk_widget_set_events (priv->popup, gtk_widget_get_events (priv->popup) | 
			 GDK_KEY_PRESS_MASK);
  
  gtk_container_add (GTK_CONTAINER (frame), priv->calendar);
  gtk_widget_show (priv->calendar);

  g_signal_connect (G_OBJECT(priv->button), "toggled",
		    G_CALLBACK (on_button_toggled_cb), self);
  
  g_signal_connect (priv->popup, "delete_event",
		    G_CALLBACK (on_delete_popup), self);
  g_signal_connect (priv->popup, "key_press_event",
		    G_CALLBACK (on_popup_key_press), self);
  g_signal_connect (priv->popup, "button_press_event",
		    G_CALLBACK (on_popup_button_press), self);
  
  g_signal_connect (G_OBJECT (priv->calendar), "day-selected",
		    G_CALLBACK (on_day_selected), self);
  g_signal_connect (G_OBJECT (priv->calendar), "day-selected-double-click",
		    G_CALLBACK (on_day_selected_double), self);
  
  priv->dispose_has_run = FALSE;
}


static void 
rubrica_calendar_set_property (GObject* obj, guint property_id,
			       GValue* value, GParamSpec* spec)
{
  RubricaCalendar* self = (RubricaCalendar*) obj;
  RubricaCalendarPrivate* priv = RUBRICA_CALENDAR_GET_PRIVATE(self);

  switch (property_id) 
    {
    case CALENDAR_DAY:
      priv->day = g_value_get_int(value);
      break;

    case CALENDAR_MONTH:
      priv->month = g_value_get_int(value);
      break;

    case CALENDAR_YEAR:
      priv->year = g_value_get_int(value);
      break;

    default: 
      break; 
    } 

  update_calendar(self);
} 
 

static void 
rubrica_calendar_get_property (GObject* obj, guint property_id,
			       GValue* value, GParamSpec* spec)
{
  RubricaCalendar* self = (RubricaCalendar*) obj;
  RubricaCalendarPrivate* priv = RUBRICA_CALENDAR_GET_PRIVATE(self);

  switch (property_id) 
    {    
    case CALENDAR_DAY:
      g_value_set_int(value, priv->day);
      break;

    case CALENDAR_MONTH:
      g_value_set_int(value, priv->month);
      break;

    case CALENDAR_YEAR:
      g_value_set_int(value, priv->year);
      break;

    default:
      break;  
    }  
}




/*   Public
*/
/**
 * rubrica_calendar_new
 *
 * create a #RubricaCalendar
 *
 * returns: a new #RubricaCalendar*
 */
GtkWidget*
rubrica_calendar_new()
{
  GtkWidget* calendar;

  calendar = GTK_WIDGET(g_object_new(RUBRICA_CALENDAR_TYPE, NULL));

  return calendar;
}


GtkWidget* 
rubrica_calendar_get_button (RubricaCalendar *cal)
{
  RubricaCalendarPrivate* priv;

  g_return_val_if_fail(IS_RUBRICA_CALENDAR(cal), NULL);
 
  priv = RUBRICA_CALENDAR_GET_PRIVATE(cal);

  return priv->button;
}

gboolean 
rubrica_calendar_has_date  (RubricaCalendar *cal)
{
  g_return_val_if_fail(IS_RUBRICA_CALENDAR(cal), FALSE);

  return date_is_valid(cal);
}


GDate* 
rubrica_calendar_get_gdate (RubricaCalendar *cal)
{
  RubricaCalendarPrivate *priv;

  g_return_val_if_fail(IS_RUBRICA_CALENDAR(cal), NULL);

  priv = RUBRICA_CALENDAR_GET_PRIVATE (cal);
  
  if (date_is_valid(cal))
    return priv->date;
  
  return NULL;
}


gchar*     
rubrica_calendar_get_date (RubricaCalendar *cal)
{
  RubricaCalendarPrivate *priv;
  gchar buffer[BUFSIZE];  

  g_return_val_if_fail(IS_RUBRICA_CALENDAR(cal), NULL);

  priv = RUBRICA_CALENDAR_GET_PRIVATE (cal);  

  if (date_is_valid(cal))
    {
      g_date_set_dmy(priv->date, priv->day, priv->month, priv->year);      
      g_date_strftime(buffer, BUFSIZE, "%Ex", priv->date);
      
      return g_strdup_printf("%s", buffer);
    }
  
  return NULL;
}


void 
rubrica_calendar_clean(RubricaCalendar* calendar)
{
  RubricaCalendarPrivate *priv;  

  g_return_if_fail(IS_RUBRICA_CALENDAR(calendar));
  
  priv = RUBRICA_CALENDAR_GET_PRIVATE (calendar);  

  priv->day   = 0;
  priv->month = 0;
  priv->year  = 0;
  
  update_calendar (calendar);
}
