/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2020      Evan Nemerson <evan@nemerson.com>
 *   2020      Himanshi Mathur <himanshi18037@iiitd.ac.in>
 */

#define SIMDE_TEST_X86_AVX512_INSN fmadd

#include <test/x86/avx512/test-avx512.h>
#include <simde/x86/avx512/set.h>
#include <simde/x86/avx512/fmadd.h>

static int
test_simde_mm512_fmadd_ps (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float32 a[16];
    const simde_float32 b[16];
    const simde_float32 c[16];
    const simde_float32 r[16];
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(    86.91), SIMDE_FLOAT32_C(   -86.70), SIMDE_FLOAT32_C(   -99.59), SIMDE_FLOAT32_C(   -96.56),
        SIMDE_FLOAT32_C(    26.42), SIMDE_FLOAT32_C(   -58.48), SIMDE_FLOAT32_C(    87.49), SIMDE_FLOAT32_C(   -42.24),
        SIMDE_FLOAT32_C(   -29.05), SIMDE_FLOAT32_C(   -90.54), SIMDE_FLOAT32_C(    -3.16), SIMDE_FLOAT32_C(    -0.51),
        SIMDE_FLOAT32_C(   -85.10), SIMDE_FLOAT32_C(    22.86), SIMDE_FLOAT32_C(   -29.23), SIMDE_FLOAT32_C(    68.63) },
      { SIMDE_FLOAT32_C(    -8.87), SIMDE_FLOAT32_C(    38.36), SIMDE_FLOAT32_C(   -41.84), SIMDE_FLOAT32_C(   -42.50),
        SIMDE_FLOAT32_C(   -40.58), SIMDE_FLOAT32_C(   -31.31), SIMDE_FLOAT32_C(     6.98), SIMDE_FLOAT32_C(    83.81),
        SIMDE_FLOAT32_C(    -8.57), SIMDE_FLOAT32_C(   -15.34), SIMDE_FLOAT32_C(   -23.78), SIMDE_FLOAT32_C(   -77.95),
        SIMDE_FLOAT32_C(   -47.16), SIMDE_FLOAT32_C(    79.32), SIMDE_FLOAT32_C(    33.92), SIMDE_FLOAT32_C(   -60.25) },
      { SIMDE_FLOAT32_C(   926.25), SIMDE_FLOAT32_C(   343.33), SIMDE_FLOAT32_C(  -568.12), SIMDE_FLOAT32_C(   190.42),
        SIMDE_FLOAT32_C(   758.48), SIMDE_FLOAT32_C(  -693.26), SIMDE_FLOAT32_C(   768.06), SIMDE_FLOAT32_C(  -532.03),
        SIMDE_FLOAT32_C(  -598.68), SIMDE_FLOAT32_C(  -263.53), SIMDE_FLOAT32_C(   462.87), SIMDE_FLOAT32_C(  -449.64),
        SIMDE_FLOAT32_C(   965.10), SIMDE_FLOAT32_C(  -829.43), SIMDE_FLOAT32_C(  -763.34), SIMDE_FLOAT32_C(  -123.62) },
      { SIMDE_FLOAT32_C(   155.36), SIMDE_FLOAT32_C( -2982.48), SIMDE_FLOAT32_C(  3598.73), SIMDE_FLOAT32_C(  4294.22),
        SIMDE_FLOAT32_C(  -313.64), SIMDE_FLOAT32_C(  1137.75), SIMDE_FLOAT32_C(  1378.74), SIMDE_FLOAT32_C( -4072.16),
        SIMDE_FLOAT32_C(  -349.72), SIMDE_FLOAT32_C(  1125.35), SIMDE_FLOAT32_C(   538.01), SIMDE_FLOAT32_C(  -409.89),
        SIMDE_FLOAT32_C(  4978.42), SIMDE_FLOAT32_C(   983.83), SIMDE_FLOAT32_C( -1754.82), SIMDE_FLOAT32_C( -4258.58) } },
    { { SIMDE_FLOAT32_C(    55.42), SIMDE_FLOAT32_C(   -18.17), SIMDE_FLOAT32_C(    45.14), SIMDE_FLOAT32_C(   -85.16),
        SIMDE_FLOAT32_C(    50.52), SIMDE_FLOAT32_C(   -47.88), SIMDE_FLOAT32_C(    98.65), SIMDE_FLOAT32_C(   -58.05),
        SIMDE_FLOAT32_C(    36.78), SIMDE_FLOAT32_C(   -25.13), SIMDE_FLOAT32_C(   -36.01), SIMDE_FLOAT32_C(    89.63),
        SIMDE_FLOAT32_C(   -45.81), SIMDE_FLOAT32_C(    97.91), SIMDE_FLOAT32_C(   -70.62), SIMDE_FLOAT32_C(   -53.18) },
      { SIMDE_FLOAT32_C(    32.24), SIMDE_FLOAT32_C(   -27.44), SIMDE_FLOAT32_C(    65.86), SIMDE_FLOAT32_C(     8.09),
        SIMDE_FLOAT32_C(     3.24), SIMDE_FLOAT32_C(    42.66), SIMDE_FLOAT32_C(    54.89), SIMDE_FLOAT32_C(    43.37),
        SIMDE_FLOAT32_C(   -83.69), SIMDE_FLOAT32_C(     1.17), SIMDE_FLOAT32_C(    98.41), SIMDE_FLOAT32_C(   -87.18),
        SIMDE_FLOAT32_C(    18.23), SIMDE_FLOAT32_C(   -77.93), SIMDE_FLOAT32_C(     0.46), SIMDE_FLOAT32_C(   -26.35) },
      { SIMDE_FLOAT32_C(    39.01), SIMDE_FLOAT32_C(  -544.03), SIMDE_FLOAT32_C(  -115.13), SIMDE_FLOAT32_C(  -455.82),
        SIMDE_FLOAT32_C(   -22.84), SIMDE_FLOAT32_C(  -128.64), SIMDE_FLOAT32_C(   -36.36), SIMDE_FLOAT32_C(  -655.01),
        SIMDE_FLOAT32_C(   620.05), SIMDE_FLOAT32_C(   603.55), SIMDE_FLOAT32_C(  -758.76), SIMDE_FLOAT32_C(  -838.04),
        SIMDE_FLOAT32_C(   582.66), SIMDE_FLOAT32_C(  -464.98), SIMDE_FLOAT32_C(  -369.88), SIMDE_FLOAT32_C(   -94.91) },
      { SIMDE_FLOAT32_C(  1825.75), SIMDE_FLOAT32_C(   -45.45), SIMDE_FLOAT32_C(  2857.79), SIMDE_FLOAT32_C( -1144.76),
        SIMDE_FLOAT32_C(   140.84), SIMDE_FLOAT32_C( -2171.20), SIMDE_FLOAT32_C(  5378.54), SIMDE_FLOAT32_C( -3172.64),
        SIMDE_FLOAT32_C( -2458.07), SIMDE_FLOAT32_C(   574.15), SIMDE_FLOAT32_C( -4302.50), SIMDE_FLOAT32_C( -8651.98),
        SIMDE_FLOAT32_C(  -252.46), SIMDE_FLOAT32_C( -8095.11), SIMDE_FLOAT32_C(  -402.37), SIMDE_FLOAT32_C(  1306.38) } },
    { { SIMDE_FLOAT32_C(    26.07), SIMDE_FLOAT32_C(   -71.13), SIMDE_FLOAT32_C(    98.60), SIMDE_FLOAT32_C(   -70.70),
        SIMDE_FLOAT32_C(    71.53), SIMDE_FLOAT32_C(    53.49), SIMDE_FLOAT32_C(    72.68), SIMDE_FLOAT32_C(    87.84),
        SIMDE_FLOAT32_C(   -45.34), SIMDE_FLOAT32_C(    71.08), SIMDE_FLOAT32_C(   -99.33), SIMDE_FLOAT32_C(    72.89),
        SIMDE_FLOAT32_C(    93.15), SIMDE_FLOAT32_C(     1.12), SIMDE_FLOAT32_C(   -53.46), SIMDE_FLOAT32_C(    -2.95) },
      { SIMDE_FLOAT32_C(    46.72), SIMDE_FLOAT32_C(    35.03), SIMDE_FLOAT32_C(    51.47), SIMDE_FLOAT32_C(   -55.56),
        SIMDE_FLOAT32_C(   -77.83), SIMDE_FLOAT32_C(   -52.16), SIMDE_FLOAT32_C(   -21.06), SIMDE_FLOAT32_C(    84.17),
        SIMDE_FLOAT32_C(   -91.81), SIMDE_FLOAT32_C(     3.06), SIMDE_FLOAT32_C(   -99.63), SIMDE_FLOAT32_C(    66.46),
        SIMDE_FLOAT32_C(    56.56), SIMDE_FLOAT32_C(   -36.62), SIMDE_FLOAT32_C(   -43.03), SIMDE_FLOAT32_C(   -17.37) },
      { SIMDE_FLOAT32_C(   -77.52), SIMDE_FLOAT32_C(  -444.33), SIMDE_FLOAT32_C(   119.34), SIMDE_FLOAT32_C(  -362.18),
        SIMDE_FLOAT32_C(  -909.45), SIMDE_FLOAT32_C(  -153.91), SIMDE_FLOAT32_C(  -483.73), SIMDE_FLOAT32_C(  -362.83),
        SIMDE_FLOAT32_C(  -443.10), SIMDE_FLOAT32_C(  -477.07), SIMDE_FLOAT32_C(  -633.88), SIMDE_FLOAT32_C(  -511.56),
        SIMDE_FLOAT32_C(   534.17), SIMDE_FLOAT32_C(  -168.46), SIMDE_FLOAT32_C(   458.99), SIMDE_FLOAT32_C(     1.39) },
      { SIMDE_FLOAT32_C(  1140.47), SIMDE_FLOAT32_C( -2936.01), SIMDE_FLOAT32_C(  5194.28), SIMDE_FLOAT32_C(  3565.91),
        SIMDE_FLOAT32_C( -6476.63), SIMDE_FLOAT32_C( -2943.95), SIMDE_FLOAT32_C( -2014.37), SIMDE_FLOAT32_C(  7030.66),
        SIMDE_FLOAT32_C(  3719.57), SIMDE_FLOAT32_C(  -259.57), SIMDE_FLOAT32_C(  9262.37), SIMDE_FLOAT32_C(  4332.71),
        SIMDE_FLOAT32_C(  5802.73), SIMDE_FLOAT32_C(  -209.47), SIMDE_FLOAT32_C(  2759.37), SIMDE_FLOAT32_C(    52.63) } },
    { { SIMDE_FLOAT32_C(   -81.82), SIMDE_FLOAT32_C(    -2.63), SIMDE_FLOAT32_C(    44.58), SIMDE_FLOAT32_C(   -59.65),
        SIMDE_FLOAT32_C(    45.21), SIMDE_FLOAT32_C(   -76.49), SIMDE_FLOAT32_C(   -75.48), SIMDE_FLOAT32_C(    53.40),
        SIMDE_FLOAT32_C(    26.57), SIMDE_FLOAT32_C(   -75.11), SIMDE_FLOAT32_C(    19.86), SIMDE_FLOAT32_C(   -16.86),
        SIMDE_FLOAT32_C(   -11.73), SIMDE_FLOAT32_C(    76.83), SIMDE_FLOAT32_C(    65.77), SIMDE_FLOAT32_C(    80.51) },
      { SIMDE_FLOAT32_C(   -67.61), SIMDE_FLOAT32_C(   -22.30), SIMDE_FLOAT32_C(   -55.70), SIMDE_FLOAT32_C(   -58.55),
        SIMDE_FLOAT32_C(    62.31), SIMDE_FLOAT32_C(    -4.08), SIMDE_FLOAT32_C(     5.17), SIMDE_FLOAT32_C(   -82.00),
        SIMDE_FLOAT32_C(    48.22), SIMDE_FLOAT32_C(    41.78), SIMDE_FLOAT32_C(   -33.16), SIMDE_FLOAT32_C(     1.63),
        SIMDE_FLOAT32_C(   -75.07), SIMDE_FLOAT32_C(   -87.26), SIMDE_FLOAT32_C(   -98.23), SIMDE_FLOAT32_C(   -56.88) },
      { SIMDE_FLOAT32_C(   101.14), SIMDE_FLOAT32_C(   463.49), SIMDE_FLOAT32_C(  -165.36), SIMDE_FLOAT32_C(  -446.77),
        SIMDE_FLOAT32_C(   698.62), SIMDE_FLOAT32_C(    79.84), SIMDE_FLOAT32_C(  -912.75), SIMDE_FLOAT32_C(   -35.63),
        SIMDE_FLOAT32_C(   328.70), SIMDE_FLOAT32_C(   285.86), SIMDE_FLOAT32_C(   795.73), SIMDE_FLOAT32_C(  -788.64),
        SIMDE_FLOAT32_C(    54.13), SIMDE_FLOAT32_C(   453.39), SIMDE_FLOAT32_C(  -983.51), SIMDE_FLOAT32_C(   378.08) },
      { SIMDE_FLOAT32_C(  5632.99), SIMDE_FLOAT32_C(   522.14), SIMDE_FLOAT32_C( -2648.47), SIMDE_FLOAT32_C(  3045.74),
        SIMDE_FLOAT32_C(  3515.66), SIMDE_FLOAT32_C(   391.92), SIMDE_FLOAT32_C( -1302.98), SIMDE_FLOAT32_C( -4414.43),
        SIMDE_FLOAT32_C(  1609.91), SIMDE_FLOAT32_C( -2852.24), SIMDE_FLOAT32_C(   137.17), SIMDE_FLOAT32_C(  -816.12),
        SIMDE_FLOAT32_C(   934.70), SIMDE_FLOAT32_C( -6250.80), SIMDE_FLOAT32_C( -7444.10), SIMDE_FLOAT32_C( -4201.33) } },
    { { SIMDE_FLOAT32_C(   -76.96), SIMDE_FLOAT32_C(   -54.06), SIMDE_FLOAT32_C(    79.26), SIMDE_FLOAT32_C(    85.35),
        SIMDE_FLOAT32_C(    41.87), SIMDE_FLOAT32_C(   -15.58), SIMDE_FLOAT32_C(   -96.66), SIMDE_FLOAT32_C(    -9.92),
        SIMDE_FLOAT32_C(   -73.80), SIMDE_FLOAT32_C(   -29.81), SIMDE_FLOAT32_C(    91.72), SIMDE_FLOAT32_C(   -48.87),
        SIMDE_FLOAT32_C(   -17.07), SIMDE_FLOAT32_C(    93.49), SIMDE_FLOAT32_C(    -5.75), SIMDE_FLOAT32_C(    93.04) },
      { SIMDE_FLOAT32_C(    39.84), SIMDE_FLOAT32_C(    77.71), SIMDE_FLOAT32_C(   -51.63), SIMDE_FLOAT32_C(     9.70),
        SIMDE_FLOAT32_C(   -14.30), SIMDE_FLOAT32_C(   -42.91), SIMDE_FLOAT32_C(   -93.87), SIMDE_FLOAT32_C(   -81.43),
        SIMDE_FLOAT32_C(    85.68), SIMDE_FLOAT32_C(    85.71), SIMDE_FLOAT32_C(   -60.30), SIMDE_FLOAT32_C(    -8.91),
        SIMDE_FLOAT32_C(    31.05), SIMDE_FLOAT32_C(   -58.65), SIMDE_FLOAT32_C(   -71.10), SIMDE_FLOAT32_C(    54.08) },
      { SIMDE_FLOAT32_C(  -127.02), SIMDE_FLOAT32_C(  -918.44), SIMDE_FLOAT32_C(   394.30), SIMDE_FLOAT32_C(  -708.34),
        SIMDE_FLOAT32_C(   -74.20), SIMDE_FLOAT32_C(   427.75), SIMDE_FLOAT32_C(   192.49), SIMDE_FLOAT32_C(   187.82),
        SIMDE_FLOAT32_C(  -870.38), SIMDE_FLOAT32_C(   109.64), SIMDE_FLOAT32_C(   699.16), SIMDE_FLOAT32_C(   -41.09),
        SIMDE_FLOAT32_C(    44.51), SIMDE_FLOAT32_C(  -358.33), SIMDE_FLOAT32_C(  -110.66), SIMDE_FLOAT32_C(  -557.13) },
      { SIMDE_FLOAT32_C( -3193.11), SIMDE_FLOAT32_C( -5119.44), SIMDE_FLOAT32_C( -3697.89), SIMDE_FLOAT32_C(   119.55),
        SIMDE_FLOAT32_C(  -672.94), SIMDE_FLOAT32_C(  1096.29), SIMDE_FLOAT32_C(  9265.96), SIMDE_FLOAT32_C(   995.61),
        SIMDE_FLOAT32_C( -7193.56), SIMDE_FLOAT32_C( -2445.38), SIMDE_FLOAT32_C( -4831.56), SIMDE_FLOAT32_C(   394.34),
        SIMDE_FLOAT32_C(  -485.51), SIMDE_FLOAT32_C( -5841.52), SIMDE_FLOAT32_C(   298.16), SIMDE_FLOAT32_C(  4474.47) } },
    { { SIMDE_FLOAT32_C(   -58.12), SIMDE_FLOAT32_C(    37.30), SIMDE_FLOAT32_C(    53.98), SIMDE_FLOAT32_C(    27.58),
        SIMDE_FLOAT32_C(    94.39), SIMDE_FLOAT32_C(    60.12), SIMDE_FLOAT32_C(    46.15), SIMDE_FLOAT32_C(    80.07),
        SIMDE_FLOAT32_C(    45.83), SIMDE_FLOAT32_C(    85.85), SIMDE_FLOAT32_C(   -28.84), SIMDE_FLOAT32_C(   -23.13),
        SIMDE_FLOAT32_C(   -72.79), SIMDE_FLOAT32_C(     0.06), SIMDE_FLOAT32_C(   -69.04), SIMDE_FLOAT32_C(    14.50) },
      { SIMDE_FLOAT32_C(     8.21), SIMDE_FLOAT32_C(    70.39), SIMDE_FLOAT32_C(    43.67), SIMDE_FLOAT32_C(   -99.21),
        SIMDE_FLOAT32_C(    13.16), SIMDE_FLOAT32_C(   -37.08), SIMDE_FLOAT32_C(    19.57), SIMDE_FLOAT32_C(    26.13),
        SIMDE_FLOAT32_C(    73.88), SIMDE_FLOAT32_C(   -10.51), SIMDE_FLOAT32_C(   -77.98), SIMDE_FLOAT32_C(   -21.67),
        SIMDE_FLOAT32_C(    53.66), SIMDE_FLOAT32_C(    10.95), SIMDE_FLOAT32_C(    22.62), SIMDE_FLOAT32_C(    95.54) },
      { SIMDE_FLOAT32_C(  -517.50), SIMDE_FLOAT32_C(  -233.95), SIMDE_FLOAT32_C(   231.18), SIMDE_FLOAT32_C(  -573.59),
        SIMDE_FLOAT32_C(  -632.76), SIMDE_FLOAT32_C(  -307.34), SIMDE_FLOAT32_C(  -772.91), SIMDE_FLOAT32_C(   825.52),
        SIMDE_FLOAT32_C(  -448.81), SIMDE_FLOAT32_C(   -61.32), SIMDE_FLOAT32_C(  -405.74), SIMDE_FLOAT32_C(  -176.75),
        SIMDE_FLOAT32_C(   939.26), SIMDE_FLOAT32_C(   -96.15), SIMDE_FLOAT32_C(   968.29), SIMDE_FLOAT32_C(    21.39) },
      { SIMDE_FLOAT32_C(  -994.67), SIMDE_FLOAT32_C(  2391.60), SIMDE_FLOAT32_C(  2588.49), SIMDE_FLOAT32_C( -3309.80),
        SIMDE_FLOAT32_C(   609.41), SIMDE_FLOAT32_C( -2536.59), SIMDE_FLOAT32_C(   130.25), SIMDE_FLOAT32_C(  2917.75),
        SIMDE_FLOAT32_C(  2937.11), SIMDE_FLOAT32_C(  -963.60), SIMDE_FLOAT32_C(  1843.20), SIMDE_FLOAT32_C(   324.48),
        SIMDE_FLOAT32_C( -2966.65), SIMDE_FLOAT32_C(   -95.49), SIMDE_FLOAT32_C(  -593.39), SIMDE_FLOAT32_C(  1406.72) } },
    { { SIMDE_FLOAT32_C(   -39.23), SIMDE_FLOAT32_C(    40.50), SIMDE_FLOAT32_C(     2.93), SIMDE_FLOAT32_C(    73.94),
        SIMDE_FLOAT32_C(   -96.58), SIMDE_FLOAT32_C(   -77.49), SIMDE_FLOAT32_C(     0.06), SIMDE_FLOAT32_C(    77.30),
        SIMDE_FLOAT32_C(    12.00), SIMDE_FLOAT32_C(    22.08), SIMDE_FLOAT32_C(   -44.36), SIMDE_FLOAT32_C(   -34.35),
        SIMDE_FLOAT32_C(   -66.97), SIMDE_FLOAT32_C(    78.26), SIMDE_FLOAT32_C(   -38.81), SIMDE_FLOAT32_C(   -18.72) },
      { SIMDE_FLOAT32_C(   -45.14), SIMDE_FLOAT32_C(    84.31), SIMDE_FLOAT32_C(    23.92), SIMDE_FLOAT32_C(    -8.41),
        SIMDE_FLOAT32_C(   -46.42), SIMDE_FLOAT32_C(    46.63), SIMDE_FLOAT32_C(   -25.86), SIMDE_FLOAT32_C(     8.70),
        SIMDE_FLOAT32_C(   -59.50), SIMDE_FLOAT32_C(    33.56), SIMDE_FLOAT32_C(    91.02), SIMDE_FLOAT32_C(   -65.58),
        SIMDE_FLOAT32_C(   -76.05), SIMDE_FLOAT32_C(    87.85), SIMDE_FLOAT32_C(    36.56), SIMDE_FLOAT32_C(   -15.28) },
      { SIMDE_FLOAT32_C(   283.51), SIMDE_FLOAT32_C(  -605.06), SIMDE_FLOAT32_C(  -413.40), SIMDE_FLOAT32_C(   317.70),
        SIMDE_FLOAT32_C(  -379.99), SIMDE_FLOAT32_C(   587.24), SIMDE_FLOAT32_C(    90.72), SIMDE_FLOAT32_C(   739.98),
        SIMDE_FLOAT32_C(  -191.96), SIMDE_FLOAT32_C(   647.07), SIMDE_FLOAT32_C(  -603.47), SIMDE_FLOAT32_C(   138.34),
        SIMDE_FLOAT32_C(   429.62), SIMDE_FLOAT32_C(     8.46), SIMDE_FLOAT32_C(   951.13), SIMDE_FLOAT32_C(   978.23) },
      { SIMDE_FLOAT32_C(  2054.35), SIMDE_FLOAT32_C(  2809.49), SIMDE_FLOAT32_C(  -343.31), SIMDE_FLOAT32_C(  -304.14),
        SIMDE_FLOAT32_C(  4103.25), SIMDE_FLOAT32_C( -3026.12), SIMDE_FLOAT32_C(    89.17), SIMDE_FLOAT32_C(  1412.49),
        SIMDE_FLOAT32_C(  -905.96), SIMDE_FLOAT32_C(  1388.07), SIMDE_FLOAT32_C( -4641.12), SIMDE_FLOAT32_C(  2391.01),
        SIMDE_FLOAT32_C(  5522.69), SIMDE_FLOAT32_C(  6883.60), SIMDE_FLOAT32_C(  -467.76), SIMDE_FLOAT32_C(  1264.27) } },
    { { SIMDE_FLOAT32_C(   -14.84), SIMDE_FLOAT32_C(    19.03), SIMDE_FLOAT32_C(   -10.59), SIMDE_FLOAT32_C(    38.74),
        SIMDE_FLOAT32_C(   -34.34), SIMDE_FLOAT32_C(    63.55), SIMDE_FLOAT32_C(   -52.57), SIMDE_FLOAT32_C(     6.16),
        SIMDE_FLOAT32_C(    -2.89), SIMDE_FLOAT32_C(   -61.55), SIMDE_FLOAT32_C(    40.58), SIMDE_FLOAT32_C(    21.06),
        SIMDE_FLOAT32_C(   -73.69), SIMDE_FLOAT32_C(   -22.85), SIMDE_FLOAT32_C(   -94.22), SIMDE_FLOAT32_C(    54.66) },
      { SIMDE_FLOAT32_C(    16.64), SIMDE_FLOAT32_C(   -35.56), SIMDE_FLOAT32_C(   -13.57), SIMDE_FLOAT32_C(    78.64),
        SIMDE_FLOAT32_C(   -76.84), SIMDE_FLOAT32_C(    95.50), SIMDE_FLOAT32_C(    52.64), SIMDE_FLOAT32_C(     3.97),
        SIMDE_FLOAT32_C(    60.20), SIMDE_FLOAT32_C(    92.29), SIMDE_FLOAT32_C(   -82.20), SIMDE_FLOAT32_C(     3.17),
        SIMDE_FLOAT32_C(    -6.86), SIMDE_FLOAT32_C(   -87.09), SIMDE_FLOAT32_C(     0.99), SIMDE_FLOAT32_C(    78.29) },
      { SIMDE_FLOAT32_C(   319.47), SIMDE_FLOAT32_C(   903.98), SIMDE_FLOAT32_C(   170.30), SIMDE_FLOAT32_C(   976.10),
        SIMDE_FLOAT32_C(   539.44), SIMDE_FLOAT32_C(   644.62), SIMDE_FLOAT32_C(    37.70), SIMDE_FLOAT32_C(  -489.48),
        SIMDE_FLOAT32_C(  -970.84), SIMDE_FLOAT32_C(  -556.47), SIMDE_FLOAT32_C(   721.09), SIMDE_FLOAT32_C(  -707.78),
        SIMDE_FLOAT32_C(   214.99), SIMDE_FLOAT32_C(   778.88), SIMDE_FLOAT32_C(   838.78), SIMDE_FLOAT32_C(  -618.61) },
      { SIMDE_FLOAT32_C(    72.53), SIMDE_FLOAT32_C(   227.27), SIMDE_FLOAT32_C(   314.01), SIMDE_FLOAT32_C(  4022.61),
        SIMDE_FLOAT32_C(  3178.13), SIMDE_FLOAT32_C(  6713.65), SIMDE_FLOAT32_C( -2729.58), SIMDE_FLOAT32_C(  -465.02),
        SIMDE_FLOAT32_C( -1144.82), SIMDE_FLOAT32_C( -6236.92), SIMDE_FLOAT32_C( -2614.59), SIMDE_FLOAT32_C(  -641.02),
        SIMDE_FLOAT32_C(   720.50), SIMDE_FLOAT32_C(  2768.89), SIMDE_FLOAT32_C(   745.50), SIMDE_FLOAT32_C(  3660.72) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512 a = simde_mm512_loadu_ps(test_vec[i].a);
    simde__m512 b = simde_mm512_loadu_ps(test_vec[i].b);
    simde__m512 c = simde_mm512_loadu_ps(test_vec[i].c);
    simde__m512 r = simde_mm512_fmadd_ps(a, b, c);
    simde_test_x86_assert_equal_f32x16(r, simde_mm512_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm512_mask_fmadd_ps (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float32 a[16];
    const simde__mmask8 k;
    const simde_float32 b[16];
    const simde_float32 c[16];
    const simde_float32 r[16];
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(   438.16), SIMDE_FLOAT32_C(   227.24), SIMDE_FLOAT32_C(   799.36), SIMDE_FLOAT32_C(   426.66),
        SIMDE_FLOAT32_C(   500.62), SIMDE_FLOAT32_C(   608.47), SIMDE_FLOAT32_C(  -161.69), SIMDE_FLOAT32_C(   411.36),
        SIMDE_FLOAT32_C(  -860.72), SIMDE_FLOAT32_C(   -33.50), SIMDE_FLOAT32_C(    31.57), SIMDE_FLOAT32_C(  -730.88),
        SIMDE_FLOAT32_C(  -717.60), SIMDE_FLOAT32_C(   156.67), SIMDE_FLOAT32_C(   890.78), SIMDE_FLOAT32_C(   110.30) },
      UINT8_C(213),
      { SIMDE_FLOAT32_C(   -29.44), SIMDE_FLOAT32_C(    -9.88), SIMDE_FLOAT32_C(   -43.91), SIMDE_FLOAT32_C(    34.52),
        SIMDE_FLOAT32_C(   -56.88), SIMDE_FLOAT32_C(     0.08), SIMDE_FLOAT32_C(   -37.10), SIMDE_FLOAT32_C(    14.16),
        SIMDE_FLOAT32_C(   -20.56), SIMDE_FLOAT32_C(    -7.94), SIMDE_FLOAT32_C(    10.90), SIMDE_FLOAT32_C(    73.59),
        SIMDE_FLOAT32_C(   -31.00), SIMDE_FLOAT32_C(   -73.00), SIMDE_FLOAT32_C(    17.41), SIMDE_FLOAT32_C(    91.72) },
      { SIMDE_FLOAT32_C(  -930.67), SIMDE_FLOAT32_C(  -399.24), SIMDE_FLOAT32_C(   417.85), SIMDE_FLOAT32_C(   677.80),
        SIMDE_FLOAT32_C(   439.07), SIMDE_FLOAT32_C(  -170.79), SIMDE_FLOAT32_C(   817.08), SIMDE_FLOAT32_C(  -594.43),
        SIMDE_FLOAT32_C(   860.77), SIMDE_FLOAT32_C(  -913.80), SIMDE_FLOAT32_C(  -312.03), SIMDE_FLOAT32_C(    17.44),
        SIMDE_FLOAT32_C(   976.98), SIMDE_FLOAT32_C(   798.27), SIMDE_FLOAT32_C(    96.05), SIMDE_FLOAT32_C(  -317.46) },
      { SIMDE_FLOAT32_C(-13830.10), SIMDE_FLOAT32_C(   227.24), SIMDE_FLOAT32_C(-34682.05), SIMDE_FLOAT32_C(   426.66),
        SIMDE_FLOAT32_C(-28036.20), SIMDE_FLOAT32_C(   608.47), SIMDE_FLOAT32_C(  6815.78), SIMDE_FLOAT32_C(  5230.43),
        SIMDE_FLOAT32_C(  -860.72), SIMDE_FLOAT32_C(   -33.50), SIMDE_FLOAT32_C(    31.57), SIMDE_FLOAT32_C(  -730.88),
        SIMDE_FLOAT32_C(  -717.60), SIMDE_FLOAT32_C(   156.67), SIMDE_FLOAT32_C(   890.78), SIMDE_FLOAT32_C(   110.30) } },
    { { SIMDE_FLOAT32_C(  -300.53), SIMDE_FLOAT32_C(   656.99), SIMDE_FLOAT32_C(  -972.24), SIMDE_FLOAT32_C(   130.63),
        SIMDE_FLOAT32_C(  -342.25), SIMDE_FLOAT32_C(  -343.28), SIMDE_FLOAT32_C(  -727.81), SIMDE_FLOAT32_C(   452.11),
        SIMDE_FLOAT32_C(   577.27), SIMDE_FLOAT32_C(   381.21), SIMDE_FLOAT32_C(   188.06), SIMDE_FLOAT32_C(  -732.74),
        SIMDE_FLOAT32_C(   651.17), SIMDE_FLOAT32_C(  -637.83), SIMDE_FLOAT32_C(  -815.51), SIMDE_FLOAT32_C(   720.50) },
      UINT8_C(204),
      { SIMDE_FLOAT32_C(    60.23), SIMDE_FLOAT32_C(    39.83), SIMDE_FLOAT32_C(   -59.80), SIMDE_FLOAT32_C(   -56.85),
        SIMDE_FLOAT32_C(    21.54), SIMDE_FLOAT32_C(   -19.24), SIMDE_FLOAT32_C(   -70.77), SIMDE_FLOAT32_C(    30.16),
        SIMDE_FLOAT32_C(    49.55), SIMDE_FLOAT32_C(    30.98), SIMDE_FLOAT32_C(    27.86), SIMDE_FLOAT32_C(    29.38),
        SIMDE_FLOAT32_C(   -59.42), SIMDE_FLOAT32_C(    96.11), SIMDE_FLOAT32_C(    99.33), SIMDE_FLOAT32_C(   -93.72) },
      { SIMDE_FLOAT32_C(   988.87), SIMDE_FLOAT32_C(   123.92), SIMDE_FLOAT32_C(  -279.45), SIMDE_FLOAT32_C(  -354.42),
        SIMDE_FLOAT32_C(   396.10), SIMDE_FLOAT32_C(  -827.34), SIMDE_FLOAT32_C(  -777.14), SIMDE_FLOAT32_C(  -222.69),
        SIMDE_FLOAT32_C(   360.72), SIMDE_FLOAT32_C(  -509.88), SIMDE_FLOAT32_C(  -571.52), SIMDE_FLOAT32_C(   722.89),
        SIMDE_FLOAT32_C(  -325.39), SIMDE_FLOAT32_C(  -851.01), SIMDE_FLOAT32_C(  -314.17), SIMDE_FLOAT32_C(  -723.05) },
      { SIMDE_FLOAT32_C(  -300.53), SIMDE_FLOAT32_C(   656.99), SIMDE_FLOAT32_C( 57860.50), SIMDE_FLOAT32_C( -7780.74),
        SIMDE_FLOAT32_C(  -342.25), SIMDE_FLOAT32_C(  -343.28), SIMDE_FLOAT32_C( 50729.97), SIMDE_FLOAT32_C( 13412.95),
        SIMDE_FLOAT32_C(   577.27), SIMDE_FLOAT32_C(   381.21), SIMDE_FLOAT32_C(   188.06), SIMDE_FLOAT32_C(  -732.74),
        SIMDE_FLOAT32_C(   651.17), SIMDE_FLOAT32_C(  -637.83), SIMDE_FLOAT32_C(  -815.51), SIMDE_FLOAT32_C(   720.50) } },
    { { SIMDE_FLOAT32_C(   547.29), SIMDE_FLOAT32_C(    87.84), SIMDE_FLOAT32_C(  -291.51), SIMDE_FLOAT32_C(  -237.32),
        SIMDE_FLOAT32_C(   895.42), SIMDE_FLOAT32_C(     0.81), SIMDE_FLOAT32_C(  -935.74), SIMDE_FLOAT32_C(   390.97),
        SIMDE_FLOAT32_C(  -689.43), SIMDE_FLOAT32_C(   342.83), SIMDE_FLOAT32_C(  -315.22), SIMDE_FLOAT32_C(  -283.62),
        SIMDE_FLOAT32_C(   303.94), SIMDE_FLOAT32_C(  -321.93), SIMDE_FLOAT32_C(  -220.83), SIMDE_FLOAT32_C(   292.81) },
      UINT8_C( 14),
      { SIMDE_FLOAT32_C(    49.97), SIMDE_FLOAT32_C(    93.84), SIMDE_FLOAT32_C(    19.81), SIMDE_FLOAT32_C(    67.24),
        SIMDE_FLOAT32_C(   -83.88), SIMDE_FLOAT32_C(    97.54), SIMDE_FLOAT32_C(     3.31), SIMDE_FLOAT32_C(   -34.86),
        SIMDE_FLOAT32_C(   -59.61), SIMDE_FLOAT32_C(   -24.40), SIMDE_FLOAT32_C(    32.60), SIMDE_FLOAT32_C(   -44.71),
        SIMDE_FLOAT32_C(    44.18), SIMDE_FLOAT32_C(    60.29), SIMDE_FLOAT32_C(   -89.98), SIMDE_FLOAT32_C(   -47.03) },
      { SIMDE_FLOAT32_C(  -688.58), SIMDE_FLOAT32_C(  -137.17), SIMDE_FLOAT32_C(  -574.91), SIMDE_FLOAT32_C(   312.23),
        SIMDE_FLOAT32_C(   -72.90), SIMDE_FLOAT32_C(   816.06), SIMDE_FLOAT32_C(   622.80), SIMDE_FLOAT32_C(  -730.07),
        SIMDE_FLOAT32_C(  -499.16), SIMDE_FLOAT32_C(  -660.83), SIMDE_FLOAT32_C(   573.87), SIMDE_FLOAT32_C(   178.90),
        SIMDE_FLOAT32_C(   118.35), SIMDE_FLOAT32_C(  -133.32), SIMDE_FLOAT32_C(   -19.11), SIMDE_FLOAT32_C(  -381.93) },
      { SIMDE_FLOAT32_C(   547.29), SIMDE_FLOAT32_C(  8105.73), SIMDE_FLOAT32_C( -6349.72), SIMDE_FLOAT32_C(-15645.17),
        SIMDE_FLOAT32_C(   895.42), SIMDE_FLOAT32_C(     0.81), SIMDE_FLOAT32_C(  -935.74), SIMDE_FLOAT32_C(   390.97),
        SIMDE_FLOAT32_C(  -689.43), SIMDE_FLOAT32_C(   342.83), SIMDE_FLOAT32_C(  -315.22), SIMDE_FLOAT32_C(  -283.62),
        SIMDE_FLOAT32_C(   303.94), SIMDE_FLOAT32_C(  -321.93), SIMDE_FLOAT32_C(  -220.83), SIMDE_FLOAT32_C(   292.81) } },
    { { SIMDE_FLOAT32_C(  -194.93), SIMDE_FLOAT32_C(  -821.03), SIMDE_FLOAT32_C(  -709.55), SIMDE_FLOAT32_C(   -33.68),
        SIMDE_FLOAT32_C(  -845.63), SIMDE_FLOAT32_C(   323.55), SIMDE_FLOAT32_C(   617.68), SIMDE_FLOAT32_C(  -441.75),
        SIMDE_FLOAT32_C(  -920.45), SIMDE_FLOAT32_C(   -56.34), SIMDE_FLOAT32_C(   111.11), SIMDE_FLOAT32_C(   521.37),
        SIMDE_FLOAT32_C(  -453.42), SIMDE_FLOAT32_C(   211.27), SIMDE_FLOAT32_C(  -948.96), SIMDE_FLOAT32_C(  -142.01) },
      UINT8_C(248),
      { SIMDE_FLOAT32_C(   -52.39), SIMDE_FLOAT32_C(   -82.98), SIMDE_FLOAT32_C(     0.12), SIMDE_FLOAT32_C(   -70.78),
        SIMDE_FLOAT32_C(    79.30), SIMDE_FLOAT32_C(    27.11), SIMDE_FLOAT32_C(   -20.70), SIMDE_FLOAT32_C(   -86.78),
        SIMDE_FLOAT32_C(   -15.50), SIMDE_FLOAT32_C(    97.19), SIMDE_FLOAT32_C(    25.05), SIMDE_FLOAT32_C(    71.17),
        SIMDE_FLOAT32_C(    -4.72), SIMDE_FLOAT32_C(    86.86), SIMDE_FLOAT32_C(   -48.33), SIMDE_FLOAT32_C(    13.18) },
      { SIMDE_FLOAT32_C(  -840.95), SIMDE_FLOAT32_C(   483.06), SIMDE_FLOAT32_C(   286.15), SIMDE_FLOAT32_C(   482.60),
        SIMDE_FLOAT32_C(   100.74), SIMDE_FLOAT32_C(   844.40), SIMDE_FLOAT32_C(   562.15), SIMDE_FLOAT32_C(  -955.60),
        SIMDE_FLOAT32_C(   -44.49), SIMDE_FLOAT32_C(    83.52), SIMDE_FLOAT32_C(  -409.02), SIMDE_FLOAT32_C(  -833.21),
        SIMDE_FLOAT32_C(   134.56), SIMDE_FLOAT32_C(   448.97), SIMDE_FLOAT32_C(  -759.11), SIMDE_FLOAT32_C(   610.69) },
      { SIMDE_FLOAT32_C(  -194.93), SIMDE_FLOAT32_C(  -821.03), SIMDE_FLOAT32_C(  -709.55), SIMDE_FLOAT32_C(  2866.47),
        SIMDE_FLOAT32_C(-66957.72), SIMDE_FLOAT32_C(  9615.84), SIMDE_FLOAT32_C(-12223.83), SIMDE_FLOAT32_C( 37379.46),
        SIMDE_FLOAT32_C(  -920.45), SIMDE_FLOAT32_C(   -56.34), SIMDE_FLOAT32_C(   111.11), SIMDE_FLOAT32_C(   521.37),
        SIMDE_FLOAT32_C(  -453.42), SIMDE_FLOAT32_C(   211.27), SIMDE_FLOAT32_C(  -948.96), SIMDE_FLOAT32_C(  -142.01) } },
    { { SIMDE_FLOAT32_C(   619.19), SIMDE_FLOAT32_C(   242.10), SIMDE_FLOAT32_C(   902.88), SIMDE_FLOAT32_C(   412.21),
        SIMDE_FLOAT32_C(  -486.77), SIMDE_FLOAT32_C(  -304.10), SIMDE_FLOAT32_C(   544.40), SIMDE_FLOAT32_C(   358.23),
        SIMDE_FLOAT32_C(  -332.18), SIMDE_FLOAT32_C(  -205.06), SIMDE_FLOAT32_C(    69.91), SIMDE_FLOAT32_C(   620.63),
        SIMDE_FLOAT32_C(  -336.46), SIMDE_FLOAT32_C(   586.66), SIMDE_FLOAT32_C(  -247.59), SIMDE_FLOAT32_C(  -177.40) },
      UINT8_C(  4),
      { SIMDE_FLOAT32_C(   -96.14), SIMDE_FLOAT32_C(   -69.48), SIMDE_FLOAT32_C(   -82.95), SIMDE_FLOAT32_C(    88.30),
        SIMDE_FLOAT32_C(    86.73), SIMDE_FLOAT32_C(   -78.51), SIMDE_FLOAT32_C(   -16.15), SIMDE_FLOAT32_C(    -4.91),
        SIMDE_FLOAT32_C(   -19.42), SIMDE_FLOAT32_C(     0.53), SIMDE_FLOAT32_C(   -91.46), SIMDE_FLOAT32_C(   -74.52),
        SIMDE_FLOAT32_C(    24.62), SIMDE_FLOAT32_C(    69.61), SIMDE_FLOAT32_C(    87.40), SIMDE_FLOAT32_C(   -51.17) },
      { SIMDE_FLOAT32_C(   598.99), SIMDE_FLOAT32_C(   286.21), SIMDE_FLOAT32_C(     1.49), SIMDE_FLOAT32_C(  -705.11),
        SIMDE_FLOAT32_C(  -169.39), SIMDE_FLOAT32_C(  -640.28), SIMDE_FLOAT32_C(   -37.29), SIMDE_FLOAT32_C(   625.55),
        SIMDE_FLOAT32_C(   429.63), SIMDE_FLOAT32_C(  -416.66), SIMDE_FLOAT32_C(  -710.90), SIMDE_FLOAT32_C(    16.29),
        SIMDE_FLOAT32_C(   335.76), SIMDE_FLOAT32_C(   111.69), SIMDE_FLOAT32_C(  -913.99), SIMDE_FLOAT32_C(   374.32) },
      { SIMDE_FLOAT32_C(   619.19), SIMDE_FLOAT32_C(   242.10), SIMDE_FLOAT32_C(-74892.41), SIMDE_FLOAT32_C(   412.21),
        SIMDE_FLOAT32_C(  -486.77), SIMDE_FLOAT32_C(  -304.10), SIMDE_FLOAT32_C(   544.40), SIMDE_FLOAT32_C(   358.23),
        SIMDE_FLOAT32_C(  -332.18), SIMDE_FLOAT32_C(  -205.06), SIMDE_FLOAT32_C(    69.91), SIMDE_FLOAT32_C(   620.63),
        SIMDE_FLOAT32_C(  -336.46), SIMDE_FLOAT32_C(   586.66), SIMDE_FLOAT32_C(  -247.59), SIMDE_FLOAT32_C(  -177.40) } },
    { { SIMDE_FLOAT32_C(   416.89), SIMDE_FLOAT32_C(  -743.53), SIMDE_FLOAT32_C(   257.29), SIMDE_FLOAT32_C(   284.23),
        SIMDE_FLOAT32_C(  -528.66), SIMDE_FLOAT32_C(  -904.23), SIMDE_FLOAT32_C(  -764.90), SIMDE_FLOAT32_C(   277.18),
        SIMDE_FLOAT32_C(   101.04), SIMDE_FLOAT32_C(  -679.48), SIMDE_FLOAT32_C(   531.99), SIMDE_FLOAT32_C(  -652.80),
        SIMDE_FLOAT32_C(  -983.36), SIMDE_FLOAT32_C(   405.99), SIMDE_FLOAT32_C(  -164.55), SIMDE_FLOAT32_C(   615.63) },
      UINT8_C(234),
      { SIMDE_FLOAT32_C(    83.69), SIMDE_FLOAT32_C(    91.05), SIMDE_FLOAT32_C(    52.28), SIMDE_FLOAT32_C(   -80.33),
        SIMDE_FLOAT32_C(   -12.68), SIMDE_FLOAT32_C(    14.84), SIMDE_FLOAT32_C(    62.63), SIMDE_FLOAT32_C(    45.66),
        SIMDE_FLOAT32_C(    43.75), SIMDE_FLOAT32_C(   -35.74), SIMDE_FLOAT32_C(   -20.77), SIMDE_FLOAT32_C(   -45.08),
        SIMDE_FLOAT32_C(   -27.14), SIMDE_FLOAT32_C(   -83.33), SIMDE_FLOAT32_C(    96.61), SIMDE_FLOAT32_C(    -1.49) },
      { SIMDE_FLOAT32_C(   423.95), SIMDE_FLOAT32_C(   250.29), SIMDE_FLOAT32_C(   456.40), SIMDE_FLOAT32_C(   519.71),
        SIMDE_FLOAT32_C(   485.39), SIMDE_FLOAT32_C(  -266.42), SIMDE_FLOAT32_C(  -379.25), SIMDE_FLOAT32_C(   805.92),
        SIMDE_FLOAT32_C(  -734.43), SIMDE_FLOAT32_C(   -32.05), SIMDE_FLOAT32_C(   822.56), SIMDE_FLOAT32_C(   671.56),
        SIMDE_FLOAT32_C(   803.40), SIMDE_FLOAT32_C(   438.19), SIMDE_FLOAT32_C(  -636.23), SIMDE_FLOAT32_C(   640.34) },
      { SIMDE_FLOAT32_C(   416.89), SIMDE_FLOAT32_C(-67448.12), SIMDE_FLOAT32_C(   257.29), SIMDE_FLOAT32_C(-22312.49),
        SIMDE_FLOAT32_C(  -528.66), SIMDE_FLOAT32_C(-13685.19), SIMDE_FLOAT32_C(-48284.94), SIMDE_FLOAT32_C( 13461.96),
        SIMDE_FLOAT32_C(   101.04), SIMDE_FLOAT32_C(  -679.48), SIMDE_FLOAT32_C(   531.99), SIMDE_FLOAT32_C(  -652.80),
        SIMDE_FLOAT32_C(  -983.36), SIMDE_FLOAT32_C(   405.99), SIMDE_FLOAT32_C(  -164.55), SIMDE_FLOAT32_C(   615.63) } },
    { { SIMDE_FLOAT32_C(   348.71), SIMDE_FLOAT32_C(   886.59), SIMDE_FLOAT32_C(   837.01), SIMDE_FLOAT32_C(  -778.06),
        SIMDE_FLOAT32_C(    34.96), SIMDE_FLOAT32_C(   463.30), SIMDE_FLOAT32_C(   678.52), SIMDE_FLOAT32_C(  -527.56),
        SIMDE_FLOAT32_C(  -894.11), SIMDE_FLOAT32_C(  -529.15), SIMDE_FLOAT32_C(    21.61), SIMDE_FLOAT32_C(  -165.52),
        SIMDE_FLOAT32_C(  -362.49), SIMDE_FLOAT32_C(   -12.33), SIMDE_FLOAT32_C(   819.55), SIMDE_FLOAT32_C(  -938.55) },
      UINT8_C(233),
      { SIMDE_FLOAT32_C(    27.60), SIMDE_FLOAT32_C(    58.12), SIMDE_FLOAT32_C(    72.34), SIMDE_FLOAT32_C(   -99.05),
        SIMDE_FLOAT32_C(   -79.81), SIMDE_FLOAT32_C(    52.93), SIMDE_FLOAT32_C(   -72.49), SIMDE_FLOAT32_C(    16.99),
        SIMDE_FLOAT32_C(    35.18), SIMDE_FLOAT32_C(    94.67), SIMDE_FLOAT32_C(    -2.67), SIMDE_FLOAT32_C(   -21.00),
        SIMDE_FLOAT32_C(   -68.96), SIMDE_FLOAT32_C(   -38.64), SIMDE_FLOAT32_C(   -86.13), SIMDE_FLOAT32_C(   -80.30) },
      { SIMDE_FLOAT32_C(  -549.39), SIMDE_FLOAT32_C(  -639.33), SIMDE_FLOAT32_C(   231.98), SIMDE_FLOAT32_C(   913.92),
        SIMDE_FLOAT32_C(  -960.81), SIMDE_FLOAT32_C(   704.41), SIMDE_FLOAT32_C(  -980.19), SIMDE_FLOAT32_C(  -489.96),
        SIMDE_FLOAT32_C(  -273.98), SIMDE_FLOAT32_C(  -145.71), SIMDE_FLOAT32_C(   147.55), SIMDE_FLOAT32_C(   713.69),
        SIMDE_FLOAT32_C(  -326.16), SIMDE_FLOAT32_C(   209.00), SIMDE_FLOAT32_C(   951.65), SIMDE_FLOAT32_C(   949.79) },
      { SIMDE_FLOAT32_C(  9075.01), SIMDE_FLOAT32_C(   886.59), SIMDE_FLOAT32_C(   837.01), SIMDE_FLOAT32_C( 77980.77),
        SIMDE_FLOAT32_C(    34.96), SIMDE_FLOAT32_C( 25226.88), SIMDE_FLOAT32_C(-50166.11), SIMDE_FLOAT32_C( -9453.20),
        SIMDE_FLOAT32_C(  -894.11), SIMDE_FLOAT32_C(  -529.15), SIMDE_FLOAT32_C(    21.61), SIMDE_FLOAT32_C(  -165.52),
        SIMDE_FLOAT32_C(  -362.49), SIMDE_FLOAT32_C(   -12.33), SIMDE_FLOAT32_C(   819.55), SIMDE_FLOAT32_C(  -938.55) } },
    { { SIMDE_FLOAT32_C(  -209.83), SIMDE_FLOAT32_C(   675.00), SIMDE_FLOAT32_C(   959.32), SIMDE_FLOAT32_C(    -7.91),
        SIMDE_FLOAT32_C(   204.27), SIMDE_FLOAT32_C(  -765.58), SIMDE_FLOAT32_C(  -838.05), SIMDE_FLOAT32_C(  -443.90),
        SIMDE_FLOAT32_C(  -818.92), SIMDE_FLOAT32_C(   135.21), SIMDE_FLOAT32_C(   346.12), SIMDE_FLOAT32_C(  -508.49),
        SIMDE_FLOAT32_C(   748.82), SIMDE_FLOAT32_C(   484.85), SIMDE_FLOAT32_C(  -311.47), SIMDE_FLOAT32_C(  -800.57) },
      UINT8_C( 77),
      { SIMDE_FLOAT32_C(    92.05), SIMDE_FLOAT32_C(   -88.67), SIMDE_FLOAT32_C(    88.47), SIMDE_FLOAT32_C(    62.49),
        SIMDE_FLOAT32_C(   -86.68), SIMDE_FLOAT32_C(   -60.52), SIMDE_FLOAT32_C(   -64.91), SIMDE_FLOAT32_C(    -1.26),
        SIMDE_FLOAT32_C(    54.23), SIMDE_FLOAT32_C(   -93.54), SIMDE_FLOAT32_C(    66.13), SIMDE_FLOAT32_C(   -24.87),
        SIMDE_FLOAT32_C(   -98.37), SIMDE_FLOAT32_C(    61.11), SIMDE_FLOAT32_C(    54.15), SIMDE_FLOAT32_C(    69.13) },
      { SIMDE_FLOAT32_C(   570.40), SIMDE_FLOAT32_C(  -466.42), SIMDE_FLOAT32_C(  -104.45), SIMDE_FLOAT32_C(   804.83),
        SIMDE_FLOAT32_C(  -304.47), SIMDE_FLOAT32_C(   451.65), SIMDE_FLOAT32_C(   985.91), SIMDE_FLOAT32_C(   830.74),
        SIMDE_FLOAT32_C(  -202.23), SIMDE_FLOAT32_C(  -522.58), SIMDE_FLOAT32_C(   579.55), SIMDE_FLOAT32_C(  -717.38),
        SIMDE_FLOAT32_C(   165.95), SIMDE_FLOAT32_C(   778.98), SIMDE_FLOAT32_C(  -871.85), SIMDE_FLOAT32_C(    86.45) },
      { SIMDE_FLOAT32_C(-18744.45), SIMDE_FLOAT32_C(   675.00), SIMDE_FLOAT32_C( 84766.59), SIMDE_FLOAT32_C(   310.53),
        SIMDE_FLOAT32_C(   204.27), SIMDE_FLOAT32_C(  -765.58), SIMDE_FLOAT32_C( 55383.74), SIMDE_FLOAT32_C(  -443.90),
        SIMDE_FLOAT32_C(  -818.92), SIMDE_FLOAT32_C(   135.21), SIMDE_FLOAT32_C(   346.12), SIMDE_FLOAT32_C(  -508.49),
        SIMDE_FLOAT32_C(   748.82), SIMDE_FLOAT32_C(   484.85), SIMDE_FLOAT32_C(  -311.47), SIMDE_FLOAT32_C(  -800.57) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512 a = simde_mm512_loadu_ps(test_vec[i].a);
    simde__m512 b = simde_mm512_loadu_ps(test_vec[i].b);
    simde__m512 c = simde_mm512_loadu_ps(test_vec[i].c);
    simde__m512 r = simde_mm512_mask_fmadd_ps(a, test_vec[i].k, b, c);
    simde_test_x86_assert_equal_f32x16(r, simde_mm512_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm512_maskz_fmadd_ps (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask16 k;
    const simde_float32 a[16];
    const simde_float32 b[16];
    const simde_float32 c[16];
    const simde_float32 r[16];
  } test_vec[] = {
    { UINT16_C(14967),
      { SIMDE_FLOAT32_C(    18.11), SIMDE_FLOAT32_C(   -41.35), SIMDE_FLOAT32_C(    32.27), SIMDE_FLOAT32_C(    -1.66),
        SIMDE_FLOAT32_C(   -21.64), SIMDE_FLOAT32_C(    77.20), SIMDE_FLOAT32_C(    35.46), SIMDE_FLOAT32_C(   -90.91),
        SIMDE_FLOAT32_C(   -64.23), SIMDE_FLOAT32_C(   -16.52), SIMDE_FLOAT32_C(    22.87), SIMDE_FLOAT32_C(    30.16),
        SIMDE_FLOAT32_C(    -6.85), SIMDE_FLOAT32_C(    51.65), SIMDE_FLOAT32_C(    -2.38), SIMDE_FLOAT32_C(    -7.27) },
      { SIMDE_FLOAT32_C(   -26.01), SIMDE_FLOAT32_C(    50.75), SIMDE_FLOAT32_C(    26.70), SIMDE_FLOAT32_C(    21.59),
        SIMDE_FLOAT32_C(    39.55), SIMDE_FLOAT32_C(    43.61), SIMDE_FLOAT32_C(   -16.85), SIMDE_FLOAT32_C(    62.00),
        SIMDE_FLOAT32_C(   -10.33), SIMDE_FLOAT32_C(   -32.63), SIMDE_FLOAT32_C(    29.56), SIMDE_FLOAT32_C(     2.54),
        SIMDE_FLOAT32_C(    72.74), SIMDE_FLOAT32_C(   -37.07), SIMDE_FLOAT32_C(   -74.38), SIMDE_FLOAT32_C(    -9.15) },
      { SIMDE_FLOAT32_C(   215.80), SIMDE_FLOAT32_C(   578.85), SIMDE_FLOAT32_C(   892.00), SIMDE_FLOAT32_C(   999.36),
        SIMDE_FLOAT32_C(   350.88), SIMDE_FLOAT32_C(   246.63), SIMDE_FLOAT32_C(  -909.73), SIMDE_FLOAT32_C(   708.55),
        SIMDE_FLOAT32_C(  -918.55), SIMDE_FLOAT32_C(   318.97), SIMDE_FLOAT32_C(    10.19), SIMDE_FLOAT32_C(    12.96),
        SIMDE_FLOAT32_C(  -164.54), SIMDE_FLOAT32_C(   986.37), SIMDE_FLOAT32_C(   940.27), SIMDE_FLOAT32_C(   575.40) },
      { SIMDE_FLOAT32_C(  -255.24), SIMDE_FLOAT32_C( -1519.66), SIMDE_FLOAT32_C(  1753.61), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(  -504.98), SIMDE_FLOAT32_C(  3613.32), SIMDE_FLOAT32_C( -1507.23), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   858.02), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    89.57),
        SIMDE_FLOAT32_C(  -662.81), SIMDE_FLOAT32_C(  -928.30), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } },
    { UINT16_C(49253),
      { SIMDE_FLOAT32_C(   -20.87), SIMDE_FLOAT32_C(   -11.07), SIMDE_FLOAT32_C(   -35.67), SIMDE_FLOAT32_C(    62.27),
        SIMDE_FLOAT32_C(   -49.07), SIMDE_FLOAT32_C(    54.00), SIMDE_FLOAT32_C(   -70.36), SIMDE_FLOAT32_C(    80.49),
        SIMDE_FLOAT32_C(   -43.47), SIMDE_FLOAT32_C(   -97.62), SIMDE_FLOAT32_C(   -56.58), SIMDE_FLOAT32_C(   -17.85),
        SIMDE_FLOAT32_C(    -6.76), SIMDE_FLOAT32_C(    65.00), SIMDE_FLOAT32_C(   -59.96), SIMDE_FLOAT32_C(   -17.56) },
      { SIMDE_FLOAT32_C(    64.94), SIMDE_FLOAT32_C(    75.12), SIMDE_FLOAT32_C(   -92.90), SIMDE_FLOAT32_C(    73.96),
        SIMDE_FLOAT32_C(    45.98), SIMDE_FLOAT32_C(   -84.75), SIMDE_FLOAT32_C(     5.86), SIMDE_FLOAT32_C(   -53.00),
        SIMDE_FLOAT32_C(    16.54), SIMDE_FLOAT32_C(    89.41), SIMDE_FLOAT32_C(   -54.37), SIMDE_FLOAT32_C(    10.57),
        SIMDE_FLOAT32_C(    46.95), SIMDE_FLOAT32_C(    95.02), SIMDE_FLOAT32_C(   -68.71), SIMDE_FLOAT32_C(   -73.93) },
      { SIMDE_FLOAT32_C(  -160.52), SIMDE_FLOAT32_C(   -43.76), SIMDE_FLOAT32_C(   883.44), SIMDE_FLOAT32_C(   348.76),
        SIMDE_FLOAT32_C(  -503.79), SIMDE_FLOAT32_C(  -820.14), SIMDE_FLOAT32_C(   153.62), SIMDE_FLOAT32_C(    61.54),
        SIMDE_FLOAT32_C(  -796.31), SIMDE_FLOAT32_C(   587.81), SIMDE_FLOAT32_C(   883.04), SIMDE_FLOAT32_C(   136.08),
        SIMDE_FLOAT32_C(   237.80), SIMDE_FLOAT32_C(  -716.60), SIMDE_FLOAT32_C(   960.45), SIMDE_FLOAT32_C(  -112.85) },
      { SIMDE_FLOAT32_C( -1515.82), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  4197.18), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C( -5396.64), SIMDE_FLOAT32_C(  -258.69), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  5080.30), SIMDE_FLOAT32_C(  1185.36) } },
    { UINT16_C(33012),
      { SIMDE_FLOAT32_C(   -37.32), SIMDE_FLOAT32_C(    49.44), SIMDE_FLOAT32_C(   -81.61), SIMDE_FLOAT32_C(    68.54),
        SIMDE_FLOAT32_C(    96.44), SIMDE_FLOAT32_C(    34.93), SIMDE_FLOAT32_C(    57.94), SIMDE_FLOAT32_C(   -57.93),
        SIMDE_FLOAT32_C(   -54.50), SIMDE_FLOAT32_C(     4.89), SIMDE_FLOAT32_C(   -62.91), SIMDE_FLOAT32_C(   -23.20),
        SIMDE_FLOAT32_C(    30.96), SIMDE_FLOAT32_C(    21.04), SIMDE_FLOAT32_C(    72.42), SIMDE_FLOAT32_C(    19.30) },
      { SIMDE_FLOAT32_C(   -44.08), SIMDE_FLOAT32_C(   -77.96), SIMDE_FLOAT32_C(    37.29), SIMDE_FLOAT32_C(    71.28),
        SIMDE_FLOAT32_C(    28.20), SIMDE_FLOAT32_C(    57.66), SIMDE_FLOAT32_C(    30.06), SIMDE_FLOAT32_C(    16.50),
        SIMDE_FLOAT32_C(   -28.73), SIMDE_FLOAT32_C(   -46.16), SIMDE_FLOAT32_C(    44.84), SIMDE_FLOAT32_C(   -32.69),
        SIMDE_FLOAT32_C(    42.55), SIMDE_FLOAT32_C(    48.31), SIMDE_FLOAT32_C(   -29.54), SIMDE_FLOAT32_C(   -94.77) },
      { SIMDE_FLOAT32_C(   -22.52), SIMDE_FLOAT32_C(  -111.49), SIMDE_FLOAT32_C(   737.68), SIMDE_FLOAT32_C(   -58.11),
        SIMDE_FLOAT32_C(  -762.15), SIMDE_FLOAT32_C(   317.10), SIMDE_FLOAT32_C(   362.64), SIMDE_FLOAT32_C(  -307.11),
        SIMDE_FLOAT32_C(  -634.03), SIMDE_FLOAT32_C(   733.57), SIMDE_FLOAT32_C(   460.86), SIMDE_FLOAT32_C(   675.57),
        SIMDE_FLOAT32_C(   -56.01), SIMDE_FLOAT32_C(   185.08), SIMDE_FLOAT32_C(  -131.39), SIMDE_FLOAT32_C(   503.16) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C( -2305.56), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(  1957.46), SIMDE_FLOAT32_C(  2331.16), SIMDE_FLOAT32_C(  2104.32), SIMDE_FLOAT32_C( -1262.95),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C( -1325.90) } },
    { UINT16_C(30249),
      { SIMDE_FLOAT32_C(    21.60), SIMDE_FLOAT32_C(   -31.25), SIMDE_FLOAT32_C(    81.81), SIMDE_FLOAT32_C(   -48.34),
        SIMDE_FLOAT32_C(    85.25), SIMDE_FLOAT32_C(   -46.92), SIMDE_FLOAT32_C(     5.49), SIMDE_FLOAT32_C(    30.09),
        SIMDE_FLOAT32_C(    20.39), SIMDE_FLOAT32_C(   -51.95), SIMDE_FLOAT32_C(   -21.61), SIMDE_FLOAT32_C(    90.85),
        SIMDE_FLOAT32_C(   -46.72), SIMDE_FLOAT32_C(    76.14), SIMDE_FLOAT32_C(   -20.30), SIMDE_FLOAT32_C(   -72.95) },
      { SIMDE_FLOAT32_C(   -29.67), SIMDE_FLOAT32_C(     3.48), SIMDE_FLOAT32_C(    58.76), SIMDE_FLOAT32_C(   -93.40),
        SIMDE_FLOAT32_C(    72.77), SIMDE_FLOAT32_C(    95.35), SIMDE_FLOAT32_C(    79.95), SIMDE_FLOAT32_C(    18.86),
        SIMDE_FLOAT32_C(    62.91), SIMDE_FLOAT32_C(   -25.65), SIMDE_FLOAT32_C(   -62.63), SIMDE_FLOAT32_C(   -50.23),
        SIMDE_FLOAT32_C(   -75.33), SIMDE_FLOAT32_C(    77.92), SIMDE_FLOAT32_C(   -26.08), SIMDE_FLOAT32_C(    46.26) },
      { SIMDE_FLOAT32_C(  -533.34), SIMDE_FLOAT32_C(  -442.67), SIMDE_FLOAT32_C(   979.18), SIMDE_FLOAT32_C(  -680.86),
        SIMDE_FLOAT32_C(    88.10), SIMDE_FLOAT32_C(    34.12), SIMDE_FLOAT32_C(   620.02), SIMDE_FLOAT32_C(  -708.00),
        SIMDE_FLOAT32_C(   514.61), SIMDE_FLOAT32_C(  -596.04), SIMDE_FLOAT32_C(  -799.52), SIMDE_FLOAT32_C(  -952.60),
        SIMDE_FLOAT32_C(  -834.61), SIMDE_FLOAT32_C(    -2.53), SIMDE_FLOAT32_C(  -682.13), SIMDE_FLOAT32_C(  -131.30) },
      { SIMDE_FLOAT32_C( -1174.21), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  3834.10),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C( -4439.70), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   736.48), SIMDE_FLOAT32_C(   553.91), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(  2684.81), SIMDE_FLOAT32_C(  5930.30), SIMDE_FLOAT32_C(  -152.71), SIMDE_FLOAT32_C(     0.00) } },
    { UINT16_C(36814),
      { SIMDE_FLOAT32_C(    -6.53), SIMDE_FLOAT32_C(    76.01), SIMDE_FLOAT32_C(    85.90), SIMDE_FLOAT32_C(   -26.58),
        SIMDE_FLOAT32_C(    -5.13), SIMDE_FLOAT32_C(    48.81), SIMDE_FLOAT32_C(    47.77), SIMDE_FLOAT32_C(    32.23),
        SIMDE_FLOAT32_C(    98.58), SIMDE_FLOAT32_C(    72.44), SIMDE_FLOAT32_C(    10.15), SIMDE_FLOAT32_C(   -27.49),
        SIMDE_FLOAT32_C(    18.70), SIMDE_FLOAT32_C(    56.82), SIMDE_FLOAT32_C(    28.24), SIMDE_FLOAT32_C(    16.62) },
      { SIMDE_FLOAT32_C(    88.73), SIMDE_FLOAT32_C(   -62.95), SIMDE_FLOAT32_C(   -79.97), SIMDE_FLOAT32_C(    50.73),
        SIMDE_FLOAT32_C(   -33.75), SIMDE_FLOAT32_C(    71.49), SIMDE_FLOAT32_C(    91.13), SIMDE_FLOAT32_C(   -13.70),
        SIMDE_FLOAT32_C(    76.23), SIMDE_FLOAT32_C(   -92.33), SIMDE_FLOAT32_C(    86.04), SIMDE_FLOAT32_C(   -91.98),
        SIMDE_FLOAT32_C(    -5.46), SIMDE_FLOAT32_C(    89.28), SIMDE_FLOAT32_C(    98.56), SIMDE_FLOAT32_C(    88.01) },
      { SIMDE_FLOAT32_C(   652.81), SIMDE_FLOAT32_C(   844.61), SIMDE_FLOAT32_C(  -385.77), SIMDE_FLOAT32_C(  -398.54),
        SIMDE_FLOAT32_C(   332.71), SIMDE_FLOAT32_C(  -908.07), SIMDE_FLOAT32_C(   923.80), SIMDE_FLOAT32_C(   318.53),
        SIMDE_FLOAT32_C(   816.30), SIMDE_FLOAT32_C(    25.32), SIMDE_FLOAT32_C(  -956.42), SIMDE_FLOAT32_C(     3.30),
        SIMDE_FLOAT32_C(  -406.50), SIMDE_FLOAT32_C(   325.97), SIMDE_FLOAT32_C(  -830.53), SIMDE_FLOAT32_C(  -519.19) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C( -3940.22), SIMDE_FLOAT32_C( -7255.19), SIMDE_FLOAT32_C( -1746.94),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  5277.08), SIMDE_FLOAT32_C(  -123.02),
        SIMDE_FLOAT32_C(  8331.05), SIMDE_FLOAT32_C( -6663.07), SIMDE_FLOAT32_C(   -83.11), SIMDE_FLOAT32_C(  2531.83),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   943.54) } },
    { UINT16_C(21105),
      { SIMDE_FLOAT32_C(    98.81), SIMDE_FLOAT32_C(   -64.11), SIMDE_FLOAT32_C(   -91.53), SIMDE_FLOAT32_C(    89.95),
        SIMDE_FLOAT32_C(    22.19), SIMDE_FLOAT32_C(    84.70), SIMDE_FLOAT32_C(    97.61), SIMDE_FLOAT32_C(     8.23),
        SIMDE_FLOAT32_C(    92.72), SIMDE_FLOAT32_C(    -7.85), SIMDE_FLOAT32_C(    -2.49), SIMDE_FLOAT32_C(    91.28),
        SIMDE_FLOAT32_C(   -19.84), SIMDE_FLOAT32_C(   -37.21), SIMDE_FLOAT32_C(    75.74), SIMDE_FLOAT32_C(    41.58) },
      { SIMDE_FLOAT32_C(    22.93), SIMDE_FLOAT32_C(     9.01), SIMDE_FLOAT32_C(    50.78), SIMDE_FLOAT32_C(    15.31),
        SIMDE_FLOAT32_C(   -59.14), SIMDE_FLOAT32_C(    32.41), SIMDE_FLOAT32_C(   -82.15), SIMDE_FLOAT32_C(   -54.78),
        SIMDE_FLOAT32_C(   -67.27), SIMDE_FLOAT32_C(   -22.80), SIMDE_FLOAT32_C(    77.82), SIMDE_FLOAT32_C(   -50.32),
        SIMDE_FLOAT32_C(    25.28), SIMDE_FLOAT32_C(    47.46), SIMDE_FLOAT32_C(   -13.34), SIMDE_FLOAT32_C(    24.09) },
      { SIMDE_FLOAT32_C(   833.57), SIMDE_FLOAT32_C(   -48.74), SIMDE_FLOAT32_C(   140.37), SIMDE_FLOAT32_C(    55.46),
        SIMDE_FLOAT32_C(  -201.75), SIMDE_FLOAT32_C(   116.51), SIMDE_FLOAT32_C(  -862.21), SIMDE_FLOAT32_C(  -274.59),
        SIMDE_FLOAT32_C(  -961.95), SIMDE_FLOAT32_C(   112.88), SIMDE_FLOAT32_C(  -361.81), SIMDE_FLOAT32_C(  -160.36),
        SIMDE_FLOAT32_C(   740.76), SIMDE_FLOAT32_C(  -604.42), SIMDE_FLOAT32_C(  -744.54), SIMDE_FLOAT32_C(   -29.89) },
      { SIMDE_FLOAT32_C(  3099.28), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C( -1514.07), SIMDE_FLOAT32_C(  2861.64), SIMDE_FLOAT32_C( -8880.87), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   291.86), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(   239.20), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C( -1754.91), SIMDE_FLOAT32_C(     0.00) } },
    { UINT16_C(37070),
      { SIMDE_FLOAT32_C(   -87.67), SIMDE_FLOAT32_C(    89.43), SIMDE_FLOAT32_C(     8.73), SIMDE_FLOAT32_C(   -69.83),
        SIMDE_FLOAT32_C(   -65.35), SIMDE_FLOAT32_C(    41.46), SIMDE_FLOAT32_C(     7.37), SIMDE_FLOAT32_C(   -87.53),
        SIMDE_FLOAT32_C(    91.14), SIMDE_FLOAT32_C(   -67.36), SIMDE_FLOAT32_C(    59.93), SIMDE_FLOAT32_C(   -22.20),
        SIMDE_FLOAT32_C(    56.74), SIMDE_FLOAT32_C(    43.29), SIMDE_FLOAT32_C(    72.93), SIMDE_FLOAT32_C(   -29.23) },
      { SIMDE_FLOAT32_C(   -51.16), SIMDE_FLOAT32_C(   -47.25), SIMDE_FLOAT32_C(    82.42), SIMDE_FLOAT32_C(   -37.38),
        SIMDE_FLOAT32_C(    25.29), SIMDE_FLOAT32_C(    86.23), SIMDE_FLOAT32_C(    73.90), SIMDE_FLOAT32_C(    89.11),
        SIMDE_FLOAT32_C(   -29.81), SIMDE_FLOAT32_C(    47.98), SIMDE_FLOAT32_C(   -71.33), SIMDE_FLOAT32_C(    -4.26),
        SIMDE_FLOAT32_C(   -55.01), SIMDE_FLOAT32_C(    77.24), SIMDE_FLOAT32_C(   -27.94), SIMDE_FLOAT32_C(   -42.68) },
      { SIMDE_FLOAT32_C(   666.71), SIMDE_FLOAT32_C(   807.87), SIMDE_FLOAT32_C(  -125.12), SIMDE_FLOAT32_C(  -986.75),
        SIMDE_FLOAT32_C(   222.48), SIMDE_FLOAT32_C(   948.55), SIMDE_FLOAT32_C(  -862.04), SIMDE_FLOAT32_C(   133.91),
        SIMDE_FLOAT32_C(  -725.01), SIMDE_FLOAT32_C(   737.31), SIMDE_FLOAT32_C(   911.94), SIMDE_FLOAT32_C(   842.35),
        SIMDE_FLOAT32_C(   170.22), SIMDE_FLOAT32_C(   641.22), SIMDE_FLOAT32_C(  -449.91), SIMDE_FLOAT32_C(   658.60) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C( -3417.70), SIMDE_FLOAT32_C(   594.41), SIMDE_FLOAT32_C(  1623.50),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -317.40), SIMDE_FLOAT32_C( -7665.89),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C( -2951.05), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  1906.14) } },
    { UINT16_C( 2209),
      { SIMDE_FLOAT32_C(   -71.52), SIMDE_FLOAT32_C(    42.17), SIMDE_FLOAT32_C(   -76.34), SIMDE_FLOAT32_C(   -97.62),
        SIMDE_FLOAT32_C(    31.28), SIMDE_FLOAT32_C(    -6.14), SIMDE_FLOAT32_C(    50.36), SIMDE_FLOAT32_C(    59.95),
        SIMDE_FLOAT32_C(    89.60), SIMDE_FLOAT32_C(    95.35), SIMDE_FLOAT32_C(    37.19), SIMDE_FLOAT32_C(   -38.34),
        SIMDE_FLOAT32_C(   -47.33), SIMDE_FLOAT32_C(     3.86), SIMDE_FLOAT32_C(   -57.56), SIMDE_FLOAT32_C(    40.16) },
      { SIMDE_FLOAT32_C(     5.19), SIMDE_FLOAT32_C(    64.69), SIMDE_FLOAT32_C(    35.01), SIMDE_FLOAT32_C(    18.99),
        SIMDE_FLOAT32_C(   -21.92), SIMDE_FLOAT32_C(    62.51), SIMDE_FLOAT32_C(    -7.28), SIMDE_FLOAT32_C(   -30.72),
        SIMDE_FLOAT32_C(    46.75), SIMDE_FLOAT32_C(   -90.26), SIMDE_FLOAT32_C(   -66.60), SIMDE_FLOAT32_C(   -98.24),
        SIMDE_FLOAT32_C(    75.60), SIMDE_FLOAT32_C(   -49.72), SIMDE_FLOAT32_C(   -60.81), SIMDE_FLOAT32_C(   -95.93) },
      { SIMDE_FLOAT32_C(   924.44), SIMDE_FLOAT32_C(  -371.47), SIMDE_FLOAT32_C(  -935.45), SIMDE_FLOAT32_C(   237.27),
        SIMDE_FLOAT32_C(   567.09), SIMDE_FLOAT32_C(   568.16), SIMDE_FLOAT32_C(  -163.20), SIMDE_FLOAT32_C(   463.06),
        SIMDE_FLOAT32_C(   521.69), SIMDE_FLOAT32_C(  -791.26), SIMDE_FLOAT32_C(  -920.37), SIMDE_FLOAT32_C(  -951.61),
        SIMDE_FLOAT32_C(   247.38), SIMDE_FLOAT32_C(  -495.92), SIMDE_FLOAT32_C(   449.97), SIMDE_FLOAT32_C(  -700.73) },
      { SIMDE_FLOAT32_C(   553.25), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   184.35), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C( -1378.60),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  2814.91),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512 a = simde_mm512_loadu_ps(test_vec[i].a);
    simde__m512 b = simde_mm512_loadu_ps(test_vec[i].b);
    simde__m512 c = simde_mm512_loadu_ps(test_vec[i].c);
    simde__m512 r = simde_mm512_maskz_fmadd_ps(test_vec[i].k, a, b, c);
    simde_test_x86_assert_equal_f32x16(r, simde_mm512_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm512_fmadd_pd (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float64 a[8];
    const simde_float64 b[8];
    const simde_float64 c[8];
    const simde_float64 r[8];
  } test_vec[] = {
    { { SIMDE_FLOAT64_C(   -15.20), SIMDE_FLOAT64_C(   -92.09), SIMDE_FLOAT64_C(    -5.68), SIMDE_FLOAT64_C(   -60.32),
        SIMDE_FLOAT64_C(    11.66), SIMDE_FLOAT64_C(   -39.24), SIMDE_FLOAT64_C(   -47.98), SIMDE_FLOAT64_C(    -9.95) },
      { SIMDE_FLOAT64_C(   -67.23), SIMDE_FLOAT64_C(   -68.40), SIMDE_FLOAT64_C(   -47.95), SIMDE_FLOAT64_C(    26.22),
        SIMDE_FLOAT64_C(   -82.14), SIMDE_FLOAT64_C(    74.05), SIMDE_FLOAT64_C(   -33.99), SIMDE_FLOAT64_C(    90.24) },
      { SIMDE_FLOAT64_C(   198.57), SIMDE_FLOAT64_C(  -636.25), SIMDE_FLOAT64_C(   470.89), SIMDE_FLOAT64_C(   299.50),
        SIMDE_FLOAT64_C(  -364.82), SIMDE_FLOAT64_C(   440.56), SIMDE_FLOAT64_C(   486.88), SIMDE_FLOAT64_C(   920.40) },
      { SIMDE_FLOAT64_C(  1220.47), SIMDE_FLOAT64_C(  5662.71), SIMDE_FLOAT64_C(   743.25), SIMDE_FLOAT64_C( -1282.09),
        SIMDE_FLOAT64_C( -1322.57), SIMDE_FLOAT64_C( -2465.16), SIMDE_FLOAT64_C(  2117.72), SIMDE_FLOAT64_C(    22.51) } },
    { { SIMDE_FLOAT64_C(   -84.36), SIMDE_FLOAT64_C(   -64.18), SIMDE_FLOAT64_C(   -17.96), SIMDE_FLOAT64_C(   -13.18),
        SIMDE_FLOAT64_C(   -53.67), SIMDE_FLOAT64_C(    26.82), SIMDE_FLOAT64_C(    -6.48), SIMDE_FLOAT64_C(    64.66) },
      { SIMDE_FLOAT64_C(   -48.92), SIMDE_FLOAT64_C(   -80.30), SIMDE_FLOAT64_C(   -78.02), SIMDE_FLOAT64_C(   -69.90),
        SIMDE_FLOAT64_C(    29.55), SIMDE_FLOAT64_C(    51.81), SIMDE_FLOAT64_C(   -58.76), SIMDE_FLOAT64_C(   -43.34) },
      { SIMDE_FLOAT64_C(  -369.54), SIMDE_FLOAT64_C(  -938.59), SIMDE_FLOAT64_C(  -949.35), SIMDE_FLOAT64_C(   294.30),
        SIMDE_FLOAT64_C(   268.45), SIMDE_FLOAT64_C(  -199.17), SIMDE_FLOAT64_C(   580.15), SIMDE_FLOAT64_C(   625.12) },
      { SIMDE_FLOAT64_C(  3757.35), SIMDE_FLOAT64_C(  4215.06), SIMDE_FLOAT64_C(   451.89), SIMDE_FLOAT64_C(  1215.58),
        SIMDE_FLOAT64_C( -1317.50), SIMDE_FLOAT64_C(  1190.37), SIMDE_FLOAT64_C(   960.91), SIMDE_FLOAT64_C( -2177.24) } },
    { { SIMDE_FLOAT64_C(    42.20), SIMDE_FLOAT64_C(    81.03), SIMDE_FLOAT64_C(    93.61), SIMDE_FLOAT64_C(    85.46),
        SIMDE_FLOAT64_C(   -54.78), SIMDE_FLOAT64_C(   -43.11), SIMDE_FLOAT64_C(    68.99), SIMDE_FLOAT64_C(   -72.24) },
      { SIMDE_FLOAT64_C(    31.24), SIMDE_FLOAT64_C(    44.02), SIMDE_FLOAT64_C(    35.86), SIMDE_FLOAT64_C(    36.00),
        SIMDE_FLOAT64_C(    49.85), SIMDE_FLOAT64_C(    71.08), SIMDE_FLOAT64_C(    95.94), SIMDE_FLOAT64_C(   -61.93) },
      { SIMDE_FLOAT64_C(   -70.68), SIMDE_FLOAT64_C(   113.51), SIMDE_FLOAT64_C(  -246.66), SIMDE_FLOAT64_C(   329.53),
        SIMDE_FLOAT64_C(   345.01), SIMDE_FLOAT64_C(   517.47), SIMDE_FLOAT64_C(  -899.02), SIMDE_FLOAT64_C(   163.03) },
      { SIMDE_FLOAT64_C(  1247.65), SIMDE_FLOAT64_C(  3680.45), SIMDE_FLOAT64_C(  3110.19), SIMDE_FLOAT64_C(  3406.09),
        SIMDE_FLOAT64_C( -2385.77), SIMDE_FLOAT64_C( -2546.79), SIMDE_FLOAT64_C(  5719.88), SIMDE_FLOAT64_C(  4636.85) } },
    { { SIMDE_FLOAT64_C(     1.76), SIMDE_FLOAT64_C(   -74.31), SIMDE_FLOAT64_C(    62.56), SIMDE_FLOAT64_C(   -37.50),
        SIMDE_FLOAT64_C(    13.98), SIMDE_FLOAT64_C(   -68.32), SIMDE_FLOAT64_C(   -66.19), SIMDE_FLOAT64_C(    -8.98) },
      { SIMDE_FLOAT64_C(   -46.81), SIMDE_FLOAT64_C(    39.06), SIMDE_FLOAT64_C(   -86.48), SIMDE_FLOAT64_C(   -52.33),
        SIMDE_FLOAT64_C(    78.57), SIMDE_FLOAT64_C(     2.96), SIMDE_FLOAT64_C(   -70.43), SIMDE_FLOAT64_C(    77.87) },
      { SIMDE_FLOAT64_C(  -708.96), SIMDE_FLOAT64_C(   589.10), SIMDE_FLOAT64_C(  -927.53), SIMDE_FLOAT64_C(   949.55),
        SIMDE_FLOAT64_C(  -830.02), SIMDE_FLOAT64_C(  -118.83), SIMDE_FLOAT64_C(   850.20), SIMDE_FLOAT64_C(  -768.46) },
      { SIMDE_FLOAT64_C(  -791.35), SIMDE_FLOAT64_C( -2313.45), SIMDE_FLOAT64_C( -6337.72), SIMDE_FLOAT64_C(  2911.92),
        SIMDE_FLOAT64_C(   268.39), SIMDE_FLOAT64_C(  -321.06), SIMDE_FLOAT64_C(  5511.96), SIMDE_FLOAT64_C( -1467.73) } },
    { { SIMDE_FLOAT64_C(    53.04), SIMDE_FLOAT64_C(    11.01), SIMDE_FLOAT64_C(   -88.73), SIMDE_FLOAT64_C(   -90.01),
        SIMDE_FLOAT64_C(    26.51), SIMDE_FLOAT64_C(    88.06), SIMDE_FLOAT64_C(    55.77), SIMDE_FLOAT64_C(    81.19) },
      { SIMDE_FLOAT64_C(    88.52), SIMDE_FLOAT64_C(    75.58), SIMDE_FLOAT64_C(    -7.51), SIMDE_FLOAT64_C(   -80.56),
        SIMDE_FLOAT64_C(    54.57), SIMDE_FLOAT64_C(   -95.80), SIMDE_FLOAT64_C(   -84.04), SIMDE_FLOAT64_C(    22.23) },
      { SIMDE_FLOAT64_C(  -554.02), SIMDE_FLOAT64_C(   517.62), SIMDE_FLOAT64_C(  -314.36), SIMDE_FLOAT64_C(   517.56),
        SIMDE_FLOAT64_C(   675.22), SIMDE_FLOAT64_C(   462.96), SIMDE_FLOAT64_C(  -994.29), SIMDE_FLOAT64_C(   896.33) },
      { SIMDE_FLOAT64_C(  4141.08), SIMDE_FLOAT64_C(  1349.76), SIMDE_FLOAT64_C(   352.00), SIMDE_FLOAT64_C(  7768.77),
        SIMDE_FLOAT64_C(  2121.87), SIMDE_FLOAT64_C( -7973.19), SIMDE_FLOAT64_C( -5681.20), SIMDE_FLOAT64_C(  2701.18) } },
    { { SIMDE_FLOAT64_C(    62.43), SIMDE_FLOAT64_C(    38.95), SIMDE_FLOAT64_C(    97.62), SIMDE_FLOAT64_C(    23.91),
        SIMDE_FLOAT64_C(   -90.70), SIMDE_FLOAT64_C(   -35.25), SIMDE_FLOAT64_C(   -55.92), SIMDE_FLOAT64_C(    91.62) },
      { SIMDE_FLOAT64_C(    49.07), SIMDE_FLOAT64_C(   -48.16), SIMDE_FLOAT64_C(    37.98), SIMDE_FLOAT64_C(   -89.53),
        SIMDE_FLOAT64_C(    30.97), SIMDE_FLOAT64_C(   -76.20), SIMDE_FLOAT64_C(   -18.45), SIMDE_FLOAT64_C(   -36.22) },
      { SIMDE_FLOAT64_C(   706.29), SIMDE_FLOAT64_C(   533.56), SIMDE_FLOAT64_C(  -534.21), SIMDE_FLOAT64_C(  -539.66),
        SIMDE_FLOAT64_C(   -64.45), SIMDE_FLOAT64_C(   847.44), SIMDE_FLOAT64_C(   900.30), SIMDE_FLOAT64_C(  -706.27) },
      { SIMDE_FLOAT64_C(  3769.73), SIMDE_FLOAT64_C( -1342.27), SIMDE_FLOAT64_C(  3173.40), SIMDE_FLOAT64_C( -2680.32),
        SIMDE_FLOAT64_C( -2873.43), SIMDE_FLOAT64_C(  3533.49), SIMDE_FLOAT64_C(  1932.02), SIMDE_FLOAT64_C( -4024.75) } },
    { { SIMDE_FLOAT64_C(   -28.07), SIMDE_FLOAT64_C(    78.32), SIMDE_FLOAT64_C(   -34.12), SIMDE_FLOAT64_C(    -3.25),
        SIMDE_FLOAT64_C(    -7.25), SIMDE_FLOAT64_C(    10.89), SIMDE_FLOAT64_C(    81.19), SIMDE_FLOAT64_C(   -24.26) },
      { SIMDE_FLOAT64_C(    64.22), SIMDE_FLOAT64_C(    13.80), SIMDE_FLOAT64_C(   -24.84), SIMDE_FLOAT64_C(   -54.57),
        SIMDE_FLOAT64_C(     4.33), SIMDE_FLOAT64_C(    28.21), SIMDE_FLOAT64_C(   -30.50), SIMDE_FLOAT64_C(   -96.96) },
      { SIMDE_FLOAT64_C(    40.88), SIMDE_FLOAT64_C(   992.17), SIMDE_FLOAT64_C(  -527.77), SIMDE_FLOAT64_C(  -236.95),
        SIMDE_FLOAT64_C(  -475.32), SIMDE_FLOAT64_C(  -695.07), SIMDE_FLOAT64_C(   -17.12), SIMDE_FLOAT64_C(   223.14) },
      { SIMDE_FLOAT64_C( -1761.78), SIMDE_FLOAT64_C(  2072.99), SIMDE_FLOAT64_C(   319.77), SIMDE_FLOAT64_C(   -59.60),
        SIMDE_FLOAT64_C(  -506.71), SIMDE_FLOAT64_C(  -387.86), SIMDE_FLOAT64_C( -2493.41), SIMDE_FLOAT64_C(  2575.39) } },
    { { SIMDE_FLOAT64_C(    35.24), SIMDE_FLOAT64_C(    65.04), SIMDE_FLOAT64_C(   -19.35), SIMDE_FLOAT64_C(    -9.52),
        SIMDE_FLOAT64_C(    46.63), SIMDE_FLOAT64_C(    88.76), SIMDE_FLOAT64_C(     6.35), SIMDE_FLOAT64_C(   -20.93) },
      { SIMDE_FLOAT64_C(   -26.93), SIMDE_FLOAT64_C(   -13.31), SIMDE_FLOAT64_C(    37.30), SIMDE_FLOAT64_C(   -56.20),
        SIMDE_FLOAT64_C(    92.98), SIMDE_FLOAT64_C(   -92.79), SIMDE_FLOAT64_C(    60.93), SIMDE_FLOAT64_C(   -90.76) },
      { SIMDE_FLOAT64_C(   453.61), SIMDE_FLOAT64_C(  -208.18), SIMDE_FLOAT64_C(  -835.16), SIMDE_FLOAT64_C(  -496.06),
        SIMDE_FLOAT64_C(   796.10), SIMDE_FLOAT64_C(    16.91), SIMDE_FLOAT64_C(   288.58), SIMDE_FLOAT64_C(   203.13) },
      { SIMDE_FLOAT64_C(  -495.40), SIMDE_FLOAT64_C( -1073.86), SIMDE_FLOAT64_C( -1556.91), SIMDE_FLOAT64_C(    38.96),
        SIMDE_FLOAT64_C(  5131.76), SIMDE_FLOAT64_C( -8219.13), SIMDE_FLOAT64_C(   675.49), SIMDE_FLOAT64_C(  2102.74) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512d a = simde_mm512_loadu_pd(test_vec[i].a);
    simde__m512d b = simde_mm512_loadu_pd(test_vec[i].b);
    simde__m512d c = simde_mm512_loadu_pd(test_vec[i].c);
    simde__m512d r = simde_mm512_fmadd_pd(a, b, c);
    simde_test_x86_assert_equal_f64x8(r, simde_mm512_loadu_pd(test_vec[i].r), 1);
  }

  return 0;
}

SIMDE_TEST_FUNC_LIST_BEGIN
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_fmadd_ps)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_fmadd_ps)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_fmadd_ps)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_fmadd_pd)
SIMDE_TEST_FUNC_LIST_END

#include <test/x86/avx512/test-avx512-footer.h>
