use ui;
use core:sync;
use core:lang;

/**
 * Exception thrown to exit.
 */
class ExitError extends Exception {
	void message(StrBuf to) : override {
		to << "Time to terminate!";
	}
}


/**
 * Represents a running thread inside the program.
 */
class ProgThread on Render {
	// Semaphore used to pause the monitored thread.
	private Sema sema;

	// Duration for automatic stepping. If zero, we will wait for user input instead.
	private Duration autoStep;

	// Current call depth. There may be additional entries in 'frames', as we might have exited a
	// function but not yet received any new information. In this case, we will still show the old
	// state since it might contain references to values that are still known by the code, but not
	// otherwise reachable (eg. the return value).
	Nat callDepth;

	// Current thread ID (the simple thread ID generated by this system).
	Nat threadId;

	// Future to watch whenever this thread terminated.
	FutureBase? watch;

	// Entries of called functions. We expect this array to contain at least one element when we
	// receive calls to 'onNewLocation'.
	StackFrame[] frames;

	// Owning Program object, so we can access the callback.
	Program owner;

	// Memory reads and writes by this thread since the last statement.
	// Not stored inside a particular frame in order to properly track reading function
	// parameters, etc.
	// TODO: We could copy/move these around to solve the issue as well, we need
	// to examine what works best in practice.
	MemTracker memory;

	// Waiting for some external event?
	Bool sleeping;

	// Is the thread alive?
	Bool alive() { callDepth > 0; }

	// Did the thread crash? (Only relevant when "alive" returns false)
	Bool crashed() { terminating; }

	// Currently running? This is 'true' even if the thread is currently waiting in an animation.
	private Bool running;

	// Terminate the thread at the next opportunity to do so.
	private Bool doTerminate;

	// Did we throw the termination exception? If so, we will not intercept execution anymore, since
	// destructors might be intercepted.
	private Bool terminating;

	// Create.
	init(Program owner, Nat threadId) {
		init() {
			sema(0);
			callDepth = 0;
			threadId = threadId;
			owner = owner;
			running = true;
		}
	}

	// Get the current location.
	SrcPos? pos() {
		if (frames.empty())
			return null;
		frames.last.pos;
	}

	// Called by a monitored thread to possible pause execution.
	// Returns 'false' if this event is not interesting to propagate.
	Bool onNewLocation(SrcPos pos, StackVar[] vars) {
		if (terminating)
			return false;

		popFrames();

		var last = frames.last;
		last.variables = vars;

		if (pos.any) {
			last.pos = pos;
			wait();
		}

		memory.clear();
		true;
	}

	// Called when a monitored thread is about to return.
	// Returns 'false' if this event is not interesting to propagate.
	void onFunctionReturn(StackVar[] vars) {
		if (terminating)
			return false;

		popFrames();

		var last = frames.last;
		last.variables = vars;
		last.returned = true;
		wait();

		memory.clear();
		true;
	}

	// Called when a new function was entered.
	MemTracker onFunctionEntered(Str name, SrcPos pos, ViewHints hints, StackVar[] vars) {
		popFrames();

		callDepth++;
		frames.push(StackFrame(name, hints, vars, pos));

		if (terminating)
			return memory;

		if (hints.pauseOnEntry())
			wait();

		memory;
	}

	// Called when a function is about to return.
	// Returns 'false' if this event is not interesting to propagate.
	Bool onFunctionExited() {
		// print("Function ${frames.last.function} exited.");
		if (callDepth > 0)
			callDepth--;

		popFrames();

		// Tell the system that we have exited if this is the last function in the stack.
		if (callDepth == 0)
			return true;
		return !terminating;
	}

	// Called when function is about to throw an exception. Disables further tracing of the function.
	void onFatalException() {
		terminating = true;
	}

	// Make the thread sleep until something else wakes it. Used to implement synchronization primitives.
	void lockSleep(Variant value) {
		sleeping = true;
		if (frames.any) {
			var last = frames.last;
			StackVar var("For", value);
			frames.push(StackFrame("Waiting...", last.hints, [var], last.pos));
		}

		wait();

		// Remove any "waiting for" frame if needed.
		popFrames();
	}

	// Wake the thread again. Assumes that 'lockSleep' was called earlier.
	void lockWake() {
		sleeping = false;
		if (!running)
			sema.up();
	}

	// Set the animation delay. Resumes the thread if it was stopped.
	void resume(Duration duration) {
		autoStep = duration;

		if (!running & !sleeping)
			sema.up();
	}

	// Resume execution, stopping at the next location.
	void resume() {
		if (running | sleeping)
			return;

		autoStep = Duration();
		sema.up();
	}

	// Pause the thread at the nearest convenient location.
	void pause() {
		autoStep = Duration();
	}

	// Terminate the thread.
	void terminate() {
		doTerminate = true;
		autoStep = Duration();
		sema.up();
		// Once more, just to be safe...
		sema.up();
	}

	// Decrease the stack frames until the proper depth is reached.
	private void popFrames() {
		while (frames.count > max(1n, callDepth))
			frames.pop();
	}

	// Determine whether or not we shall wait.
	private void wait() {
		// Remember if this 'wait' was due to us needing to sleep for a semaphore etc. If so,
		// we should not notify the owner when we wake, as that wake was not due to a user action.
		Bool wasSleeping = sleeping;
		owner.notifyChange();
		if (wasSleeping | autoStep == Duration()) {
			running = false;
			sema.down();
		} else {
			sleep(autoStep);

			// If paused during sleep, keep waiting.
			if (autoStep == Duration()) {
				running = false;
				sema.down();
			}
		}
		running = true;

		if (doTerminate) {
			terminating = true;
			throw ExitError();
		} else if (!wasSleeping) {
			owner.notifyAdvance(threadId);
		}
	}
}

/**
 * A variable in a stack-frame.
 *
 * Note: We fill in this type directly from machine code, don't alter the layout!
 */
class StackVar {
	Str name;
	unsafe:RawPtr value;

	// If we need to properly destroy 'value' when we're done with it, this Variant will contain the
	// same value as 'value'. Otherwise, this variant is empty.
	Variant destroy;

	init(Str name) {
		init { name = name; }
	}

	init(Str name, Variant val) {
		init {
			name = name;
			destroy = val;
		}

		// Make sure to use the correct value.
		value = unsafe:RawPtr(destroy);
	}
}


/**
 * A stack frame of a thread.
 */
class StackFrame on Render {
	// Name of the function. (TODO: Add a SrcPos?)
	Str function;

	// View hints for this function. Used to scrape variables of the stack.
	ViewHints hints;

	// The current state of all variables. Ordered roughly as they appear in the source.
	StackVar[] variables;

	// Location in this frame.
	SrcPos pos;

	// Did we return from this frame?
	Bool returned;

	// Create.
	init(Str name, ViewHints hints, StackVar[] vars, SrcPos pos) {
		init {
			function = name;
			hints = hints;
			variables = vars;
			pos = pos;
			returned = false;
		}
	}
}
