/*
*         OpenPBS (Portable Batch System) v2.3 Software License
*
* Copyright (c) 1999-2000 Veridian Information Solutions, Inc.
* All rights reserved.
*
* ---------------------------------------------------------------------------
* For a license to use or redistribute the OpenPBS software under conditions
* other than those described below, or to purchase support for this software,
* please contact Veridian Systems, PBS Products Department ("Licensor") at:
*
*    www.OpenPBS.org  +1 650 967-4675                  sales@OpenPBS.org
*                        877 902-4PBS (US toll-free)
* ---------------------------------------------------------------------------
*
* This license covers use of the OpenPBS v2.3 software (the "Software") at
* your site or location, and, for certain users, redistribution of the
* Software to other sites and locations.  Use and redistribution of
* OpenPBS v2.3 in source and binary forms, with or without modification,
* are permitted provided that all of the following conditions are met.
* After December 31, 2001, only conditions 3-6 must be met:
*
* 1. Commercial and/or non-commercial use of the Software is permitted
*    provided a current software registration is on file at www.OpenPBS.org.
*    If use of this software contributes to a publication, product, or
*    service, proper attribution must be given; see www.OpenPBS.org/credit.html
*
* 2. Redistribution in any form is only permitted for non-commercial,
*    non-profit purposes.  There can be no charge for the Software or any
*    software incorporating the Software.  Further, there can be no
*    expectation of revenue generated as a consequence of redistributing
*    the Software.
*
* 3. Any Redistribution of source code must retain the above copyright notice
*    and the acknowledgment contained in paragraph 6, this list of conditions
*    and the disclaimer contained in paragraph 7.
*
* 4. Any Redistribution in binary form must reproduce the above copyright
*    notice and the acknowledgment contained in paragraph 6, this list of
*    conditions and the disclaimer contained in paragraph 7 in the
*    documentation and/or other materials provided with the distribution.
*
* 5. Redistributions in any form must be accompanied by information on how to
*    obtain complete source code for the OpenPBS software and any
*    modifications and/or additions to the OpenPBS software.  The source code
*    must either be included in the distribution or be available for no more
*    than the cost of distribution plus a nominal fee, and all modifications
*    and additions to the Software must be freely redistributable by any party
*    (including Licensor) without restriction.
*
* 6. All advertising materials mentioning features or use of the Software must
*    display the following acknowledgment:
*
*     "This product includes software developed by NASA Ames Research Center,
*     Lawrence Livermore National Laboratory, and Veridian Information
*     Solutions, Inc.
*     Visit www.OpenPBS.org for OpenPBS software support,
*     products, and information."
*
* 7. DISCLAIMER OF WARRANTY
*
* THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND. ANY EXPRESS
* OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
* OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT
* ARE EXPRESSLY DISCLAIMED.
*
* IN NO EVENT SHALL VERIDIAN CORPORATION, ITS AFFILIATED COMPANIES, OR THE
* U.S. GOVERNMENT OR ANY OF ITS AGENCIES BE LIABLE FOR ANY DIRECT OR INDIRECT,
* INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
* LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
* OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
* LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
* NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
* EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/

/*
 * @file svr_task.c
 *
 * Contains functions to deal with the server's task list
 *
 * A task list is a set of pending functions usually associated with
 * processing a reply message.
 *
 * Functions included are:
 */

#include <pbs_config.h>   /* the master config generated by configure */

#include "portability.h"
#include <stdlib.h>
#include <time.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/wait.h>
#include "server_limits.h"
#include "list_link.h"
#include "work_task.h"


/* Global Data Items: */

extern tlist_head task_list_immed;
extern tlist_head task_list_timed;
extern tlist_head task_list_event;




/*
 * set_task - add the job entry to the task list
 *
 * Task time depends on the type of task.  The list is time ordered.
 */

struct work_task *set_task(

        enum work_type  type,
        long          event_id,  /* I - based on type can be time of event */
        void (*func)(),
        void         *parm)

  {

  struct work_task *pnew;

  struct work_task *pold;

  pnew = (struct work_task *)malloc(sizeof(struct work_task));

  if (pnew == NULL)
    {
    return(NULL);
    }

  CLEAR_LINK(pnew->wt_linkall);

  CLEAR_LINK(pnew->wt_linkobj);

  pnew->wt_event = event_id;
  pnew->wt_type  = type;
  pnew->wt_func  = func;
  pnew->wt_parm1 = parm;
  pnew->wt_parm2 = NULL;
  pnew->wt_aux   = 0;

  if (type == WORK_Immed)
    {
    append_link(&task_list_immed, &pnew->wt_linkall, pnew);
    }
  else if (type == WORK_Timed)
    {
    pold = (struct work_task *)GET_NEXT(task_list_timed);

    while (pold != NULL)
      {
      if (pold->wt_event > pnew->wt_event)
        break;

      pold = (struct work_task *)GET_NEXT(pold->wt_linkall);
      }

    if (pold != NULL)
      {
      insert_link(
        &pold->wt_linkall,
        &pnew->wt_linkall,
        pnew,
        LINK_INSET_BEFORE);
      }
    else
      {
      append_link(&task_list_timed, &pnew->wt_linkall, pnew);
      }
    }
  else
    {
    append_link(&task_list_event, &pnew->wt_linkall, pnew);
    }

  return(pnew);
  }  /* END set_task() */




/*
 * dispatch_task - dispatch a work task found on a work list
 *
 * delinks the work task entry, calls the associated function with
 * the parameters from the work task entry, and then frees the entry.
 */

void dispatch_task(

  struct work_task *ptask)

  {
  delete_link(&ptask->wt_linkall);
  delete_link(&ptask->wt_linkobj);

  if (ptask->wt_func != NULL)
    ptask->wt_func(ptask);  /* dispatch process function */

  free(ptask);

  return;
  }  /* END dispatch_task() */





/*
 * delete_task - unlink and free a work_task structure.
 */

void
delete_task(struct work_task *ptask)
  {
  delete_link(&ptask->wt_linkobj);
  delete_link(&ptask->wt_linkall);
  (void)free(ptask);
  }
