/*
 * Copyright (C) 2000-2024 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>
#include <unistd.h>

#include <xine/sorted_array.h>

#include "_xitk.h"
#include "label.h"
#include "font.h"
#include "backend.h"

typedef enum {
  _LABEL_ANIM_NONE = 0, /** << no anim thread; change_mutex and anim_wake unset */
  _LABEL_ANIM_RUN,      /** << anim running */
  _LABEL_ANIM_IDLE,     /** << anim standby */
  _LABEL_ANIM_STOP,     /** << waiting for anim to shut down */
  _LABEL_ANIM_LAST
} _label_anim_state_t;

typedef struct _label_anim_s _label_anim_t;

typedef struct {
  xitk_widget_t           w;

  xitk_short_string_t     fontname, label;

  xitk_image_t           *labelpix;

  xitk_pix_font_t        *pix_font;

  int                     length; /** << in chars (pixfont) or pixels (sysfont) */
  int                     align;
  xitk_image_t           *font, *highlight_font;
  size_t                  label_len;

  xitk_simple_callback_t  callback;

  uint32_t                text_color[2], bg_color;
#define _LABEL_FLAG_PAINTED 1
#define _LABEL_FLAG_HIDDEN  2
  uint32_t                flags;

  _label_anim_t          *anim;

  int                     animation;   /** << skin enable */
  int                     anim_want;   /** << text too wide for display */
  int                     anim_step;   /** << pixels per anim step */
  int                     anim_timer;  /** << microseconds between steps */
  int                     anim_offset; /** << pixels from left of full text */
  int                     anim_max;    /** << pixels to loop offset at */
  int                     anim_base;   /** << text base line */

  int                     label_visible;
} _label_private_t;

struct _label_anim_s {
  unsigned int      usecs, used;
#define _LABEL_MAX_CLIENTS 32
  _label_private_t *clients[_LABEL_MAX_CLIENTS];
  pthread_t         thread;
  pthread_mutex_t   mutex;
  pthread_cond_t    wake;
  _label_anim_state_t state;
};

static uint32_t _decompose_hangeul (uint16_t *s, uint32_t bufsize) {
  uint16_t *p, *q, *t, *e = s + bufsize;
  static const uint16_t
  /* initial consonant(s). */
  tab_in[] = {
    /* ㄱ */ 12593,
    /* ㄲ */ 12594,
    /* ㄴ */ 12596,
    /* ㄷ */ 12599,
    /* ㄸ */ 12600,
    /* ㄹ */ 12601,
    /* ㅁ */ 12609,
    /* ㅂ */ 12610,
    /* ㅃ */ 12611,
    /* ㅅ */ 12613,
    /* ㅆ */ 12614,
    /* ㅇ */ 12615,
    /* ㅈ */ 12616,
    /* ㅉ */ 12617,
    /* ㅊ */ 12618,
    /* ㅋ */ 12619,
    /* ㅌ */ 12620,
    /* ㅍ */ 12621,
    /* ㅎ */ 12622
  },
  /* medial vovel(s). can be simplified to (i + 12623) for now. */
#if 0
  tab_me[] = {
    /* ㅏ */ 12623,
    /* ㅐ */ 12624,
    /* ㅑ */ 12625,
    /* ㅒ */ 12626,
    /* ㅓ */ 12627,
    /* ㅔ */ 12628,
    /* ㅕ */ 12629,
    /* ㅖ */ 12630,
    /* ㅗ */ 12631,
    /* ㅘ */ 12632,
    /* ㅙ */ 12633,
    /* ㅚ */ 12634,
    /* ㅛ */ 12635,
    /* ㅜ */ 12636,
    /* ㅝ */ 12637,
    /* ㅞ */ 12638,
    /* ㅟ */ 12639,
    /* ㅠ */ 12640,
    /* ㅡ */ 12641,
    /* ㅢ */ 12642,
    /* ㅣ */ 12643
  },
#endif
  /* final consonant(s). */
  tab_fi[] = {
    /*   */ 0,
    /* ㄱ */ 12593,
    /* ㄲ */ 12594,
    /* ㄳ */ 12595,
    /* ㄴ */ 12596,
    /* ㄵ */ 12597,
    /* ㄶ */ 12598,
    /* ㄷ */ 12599,
    /* ㄹ */ 12601,
    /* ㄺ */ 12602,
    /* ㄻ */ 12603,
    /* ㄼ */ 12604,
    /* ㄽ */ 12605,
    /* ㄾ */ 12606,
    /* ㄿ */ 12607,
    /* ㅀ */ 12608,
    /* ㅁ */ 12609,
    /* ㅂ */ 12610,
    /* ㅄ */ 12612,
    /* ㅅ */ 12613,
    /* ㅆ */ 12614,
    /* ㅇ */ 12615,
    /* ㅈ */ 12616,
    /* ㅊ */ 12618,
    /* ㅋ */ 12619,
    /* ㅌ */ 12620,
    /* ㅍ */ 12621,
    /* ㅎ */ 12622
  };

  /* skip non hangeul part. */
  for (p = s; *p; p++) {
    uint32_t v = (uint32_t)((int32_t)*p - 44032);
    if (v < 19 * 21 * 28)
      break;
  }
  if (!*p)
    return p - s;
  /* now the nasty part. move away the rest. */
  q = p;
  for (; *p; p++) ;
  *--e = 0;
  for (t = e; p > q; )
    *--t = *--p;
  /* full scan. */
  for (p = t; *p; p++) {
    uint32_t v = (uint32_t)((int32_t)*p - 44032);
    if (v >= 19 * 21 * 28) {
      *q++ = *p;
    } else {
      uint32_t w;
      if (q + 3 >= e)
        break;
      w = v / (21 * 28);
      v %= 21 * 28;
      *q++ = tab_in[w];
      w = v / 28;
      v %= 28;
#if 0
      *q++ = tab_me[w];
#else
      *q++ = w + 12623;
#endif
      w = tab_fi[v];
      if (w)
        *q++ = w;
    }
  }
  *q = 0;
  return q - s;
}

static int _find_pix_font_char (xitk_pix_font_t *pf, xitk_point_t *found, int this_char) {
  int range, n = 0;

  for (range = 0; pf->unicode_ranges[range].first > 0; range++) {
    if ((this_char >= pf->unicode_ranges[range].first) && (this_char <= pf->unicode_ranges[range].last))
      break;
    n += pf->unicode_ranges[range].last - pf->unicode_ranges[range].first + 1;
  }

  if (pf->unicode_ranges[range].first <= 0) {
    *found = pf->unknown;
    return 0;
  }

  n += this_char - pf->unicode_ranges[range].first;
  found->x = (n % pf->chars_per_row) * pf->char_width;
  found->y = (n / pf->chars_per_row) * pf->char_height;
  return 1;
}

static int _need_decompose_hangeul (xitk_pix_font_t *pf) {
  xitk_point_t found;
  return _find_pix_font_char (pf, &found, 12593) && !_find_pix_font_char (pf, &found, 44032);
}

static void _create_label_pixmap (_label_private_t *wp) {
  xitk_image_t          *font;
  xitk_pix_font_t       *pix_font;
  int                    pixwidth, x_dest, start;
  int                    len, anim_add = 0;
  uint16_t               buf[2048];

  font = (wp->w.state & (XITK_WIDGET_STATE_MOUSE | XITK_WIDGET_STATE_FOCUS)) && wp->highlight_font ? wp->highlight_font : wp->font;
  wp->anim_offset = 0;
  wp->anim_want = 0;

  if (!font->pix_font) {
    /* old style */
    xitk_image_set_pix_font (font, "");
    if (!font->pix_font)
      return;
  }
  pix_font = font->pix_font;

  {
    const uint8_t *p = (const uint8_t *)wp->label.s;
    uint16_t *q = buf, *e = buf + sizeof (buf) / sizeof (buf[0]) - 1;
    while (*p && (q < e)) {
      if (!(p[0] & 0x80)) {
        *q++ = p[0];
        p += 1;
      } else if (((p[0] & 0xe0) == 0xc0) && ((p[1] & 0xc0) == 0x80)) {
        *q++ = ((uint32_t)(p[0] & 0x1f) << 6) | (p[1] & 0x3f);
        p += 2;
      } else if (((p[0] & 0xf0) == 0xe0) && ((p[1] & 0xc0) == 0x80) && ((p[2] & 0xc0) == 0x80)) {
        *q++ = ((uint32_t)(p[0] & 0x0f) << 12) | ((uint32_t)(p[1] & 0x3f) << 6) | (p[2] & 0x3f);
        p += 3;
      } else {
        *q++ = p[0];
        p += 1;
      }
    }
    *q = 0;
    if (_need_decompose_hangeul (wp->pix_font))
        q = buf + _decompose_hangeul (buf, sizeof (buf) / sizeof (buf[0]));
    wp->label_len = len = q - buf;
  }

  if (wp->animation && (len > wp->length)) {
    anim_add = wp->length + 5;
    wp->anim_want = 1;
    wp->anim_max = (len + 5) * pix_font->char_width;
    start = 0;
  } else {
    start = (wp->align == ALIGN_CENTER) ? ((len - wp->length) >> 1)
          : (wp->align == ALIGN_RIGHT) ? (len - wp->length)
          : 0; /** << default = left */
    if (start > 0)
      wp->anim_offset = start * pix_font->char_width;
  }

  /* reuse or reallocate pixmap */
  pixwidth = len + anim_add;
  if (pixwidth < wp->length)
    pixwidth = wp->length;
  if (pixwidth < 1)
    pixwidth = 1;
  pixwidth *= pix_font->char_width;
  if (wp->labelpix) {
    if ((wp->labelpix->width != pixwidth) || (wp->labelpix->height != pix_font->char_height))
      xitk_image_free_image (&wp->labelpix);
  }
  if (!wp->labelpix) {
    wp->labelpix = xitk_image_new (wp->w.wl->xitk, NULL, 0, pixwidth, pix_font->char_height);
#if 0
    printf ("xine.label: new pixmap %d:%d -> %d:%d for \"%s\"\n", pixwidth, pix_font->char_height,
      wp->labelpix->width, wp->labelpix->height, wp->label);
#endif
    if (!wp->labelpix)
      return;
  }

  do {
    int w;
    xitk_point_t found;
    const union { uint16_t w[2]; uint32_t l; } s70 = { .w = { 7, 0 }};
    /* lamp test pattern */
    if (memcmp (buf, &s70.w, 4))
      break;
    if (!_find_pix_font_char (wp->pix_font, &found, 0xa0))
      break;
    /* HACK: write the first _2_ chars manually. This way, backend_x11 will add a transparency
     * mask to wp->labelpix if font has one. */
    for (w = 0; w < 2; w++)
      xitk_image_copy_rect (font, wp->labelpix,
        found.x, found.y, pix_font->char_width, pix_font->char_height, w * pix_font->char_width, 0);
    /* now fast fill the rest. */
    for (; w * 2 <= wp->length; w *= 2)
      xitk_image_copy_rect (wp->labelpix, wp->labelpix,
        0, 0, w * pix_font->char_width, pix_font->char_height, w * pix_font->char_width, 0);
    if (w < wp->length)
      xitk_image_copy_rect (wp->labelpix, wp->labelpix,
        0, 0, (wp->length - w) * pix_font->char_width, pix_font->char_height, w * pix_font->char_width, 0);
    return;
  } while (0);

  x_dest = 0;
  /* (   ) */
  while (start < 0) {
    xitk_image_copy_rect (font, wp->labelpix,
      pix_font->space.x, pix_font->space.y,
      pix_font->char_width, pix_font->char_height, x_dest, 0);
    x_dest += pix_font->char_width;
    start++;
  }
  /* [foo] */
  {
    const uint16_t *p = buf;
    while (*p) {
      xitk_point_t pt;
      _find_pix_font_char (font->pix_font, &pt, *p);
      p++;

      xitk_image_copy_rect (font, wp->labelpix,
        pt.x, pt.y, pix_font->char_width, pix_font->char_height, x_dest, 0);
      x_dest += pix_font->char_width;
    }
  }
  /* foo[ *** fo] */
  if (anim_add) {
    xitk_image_copy_rect (font, wp->labelpix,
      pix_font->space.x, pix_font->space.y,
      pix_font->char_width, pix_font->char_height, x_dest, 0);
    x_dest += pix_font->char_width;
    xitk_image_copy_rect (font, wp->labelpix,
      pix_font->asterisk.x, pix_font->asterisk.y,
      pix_font->char_width, pix_font->char_height, x_dest, 0);
    x_dest += pix_font->char_width;
    xitk_image_copy_rect (font, wp->labelpix,
      pix_font->asterisk.x, pix_font->asterisk.y,
      pix_font->char_width, pix_font->char_height, x_dest, 0);
    x_dest += pix_font->char_width;
    xitk_image_copy_rect (font, wp->labelpix,
      pix_font->asterisk.x, pix_font->asterisk.y,
      pix_font->char_width, pix_font->char_height, x_dest, 0);
    x_dest += pix_font->char_width;
    xitk_image_copy_rect (font, wp->labelpix,
      pix_font->space.x, pix_font->space.y,
      pix_font->char_width, pix_font->char_height, x_dest, 0);
    x_dest += pix_font->char_width;
    xitk_image_copy_rect (wp->labelpix, wp->labelpix,
      0, 0, pixwidth - x_dest, pix_font->char_height, x_dest, 0);
    x_dest = pixwidth;
  }

  /* fill gap with spaces */
  while (x_dest < pixwidth) {
      xitk_image_copy_rect (font, wp->labelpix,
        pix_font->space.x, pix_font->space.y,
        pix_font->char_width, pix_font->char_height, x_dest, 0);
      x_dest += pix_font->char_width;
  }
}

static void _create_label_image (_label_private_t *wp) {
  xitk_font_t *fs = xitk_font_load_font (wp->w.wl->xitk, wp->fontname.s);
  int lbear, rbear, wid, asc, des;
  int textl, textw, astl, astw, imagew, w, h;
  unsigned int state;
  XITK_HV_INIT;

  wp->anim_want = 0;
  if (!fs)
    return;

  w = XITK_HV_H (wp->w.size);
  h = XITK_HV_V (wp->w.size);
  xitk_font_text_extent (fs, wp->label.s, wp->label_len, &lbear, &rbear, &wid, &asc, &des);
  wp->anim_base = ((h + asc + des) >> 1) - des;
  textw = rbear - lbear;
  textl = (wp->align == ALIGN_CENTER) ? ((w - textw) >> 1)
        : (wp->align == ALIGN_RIGHT) ? (w - 2 - textw)
        : 2; /** << default = left */
  textl -= lbear;

  if ((textw + 4 <= w) || !wp->animation) {
    /* fits the space, or clip. defer render to paint. */
    wp->anim_offset = textl;
    return;
  }

  /* prepare animation. */
  xitk_font_text_extent (fs, " *** ", 5, &lbear, &rbear, &wid, &asc, &des);
  astl = -lbear;
  astw = rbear - lbear;
  imagew = textw + astw + w;
  if (wp->labelpix && (wp->labelpix->width >= imagew)) {
    ;
  } else {
    xitk_image_free_image (&wp->labelpix);
    wp->labelpix = xitk_image_new (wp->w.wl->xitk, NULL, 0, imagew, h);
    if (!wp->labelpix) {
      xitk_font_unload_font (fs);
      return;
    }
  }
  state = xitk_image_find_state (sizeof (wp->text_color) / sizeof (wp->text_color[0]) - 1, wp->w.state);
  xitk_image_fill_rectangle (wp->labelpix, 0, 0, imagew, h, wp->bg_color);
  xitk_image_draw_string (wp->labelpix, fs, textl, wp->anim_base,
    wp->label.s, wp->label_len, wp->text_color[state]);
  xitk_image_draw_string (wp->labelpix, fs, textw + astl, wp->anim_base,
    " *** ", 5, wp->text_color[state]);
  xitk_image_copy_rect (wp->labelpix, wp->labelpix,
    0, 0, imagew - (textw + astw), h, textw + astw, 0);
  xitk_font_unload_font (fs);
  wp->anim_want = 1;
  wp->anim_max = textw + astw;
}

static void _create_label_gfx (_label_private_t *wp) {
  if (wp->pix_font)
    _create_label_pixmap (wp);
  else
    _create_label_image (wp);
}

/*
 *
 */
const char *xitk_label_get_label (xitk_widget_t *w) {
  _label_private_t *wp;

  xitk_container (wp, w, w);
  if (wp && ((wp->w.type & WIDGET_TYPE_MASK) == WIDGET_TYPE_LABEL))
    return wp->label.s;
  return NULL;
}

/*
 *
 */
static void _label_paint (_label_private_t *wp, const widget_event_t *event) {
  XITK_HV_INIT;
#ifdef XITK_PAINT_DEBUG
    printf ("xitk.label.paint (%d, %d, %d, %d).\n", event->x, event->y, event->width, event->height);
#endif
  if ((wp->w.state & XITK_WIDGET_STATE_VISIBLE) && XITK_HV_H (wp->w.size) && wp->label_visible) {
    /* non skinable widget */
    if (!wp->pix_font && !wp->anim) {
      xitk_font_t *fs = xitk_font_load_font (wp->w.wl->xitk, wp->fontname.s);

      if (fs) {
        xitk_image_t *temp = xitk_image_temp_get (wp->w.wl->xitk);
        unsigned int state = xitk_image_find_state (sizeof (wp->text_color) / sizeof (wp->text_color[0]) - 1, wp->w.state);

        if (wp->bg_color == ~0u)
          xitk_image_copy_rect (wp->font, temp, 0, 0, XITK_HV_H (wp->w.size), XITK_HV_V (wp->w.size), 0, 0);
        else
          xitk_image_fill_rectangle (temp, 0, 0, XITK_HV_H (wp->w.size), XITK_HV_V (wp->w.size), wp->bg_color);
        xitk_image_draw_string (temp, fs, wp->anim_offset, wp->anim_base,
          wp->label.s, wp->label_len, wp->text_color[state]);
        xitk_image_draw_image (wp->w.wl, temp,
          event->x - XITK_HV_H (wp->w.pos), event->y - XITK_HV_V (wp->w.pos), event->width, event->height, event->x, event->y, 0);
        xitk_font_unload_font (fs);
      }
      return;
    }

    if (((wp->w.state ^ wp->w.shown_state) & (XITK_WIDGET_STATE_MOUSE | XITK_WIDGET_STATE_FOCUS))) {
      if (wp->highlight_font || (wp->text_color[0] != wp->text_color[1]))
        _create_label_gfx (wp);
    }
    xitk_image_draw_image (wp->w.wl, wp->labelpix,
    wp->anim_offset + event->x - XITK_HV_H (wp->w.pos), event->y - XITK_HV_V (wp->w.pos),
      event->width, event->height,
      event->x, event->y, !!wp->anim);
    wp->w.shown_state = wp->w.state;
  }
}

/*
 *
 */
static void *xitk_label_animation_loop (void *data) {
  _label_anim_t *anim = (_label_anim_t *)data;
  static const uint8_t run[_LABEL_ANIM_LAST] = {
    [_LABEL_ANIM_RUN] = 1,
    [_LABEL_ANIM_IDLE] = 1
  };
  struct timespec line = {0, 0};
  XITK_HV_INIT;

  xitk_gettime_ts (&line);
  pthread_mutex_lock (&anim->mutex);

  while (run[anim->state]) {
    if (anim->state == _LABEL_ANIM_RUN) {
      struct timespec now = {0, 0};
      unsigned int u;

      xitk_gettime_ts (&now);
      for (u = 0; u < anim->used; u++) {
        _label_private_t *wp = anim->clients[u];

        if ((wp->w.state & XITK_WIDGET_STATE_VISIBLE) && wp->labelpix) {
          widget_event_t event;

          event.x = XITK_HV_H (wp->w.pos);
          event.y = XITK_HV_V (wp->w.pos);
          event.width = XITK_HV_H (wp->w.size);
          event.height = XITK_HV_V (wp->w.size);
          _label_paint (wp, &event);
          wp->anim_offset = (unsigned int)(wp->anim_offset + wp->anim_step + wp->anim_max) % (unsigned int)wp->anim_max;
          wp->w.shown_state = wp->w.state;
        }
      }
      line.tv_sec += anim->usecs / 1000000;
      line.tv_nsec += (anim->usecs % 1000000) * 1000;
      if (line.tv_nsec >= 1000000000) {
        line.tv_nsec -= 1000000000;
        line.tv_sec += 1;
      }
      if  ((now.tv_sec > line.tv_sec)
        || ((now.tv_sec == line.tv_sec) && (now.tv_nsec > line.tv_nsec))) {
        /* late wakeup, skip missed cycles. */
        line = now;
        continue;
      }
      now = line;
      pthread_cond_timedwait (&anim->wake, &anim->mutex, &now);
    } else {
      pthread_cond_wait (&anim->wake, &anim->mutex);
    }
  }
  pthread_mutex_unlock (&anim->mutex);

  return NULL;
}

static int _label_anim_cmp (void *a, void *b) {
  _label_anim_t *d = (_label_anim_t *)a, *e = (_label_anim_t *)b;
  return (int)d->usecs - (int)e->usecs;
}

static int _label_anim_join (_label_private_t *wp) {
  _label_anim_t *anim, dummy = {.usecs = wp->anim_timer};
  xine_sarray_t **array = xitk_anim_threads (wp->w.wl->xitk);
  int i;

  if (!array)
    return 0;
  if (!*array) {
    *array = xine_sarray_new (32, _label_anim_cmp);
    if (!*array)
      return 0;
  }
  i = xine_sarray_binary_search (*array, &dummy);
  if (i >= 0) {
    anim = xine_sarray_get (*array, i);
    pthread_mutex_lock (&anim->mutex);
    if (anim->used >= _LABEL_MAX_CLIENTS) {
      pthread_mutex_unlock (&anim->mutex);
      return _LABEL_MAX_CLIENTS;
    }
    anim->clients[anim->used++] = wp;
    pthread_mutex_unlock (&anim->mutex);
  } else {
    pthread_attr_t pth_attrs;
#if defined(_POSIX_THREAD_PRIORITY_SCHEDULING) && (_POSIX_THREAD_PRIORITY_SCHEDULING > 0)
    struct sched_param pth_params;
#endif
    anim = xitk_xmalloc (sizeof (*anim));
    if (!anim)
      return 0;
    anim->usecs = wp->anim_timer;
    anim->used = 1;
    anim->clients[0] = wp;
    anim->state = _LABEL_ANIM_RUN;
    pthread_mutex_init (&anim->mutex, NULL);
    pthread_cond_init (&anim->wake, NULL);
    pthread_attr_init (&pth_attrs);
#if defined(_POSIX_THREAD_PRIORITY_SCHEDULING) && (_POSIX_THREAD_PRIORITY_SCHEDULING > 0)
    pthread_attr_getschedparam (&pth_attrs, &pth_params);
    pth_params.sched_priority = sched_get_priority_min (SCHED_OTHER);
    pthread_attr_setschedparam (&pth_attrs, &pth_params);
#endif
    i = pthread_create (&anim->thread, &pth_attrs, xitk_label_animation_loop, (void *)anim);
    pthread_attr_destroy (&pth_attrs);
    if (i) {
      pthread_cond_destroy (&anim->wake);
      pthread_mutex_destroy (&anim->mutex);
      free (anim);
      return 0;
    }
    xine_sarray_add (*array, anim);
  }
  wp->anim = anim;
  return anim->used;
}

static int _label_anim_leave (_label_private_t *wp) {
  _label_anim_t *anim;
  xine_sarray_t **array;
  void *dummy;
  int i;

  if (!(anim = wp->anim))
    return 0;

  wp->anim = NULL;
  pthread_mutex_lock (&anim->mutex);
  for (i = 0; i < (int)anim->used; i++) {
    if (anim->clients[i] == wp)
      break;
  }
  if (i >= (int)anim->used) {
    i = anim->used;
    pthread_mutex_unlock (&anim->mutex);
    return i;
  }
  anim->used--;
  anim->clients[i] = anim->clients[anim->used];
  if ((i = anim->used) > 0) {
    pthread_mutex_unlock (&anim->mutex);
    return i;
  }
  anim->state = _LABEL_ANIM_STOP;
  pthread_cond_signal (&anim->wake);
  pthread_mutex_unlock (&anim->mutex);

  pthread_join (anim->thread, &dummy);

  pthread_cond_destroy (&anim->wake);
  pthread_mutex_destroy (&anim->mutex);

  do {
    if (!(array = xitk_anim_threads (wp->w.wl->xitk)))
      break;
    if (!*array)
      break;
    i = xine_sarray_binary_search (*array, anim);
    if (i < 0)
      break;
    xine_sarray_remove (*array, i);
  } while (0);

  free (anim);
  return 0;
}

/*
 *
 */
static void _label_destroy (_label_private_t *wp) {
  if (wp->anim) {
    int n = _label_anim_leave (wp);

    if (wp->w.wl->xitk->verbosity >= 2)
      printf ("xitk.label.anim.leave (%iusec, %i).\n", wp->anim_timer, n);
  }

  xitk_image_free_image (&(wp->labelpix));

  if (!wp->w.skin_element_name[0]) {
    xitk_image_free_image (&(wp->font));
    xitk_image_free_image (&(wp->highlight_font));
  }

  xitk_short_string_deinit (&wp->label);
  xitk_short_string_deinit (&wp->fontname);
}

/*
 *
 */
static void _label_setup_label (_label_private_t *wp) {
  XITK_HV_INIT;
  int leave, join;

  if (!XITK_HV_H (wp->w.size))
    return;

  if (wp->anim)
    pthread_mutex_lock (&wp->anim->mutex);
  _create_label_gfx (wp);
  if (wp->anim)
    pthread_mutex_unlock (&wp->anim->mutex);

  /* the reuse logic. */
  join = (wp->animation && wp->anim_want);
  if (join) {
    if (wp->anim) {
      if ((int)wp->anim->usecs != wp->anim_timer)
        leave = 1;
      else
        join = leave = 0;
    } else {
      leave = 0;
    }
  } else {
    leave = !!wp->anim;
  }
  if (leave) {
    int usecs = wp->anim->usecs, n = _label_anim_leave (wp);

    if (wp->w.wl->xitk->verbosity >= 2)
      printf ("xitk.label.anim.leave (%iusec, %i).\n", usecs, n);
  }
  if (join) {
    int n = _label_anim_join (wp);

    if (wp->w.wl->xitk->verbosity >= 2)
      printf ("xitk.label.anim.join (%iusec, %i).\n", wp->anim_timer, n);
  }

  if (wp->anim)
    pthread_mutex_lock (&wp->anim->mutex);
  {
    widget_event_t event;

    event.x = XITK_HV_H (wp->w.pos);
    event.y = XITK_HV_V (wp->w.pos);
    event.width = XITK_HV_H (wp->w.size);
    event.height = XITK_HV_V (wp->w.size);
    _label_paint (wp, &event);
    wp->w.shown_state = wp->w.state;
  }
  if (wp->anim)
    pthread_mutex_unlock (&wp->anim->mutex);
}

/*
 *
 */
static void _label_set_skin (_label_private_t *wp, const xitk_skin_element_info_t *info) {
  XITK_HV_INIT;

  if (info) {
    xitk_short_string_set (&wp->fontname, info->label_fontname);
    wp->bg_color      = xitk_color_db_get (wp->w.wl->xitk, info->label_color_bg);
    wp->text_color[0] = xitk_color_db_get (wp->w.wl->xitk, info->label_color);
    wp->text_color[1] = xitk_color_db_get (wp->w.wl->xitk, info->label_color_focus);
    wp->label_visible = info->label_printable;
    wp->length        = info->label_length;
    wp->align         = info->label_alignment;

    wp->animation  = info->label_animation;
    wp->anim_step  = info->label_animation_step;
    wp->anim_timer = info->label_animation_timer;
    if (wp->anim_timer <= 0)
      wp->anim_timer = xitk_get_cfg_num (wp->w.wl->xitk, XITK_TIMER_LABEL_ANIM);
    if (!wp->anim_step || (wp->anim_timer <= 0))
      wp->animation = 0;

    XITK_HV_H (wp->w.pos) = info->x;
    XITK_HV_V (wp->w.pos) = info->y;
    xitk_widget_state_from_info (&wp->w, info);

    if (info->label_pixmap_font_img) {
      wp->font = info->label_pixmap_font_img;
      wp->highlight_font = info->label_pixmap_highlight_font_img;
      wp->pix_font = wp->font->pix_font;
      if (!wp->pix_font) {
        xitk_image_set_pix_font (wp->font, "");
        wp->pix_font    = wp->font->pix_font;
      }
      XITK_HV_H (wp->w.size) = wp->pix_font->char_width * wp->length;
      XITK_HV_V (wp->w.size) = wp->pix_font->char_height;
      return;
    } else if (wp->fontname.s[0]) {
      wp->font = NULL;
      wp->highlight_font = NULL;
      wp->pix_font = NULL;
      XITK_HV_H (wp->w.size) = wp->length;
      XITK_HV_V (wp->w.size) = info->label_height;
      return;
    }
  }
  /* not used by this skin. */
  wp->w.pos.w = 0;
  wp->w.size.w = 0;
  wp->w.state      &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
  wp->label_visible = 0;
  wp->length        = 0;
  wp->animation     = 0;
  wp->anim_step     = 0;
  wp->anim_timer    = 0;
  wp->font          = NULL;
  wp->highlight_font = NULL;
  wp->pix_font      = NULL;
  xitk_image_free_image (&wp->labelpix);
}

/*
 *
 */
int xitk_label_change_label(xitk_widget_t *w, const char *newlabel) {
  _label_private_t *wp;

  xitk_container (wp, w, w);
  if (wp && ((wp->w.type & WIDGET_TYPE_MASK) == WIDGET_TYPE_LABEL)) {
    size_t s;
    /* application may stop hidden labels. */
    if (wp->w.wl->flags & XITK_WL_HIDDEN) {
      if (wp->flags & _LABEL_FLAG_HIDDEN)
        return 1;
      if (newlabel && !newlabel[0])
        wp->flags |= _LABEL_FLAG_HIDDEN;
    } else {
      wp->flags &= ~_LABEL_FLAG_HIDDEN;
    }
    s = xitk_short_string_set (&wp->label, newlabel);
    if (s != (size_t)-1) {
      wp->label_len = s;
      if (wp->flags & _LABEL_FLAG_PAINTED)
        _label_setup_label (wp);
    }
    return 1;
  }
  return 0;
}

/*
 *
 */
static int _label_input (_label_private_t *wp, const widget_event_t *event) {
  int  ret = 0, fire;

  if (!(wp->w.state & XITK_WIDGET_STATE_FOCUS))
    return 0;

  if (event->type == WIDGET_EVENT_KEY) {
    static const char k[] = {
      XITK_CTRL_KEY_PREFIX, XITK_KEY_RETURN,
      ' ', 0
    };
    int i, n = sizeof (k) / sizeof (k[0]);

    if (event->modifier & ~(MODIFIER_SHIFT | MODIFIER_NUML))
      return 0;
    if (!event->string)
      return 0;
    for (i = 0; i < n; i += 2) {
      if (!memcmp (event->string, k + i, 2))
        break;
    }
    if (i >= n)
      return 0;
  } else { /* WIDGET_EVENT_CLICK */
    if (event->button != 1)
      return 0;
  }

  fire = event->pressed ? ~0u : 0;
  wp->w.state &= ~XITK_WIDGET_STATE_CLICK;
  wp->w.state |= fire & XITK_WIDGET_STATE_CLICK;
  fire = (wp->w.state ^ ~fire) & XITK_WIDGET_STATE_IMMEDIATE;
  if (wp->callback) {
    if (fire)
      wp->callback (&wp->w, wp->w.userdata);
    ret = 1;
  }
  return ret;
}

static int notify_event (xitk_widget_t *w, const widget_event_t *event) {
  _label_private_t *wp;

  xitk_container (wp, w, w);
  if (!wp)
    return 0;
  if ((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_LABEL)
    return 0;
  if (!event)
    return 0;

  switch (event->type) {
    case WIDGET_EVENT_PAINT:
      if (!(wp->flags & _LABEL_FLAG_PAINTED)) {
        /* defer until first paint event. */
        wp->flags |= _LABEL_FLAG_PAINTED;
        _label_setup_label (wp);
      } else if (!wp->anim) {
        _label_paint (wp, event);
      } else if (!pthread_mutex_trylock (&wp->anim->mutex)) {
        _label_paint (wp, event);
        pthread_mutex_unlock (&wp->anim->mutex);
      }
      break;
    case WIDGET_EVENT_CLICK:
    case WIDGET_EVENT_KEY:
      return _label_input (wp, event);
    case WIDGET_EVENT_CHANGE_SKIN:
      if (wp->w.skin_element_name[0]) {
        XITK_HV_INIT;
        if (wp->anim) {
          int n = _label_anim_leave (wp);
          if (wp->w.wl->xitk->verbosity >= 2)
            printf ("xitk.label.anim.leave (%iusec, %i).\n", wp->anim_timer, n);
        }
        wp->flags &= ~_LABEL_FLAG_PAINTED;
        _label_set_skin (wp, xitk_skin_get_info (event->skonfig, wp->w.skin_element_name));
        xitk_set_widget_pos (&wp->w, XITK_HV_H (wp->w.pos), XITK_HV_V (wp->w.pos));
        if (wp->callback) {
          if (wp->highlight_font)
            wp->w.type |= WIDGET_TABABLE | WIDGET_KEYABLE;
          else
            wp->w.type &= ~(WIDGET_TABABLE | WIDGET_KEYABLE);
        }
      }
      break;
    case WIDGET_EVENT_DESTROY:
      _label_destroy (wp);
      break;
    case WIDGET_EVENT_GET_SKIN:
      if (event->image) {
        *event->image = (event->skin_layer == FOREGROUND_SKIN) ? wp->font : NULL;
        return 1;
      }
      break;
    default: ;
  }
  return 0;
}

/*
 *
 */
static xitk_widget_t *_label_create (_label_private_t *wp, const xitk_label_widget_t *l) {
  wp->callback = l->callback;

  wp->anim = NULL;

  wp->w.type = WIDGET_TYPE_LABEL | WIDGET_CLICKABLE | WIDGET_PARTIAL_PAINTABLE;
  if (wp->callback) {
    wp->w.type |= WIDGET_FOCUSABLE;
    if (wp->highlight_font || (wp->text_color[0] != wp->text_color[1]))
      wp->w.type |= WIDGET_TABABLE | WIDGET_KEYABLE;
  }
  wp->w.event = notify_event;

  xitk_short_string_init (&wp->label);
  wp->label_len = xitk_short_string_set (&wp->label, l->label);
  if (wp->label_len == (size_t)-1)
    wp->label_len = 0;
  wp->flags &= ~_LABEL_FLAG_PAINTED;

  _xitk_new_widget_apply (&l->nw, &wp->w);

  return &wp->w;
}

/*
 *
 */
xitk_widget_t *xitk_label_create (const xitk_label_widget_t *l, xitk_skin_config_t *skonfig) {
  const xitk_skin_element_info_t *info;
  _label_private_t *wp;

  wp = (_label_private_t *)xitk_widget_new (&l->nw, sizeof (*wp));
  if (!wp)
    return NULL;
  xitk_short_string_init (&wp->fontname);

  info = xitk_skin_get_info (skonfig, l->nw.skin_element_name);
  _label_set_skin (wp, info);

  return _label_create (wp, l);
}

/*
 *
 */
static uint32_t _xitk_label_color (xitk_t *xitk, uint32_t rgb) {
  if (rgb & XITK_NOSKIN_DEFAULT)
    return xitk_get_cfg_num (xitk,
        (rgb == XITK_NOSKIN_DEFAULT) ? XITK_BG_COLOR :
        (rgb == XITK_NOSKIN_TEXT_INV) ? XITK_WHITE_COLOR :
        XITK_BLACK_COLOR);
  return xitk_color_db_get (xitk, rgb);
}

xitk_widget_t *xitk_noskin_label_color_create (const xitk_label_widget_t *l,
  int x, int y, int width, int height, const char *fontname, uint32_t text_color, uint32_t bg_color) {
  _label_private_t *wp;
  XITK_HV_INIT;

  wp = (_label_private_t *)xitk_widget_new (&l->nw, sizeof (*wp));
  if (!wp)
    return NULL;

  wp->w.skin_element_name[0] = 0;
  wp->w.state    &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
  XITK_HV_H (wp->w.pos) = x;
  XITK_HV_V (wp->w.pos) = y;
  XITK_HV_V (wp->w.size) = height;
  XITK_HV_H (wp->w.size) =
  wp->length    = width;
  wp->pix_font  = NULL;
  wp->text_color[0] =
  wp->text_color[1] = _xitk_label_color (l->nw.wl->xitk, text_color);
  wp->bg_color   = _xitk_label_color (l->nw.wl->xitk, bg_color);
  wp->animation = 0;
  wp->anim_want = 0;
  wp->anim_step = 1;
  wp->anim_timer = xitk_get_cfg_num (wp->w.wl->xitk, XITK_TIMER_LABEL_ANIM);
  wp->label_visible = 1;

  xitk_short_string_init (&wp->fontname);
  xitk_short_string_set (&wp->fontname, fontname);

  xitk_image_temp_size (wp->w.wl->xitk, width, height);
  if (l->nw.skin_element_name && !strcmp (l->nw.skin_element_name, "XITK_NOSKIN_INNER")) {
    wp->bg_color = ~0;
    if (xitk_shared_image (wp->w.wl, "xitk_label_i", width, height, &wp->font) == 1) {
      xitk_image_draw_relief (wp->font, width, height, XITK_DRAW_FLATTER);
      xitk_image_draw_rectangular_box (wp->font, 0, 0, width, height, XITK_DRAW_INNER);
    }
  } else {
    /* plain background color. */
    wp->font = NULL;
  }

  wp->highlight_font = NULL;

  return _label_create (wp, l);
}
